/* example.c -- usage example of the zlib compression library
 * Copyright (C) 1995-2004 Jean-loup Gailly.
 * For conditions of distribution and use, see copyright notice in zlib.h
 */

/* @(#) $Id: example.c,v 1.3 2015/10/02 08:58:46 penou Exp $ */

#include <stdio.h>
#include "zlib.h"

#ifdef STDC
#  include <string.h>
#  include <stdlib.h>
#endif

#if defined(VMS) || defined(RISCOS)
#  define TESTFILE "foo-gz"
#else
#  define TESTFILE "foo.gz"
#endif

#define CHECK_ERR(err, msg) { \
    if (err != Z_OK) { \
        fprintf(stderr, "%s error: %d\n", msg, err); \
        exit(1); \
    } \
}

const char hello[] = "hello, hello!";
/* "hello world" would be more standard, but the repeated "hello"
 * stresses the compression code better, sorry...
 */

const char dictionary[] = "hello";
uLong dictId; /* Adler32 value of the dictionary */

void test_compressXXX      OF((Byte *compr, uLong comprLen,
                            Byte *uncompr, uLong uncomprLen));
void test_gzio          OF((const char *fname,
                            Byte *uncompr, uLong uncomprLen));
void test_deflateXXX       OF((Byte *compr, uLong comprLen));
void test_inflateXXX       OF((Byte *compr, uLong comprLen,
                            Byte *uncompr, uLong uncomprLen));
void test_large_deflateXXX OF((Byte *compr, uLong comprLen,
                            Byte *uncompr, uLong uncomprLen));
void test_large_inflateXXX OF((Byte *compr, uLong comprLen,
                            Byte *uncompr, uLong uncomprLen));
void test_flush         OF((Byte *compr, uLong *comprLen));
void test_sync          OF((Byte *compr, uLong comprLen,
                            Byte *uncompr, uLong uncomprLen));
void test_dict_deflateXXX  OF((Byte *compr, uLong comprLen));
void test_dict_inflateXXX  OF((Byte *compr, uLong comprLen,
                            Byte *uncompr, uLong uncomprLen));
int  main               OF((int argc, char *argv[]));

/* ===========================================================================
 * Test compressXXX() and uncompressXXX()
 */
void test_compressXXX(compr, comprLen, uncompr, uncomprLen)
    Byte *compr, *uncompr;
    uLong comprLen, uncomprLen;
{
    int err;
    uLong len = (uLong)strlen(hello)+1;

    err = compressXXX(compr, &comprLen, (const Bytef*)hello, len);
    CHECK_ERR(err, "compressXXX");

    strcpy((char*)uncompr, "garbage");

    err = uncompressXXX(uncompr, &uncomprLen, compr, comprLen);
    CHECK_ERR(err, "uncompressXXX");

    if (strcmp((char*)uncompr, hello)) {
        fprintf(stderr, "bad uncompressXXX\n");
        exit(1);
    } else {
        printf("uncompressXXX(): %s\n", (char *)uncompr);
    }
}

/* ===========================================================================
 * Test read/write of .gz files
 */
void test_gzio(fname, uncompr, uncomprLen)
    const char *fname; /* compressed file name */
    Byte *uncompr;
    uLong uncomprLen;
{
#ifdef NO_GZCOMPRESS
    fprintf(stderr, "NO_GZCOMPRESS -- gz* functions cannot compressXXX\n");
#else
    int err;
    int len = (int)strlen(hello)+1;
    gzFile file;
    z_off_t pos;

    file = gzopenXXX(fname, "wb");
    if (file == NULL) {
        fprintf(stderr, "gzopenXXX error\n");
        exit(1);
    }
    gzputcXXX(file, 'h');
    if (gzputsXXX(file, "ello") != 4) {
        fprintf(stderr, "gzputsXXX err: %s\n", gzerrorXXX(file, &err));
        exit(1);
    }
    if (gzprintfXXX(file, ", %s!", "hello") != 8) {
        fprintf(stderr, "gzprintfXXX err: %s\n", gzerrorXXX(file, &err));
        exit(1);
    }
    gzseekXXX(file, 1L, SEEK_CUR); /* add one zero byte */
    gzcloseXXX(file);

    file = gzopenXXX(fname, "rb");
    if (file == NULL) {
        fprintf(stderr, "gzopenXXX error\n");
        exit(1);
    }
    strcpy((char*)uncompr, "garbage");

    if (gzreadXXX(file, uncompr, (unsigned)uncomprLen) != len) {
        fprintf(stderr, "gzreadXXX err: %s\n", gzerrorXXX(file, &err));
        exit(1);
    }
    if (strcmp((char*)uncompr, hello)) {
        fprintf(stderr, "bad gzreadXXX: %s\n", (char*)uncompr);
        exit(1);
    } else {
        printf("gzreadXXX(): %s\n", (char*)uncompr);
    }

    pos = gzseekXXX(file, -8L, SEEK_CUR);
    if (pos != 6 || gztellXXX(file) != pos) {
        fprintf(stderr, "gzseekXXX error, pos=%ld, gztellXXX=%ld\n",
                (long)pos, (long)gztellXXX(file));
        exit(1);
    }

    if (gzgetcXXX(file) != ' ') {
        fprintf(stderr, "gzgetcXXX error\n");
        exit(1);
    }

    if (gzungetcXXX(' ', file) != ' ') {
        fprintf(stderr, "gzungetcXXX error\n");
        exit(1);
    }

    gzgetsXXX(file, (char*)uncompr, (int)uncomprLen);
    if (strlen((char*)uncompr) != 7) { /* " hello!" */
        fprintf(stderr, "gzgetsXXX err after gzseekXXX: %s\n", gzerrorXXX(file, &err));
        exit(1);
    }
    if (strcmp((char*)uncompr, hello + 6)) {
        fprintf(stderr, "bad gzgetsXXX after gzseekXXX\n");
        exit(1);
    } else {
        printf("gzgetsXXX() after gzseekXXX: %s\n", (char*)uncompr);
    }

    gzcloseXXX(file);
#endif
}

/* ===========================================================================
 * Test deflateXXX() with small buffers
 */
void test_deflateXXX(compr, comprLen)
    Byte *compr;
    uLong comprLen;
{
    z_stream c_stream; /* compression stream */
    int err;
    uLong len = (uLong)strlen(hello)+1;

    c_stream.zalloc = (alloc_func)0;
    c_stream.zfree = (free_func)0;
    c_stream.opaque = (voidpf)0;

    err = deflateInit(&c_stream, Z_DEFAULT_COMPRESSION);
    CHECK_ERR(err, "deflateInit");

    c_stream.next_in  = (Bytef*)hello;
    c_stream.next_out = compr;

    while (c_stream.total_in != len && c_stream.total_out < comprLen) {
        c_stream.avail_in = c_stream.avail_out = 1; /* force small buffers */
        err = deflateXXX(&c_stream, Z_NO_FLUSH);
        CHECK_ERR(err, "deflateXXX");
    }
    /* Finish the stream, still forcing small buffers: */
    for (;;) {
        c_stream.avail_out = 1;
        err = deflateXXX(&c_stream, Z_FINISH);
        if (err == Z_STREAM_END) break;
        CHECK_ERR(err, "deflateXXX");
    }

    err = deflateEndXXX(&c_stream);
    CHECK_ERR(err, "deflateEndXXX");
}

/* ===========================================================================
 * Test inflateXXX() with small buffers
 */
void test_inflateXXX(compr, comprLen, uncompr, uncomprLen)
    Byte *compr, *uncompr;
    uLong comprLen, uncomprLen;
{
    int err;
    z_stream d_stream; /* decompression stream */

    strcpy((char*)uncompr, "garbage");

    d_stream.zalloc = (alloc_func)0;
    d_stream.zfree = (free_func)0;
    d_stream.opaque = (voidpf)0;

    d_stream.next_in  = compr;
    d_stream.avail_in = 0;
    d_stream.next_out = uncompr;

    err = inflateInit(&d_stream);
    CHECK_ERR(err, "inflateInit");

    while (d_stream.total_out < uncomprLen && d_stream.total_in < comprLen) {
        d_stream.avail_in = d_stream.avail_out = 1; /* force small buffers */
        err = inflateXXX(&d_stream, Z_NO_FLUSH);
        if (err == Z_STREAM_END) break;
        CHECK_ERR(err, "inflateXXX");
    }

    err = inflateEndXXX(&d_stream);
    CHECK_ERR(err, "inflateEndXXX");

    if (strcmp((char*)uncompr, hello)) {
        fprintf(stderr, "bad inflateXXX\n");
        exit(1);
    } else {
        printf("inflateXXX(): %s\n", (char *)uncompr);
    }
}

/* ===========================================================================
 * Test deflateXXX() with large buffers and dynamic change of compression level
 */
void test_large_deflateXXX(compr, comprLen, uncompr, uncomprLen)
    Byte *compr, *uncompr;
    uLong comprLen, uncomprLen;
{
    z_stream c_stream; /* compression stream */
    int err;

    c_stream.zalloc = (alloc_func)0;
    c_stream.zfree = (free_func)0;
    c_stream.opaque = (voidpf)0;

    err = deflateInit(&c_stream, Z_BEST_SPEED);
    CHECK_ERR(err, "deflateInit");

    c_stream.next_out = compr;
    c_stream.avail_out = (uInt)comprLen;

    /* At this point, uncompr is still mostly zeroes, so it should compressXXX
     * very well:
     */
    c_stream.next_in = uncompr;
    c_stream.avail_in = (uInt)uncomprLen;
    err = deflateXXX(&c_stream, Z_NO_FLUSH);
    CHECK_ERR(err, "deflateXXX");
    if (c_stream.avail_in != 0) {
        fprintf(stderr, "deflateXXX not greedy\n");
        exit(1);
    }

    /* Feed in already compressed data and switch to no compression: */
    deflateParamsXXX(&c_stream, Z_NO_COMPRESSION, Z_DEFAULT_STRATEGY);
    c_stream.next_in = compr;
    c_stream.avail_in = (uInt)comprLen/2;
    err = deflateXXX(&c_stream, Z_NO_FLUSH);
    CHECK_ERR(err, "deflateXXX");

    /* Switch back to compressing mode: */
    deflateParamsXXX(&c_stream, Z_BEST_COMPRESSION, Z_FILTERED);
    c_stream.next_in = uncompr;
    c_stream.avail_in = (uInt)uncomprLen;
    err = deflateXXX(&c_stream, Z_NO_FLUSH);
    CHECK_ERR(err, "deflateXXX");

    err = deflateXXX(&c_stream, Z_FINISH);
    if (err != Z_STREAM_END) {
        fprintf(stderr, "deflateXXX should report Z_STREAM_END\n");
        exit(1);
    }
    err = deflateEndXXX(&c_stream);
    CHECK_ERR(err, "deflateEndXXX");
}

/* ===========================================================================
 * Test inflateXXX() with large buffers
 */
void test_large_inflateXXX(compr, comprLen, uncompr, uncomprLen)
    Byte *compr, *uncompr;
    uLong comprLen, uncomprLen;
{
    int err;
    z_stream d_stream; /* decompression stream */

    strcpy((char*)uncompr, "garbage");

    d_stream.zalloc = (alloc_func)0;
    d_stream.zfree = (free_func)0;
    d_stream.opaque = (voidpf)0;

    d_stream.next_in  = compr;
    d_stream.avail_in = (uInt)comprLen;

    err = inflateInit(&d_stream);
    CHECK_ERR(err, "inflateInit");

    for (;;) {
        d_stream.next_out = uncompr;            /* discard the output */
        d_stream.avail_out = (uInt)uncomprLen;
        err = inflateXXX(&d_stream, Z_NO_FLUSH);
        if (err == Z_STREAM_END) break;
        CHECK_ERR(err, "large inflateXXX");
    }

    err = inflateEndXXX(&d_stream);
    CHECK_ERR(err, "inflateEndXXX");

    if (d_stream.total_out != 2*uncomprLen + comprLen/2) {
        fprintf(stderr, "bad large inflateXXX: %ld\n", d_stream.total_out);
        exit(1);
    } else {
        printf("large_inflateXXX(): OK\n");
    }
}

/* ===========================================================================
 * Test deflateXXX() with full flush
 */
void test_flush(compr, comprLen)
    Byte *compr;
    uLong *comprLen;
{
    z_stream c_stream; /* compression stream */
    int err;
    uInt len = (uInt)strlen(hello)+1;

    c_stream.zalloc = (alloc_func)0;
    c_stream.zfree = (free_func)0;
    c_stream.opaque = (voidpf)0;

    err = deflateInit(&c_stream, Z_DEFAULT_COMPRESSION);
    CHECK_ERR(err, "deflateInit");

    c_stream.next_in  = (Bytef*)hello;
    c_stream.next_out = compr;
    c_stream.avail_in = 3;
    c_stream.avail_out = (uInt)*comprLen;
    err = deflateXXX(&c_stream, Z_FULL_FLUSH);
    CHECK_ERR(err, "deflateXXX");

    compr[3]++; /* force an error in first compressed block */
    c_stream.avail_in = len - 3;

    err = deflateXXX(&c_stream, Z_FINISH);
    if (err != Z_STREAM_END) {
        CHECK_ERR(err, "deflateXXX");
    }
    err = deflateEndXXX(&c_stream);
    CHECK_ERR(err, "deflateEndXXX");

    *comprLen = c_stream.total_out;
}

/* ===========================================================================
 * Test inflateSyncXXX()
 */
void test_sync(compr, comprLen, uncompr, uncomprLen)
    Byte *compr, *uncompr;
    uLong comprLen, uncomprLen;
{
    int err;
    z_stream d_stream; /* decompression stream */

    strcpy((char*)uncompr, "garbage");

    d_stream.zalloc = (alloc_func)0;
    d_stream.zfree = (free_func)0;
    d_stream.opaque = (voidpf)0;

    d_stream.next_in  = compr;
    d_stream.avail_in = 2; /* just read the zlib header */

    err = inflateInit(&d_stream);
    CHECK_ERR(err, "inflateInit");

    d_stream.next_out = uncompr;
    d_stream.avail_out = (uInt)uncomprLen;

    inflateXXX(&d_stream, Z_NO_FLUSH);
    CHECK_ERR(err, "inflateXXX");

    d_stream.avail_in = (uInt)comprLen-2;   /* read all compressed data */
    err = inflateSyncXXX(&d_stream);           /* but skip the damaged part */
    CHECK_ERR(err, "inflateSyncXXX");

    err = inflateXXX(&d_stream, Z_FINISH);
    if (err != Z_DATA_ERROR) {
        fprintf(stderr, "inflateXXX should report DATA_ERROR\n");
        /* Because of incorrect adlerXXX32 */
        exit(1);
    }
    err = inflateEndXXX(&d_stream);
    CHECK_ERR(err, "inflateEndXXX");

    printf("after inflateSyncXXX(): hel%s\n", (char *)uncompr);
}

/* ===========================================================================
 * Test deflateXXX() with preset dictionary
 */
void test_dict_deflateXXX(compr, comprLen)
    Byte *compr;
    uLong comprLen;
{
    z_stream c_stream; /* compression stream */
    int err;

    c_stream.zalloc = (alloc_func)0;
    c_stream.zfree = (free_func)0;
    c_stream.opaque = (voidpf)0;

    err = deflateInit(&c_stream, Z_BEST_COMPRESSION);
    CHECK_ERR(err, "deflateInit");

    err = deflateSetDictionaryXXX(&c_stream,
                               (const Bytef*)dictionary, sizeof(dictionary));
    CHECK_ERR(err, "deflateSetDictionaryXXX");

    dictId = c_stream.adlerXXX;
    c_stream.next_out = compr;
    c_stream.avail_out = (uInt)comprLen;

    c_stream.next_in = (Bytef*)hello;
    c_stream.avail_in = (uInt)strlen(hello)+1;

    err = deflateXXX(&c_stream, Z_FINISH);
    if (err != Z_STREAM_END) {
        fprintf(stderr, "deflateXXX should report Z_STREAM_END\n");
        exit(1);
    }
    err = deflateEndXXX(&c_stream);
    CHECK_ERR(err, "deflateEndXXX");
}

/* ===========================================================================
 * Test inflateXXX() with a preset dictionary
 */
void test_dict_inflateXXX(compr, comprLen, uncompr, uncomprLen)
    Byte *compr, *uncompr;
    uLong comprLen, uncomprLen;
{
    int err;
    z_stream d_stream; /* decompression stream */

    strcpy((char*)uncompr, "garbage");

    d_stream.zalloc = (alloc_func)0;
    d_stream.zfree = (free_func)0;
    d_stream.opaque = (voidpf)0;

    d_stream.next_in  = compr;
    d_stream.avail_in = (uInt)comprLen;

    err = inflateInit(&d_stream);
    CHECK_ERR(err, "inflateInit");

    d_stream.next_out = uncompr;
    d_stream.avail_out = (uInt)uncomprLen;

    for (;;) {
        err = inflateXXX(&d_stream, Z_NO_FLUSH);
        if (err == Z_STREAM_END) break;
        if (err == Z_NEED_DICT) {
            if (d_stream.adlerXXX != dictId) {
                fprintf(stderr, "unexpected dictionary");
                exit(1);
            }
            err = inflateSetDictionaryXXX(&d_stream, (const Bytef*)dictionary,
                                       sizeof(dictionary));
        }
        CHECK_ERR(err, "inflateXXX with dict");
    }

    err = inflateEndXXX(&d_stream);
    CHECK_ERR(err, "inflateEndXXX");

    if (strcmp((char*)uncompr, hello)) {
        fprintf(stderr, "bad inflateXXX with dict\n");
        exit(1);
    } else {
        printf("inflateXXX with dictionary: %s\n", (char *)uncompr);
    }
}

/* ===========================================================================
 * Usage:  example [output.gz  [input.gz]]
 */

int main(argc, argv)
    int argc;
    char *argv[];
{
    Byte *compr, *uncompr;
    uLong comprLen = 10000*sizeof(int); /* don't overflow on MSDOS */
    uLong uncomprLen = comprLen;
    static const char* myVersion = ZLIB_VERSION;

    if (zlibVersionXXX()[0] != myVersion[0]) {
        fprintf(stderr, "incompatible zlib version\n");
        exit(1);

    } else if (strcmp(zlibVersionXXX(), ZLIB_VERSION) != 0) {
        fprintf(stderr, "warning: different zlib version\n");
    }

    printf("zlib version %s = 0x%04x, compile flags = 0x%lx\n",
            ZLIB_VERSION, ZLIB_VERNUM, zlibCompileFlagsXXX());

    compr    = (Byte*)calloc((uInt)comprLen, 1);
    uncompr  = (Byte*)calloc((uInt)uncomprLen, 1);
    /* compr and uncompr are cleared to avoid reading uninitialized
     * data and to ensure that uncompr compresses well.
     */
    if (compr == Z_NULL || uncompr == Z_NULL) {
        printf("out of memory\n");
        exit(1);
    }
    test_compressXXX(compr, comprLen, uncompr, uncomprLen);

    test_gzio((argc > 1 ? argv[1] : TESTFILE),
              uncompr, uncomprLen);

    test_deflateXXX(compr, comprLen);
    test_inflateXXX(compr, comprLen, uncompr, uncomprLen);

    test_large_deflateXXX(compr, comprLen, uncompr, uncomprLen);
    test_large_inflateXXX(compr, comprLen, uncompr, uncomprLen);

    test_flush(compr, &comprLen);
    test_sync(compr, comprLen, uncompr, uncomprLen);
    comprLen = uncomprLen;

    test_dict_deflateXXX(compr, comprLen);
    test_dict_inflateXXX(compr, comprLen, uncompr, uncomprLen);

    free(compr);
    free(uncompr);

    return 0;
}
