;+
;Function:	get_thm_esa_cal
;PURPOSE:	
;	Returns esa calibration factors
;INPUT:		
;
;KEYWORDS:
;	sc:		string		themis spacecraft - "a", "b", "c", "d", "e"
;					if not set defaults to "a"
;	ion:		0/1		0= electron esa, 1= ion esa, default=0
;	time:		dblarr		times that relative geometric factor are to be returned	
;
;CREATED BY:	J. McFadden
;VERSION:	1
;LAST MODIFICATION:  07/07/08
;MOD HISTORY:
;
;NOTES:	  
;	
;-

function get_thm_esa_cal,sc=sc,ion=ion,time=time

common tha_esa_cal,esa_cal_time,esa_cal_gf

if not keyword_set(sc) then sc='a' 
case sc of
	'a': scn=0
	'b': scn=1
	'c': scn=2
	'd': scn=3
	'e': scn=4
	'f': scn=5
endcase

; the following an_en_eff arrays store the calibration factors that correct for electron ESA low energy degradation of sensitivity
; these arrays are generated using the eesa_degradation.crib and sheath data, assuming isotropic flux at low energies
; some additional arrays are included that provide a check on the accuracy/repeatability of the calibration values 
; an_en_eff_thm(8energy,8anodes,4times), where the energies are the lowest 8 from highest to lowest
; note that the lowest energy channel of anode 0, an_en_eff_thm(0,8), may get contamination photo-e from axial boom if the usher is negative.

; default values are unity
an_en_eff_tha=fltarr(64,4) & an_en_eff_tha[*,*]=1.
an_en_eff_thb=fltarr(64,4) & an_en_eff_thb[*,*]=1.
an_en_eff_thc=fltarr(64,4) & an_en_eff_thc[*,*]=1.
an_en_eff_thd=fltarr(64,4) & an_en_eff_thd[*,*]=1.
an_en_eff_the=fltarr(64,4) & an_en_eff_the[*,*]=1.

; tha*****************************************

an_en_eff_tha[*,0]=$     
    [0.970246,     0.959944,     0.961091,     0.965873,     0.978508,     0.999449,      1.03346,      1.05356,$
     0.985492,     0.981186,     0.980432,     0.981259,     0.983945,     0.990044,     0.996385,     0.997158,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
      1.00887,      1.01194,      1.00592,     0.992630,     0.980136,     0.969536,     0.959396,     0.953436,$
      1.01130,      1.01036,     0.999812,     0.983661,     0.966088,     0.948519,     0.929055,     0.912469,$
     0.997223,     0.994653,     0.986667,     0.977979,     0.967472,     0.958448,     0.944394,     0.924410,$
     0.970484,     0.956822,     0.941076,     0.924177,     0.910496,     0.901672,     0.893342,     0.881946,$
     0.934111,     0.908736,     0.888911,     0.873541,     0.865833,     0.875016,     0.905417,     0.940829]	;20070821

															;20070731	tha-x94
															;20070829	tha-x98
															;20080516	tha-x9c
an_en_eff_tha[*,2]=$     
    [0.920547,     0.968215,     0.979017,     0.973129,     0.963922,     0.958283,     0.977678,      1.02312,$
     0.943339,     0.982025,     0.996464,     0.991404,     0.981213,     0.978590,     0.974821,     0.981053,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
      1.03514,     0.999551,     0.986523,     0.985106,     0.976622,     0.969736,     0.960268,     0.953308,$
      1.00915,     0.977033,     0.962180,     0.951822,     0.928991,     0.909241,     0.877918,     0.848887,$
     0.922926,     0.932795,     0.934485,     0.923287,     0.898535,     0.870875,     0.836776,     0.798979,$
     0.826272,     0.863632,     0.875773,     0.847904,     0.795667,     0.741664,     0.676716,     0.548037,$
     0.769627,     0.818093,     0.834261,     0.808607,     0.757668,     0.704861,     0.631044,     0.537318]	;20081011
an_en_eff_tha[*,1]=an_en_eff_tha[*,2]         

an_en_eff_tha[*,3]=$   
    [0.968658,     0.962897,     0.950162,     0.944699,     0.933456,     0.943188,     0.953039,      1.00751,$
     0.971436,     0.974225,     0.966291,     0.964621,     0.964765,     0.964396,     0.964406,     0.970778,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
      1.03522,      1.02040,      1.00539,     0.991723,     0.980574,     0.972867,     0.961748,     0.958294,$
      1.08368,      1.05140,      1.02259,     0.995496,     0.973655,     0.955999,     0.934171,     0.923291,$
      1.07491,      1.03868,      1.00534,     0.976299,     0.952197,     0.932522,     0.909239,     0.898049,$
      1.05222,      1.01056,     0.957812,     0.913579,     0.878476,     0.849619,     0.803741,     0.778753,$
      1.03629,     0.990560,     0.934096,     0.881347,     0.831936,     0.801176,     0.773552,     0.806499]	;20090909

; thb*****************************************

an_en_eff_thb[*,1]=$     
    [0.868743,     0.923108,     0.950444,     0.940771,     0.929143,     0.916705,     0.937744,      1.06922,$
     0.909902,     0.940325,     0.955193,     0.943212,     0.925497,     0.896189,     0.866267,     0.853189,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
      1.09402,      1.03050,     0.981454,     0.953723,     0.937718,     0.919758,     0.890968,     0.868093,$
      1.16877,      1.07000,     0.999464,     0.963760,     0.946812,     0.929274,     0.906458,     0.888203,$
      1.20638,      1.11084,      1.04089,      1.00988,     0.994768,     0.990803,     0.980493,     0.974541,$
      1.22294,      1.11776,      1.00952,     0.938111,     0.891947,     0.846102,     0.809480,     0.757584,$
      1.20773,      1.12723,      1.00227,     0.904451,     0.849986,     0.811222,     0.818501,     0.942578]	;20080516


an_en_eff_thb[*,2]=$     
    [0.966468,     0.963295,     0.952980,     0.942607,     0.930156,     0.928338,     0.955715,      1.05304,$
     0.978736,     0.973321,     0.950263,     0.933489,     0.912223,     0.891515,     0.863400,     0.851573,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
     0.995133,     0.982071,     0.968979,     0.952431,     0.933880,     0.909767,     0.876671,     0.856082,$
     0.990179,     0.981464,     0.969968,     0.953023,     0.935636,     0.917487,     0.891614,     0.869601,$
     0.977501,     0.982055,     0.981035,     0.975537,     0.971973,     0.968725,     0.966630,     0.960207,$
     0.954211,     0.945811,     0.914853,     0.888422,     0.851928,     0.821655,     0.797862,     0.731251,$
     0.920049,     0.910694,     0.889455,     0.857747,     0.826485,     0.805688,     0.832386,     0.894656]	;20081010

an_en_eff_thb[*,3]=$     
    [0.965489,     0.955962,     0.946287,     0.930873,     0.923395,     0.910015,     0.922744,     0.971359,$
     0.976798,     0.969149,     0.956386,     0.926246,     0.911847,     0.890472,     0.867139,     0.854935,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
      1.00468,     0.989727,     0.975924,     0.959714,     0.943337,     0.921426,     0.888146,     0.855797,$
      1.00978,     0.992018,     0.977701,     0.960191,     0.946397,     0.928760,     0.904264,     0.874419,$
      1.00380,     0.991637,     0.985467,     0.981153,     0.977588,     0.974576,     0.969548,     0.962416,$
     0.985766,     0.954398,     0.926598,     0.883301,     0.858826,     0.821204,     0.800336,     0.754719,$
     0.961244,     0.923256,     0.897785,     0.859745,     0.830524,     0.800667,     0.806499,     0.832744]	;20090713
an_en_eff_thb[*,3]=$     
    [0.965489,     0.955962,     0.946287,     0.930873,     0.923395,     0.910015,     0.900000,     0.890000,$
     0.976798,     0.969149,     0.956386,     0.926246,     0.911847,     0.890472,     0.867139,     0.854935,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
      1.00468,     0.989727,     0.975924,     0.959714,     0.943337,     0.921426,     0.888146,     0.855797,$
      1.00978,     0.992018,     0.977701,     0.960191,     0.946397,     0.928760,     0.904264,     0.874419,$
      1.00380,     0.991637,     0.985467,     0.981153,     0.977588,     0.974576,     0.969548,     0.962416,$
     0.985766,     0.954398,     0.926598,     0.883301,     0.858826,     0.821204,     0.800336,     0.754719,$
     0.961244,     0.923256,     0.897785,     0.859745,     0.830524,     0.800667,     0.770000,     0.740000]	;20090713, corrected
an_en_eff_thb[*,3]=$     
    [0.941352,     0.932063,     0.922630,     0.907601,     0.900310,     0.887265,     0.870000,     0.860000,$
     0.952378,     0.944920,     0.932476,     0.903090,     0.889051,     0.868210,     0.845461,     0.833561,$
     0.975000,     0.975000,     0.975000,     0.975000,     0.975000,     0.975000,     0.975000,     0.975000,$
     0.979566,     0.964984,     0.951526,     0.935721,     0.919754,     0.898391,     0.865942,     0.834402,$
      1.03502,      1.01682,      1.00214,     0.984196,     0.970057,     0.951979,     0.926870,     0.896280,$
      1.02889,      1.01643,      1.01010,      1.00568,      1.00203,     0.998941,     0.993787,     0.986477,$
      1.01041,     0.978258,     0.949763,     0.905383,     0.880296,     0.841734,     0.820345,     0.773587,$
     0.985275,     0.946337,     0.920230,     0.881238,     0.851287,     0.820684,     0.790000,     0.760000]	;20090713, corrected, delta=0.05
an_en_eff_thb[*,3]=$     
    [0.926870,     0.917723,     0.908436,     0.893638,     0.886459,     0.873615,     0.860000,     0.850000,$
     0.937726,     0.930383,     0.918130,     0.889197,     0.875373,     0.854853,     0.832453,     0.820737,$
     0.960000,     0.960000,     0.960000,     0.960000,     0.960000,     0.960000,     0.960000,     0.960000,$
     0.964495,     0.950138,     0.936888,     0.921325,     0.905604,     0.884569,     0.852620,     0.821565,$
      1.05017,      1.03170,      1.01681,     0.998599,     0.984253,     0.965910,     0.940434,     0.909396,$
      1.04395,      1.03130,      1.02489,      1.02040,      1.01669,      1.01356,      1.00833,      1.00091,$
      1.02520,     0.992574,     0.963662,     0.918633,     0.893178,     0.854052,     0.832350,     0.784908,$
     0.999693,     0.960186,     0.933697,     0.894134,     0.863744,     0.832694,     0.800000,     0.777777]	;20090713, corrected, delta=0.08


;an_en_eff_thb[*,3]=$     
;    [0.923249,     0.914138,     0.904887,     0.890147,     0.882996,     0.870202,     0.882374,     0.928862,$
;     0.946273,     0.938863,     0.926499,     0.897301,     0.883352,     0.862645,     0.840041,     0.828218,$
;     0.981250,     0.981250,     0.981250,     0.981250,     0.981250,     0.981250,     0.981250,     0.981250,$
;     0.998403,     0.983541,     0.969825,     0.953715,     0.937442,     0.915667,     0.882595,     0.850449,$
;      1.01609,     0.998218,     0.983812,     0.966192,     0.952312,     0.934565,     0.909915,     0.879884,$
;      1.02262,      1.01023,      1.00394,     0.999549,     0.995918,     0.992850,     0.987727,     0.980462,$
;      1.01657,     0.984223,     0.955554,     0.910904,     0.885664,     0.846866,     0.825347,     0.778304,$
;      1.00330,     0.963648,     0.937064,     0.897358,     0.866859,     0.835696,     0.841783,     0.869176]	;20090713, delta=0.05, 1st order
;an_en_eff_thb[*,3]=$     
;    [0.923249,     0.914138,     0.904887,     0.890147,     0.882996,     0.870202,     0.860000,     0.850000,$
;     0.946273,     0.938863,     0.926499,     0.897301,     0.883352,     0.862645,     0.840041,     0.828218,$
;     0.981250,     0.981250,     0.981250,     0.981250,     0.981250,     0.981250,     0.981250,     0.981250,$
;     0.998403,     0.983541,     0.969825,     0.953715,     0.937442,     0.915667,     0.882595,     0.850449,$
;      1.01609,     0.998218,     0.983812,     0.966192,     0.952312,     0.934565,     0.909915,     0.879884,$
;      1.02262,      1.01023,      1.00394,     0.999549,     0.995918,     0.992850,     0.987727,     0.980462,$
;      1.01657,     0.984223,     0.955554,     0.910904,     0.885664,     0.846866,     0.825347,     0.778304,$
;      1.00330,     0.963648,     0.937064,     0.897358,     0.866859,     0.835696,     0.810000,     0.780000]	;20090713, corrected, delta=0.05, 1st order

; thc*****************************************
an_en_eff_thc[*,0]=$     
    [0.964611,     0.945423,     0.933969,     0.921643,     0.903965,     0.886341,     0.892425,     0.944135,$
     0.974425,     0.958047,     0.948767,     0.923040,     0.891187,     0.867447,     0.838166,     0.817591,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
      1.00836,     0.998032,     0.985628,     0.974962,     0.960956,     0.949312,     0.933085,     0.923129,$
      1.02517,      1.01352,     0.996301,     0.980745,     0.963313,     0.950401,     0.934850,     0.924204,$
      1.00932,      1.01388,      1.00548,     0.985083,     0.964953,     0.953297,     0.946468,     0.948778,$
     0.956140,     0.968233,     0.967860,     0.922729,     0.866936,     0.828103,     0.792579,     0.781275,$
     0.931283,     0.962786,     0.976845,     0.935645,     0.885637,     0.851251,     0.848226,     0.883111]	;20070824

an_en_eff_thc[*,1]=$
    [0.943781,      1.05634,      1.01995,     0.935159,     0.884636,     0.824889,     0.733618,     0.689261,$
     0.936985,     0.987317,     0.940816,     0.867180,     0.802260,     0.697928,     0.463366,     0.295262,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
      1.00789,     0.910466,     0.885194,     0.890429,     0.884833,     0.864748,     0.837117,     0.777265,$
      1.03059,     0.881429,     0.855083,     0.873627,     0.876500,     0.861411,     0.834896,     0.793875,$
      1.03959,     0.884703,     0.868777,     0.906072,     0.924023,     0.930281,     0.926551,     0.919645,$
     0.973865,     0.799282,     0.755204,     0.745112,     0.711413,     0.600119,     0.319426,     0.196710,$
     0.967155,     0.815912,     0.787185,     0.792306,     0.792903,     0.755620,     0.645172,     0.580157]	;20080516

an_en_eff_thc[*,2]=$
    [0.863475,     0.865798,     0.863485,     0.856004,     0.858266,     0.856811,     0.885689,      1.01819,$
     0.887959,     0.866176,     0.842182,     0.804601,     0.754221,     0.653341,     0.390111,     0.267276,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
      1.00638,     0.987167,     0.967499,     0.936192,     0.907011,     0.865055,     0.807141,     0.716948,$
     0.998830,     0.982328,     0.961760,     0.936418,     0.907093,     0.872231,     0.820883,     0.753489,$
     0.962212,     0.958905,     0.956652,     0.951530,     0.950866,     0.952162,     0.947613,     0.946810,$
     0.835701,     0.812661,     0.786462,     0.749283,     0.688948,     0.545997,     0.259145,     0.180925,$
     0.804308,     0.807038,     0.804560,     0.800984,     0.793363,     0.768772,     0.726667,     0.891569]	;20081014
an_en_eff_thc[*,2]=$
    [0.863475,     0.865798,     0.863485,     0.856004,     0.858266,     0.856811,     0.856000,     0.856000,$
     0.887959,     0.866176,     0.842182,     0.804601,     0.754221,     0.653341,     0.390111,     0.267276,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
      1.00638,     0.987167,     0.967499,     0.936192,     0.907011,     0.865055,     0.807141,     0.716948,$
     0.998830,     0.982328,     0.961760,     0.936418,     0.907093,     0.872231,     0.820883,     0.753489,$
     0.962212,     0.958905,     0.956652,     0.951530,     0.950866,     0.952162,     0.947613,     0.946810,$
     0.835701,     0.812661,     0.786462,     0.749283,     0.688948,     0.545997,     0.259145,     0.180925,$
     0.804308,     0.807038,     0.804560,     0.800984,     0.793363,     0.768772,     0.726667,     0.700000]	;20081014, corrected

															;20081121 	thc-x94
an_en_eff_thc[*,3]=$
    [0.897731,     0.890470,     0.876801,     0.859903,     0.858163,     0.844853,     0.868255,     0.972552,$
     0.910839,     0.891537,     0.860654,     0.820483,     0.778195,     0.688982,     0.474570,     0.313027,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
     0.999212,     0.970871,     0.949138,     0.926772,     0.903012,     0.867045,     0.826501,     0.766258,$
      1.00009,     0.965213,     0.942511,     0.923380,     0.904459,     0.873835,     0.835556,     0.795282,$
     0.982695,     0.957687,     0.949670,     0.945694,     0.941835,     0.938549,     0.932200,     0.934003,$
     0.874476,     0.832780,     0.797412,     0.755118,     0.707530,     0.587108,     0.302585,     0.191390,$
     0.851099,     0.831145,     0.817510,     0.802655,     0.799070,     0.770771,     0.749356,     0.854352]	;20090727
an_en_eff_thc[*,3]=$
    [0.897731,     0.890470,     0.876801,     0.859903,     0.858163,     0.844853,     0.830000,     0.820000,$
     0.910839,     0.891537,     0.860654,     0.820483,     0.778195,     0.688982,     0.474570,     0.313027,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
     0.999212,     0.970871,     0.949138,     0.926772,     0.903012,     0.867045,     0.826501,     0.766258,$
      1.00009,     0.965213,     0.942511,     0.923380,     0.904459,     0.873835,     0.835556,     0.795282,$
     0.982695,     0.957687,     0.949670,     0.945694,     0.941835,     0.938549,     0.932200,     0.934003,$
     0.874476,     0.832780,     0.797412,     0.755118,     0.707530,     0.587108,     0.302585,     0.191390,$
     0.851099,     0.831145,     0.817510,     0.802655,     0.799070,     0.770771,     0.749356,     0.700000]	;20090727, corrected
an_en_eff_thc[*,3]=$
    [0.861822,     0.854851,     0.841729,     0.825507,     0.823836,     0.811058,     0.805000,     0.800000,$
     0.801538,     0.784553,     0.757376,     0.722025,     0.684812,     0.606304,     0.417622,     0.275464,$
     0.960000,     0.960000,     0.960000,     0.960000,     0.960000,     0.960000,     0.960000,     0.960000,$
     0.959244,     0.932036,     0.911172,     0.889701,     0.866892,     0.832363,     0.793441,     0.735607,$
      1.04010,      1.00382,     0.980211,     0.960316,     0.940638,     0.908788,     0.868978,     0.827093,$
      1.02200,     0.995995,     0.987656,     0.983522,     0.979508,     0.976091,     0.969487,     0.971364,$
     0.979413,     0.932714,     0.893101,     0.845732,     0.792434,     0.657560,     0.338895,     0.214356,$
     0.885143,     0.864391,     0.850210,     0.834761,     0.831033,     0.801602,     0.779330,     0.750000]	;20090727, corrected, delta=0.08
an_en_eff_thc[*,3]=$
    [0.890998,     0.881682,     0.865463,     0.844905,     0.838659,     0.820482,     0.810000,     0.800000,$
     0.890345,     0.865143,     0.827267,     0.777551,     0.725137,     0.629359,     0.418371,     0.266073,$
     0.992500,     0.990132,     0.987069,     0.982558,     0.977273,     0.971154,     0.960526,     0.950000,$
     0.991718,     0.961290,     0.936864,     0.910608,     0.882490,     0.842034,     0.793876,     0.727945,$
      1.00759,     0.974738,     0.954698,     0.939486,     0.925015,     0.899041,     0.868538,     0.835046,$
     0.990065,     0.967138,     0.961950,     0.962189,     0.963240,     0.965623,     0.968997,     0.970000,$
     0.894151,     0.857435,     0.828346,     0.794629,     0.755771,     0.637915,     0.338417,     0.220098,$
     0.857482,     0.839347,     0.828081,     0.816655,     0.817231,     0.793005,     0.778936,     0.770000]	;20090727, corrected, delta=0.1, energy scaled
an_en_eff_thc[*,3]=$
    [0.888978,     0.879046,     0.862062,     0.840406,     0.832808,     0.813170,     0.810000,     0.800000,$
     0.884197,     0.857225,     0.817251,     0.764671,     0.709219,     0.611472,     0.401511,     0.251987,$
     0.990250,     0.987171,     0.983190,     0.977326,     0.970455,     0.962500,     0.948684,     0.935000,$
     0.989470,     0.958416,     0.933182,     0.905758,     0.876333,     0.834531,     0.784088,     0.716451,$
      1.00985,     0.977596,     0.958355,     0.944317,     0.931182,     0.906604,     0.878433,     0.846975,$
     0.992276,     0.969974,     0.965634,     0.967137,     0.969662,     0.973745,     0.970000,     0.970000,$
     0.900054,     0.864831,     0.837626,     0.806483,     0.770243,     0.653157,     0.349167,     0.228711,$
     0.859397,     0.841808,     0.831253,     0.820854,     0.822679,     0.799675,     0.787810,     0.780000]	;20090727, corrected, delta=0.13, energy scaled
an_en_eff_thc[*,3]=$
    [0.887631,     0.877288,     0.859794,     0.837406,     0.828907,     0.808296,     0.800000,     0.790000,$
     0.880098,     0.851946,     0.810573,     0.756085,     0.698607,     0.599547,     0.390272,     0.242596,$
     0.988750,     0.985197,     0.980603,     0.973837,     0.965909,     0.956731,     0.940789,     0.925000,$
     0.987971,     0.956500,     0.930728,     0.902525,     0.872228,     0.829529,     0.777563,     0.708788,$
      1.01135,     0.979500,     0.960792,     0.947539,     0.935293,     0.911645,     0.885029,     0.854928,$
     0.993750,     0.971864,     0.968090,     0.970436,     0.973943,     0.979159,     0.980000,     0.980000,$
     0.903989,     0.869762,     0.843813,     0.814385,     0.779891,     0.663319,     0.356334,     0.234452,$
     0.860673,     0.843448,     0.833367,     0.823654,     0.826311,     0.804121,     0.800000,     0.790000]	;20090727, corrected, delta=0.15, energy scaled

;an_en_eff_thc[*,3]=$
;    [0.819179,     0.812554,     0.800081,     0.784662,     0.783074,     0.770928,     0.760000,     0.750000,$
;     0.853911,     0.835816,     0.806863,     0.769203,     0.729558,     0.645921,     0.444910,     0.293463,$
;     0.962500,     0.962500,     0.962500,     0.962500,     0.962500,     0.962500,     0.962500,     0.962500,$
;     0.986722,     0.958735,     0.937273,     0.915187,     0.891725,     0.856207,     0.816170,     0.756680,$
;      1.01260,     0.977278,     0.954292,     0.934923,     0.915765,     0.884758,     0.846000,     0.805223,$
;      1.01955,     0.993601,     0.985282,     0.981157,     0.977154,     0.973745,     0.967157,     0.969029,$
;     0.929130,     0.884829,     0.847250,     0.802312,     0.751751,     0.623802,     0.321497,     0.203352,$
;     0.925570,     0.903870,     0.889042,     0.872887,     0.868989,     0.838213,     0.814925,     0.800000]	;20090727, corrected, delta=0.10 1st order

; thd*****************************************
an_en_eff_thd[*,0]=$
    [0.968418,     0.915813,     0.897744,     0.886849,     0.870391,     0.850029,     0.885701,      1.01425,$
      1.01804,     0.959762,     0.932737,     0.915777,     0.892625,     0.862741,     0.845373,     0.821916,$
      1.03326,     0.990819,     0.976248,     0.973987,     0.970266,     0.965129,     0.970380,     0.984393,$
      1.01736,     0.982580,     0.968558,     0.956853,     0.941554,     0.921249,     0.908478,     0.898594,$
      1.02279,      1.00326,     0.993152,     0.981834,     0.967913,     0.955209,     0.943644,     0.930132,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
     0.982798,     0.982424,     0.968290,     0.944451,     0.919887,     0.889823,     0.856745,     0.822931,$
     0.956130,     0.970110,     0.970125,     0.959711,     0.938076,     0.910955,     0.900651,     0.908298]	;20070821															;20080129	thd-x94

test=$
    [0.927968,     0.905787,     0.878803,     0.863941,     0.854132,     0.878814,      1.06430,      1.49606,$
     0.952208,     0.921782,     0.903805,     0.879441,     0.853071,     0.829253,     0.800018,     0.722588,$
     0.955561,     0.958942,     0.960299,     0.956773,     0.956287,     0.956123,     0.986354,      1.03238,$
     0.947632,     0.951219,     0.944572,     0.934026,     0.914633,     0.893161,     0.876950,     0.899251,$
     0.981251,     0.982326,     0.978504,     0.969099,     0.952123,     0.933338,     0.917381,     0.920066,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
      1.03206,     0.987081,     0.960077,     0.934036,     0.894991,     0.857224,     0.811581,     0.729913,$
      1.04751,      1.00264,     0.967557,     0.947931,     0.915378,     0.899871,     0.893663,     0.992410]	;20071018burst

test=$
    [0.961500,     0.926549,     0.898791,     0.871279,     0.859049,     0.878230,      1.03899,      1.42318,$
     0.982756,     0.943236,     0.914717,     0.888424,     0.858687,     0.831765,     0.802131,     0.715226,$
     0.976250,     0.966964,     0.962971,     0.960757,     0.959221,     0.961240,     0.986477,      1.02485,$
     0.954795,     0.949134,     0.939964,     0.928661,     0.916520,     0.894385,     0.881556,     0.892191,$
     0.984631,     0.979940,     0.970500,     0.960826,     0.951027,     0.935977,     0.919618,     0.920537,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
      1.02357,     0.998594,     0.967195,     0.932751,     0.899947,     0.865591,     0.815186,     0.729890,$
      1.03657,      1.01878,     0.988549,     0.951034,     0.925799,     0.902867,     0.899046,     0.961847]	;20071018full

test=$
    [0.964758,     0.856831,     0.787105,     0.672043,     0.508604,     0.376476,     0.346472,     0.397914,$
     0.959792,     0.851574,     0.774326,     0.633861,     0.414157,     0.223943,     0.142470,     0.109633,$
     0.994337,     0.935559,     0.922281,     0.918450,     0.906365,     0.871295,     0.812315,     0.788816,$
     0.919517,     0.862082,     0.832577,     0.803079,     0.719822,     0.522307,     0.317996,     0.232556,$
     0.952180,     0.918569,     0.896515,     0.871885,     0.824948,     0.713856,     0.524026,     0.393051,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
     0.952324,     0.910850,     0.842903,     0.727562,     0.506636,     0.288425,     0.171439,     0.130072,$
     0.974979,     0.949492,     0.888989,     0.786730,     0.618619,     0.424169,     0.309504,     0.288879]	;20080614

an_en_eff_thd[*,1]=$
    [0.909786,     0.840362,     0.774951,     0.643039,     0.478141,     0.340998,     0.269092,     0.249174,$
     0.948826,     0.860348,     0.773361,     0.598618,     0.380103,     0.201043,     0.128011,    0.0978733,$
      1.04252,     0.986696,     0.954384,     0.938790,     0.916998,     0.867988,     0.806320,     0.760204,$
     0.991080,     0.918308,     0.866229,     0.815912,     0.711696,     0.484804,     0.301537,     0.216297,$
      1.00444,     0.957985,     0.921041,     0.882538,     0.823119,     0.691906,     0.504074,     0.373237,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
     0.882044,     0.858326,     0.809336,     0.679641,     0.448168,     0.259999,     0.161393,     0.119407,$
     0.855236,     0.859674,     0.835743,     0.738258,     0.557828,     0.386891,     0.275340,     0.219221]	;20080805

test=$
    [0.855986,     0.810470,     0.759218,     0.632638,     0.474107,     0.343505,     0.290918,     0.297849,$
     0.892867,     0.829830,     0.756695,     0.592370,     0.379351,     0.202312,     0.127231,     0.101586,$
     0.991336,     0.965469,     0.953689,     0.947726,     0.929847,     0.890117,     0.838768,     0.807800,$
     0.948210,     0.908675,     0.869071,     0.820899,     0.714327,     0.487047,     0.308265,     0.226775,$
     0.980189,     0.953631,     0.923595,     0.884685,     0.820900,     0.678713,     0.487549,     0.359370,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
     0.897107,     0.852142,     0.791486,     0.669996,     0.437698,     0.255668,     0.157805,     0.117751,$
     0.880555,     0.852280,     0.814006,     0.721176,     0.546380,     0.384316,     0.276849,     0.231832]	;20080826

															;20080916	thd-x9C
an_en_eff_thd[*,2]=$
    [0.905917,     0.894403,     0.821241,     0.653271,     0.483981,     0.348578,     0.288833,     0.300878,$
     0.941334,     0.899432,     0.801448,     0.602983,     0.382458,     0.201774,     0.125213,    0.0998344,$
      1.04099,      1.02106,     0.990431,     0.956856,     0.928688,     0.887623,     0.832987,     0.812081,$
      1.00340,     0.942664,     0.882934,     0.815412,     0.708950,     0.483488,     0.304297,     0.226518,$
      1.01408,     0.968438,     0.923679,     0.880367,     0.813657,     0.678267,     0.486739,     0.360808,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
     0.884557,     0.855532,     0.799406,     0.674824,     0.441937,     0.257560,     0.157322,     0.118194,$
     0.853907,     0.858557,     0.832317,     0.739170,     0.553811,     0.391208,     0.278928,     0.232178]	;20080921

test=$
    [0.878117,     0.825392,     0.769511,     0.633706,     0.473278,     0.345575,     0.287450,     0.300141,$
     0.906426,     0.842029,     0.766117,     0.591474,     0.377628,     0.202323,     0.126366,    0.0998364,$
     0.996617,     0.973354,     0.959857,     0.947021,     0.926942,     0.890573,     0.837748,     0.807014,$
     0.950348,     0.910215,     0.873455,     0.818034,     0.711189,     0.485813,     0.307703,     0.224393,$
     0.977892,     0.951163,     0.921877,     0.882235,     0.816415,     0.679142,     0.489865,     0.364647,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
     0.904845,     0.862417,     0.801171,     0.671853,     0.437307,     0.258220,     0.158915,     0.118385,$
     0.895489,     0.869796,     0.828295,     0.725703,     0.544202,     0.387515,     0.278313,     0.230164]	;20080927

test=$
    [0.774712,     0.762438,     0.723115,     0.607581,     0.468259,     0.363745,     0.383583,     0.449589,$
     0.820445,     0.783153,     0.727670,     0.556133,     0.355772,     0.193720,     0.125918,     0.103180,$
     0.957205,     0.948749,     0.941831,     0.934955,     0.919199,     0.882803,     0.846016,     0.819984,$
     0.958896,     0.913015,     0.870097,     0.809285,     0.687060,     0.449707,     0.286429,     0.220489,$
      1.00516,     0.965910,     0.927450,     0.882423,     0.801750,     0.644977,     0.453759,     0.336954,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
     0.858341,     0.824303,     0.777449,     0.644031,     0.411266,     0.243653,     0.153264,     0.117350,$
     0.826715,     0.818095,     0.785978,     0.697912,     0.518405,     0.378686,     0.293496,     0.286762]	;20081028


an_en_eff_thd[*,3]=$
    [0.794567,     0.947868,     0.912352,     0.720072,     0.516072,     0.361318,     0.300995,     0.267368,$
     0.845925,     0.886948,     0.823497,     0.595790,     0.366668,     0.186587,     0.114448,    0.0909717,$
     0.961280,     0.943948,     0.953048,     0.957960,     0.937575,     0.881792,     0.817600,     0.779927,$
     0.897168,     0.787993,     0.781170,     0.759973,     0.660908,     0.435820,     0.273432,     0.209692,$
     0.923467,     0.835993,     0.839362,     0.836838,     0.789405,     0.650563,     0.467923,     0.356342,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
      1.01787,      1.00687,     0.896908,     0.709912,     0.443345,     0.254820,     0.152032,     0.117575,$
      1.09150,      1.16991,      1.03108,     0.847632,     0.606587,     0.409533,     0.289428,     0.271362]	;20090806


; the*****************************************

an_en_eff_the[*,0]=$
     [1.05963,     0.972917,     0.923245,     0.902717,     0.879371,     0.853237,     0.835533,     0.859425,$
      1.06953,     0.994619,     0.946424,     0.922784,     0.903797,     0.874083,     0.844417,     0.804915,$
      1.07153,      1.02406,      1.00131,      1.00151,      1.00296,      1.00091,     0.997453,     0.996781,$
      1.05136,      1.01845,     0.997033,     0.988177,     0.979698,     0.966929,     0.947206,     0.926871,$
      1.02891,      1.01394,      1.00339,     0.993862,     0.983886,     0.968797,     0.954620,     0.942368,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
     0.994500,     0.983609,     0.968894,     0.950600,     0.941373,     0.928695,     0.903650,     0.885161,$
     0.984697,     0.972716,     0.967318,     0.968354,     0.968453,     0.970134,     0.982432,      1.01466]	;20070821

an_en_eff_the[*,1]=$
     [1.02951,     0.929991,     0.853634,     0.770800,     0.652248,     0.440669,     0.295451,     0.229215,$
      1.04311,     0.956951,     0.885709,     0.802306,     0.706243,     0.510449,     0.315604,     0.207394,$
      1.07964,      1.04553,      1.02057,      1.01064,      1.00295,      1.00312,      1.00218,     0.979481,$
      1.05362,      1.03151,      1.01297,      1.00229,     0.991615,     0.980678,     0.960771,     0.952930,$
      1.02615,      1.01676,      1.00582,     0.999591,     0.987305,     0.976186,     0.956789,     0.942428,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
     0.980163,     0.971334,     0.959071,     0.928977,     0.905015,     0.877851,     0.858863,     0.802349,$
     0.961494,     0.949714,     0.941265,     0.924067,     0.908248,     0.878264,     0.870204,     0.837597]	;20080806

; neglect the following test run, this data has significant internally produced secondaries which distort the measurement
test=$
    [0.959471,     0.913580,     0.853305,     0.773775,     0.659832,     0.463799,     0.368786,     0.385011,$
     0.949486,     0.920973,     0.874984,     0.804239,     0.693759,     0.502581,     0.301634,     0.190230,$
     0.972269,     0.995451,      1.00009,     0.995429,     0.985036,     0.975616,     0.952510,     0.910184,$
     0.943034,     0.964279,     0.971774,     0.965886,     0.957758,     0.942284,     0.908624,     0.889097,$
     0.954235,     0.971841,     0.974109,     0.969760,     0.963260,     0.950930,     0.931826,     0.915375,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
      1.04351,      1.00283,     0.979722,     0.952749,     0.930956,     0.906954,     0.882659,     0.803619,$
      1.07050,      1.03073,     0.990629,     0.963291,     0.954522,     0.951480,     0.993710,      1.05203]	;20080809 															;20080916	the-xA0

															;20080916	the-x98->xA0
; note the slight increase in efficiency for some anodes is probably due to the MCP voltage increase 
an_en_eff_the[*,2]=$
    [0.946424,     0.897998,     0.845360,     0.774872,     0.639604,     0.430745,     0.351562,     0.361390,$
     0.958430,     0.906470,     0.866306,     0.799868,     0.670121,     0.455122,     0.263959,     0.162538,$
      1.01808,      1.01493,      1.01378,      1.01123,      1.00395,     0.990736,     0.956320,     0.902860,$
      1.01719,      1.00804,      1.00106,     0.989213,     0.967620,     0.939122,     0.899300,     0.856866,$
      1.01827,      1.01010,      1.00300,     0.993425,     0.976108,     0.953530,     0.922786,     0.903577,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
     0.976403,     0.952121,     0.938408,     0.921742,     0.894454,     0.875594,     0.815736,     0.680770,$
     0.966446,     0.956029,     0.941142,     0.938293,     0.926736,     0.936021,     0.949350,      1.00379]	;20081011

an_en_eff_the[*,3]=$
    [0.943582,     0.914780,     0.864765,     0.797426,     0.676668,     0.465241,     0.344488,     0.336780,$
     0.951325,     0.921160,     0.880043,     0.817363,     0.706099,     0.506271,     0.308568,     0.194585,$
      1.01890,      1.02907,      1.02690,      1.02232,      1.01721,      1.01135,     0.991601,     0.946312,$
      1.01278,      1.01213,      1.00018,     0.983694,     0.960896,     0.929791,     0.893222,     0.829519,$
      1.01345,      1.01041,      1.00134,     0.989476,     0.972376,     0.953590,     0.924550,     0.895879,$
      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,$
     0.982092,     0.958189,     0.937354,     0.911606,     0.878841,     0.857353,     0.791168,     0.649344,$
     0.981909,     0.969937,     0.952559,     0.944403,     0.931631,     0.933380,     0.943578,     0.974647]	;20090903


; initial test run for the
;test=$
;    [0.959965,     0.924864,     0.875415,     0.795460,     0.675812,     0.463267,     0.328321,     0.298193,$
;     0.970915,     0.937790,     0.886294,     0.818456,     0.709737,     0.508639,     0.309672,     0.198051,$
;      1.03916,      1.04071,      1.03675,      1.02787,      1.02514,      1.01806,     0.998219,     0.959940,$
;      1.03046,      1.02248,      1.00864,     0.989618,     0.970412,     0.936122,     0.902011,     0.841001,$
;      1.02226,      1.01583,      1.00522,     0.992326,     0.979820,     0.957286,     0.928284,     0.900147,$
;      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,      1.00000,     1.00000 ,      1.00000,$
;     0.980870,     0.962875,     0.935286,     0.908879,     0.879179,     0.852086,     0.789176,     0.647795,$
;     0.976151,     0.967618,     0.954430,     0.940759,     0.927784,     0.920864,     0.921881,     0.923356]	;20080903
an_en_eff_the[*,3]=an_en_eff_the[*,2]

;********************************************************************************************
; any values of an_en_eff>1. are probably due to statistical errors, anisotropies, low sheath temperature, or internal scattering
; Since >1 values are unrealistic, we limit the values for efficiency to <=1

	an_en_eff_tha = an_en_eff_tha < 1.
	an_en_eff_thb = an_en_eff_thb < 1.
	an_en_eff_thc = an_en_eff_thc < 1. 
	an_en_eff_thd = an_en_eff_thd < 1.
	an_en_eff_the = an_en_eff_the < 1.

; Assume the efficiencies of the 48eV and 36eV bins are near the shoulder of the flat top distribution
; Therefore these two bins will not be isotropic and will be affected by flow.
; Assume the efficiency of the 28eV bin is correct and estimate the 48eV and 36eV bin corrections from it.
	ind_48=indgen(8)*8
	ind_36=indgen(8)*8+1
	ind_28=indgen(8)*8+2
	an_en_eff_tha[ind_48] = (an_en_eff_tha[ind_28]+3.)/4.
	an_en_eff_thb[ind_48] = (an_en_eff_thb[ind_28]+3.)/4.
	an_en_eff_thc[ind_48] = (an_en_eff_thc[ind_28]+3.)/4.
	an_en_eff_thd[ind_48] = (an_en_eff_thd[ind_28]+3.)/4.
	an_en_eff_the[ind_48] = (an_en_eff_the[ind_28]+3.)/4.
	an_en_eff_tha[ind_36] = (an_en_eff_tha[ind_28]+1.)/2.
	an_en_eff_thb[ind_36] = (an_en_eff_thb[ind_28]+1.)/2.
	an_en_eff_thc[ind_36] = (an_en_eff_thc[ind_28]+1.)/2.
	an_en_eff_thd[ind_36] = (an_en_eff_thd[ind_28]+1.)/2.
	an_en_eff_the[ind_36] = (an_en_eff_the[ind_28]+1.)/2.

;********************************************************************************************
;********************************************************************************************

if not keyword_set(time) then time=time_double('2007-5-24')

; geom_factor is the baseline geometric factor for the sensors at the beginning of the mission

;	geom_factor=[.00115,.00107]			; original geom_factor estimate
	geom_factor=[.00164,.00153]			; new GF after 0.7 correction from cross calibration w/ Wind-SWE


; an_eff corrects for high energy angle variations in MCP efficiency

	an_eff_arr=fltarr(8,2,6)
	an_eff_arr[*,*,*]=1.

;   Electron relative anode calibration
;	delta values are used to correct for asymmetric response not accounted for in calibrations
;	delta values add small delta% asymmetric linear adjustment to the relative anode sensitivity
;	delta values are chosen to match the Vz ion and electron velocities in the sheath 
; 	to decrease Vz for electrons, make delta more positive

;   tha	
;		an_eff_arr[*,0,0]=[0.932534,1.01403 ,1.02331 ,0.985032,1.00045 ,1.03397 ,1.04974 ,0.960941]	; delta=0
;		an_eff_arr[*,0,0]=[0.924313,1.00763 ,1.01940 ,0.983736,1.00163 ,1.03778 ,1.05623 ,0.969285]	; delta=.010
		an_eff_arr[*,0,0]=[0.920202,1.00442 ,1.01745 ,0.983087,1.00223 ,1.03968 ,1.05947 ,0.973456]	; delta=.015
;		an_eff_arr[*,0,0]=[0.916093,1.00122 ,1.01550 ,0.982439,1.00282 ,1.04159 ,1.06272 ,0.977628]	; delta=.020
	if time[0] gt time_double('2008-01-01/0:00') then begin
;		an_eff_arr[*,0,0]=[0.945427,1.02060 ,1.02167 ,0.974944,0.985589,1.02850 ,1.05271 ,0.970561]	; delta=0
;		an_eff_arr[*,0,0]=[0.937101,1.01416 ,1.01778 ,0.973670,0.986765,1.03230 ,1.05923 ,0.978997]	; delta=.010
;		an_eff_arr[*,0,0]=[0.932938,1.01094 ,1.01584 ,0.973032,0.987352,1.03420 ,1.06249 ,0.983215]	; delta=.015
;		an_eff_arr[*,0,0]=[0.928775,1.00772 ,1.01389 ,0.972395,0.987940,1.03609 ,1.06574 ,0.987432]	; delta=.020
;		an_eff_arr[*,0,0]=[0.912127,0.994854,1.00612 ,0.969847,0.990290,1.04369 ,1.07878 ,1.00430 ]	; delta=.040
;		an_eff_arr[*,0,0]=[0.899644,0.985203,1.00029 ,0.967936,0.992052,1.04938 ,1.08855 ,1.01695 ]	; delta=.055
		an_eff_arr[*,0,0]=[0.891323,0.978770,0.996398,0.966662,0.993226,1.05318 ,1.09507 ,1.02538 ]	; delta=.065
;		an_eff_arr[*,0,0]=[0.887163,0.975554,0.994455,0.966025,0.993814,1.05507 ,1.09832 ,1.02959 ]	; delta=.070
;		an_eff_arr[*,0,0]=[0.883003,0.972338,0.992511,0.965388,0.994401,1.05697 ,1.10158 ,1.03381 ]	; delta=.075
	endif
	if time[0] gt time_double('2008-11-21/0:00') then begin
;		an_eff_arr[*,0,0]=[0.939673,1.01614 ,1.01822 ,0.989315,0.991789,1.02850 ,1.05188 ,0.964473]	; delta=0
		an_eff_arr[*,0,0]=[0.927256,1.00652 ,1.01240 ,0.987371,0.993559,1.03420 ,1.06165 ,0.977043]	; delta=.015
;		an_eff_arr[*,0,0]=[0.918980,1.00011 ,1.00853 ,0.986075,0.994739,1.03799 ,1.06816 ,0.985422]	; delta=.025
;		an_eff_arr[*,0,0]=[0.906566,0.990499,1.00271 ,0.984132,0.996508,1.04368 ,1.07792 ,0.997989]	; delta=.040
;		an_eff_arr[*,0,0]=[0.885883,0.974479,0.993016,0.980893,0.999456,1.05316 ,1.09418 ,1.01893 ]	; delta=.065
	endif

;   thb
;		an_eff_arr[*,0,1]=[0.948788,1.02241 ,1.01765 ,0.991669,0.998679,1.03073 ,1.03701 ,0.953064]	; delta=0
;		an_eff_arr[*,0,1]=[0.957112,1.02882 ,1.02149 ,0.992932,0.997454,1.02688 ,1.03056 ,0.944747]	; delta=-.010
		an_eff_arr[*,0,1]=[0.961274,1.03203 ,1.02341 ,0.993564,0.996842,1.02496 ,1.02733 ,0.940588]	; delta=-.015
;		an_eff_arr[*,0,1]=[0.965437,1.03524 ,1.02533 ,0.994195,0.996229,1.02304 ,1.02410 ,0.936429]	; delta=-.020
	if time[0] gt time_double('2008-01-01/0:00') then begin
;		an_eff_arr[*,0,1]=[0.904159,1.00201 ,1.01720 ,1.01247 ,1.02168 ,1.04823 ,1.04462 ,0.949633]	; delta=0.030
;		an_eff_arr[*,0,1]=[0.912288,1.00840 ,1.02106 ,1.01375 ,1.02041 ,1.04435 ,1.03821 ,0.941540]	; delta=0.020
;		an_eff_arr[*,0,1]=[0.920416,1.01478 ,1.02492 ,1.01502 ,1.01914 ,1.04046 ,1.03181 ,0.933447]	; delta=0.010
;		an_eff_arr[*,0,1]=[0.928544,1.02117 ,1.02878 ,1.01630 ,1.01787 ,1.03658 ,1.02540 ,0.925353]	; delta=0
		an_eff_arr[*,0,1]=[0.932608,1.02436 ,1.03071 ,1.01693 ,1.01724 ,1.03464 ,1.02220 ,0.921307]	; delta=-.005
;		an_eff_arr[*,0,1]=[0.936672,1.02756 ,1.03264 ,1.01757 ,1.01660 ,1.03270 ,1.01900 ,0.917260]	; delta=-.010
;		an_eff_arr[*,0,1]=[0.944800,1.03394 ,1.03650 ,1.01885 ,1.01534 ,1.02881 ,1.01259 ,0.909166]	; delta=-.020
	endif
	if time[0] gt time_double('2008-11-21/0:00') then begin
;		an_eff_arr[*,0,1]=[0.902553,0.997227,1.01235 ,1.00581 ,1.01687 ,1.04753 ,1.05383 ,0.963831]	; delta=+.04
;		an_eff_arr[*,0,1]=[0.910738,1.00362 ,1.01620 ,1.00708 ,1.01560 ,1.04366 ,1.04741 ,0.955685]	; delta=+.03
;		an_eff_arr[*,0,1]=[0.918924,1.01002 ,1.02006 ,1.00835 ,1.01434 ,1.03979 ,1.04099 ,0.947538]	; delta=+.02
;		an_eff_arr[*,0,1]=[0.927109,1.01641 ,1.02392 ,1.00961 ,1.01308 ,1.03592 ,1.03456 ,0.939392]	; delta=+.01
;		an_eff_arr[*,0,1]=[0.935295,1.02281 ,1.02777 ,1.01088 ,1.01181 ,1.03205 ,1.02814 ,0.931245]	; delta=0
;		an_eff_arr[*,0,1]=[0.943480,1.02920 ,1.03163 ,1.01214 ,1.01055 ,1.02818 ,1.02171 ,0.923098]	; delta=-.01
		an_eff_arr[*,0,1]=[0.951666,1.03559 ,1.03549 ,1.01341 ,1.00929 ,1.02432 ,1.01529 ,0.914952]	; delta=-.02
	endif

;   thc 
;	- showed energy-angle dependent sensitivity in the 40-200 eV range, so delta not well determined
;		an_eff_arr[*,0,2]=[0.937234,1.03726 ,1.02144 ,0.994146,0.987515,1.01184 ,1.03937 ,0.971188]	; delta=0
;		an_eff_arr[*,0,2]=[0.933118,1.03400 ,1.01951 ,0.993509,0.988116,1.01372 ,1.04260 ,0.975421]	; delta=.005
;		an_eff_arr[*,0,2]=[0.929002,1.03074 ,1.01758 ,0.992871,0.988717,1.01560 ,1.04583 ,0.979653]	; delta=.01
;		an_eff_arr[*,0,2]=[0.920771,1.02423 ,1.01372 ,0.991595,0.989918,1.01936 ,1.05229 ,0.988118]	; delta=.02
;		an_eff_arr[*,0,2]=[0.912541,1.01771 ,1.00985 ,0.990320,0.991120,1.02312 ,1.05875 ,0.996582]	; delta=.03
;		an_eff_arr[*,0,2]=[0.904310,1.01120 ,1.00599 ,0.989044,0.992321,1.02688 ,1.06521 ,1.00505 ]	; delta=.04
		an_eff_arr[*,0,2]=[0.896081,1.00468 ,1.00213 ,0.987769,0.993522,1.03064 ,1.07167 ,1.01351 ]	; delta=.05
	if time[0] gt time_double('2008-10-23/0:00') then begin
;		an_eff_arr[*,0,2]=[0.935968,1.03715 ,1.02226 ,0.998755,0.993755,1.01390 ,1.03435 ,0.963861]	; delta=0
;		an_eff_arr[*,0,2]=[0.927756,1.03064 ,1.01840 ,0.997483,0.994973,1.01768 ,1.04079 ,0.972272]	; delta=.01
;		an_eff_arr[*,0,2]=[0.919545,1.02413 ,1.01455 ,0.996211,0.996192,1.02146 ,1.04723 ,0.980682]	; delta=.02
;		an_eff_arr[*,0,2]=[0.911334,1.01763 ,1.01069 ,0.994939,0.997411,1.02524 ,1.05367 ,0.989092]	; delta=.03
		an_eff_arr[*,0,2]=[0.903123,1.01112 ,1.00683 ,0.993667,0.998629,1.02901 ,1.06011 ,0.997502]	; delta=.04
;		an_eff_arr[*,0,2]=[0.894913,1.00462 ,1.00298 ,0.992395,0.999848,1.03279 ,1.06655 ,1.00591 ]	; delta=.05
;		an_eff_arr[*,0,2]=[0.886704,0.998111,0.999119,0.991123,1.00107 ,1.03657 ,1.07299 ,1.01432 ]	; delta=.06
;		an_eff_arr[*,0,2]=[0.878494,0.991607,0.995263,0.989852,1.00228 ,1.04035 ,1.07943 ,1.02273 ]	; delta=.07
	endif
	if time[0] gt time_double('2008-11-21/0:00') then begin
		an_eff_arr[*,0,2]=[0.937799,1.03402 ,1.01578 ,0.996553,0.993856,1.01254 ,1.03769 ,0.971764]	; delta=0
;		an_eff_arr[*,0,2]=[0.929557,1.02751 ,1.01194 ,0.995269,0.995061,1.01630 ,1.04413 ,0.980230]	; delta=.01
;		an_eff_arr[*,0,2]=[0.921317,1.02101 ,1.00809 ,0.993985,0.996265,1.02006 ,1.05058 ,0.988695]	; delta=.02
;		an_eff_arr[*,0,2]=[0.913077,1.01451 ,1.00424 ,0.992702,0.997469,1.02382 ,1.05702 ,0.997159]	; delta=.03
;		an_eff_arr[*,0,2]=[0.904838,1.00801 ,1.00039 ,0.991419,0.998673,1.02758 ,1.06347 ,1.00562 ]	; delta=.04
;		an_eff_arr[*,0,2]=[0.896599,1.00151 ,0.996547,0.990136,0.999877,1.03133 ,1.06991 ,1.01409 ]	; delta=.05
	endif

;   thd
;		an_eff_arr[*,0,3]=[0.923366,1.00238 ,1.02970 ,0.998831,1.00886 ,1.05680 ,1.02135 ,0.958724]	; delta=-.00
;		an_eff_arr[*,0,3]=[0.939652,1.01505 ,1.03756 ,1.00146 ,1.00647 ,1.04901 ,1.00872 ,0.942074]	; delta=-.02
;		an_eff_arr[*,0,3]=[0.947797,1.02138 ,1.04150 ,1.00278 ,1.00528 ,1.04512 ,1.00240 ,0.933748]	; delta=-.03 
		an_eff_arr[*,0,3]=[0.955943,1.02772 ,1.04543 ,1.00410 ,1.00408 ,1.04123 ,0.996084,0.925420]	; delta=-.04
;		an_eff_arr[*,0,3]=[0.964090,1.03405 ,1.04936 ,1.00541 ,1.00289 ,1.03733 ,0.989766,0.917091]	; delta=-.05
	if time[0] gt time_double('2008-01-29/19:28') then begin
;		an_eff_arr[*,0,3]=[0.931482,0.997922,1.02776 ,0.999371,1.00756 ,1.05000 ,1.02162 ,0.964283]	; delta=0
;		an_eff_arr[*,0,3]=[0.947908,1.01053 ,1.03561 ,1.00200 ,1.00517 ,1.04226 ,1.00898 ,0.947533]	; delta=-.020
;		an_eff_arr[*,0,3]=[0.952015,1.01368 ,1.03757 ,1.00266 ,1.00458 ,1.04033 ,1.00582 ,0.943346]	; delta=-.025
		an_eff_arr[*,0,3]=[0.956123,1.01683 ,1.03953 ,1.00332 ,1.00398 ,1.03839 ,1.00266 ,0.939157]	; delta=-.030 
;		an_eff_arr[*,0,3]=[0.964339,1.02314 ,1.04346 ,1.00463 ,1.00279 ,1.03452 ,0.996342,0.930779]	; delta=-.040 
	endif
	if time[0] gt time_double('2008-11-21/0:00') then begin
;		an_eff_arr[*,0,3]=[0.916773,0.997877,1.02651 ,1.00461 ,1.01064 ,1.05436 ,1.02805 ,0.961168]	; delta=0
;		an_eff_arr[*,0,3]=[0.924875,1.00420 ,1.03045 ,1.00595 ,1.00947 ,1.05050 ,1.02171 ,0.952840]	; delta=-.010
;		an_eff_arr[*,0,3]=[0.932978,1.01052 ,1.03439 ,1.00729 ,1.00829 ,1.04664 ,1.01538 ,0.944511]	; delta=-.020
;		an_eff_arr[*,0,3]=[0.941082,1.01685 ,1.03833 ,1.00864 ,1.00711 ,1.04277 ,1.00904 ,0.936180]	; delta=-.030 
;		an_eff_arr[*,0,3]=[0.957294,1.02950 ,1.04621 ,1.01132 ,1.00476 ,1.03504 ,0.996355,0.919513]	; delta=-.050 
;		an_eff_arr[*,0,3]=[0.961349,1.03267 ,1.04818 ,1.01199 ,1.00417 ,1.03311 ,0.993183,0.915346]	; delta=-.055 
;		an_eff_arr[*,0,3]=[0.965403,1.03583 ,1.05015 ,1.01267 ,1.00358 ,1.03117 ,0.990012,0.911178]	; delta=-.060
;		an_eff_arr[*,0,3]=[0.969458,1.03900 ,1.05212 ,1.01334 ,1.00299 ,1.02924 ,0.986840,0.907010]	; delta=-.065
		an_eff_arr[*,0,3]=[0.973513,1.04216 ,1.05410 ,1.01401 ,1.00240 ,1.02731 ,0.983668,0.902841]	; delta=-.070
;		an_eff_arr[*,0,3]=[0.977568,1.04533 ,1.05607 ,1.01468 ,1.00182 ,1.02537 ,0.980496,0.898672]	; delta=-.075
	endif


;   the
;		an_eff_arr[*,0,4]=[0.940985,1.04017 ,1.03806 ,1.00305 ,0.981721,1.01644 ,1.03942 ,0.940142] 	; delta=0
;		an_eff_arr[*,0,4]=[0.949204,1.04666 ,1.04194 ,1.00429 ,0.980479,1.01262 ,1.03291 ,0.931902]	; delta=-.01
;		an_eff_arr[*,0,4]=[0.957423,1.05315 ,1.04582 ,1.00552 ,0.979237,1.00879 ,1.02640 ,0.923662] 	; delta=-.02
;		an_eff_arr[*,0,4]=[0.953314,1.04990 ,1.04388 ,1.00491 ,0.979858,1.01070 ,1.02966 ,0.927782]	; delta=-.015
		an_eff_arr[*,0,4]=[0.957423,1.05315 ,1.04582 ,1.00552 ,0.979237,1.00879 ,1.02640 ,0.923662]	; delta=-.020
;		an_eff_arr[*,0,4]=[0.961533,1.05639 ,1.04776 ,1.00614 ,0.978616,1.00688 ,1.02314 ,0.919542]	; delta=-.025
;		an_eff_arr[*,0,4]=[0.965642,1.05963 ,1.04969 ,1.00676 ,0.977995,1.00496 ,1.01989 ,0.915422]	; delta=-.030
;		an_eff_arr[*,0,4]=[0.982079,1.07260 ,1.05745 ,1.00924 ,0.975512,0.997312,1.00687 ,0.898943]	; delta=-.050
;	if time(0) gt time_double('2007-08-30/0:00') then begin
	if time[0] gt time_double('2007-11-21/0:00') then begin
;		an_eff_arr[*,0,4]=[0.929803,1.03085 ,1.03066 ,1.00123 ,0.985556,1.01810 ,1.04779 ,0.956012]	; delta=0.01
;		an_eff_arr[*,0,4]=[0.933908,1.03409 ,1.03260 ,1.00186 ,0.984941,1.01620 ,1.04454 ,0.951867]	; delta=0.005
;		an_eff_arr[*,0,4]=[0.938012,1.03733 ,1.03455 ,1.00248 ,0.984326,1.01429 ,1.04129 ,0.947721]	; delta=0
;		an_eff_arr[*,0,4]=[0.942116,1.04057 ,1.03649 ,1.00311 ,0.983712,1.01239 ,1.03803 ,0.943576]	; delta=-.005
;		an_eff_arr[*,0,4]=[0.946221,1.04382 ,1.03843 ,1.00374 ,0.983097,1.01049 ,1.03478 ,0.939430]	; delta=-.010
;		an_eff_arr[*,0,4]=[0.950325,1.04706 ,1.04037 ,1.00436 ,0.982483,1.00859 ,1.03153 ,0.935284]	; delta=-.015 
		an_eff_arr[*,0,4]=[0.954430,1.05030 ,1.04231 ,1.00499 ,0.981868,1.00669 ,1.02827 ,0.931139]	; delta=-.020
;		an_eff_arr[*,0,4]=[0.958534,1.05354 ,1.04425 ,1.00562 ,0.981254,1.00479 ,1.02502 ,0.926993]	; delta=-.025
	endif
	if time[0] gt time_double('2008-11-21/0:00') then begin
;		an_eff_arr[*,0,4]=[0.920750,1.02744 ,1.02261 ,0.999928,0.991053,1.02270 ,1.05228 ,0.963232]	; delta=0.015
;		an_eff_arr[*,0,4]=[0.924839,1.03069 ,1.02455 ,1.00056 ,0.990443,1.02080 ,1.04903 ,0.959080]	; delta=0.010
;		an_eff_arr[*,0,4]=[0.928929,1.03394 ,1.02649 ,1.00120 ,0.989833,1.01891 ,1.04578 ,0.954928]	; delta=0.005
;		an_eff_arr[*,0,4]=[0.933018,1.03719 ,1.02842 ,1.00183 ,0.989223,1.01701 ,1.04253 ,0.950777]	; delta=0
;		an_eff_arr[*,0,4]=[0.937108,1.04044 ,1.03036 ,1.00247 ,0.988612,1.01511 ,1.03928 ,0.946625]	; delta=-.005
;		an_eff_arr[*,0,4]=[0.941197,1.04369 ,1.03230 ,1.00310 ,0.988002,1.01321 ,1.03603 ,0.942473]	; delta=-.010
;		an_eff_arr[*,0,4]=[0.945287,1.04694 ,1.03423 ,1.00373 ,0.987392,1.01131 ,1.03279 ,0.938320]	; delta=-.015
;		an_eff_arr[*,0,4]=[0.949377,1.05019 ,1.03617 ,1.00437 ,0.986782,1.00941 ,1.02954 ,0.934168]	; delta=-.020
		an_eff_arr[*,0,4]=[0.953467,1.05344 ,1.03811 ,1.00500 ,0.986172,1.00751 ,1.02629 ,0.930016]	; delta=-.025
;		an_eff_arr[*,0,4]=[0.957557,1.05669 ,1.04004 ,1.00564 ,0.985561,1.00561 ,1.02304 ,0.925864]	; delta=-.030
	endif

;   Ion relative anode calibration

;   tha
		an_eff_arr[*,1,0]=[0.885003,0.983638,1.02201 ,1.04247 ,1.05520 ,1.04390 ,1.04236 ,0.925424]
	if time[0] gt time_double('2008-01-01/00:00') then begin
		an_eff_arr[*,1,0]=[0.889871,0.996283,1.01743 ,1.03879 ,1.04740 ,1.03966 ,1.04302 ,0.927555]
	endif
	if time[0] gt time_double('2008-11-21/18:04:46') then begin
		an_eff_arr[*,1,0]=[0.887466,1.00205 ,1.02653 ,1.04869 ,1.05523 ,1.03547 ,1.03171 ,0.912871]
	endif

;   thb
		an_eff_arr[*,1,1]=[0.883371,0.978252,1.01493 ,1.04915 ,1.04927 ,1.05070 ,1.03988 ,0.934454]
	if time[0] gt time_double('2008-01-01') then begin
		an_eff_arr[*,1,1]=[0.857464,0.969802,1.00349 ,1.04081 ,1.04351 ,1.06247 ,1.06633 ,0.956126]
	endif
	if time[0] gt time_double('2008-11-21/18:04:46') then begin
		an_eff_arr[*,1,1]=[0.865988,0.976858,1.01516 ,1.02449 ,1.04483 ,1.06497 ,1.06176 ,0.945951]
	endif

;   thc
		an_eff_arr[*,1,2]=[0.896830,0.977188,1.01154 ,1.07575 ,1.09337 ,1.03015 ,0.996858,0.918310]
	if time[0] gt time_double('2008-01-01') then begin
		an_eff_arr[*,1,2]=[0.880834,0.974426,1.00970 ,1.07004 ,1.08506 ,1.04305 ,1.01568 ,0.921209]
	endif
	if time[0] gt time_double('2008-11-21') then begin
		an_eff_arr[*,1,2]=[0.883216,0.969479,1.01473 ,1.06325 ,1.08870 ,1.04015 ,1.01708 ,0.923404]
	endif

;   thd
		an_eff_arr[*,1,3]=[0.881499,0.960724,0.982417,1.02190 ,1.06730 ,1.05802 ,1.06450 ,0.963637]
	if time[0] gt time_double('2008-02-16/18:24') then begin
		an_eff_arr[*,1,3]=[0.879061,0.961538,0.985881,1.02784 ,1.06654 ,1.05243 ,1.05915 ,0.967562]
	endif
	if time[0] gt time_double('2008-11-21/19:11:46') then begin
		an_eff_arr[*,1,3]=[0.865675,0.962226,0.994435,1.03504 ,1.07931 ,1.05717 ,1.05824 ,0.947896]
	endif
 

;   the
		an_eff_arr[*,1,4]=[0.904047,0.996663,1.00987 ,1.06548 ,1.07002 ,1.03860 ,1.00385 ,0.911470]
	if time[0] gt time_double('2007-12-12/22:32') then begin
		an_eff_arr[*,1,4]=[0.896659,0.993430,1.01416 ,1.07099 ,1.07492 ,1.04506 ,1.00283 ,0.901947]
	endif
	if time[0] gt time_double('2008-11-21/19:45:55') then begin
		an_eff_arr[*,1,4]=[0.895642,0.998058,1.01726 ,1.07183 ,1.07612 ,1.04555 ,1.00305 ,0.892481]
	endif

	an_eff=reform(an_eff_arr[*,ion,scn])




; rel_gf corrects for initial variations in geom_factor and the slow time dependent degradation in instrument sensitivity

	gf_time=esa_cal_time
	gf_val = reform(esa_cal_gf[scn,ion,*])
	rel_gf=interp(/no_extrapolate,gf_val,gf_time,time)



; an_en_eff corrects for the angle-energy dependent geometric factor changes in electron ESAs at <40 eV
; 	an_en_eff[nenergy,anode,energy_mode,scn]
	
	an_en_eff = fltarr(64) & an_en_eff[*]=1.

	if not ion then begin
		if sc eq 'a' then begin
			ttarr=[time_double('2007-12-01'),time_double('2008-05-16'),time_double('2008-10-10'),time_double('2009-07-13')]
			if time[0] lt ttarr[0] then begin
				an_en_eff=reform(an_en_eff_tha[*,0])
			endif else if time[0] ge ttarr[0] and time[0] lt ttarr[1] then begin
				delta=ttarr[1]-ttarr[0]
				delta1=(ttarr[1]-time[0])/delta & delta2=(time[0]-ttarr[0])/delta
				an_en_eff=delta1*reform(an_en_eff_tha[*,0]) + delta2*reform(an_en_eff_tha[*,1])
			endif else if time[0] ge ttarr[1] and time[0] lt ttarr[2] then begin
				delta=ttarr[2]-ttarr[1]
				delta1=(ttarr[2]-time[0])/delta & delta2=(time[0]-ttarr[1])/delta
				an_en_eff=delta1*reform(an_en_eff_tha[*,1]) + delta2*reform(an_en_eff_tha[*,2])
			endif else if time[0] ge ttarr[2] and time[0] lt ttarr[3] then begin
				delta=ttarr[3]-ttarr[2]
				delta1=(ttarr[3]-time[0])/delta & delta2=(time[0]-ttarr[2])/delta
				an_en_eff=delta1*reform(an_en_eff_tha[*,2]) + delta2*reform(an_en_eff_tha[*,3])
			endif else an_en_eff=reform(an_en_eff_tha[*,3])
		endif	
		if sc eq 'b' then begin
			ttarr=[time_double('2007-12-01'),time_double('2008-05-16'),time_double('2008-10-10'),time_double('2009-07-13')]
			if time[0] lt ttarr[0] then begin
				an_en_eff=reform(an_en_eff_thb[*,0])
			endif else if time[0] ge ttarr[0] and time[0] lt ttarr[1] then begin
				delta=ttarr[1]-ttarr[0]
				delta1=(ttarr[1]-time[0])/delta & delta2=(time[0]-ttarr[0])/delta
				an_en_eff=delta1*reform(an_en_eff_thb[*,0]) + delta2*reform(an_en_eff_thb[*,1])
			endif else if time[0] ge ttarr[1] and time[0] lt ttarr[2] then begin
				delta=ttarr[2]-ttarr[1]
				delta1=(ttarr[2]-time[0])/delta & delta2=(time[0]-ttarr[1])/delta
				an_en_eff=delta1*reform(an_en_eff_thb[*,1]) + delta2*reform(an_en_eff_thb[*,2])
			endif else if time[0] ge ttarr[2] and time[0] lt ttarr[3] then begin
				delta=ttarr[3]-ttarr[2]
				delta1=(ttarr[3]-time[0])/delta & delta2=(time[0]-ttarr[2])/delta
				an_en_eff=delta1*reform(an_en_eff_thb[*,2]) + delta2*reform(an_en_eff_thb[*,3])
			endif else an_en_eff=reform(an_en_eff_thb[*,3])
		endif	
		if sc eq 'c' then begin
			ttarr=[time_double('2007-08-24'),time_double('2008-05-16'),time_double('2008-10-14'),time_double('2009-07-27')]
			if time[0] lt ttarr[0] then begin
				an_en_eff=reform(an_en_eff_thc[*,0])
			endif else if time[0] ge ttarr[0] and time[0] lt ttarr[1] then begin
				delta=ttarr[1]-ttarr[0]
				delta1=(ttarr[1]-time[0])/delta & delta2=(time[0]-ttarr[0])/delta
				an_en_eff=delta1*reform(an_en_eff_thc[*,0]) + delta2*reform(an_en_eff_thc[*,1])
			endif else if time[0] ge ttarr[1] and time[0] lt ttarr[2] then begin
				delta=ttarr[2]-ttarr[1]
				delta1=(ttarr[2]-time[0])/delta & delta2=(time[0]-ttarr[1])/delta
				an_en_eff=delta1*reform(an_en_eff_thc[*,1]) + delta2*reform(an_en_eff_thc[*,2])
			endif else if time[0] ge ttarr[2] and time[0] lt ttarr[3] then begin
				delta=ttarr[3]-ttarr[2]
				delta1=(ttarr[3]-time[0])/delta & delta2=(time[0]-ttarr[2])/delta
				an_en_eff=delta1*reform(an_en_eff_thc[*,2]) + delta2*reform(an_en_eff_thc[*,3])
			endif else an_en_eff=reform(an_en_eff_thc[*,3])
		endif	
		if sc eq 'd' then begin
			ttarr=[time_double('2007-10-18'),time_double('2008-06-14'),time_double('2008-08-21'),time_double('2009-08-06')]
			if time[0] lt ttarr[0] then begin
				an_en_eff=reform(an_en_eff_thd[*,0])
			endif else if time[0] ge ttarr[0] and time[0] lt ttarr[1] then begin
				delta=ttarr[1]-ttarr[0]
				delta1=(ttarr[1]-time[0])/delta & delta2=(time[0]-ttarr[0])/delta
				an_en_eff=delta1*reform(an_en_eff_thd[*,0]) + delta2*reform(an_en_eff_thd[*,1])
			endif else if time[0] ge ttarr[1] and time[0] lt ttarr[2] then begin
				delta=ttarr[2]-ttarr[1]
				delta1=(ttarr[2]-time[0])/delta & delta2=(time[0]-ttarr[1])/delta
				an_en_eff=delta1*reform(an_en_eff_thd[*,1]) + delta2*reform(an_en_eff_thd[*,2])
			endif else if time[0] ge ttarr[2] and time[0] lt ttarr[3] then begin
				delta=ttarr[3]-ttarr[2]
				delta1=(ttarr[3]-time[0])/delta & delta2=(time[0]-ttarr[2])/delta
				an_en_eff=delta1*reform(an_en_eff_thd[*,2]) + delta2*reform(an_en_eff_thd[*,3])
			endif else an_en_eff=reform(an_en_eff_thd[*,3])
		endif	
		if sc eq 'e' then begin
			ttarr=[time_double('2007-12-01'),time_double('2008-09-16'),time_double('2008-09-16'),time_double('2009-10-11')]
			if time[0] lt ttarr[0] then begin
				an_en_eff=reform(an_en_eff_the[*,0])
			endif else if time[0] ge ttarr[0] and time[0] lt ttarr[1] then begin
				delta=ttarr[1]-ttarr[0]
				delta1=(ttarr[1]-time[0])/delta & delta2=(time[0]-ttarr[0])/delta
				an_en_eff=delta1*reform(an_en_eff_the[*,0]) + delta2*reform(an_en_eff_the[*,1])
			endif else if time[0] ge ttarr[1] and time[0] lt ttarr[2] then begin
				delta=ttarr[2]-ttarr[1]
				delta1=(ttarr[2]-time[0])/delta & delta2=(time[0]-ttarr[1])/delta
				an_en_eff=delta1*reform(an_en_eff_the[*,1]) + delta2*reform(an_en_eff_the[*,2])
			endif else if time[0] ge ttarr[2] and time[0] lt ttarr[3] then begin
				delta=ttarr[3]-ttarr[2]
				delta1=(ttarr[3]-time[0])/delta & delta2=(time[0]-ttarr[2])/delta
				an_en_eff=delta1*reform(an_en_eff_the[*,2]) + delta2*reform(an_en_eff_the[*,3])
			endif else an_en_eff=reform(an_en_eff_the[*,3])
		endif	
	endif



calib={geom_factor:geom_factor[ion],an_eff:an_eff,rel_gf:rel_gf,an_en_eff:an_en_eff}
return,calib

end
