/*------------------------------------------------------------------------------
 *
 *	Fichier	: $RCSfile: libascii.c,v $, v $Revision: 1.32 $
 *
 *	Date	: $Date: 2021/04/27 09:48:59 $
 *
 *	Auteur	: $Author: penou $
 *
 *	Version : %Z% version %I% de %M% du %G%
 *
 *------------------------------------------------------------------------------
*/

// -----------
// Compilation
// -----------
//
// This module uses some functions from zlib (gzopen,gzgets,gzread,gzclose,gzeof).
//
// But, if this file is linked with standard zlib, execution fails inside IDL. 
// Why ? I think because there is a confusion between standard zlib and IDL zlib
// Solution ? links with a modified version of zlib where some function names has just been renamed.
//
// So, compiles libascii.c with adler32.c compress.c crc32.c deflate.c gzio.c infback.c inffast.c inflate.c inftrees.c trees.c uncompr.c zutil.c
// (needs crc32.h deflate.h inffast.h inffixed.h inflate.h inftrees.h trees.h zconf.h zlib.h zutil.h)

//
// Linux gcc:
// ----------
// gcc -c -Wall -Wno-unused  -O3 -finline -fomit-frame-pointer -D_FILE_OFFSET_BITS=64 -D_LARGEFILE_SOURCE -I$IDL_DIR/external -I$IDL_DIR/external/include -fPIC libascii.c adler32.c compress.c crc32.c deflate.c gzio.c infback.c inffast.c inflate.c inftrees.c trees.c uncompr.c zutil.c
// gcc -shared libascii.o adler32.o compress.o crc32.o deflate.o gzio.o infback.o inffast.o inflate.o inftrees.o trees.o uncompr.o zutil.o -o libascii.so
//
//
// Solaris cc:
// -----------
// cc -c  -Xa -xO3 -D_FILE_OFFSET_BITS=64 -D_LARGEFILE_SOURCE -I$IDL_DIR/external -I$IDL_DIR/external/include -K PIC -G libascii.c adler32.c compress.c crc32.c deflate.c gzio.c infback.c inffast.c inflate.c inftrees.c trees.c uncompr.c zutil.c
// cc -G  libascii.o adler32.o compress.o crc32.o deflate.o gzio.o infback.o inffast.o inflate.o inftrees.o trees.o uncompr.o zutil.o -o libascii.so
//
//
// Sparc Solaris gcc:
// ------------------
// gcc -c -Wall -Wno-unused -O3 -mcpu=v8 -finline -fomit-frame-pointer -D_FILE_OFFSET_BITS=64 -D_LARGEFILE_SOURCE -I$IDL_DIR/external -I$IDL_DIR/external/include -fPIC libascii.c adler32.c compress.c crc32.c deflate.c gzio.c infback.c inffast.c inflate.c inftrees.c trees.c uncompr.c zutil.c
// gcc -G libascii.o adler32.o compress.o crc32.o deflate.o gzio.o infback.o inffast.o inflate.o inftrees.o trees.o uncompr.o zutil.o -o libascii.so
//
//
// Windows bcc32 (free Borland C++ 5.5.1 for win32)
// ------------------------------------------------
// implib -a idlborland %IDL_DIR%\BIN\BIN.X86\idl.dll
// bcc32 -w-par -w-aus -6 -O2 -OS -tWD -ps -I%IDL_DIR%\external -I%IDL_DIR%\external\include libascii.c adler32.c compress.c crc32.c deflate.c gzio.c infback.c inffast.c inflate.c inftrees.c trees.c uncompr.c zutil.c idlborland.lib





// ----------------------------
// Performances of read_ascii()
// ----------------------------
//
//	Results for "C3_CP_FGM_FULL__20020821_000000_20020822_000000_V060622.cef" (184_672_936 bytes) on Dell PowerEdge 2950 Xeon 5160 3.0 Ghz/4MB 1333FSB
//	gcc_3.4.6 and IDL_6.3
//
// 						MACRO	MACRO		PENOU	PENOU		STRTOD	STRTOD
// 						RAM1	RAM0		RAM1	RAM0		RAM1	RAM0
//
// => duration (in sec)				0.95	1.04		1.32	1.36		5.09	5.12
//
//
//
// stereo	gcc_3.4.6	IDL_6.3		0.95	Linux	Intel(R) Xeon(R) CPU            5160  @ 3.00GHz
// caa		gcc_4.1.2	IDL_5.4		1.66	Linux	Intel(R) Xeon(R) CPU            5130  @ 2.00GHz
// pc-92	bcc32		IDL_6.3 	2.67	Windows	Core2Duo T7250@2.0 Ghz
// aspera	gcc_3.2.3	IDL_6.3		2.76	Linux	Intel(R) Xeon(TM) CPU 3.06GHz
// cisnew 	cc_5.4    	IDL_5.4		6.86	Solaris	Sun-Fire-V250
// cisnew	cc_5.4  	IDL_6.3		7.16	Solaris	Sun-Fire-V250
// cisnew 	gcc_3.4.6 	IDL_5.4		8.37	Solaris	Sun-Fire-V250
// cisnew 	gcc_3.4.6 	IDL_6.3		8.65	Solaris	Sun-Fire-V250
// interball2	gcc_2.95.2 	IDL_5.6		13.44	Solaris	Sun-Blade-1000
// interball2	cc_5.0  	IDL_6.3		cdflib	ne se compile pas (le compilateur ne comprend pas les commentaires //)


// Define DEBUG for debugging
//#define DEBUG

// Define USE_MACRO or USE_PENOU or USE_STRTOD for "SSCANF_REEL" definition
// Define USE_MACRO or not define USE_MACRO for "SSCANF_ENTIER" definition
#define USE_MACRO

// Definir USE_RAM0 ou USE_RAM1
#define USE_RAM1

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#ifdef GDL
#include "libgdl.h"
#else
#include "export.h"
#endif

#ifdef USE_STRTOD
#include <errno.h>
#endif

#include <sys/time.h>

#include <unistd.h>

#include "zlib.h"

#define ISGZ(nom)			(strlen(nom)>=4 && strcasecmp(nom+strlen(nom)-3,".gz")==0)
#define OPEN(isgz,nom)			(isgz ? gzopenXXX (nom,"rb") : fopen (nom,"rb"))
#define READ(isgz,fd,ligne,taille)	(isgz ? gzreadXXX (fd, ligne, taille) : fread (ligne, 1, taille, fd))
#define GETS(isgz,fd,ligne,taille)	(isgz ? gzgetsXXX (fd, ligne, taille) : fgets (ligne, taille, fd))
#define CLOSE(isgz,fd)			(isgz ? gzcloseXXX (fd) : fclose (fd))
#define FEOF(isgz,fd)			(isgz ? gzeofXXX (fd) : feof (fd))

#ifdef USE_RAM1
	char TABDIGIT[256]={ 		// '0'=48 '9'=57
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,1,1,
		1,1,1,1,1,1,1,1,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0};
	#define ISDIGIT(x) TABDIGIT[(int)(x)]
#endif

#ifdef USE_RAM0
	#define ISDIGIT(x) (((x)>='0')&&((x)<='9'))
#endif

#define MAX10 50
double MULTI[MAX10] = {
	1e0,	1e1,	1e2,	1e3,	1e4,	1e5,	1e6,	1e7,	1e8,	1e9,
	1e10,	1e11,	1e12,	1e13,	1e14,	1e15,	1e16,	1e17,	1e18,	1e19,
	1e20,	1e21,	1e22,	1e23,	1e24,	1e25,	1e26,	1e27,	1e28,	1e29,
	1e30,	1e31,	1e32,	1e33,	1e34,	1e35,	1e36,	1e37,	1e38,	1e39,
	1e40,	1e41,	1e42,	1e43,	1e44,	1e45,	1e46,	1e47,	1e48,	1e49};
double DIVIS[MAX10] = {
	1e0,	1e-1,	1e-2,	1e-3,	1e-4,	1e-5,	1e-6,	1e-7,	1e-8,	1e-9,
	1e-10,	1e-11,	1e-12,	1e-13,	1e-14,	1e-15,	1e-16,	1e-17,	1e-18,	1e-19,
	1e-20,	1e-21,	1e-22,	1e-23,	1e-24,	1e-25,	1e-26,	1e-27,	1e-28,	1e-29,
	1e-30,	1e-31,	1e-32,	1e-33,	1e-34,	1e-35,	1e-36,	1e-37,	1e-38,	1e-39,
	1e-40,	1e-41,	1e-42,	1e-43,	1e-44,	1e-45,	1e-46,	1e-47,	1e-48,	1e-49};

#define ISPOINT(x) ((x)=='.')
#define ISE(x) (((x)=='e'||(x)=='E'))
#define ISPLUS(x) ((x)=='+')
#define ISMINUS(x) ((x)=='-')

int	isxml;

char ISTAG[256]={ 		
	0,0,0,0,0,0,0,0,0,1,	// '\t'=9
	1,0,0,1,0,0,0,0,0,0,	// 10 + 13
	0,0,0,0,0,0,0,0,0,0,
	0,0,1,0,0,0,0,0,0,0,	// ' '=32
	0,0,0,0,0,0,0,1,0,0,	// '/'=47
	0,0,0,0,0,0,0,0,0,0,
	1,0,1,0,0,0,0,0,1,0,	// '<'=60 '>'=62 'D'=68
	0,0,0,0,0,0,0,0,0,0,
	0,0,1,0,1,0,0,0,0,0,	// 'T'=84 'R'=82
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0};


#define IS_KEYWORD(X,KEYWORD) (strcmp(X, KEYWORD) == 0)

/*---------------------------------------------------------------------------*/
int CORRIGER_NOMCHAMP (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	// Entres
	IDL_STRING	*nom		= (IDL_STRING *)	argv[0]; /* nom */

	// Sorties
	IDL_STRING	*nomcorrect	= (IDL_STRING *)	argv[1]; /* nomcorrect */

	// Retourne dans nomcorrect une correction de nom.
	// Les caractres autres que a..z A..Z 0..9 sont transforms en '_'.
	// Si nomcorrect commence par 0..9 ou si nomcorrect est un mot cl IDL alors on rajoute un '_' devant.

	int	i;
	char	c;
	int	ok;
	char	tmp[1024];
	char	correct[1024];
	int	j=0;
	int	len = nom->slen;

	if (len==0)
		return 1;

	for (i=0 ; i<len ; i++)
		tmp[i] = toupper(nom->s[i]);
	tmp[i]=0;

	     if (IS_KEYWORD (tmp, "AND"			)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "BEGIN"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "BREAK"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "CASE"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "COMMON"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "COMPILE_OPT"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "CONTINUE"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "DO"			)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "ELSE"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "END"			)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "ENDCASE"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "ENDELSE"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "ENDFOR"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "ENDIF"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "ENDREP"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "ENDSWITCH"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "ENDWHILE"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "EQ"			)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "FOR"			))	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "FORWARD_FUNCTION"	))	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "FUNCTION"		))	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "GE"			))	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "GOTO"		))	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "GT"			))	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "IF"			)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "INHERITS"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "LE"			))	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "LT"			)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "MOD"			)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "NE"			)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "NOT"			)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "OF"			)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "ON_IOERROR"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "OR"			)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "PRO"			)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "REPEAT"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "SWITCH"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "THEN"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "UNTIL"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "WHILE"		)) 	correct[j++] = '_';
	else if (IS_KEYWORD (tmp, "XOR"			)) 	correct[j++] = '_';

	for (i=0 ; i<len ; i++) {
		c = nom->s[i];
		if (i==0 && c>='0' && c<='9') {
			correct[j++] = '_';
		}
		ok = (c>='a' && c<='z') || (c>='A' && c<='Z') || (c=='_') || (c>='0' && c<='9');
		correct[j++] = ok ? c : '_';
	}

	correct[j] = 0;

	IDL_StrStore (nomcorrect, correct);

	return 1;

}

/*---------------------------------------------------------------------------*/
int CORRIGER_NOMCHAMP_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p_nom,
	void	*p_nomcorrect) {

	int		argc=2;
	void		*argv[2];
	int		code;
	IDL_STRING	*nom;
	IDL_STRING	*nomcorrect;

	// p_nom -> nom
	nom = malloc (sizeof(IDL_STRING));
	IDL_StrStore (nom, p_nom);

	// p_nomcorrect -> nomcorrect
	nomcorrect = malloc (sizeof(IDL_STRING));
	IDL_StrStore (nomcorrect, p_nomcorrect);

	argv[0] = nom;
	argv[1] = nomcorrect;
	code = CORRIGER_NOMCHAMP (argc, argv);

	// nomcorrect -> p_nomcorrect
	sprintf (p_nomcorrect,"%s",IDL_STRING_STR(nomcorrect));

	free (nom);
	free (nomcorrect);

	return code;

}


/*---------------------------------------------------------------------------*/
int S_TO_MOT1 (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	// Entrees
	IDL_STRING	*s	= (IDL_STRING *)	argv[0]; /* s */

	// Sorties
	int		*i1	= (int *)		argv[1];
	int		*l1	= (int *)		argv[2];

	// |blanc|mot1|blanc|
	//        i1(l1)  
	//
	// Tout arrter  partir de '!'
	//
	// mot1: s[i1:i1+l1-1]
	//
	// Retourne:
	// ---------
	//	0 si OK
	//	-1 si pas de mot 1

	int	i=0;

	// mot1
	while ((i<s->slen) && (s->s[i] == ' ' || s->s[i] == '\t')) i++;							// blanc
	*i1 = i;
	while ((i<s->slen) && (s->s[i] != '!')) i++;									// mot1
	i--;
	while ((i>0) && (i<s->slen) && (s->s[i] == ' ' || s->s[i] == '\t')) i--;					// blanc
	i++;
	*l1 = i-*i1;
	if (*l1<=0) return -1;

	return 0;

}

/*---------------------------------------------------------------------------*/
int S_TO_MOT1_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p_s,
	void	*p02,
	void	*p03) {

	int		argc=3;
	void		*argv[3];
	int		code;
	IDL_STRING	*s;

	// p_s -> s
	s = malloc (sizeof(IDL_STRING));
	IDL_StrStore (s, p_s);

	argv[0] = s;
	argv[1] = p02;
	argv[2] = p03;
	code = S_TO_MOT1 (argc, argv);

	free (s);

	return code;

}


/*---------------------------------------------------------------------------*/
int S_TO_MOT1_MOT2 (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	// Entrees
	IDL_STRING	*s	= (IDL_STRING *)	argv[0]; /* s */

	// Sorties
	int		*i1	= (int *)		argv[1];
	int		*l1	= (int *)		argv[2];
	int		*i2	= (int *)		argv[3];
	int		*l2	= (int *)		argv[4];

	// |blanc|mot1|blanc|mot2|blanc|
	//        i1(l1)     i2+l2
	//
	// Tout arrter  partir de '!'
	//
	// mot1: s[i1:i1+l1-1]
	// mot2: s[i2:i2+l2-1]
	//
	// Retourne:
	// ---------
	//	0 si OK
	//	-1 si pas de mot 1
	//	-2 si pas de mot 2

	int	i=0;

	// mot1
	while ((i<s->slen) && (s->s[i] == ' ' || s->s[i] == '\t')) i++;							// blanc
	*i1 = i;
	while ((i<s->slen) && (s->s[i] != '=') && (s->s[i] != ' ') && (s->s[i] != '\t') && (s->s[i] != '!')) i++;	// mot1
	*l1 = i-*i1;
	if (*l1<=0) return -1;

	// mot2
	while ((i<s->slen) && (s->s[i] == ' ' || s->s[i] == '\t')) i++;							// blanc
	*i2 = i;
	while ((i<s->slen) && (s->s[i] != '!')) i++;									// mot2
	i--;
	while ((i>=0) && (i<s->slen) && (s->s[i] == ' ' || s->s[i] == '\t')) i--;					// blanc
	i++;
	*l2 = i-*i2;
	if (*l2<=0) return -2;

	return 0;

}


/*---------------------------------------------------------------------------*/
int S_TO_MOT1_MOT2_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p_s,
	void	*p02,
	void	*p03,
	void	*p04,
	void	*p05) {

	int		argc=5;
	void		*argv[5];
	int		code;
	IDL_STRING	*s;

	// p_s -> s
	s = malloc (sizeof(IDL_STRING));
	IDL_StrStore (s, p_s);

	argv[0] = s;
	argv[1] = p02;
	argv[2] = p03;
	argv[3] = p04;
	argv[4] = p05;
	code = S_TO_MOT1_MOT2 (argc, argv);

	free (s);

	return code;

}


/*---------------------------------------------------------------------------*/
int S_TO_MOT1_EGAL_MOT2 (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	// Entrees
	IDL_STRING	*s	= (IDL_STRING *)	argv[0]; /* s */

	// Sorties
	int		*i1	= (int *)		argv[1];
	int		*l1	= (int *)		argv[2];
	int		*i2	= (int *)		argv[3];
	int		*l2	= (int *)		argv[4];

	// |blanc|mot1|blanc|=|blanc|mot2|blanc|
	//        i1(l1)             i2+l2
	//
	// Tout arrter  partir de '!'
	//
	// mot2 peut tre entour par ", cad "mot2". Dans ce cas, renvoyer mot2 sans les "
	//
	// mot1: s[i1:i1+l1-1]
	// mot2: s[i2:i2+l2-1]
	//
	// Retourne:
	// ---------
	//	0 si OK
	//	-1 si pas de mot 1
	//	-2 si pas de '='
	//	-3 si pas de mot 2

	int	i=0;

	// mot1
	while ((i<s->slen) && (s->s[i] == ' ' || s->s[i] == '\t')) i++;							// blanc
	*i1 = i;
	while ((i<s->slen) && (s->s[i] != '=') && (s->s[i] != ' ') && (s->s[i] != '\t') && (s->s[i] != '!')) i++;	// mot1
	*l1 = i-*i1;
	if (*l1<=0) return -1;

	// '='
	while ((i<s->slen) && (s->s[i] == ' ' || s->s[i] == '\t')) i++;							// blanc
	if (i==s->slen || s->s[i]!='=') return -2;									// '='
	i++;

	// mot2
	while ((i<s->slen) && (s->s[i]!='"') && (s->s[i] == ' ' || s->s[i] == '\t')) i++;				// blanc
	if (s->s[i]=='"') {
		i++;
		*i2 = i;
		
		// Septembre 2012
		while ((i<s->slen) && (s->s[i] != '!')) i++;								
		i--;
		while ((i>=0) && (i<s->slen) && (s->s[i] != '"')) i--;							

		//while ((i<s->slen) && (s->s[i] != '"')) i++;								// mot2
	} else {
		*i2 = i;
		while ((i<s->slen) && (s->s[i] != '!')) i++;								// mot2
		i--;
		while ((i>=0) && (i<s->slen) && (s->s[i] == ' ' || s->s[i] == '\t')) i--;				// blanc
		i++;
	}
	*l2 = i-*i2;
	if (*l2<=0) return -3;

	return 0;

}

/*---------------------------------------------------------------------------*/
int S_TO_MOT1_EGAL_MOT2_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p_s,
	void	*p02,
	void	*p03,
	void	*p04,
	void	*p05) {

	int		argc=5;
	void		*argv[5];
	int		code;
	IDL_STRING	*s;

	// p_s -> s
	s = malloc (sizeof(IDL_STRING));
	IDL_StrStore (s, p_s);

	argv[0] = s;
	argv[1] = p02;
	argv[2] = p03;
	argv[3] = p04;
	argv[4] = p05;
	code = S_TO_MOT1_EGAL_MOT2 (argc, argv);

	free (s);

	return code;

}


/*---------------------------------------------------------------------------*/
int FICHIER_EXISTE (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	int		nbarg		=	0;
	IDL_STRING	*nom		=	(IDL_STRING *)	argv[nbarg++];
	

	return access (IDL_STRING_STR(nom), F_OK) != -1;

}


/*---------------------------------------------------------------------------*/
int FICHIER_EXISTE_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p_nom) {

	int		argc=1;
	void		*argv[1];
	int		code;
	IDL_STRING	*nom;

	// p_nom -> nom

	nom = malloc (sizeof(IDL_STRING));
	IDL_StrStore (nom, p_nom);

	argv[0] = nom;
	code = FICHIER_EXISTE (argc, argv);

	free (nom);

	return code;

}


/*---------------------------------------------------------------------------*/
double NBSEC (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	struct timeval tv;
	struct timezone tz;
	gettimeofday (&tv,&tz);
	return tv.tv_sec+tv.tv_usec/1000000.;

}

/*---------------------------------------------------------------------------*/
double NBSEC_AUTO_GLUE () {
/*---------------------------------------------------------------------------*/
	int	argc=0;
	void	*argv[1];

	return NBSEC (argc, argv);

}


#ifndef TEST
/*---------------------------------------------------------------------------*/
char *MY_GETS (int isgz, FILE *fd, int init, int *len) {
/*---------------------------------------------------------------------------*/

	static	char	buffer[10*1024*1024];
	static	int	nb=0;
	static	int	pos=0;
	int		i;
	int		nbnew;
	char		*ligne;

	if (init) {
		nb=pos=0;
		return NULL;
	}

	for ( ; ; ) {

		if (pos==nb+1) {	// Nothing to read
			return NULL;
		}

		ligne = buffer+pos;
		for (i=pos ; i<nb; i++) {
			if (buffer[i]==10) {
				buffer[i]=0;
				*len = i-pos;
				pos = i+1;
				return ligne;
			}
		}

		/* Cas des fichiers ne se terminant pas par un retour chariot. cela arrive avec des fichiers ascii crs avec notepad sous Windows.
			Cela n'arrive pas avec les fichiers crs avec vi. */
		if (FEOF(isgz,fd)==1) {
			buffer[nb]=0;
			*len = nb-pos;
			pos=nb+1;
			return ligne;
		}

		/* Lecture */
		memmove (buffer, buffer+pos, nb-pos);						/* tout ce qui est >= pos est dplac au dbut de buffer */
		if ((nbnew = READ(isgz,fd,buffer+(nb-pos),sizeof(buffer)-1-(nb-pos))) <= 0)	/* Lecture a buffer+pos, garder un car de rserve */
			return NULL;

		nb = nb-pos+nbnew;
		pos = 0;

		if (isxml) {
			// Correction pour les fichiers SINP ou toutes les donnes sont sur une ligne: on remplace les "</TR>" par "</TR\n"
			for (i=0; i<nb ; i++) {
				if ((buffer[i]=='>') && (buffer[i-1]=='R'||buffer[i-1]=='r') && (buffer[i-2]=='T'||buffer[i-2]=='t') && (buffer[i-3]=='<')) {
					buffer[i] = '\n';
				}
			}
		}

	}

}
#endif


#ifndef TEST
/*---------------------------------------------------------------------------*/
void WC_L (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	IDL_STRING	*nom_IDL	=	(IDL_STRING *)	argv[0];
	int		nblignes0	=	*(int *)	argv[1];
	int		usecomma	=	*(int *)	argv[2];
	int		usechar		=	*(int *)	argv[3];
	int		*nblignes	=	(int *)		argv[4];
	int		*nbspecial	=	(int *)		argv[5];	// nombre de fois ou chaque dernier caractre
	int		*nbcomma	=	(int *)		argv[6];	// nombre de fois ou ',' a t trouv

	FILE		*fd		=	NULL;
	int		isgz		=	0;
	int		valid;
	int		i;
	char		*p;
	int		booster[256];
	int		code;
	int		len;

	#define	SKIP	1
	#define	END	2
	#define	STRING	3
	#define	COMMA	4

	for (i=0 ; i<256 ; i++) {

		booster[i]=0;
		if (i==' ' || i=='\t')			booster[i] = SKIP;
		if (i=='!' || i==0 || i==10 || i==13)	booster[i] = END;
		if (i=='"')				booster[i] = STRING;
		if (i==',')				booster[i] = COMMA;

	}

	*nblignes = 0;
	isgz = ISGZ(IDL_STRING_STR(nom_IDL));
	fd = OPEN(isgz,IDL_STRING_STR(nom_IDL));

	MY_GETS(isgz,fd,1,&len);

	// On saute les lignes d'entte
	// ----------------------------
	for (i=0 ; i<nblignes0 ; i++) {
		if ((p=MY_GETS(isgz,fd,0,&len)) == NULL) goto FIN;
	}

	*nblignes = nblignes0;

	if (usecomma==0 && usechar==0) {

		//printf ("wc_l  QUICK  %s\n", IDL_STRING_STR(nom_IDL));
		while ((p=MY_GETS(isgz,fd,0,&len)) != NULL) {
			for (i=0 ; i<len && p[i]!='!' ; i++);
			for (i-- ; i>=0 && booster[(int)p[i]]==SKIP ; i--);

			// 02/02/2010
			// Correction bug dans wc_l avec des fichiers ascii cres sous Linux puis rcuprs sous Windows par ftp ascii
			// Les retours charriots "10" sont transforms en "13-10"
			if (i>0 && p[i]==13) i--;

			if (i>=0) {
				(*nblignes)++;
				nbspecial[(unsigned char)p[i]]++;	// unsigned char pour avoir une valeur positive meme si p est un char *
			}
		}

	} else {

		//printf ("wc_l  COMPLETE  %s  %s\n", (usecomma && usechar) ? "comma+char" : (usecomma ? "comma" : ( usechar ? "char" : "")), IDL_STRING_STR(nom_IDL));
		while ((p=MY_GETS(isgz,fd,0,&len)) != NULL) {

			valid=0;

			// On scanne les caractres valides
			// --------------------------------
			for ( ; ; ) {

				if ((code = booster[(int)*p]) == SKIP) {
					p++;
					while (booster[(int)*p]==SKIP) p++;
					code = booster[(int)*p];
				}

				if (code == STRING) {
					p++;
					while (*p && *p!='"') p++;
					if (*p) p++;
					continue;
				}

				if (code == END) break;
				valid = *(p++);
				if (usecomma && valid==',') (*nbcomma)++;

			}

			if (valid) {
				(*nblignes)++;
				nbspecial[valid]++;
			}

		}
	}

FIN:
	if (fd != NULL) CLOSE(isgz,fd);

}

/*---------------------------------------------------------------------------*/
void WC_L_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p_nom_IDL,
	void	*p02,
	void	*p03,
	void	*p04,
	void	*p05,
	void	*p06,
	void	*p07) {

	int		argc=7;
	void		*argv[7];
	IDL_STRING	*nom_IDL;

	// p_nom_IDL -> nom_IDL
	nom_IDL = malloc (sizeof(IDL_STRING));
	IDL_StrStore (nom_IDL, p_nom_IDL);

	argv[0] = nom_IDL;
	argv[1] = p02;
	argv[2] = p03;
	argv[3] = p04;
	argv[4] = p05;
	argv[5] = p06;
	argv[6] = p07;
	WC_L (argc, argv);

	free (nom_IDL);

}
#endif


/*---------------------------------------------------------------------------*/
int my_strncasecmp (char *s1, char *s2, int n) {
/*---------------------------------------------------------------------------*/

	int	i;

	for (i=0 ; i<n; i++) {

		if (s1[i]==0 || s2[i]==0 || (toupper(s1[i]) != toupper(s2[i])))
			return 1;

	}

	return 0;

}


/*---------------------------------------------------------------------------*/
int julday (int mm, int id, int iyyy, int *julian) {
/*---------------------------------------------------------------------------*/

	#define igreg1 588829

	int ja,jm,jy,jul;

	if (iyyy==0)
		return -1;

	if (iyyy < 0)
		iyyy = iyyy + 1; 
	if (mm > 2) {
		jy = iyyy;
		jm = mm+1;
	} else {
		jy = iyyy-1;
		jm = mm+13;
	}
	jul = (int)(365.25*jy) + (int)(30.6001*jm) + id + 1720995;

	if (id+31*(mm+12*iyyy) >= igreg1) {
		ja = (int)(0.01*jy);
		jul = jul + 2 - ja + (int)(0.25*ja);
	}
	*julian = jul;

	return 0;

}

/*---------------------------------------------------------------------------*/
void caldat (int *mm, int *id, int *iyyy, int julian) {
/*---------------------------------------------------------------------------*/

	#define igreg2 2299161

	int je,jd,jc,jb,jalpha,ja;

	if (julian > igreg2) {
		jalpha = (int)(((julian-1867216)-0.25)/36524.25);
		ja = julian + 1 + jalpha - (int)(0.25*jalpha);
	} else
		ja = julian;

	jb = ja + 1524;
	jc = (int)(6680.0 + ((jb - 2439870)-122.1)/365.25);
	jd = 365*jc + (int)(0.25*jc);
	je = (int)((jb - jd)/30.6001);
	*id = jb - jd - (int)(30.6001*je);
	*mm = je - 1;
	if (*mm > 12)
		*mm = *mm - 12;
	*iyyy = jc - 4715;
	if (*mm > 2)
		*iyyy = *iyyy - 1;
	if (*iyyy <= 0)
		*iyyy = *iyyy - 1;

}


#define julian1958 2436205

/*---------------------------------------------------------------------------*/
void date_to_tu (double date, int *annee, int *mois, int *jour, int *heure, int *minute, double *seconde) {
/*---------------------------------------------------------------------------*/

	int	julian;

	julian = julian1958 + (int)(date/86400000);
	caldat (mois, jour, annee, julian);

	*seconde = (date - (double)(julian-julian1958)*86400000)/1000;
	*heure = *seconde / 3600;

	*seconde = *seconde - *heure*3600;

	*minute = *seconde / 60;

	*seconde = *seconde - *minute*60;

}

/*---------------------------------------------------------------------------*/
double tu_to_date (int ann, int moi, int jou, int heu, int min, double sec) {
/*---------------------------------------------------------------------------*/

	int	julian=0; // initialis sinon warning gcc
	int	nbjours;
	double	nbms;
	double	date;

	#define	julian1958 2436205

	// static variables to call julday() only when necessary
	static int ann1 = -1;
	static int moi1 = -1;
	static int jou1 = -1;
	static int julian1 = -1;

	if (ann==ann1 && moi==moi1 && jou==jou1) {
		julian = julian1;
	} else {
		julday (moi, jou, ann, &julian);
		ann1 = ann;
		moi1 = moi;
		jou1 = jou;
		julian1 = julian;
	}

	nbjours = julian - julian1958;
	nbms    = heu*3600000. + min*60000. + sec*1000.;
	date    = nbjours*86400000. + nbms;
	return date;

}


#ifdef USE_MACRO
/*---------------------------------------------------------------------------*/
#define SSCANF_REEL(buffer,fillval,resultat,dep,code) {											\
/*---------------------------------------------------------------------------*/								\
	int	SIGNE;															\
	double 	RES;															\
	int	EXPOSANT;														\
	int 	EXPSIGNE;														\
	int 	J;															\
	int	CHIFFRE;														\
	int	POINT;															\
																	\
	code = 1;															\
	for (dep=0 ; (buffer[dep]==' ') || (buffer[dep]=='\t') ; dep++);								\
	if (ISPLUS(buffer[dep])) {													\
		dep++;															\
		SIGNE='+';														\
	} else if (ISMINUS(buffer[dep])) {												\
		dep++;															\
		SIGNE='-';														\
	} else {															\
		SIGNE=' ';														\
	}																\
	if ((CHIFFRE=ISDIGIT(buffer[dep]))) {												\
		resultat = buffer[dep++] - '0'; while (ISDIGIT(buffer[dep])) { resultat *= 10; resultat += buffer[dep++] - '0';}	\
	} else if (buffer[dep]=='N' || buffer[dep]=='n' || buffer[dep]=='I' || buffer[dep]=='i') {					\
		if ((my_strncasecmp (buffer+dep, "NaN", 3) == 0) || (my_strncasecmp (buffer+dep, "Inf", 3) == 0)) {			\
			resultat = fillval;												\
			dep += 3;													\
		} else {														\
			resultat = 0;													\
			code = 0;													\
		}															\
	} else {															\
		resultat = 0;														\
		code = ISPOINT(buffer[dep]) || ISE(buffer[dep]);									\
	}																\
	if ((POINT=ISPOINT(buffer[dep]))) {												\
		dep++;															\
		J = dep;														\
		if (ISDIGIT(buffer[dep])) {												\
			RES=buffer[dep++] - '0'; while (ISDIGIT(buffer[dep])) { RES *= 10; RES += buffer[dep++] - '0'; }		\
			if (dep-J<MAX10)												\
				RES *= DIVIS[dep-J];											\
			else														\
				RES = 0;				/* Underflow */ 						\
			resultat += RES;												\
		} else if (CHIFFRE==0) {												\
			code = 0;					/* Pas de chiffre et pas de chiffre aprs le point */		\
		}															\
	}																\
	if (SIGNE == '-')														\
		resultat = -resultat;													\
	if (ISE(buffer[dep])) {														\
		if (CHIFFRE==0 && POINT==0) {												\
			code = 0;													\
		} else {														\
			dep++;														\
			if (ISPLUS(buffer[dep])) {											\
				dep++;													\
				EXPSIGNE='+';												\
			} else if (ISMINUS(buffer[dep])) {										\
				dep++;													\
				EXPSIGNE='-';												\
			} else {													\
				EXPSIGNE=' ';												\
			}														\
			if (ISDIGIT(buffer[dep])) {											\
				EXPOSANT=buffer[dep++]-'0';while (ISDIGIT(buffer[dep])) {EXPOSANT *= 10;EXPOSANT+=buffer[dep++]-'0';}	\
				if (EXPSIGNE == '-') {											\
					if (EXPOSANT<MAX10)										\
						resultat *= DIVIS[EXPOSANT];								\
					else												\
						resultat = 0;			/* Underflow */						\
				} else {												\
					if (EXPOSANT<MAX10)										\
						resultat *= MULTI[EXPOSANT];								\
					else												\
						resultat = fillval;		/* Overflow */						\
				}													\
			}														\
		}															\
	}																\
}
#endif


#ifdef USE_PENOU
/*---------------------------------------------------------------------------*/
int SSCANF_REEL (char *buffer, double fillval, double *resultat, int *dep) {
/*---------------------------------------------------------------------------*/

	int	signe;
	double 	res;
	int	exposant;
	int 	expsigne;	// int plus rapide que char
	int 	j;
	int	chiffre;
	int	point;

	for (*dep=0 ; (buffer[*dep]==' ') || (buffer[*dep]=='\t') ; (*dep)++);
	if (ISPLUS(buffer[*dep])) {
		(*dep)++;
		signe='+';
	} else if (ISMINUS(buffer[*dep])) {
		(*dep)++;
		signe='-';
	} else {
		signe=' ';
	}

	if ((chiffre=ISDIGIT(buffer[*dep]))) {
		*resultat = buffer[(*dep)++] - '0'; while (ISDIGIT(buffer[*dep])) { *resultat *= 10; *resultat += buffer[(*dep)++] - '0'; }
	} else if (buffer[*dep]=='N' || buffer[*dep]=='n' || buffer[*dep]=='I' || buffer[*dep]=='i') { // gain de temps
		if ((my_strncasecmp (buffer+(*dep), "NaN", 3) == 0) || (my_strncasecmp (buffer+(*dep), "Inf", 3) == 0)) {
			*resultat = fillval;
			*dep += 3;
			return 1;
		} else {
			return 0;
		}
	} else {
		if (!(ISPOINT(buffer[*dep]) || ISE(buffer[*dep])))
			return 0;
		*resultat = 0;
	}

	if ((point=ISPOINT(buffer[*dep]))) {
		(*dep)++;
		j = *dep;
		if (ISDIGIT(buffer[*dep])) {
			res=buffer[(*dep)++] - '0'; while (ISDIGIT(buffer[*dep])) { res *= 10; res += buffer[(*dep)++] - '0'; }
			if (*dep-j<MAX10)
				res *= DIVIS[*dep-j];
			else
				res = 0;					// Underflow
			*resultat += res;
		} else if (chiffre==0) {
			return 0;						// Pas de chiffre et pas de chiffre aprs le point
		}
	}

	if (ISE(buffer[*dep])) {
		if (chiffre==0 && point==0) {
			return 0;
		} else {
			(*dep)++;
			if (ISPLUS(buffer[*dep])) {
				(*dep)++;
				expsigne='+';
			} else if (ISMINUS(buffer[*dep])) {
				(*dep)++;
				expsigne='-';
			} else {
				expsigne=' ';
			}
			if (ISDIGIT(buffer[*dep])) {
				exposant = buffer[(*dep)++] - '0'; while (ISDIGIT(buffer[*dep])) { exposant *= 10; exposant += buffer[(*dep)++] - '0'; }
				if (expsigne == '-') {
					if (exposant<MAX10)
						*resultat *= DIVIS[exposant];
					else
						*resultat = 0;					// Underflow
				} else {
					if (exposant<MAX10)
						*resultat *= MULTI[exposant];
					else
						*resultat = fillval;				// Overflow
				}
			}
		}
	}

	if (signe == '-')
		*resultat = - *resultat;

	return 1;

}
#endif


#ifdef USE_STRTOD
/*---------------------------------------------------------------------------*/
int SSCANF_REEL (char *buffer, double fillval, double *resultat, int *dep) {
/*---------------------------------------------------------------------------*/
	char	*ptr;

	errno=0;
	*resultat = strtod(buffer,&ptr);
	*dep = (int)(ptr-buffer);
	return errno==0;
} 
#endif


#ifdef USE_MACRO
/*---------------------------------------------------------------------------*/
#define SSCANF_ENTIER(buffer,resultat,dep,code) {											\
/*---------------------------------------------------------------------------*/								\
	for (dep=0 ; (buffer[dep]==' ')||(buffer[dep]=='\t') ; dep++);									\
	if ((code=ISDIGIT(buffer[dep]))) {												\
		resultat=buffer[dep++]-'0'; while (ISDIGIT(buffer[dep])) { resultat *= 10; resultat += buffer[dep++]-'0'; }		\
	}																\
}
#else
/*---------------------------------------------------------------------------*/
int SSCANF_ENTIER (char *buffer, int *resultat, int *dep) {
/*---------------------------------------------------------------------------*/
	for (*dep=0 ; (buffer[*dep]==' ')||(buffer[*dep]=='\t') ; (*dep)++);
	if (!ISDIGIT(buffer[*dep])) return 0;
	*resultat = buffer[(*dep)++] - '0'; while (ISDIGIT(buffer[*dep])) { *resultat *= 10; *resultat += buffer[(*dep)++] - '0'; }
	return 1;
}
#endif

#ifdef USE_MACRO
/*---------------------------------------------------------------------------*/
#define READ_DATE_FORMAT_CCSDS() {								\
/*---------------------------------------------------------------------------*/			\
	SSCANF_ENTIER(p,annee, dep,code);		if (code != 1) goto ERROR;		\
	p += dep; if (*p != '-') goto ERROR; p++;						\
	SSCANF_ENTIER(p,mois,dep,code);			if (code != 1) goto ERROR;		\
	p += dep; if (*p != '-') goto ERROR; p++;						\
	SSCANF_ENTIER(p,jour,dep,code);			if (code != 1) goto ERROR;		\
	p += dep; if (*p != 'T') goto ERROR; p++;						\
	SSCANF_ENTIER(p,heure,dep,code);		if (code != 1) goto ERROR;		\
	p += dep; if (*p != ':') goto ERROR; p++;						\
	SSCANF_ENTIER(p,minute,dep,code);		if (code != 1) goto ERROR;		\
	p += dep; if (*p != ':') goto ERROR; p++;						\
	SSCANF_REEL(p,fillval,seconde,dep,code);	if (code != 1) goto ERROR;		\
	p += dep;										\
}
#else
/*---------------------------------------------------------------------------*/
#define READ_DATE_FORMAT_CCSDS() {								\
/*---------------------------------------------------------------------------*/			\
	if (SSCANF_ENTIER(p,&annee,&dep) != 1) goto ERROR;					\
	p += dep; if (*p != '-') goto ERROR; p++;						\
	if (SSCANF_ENTIER(p,&mois,&dep) != 1) goto ERROR;					\
	p += dep; if (*p != '-') goto ERROR; p++;						\
	if (SSCANF_ENTIER(p,&jour,&dep) != 1) goto ERROR;					\
	p += dep; if (*p != 'T') goto ERROR; p++;						\
	if (SSCANF_ENTIER(p,&heure,&dep) != 1) goto ERROR;					\
	p += dep; if (*p != ':') goto ERROR; p++;						\
	if (SSCANF_ENTIER(p,&minute,&dep) != 1) goto ERROR;					\
	p += dep; if (*p != ':') goto ERROR; p++;						\
	if (SSCANF_REEL(p,fillval,&seconde,&dep) != 1) goto ERROR;				\
	p += dep;										\
}
#endif

#ifdef USE_MACRO
/*---------------------------------------------------------------------------*/
#define READ_DATE_FORMAT_CEF() {								\
/*---------------------------------------------------------------------------*/			\
	do {											\
		SSCANF_ENTIER(p,annee,dep,code);						\
		if (code != 1) {								\
			if ((p=MY_GETS(isgz,fd,0,&len)) == NULL) goto ERROR;			\
		}										\
	} while (code != 1);									\
	p += dep; if (*p != '-') goto ERROR; p++;						\
	SSCANF_ENTIER(p,mois,dep,code);			if (code != 1) goto ERROR;		\
	p += dep; if (*p != '-') goto ERROR; p++;						\
	SSCANF_ENTIER(p,jour,dep,code);			if (code != 1) goto ERROR;		\
	p += dep; if (*p != 'T') goto ERROR; p++;						\
	SSCANF_ENTIER(p,heure,dep,code);		if (code != 1) goto ERROR;		\
	p += dep; if (*p != ':') goto ERROR; p++;						\
	SSCANF_ENTIER(p,minute,dep,code);		if (code != 1) goto ERROR;		\
	p += dep; if (*p != ':') goto ERROR; p++;						\
	SSCANF_REEL(p,fillval,seconde,dep,code);	if (code != 1) goto ERROR;		\
	p += dep; if (*p != 'Z') goto ERROR; dep=1;						\
}
#else
/*---------------------------------------------------------------------------*/			\
#define READ_DATE_FORMAT_CEF() {								\
/*---------------------------------------------------------------------------*/			\
	do {											\
		code = SSCANF_ENTIER (p, &annee, &dep);						\
		if (code != 1) {								\
			if ((p=MY_GETS(isgz,fd,0,&len)) == NULL) goto ERROR;			\
		}										\
	} while (code != 1);									\
	p += dep; if (*p != '-') goto ERROR; p++;						\
	if (SSCANF_ENTIER (p, &mois, &dep) != 1)	goto ERROR;				\
	p += dep; if (*p != '-') goto ERROR; p++;						\
	if (SSCANF_ENTIER (p, &jour, &dep) != 1)	goto ERROR;				\
	p += dep; if (*p != 'T') goto ERROR; p++;						\
	if (SSCANF_ENTIER (p, &heure, &dep) != 1)	goto ERROR;				\
	p += dep; if (*p != ':') goto ERROR; p++;						\
	if (SSCANF_ENTIER (p, &minute, &dep) != 1)	goto ERROR;				\
	p += dep; if (*p != ':') goto ERROR; p++;						\
	if (SSCANF_REEL (p,fillval,&seconde,&dep) != 1)	goto ERROR;				\
	p += dep; if (*p != 'Z') goto ERROR; dep=1;						\
}
#endif

#ifdef USE_MACRO
/*---------------------------------------------------------------------------*/			\
#define READ_DATE_FORMAT_7() {									\
/*---------------------------------------------------------------------------*/			\
	SSCANF_ENTIER(p,annee,dep,code);		if (code != 1) goto ERROR;		\
	p += dep; if (*p != '-') goto ERROR; p++;						\
	SSCANF_ENTIER(p,mois,dep,code);			if (code != 1) goto ERROR;		\
	p += dep; if (*p != '-') goto ERROR; p++;						\
	SSCANF_ENTIER(p,jour,dep,code);			if (code != 1) goto ERROR;		\
	p += dep; if (*p != 'T') goto ERROR; p++;						\
	SSCANF_ENTIER(p,heure,dep,code);		if (code != 1) goto ERROR;		\
	p += dep; if (*p != ':') goto ERROR; p++;						\
	SSCANF_ENTIER(p,minute,dep,code);		if (code != 1) goto ERROR;		\
	p += dep; if (*p != ':') goto ERROR; p++;						\
	SSCANF_REEL(p,fillval,seconde,dep,code);	if (code != 1) goto ERROR;		\
}
#else
/*---------------------------------------------------------------------------*/			\
#define READ_DATE_FORMAT_7() {									\
/*---------------------------------------------------------------------------*/			\
	if (SSCANF_ENTIER (p, &annee, &dep) != 1)	goto ERROR;				\
	p += dep; if (*p != '-') goto ERROR; p++;						\
	if (SSCANF_ENTIER (p, &mois, &dep) != 1)	goto ERROR;				\
	p += dep; if (*p != '-') goto ERROR; p++;						\
	if (SSCANF_ENTIER (p, &jour, &dep) != 1)	goto ERROR;				\
	p += dep; if (*p != 'T') goto ERROR; p++;						\
	if (SSCANF_ENTIER (p, &heure, &dep) != 1)	goto ERROR;				\
	p += dep; if (*p != ':') goto ERROR; p++;						\
	if (SSCANF_ENTIER (p, &minute, &dep) != 1)	goto ERROR;				\
	p += dep; if (*p != ':') goto ERROR; p++;						\
	if (SSCANF_REEL (p,fillval,&seconde,&dep) != 1)	goto ERROR;				\
}
#endif

#ifdef USE_MACRO
/*---------------------------------------------------------------------------*/			\
#define READ_DATE_FORMAT_10() {									\
/*---------------------------------------------------------------------------*/			\
	SSCANF_ENTIER(p,annee,dep,code);		if (code != 1) goto ERROR;		\
	p += dep; if (*p != '-') goto ERROR; p++;						\
	SSCANF_ENTIER(p,mois,dep,code);			if (code != 1) goto ERROR;		\
	p += dep; if (*p != '-') goto ERROR; p++;						\
	SSCANF_ENTIER(p,jour,dep,code);			if (code != 1) goto ERROR;		\
	p += dep; if (*p != ' ') goto ERROR; p++;						\
	SSCANF_ENTIER(p,heure,dep,code);		if (code != 1) goto ERROR;		\
	p += dep; if (*p != ':') goto ERROR; p++;						\
	SSCANF_ENTIER(p,minute,dep,code);		if (code != 1) goto ERROR;		\
	p += dep; if (*p != ':') goto ERROR; p++;						\
	SSCANF_REEL(p,fillval,seconde,dep,code);	if (code != 1) goto ERROR;		\
}
#else
/*---------------------------------------------------------------------------*/			\
#define READ_DATE_FORMAT_10() {									\
/*---------------------------------------------------------------------------*/			\
	if (SSCANF_ENTIER (p, &annee, &dep) != 1)	goto ERROR;				\
	p += dep; if (*p != '-') goto ERROR; p++;						\
	if (SSCANF_ENTIER (p, &mois, &dep) != 1)	goto ERROR;				\
	p += dep; if (*p != '-') goto ERROR; p++;						\
	if (SSCANF_ENTIER (p, &jour, &dep) != 1)	goto ERROR;				\
	p += dep; if (*p != ' ' && *p!= ' ') goto ERROR; p++;					\
	if (SSCANF_ENTIER (p, &heure, &dep) != 1)	goto ERROR;				\
	p += dep; if (*p != ':') goto ERROR; p++;						\
	if (SSCANF_ENTIER (p, &minute, &dep) != 1)	goto ERROR;				\
	p += dep; if (*p != ':') goto ERROR; p++;						\
	if (SSCANF_REEL (p,fillval,&seconde,&dep) != 1)	goto ERROR;				\
}
#endif


#ifdef USE_MACRO
/*---------------------------------------------------------------------------*/			\
#define READ_DATE_FORMAT_4() {									\
/*---------------------------------------------------------------------------*/			\
	SSCANF_ENTIER(p,annee, dep,code);		if (code != 1) goto ERROR;		\
	p += dep;										\
	SSCANF_ENTIER(p,mois,dep,code);			if (code != 1) goto ERROR;		\
	p += dep;										\
	SSCANF_ENTIER(p,jour,dep,code);			if (code != 1) goto ERROR;		\
	p += dep;										\
	SSCANF_ENTIER(p,heure,dep,code);		if (code != 1) goto ERROR;		\
	p += dep;										\
	SSCANF_ENTIER(p,minute,dep,code);		if (code != 1) goto ERROR;		\
	p += dep;										\
	SSCANF_REEL(p,fillval,seconde, dep,code);	if (code != 1) goto ERROR;		\
}
#else
/*---------------------------------------------------------------------------*/			\
#define READ_DATE_FORMAT_4() {									\
/*---------------------------------------------------------------------------*/			\
	if (SSCANF_ENTIER (p, &annee, &dep) != 1)	goto ERROR;				\
	p += dep;										\
	if (SSCANF_ENTIER (p, &mois, &dep) != 1)	goto ERROR;				\
	p += dep;										\
	if (SSCANF_ENTIER (p, &jour, &dep) != 1)	goto ERROR;				\
	p += dep;										\
	if (SSCANF_ENTIER (p, &heure, &dep) != 1)	goto ERROR;				\
	p += dep;										\
	if (SSCANF_ENTIER (p, &minute, &dep) != 1)	goto ERROR;				\
	p += dep;										\
	if (SSCANF_REEL (p,fillval,&seconde,&dep) != 1)	goto ERROR;				\
}
#endif


#ifdef USE_MACRO
/*---------------------------------------------------------------------------*/			\
#define READ_DATE_FORMAT_8() {									\
/*---------------------------------------------------------------------------*/			\
	SSCANF_ENTIER(p,annee, dep,code);		if (code != 1) goto ERROR;		\
	p += dep;										\
	SSCANF_ENTIER(p,doy,dep,code);			if (code != 1) goto ERROR;		\
	p += dep;										\
	SSCANF_ENTIER(p,heure,dep,code);		if (code != 1) goto ERROR;		\
	p += dep;										\
	SSCANF_ENTIER(p,minute,dep,code);		if (code != 1) goto ERROR;		\
	p += dep;										\
	SSCANF_REEL(p,fillval,seconde, dep,code);	if (code != 1) goto ERROR;		\
}
#else
/*---------------------------------------------------------------------------*/			\
#define READ_DATE_FORMAT_8() {									\
/*---------------------------------------------------------------------------*/			\
	if (SSCANF_ENTIER (p, &annee, &dep) != 1)	goto ERROR;				\
	p += dep;										\
	if (SSCANF_ENTIER (p, &doy, &dep) != 1)		goto ERROR;				\
	p += dep;										\
	if (SSCANF_ENTIER (p, &jour, &dep) != 1)	goto ERROR;				\
	p += dep;										\
	if (SSCANF_ENTIER (p, &heure, &dep) != 1)	goto ERROR;				\
	p += dep;										\
	if (SSCANF_ENTIER (p, &minute, &dep) != 1)	goto ERROR;				\
	p += dep;										\
	if (SSCANF_REEL (p,fillval,&seconde,&dep) != 1)	goto ERROR;				\
}
#endif


#ifdef USE_MACRO
/*---------------------------------------------------------------------------*/			\
#define READ_DATE_FORMAT_1() {									\
/*---------------------------------------------------------------------------*/			\
	SSCANF_ENTIER(p,jour,dep,code);			if (code != 1) goto ERROR;		\
	p += dep;										\
	SSCANF_ENTIER(p,mois,dep,code);			if (code != 1) goto ERROR;		\
	p += dep;										\
	SSCANF_ENTIER(p,annee,dep,code);		if (code != 1) goto ERROR;		\
	p += dep;										\
	SSCANF_ENTIER(p,heure,dep,code);		if (code != 1) goto ERROR;		\
	p += dep;										\
	SSCANF_ENTIER(p,minute,dep,code);		if (code != 1) goto ERROR;		\
	p += dep;										\
	SSCANF_REEL(p,fillval,seconde,dep,code);	if (code != 1) goto ERROR;		\
}
#else
/*---------------------------------------------------------------------------*/			\
#define READ_DATE_FORMAT_1() {									\
/*---------------------------------------------------------------------------*/			\
	if (SSCANF_ENTIER (p, &jour, &dep) != 1)	goto ERROR;				\
	p += dep;										\
	if (SSCANF_ENTIER (p, &mois, &dep) != 1)	goto ERROR;				\
	p += dep;										\
	if (SSCANF_ENTIER (p, &annee, &dep) != 1)	goto ERROR;				\
	p += dep;										\
	if (SSCANF_ENTIER (p, &heure, &dep) != 1)	goto ERROR;				\
	p += dep;										\
	if (SSCANF_ENTIER (p, &minute, &dep) != 1)	goto ERROR;				\
	p += dep;										\
	if (SSCANF_REEL (p,fillval,&seconde,&dep) != 1)	goto ERROR;				\
}
#endif

/*---------------------------------------------------------------------------*/			\
#define SKIP_BLANK_TAB() {									\
/*---------------------------------------------------------------------------*/			\
	/* Sauter tous les ' ' et '\t' */							\
	while (*p==' ' || *p=='\t') p++;							\
}


/*---------------------------------------------------------------------------*/			\
#define SKIP_COMMA() {										\
/*---------------------------------------------------------------------------*/			\
	/* Sauter le ',' s'il existe */								\
	if (*p == ',') p++;									\
	while (isxml) {										\
		/* Sauter tous les ' ', '\t', '<TR>', '<TD>', '</TR>' '</TD>' */		\
		/* Lire une nouvelle ligne si ncessaire */					\
		if (*p==0) {									\
			if ((p=MY_GETS(isgz,fd,0,&len)) == NULL) goto ERROR;			\
			continue;								\
		}										\
		if (ISTAG[(int)*p]) {								\
			p++;									\
			continue;								\
		}										\
		break;										\
	}											\
}


/*---------------------------------------------------------------------------*/			\
#define SKIP_SLASH() {										\
/*---------------------------------------------------------------------------*/			\
	/* Sauter le '/' des ISO_TIME_RANGE */							\
	if (*p == '/') p++;									\
}


/*---------------------------------------------------------------------------*/			\
#define FIND_BLANK_TAB_COMMA() {								\
/*---------------------------------------------------------------------------*/			\
	/* Sauter tous les caractres jusqu' trouver ' ' ou '\t' ou ',' pour MAG VEX */	\
	if (isxml==0) {										\
		while (*p!=' ' && *p!='\t' && *p!=',') p++;					\
	} else {										\
		while (ISTAG[(int)*p]) p++;							\
	}											\
}


/*---------------------------------------------------------------------------*/			\
#define READ_STRING() {										\
/*---------------------------------------------------------------------------*/			\
	/* Sauter tous les ' ' , '\t' et 's' */							\
	while (*p==' ' || *p=='\t' || *p==',') p++;						\
												\
	/* suppression s'il existe du retour chariot en fin de ligne (fichier mode dos) */	\
	if (p[strlen(p)-1]==13) p[strlen(p)-1] = '\0';						\
												\
	pIS = (IDL_STRING *)(data + cte_STRING_CHAR + (nbC+position[j])*sizeof_string);		\
	if (is_fdl) {										\
		sprintf (pIS->s,"%s%c", p, 255);						\
	} else {										\
		IDL_StrStore (pIS,p);								\
	}											\
}


/*---------------------------------------------------------------------------*/			\
#define READ_CHAR(ok) {										\
/*---------------------------------------------------------------------------*/			\
	/* la chaine doit tre sur une seule ligne, entoure ventuellemnt par '"' et '"' */	\
												\
	do {											\
		/* Sauter tous les caractres jusqu' trouver '"', ou non blanc */		\
		while (*p && *p!='"' && (*p==' ' || *p=='\t')) p++;				\
		code = (*p!=0 && *p!='!' && *p!=10 && *p!=13);					\
		if (code != 1) {								\
			if ((p=MY_GETS(isgz,fd,0,&len)) == NULL) goto ERROR;			\
		}										\
	} while (code!=1);									\
												\
	if (ok) pIS = (IDL_STRING *)(data + cte_STRING_CHAR + position[j]*sizeof_string);	\
	if (*p=='"') {										\
		p++;										\
		pstring = p;									\
		/* Sauter tous les caractres jusqu' trouver '"' */				\
		while (*p && *p!='"') p++;							\
		tmp = *p;									\
		*p = 0;										\
		if (ok) {									\
			if (is_fdl) {								\
				sprintf (pIS->s,"%s%c", pstring, 255);				\
			} else {								\
				IDL_StrStore (pIS,pstring);					\
			}									\
		}										\
		*p = tmp;									\
		if (*p) p++;									\
	} else {										\
		pstring = p;									\
		p++;										\
		if (isxml) {									\
			/* Sauter tous les caractres jusqu' trouver '<' */			\
			while (*p && *p!='<') p++;						\
		} else {									\
			/* Sauter tous les caractres jusqu' trouver ' ', '\t' ou ',' */	\
			while (*p && *p!='\t' && *p!=' ' && *p!=',') p++;			\
		}										\
		tmp = *p;									\
		*p = 0;										\
		if (ok) {									\
			if (is_fdl) {								\
				sprintf (pIS->s,"%s%c", pstring, 255);				\
			} else {								\
				IDL_StrStore (pIS,pstring);					\
			}									\
		}										\
		*p = tmp;									\
	}											\
	dep = 0;										\
}


#ifdef USE_MACRO
/*---------------------------------------------------------------------------*/			\
#define READ_FLOAT() {										\
/*---------------------------------------------------------------------------*/			\
	do {											\
		SSCANF_REEL(p,fillval,reel,dep,code);						\
		if (code != 1) {								\
			if ((p=MY_GETS(isgz,fd,0,&len)) == NULL) goto ERROR;			\
		}										\
	} while (code != 1);									\
}
#else
/*---------------------------------------------------------------------------*/			\
#define READ_FLOAT() {										\
/*---------------------------------------------------------------------------*/			\
	do {											\
		code = SSCANF_REEL (p,fillval,&reel,&dep);					\
		if (code != 1) {								\
			if ((p=MY_GETS(isgz,fd,0,&len)) == NULL) goto ERROR;			\
		}										\
	} while (code != 1);									\
}
#endif


/*---------------------------------------------------------------------------*/
int STR1_TO_DATE (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	int		nb		= *(int *)		argv[0];
	IDL_STRING	*str		= (IDL_STRING *)	argv[1];
	double		infini		= *(double *)		argv[2];
	double		*tabdate	= (double *)		argv[3];
	
	int		annee;
	int		mois;
	int		jour;
	int		heure;
	int		minute;
	double		seconde;
	int		i;

	int		dep;
	char		*p;
	int		code;
	float		fillval=-1e31;

	for (i=0 ; i<nb ; i++) {

		p=IDL_STRING_STR(str+i);
		READ_DATE_FORMAT_1();
		tabdate[i] = tu_to_date (annee, mois, jour, heure, minute, seconde);
		continue;
ERROR:
		tabdate[i] = infini;
	}

	return 0;

}

/*---------------------------------------------------------------------------*/
int STR1_TO_DATE_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p_nb,
	void	*p_str,
	void	*p03,
	void	*p04) {

	int		argc=4;
	void		*argv[4];
	int		nb = * (int *) p_nb;
	int		i;
	int		code;
	IDL_STRING	*str;

	// p_str -> str
	str = malloc (nb * sizeof(IDL_STRING));
	for (i=0 ; i<nb ; i++) IDL_StrStore (str+i, p_str+i*1024);

	argv[0] = p_nb;
	argv[1] = str;
	argv[2] = p03;
	argv[3] = p04;
	code = STR1_TO_DATE (argc, argv);

	free (str);

	return code;
}

/*---------------------------------------------------------------------------*/
int STRCCSDS_TO_DATE (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	int		nb		= *(int *)		argv[0];
	IDL_STRING	*str		= (IDL_STRING *)	argv[1];
	double		infini		= *(double *)		argv[2];
	double		*tabdate	= (double *)		argv[3];
	
	int		annee;
	int		mois;
	int		jour;
	int		heure;
	int		minute;
	double		seconde;
	int		i;

	int		dep;
	char		*p;
	int		code;
	float		fillval=-1e31;

	for (i=0 ; i<nb ; i++) {

		p=IDL_STRING_STR(str+i);
		READ_DATE_FORMAT_CCSDS();
		tabdate[i] = tu_to_date (annee, mois, jour, heure, minute, seconde);
		continue;
ERROR:
		tabdate[i] = infini;
	}

	return 0;

}

/*---------------------------------------------------------------------------*/
int STRCCSDS_TO_DATE_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p_nb,
	void	*p_str,
	void	*p03,
	void	*p04) {

	int		argc=4;
	void		*argv[4];
	int		nb = * (int *) p_nb;
	int		i;
	int		code;
	IDL_STRING	*str;

	// p_str -> str
	str = malloc (nb * sizeof(IDL_STRING));
	for (i=0 ; i<nb ; i++) IDL_StrStore (str+i, p_str+i*1024);

	argv[0] = p_nb;
	argv[1] = str;
	argv[2] = p03;
	argv[3] = p04;
	code = STRCCSDS_TO_DATE (argc, argv);

	free (str);

	return code;
}


/*---------------------------------------------------------------------------*/
int DATE_TO_STR1 (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	int		nb		= *(int *)		argv[0];
	double		*date		= (double *)		argv[1];
	IDL_STRING	*str		= (IDL_STRING *)	argv[2];

	int		annee;
	int		mois;
	int		jour;
	int		heure;
	int		minute;
	double		seconde;
	int		i;
	IDL_STRING	*pIS			=	NULL;	// gcc
	char		buffer[1024];

	for (i=0 ; i<nb ; i++) {
		date_to_tu (date[i], &annee, &mois, &jour, &heure, &minute, &seconde);
		sprintf (buffer,"%02d %02d %04d %02d %02d %06.3f", jour,mois,annee,heure,minute,seconde);

		pIS = (IDL_STRING *)(str+i);
		IDL_StrStore (pIS, buffer);
	}

	return 0;

}

/*---------------------------------------------------------------------------*/
int DATE_TO_STR1_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p_nb,
	void	*p02,
	void	*p_str) {

	int		argc=3;
	void		*argv[3];
	int		nb = * (int *) p_nb;
	int		i;
	int		code;
	IDL_STRING	*str;

	str = malloc (nb * sizeof(IDL_STRING));
	argv[0] = p_nb;
	argv[1] = p02;
	argv[2] = str;
	code = DATE_TO_STR1 (argc, argv);

	// str -> p_str
	for (i=0 ; i<nb ; i++) sprintf (p_str+i*1024, "%s", IDL_STRING_STR(str+i));

	free (str);

	return code;
}

/*---------------------------------------------------------------------------*/
int DATE_TO_STR3 (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	int		nb		= *(int *)		argv[0];
	double		*date		= (double *)		argv[1];
	IDL_STRING	*str		= (IDL_STRING *)	argv[2];

	int		annee;
	int		mois;
	int		jour;
	int		heure;
	int		minute;
	double		seconde;
	int		i;
	IDL_STRING	*pIS			=	NULL;	// gcc
	char		buffer[1024];

	for (i=0 ; i<nb ; i++) {
		date_to_tu (date[i], &annee, &mois, &jour, &heure, &minute, &seconde);
		sprintf (buffer,"%04d-%02d-%02dT%02d:%02d:%06.3fZ", annee,mois,jour,heure,minute,seconde);

		pIS = (IDL_STRING *)(str+i);
		IDL_StrStore (pIS, buffer);
	}

	return 0;

}

/*---------------------------------------------------------------------------*/
int DATE_TO_STR3_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p_nb,
	void	*p02,
	void	*p_str) {

	int		argc=3;
	void		*argv[3];
	int		nb = * (int *) p_nb;
	int		i;
	int		code;
	IDL_STRING	*str;

	str = malloc (nb * sizeof(IDL_STRING));
	argv[0] = p_nb;
	argv[1] = p02;
	argv[2] = str;
	code = DATE_TO_STR3 (argc, argv);

	// str -> p_str
	for (i=0 ; i<nb ; i++) sprintf (p_str+i*1024, "%s", IDL_STRING_STR(str+i));

	free (str);

	return code;
}


/*---------------------------------------------------------------------------*/
int DATE_TO_STR5 (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	int		nb		= *(int *)		argv[0];
	double		*date		= (double *)		argv[1];
	IDL_STRING	*str		= (IDL_STRING *)	argv[2];

	int		annee;
	int		mois;
	int		jour;
	int		heure;
	int		minute;
	double		seconde;
	int		i;
	IDL_STRING	*pIS			=	NULL;	// gcc
	char		buffer[1024];

	for (i=0 ; i<nb ; i++) {
		date_to_tu (date[i], &annee, &mois, &jour, &heure, &minute, &seconde);
		sprintf (buffer,"%04d-%02d-%02dT%02d:%02d:%02dZ", annee,mois,jour,heure,minute,(int)seconde);

		pIS = (IDL_STRING *)(str+i);
		IDL_StrStore (pIS, buffer);
	}

	return 0;

}

/*---------------------------------------------------------------------------*/
int DATE_TO_STR5_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p_nb,
	void	*p02,
	void	*p_str) {

	int		argc=3;
	void		*argv[3];
	int		nb = * (int *) p_nb;
	int		i;
	int		code;
	IDL_STRING	*str;

	str = malloc (nb * sizeof(IDL_STRING));
	argv[0] = p_nb;
	argv[1] = p02;
	argv[2] = str;
	code = DATE_TO_STR5 (argc, argv);

	// str -> p_str
	for (i=0 ; i<nb ; i++) sprintf (p_str+i*1024, "%s", IDL_STRING_STR(str+i));

	free (str);

	return code;
}

#ifndef TEST
/*---------------------------------------------------------------------------*/
int READ_ASCII (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

/*
	Retourne:	1 si OK
			0 si PB
*/

	IDL_STRING	*nom_IDL		= (IDL_STRING *)	argv[ 0];
	int		nb			= *(int *)		argv[ 1];	/* nombre de champ dans le fichier */
	char		*type			= (char *)		argv[ 2];	/* type[nb]
												- 'D':	date
												- 'R':	date/date
												- 'S':	string
												- 'C':	char
												- 'F':	float
												- 'I':	int
											*/
	char		*memorise		= (char *)		argv[ 3];	/* memorise[nb]:
												1 pour lire et stocker la valeur
												0 pour lire et sauter la valeur
											*/
	int		*position		= (int *)		argv[ 4];	/* position[nb]: position */
	char		*data			= (char *)		argv[ 5];	/* pointeur sur data */
	int		format			= *(int *)		argv[ 6];	/* 1,3,4,5,6 or 7

												1: 'DD MM YYYY HH MM SS[.MSC]'

												4: 'YYYY MM DD HH MM SS[.MSC]'

												3: 'YYYY-MM-DDTHH:MM:SS.MSCZ'
												5: 'YYYY-MM-DDTHH:MM:SSZ'

												7: 'YYYY-MM-DDTHH:MM:SS.MSC'

												6: file with no Epoch
											*/
	int		nblignes0		= *(int *)		argv[ 7];	/* nombre de lignes d'entete dans le fichier */
	int		REC_COUNT		= *(int *)		argv[ 8];	/* nombre d'enregistrements a lire */
	int		REC_START		= *(int *)		argv[ 9];	/* numero de la premiere ligne a lire (premiere=0) */
	int		taille			= *(int *)		argv[10];	/* taille d'un enregistrement (double+float: Linux=12, Win=8) */
	IDL_STRING	*sfillval		= (IDL_STRING *)	argv[11];	/* tableau de fillval en string */
	int		modify_float_fillval	= *(int *)		argv[12];	/* 1 to modify float fillval by -1e31 */
	int		votable			= *(int *)		argv[13];	/* 1 for VOTABLE format, 0 for CEF format */
	int		is_fdl			= *(int *)		argv[14];	/* 1 si FDL, 0 sinon */

	char		*p;
	char		*pstring;
	int		i;
	int		j;
	int		dep			=	0;	// gcc
	FILE		*fd			=	NULL;
	int		annee			=	0;	// gcc
	int		mois			=	0;	// gcc
	int		jour			=	0;	// gcc
	int		doy			=	0;	// gcc
	int		heure			=	0;	// gcc
	int		minute			=	0;	// gcc
	double		seconde			=	0;	// gcc
	double		*pdouble		=	NULL;	// gcc
	float		*pfloat;
	int		*pint;
	IDL_STRING	*pIS			=	NULL;	// gcc
	double		reel;
	int		isgz;
	int		code;
	int		ok;
	int		nbD			=	0;
	int		nbR			=	0;
	int		nbC			=	0;
	int		nbS			=	0;
	int		nbF			=	0;
	int		sauter;
	long long	cte_DATE;
	long long	cte_RANGE;
	long long	cte_STRING_CHAR;
	long long	cte_FLOAT_INT;
	float		*tabfillval		=	NULL;
	char		*tabopfillval		=	NULL; // '=' ou '>'
	float		fillval;
	char		opfillval;
	char		tmp;
	int		len;
	double		tmpdouble;
	float		tmpfloat;
	int		noligneERROR		=	0;

	int		votable_debut		=	1; // 1 pour chercher le tag <TABLEDATA>, 0 sinon

	int		taille_enreg_fichier	=	0; // taille d'un enregistrement pour ls fichiers binaires
	char		*tmpchar=NULL; // pour stocker un enregistrement binaire, initialis sinon warning gcc
	char		tmpligne[10240]; // pour stocker une ligne d'entte (1024 ne suffit pas pour JUNO)
	int		sizeof_string;

	sizeof_string = sizeof(IDL_STRING) * (is_fdl ? 2 : 1);

	#ifdef USE_MACRO
		#ifdef USE_RAM1
			//printf ("MACRO_RAM1\n");
		#endif
		#ifdef USE_RAM0
			printf ("WARNING read_ascii() uses MACRO_RAM0\n");
		#endif
	#endif
	#ifdef USE_PENOU
		#ifdef USE_RAM1
			printf ("WARNING read_ascii() uses PENOU_RAM1\n");
		#endif
		#ifdef USE_RAM0
			printf ("WARNING read_ascii() uses PENOU_RAM0\n");
		#endif
	#endif
	#ifdef USE_STRTOD
		#ifdef USE_RAM1
			printf ("WARNING read_ascii() uses STRTOD_RAM1\n");
		#endif
		#ifdef USE_RAM0
			printf ("WARNING read_ascii() uses STRTOD_RAM0\n");
		#endif
	#endif

	//printf ("taille=%d\n",taille);

	isxml = votable;

	isgz = ISGZ(IDL_STRING_STR(nom_IDL));	// pour que isgz soit connu dans l'tiquette ERROR

	if ((tabfillval = (float *) malloc(nb*sizeof(float))) == NULL) goto ERROR;
	if ((tabopfillval = (char *) malloc(nb)) == NULL) goto ERROR;

	for (j=0 ; j<nb ; j++) {

		if (type[j]=='F') {

			// A cause de "caa.estec.esa.int/C3/RAP/C3_CP_RAP_I3D_CNO_R/2001/02/C3_CP_RAP_I3D_CNO_R_20010201_V01.cef.gz"
			//START_VARIABLE  = Rec_Key__C3_CP_RAP_I3D_CNO_R
    			//	PARAMETER_TYPE      = "Support_Data"
    			//	CATDESC             = "Key for split ion distributions: eg '32' means record 2 of 3 records"
    			//	FIELDNAM            = "3D ion record key"
    			//	VALUE_TYPE          = INT
    			//	SIGNIFICANT_DIGITS  = 2
    			//	FILLVAL             = ???????
    			//	UNITS               = "unitless"
    			//	SI_CONVERSION       = "1>unitless"
    			//	DEPEND_0            = Time_tags__C3_CP_RAP_I3D_CNO_R
			//END_VARIABLE    = Rec_Key__C3_CP_RAP_I3D_CNO_R
			//
			// Impossible de transformer FILLVAL en float: continuer quand mme

			p=IDL_STRING_STR(sfillval+j);
			if (p[0]=='>') {
				tabopfillval[j] = '>';
				p++;
			} else {
				tabopfillval[j] = '=';
			}
			#ifdef USE_MACRO
			SSCANF_REEL(p,-1e30,reel,dep,code) ; if (code != 1) reel=-1e31;
			#else
			if (SSCANF_REEL (p,-1e30,&reel,&dep) != 1) reel=-1e31;
			#endif
		} else {
			tabopfillval[j] = '=';
			reel = -1e31;
		}
		tabfillval[j] = reel;
		//printf ("tabfillval[%d]=%f tabopfillval[%d]=\"%c\"\n",j,tabfillval[j],j,tabopfillval[j]);
	
		if (type[j]=='R' && memorise[j]==1) nbR++;	// 'R' = 'ISO_TIME_RANGE'
		if (type[j]=='D' && memorise[j]==1) nbD++;	// 'D' = 'ISO_TIME'
		if (type[j]=='C' && memorise[j]==1) nbC++;	// 'C' = 'CHAR'
		if (type[j]=='F' && memorise[j]==1) nbF++;	// 'F' = 'FLOAT'
		if (type[j]=='I' && memorise[j]==1) nbF++;	// 'I' = 'INT'
		if (type[j]=='S' && memorise[j]==1) nbS++;	// 'S' = 'STRING'
	}

	if ((fd = OPEN(isgz,IDL_STRING_STR(nom_IDL))) == NULL) goto ERROR;

	if (format==-1) {

		// Calcul de taille_enreg_fichier
		for (j=0 ; j<nb ; j++) {
			switch (type[j]) {
				case 'D':	// 'ISO_TIME'
						taille_enreg_fichier += 8;
						break;

				case 'F':	// 'FLOAT'
						taille_enreg_fichier += 4;
						break;
			}
		}
		//printf ("taille_enreg_fichier = %d\n",taille_enreg_fichier);

		if ((tmpchar = (char *) malloc (taille_enreg_fichier)) == NULL) goto ERROR;
		//printf ("nblignes0 = %d\n",nblignes0);

		for (i=0 ; i<nblignes0 ; i++) {
			if (GETS(isgz,fd,tmpligne,sizeof(tmpligne)) == NULL) goto ERROR;
			//printf ("i=%d %s",i,tmpligne);
		}



	} else {

		MY_GETS(isgz,fd,1,&len);

		for (i=0 ; i<nblignes0 ; i++) {
			if ((p=MY_GETS(isgz,fd,0,&len)) == NULL) goto ERROR;
		}

	}

	for (i=0 ; i<REC_START+REC_COUNT ; i++) {

		noligneERROR	= i+1;

		cte_DATE	= (long long)taille*(i-REC_START);
		cte_RANGE	= (long long)taille*(i-REC_START) + (nbD)	* sizeof(double);
		cte_STRING_CHAR	= (long long)taille*(i-REC_START) + (nbD+2*nbR)	* sizeof(double);
		cte_FLOAT_INT	= (long long)taille*(i-REC_START) + (nbD+2*nbR)	* sizeof(double) + (nbC+nbS)*sizeof_string;

		if (format == -1) {
			// Read a record
			// -------------
			if (READ(isgz,fd,tmpchar,taille_enreg_fichier) == 0) goto ERROR;
			p = tmpchar;
		} else {
			// Read a line
			// -----------
			if ((p=MY_GETS(isgz,fd,0,&len)) == NULL) goto ERROR;
			if (isxml && votable_debut) {
				for ( ; ; ) {
					p = strstr (p,"<TABLEDATA>");
					if (p==NULL) {
						if ((p=MY_GETS(isgz,fd,0,&len)) == NULL) goto ERROR;
					} else {
						p += strlen("<TABLEDATA>");
						votable_debut=0;
						break;
					}
				}
			}
		}

		sauter = 0;
		for (j=0 ; j<nb ; j++) {

			#ifdef DEBUG
			printf ("%c %s\n", type[j], p);
			#endif

			fillval = tabfillval[j];
			opfillval = tabopfillval[j];

			//printf ("j=%d fillval=%f opfillval=%c\n",j,fillval,opfillval);

			ok = (i>=REC_START) && (memorise[j]==1);

			if (format==-1) {
				
				switch (type[j]) {
					case 'D':	// 'ISO_TIME'

							tmpdouble = *(double *)p;
							p += 8;
							if (ok) {
								pdouble = (double *)(data + cte_DATE + position[j]*sizeof(double));
								*pdouble = tmpdouble;
							}
							break;

					case 'F':	// 'FLOAT'

							tmpfloat = *(float *)p;
							p += 4;
							if (ok) {
								pfloat = (float *)(data + cte_FLOAT_INT + position[j]*sizeof(float));
								// tmpfloat INDISPENSABLE sinon pas de reconnaissance exacte des fillvals ...
								if (opfillval == '=') {
									*pfloat = (modify_float_fillval && tmpfloat == fillval) ? -1e31 : tmpfloat;
								} else if (opfillval == '>') {
									*pfloat = (modify_float_fillval && tmpfloat > fillval) ? -1e31 : tmpfloat;
								}
							}
							break;

				}

				continue;
			}

			switch (type[j]) {


				case 'R':	// 'ISO_TIME_RANGE'

						SKIP_BLANK_TAB();
						SKIP_COMMA();
						sauter = 1;
						if (ok) pdouble = (double *)(data + cte_RANGE + (2*position[j]+0)*sizeof(double));
						READ_DATE_FORMAT_CEF();
						if (ok) *pdouble = tu_to_date (annee, mois, jour, heure, minute, seconde);
						p += dep; if (*p != '/') goto ERROR; p++;
						if (ok) pdouble = (double *)(data + cte_RANGE + (2*position[j]+1)*sizeof(double));
						READ_DATE_FORMAT_CEF();
						if (ok) *pdouble = tu_to_date (annee, mois, jour, heure, minute, seconde);
						break;


				case 'D':	// 'ISO_TIME'

						SKIP_BLANK_TAB();
						SKIP_COMMA();
						SKIP_SLASH()
						sauter = 1;
						if (ok) pdouble = (double *)(data + cte_DATE + position[j]*sizeof(double));
						switch (format) {
							case 3:
							case 5: 
								READ_DATE_FORMAT_CEF();
								if (ok) *pdouble = tu_to_date (annee, mois, jour, heure, minute, seconde);
								break;
							case 7:
								READ_DATE_FORMAT_7();
								if (ok) *pdouble = tu_to_date (annee, mois, jour, heure, minute, seconde);
								break;
							case 4:
								READ_DATE_FORMAT_4();
								if (ok) *pdouble = tu_to_date (annee, mois, jour, heure, minute, seconde);
								break;
							case 1:
								READ_DATE_FORMAT_1();
								if (ok) *pdouble = tu_to_date (annee, mois, jour, heure, minute, seconde);
								break;
							case 6:
								sauter = 0;
								if (ok) *pdouble = tu_to_date (2000, 1, 1, 0, 0, 0) + 1000. * i;
								dep = 0;
								break;
							case 8:
								READ_DATE_FORMAT_8();
								if (ok) *pdouble = tu_to_date (annee, 1, 1, heure, minute, seconde) + (doy-1)*86400000.0;
								break;
							case 10:
								READ_DATE_FORMAT_10();
								if (ok) *pdouble = tu_to_date (annee, mois, jour, heure, minute, seconde);
								break;
						}
						break;


				case 'S':	// 'STRING'

						if (ok) {
							READ_STRING();
						}
						break;


				case 'C':	// 'CHAR'

						SKIP_BLANK_TAB();
						SKIP_COMMA();
						sauter = 1;
						READ_CHAR(ok);
						break;


				case 'F':	// 'FLOAT'

						if (sauter) FIND_BLANK_TAB_COMMA();
						SKIP_BLANK_TAB();
						SKIP_COMMA();
						sauter = 1;
						READ_FLOAT();
						if (ok) {
							pfloat = (float *)(data + cte_FLOAT_INT + position[j]*sizeof(float));
							// tmpfloat INDISPENSABLE sinon pas de reconnaissance exacte des fillvals ...
							tmpfloat = reel;
							if (opfillval == '=') {
								*pfloat = (modify_float_fillval && tmpfloat == fillval) ? -1e31 : tmpfloat;
							} else if (opfillval == '>') {
								*pfloat = (modify_float_fillval && tmpfloat > fillval) ? -1e31 : tmpfloat;
							}
						}
						break;


				case 'I':	// 'INT'

						if (sauter) FIND_BLANK_TAB_COMMA();
						SKIP_BLANK_TAB();
						SKIP_COMMA();
						sauter = 1;
						READ_FLOAT();
						if (ok) {
							pint = (int *)(data + cte_FLOAT_INT + position[j]*sizeof(float));
							*pint = reel;
						}
						break;


			}
			p += dep;
		}
	}

	#ifdef DEBUG
	printf ("done...\n");
	#endif

	if (format==-1) free (tmpchar);

	if (fd != NULL) CLOSE(isgz,fd);
	if (tabfillval != NULL) free(tabfillval);
	if (tabopfillval != NULL) free(tabopfillval);
	#ifdef DEBUG
	printf ("finish...\n");
	#endif
	return 1;

ERROR:
	if (fd != NULL) CLOSE(isgz,fd);
	if (tabfillval != NULL) free(tabfillval);
	if (tabopfillval != NULL) free(tabopfillval);
	return -noligneERROR;

}
#endif

/*---------------------------------------------------------------------------*/
int READ_ASCII_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
/* Plante s'il a des STRING dans le fichier  lire */
	void	*p_nom_IDL,
	void	*p_nb,
	void	*p03,
	void	*p04,
	void	*p05,
	void	*p06,
	void	*p07,
	void	*p08,
	void	*p09,
	void	*p10,
	void	*p11,
	void	*p_sfillval,
	void	*p13,
	void	*p14,
	void	*p15) {

	int		argc=15;
	void		*argv[15];
	int		nb = * (int *) p_nb;
	int		i;
	int		code;
	IDL_STRING	*nom_IDL;
	IDL_STRING	*sfillval;

	// p_nom_IDL -> nom_IDL
	nom_IDL = malloc (sizeof(IDL_STRING));
	IDL_StrStore (nom_IDL, p_nom_IDL);

	// p_sfillval -> sfillval
	sfillval = malloc (nb * sizeof(IDL_STRING));
	for (i=0 ; i<nb ; i++) IDL_StrStore (sfillval+i, p_sfillval+i*1024);

	argv[ 0] = nom_IDL;
	argv[ 1] = p_nb;
	argv[ 2] = p03;
	argv[ 3] = p04;
	argv[ 4] = p05;
	argv[ 5] = p06;
	argv[ 6] = p07;
	argv[ 7] = p08;
	argv[ 8] = p09;
	argv[ 9] = p10;
	argv[10] = p11;
	argv[11] = sfillval;
	argv[12] = p13;
	argv[13] = p14;
	argv[14] = p15;
	code = READ_ASCII (argc, argv);

	free(nom_IDL);
	free(sfillval);

	return code;
}

/*---------------------------------------------------------------------------*/
int MY_READ_ASCII (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/
	IDL_STRING	*nom_IDL		= (IDL_STRING *)	argv[ 0];	/* nom du fichier  lire */
	IDL_STRING	*lignes			= (IDL_STRING *)	argv[ 1];	/* lignes lues dans le fichier */
	int		*nblignes		= (int *)		argv[ 2];	/* nombre de lignes lues dans fichier */

	FILE		*fd			=	NULL;
	char		*p;
	int		len;
	int		isgz;


	isgz = ISGZ(IDL_STRING_STR(nom_IDL));	// pour que isgz soit connu dans l'tiquette ERROR
	if ((fd = OPEN(isgz,IDL_STRING_STR(nom_IDL))) == NULL) goto ERROR;

	// initialisation de MY_GETS
	MY_GETS(isgz,fd,1,&len);

	for ( ; ; ) {

		if ((p=MY_GETS(isgz,fd,0,&len)) == NULL)
			break;

		if (p[0]==0) break; // sinon la dernire ligne est vide
		// Je pense que ca n'arriverait pas si j'utilisais wc-l

		IDL_StrStore (lignes+(*nblignes)++, p);

	}


ERROR:
	if (fd != NULL) CLOSE(isgz,fd);

	return 1;


}

/*---------------------------------------------------------------------------*/
int MY_READ_ASCII_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p_nom_IDL,
	void	*p_lignes,
	void	*p03,
	void	*p_nbrecords) {

	int		argc=3;
	void		*argv[3];
	int		nb = * (int *) p_nbrecords;
	int		i;
	int		code;
	IDL_STRING	*nom_IDL;
	IDL_STRING	*lignes;

	// p_nom_IDL -> nom_IDL
	nom_IDL = malloc (sizeof(IDL_STRING));
	IDL_StrStore (nom_IDL, p_nom_IDL);

	lignes = malloc (nb * sizeof(IDL_STRING));
	argv[0] = nom_IDL;
	argv[1] = lignes;
	argv[2] = p03;
	code = MY_READ_ASCII (argc, argv);

	// lignes -> p_lignes
	for (i=0 ; i<nb ; i++) sprintf (p_lignes+i*1024, "%s", IDL_STRING_STR(lignes+i));

	free (nom_IDL);
	free (lignes);

	return code;

}
