/*------------------------------------------------------------------------------
 *
 *	Fichier	: $RCSfile: interpoler2dxy.c,v $, v $Revision: 1.14 $
 *
 *	Date	: $Date: 2021/04/27 09:48:59 $
 *
 *	Auteur	: $Author: penou $
 *
 *	Version : %Z% version %I% de %M% du %G%
 *
 *------------------------------------------------------------------------------
 */


#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "infini.h"

/*---------------------------------------------------------------------------*/
int interpoler2d (
	double *tabx, double *tabxmin, double *tabxmax, int dimx,
	float *taby, float *tabymin, float *tabymax, int dimy,
	float *tabz,
	double xmin, double dx, int nbx,
	float ymin, float dy, int nby,
	float *image, int *nb, int *utilise, float infini,
	short *produitdata, float *produitimage) {
/*---------------------------------------------------------------------------*/

	/*
	Interpole les donnees (tabx, taby, tabz) suivant la grille (xmin, dx, nbx,
	ymin, dy, nby) en renvoyant les mises a jour de image et nb.

	Autour des points a l'infini, les donnees ne sont pas interpolees mais
	extrapolees en utilisant tabxmin, tabxmax, tabymin, tabymax.

	infini contient la valeur de Infini.

	Dimensions des tableaux:
		tabx[dimx]
		tabxmin[dimx]
		tabxmax[dimx]
		taby[dimy]
		tabymin[dimy]
		tabymax[dimy]
		tabz[dimx,dimy]
		image[nbx,nby]
		nb[nbx,nby]
	*/

	int i,j,k,x,y;
	double *tabxvrai,*tabyvrai,*tabt,*tabu,t,u,a,b,c,d,val;
	int *tabj,*tabk;

	/* Calcule tabxvrai: valeurs exactes de la grille en X */
	if ((tabxvrai = (double *)malloc(nbx*sizeof(double))) == NULL) return -1;
	for (i=0;i<nbx;i++) tabxvrai[i]=xmin+(i+0.5)*dx;

	/* Calcule tabyvrai: valeurs exactes de la grille en Y */
	if ((tabyvrai = (double *)malloc(nby*sizeof(double))) == NULL) return -1;
	for (i=0;i<nby;i++) tabyvrai[i]=ymin+(i+0.5)*dy;

	/* Calcule tabj: indices de tabx utilises pour interpoler tabxvrai */
	if ((tabj = (int *)malloc(nbx*sizeof(int))) == NULL) return -1;
	for (i=0;i<nbx;i++) {
		tabj[i]=-2;
		for (j=0;j<dimx;j++)
			if (tabx[j]>=tabxvrai[i]) {
				tabj[i]=j-1;
				break;
			}
	}

	/* Calcule tabk: indices de taby utilises pour interpoler tabyvrai */
	if ((tabk = (int *)malloc(nby*sizeof(int))) == NULL) return -1;
	for (i=0;i<nby;i++) {
		tabk[i]=-2;
		for (j=0;j<dimy;j++)
			if (taby[j]>=tabyvrai[i]) {
				tabk[i]=j-1;
				break;
			}
	}

	/* Calcule tabt: utilise pour l'interpolation */
	if ((tabt = (double *)malloc(nbx*sizeof(double))) == NULL) return -1;
	for (x=0;x<nbx;x++)
		if ((j=tabj[x])>=0)
			tabt[x]=(tabxvrai[x]-tabx[j])/(tabx[j+1]-tabx[j]);

	/* Calcule tabu: utilise pour l'interpolation */
	if ((tabu = (double *)malloc(nby*sizeof(double))) == NULL) return -1;
	for (y=0;y<nby;y++)
		if ((k=tabk[y])>=0)
			tabu[y]=(tabyvrai[y]-taby[k])/(taby[k+1]-taby[k]);


	/* Interpole chaque point de la grille */
	for (x=0;x<nbx;x++)
		if ((j=tabj[x])>=0)
			for (y=0;y<nby;y++)
				if ((k=tabk[y])>=0) {
					t=tabt[x];
					u=tabu[y];

					/* tabz[i,j] = tabz[i+dimx*j] */
					/* image[x,y] = image[x+nbx*y] */

					/* Valeurs des 4 points utilises pour interpoler */
					a = tabz[j     + dimx*k    ]; /* tabz[j,k] */
					b = tabz[(j+1) + dimx*k    ]; /* tabz[j+1,k] */
					c = tabz[(j+1) + dimx*(k+1)]; /* tabz[j+1,k+1] */
					d = tabz[j     + dimx*(k+1)]; /* tabz[j,k+1] */

					/* Si une des 4 valeurs est infini */
					if ((nonfini(a,infini) || nonfini(b,infini) || nonfini(c,infini) || nonfini(d,infini))) {
						if (tabyvrai[y]<=tabymax[k]) {
							if (tabxvrai[x]<=tabxmax[j])
								val = a;
							else if (tabxvrai[x]>=tabxmin[j+1])
								val = b;
 							else
								val = infini;
						} else if (tabyvrai[y]>=tabymin[k+1]) {
							if (tabxvrai[x]<=tabxmax[j])
								val = d;
							else if (tabxvrai[x]>=tabxmin[j+1])
								val = c;
							else
								val = infini;
						} else 
							val = infini;
					} else 
						val =	(1-t) * (1-u) * a + 
							t     * (1-u) * b +
							t     * u     * c + 
							(1-t) * u     * d;

					if (nonfini(produitimage[x],infini))
						if (produitdata[j]==produitdata[j+1])
							produitimage[x]=produitdata[j];

					if (fini(val,infini)) {
						utilise[x]=1;
						nb[x+nbx*y]++;
						if (nonfini(image[x+nbx*y],infini))
							image[x+nbx*y] = val; /* Premier acces */
						else
							image[x+nbx*y] += val;
					}
				}

	free(tabxvrai);
	free(tabyvrai);
	free(tabj);
	free(tabk);
	free(tabt);
	free(tabu);

	return 0;

}

/*---------------------------------------------------------------------------*/
int interpoler2dxybloc (
	double *Xmin_data, double *Xmoy_data, double *Xmax_data,
	int nbx_xdata, int deb, int fin,
	float *Ymin_data, float *Ymoy_data, float *Ymax_data, 
	int nbx_ydata, int nby_ydata, int *indY,
	float *Zdata,
	double xmin, double dx, int nbx,
	float ymin, float dy, int nby,
	float *image, int *nb, int *utilise, float infini, 
	short *produitdata, float *produitimage) {
/*---------------------------------------------------------------------------*/

	/*
	Les donnees en Y???_data[deb:fin,*] peuvent contenir des valeurs Inf. Soit
	nby1_data tel que Y???_data[deb:fin,0:nby1_data-1] ne contiene que des 
	valeurs finies.

	- utilisation des donnees avec deb<=x<=fin et 0<=y<=nby1_data-1
	- rajout des points en x=Xmin_data[deb] et x=Xmax_data[fin]
	- rajout des points en y=Ymin_data[0] et y=Ymax_data[nby1_data-1]
	- interpole ces donnees en renvoyant les mises a jour de image et nb

	infini contient la valeur de Infini.

	Dimensions des tableaux: 
		Xmin_data[nbx_xdata]
		Xmoy_data[nbx_xdata]
		Xmax[nbx_xdata]
		Ymin_data[nbx_ydata,nby_ydata]
		Ymoy_data[nbx_ydata,nby_ydata]
		Ymax_data[nbx_ydata,nby_ydata]
		indY[nbx_xdata]
		image[nbx,nby]
		nb[nbx,nby]
	*/

	int i,j,dimx, dimy,nby1_data;
	double *tabx, *tabxmin, *tabxmax;
	float *taby, *tabymin, *tabymax;
	float *tabz;
	short *tabproduitdata;

	/* Creation de tableaux contenant 'x' et '#'

	*	                0     1          deb         fin        nbx_xdata-1
	*	             ------------------#--#-----#-----#--#------------
	*	           0 |     |     |     #  x  |  x  |  x  #     |     |
	*	             -------------------------------------------------
	*	           1 |     |     |     #  x  |  x  |  x  #     |     |
	*	             -------------------------------------------------
	*	             |     |     |     #  x  |  x  |  x  #     |     |
	*	             -------------------------------------------------
	*	             |     |     |     #  x  |  x  |  x  #     |     |
	*	             -------------------------------------------------
	*	             |     |     |     #  x  |  x  |  x  #     |     |
	*	             -------------------------------------------------
	*	nby1_data-1  |     |     |     #  x  |  x  |  x  #     |     |
	*	             ------------------#--#-----#-----#--#------------


	*	                            0  1              1+fin-deb+1
	*	                    0       #--#-----#-----#--#
	*	                    1       #  x  |  x  |  x  #
	*	                            -------------------
	*	                    2       #  x  |  x  |  x  #
	*	                            -------------------
	*	                            #  x  |  x  |  x  #
	*	                            -------------------
	*	                            #  x  |  x  |  x  #
	*	                            -------------------
	*	                            #  x  |  x  |  x  #
	*	                            -------------------
	*	            nby1_data       #  x  |  x  |  x  #
	*	            nby1_data+1     #--#-----#-----#--#

	*/

	dimx = 1+fin-deb+2;
	for (nby1_data=0;nby1_data<nby_ydata;nby1_data++)
		if (nonfini(Ymoy_data[indY[deb]+nbx_ydata*nby1_data],infini)) break;
	dimy = nby1_data+2;

	/* Creation des tableaux tabymin, taby, tabymax */
	if ((tabymin = (float *)malloc(dimy*sizeof(float))) == NULL) return -1;
	if ((taby    = (float *)malloc(dimy*sizeof(float))) == NULL) return -1;
	if ((tabymax = (float *)malloc(dimy*sizeof(float))) == NULL) return -1;

	for (j=1;j<=nby1_data;j++) {
		tabymin[j] = Ymin_data[indY[deb]+nbx_ydata*(j-1)];
		taby[j]    = Ymoy_data[indY[deb]+nbx_ydata*(j-1)];
		tabymax[j] = Ymax_data[indY[deb]+nbx_ydata*(j-1)];
		}

	tabymin[0] = tabymin[1];
	taby[0]    = tabymin[1];
	tabymax[0] = tabymax[1];

	tabymin[nby1_data+1] = tabymin[nby1_data];
	taby[nby1_data+1]    = tabymax[nby1_data];
	tabymax[nby1_data+1] = tabymax[nby1_data];

	/* Creation des tableaux tabxmin, tabx, tabxmax */
	if ((tabxmin = (double *)malloc(dimx*sizeof(double))) == NULL) return -1;
	if ((tabx    = (double *)malloc(dimx*sizeof(double))) == NULL) return -1;
	if ((tabxmax = (double *)malloc(dimx*sizeof(double))) == NULL) return -1;

	for (i=1;i<=1+fin-deb;i++) {
		tabxmin[i] = Xmin_data[deb+i-1];
		tabx[i]    = Xmoy_data[deb+i-1];
		tabxmax[i] = Xmax_data[deb+i-1];
	}

	tabxmin[0] = tabxmin[1];
	tabx[0]    = tabxmin[1];
	tabxmax[0] = tabxmax[1];

	tabxmin[1+fin-deb+1] = tabxmin[1+fin-deb];
	tabx[1+fin-deb+1]    = tabxmax[1+fin-deb];
	tabxmax[1+fin-deb+1] = tabxmax[1+fin-deb];

	/* Creation du tableau tabz */
	if ((tabz = (float *)malloc(dimx*dimy*sizeof(float))) == NULL) return -1;

	for (i=1;i<=dimx-2;i++)
		for (j=1;j<=dimy-2;j++)
			tabz[i+dimx*j] = Zdata[deb+(i-1)+nbx_xdata*(j-1)];

	for (i=1;i<=dimx-2;i++) {
		tabz[i+dimx*0]        = tabz[i+dimx*1];
		tabz[i+dimx*(dimy-1)] = tabz[i+dimx*(dimy-2)];
	}

	for (j=0;j<dimy;j++) {
		tabz[0+dimx*j]      = tabz[1+dimx*j];
		tabz[dimx-1+dimx*j] = tabz[dimx-2+dimx*j];
	}

	/* Creation du tableau tabproduitdata */
	if ((tabproduitdata = (short *)malloc(dimx*sizeof(float))) == NULL) return -1;
	for (i=1;i<=dimx-2;i++)
		tabproduitdata[i] = produitdata[deb+(i-1)];
	tabproduitdata[0] = tabproduitdata[1];
	tabproduitdata[dimx-1] = tabproduitdata[dimx-2];

	/* Interpole */
	if (interpoler2d (tabx,tabxmin,tabxmax,dimx,
				taby,tabymin,tabymax,dimy,
				tabz,
				xmin,dx,nbx,
				ymin,dy,nby,
				image,nb,utilise,infini,tabproduitdata,produitimage)==-1)
		return -1;

	free (tabx);
	free (tabxmin);
	free (tabxmax);
	free (taby);
	free (tabymin);
	free (tabymax);
	free (tabz);
	free (tabproduitdata);

	return 0;

}

/*---------------------------------------------------------------------------*/
int chg_tables_trou (
	double *Xmin_data, double *Xmax_data, int nbx_xdata, double dttrou, 
	int *indY, int no) {
/*---------------------------------------------------------------------------*/

	/*
	Cherche un bloc de donnees commencant a l'indice no sans trous de donnees en X
	ni changement de tables en Y.

	Dimensions des tableaux:
		Xmin_data[nbx_xdata]
		Xmax_data[nbx_xdata]
		Ymin_data[nbx_xdata,nby_data]
		Ymoy_data[nbx_xdata,nby_data]
		Ymax_data[nbx_xdata,nby_data]
	*/

	int i;

	for (i=no;i<nbx_xdata;i++) {
		if ((i==nbx_xdata-1) || (Xmax_data[i]+dttrou < Xmin_data[i+1]))
			return i;
		if (indY[i]!=indY[no])
			return i;
	}

	return -1;

}

/*---------------------------------------------------------------------------*/
int INTERPOLER2DXY (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	/*
	Les donnes  interpols sont definies par:
		- X mini (Xmin_data), moyenne (Xmoy_data), maxi (Xmax_data)
		- Y mini (Ymin_data), moyenne (Ymoy_data), maxi (Ymax_data)
		- Z (Zdata)
		- dimension des tableaux en X: [nbx_xdata]
		- dimension des tableaux en Y et Z: [nbx_xdata,nby_data]

	La grille est definie par:
		- nbx points commencant a dx par pas de dx
		- nby points commencant a dy par pas de dy

	L'image est cr dans image[nbx_xdata,nby_data]

	Les donnes sont interpoles par blocs sans trous de donnes (dttrou)
	et sans changement de tables d'energies.
	*/

	double	*Xmin_data	= (double *)	argv[0];
	double	*Xmoy_data	= (double *)	argv[1];
	double	*Xmax_data	= (double *)	argv[2];
	int	nbx_xdata	= *(int *)	argv[3];
	double	dttrou		= *(double *)	argv[4];
	float	*Ymin_data	= (float *)	argv[5];
	float	*Ymoy_data	= (float *)	argv[6];
	float	*Ymax_data	= (float *)	argv[7];
	int	nbx_ydata	= *(int *)	argv[8];
	int	nby_ydata	= *(int *)	argv[9];
	int	*indY		= (int *)	argv[10];
	float	*Zdata		= (float *)	argv[11];
	double	xmin		= *(double *)	argv[12];
	double	dx		= *(double *)	argv[13];
	int	nbx		= *(int *)	argv[14];
	float	ymin		= *(float *)	argv[15];
	float	dy		= *(float *)	argv[16];
	int	nby		= *(int *)	argv[17];
	float	*image		= (float *)	argv[18];
	int	*nb		= (int *)	argv[19];
	int	*utilise	= (int *)	argv[20];
	float	infini		= *(float *)	argv[21];
	short	*produitdata	= (short *)	argv[22];
	float	*produitimage	= (float *)	argv[23];

	int fin,deb;

	for (deb=0;deb<nbx_xdata;) {

		if (indY[deb]==-1) { /* Table d'energie inconnue: on ne peut rien faire */
			deb++;
			continue;
		}

		fin = chg_tables_trou (Xmin_data, Xmax_data, nbx_xdata, dttrou, indY, deb);
	
		if (interpoler2dxybloc (Xmin_data, Xmoy_data, Xmax_data, nbx_xdata, 
						deb, fin,
						Ymin_data, Ymoy_data, Ymax_data, nbx_ydata, nby_ydata, indY, 
						Zdata,
						xmin, dx, nbx,
						ymin, dy, nby,
						image,nb,utilise,infini,produitdata,produitimage)==-1)
			return -1;
		deb = fin+1;
	}

	return 0;

}

/*---------------------------------------------------------------------------*/
int INTERPOLER2DXY_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void *p01,
	void *p02,
	void *p03,
	void *p04,
	void *p05,
	void *p06,
	void *p07,
	void *p08,
	void *p09,
	void *p10,
	void *p11,
	void *p12,
	void *p13,
	void *p14,
	void *p15,
	void *p16,
	void *p17,
	void *p18,
	void *p19,
	void *p20,
	void *p21,
	void *p22,
	void *p23,
	void *p24) {

	int	argc=24;
	void	*argv[24];

	argv[ 0] = p01;
	argv[ 1] = p02;
	argv[ 2] = p03;
	argv[ 3] = p04;
	argv[ 4] = p05;
	argv[ 5] = p06;
	argv[ 6] = p07;
	argv[ 7] = p08;
	argv[ 8] = p09;
	argv[ 9] = p10;
	argv[10] = p11;
	argv[11] = p12;
	argv[12] = p13;
	argv[13] = p14;
	argv[14] = p15;
	argv[15] = p16;
	argv[16] = p17;
	argv[17] = p18;
	argv[18] = p19;
	argv[19] = p20;
	argv[20] = p21;
	argv[21] = p22;
	argv[22] = p23;
	argv[23] = p24;
	return INTERPOLER2DXY (argc, argv);

}
