/*------------------------------------------------------------------------------
 *
 *	Fichier	: $RCSfile: filecdf.c,v $, v $Revision: 1.24 $
 *
 *	Date	: $Date: 2021/04/27 09:48:59 $
 *
 *	Auteur	: $Author: penou $
 *
 *	Version : %Z% version %I% de %M% du %G%
 *
 *------------------------------------------------------------------------------
 */

#include <unistd.h>

#include <stdio.h>
#include <stdlib.h>
#include <glob.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef GDL
#include "libgdl.h"
#else
#include "export.h"
#endif

#include "cdf.h"

#include <sys/time.h>

#define indice2(i1,i2,nb1,nb2)	((i1)+(nb1)*(i2))

CDFid	fd;	/* CDF identifier. */

/*---------------------------------------------------------------------------*/
double nbsec (void) {
/*---------------------------------------------------------------------------*/

	struct timeval tv;
	struct timezone tz;
	gettimeofday (&tv,&tz);
	return tv.tv_sec+tv.tv_usec/1000000.;

}


/*---------------------------------------------------------------------------*/
float CDFtofloat (unsigned char *p, long datatype) {
/*---------------------------------------------------------------------------*/

	switch (datatype) {

		case CDF_UINT1 :	return * (unsigned char *)	p;
		case CDF_UINT2 :	return * (unsigned short *)	p;
		case CDF_UINT4 :	return * (unsigned int *)	p;
		case CDF_BYTE:
		case CDF_INT1 :		return * (char *)		p;
		case CDF_INT2 :		return * (short *)		p;
		case CDF_INT4 :		return * (int *)		p;
		case CDF_INT8 :		return * (long long *)		p;
		case CDF_REAL4 :
		case CDF_FLOAT :	return * (float *)		p;
		case CDF_REAL8 :
		case CDF_DOUBLE :
		case CDF_EPOCH :	return * (double *)		p;
		case CDF_EPOCH16 :	return 1e3**(double *)p + 1e-9**(double *)(p+8);
		case CDF_TIME_TT2000 :	return 1e3*1e-9**(long long *)p;

	}

	return -1e31;

}


/*---------------------------------------------------------------------------*/
double CDFtodouble (unsigned char *p, long datatype) {
/*---------------------------------------------------------------------------*/

	switch (datatype) {

		case CDF_UINT1 :	return * (unsigned char *)	p;
		case CDF_UINT2 :	return * (unsigned short *)	p;
		case CDF_UINT4 :	return * (unsigned int *)	p;
		case CDF_BYTE:
		case CDF_INT1 :		return * (char *)		p;
		case CDF_INT2 :		return * (short *)		p;
		case CDF_INT4 :		return * (int *)		p;
		case CDF_INT8 :		return * (long long *)		p;
		case CDF_REAL4 :
		case CDF_FLOAT :	return * (float *)		p;
		case CDF_REAL8 :
		case CDF_DOUBLE :
		case CDF_EPOCH :	return * (double *)		p;
		case CDF_EPOCH16 :	return 1e3**(double *)p + 1e-9**(double *)(p+8);
		case CDF_TIME_TT2000 :	return 1e3*1e-9**(long long *)p;

	}

	return -1e31;

}


/*---------------------------------------------------------------------------*/
short CDFtoshort (unsigned char *p, long datatype) {
/*---------------------------------------------------------------------------*/

	switch (datatype) {

		case CDF_UINT1 :	return * (unsigned char *)	p;
		case CDF_UINT2 :	return * (unsigned short *)	p;
		case CDF_UINT4 :	return * (unsigned int *)	p;
		case CDF_BYTE:
		case CDF_INT1 :		return * (char *)		p;
		case CDF_INT2 :		return * (short *)		p;
		case CDF_INT4 :		return * (int *)		p;
		case CDF_INT8 :		return * (long long *)		p;
		case CDF_REAL4 :
		case CDF_FLOAT :	return * (float *)		p;
		case CDF_REAL8 :
		case CDF_DOUBLE :
		case CDF_EPOCH :	return * (double *)		p;
		case CDF_EPOCH16 :	return 1e3**(double *)p + 1e-9**(double *)(p+8);
		case CDF_TIME_TT2000 :	return 1e3*1e-9**(long long *)p;

	}

	return 65535;

}


/*---------------------------------------------------------------------------*/
unsigned CDFtochar (unsigned char *p, long datatype) {
/*---------------------------------------------------------------------------*/

	switch (datatype) {

		case CDF_UINT1 :	return * (unsigned char *)	p;
		case CDF_UINT2 :	return * (unsigned short *)	p;
		case CDF_UINT4 :	return * (unsigned int *)	p;
		case CDF_BYTE:
		case CDF_INT1 :		return * (char *)		p;
		case CDF_INT2 :		return * (short *)		p;
		case CDF_INT4 :		return * (int *)		p;
		case CDF_INT8 :		return * (long long *)		p;
		case CDF_REAL4 :
		case CDF_FLOAT :	return * (float *)		p;
		case CDF_REAL8 :
		case CDF_DOUBLE :
		case CDF_EPOCH :	return * (double *)		p;
		case CDF_EPOCH16 :	return 1e3**(double *)p + 1e-9**(double *)(p+8);
		case CDF_TIME_TT2000 :	return 1e3*1e-9**(long long *)p;

	}

	return 255;

}


/*---------------------------------------------------------------------------*/
int CDFtoint (unsigned char *p, long datatype) {
/*---------------------------------------------------------------------------*/

	switch (datatype) {

		case CDF_UINT1 :	return * (unsigned char *)	p;
		case CDF_UINT2 :	return * (unsigned short *)	p;
		case CDF_UINT4 :	return * (unsigned int *)	p;
		case CDF_BYTE:
		case CDF_INT1 :		return * (char *)		p;
		case CDF_INT2 :		return * (short *)		p;
		case CDF_INT4 :		return * (int *)		p;
		case CDF_INT8 :		return * (long long *)		p;
		case CDF_REAL4 :
		case CDF_FLOAT :	return * (float *)		p;
		case CDF_REAL8 :
		case CDF_DOUBLE :
		case CDF_EPOCH :	return * (double *)		p;
		case CDF_EPOCH16 :	return 1e3**(double *)p + 1e-9**(double *)(p+8);
		case CDF_TIME_TT2000 :	return 1e3*1e-9**(long long *)p;

	}

	return 65535;

}


/*---------------------------------------------------------------------------*/
long long int CDFtolonglongint (unsigned char *p, long datatype) {
/*---------------------------------------------------------------------------*/

	switch (datatype) {

		case CDF_UINT1 :	return * (unsigned char *)	p;
		case CDF_UINT2 :	return * (unsigned short *)	p;
		case CDF_UINT4 :	return * (unsigned int *)	p;
		case CDF_BYTE:
		case CDF_INT1 :		return * (char *)		p;
		case CDF_INT2 :		return * (short *)		p;
		case CDF_INT4 :		return * (int *)		p;
		case CDF_INT8 :		return * (long long *)		p;
		case CDF_REAL4 :
		case CDF_FLOAT :	return * (float *)		p;
		case CDF_REAL8 :
		case CDF_DOUBLE :
		case CDF_EPOCH :	return * (double *)		p;
		case CDF_EPOCH16 :	return 1e3**(double *)p + 1e-9**(double *)(p+8);
		case CDF_TIME_TT2000 :	return 1e3*1e-9**(long long *)p;

	}

	return 65535;

}


/*---------------------------------------------------------------------------*/
float CDFtofloat_FILLVAL (unsigned char *p, long datatype, int have_FILLVAL, float FILLVAL) {
/*---------------------------------------------------------------------------*/

	float	v;

	v = CDFtofloat (p, datatype);

	return (have_FILLVAL && (v==FILLVAL)) ? -1e31 : v;

}


/*---------------------------------------------------------------------------*/
void CDFtostring (unsigned char *p, long datatype, long nb, char *pstring) {
/*---------------------------------------------------------------------------*/

	int	i;

	for (i=0 ; i<nb ; i++) {
		pstring[i] = p[i];
	}
	pstring[i]=0;

}


/*---------------------------------------------------------------------------*/
void incrementer (long nbdim, long *dimSizes, long *icdf, long majority) {
/*---------------------------------------------------------------------------*/

	int	i;

	if (majority == ROW_MAJOR) {

		/* La dernire dimension varie la plus vite (ordre C) */
		for (i=nbdim-1 ; i>=0 ; i--) {

			icdf[i]++;
			if (icdf[i] != dimSizes[i]) return;
			icdf[i]=0;
		}

	} else {

		/* La premire dimension varie la plus vite (ordre FORTRAN) */
		for (i=0 ; i<nbdim ; i++) {

			icdf[i]++;
			if (icdf[i] != dimSizes[i]) return;
			icdf[i]=0;
		}

	}
}

/*---------------------------------------------------------------------------*/
void MY_CDF_INQUIRE (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	int	nbarg		= 0;
	int	*ndims		= (int *) 	argv[nbarg++];	// Sorties
	int	*nvars		= (int *) 	argv[nbarg++];	// Sorties
	int	*nzvars		= (int *) 	argv[nbarg++];	// Sorties
	int	*natts		= (int *) 	argv[nbarg++];	// Sorties
	int	*dim		= (int *) 	argv[nbarg++];	// Sorties

	long 	numDims;
	long	dimSizes[CDF_MAX_DIMS];
	long	encoding;
	long 	majority;
	long	maxrRec;
	long	numrVars;
	long	maxzRec;
	long	numzVars;
	long 	numAttrs;
	int	i;

	if (CDFinquireCDF (fd, &numDims, dimSizes, &encoding, &majority, &maxrRec, &numrVars, &maxzRec, &numzVars, &numAttrs) >= CDF_WARN) {
		*nvars = numrVars;
		*nzvars = numzVars;
		*natts = numAttrs;
		*ndims = numDims;
		for (i=0 ; i<numDims ; i++) {
			dim[i] = dimSizes[i];
		}
	}

}

/*---------------------------------------------------------------------------*/
void MY_CDF_INQUIRE_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p01,
	void	*p02,
	void	*p03,
	void	*p04,
	void	*p05) {

	int	argc=5;
	void	*argv[5];

	argv[0] = p01;
	argv[1] = p02;
	argv[2] = p03;
	argv[3] = p04;
	argv[4] = p05;
	MY_CDF_INQUIRE (argc, argv);

}

/*---------------------------------------------------------------------------*/
void MY_CDF_ATTINQ (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	int		nbarg		= 0;
	int		num		= *(int *) 		argv[nbarg++];	// Entres
	IDL_STRING	*name		= (IDL_STRING *) 	argv[nbarg++];	// Sorties
	IDL_STRING	*scope		= (IDL_STRING *) 	argv[nbarg++];	// Sorties
	int		*maxgentry	= (int *) 		argv[nbarg++];	// Sorties
	int		*maxrentry	= (int *) 		argv[nbarg++];	// Sorties
	int		*maxzentry	= (int *) 		argv[nbarg++];	// Sorties


	char	attrName[CDF_ATTR_NAME_LEN256+1];
	long	attrScope;
	long	maxgEntry;
	long	maxrEntry;
	long	maxzEntry;

	if (CDFinquireAttr (fd, (long)num, attrName, &attrScope, &maxgEntry, &maxrEntry, &maxzEntry) >=  CDF_OK) {
		IDL_StrStore (name, attrName);
		if (attrScope == GLOBAL_SCOPE) {
			IDL_StrStore (scope, "GLOBAL_SCOPE");
		} else if (attrScope == VARIABLE_SCOPE) {
			IDL_StrStore (scope, "VARIABLE_SCOPE");
		}
		*maxgentry = maxgEntry;
		*maxrentry = maxrEntry;
		*maxzentry = maxzEntry;
	}
}

/*---------------------------------------------------------------------------*/
void MY_CDF_ATTINQ_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p01,
	void	*p_name,
	void	*p_scope,
	void	*p04,
	void	*p05,
	void	*p06) {

	int		argc=6;
	void		*argv[6];
	IDL_STRING	*name;
	IDL_STRING	*scope;

	name = malloc (sizeof(IDL_STRING));
	scope = malloc (sizeof(IDL_STRING));

	argv[0] = p01;
	argv[1] = name;
	argv[2] = scope;
	argv[3] = p04;
	argv[4] = p05;
	argv[5] = p06;
	MY_CDF_ATTINQ (argc, argv);

	// name -> p_name
	strcpy (p_name, IDL_STRING_STR(name));

	// scope -> p_scope
	strcpy (p_scope, IDL_STRING_STR(scope));

	free (name);
	free (scope);

}

/*---------------------------------------------------------------------------*/
void MY_CDF_VARINQ (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	int		nbarg		= 0;
	int		novar		= *(int *) 		argv[nbarg++];	// Entres
	int		isz		= *(int *) 		argv[nbarg++];	// Entres
	IDL_STRING	*name		= (IDL_STRING *) 	argv[nbarg++];	// Sorties
	IDL_STRING	*type		= (IDL_STRING *) 	argv[nbarg++];	// Sorties
	IDL_STRING	*recvary	= (IDL_STRING *) 	argv[nbarg++];	// Sorties
	int		*numelem	= (int *) 		argv[nbarg++];	// Sorties
	int		ndims		= *(int *) 		argv[nbarg++];	// Entres
	char		*dimvar		= (char *) 		argv[nbarg++];	// Sorties dimvar[ndims]

	char	nom[CDF_VAR_NAME_LEN256+1];
	long	datatype;
	long	nombre_elements;
	long	nbdim;
	long	dimSizes[CDF_MAX_DIMS];
	long	recVary;
	long	dimVarys[CDF_MAX_DIMS];
	int	ok = 0;
	int	i;

	if (isz==0) {
		if (CDFvarInquire (fd, novar, nom, &datatype, &nombre_elements, &recVary, dimVarys) >= CDF_WARN) {
			ok = 1;
		}
	} else {
		if (CDFinquirezVar (fd, novar, nom, &datatype, &nombre_elements, &nbdim, dimSizes, &recVary, dimVarys) >= CDF_WARN) {
			ok = 1;
		}
	}

	//printf ("MY_CDF_VARINQ: isz=%d novar=%d ok=%d\n", isz, novar, ok);

	if (ok) {
		IDL_StrStore (name, nom);

		switch (datatype) {

			case CDF_UINT1 :	IDL_StrStore (type, "CDF_UINT1");	break;
			case CDF_UINT2 :	IDL_StrStore (type, "CDF_UINT2");	break;
			case CDF_UINT4 :	IDL_StrStore (type, "CDF_UINT4");	break;
			case CDF_BYTE :		IDL_StrStore (type, "CDF_BYTE");	break;
			case CDF_INT1 :		IDL_StrStore (type, "CDF_INT1");	break;
			case CDF_INT2 :		IDL_StrStore (type, "CDF_INT2");	break;
			case CDF_INT4 :		IDL_StrStore (type, "CDF_INT4");	break;
			case CDF_INT8 :		IDL_StrStore (type, "CDF_INT8");	break;
			case CDF_REAL4 :	IDL_StrStore (type, "CDF_REAL4");	break;
			case CDF_FLOAT :	IDL_StrStore (type, "CDF_FLOAT");	break;
			case CDF_REAL8 :	IDL_StrStore (type, "CDF_REAL8");	break;
			case CDF_DOUBLE :	IDL_StrStore (type, "CDF_DOUBLE");	break;
			case CDF_EPOCH :	IDL_StrStore (type, "CDF_EPOCH");	break;
			case CDF_EPOCH16 :	IDL_StrStore (type, "CDF_EPOCH16");	break;
			case CDF_TIME_TT2000 :	IDL_StrStore (type, "CDF_TIME_TT2000");	break;
			case CDF_CHAR :		IDL_StrStore (type, "CDF_CHAR");	break;
			case CDF_UCHAR :	IDL_StrStore (type, "CDF_UCHAR");	break;
		}

		IDL_StrStore (recvary, recVary==VARY ? "VARY" : "NOVARY");

		*numelem = nombre_elements;

		for (i=0 ; i<ndims ; i++) {
			dimvar[i] = dimVarys[i];
			if (dimvar[i] == -1) dimvar[i]=1;
		}
	}

}

/*---------------------------------------------------------------------------*/
void MY_CDF_VARINQ_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p01,
	void	*p02,
	void	*p_name,
	void	*p_type,
	void	*p_recvary,
	void	*p06,
	void	*p07,
	void	*p08) {

	int		argc=8;
	void		*argv[8];
	IDL_STRING	*name;
	IDL_STRING	*type;
	IDL_STRING	*recvary;

	name = malloc (sizeof(IDL_STRING));
	type = malloc (sizeof(IDL_STRING));
	recvary = malloc (sizeof(IDL_STRING));

	argv[0] = p01;
	argv[1] = p02;
	argv[2] = name;
	argv[3] = type;
	argv[4] = recvary;
	argv[5] = p06;
	argv[6] = p07;
	argv[7] = p08;
	MY_CDF_VARINQ (argc, argv);

	// name -> p_name
	strcpy (p_name, IDL_STRING_STR(name));

	// type -> p_type
	strcpy (p_type, IDL_STRING_STR(type));

	// recvary -> p_recvary
	strcpy (p_recvary, IDL_STRING_STR(recvary));

	free (name);
	free (type);
	free (recvary);

}

/*---------------------------------------------------------------------------*/
int MY_CDF_ATTGET (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	int		nbarg		= 0;
	int		novar		= *(int *) 		argv[nbarg++];	// Entres
	int		isz		= *(int *) 		argv[nbarg++];	// Entres
	IDL_STRING	*attname	= (IDL_STRING *) 	argv[nbarg++];	// Entres
	IDL_STRING	*valeur		= (IDL_STRING *) 	argv[nbarg++];	// Sorties

	long		datatype;
	long		numElems;
	int		ok;
	char		buffer[1024];
	long		noatt;
	char		pstring[1024];
	int		i;
	void		*p;

	for (i=0 ; i<1024 ; i++) {
		buffer[i] = 0 ; //  cause des CHAR/11
	}

	ok = 0;
	if ((noatt = CDFgetAttrNum (fd, IDL_STRING_STR(attname))) >= CDF_OK) {
		if (isz == 0) {
			if (CDFattrEntryInquire (fd, noatt, novar, &datatype, &numElems) >= CDF_WARN) {
				ok = CDFgetAttrrEntry (fd, noatt, novar, buffer) >= CDF_WARN;
			}
		} else {
			if (CDFinquireAttrzEntry (fd, noatt, novar, &datatype, &numElems) >= CDF_WARN) {
				ok = CDFgetAttrzEntry (fd, noatt, novar, buffer) >= CDF_WARN;
			}
		}
	}

	//printf ("MY_CDF_ATTGET novar=%d isz=%d attname=%s ok=%d\n", novar, isz, IDL_STRING_STR(attname), ok);

	if (ok) {

		p = buffer;
		switch (datatype) {

			case CDF_UINT1 :	sprintf (pstring, "%d", * (unsigned char *) p);		IDL_StrStore (valeur, pstring);		break;
			case CDF_UINT2 :	sprintf (pstring, "%d", * (unsigned short *) p);	IDL_StrStore (valeur, pstring);		break;
			case CDF_UINT4 :	sprintf (pstring, "%d", * (unsigned int *) p);		IDL_StrStore (valeur, pstring);		break;
			case CDF_BYTE :		sprintf (pstring, "%d", * (char *) p);			IDL_StrStore (valeur, pstring);		break;
			case CDF_INT1 :		sprintf (pstring, "%d", * (char *) p);			IDL_StrStore (valeur, pstring);		break;
			case CDF_INT2 :		sprintf (pstring, "%d", * (short *) p);			IDL_StrStore (valeur, pstring);		break;
			case CDF_INT4 :		sprintf (pstring, "%d", * (int *) p);			IDL_StrStore (valeur, pstring);		break;
			#if defined (__WIN32__) || defined (__WIN64__)
			case CDF_INT8 :		sprintf (pstring, "%I64d", * (long long *) p);		IDL_StrStore (valeur, pstring);		break;
			#else
			case CDF_INT8 :		sprintf (pstring, "%lld", * (long long *) p);		IDL_StrStore (valeur, pstring);		break;
			#endif
			case CDF_REAL4 :	sprintf (pstring, "%e", * (float *) p);			IDL_StrStore (valeur, pstring);		break;
			case CDF_FLOAT :	sprintf (pstring, "%e", * (float *) p);			IDL_StrStore (valeur, pstring);		break;
			case CDF_REAL8 :	sprintf (pstring, "%e", * (double *) p);		IDL_StrStore (valeur, pstring);		break;
			case CDF_DOUBLE :	sprintf (pstring, "%e", * (double *) p);		IDL_StrStore (valeur, pstring);		break;
			case CDF_EPOCH :	sprintf (pstring, "%s", "CDF_EPOCH...");		IDL_StrStore (valeur, pstring);		break;
			case CDF_EPOCH16 :	sprintf (pstring, "%s", "CDF_EPOCH16...");		IDL_StrStore (valeur, pstring);		break;
			case CDF_TIME_TT2000 :	sprintf (pstring, "%s", "CDF_TIME_TT2000...");		IDL_StrStore (valeur, pstring);         break;
			case CDF_CHAR :		sprintf (pstring, "%s", buffer);			IDL_StrStore (valeur, pstring);		break;
			case CDF_UCHAR :	sprintf (pstring, "%s", buffer);			IDL_StrStore (valeur, pstring);		break;
		}
	}

	return ok;

}

/*---------------------------------------------------------------------------*/
void MY_CDF_ATTGET_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p01,
	void	*p02,
	void	*p_attname,
	void	*p_valeur) {

	int		argc=4;
	void		*argv[4];
	int		code;
	IDL_STRING	*attname;
	IDL_STRING	*valeur;

        // p_attname -> name
	attname = malloc (sizeof(IDL_STRING));
	IDL_StrStore (attname, p_attname);

	valeur = malloc (sizeof(IDL_STRING));

	argv[0] = p01;
	argv[1] = p02;
	argv[2] = attname;
	argv[3] = valeur;
	code = MY_CDF_ATTGET (argc, argv);

	if (code == 1) {
		// valeur -> p_valeur
		strcpy (p_valeur, IDL_STRING_STR(valeur));
	} else {
		strcpy (p_valeur, "");
	}

	free (attname);
	free (valeur);

}

/*---------------------------------------------------------------------------*/
int OUVRIR_CDF_C (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	int		nbarg		= 0;
	IDL_STRING	*fichier	= (IDL_STRING *) 	argv[nbarg++]; /* fichier CDF  ouvrir */
	//double		top1;

	//top1 = nbsec();
	if (CDFopenCDF (IDL_STRING_STR(fichier), &fd) < CDF_WARN) {
		return -1;
	}
	//printf ("CDFopenCDF() in %.2f sec\n",nbsec()-top1);
	return 0;

}

/*---------------------------------------------------------------------------*/
int OUVRIR_CDF_C_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p_fichier) {

	int		argc=1;
	void		*argv[1];
	int		code;
	IDL_STRING	*fichier;

	// p_fichier -> fichier
	fichier = malloc (sizeof(IDL_STRING));
	IDL_StrStore (fichier, p_fichier);

	argv[0] = fichier;
	code = OUVRIR_CDF_C (argc, argv);

	free (fichier);

	return code;
}


/*---------------------------------------------------------------------------*/
int FERMER_CDF_C (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	int		nbarg		= 0;
	IDL_STRING	*fichier	= (IDL_STRING *) 	argv[nbarg++]; /* fichier CDF  ouvrir */
	//double		top1;

	//top1 = nbsec();
	CDFcloseCDF (fd);
	//printf ("CDFcloseCDF() in %.2f sec\n",nbsec()-top1);
	return 0;

}

/*---------------------------------------------------------------------------*/
int FERMER_CDF_C_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p_fichier) {

	int		argc=1;
	void		*argv[1];
	int		code;
	IDL_STRING	*fichier;

	// p_fichier -> fichier
	fichier = malloc (sizeof(IDL_STRING));
	IDL_StrStore (fichier, p_fichier);

	argv[0] = fichier;
	code = FERMER_CDF_C (argc, argv);

	free (fichier);

	return code;

}

/*---------------------------------------------------------------------------*/
int LIRE_VAR_CDF_C (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	int		nbarg		= 0;
	IDL_STRING	*fichier	= (IDL_STRING *) 	argv[nbarg++]; /* fichier CDF  ouvrir */
	int		ouvrir_fermer	= *(int *)		argv[nbarg++]; /* faut-il ouvrir le fichier ? */
	int		offset		= *(int *)		argv[nbarg++]; /* commencre  crire  l'enregistrement offset de data */
	int		REC_START	= *(int *)		argv[nbarg++]; /*  partir de REC_START */
	int		REC_COUNT	= *(int *)		argv[nbarg++]; /* en lire REC_COUNT */
	IDL_STRING	*variable	= (IDL_STRING *) 	argv[nbarg++]; /* variable [r ou z]  lire */
	unsigned char	*data		= (unsigned char *)	argv[nbarg++]; /* pointeur sur la structure  renseigner */
	int		taille_struct	= *(int *)		argv[nbarg++]; /* nombre d'octets dans la structure */
	int		type_champ	= *(int *)		argv[nbarg++]; /* code du type du champ dans la structure */
	int		offset_champ	= *(int *)		argv[nbarg++]; /* position en octets du champ dans la structure */
	int		nblignesonly	= *(int *)		argv[nbarg++]; /* 1 pour renvoyer seulement le nombre de lignes */
	int		*nblignesout	= (int *)		argv[nbarg++]; /* retourne le nombre de lignes */
	int		*info_alire	= (int *)		argv[nbarg++];
	int		*datatype1	= (int *)		argv[nbarg++]; /* valeur retourne de datatype */
	int		*nbdim1		= (int *)		argv[nbarg++]; /* valeur retourne de nbdim */
	int		*dimSizes1	= (int *)		argv[nbarg++]; /* valeur retourne de dimSizes */
	char		*majority1	= (char *)		argv[nbarg++]; /* valeur retourne de majority 'C' ou 'F' */
	float		*validmin	= (float *)		argv[nbarg++]; /* valeur retourne de validmin */
	float		*validmax	= (float *)		argv[nbarg++]; /* valeur retourne de validmax */
	float		*scalemin	= (float *)		argv[nbarg++]; /* valeur retourne de scalemin */
	float		*scalemax	= (float *)		argv[nbarg++]; /* valeur retourne de scalemax */
	int		is_fdl		= *(int *)		argv[nbarg++]; /* 1 si FDL, 0 sinon */

/*

	Valeurs possibles pour type_champ:

		#define IDL_TYP_UNDEF           0
		#define IDL_TYP_BYTE            1
		#define IDL_TYP_INT             2
		#define IDL_TYP_LONG            3
		#define IDL_TYP_FLOAT           4
		#define IDL_TYP_DOUBLE          5
		#define IDL_TYP_COMPLEX         6
		#define IDL_TYP_STRING          7
		#define IDL_TYP_STRUCT          8
		#define IDL_TYP_DCOMPLEX        9
		#define IDL_TYP_PTR             10
		#define IDL_TYP_OBJREF          11
		#define IDL_TYP_UINT            12
		#define IDL_TYP_ULONG           13
		#define IDL_TYP_LONG64          14
		#define IDL_TYP_ULONG64         15

*/

/*
	Valeurs possibles pour datatype:

		#define CDF_INT1                1L
		#define CDF_INT2                2L
		#define CDF_INT4                4L
		#define CDF_INT8                8L
		#define CDF_UINT1               11L
		#define CDF_UINT2               12L
		#define CDF_UINT4               14L
		#define CDF_REAL4               21L
		#define CDF_REAL8               22L
		#define CDF_EPOCH               31L     Standard style.
		#define CDF_EPOCH16             32L     Extended style. *
		#define CDF_BYTE                41L     same as CDF_INT1 (signed)
		#define CDF_FLOAT               44L     same as CDF_REAL4
		#define CDF_DOUBLE              45L     same as CDF_REAL8
		#define CDF_CHAR                51L     a "string" data type
		#define CDF_UCHAR               52L     a "string" data type

*/



/*

	info_alire:

		-2 pour lire toutes les dimensions
		info_alire[0..nbdim-1]:	-1 pour extraire les valeurs  tous les indices de cette dimension
					no pour extraire les valeurs  l'indice no (>=0)

*/
				

/*	
	Retourne:
		0:	OK
		-1:	erreur avec CDFopenCDF()
		-2:	erreur avec CDFgetVarNum()
		-3:	erreur avec CDFinquire()
		-4:	erreur avec CDFgetzVarNumRecsWritten()
		-5:	erreur avec CDFvarInquire et CDFinquirezVar() 
		-6:	erreur avec malloc()
		-7:	erreur avec CDFvarHyperGet()
		-8:	erreur avec CDFhyperGetzVarData()
		-9:	erreur avec CDFcloseCDF()
		-10:	erreur avec CDFgetDataTypeSize()
		-11:	erreur avec CDFgetMajority()
*/


	char		nom[CDF_VAR_NAME_LEN256+1];
	int		i;
	int		j;
	int		k;
	long		novar;
	long		datatype;
	long		nombre_elements;
	long		nbdim;
	long		dimSizes[CDF_MAX_DIMS];
	long		dimSizes_reel[CDF_MAX_DIMS];
	long		recVary;
	long		dimVarys[CDF_MAX_DIMS];
	int		dimVarysOK; /* 1 si tous les dimVarys[0:nbdim-1] valent VARY */
	long		arraySize; /* produit de toutes les dimensions */
	long		counts[CDF_MAX_DIMS];
	long		indices[CDF_MAX_DIMS];
	long		intervals[CDF_MAX_DIMS];
	int		isz;

	long		encoding;
	long		majority;
	long		numVars;
	long		numAttrs;

	long		nblignes;	// 20200205: passage de int en long sinon problme avec compilateur sous CentOS8 (gcc 8.2.1)
					// et il faut effectivement que ce soit un long dans l'appel  CDFgetzVarNumRecsWritten()


	unsigned char	*datamalloc=NULL;
	unsigned char	*alire=NULL;
	int		flag_alire[CDF_MAX_DIMS];
	long		taillecdf;
	double		decalage_1958 = computeEPOCH (1958, 1, 1, 0, 0, 0, 0);
	unsigned char	*psrc;
	unsigned char	*pdst;
	long		icdf[CDF_MAX_DIMS];
	int		virtual;
	long		noatt;
	long		datatype_att;
	long		numElems_att;

	unsigned char	buffer_FILLVAL[1024];
	float		FILLVAL=-1e31;
	int		have_FILLVAL=0;

	unsigned char	buffer_VALIDMIN[1024];
	float		VALIDMIN=-1e31;
	int		have_VALIDMIN=0;

	unsigned char	buffer_VALIDMAX[1024];
	float		VALIDMAX=-1e31;
	int		have_VALIDMAX=0;

	unsigned char	buffer_SCALEMIN[1024];
	float		SCALEMIN=-1e31;
	int		have_SCALEMIN=0;

	unsigned char	buffer_SCALEMAX[1024];
	float		SCALEMAX=-1e31;
	int		have_SCALEMAX=0;

	char		*pstring=NULL;

	double		top1;
	double		top2;

	double		epoch16[2];

	int		alire_all;
	int		use_memcpy;

	// Dfinir RESIZE_RVAR pour supprimer les dimensions NOVARY des rvar.
	// Ces variables seront plus petites et la lecture sera plus rapide.
	//
	// ex: lecture variable "Epoch" dans le fichier "ac_h3_mfi_20120305_v01.cdf"
	//
	// sans RESIZE_RVAR:
	// ----------------
	//			malloc of 8 * 1 * 86400 * 3 bytes
	//			Debut CDFvarHyperGet(Epoch)
	//			Fin CDFvarHyperGet() en 8.47 sec
	//
	// avec RESIZE_RVAR:
	// ----------------
	//			malloc of 8 * 1 * 86400 * 1 bytes
	//			Debut CDFvarHyperGet(Epoch)
	//			Fin CDFvarHyperGet() en 0.28 sec



	#define		RESIZE_RVAR

	int	debug=0;

	/*
		fichier CDF  lire
		-------------------
	*/
	if (ouvrir_fermer==1) {
		if (debug) top1=nbsec();
		if (CDFopenCDF (IDL_STRING_STR(fichier), &fd) < CDF_WARN)
			return -1;
		if (debug) printf ("CDFopenCDF() in %.2f sec\n",nbsec()-top1);
	}

	
	/*
		variable [r ou z] CDF  lire
		----------------------------
	*/
	if ((novar = CDFgetVarNum (fd, IDL_STRING_STR(variable))) < CDF_WARN) {
		if (ouvrir_fermer==1) CDFcloseCDF (fd);
		return -2;
	}


	/*
		Informations sur la variable [r ou z] CDF  lire
		------------------------------------------------
	*/

	// Il est impratif de tester d'abord rVar puis ensuite zVar
	// sinon sur "/DATA/WIND/DATA/CDF/EHSP_3DP/wi_ehsp_3dp_20000101_v02.cdf" impossible de lire la variable "Epoch"

	// Dans istp_public/data/ace/or_ssc/2012/ac_or_ssc_20120727_v01.cdf on a:
	// rvar: Epoch(novar=0) GSE_LAT(novar=1) GSE_LON(novar=2) RADIUS(novar=3)
	// zvar: XYZ_GSE(novar=0)
	// ATTENTION lorsque l'on interroge la variable XYZ_GSE!
	if (CDFvarInquire (fd, novar, nom, &datatype, &nombre_elements, &recVary, dimVarys) >= CDF_WARN && strcmp(nom,IDL_STRING_STR(variable))==0) {
		isz = 0;
		if (CDFinquire (fd, &nbdim, dimSizes, &encoding, &majority, &nblignes, &numVars, &numAttrs) < CDF_WARN) {
			if (ouvrir_fermer==1) CDFcloseCDF (fd);
			return -3;
		}
		nblignes++;
	} else if (CDFinquirezVar (fd, novar, nom, &datatype, &nombre_elements, &nbdim, dimSizes, &recVary, dimVarys) >= CDF_WARN) {

		isz = 1;

		// rapide
		if (CDFgetzVarNumRecsWritten (fd, novar, &nblignes) < CDF_WARN) {
			if (ouvrir_fermer==1) CDFcloseCDF (fd);
			return -4;
		}

		// rapide
		if (CDFgetMajority (fd, &majority) < CDF_WARN) {
			if (ouvrir_fermer==1) CDFcloseCDF (fd);
			return -11;
		}

	} else {
		if (ouvrir_fermer==1) CDFcloseCDF (fd);
		return -5;
	}
	*datatype1 = datatype;
	*nbdim1 = nbdim;
	*majority1 = majority==ROW_MAJOR ? 'C' : 'F';
	for (i=0; i < nbdim; i++) {
		dimSizes1[i] = dimSizes[i];
		if (dimVarys[i]!=VARY) {
			// Pour les rVariables: inversion du signe des dimensions qui ne sont pas VARY
			dimSizes1[i]=-dimSizes1[i];
			dimSizes1[i]=-1;
		}
		#ifdef RESIZE_RVAR
		dimSizes_reel[i] = dimVarys[i]==VARY ? dimSizes[i] : 1;
		#else
		dimSizes_reel[i] = dimSizes[i];
		#endif
	}


	/*	Nombre d'octets pour le type datatype
		-------------------------------------
	*/
	// rapide
	if (CDFgetDataTypeSize(datatype, &taillecdf) < CDF_WARN) {
		if (ouvrir_fermer==1) CDFcloseCDF (fd);
		return -10;
	}

	if (debug) printf ("\n");
	if (debug) printf ("CDF file     = %s\n", IDL_STRING_STR(fichier));
	if (debug) printf ("variable     = %s\n", IDL_STRING_STR(variable));
	if (debug) printf ("novar        = %ld\n", novar);
	if (debug) printf ("CDF variable = %s (%s)\n", nom, isz ? "Zvar" : "rVar");
	if (debug) printf ("datatype     = %ld (%ld * %ld bytes)\n", datatype, taillecdf, nombre_elements);
	if (debug) printf ("majority     = %s\n", majority==ROW_MAJOR ? "ROW (C)":"COLUMN (FORTRAN)");
	if (debug) printf ("nblines      = %ld\n",nblignes);
	if (debug) printf ("recVary      = %s\n", recVary==VARY ? "VARY" : "NOVARY");
	arraySize = 1;
	dimVarysOK = 1;
	if (debug) printf ("[");
	for (i=0; i < nbdim; i++) {
		if (debug) printf ("%ld", dimSizes[i]);
		if (debug) printf (":%s", dimVarys[i]==VARY ? "VARY" : "NOVARY");
		if (dimVarys[i] != VARY) dimVarysOK=0;
		arraySize *= dimSizes_reel[i];
		counts[i] = dimSizes_reel[i];
		indices[i] = 0;
		intervals[i] = 1;
		if (debug) printf (i==nbdim-1 ? "" : ", ");
	}
	if (debug) printf ("] => arraySize = %ld bytes\n", arraySize);

	*nblignesout = nblignes;

	if (nblignesonly) {
		if (ouvrir_fermer==1) {
			if (debug) top1 = nbsec();
			CDFcloseCDF (fd);
			if (debug) printf ("CDFcloseCDF() in %.2f sec\n",nbsec()-top1);
		}
		return 0;
	}


	/*	Valeur de l'attribut FILLVAL pour la variable demande
		------------------------------------------------------
	*/

	if ((noatt = CDFgetAttrNum (fd, "FILLVAL")) >= CDF_OK) {
		if (isz == 0) {
			if (CDFattrEntryInquire (fd, noatt, novar, &datatype_att, &numElems_att) >= CDF_WARN)
				have_FILLVAL = CDFgetAttrrEntry (fd, noatt, novar, buffer_FILLVAL) >= CDF_WARN;
		} else {
			if (CDFinquireAttrzEntry (fd, noatt, novar, &datatype_att, &numElems_att) >= CDF_WARN)
				have_FILLVAL = CDFgetAttrzEntry (fd, noatt, novar, buffer_FILLVAL) >= CDF_WARN;
		}
	}

	if (have_FILLVAL) {
		FILLVAL = CDFtofloat (buffer_FILLVAL,datatype_att);
		if (debug) printf ("%s  FILLVAL=%f\n", IDL_STRING_STR(variable),FILLVAL);
	}

	/*	Valeur de l'attribut VALIDMIN pour la variable demande
		------------------------------------------------------
	*/

	if ((noatt = CDFgetAttrNum (fd, "VALIDMIN")) >= CDF_OK) {
		if (isz == 0) {
			if (CDFattrEntryInquire (fd, noatt, novar, &datatype_att, &numElems_att) >= CDF_WARN)
				have_VALIDMIN = CDFgetAttrrEntry (fd, noatt, novar, buffer_VALIDMIN) >= CDF_WARN;
		} else {
			if (CDFinquireAttrzEntry (fd, noatt, novar, &datatype_att, &numElems_att) >= CDF_WARN)
				have_VALIDMIN = CDFgetAttrzEntry (fd, noatt, novar, buffer_VALIDMIN) >= CDF_WARN;
		}
	}

	if (have_VALIDMIN) {
		VALIDMIN = CDFtofloat (buffer_VALIDMIN,datatype_att);
		*validmin = VALIDMIN;
		if (debug) printf ("%s  VALIDMIN=%f\n", IDL_STRING_STR(variable),VALIDMIN);
	}

	/*	Valeur de l'attribut VALIDMAX pour la variable demande
		------------------------------------------------------
	*/

	if ((noatt = CDFgetAttrNum (fd, "VALIDMAX")) >= CDF_OK) {
		if (isz == 0) {
			if (CDFattrEntryInquire (fd, noatt, novar, &datatype_att, &numElems_att) >= CDF_WARN)
				have_VALIDMAX = CDFgetAttrrEntry (fd, noatt, novar, buffer_VALIDMAX) >= CDF_WARN;
		} else {
			if (CDFinquireAttrzEntry (fd, noatt, novar, &datatype_att, &numElems_att) >= CDF_WARN)
				have_VALIDMAX = CDFgetAttrzEntry (fd, noatt, novar, buffer_VALIDMAX) >= CDF_WARN;
		}
	}

	if (have_VALIDMAX) {
		VALIDMAX = CDFtofloat (buffer_VALIDMAX,datatype_att);
		*validmax = VALIDMAX;
		if (debug) printf ("%s  VALIDMAX=%f\n", IDL_STRING_STR(variable),VALIDMAX);
	}

	/*	Valeur de l'attribut SCALEMIN pour la variable demande
		------------------------------------------------------
	*/

	if ((noatt = CDFgetAttrNum (fd, "SCALEMIN")) >= CDF_OK) {
		if (isz == 0) {
			if (CDFattrEntryInquire (fd, noatt, novar, &datatype_att, &numElems_att) >= CDF_WARN)
				have_SCALEMIN = CDFgetAttrrEntry (fd, noatt, novar, buffer_SCALEMIN) >= CDF_WARN;
		} else {
			if (CDFinquireAttrzEntry (fd, noatt, novar, &datatype_att, &numElems_att) >= CDF_WARN)
				have_SCALEMIN = CDFgetAttrzEntry (fd, noatt, novar, buffer_SCALEMIN) >= CDF_WARN;
		}
	}

	if (have_SCALEMIN) {
		SCALEMIN = CDFtofloat (buffer_SCALEMIN,datatype_att);
		*scalemin = SCALEMIN;
		if (debug) printf ("%s  SCALEMIN=%f\n", IDL_STRING_STR(variable),SCALEMIN);
	}

	/*	Valeur de l'attribut SCALEMAX pour la variable demande
		------------------------------------------------------
	*/

	if ((noatt = CDFgetAttrNum (fd, "SCALEMAX")) >= CDF_OK) {
		if (isz == 0) {
			if (CDFattrEntryInquire (fd, noatt, novar, &datatype_att, &numElems_att) >= CDF_WARN)
				have_SCALEMAX = CDFgetAttrrEntry (fd, noatt, novar, buffer_SCALEMAX) >= CDF_WARN;
		} else {
			if (CDFinquireAttrzEntry (fd, noatt, novar, &datatype_att, &numElems_att) >= CDF_WARN)
				have_SCALEMAX = CDFgetAttrzEntry (fd, noatt, novar, buffer_SCALEMAX) >= CDF_WARN;
		}
	}

	if (have_SCALEMAX) {
		SCALEMAX = CDFtofloat (buffer_SCALEMAX,datatype_att);
		*scalemax = SCALEMAX;
		if (debug) printf ("%s  SCALEMAX=%f\n", IDL_STRING_STR(variable),SCALEMAX);
	}

	if ((alire = (unsigned char *) malloc (arraySize)) == NULL) {
		if (ouvrir_fermer==1) CDFcloseCDF (fd);
		return -6;
	}

	for (j=0 ; j<nbdim ; j++) {
		if (info_alire[0] == -2) {
			flag_alire[j] = -1;
		} else {
			flag_alire[j] = info_alire[j];
		}
	}


	for (j=0; j<nbdim; j++) icdf[j]=0;
	for (j=0 ; j<arraySize ; j++) {

		virtual=0;
		#ifdef RESIZE_RVAR
		#else
		for (k=0 ; k<nbdim ; k++) {
			/* Gestion des VARY et NOVARY: sauter les valeurs aux indices non nuls avec NOVARY */
			if (dimVarys[k]==NOVARY && icdf[k]) {
				virtual=1;
				break;
			}
		}
		#endif

		alire[j] = virtual==0;
		if (alire[j]) {
			for (k=0 ; k<nbdim ; k++) {
				if ((flag_alire[k] != -1) && (icdf[k] != flag_alire[k])) {
					alire[j] = 0;
					break;
				}
			}
		}

		incrementer (nbdim, dimSizes_reel, icdf, majority);

	}

	alire_all = 1;
	for (j=0 ; j<arraySize ; j++) {
		if (alire[j] == 0) {
			alire_all = 0;
			break;
		}
	}

	use_memcpy = alire_all && recVary && (
			((type_champ == IDL_TYP_FLOAT)						&& (datatype == CDF_REAL4 || datatype == CDF_FLOAT)) ||
			((type_champ == IDL_TYP_DOUBLE)						&& (datatype == CDF_REAL8 || datatype == CDF_DOUBLE)) ||
			((type_champ == IDL_TYP_LONG || type_champ == IDL_TYP_ULONG)		&& (datatype == CDF_INT4  || datatype == CDF_UINT4)) ||
			((type_champ == IDL_TYP_LONG64 || type_champ == IDL_TYP_ULONG64)	&& (CDF_INT8)) ||
			((type_champ == IDL_TYP_INT || type_champ == IDL_TYP_UINT)		&& (datatype == CDF_INT2  || datatype == CDF_UINT2)) ||
			((type_champ == IDL_TYP_BYTE)						&& (datatype == CDF_INT1  || datatype == CDF_BYTE)) );

	if (use_memcpy == 0) {

		/*
			Allocation memoire avant de lire la zvariable CDF
			-------------------------------------------------
		*/

		if (debug) printf ("malloc of %ld * %ld * %d * %ld bytes\n", taillecdf, nombre_elements, REC_COUNT, arraySize);
		if ((datamalloc = (unsigned char *) malloc (taillecdf * nombre_elements * REC_COUNT * arraySize)) == NULL) {
			if (ouvrir_fermer==1) CDFcloseCDF (fd);
			return -6;
		}

		// nombre_elements+1: pour stocker le 0 de fin de chaine de caracteres
		pstring = (char *) malloc (nombre_elements+1);

		/*
			Lecture de REC_COUNT lignes  partir de la ligne REC_START de la zvariable CDF dans datamalloc
			----------------------------------------------------------------------------------------------
		*/

		if (isz == 0) {
			if (debug) top1 = nbsec();
			if (debug) printf ("Debut CDFvarHyperGet(%s)\n",IDL_STRING_STR(variable));
			if (CDFvarHyperGet (fd, novar, REC_START, REC_COUNT, 1, indices, counts, intervals, datamalloc) < CDF_WARN) {
				if (ouvrir_fermer==1) CDFcloseCDF (fd);
				return -7;
			}
			if (debug) top2 = nbsec();
			if (debug) printf ("Fin CDFvarHyperGet() en %.2f sec\n",top2-top1);
		} else {
			if (debug) top1 = nbsec();
			if (debug) printf ("Debut CDFhyperGetzVarData()\n");
			if (CDFhyperGetzVarData (fd, novar, REC_START, REC_COUNT, 1, indices, counts, intervals, datamalloc) < CDF_WARN) {
				if (ouvrir_fermer==1) CDFcloseCDF (fd);
				return -8;
			}
			if (debug) top2 = nbsec();
			if (debug) printf ("Fin CDFhyperGetzVarData() en %.2f sec\n",top2-top1);
		}
	}



	if (debug) top1 = nbsec();
	if (debug) printf ("Debut recopie\n");

	for (i=REC_START ; i<REC_START+REC_COUNT ; i++) {

		pdst = data + offset_champ + (size_t)(offset+i-REC_START) * taille_struct;

		if (use_memcpy) {

			/*
				Lecture directe dans pdst
				--------------------------
			*/
			if (isz == 0) {
				if (CDFvarHyperGet (fd, novar, i, 1, 1, indices, counts, intervals, pdst) < CDF_WARN) {
					if (ouvrir_fermer==1) CDFcloseCDF (fd);
					return -7;
				}
			} else {
				if (CDFhyperGetzVarData (fd, novar, i, 1, 1, indices, counts, intervals, pdst) < CDF_WARN) {
					if (ouvrir_fermer==1) CDFcloseCDF (fd);
					return -8;
				}
			}
			if (type_champ == IDL_TYP_FLOAT && have_FILLVAL && FILLVAL>-1e30) {
				for (j=0 ; j<arraySize ; j++) {
					if (* (float *) pdst == FILLVAL) {
						* (float *) pdst = -1e31;
					}
				}
			}
			continue;

		}

		/*
			Recopie des donnes de datamalloc dans data
			-------------------------------------------
		*/
		psrc = datamalloc + (size_t)(i-REC_START) * arraySize * taillecdf * nombre_elements;
		for (j=0 ; j<arraySize ; j++) {

			if (alire[j]) {

				if (datatype == CDF_EPOCH || datatype == CDF_EPOCH16) {

					//printf ("r EPOCH  type_champ=%d %s\n", type_champ,IDL_STRING_STR(variable));
					// Si epoch vaut 0 (arrive sur "C4_CP_PEA_3DRH_DEFlux__20010214_000000_20010215_000000_V090617.cdf"), ne pas mettre
					// 0 - decalage_1958 sinon IDL ne va pas apprcier la date
					// Laisser 0 qui correspond au 01/01/1958
					if (type_champ == IDL_TYP_FLOAT) {
						// arrive sur "/DATA/WIND/DATA/CDF/EHSP_3DP/wi_ehsp_3dp_20000101_v02.cdf" avec la variable "TIME"
						if (CDFtofloat (psrc,datatype)==0) {
							//printf ("*** WARNING EPOCH==0 *** var=[%s] filename=[%s]\n",IDL_STRING_STR(variable),IDL_STRING_STR(fichier));
							*(float *) pdst = 0;
						} else {
							*(float *) pdst = CDFtofloat (psrc,datatype) - decalage_1958;
						}
						pdst += 4;
					} else {
						if (CDFtodouble (psrc,datatype)==0) {
							//printf ("*** WARNING EPOCH==0 *** var=[%s] filename=[%s]\n",IDL_STRING_STR(variable),IDL_STRING_STR(fichier));
							*(double *) pdst = 0;
						} else {
							*(double *) pdst = CDFtodouble (psrc,datatype) - decalage_1958;
						}
						pdst += 8;
					}

				} else if (datatype == CDF_TIME_TT2000) {

					// 01/01/2000 12h	->	1325419200000
					// 32.184s		-> 	32184
					// 10 leap secondes avant 1972 et 22 leap secondes entre 1972 et 2000
					//
					// En faisant : 
					//  si type_champ == IDL_TYP_FLOAT  *(float *)  pdst = 1325419200000 - 32184 - 32000 + CDFtofloat  (psrc,datatype);
					//  si type_champ == IDL_TYP_DOUBLE *(double *) pdst = 1325419200000 - 32184 - 32000 + CDFtodouble (psrc,datatype);
					// on obtient un temps qui ne tient pas compte des leap secondes aprs 2000 et ne donne pas le mme rsultat que CDF_TT2000_to_UTC_EPOCH16

					if (type_champ == IDL_TYP_FLOAT) {
						if (CDFtofloat (psrc,datatype)==0) {
							//printf ("*** WARNING EPOCH==0 *** var=[%s] filename=[%s]\n",IDL_STRING_STR(variable),IDL_STRING_STR(fichier));
							*(float *) pdst = 0;
						} else {
							CDF_TT2000_to_UTC_EPOCH16(*(long long *)psrc,(double *)&epoch16);
							*(float *) pdst = CDFtofloat ((unsigned char *)epoch16,CDF_EPOCH16) - decalage_1958;
						}
						pdst += 4;
					} else {
						if (CDFtodouble (psrc,datatype)==0) {
							//printf ("*** WARNING EPOCH==0 *** var=[%s] filename=[%s]\n",IDL_STRING_STR(variable),IDL_STRING_STR(fichier));
							*(double *) pdst = 0;
						} else {
							CDF_TT2000_to_UTC_EPOCH16(*(long long *)psrc,(double *)&epoch16);
							*(double *) pdst = CDFtodouble ((unsigned char *)epoch16,CDF_EPOCH16) - decalage_1958;
						}
						pdst += 8;
					}

				} else if (type_champ == IDL_TYP_DOUBLE) {
					//if (debug) printf ("r DOUBLE  %s\n", IDL_STRING_STR(variable));
					*(double *) pdst = CDFtodouble (psrc,datatype);
					pdst += 8;

				} else if (type_champ == IDL_TYP_FLOAT) {

					//if (debug) printf ("r FLOAT  %s\n", IDL_STRING_STR(variable));
					*(float *) pdst = CDFtofloat_FILLVAL (psrc, datatype, have_FILLVAL, FILLVAL);
					pdst += 4;

				} else if (type_champ == IDL_TYP_LONG || type_champ == IDL_TYP_ULONG) {

					//if (debug) printf ("r INT  %s\n", IDL_STRING_STR(variable));
					*(int *) pdst = CDFtoint (psrc, datatype);
					pdst += 4;

				} else if (type_champ == IDL_TYP_LONG64 || type_champ == IDL_TYP_ULONG64) {

					//if (debug) printf ("r LONG LONG INT  %s\n", IDL_STRING_STR(variable));
					*(long long int *) pdst = CDFtolonglongint (psrc, datatype);
					pdst += 8;

				} else if (type_champ == IDL_TYP_INT || type_champ == IDL_TYP_UINT) {

					//if (debug) printf ("r SHORT  %s\n", IDL_STRING_STR(variable));
					*(short *) pdst = CDFtoshort (psrc, datatype);
					pdst += 2;

				} else if (type_champ == IDL_TYP_BYTE) {

					//if (debug) printf ("r CHAR  %s\n", IDL_STRING_STR(variable));
					*(unsigned char *) pdst = CDFtochar (psrc, datatype);
					pdst += 1;

				} else if (type_champ == IDL_TYP_STRING) {

					CDFtostring (psrc, datatype, nombre_elements, pstring);
					//printf ("i=%d j=%d %s=[%s] -> STRING (%ld) taille_cdf=%ld\n", i,j,IDL_STRING_STR(variable),pstring,sizeof(IDL_STRING),taillecdf);
					if (is_fdl) {
						sprintf (((IDL_STRING *)pdst)->s,"%s%c", pstring, 255);
					} else {
						IDL_StrStore ((IDL_STRING *)pdst, pstring);
					}
					pdst += sizeof(IDL_STRING) * (is_fdl ? 2 : 1);

				}

			}
			psrc += taillecdf * nombre_elements;

		}

		if (recVary == NOVARY)
			break;

	}
	if (debug) top2 = nbsec();
	if (debug) printf ("Fin recopie en %.2f sec\n",top2-top1);


	/*
		Liberation memoire de datamalloc
		--------------------------------
	*/
	if (datamalloc != NULL) free (datamalloc);
	if (alire != NULL) free (alire);
	if (pstring != NULL) free (pstring);

	/*
		Fermeture fichier CDF
		---------------------
	*/
	if (ouvrir_fermer==1) {
		if (CDFcloseCDF (fd) < CDF_WARN)
			return -9;
	}

	return 0;

}

/*---------------------------------------------------------------------------*/
int LIRE_VAR_CDF_C_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p_fichier,
	void	*p02,
	void	*p03,
	void	*p04,
	void	*p05,
	void	*p_variable,
	void	*p07,
	void	*p08,
	void	*p09,
	void	*p10,
	void	*p11,
	void	*p12,
	void	*p13,
	void	*p14,
	void	*p15,
	void	*p16,
	void	*p17,
	void	*p18,
	void	*p19,
	void	*p20,
	void	*p21,
	void	*p22) {

	int		argc=22;
	void		*argv[22];
	int		code;
	IDL_STRING	*fichier;
	IDL_STRING	*variable;

	// p_fichier -> fichier
	fichier = malloc (sizeof(IDL_STRING));
	IDL_StrStore (fichier, p_fichier);

	// p_variable -> variable
	variable = malloc (sizeof(IDL_STRING));
	IDL_StrStore (variable, p_variable);

	argv[ 0] = fichier;
	argv[ 1] = p02;
	argv[ 2] = p03;
	argv[ 3] = p04;
	argv[ 4] = p05;
	argv[ 5] = variable;
	argv[ 6] = p07;
	argv[ 7] = p08;
	argv[ 8] = p09;
	argv[ 9] = p10;
	argv[10] = p11;
	argv[11] = p12;
	argv[12] = p13;
	argv[13] = p14;
	argv[14] = p15;
	argv[15] = p16;
	argv[16] = p17;
	argv[17] = p18;
	argv[18] = p19;
	argv[19] = p20;
	argv[20] = p21;
	argv[21] = p22;
	code = LIRE_VAR_CDF_C (argc, argv);

	free (fichier);
	free (variable);

	return code;
}

/*---------------------------------------------------------------------------*/
int LIRE_VARS_CDF_C (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	int		nbarg		= 0;
	IDL_STRING	*fichier	= (IDL_STRING *) 	argv[nbarg++]; /* fichier CDF  ouvrir */
	int		ouvrir_fermer	= *(int *)		argv[nbarg++]; /* faut-il ouvrir le fichier ? */
	int		offset		= *(int *) 		argv[nbarg++]; /* commencre  crire  l'enregistrement offset de data */
	int		REC_START	= *(int *) 		argv[nbarg++]; /*  partir de REC_START */
	int		REC_COUNT	= *(int *) 		argv[nbarg++]; /* en lire REC_COUNT */
	IDL_STRING	*tab_nom	= (IDL_STRING *) 	argv[nbarg++]; /* tab_nom[nbvars]: variable [r ou z]  lire */
	unsigned char	*data		= (unsigned char *)	argv[nbarg++]; /* pointeur sur la structure  renseigner */
	int		taille_struct	= *(int *)		argv[nbarg++]; /* nombre d'octets dans la structure */
	int		*tab_type	= (int *)		argv[nbarg++]; /* tab_type(nbvars]: code du type du champ dans la structure */
	int		*tab_offset	= (int *)		argv[nbarg++]; /* tab_offset[nbvars]: position en octets du champ dans la structure */
	int		nblignesonly	= *(int *)		argv[nbarg++]; /* 1 pour renvoyer seulement le nombre de lignes */
	int		*nblignesout	= (int *)		argv[nbarg++]; /* retourne le nombre de lignes */
	int		*tab_info	= (int *)		argv[nbarg++]; /* tab_info[nbvars,nbdims] */
	int		*datatype1	= (int *)		argv[nbarg++]; /* valeur retourne de datatype */
	int		*nbdim1		= (int *)		argv[nbarg++]; /* valeur retourne de nbdim */
	int		*dimSizes1	= (int *)		argv[nbarg++]; /* valeur retourne de dimSizes */
	char		*majority1	= (char *)		argv[nbarg++]; /* valeur retourne de majority 'C' ou 'F' */
	float		*tab_validmin	= (float *)		argv[nbarg++]; /* tab_validmin[nbvars]: valeur retourne de validmin */
	float		*tab_validmax	= (float *)		argv[nbarg++]; /* tab_validmax[nbvars]: valeur retourne de validmax */
	float		*tab_scalemin	= (float *)		argv[nbarg++]; /* tab_scalemin[nbvars]: valeur retourne de scalemin */
	float		*tab_scalemax	= (float *)		argv[nbarg++]; /* tab_scalemax[nbvars]: valeur retourne de scalemax */
	int		is_fdl		= *(int *)		argv[nbarg++]; /* 1 si FDL, 0 sinon */
	int		nbvars		= *(int *)		argv[nbarg++]; /* nombre de variables  lire */

	int		i;
	int		code=0;
	int		argc1=22;
	void		*argv1[22];

	for (i=0 ; i<nbvars ; i++) {

		argv1[ 0] = argv[ 0];
		argv1[ 1] = argv[ 1];
		argv1[ 2] = argv[ 2];
		argv1[ 3] = argv[ 3];
		argv1[ 4] = argv[ 4];
		argv1[ 5] = tab_nom+i;
		argv1[ 6] = argv[ 6];
		argv1[ 7] = argv[ 7];
		argv1[ 8] = &(tab_type[i]);
		argv1[ 9] = &(tab_offset[i]);
		argv1[10] = argv[10];
		argv1[11] = argv[11];
		argv1[12] = &(tab_info[indice2(i,0,nbvars,INUTILE)]);
		argv1[13] = argv[13];
		argv1[14] = argv[14];
		argv1[15] = argv[15];
		argv1[16] = argv[16];
		argv1[17] = &(tab_validmin[i]);
		argv1[18] = &(tab_validmax[i]);
		argv1[19] = &(tab_scalemin[i]);
		argv1[20] = &(tab_scalemax[i]);
		argv1[21] = argv[21];
		code = LIRE_VAR_CDF_C (argc1, argv1);

		if (code != 0) {
			printf ("lire_struct_cdf_c %s in %s returns %d\n", IDL_STRING_STR(tab_nom+i), IDL_STRING_STR(fichier),code);
		}

	}

	return code;

}

/*---------------------------------------------------------------------------*/
void TT2000_TO_DATE (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	int		nbarg		= 0;
	long long	*tt2000		= (long long *) 	argv[nbarg++]; /* tableau de dates en long long */
	int		nb		= *(int *)		argv[nbarg++]; /* nombre d'lments dans le tableau tt2000 */
	double		*date		= (double *) 		argv[nbarg++]; /* tableau de dates en ms depuis secondes depuis 1/1/1958*/

	int		i;
	double		epoch16[2];
	double		decalage_1958 = computeEPOCH (1958, 1, 1, 0, 0, 0, 0);

	for (i=0 ; i<nb ; i++) {
		CDF_TT2000_to_UTC_EPOCH16(tt2000[i],(double *)&epoch16);
		date[i] = CDFtodouble ((unsigned char *)&epoch16,CDF_EPOCH16) - decalage_1958;
	}

}

/*---------------------------------------------------------------------------*/
void TT2000_TO_DATE_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p01,
	void	*p02,
	void	*p03) {

	int	argc=3;
	void	*argv[3];

	argv[0] = p01;
	argv[1] = p02;
	argv[2] = p03;
	TT2000_TO_DATE (argc, argv);

}
	

/*---------------------------------------------------------------------------*/
void EPOCH16_TO_DATE (int argc, void *argv[]) {
/*---------------------------------------------------------------------------*/

	int		nbarg		= 0;
	double 		*epoch16	= (double*) 		argv[nbarg++]; /* tableau de dates en paires de double */
	int		nb		= *(int *)		argv[nbarg++]; /* nombre d'lments dans le tableau tt2000 */
	double		*date		= (double *) 		argv[nbarg++]; /* tableau de dates en ms depuis secondes depuis 1/1/1958*/

	int		i;
	double		decalage_1958 = computeEPOCH (1958, 1, 1, 0, 0, 0, 0);

	for (i=0 ; i<nb ; i++) {
		date[i] = CDFtodouble ((unsigned char *)&epoch16[2*i],CDF_EPOCH16) - decalage_1958;
	}

}

/*---------------------------------------------------------------------------*/
void EPOCH16_TO_DATE_AUTO_GLUE (
/*---------------------------------------------------------------------------*/
	void	*p01,
	void	*p02,
	void	*p03) {

	int	argc=3;
	void	*argv[3];

	argv[0] = p01;
	argv[1] = p02;
	argv[2] = p03;
	EPOCH16_TO_DATE (argc, argv);

}
