/***************************************************************************************************
 *
 *	Fichier	: $RCSfile: ORBITE.c,v $
 *
 *	Version	: $Revision: 1.16 $
 *
 *	Auteur	: $Author: penou $
 *
 *	Date	: $Date: 2007/07/19 10:35:57 $
 *
 *	==========================================================================================
 *
 *	Ce module a pour but la preparation d'un fichier d'orbite a partir 
 *	des donnees fournies par le fichier STOF
 */

#define	MODULE_NAME	"ORBITE"

#include "CISLIB.h"
#include "INTERFACE.h"
#include "CONTEXTE.h"
#include "ORBITE.h"


/*	Declaration des routines Fortran externes
 *	-----------------------------------------
 */
extern	void	openf_	();	
extern	void	closef_	();
extern	void	jd2000_	();	
extern	void	orbit_	();
extern	void	pr2000_ ();
extern	void	csundi_	();


/***************************************************************************************************
 *
 *	Calcul de la matrice de passage GEI a GSE
 *	-----------------------------------------
 */
int	Matrice_GEI_to_GSE (t_date date, t_time time, M33 matrice)
{
	char *		fonction = FNAME ("Matrice_GEI_to_GSE");
	int		code = OK;
	int		nb_jour;
	float		gst, slong, sra, sdec, obliq;
	M33		M5E, M5L;
	double		L_sun_ecl;

	if (date.annee < 1901 || date.annee > 2099) {

		Affiche_erreur (fonction, "Annee %d non traitee par fonction CSUNDI()", date.annee);
		code = ERROR;
		goto EXIT;
	}
	nb_jour = Jour_annee (date);

	Affiche_trace (3, fonction, "Nombre jours depuis 1er janvier = %d", nb_jour);

	csundi_ (& date.annee, & nb_jour, & time.heure, & time.minute, & time.seconde,
		 & gst, & slong, & sra, & sdec, & obliq);

	Affiche_trace (3, fonction, "GST   = % e", gst);
	Affiche_trace (3, fonction, "SLONG = % e", slong);
	Affiche_trace (3, fonction, "SRA   = % e", sra);
	Affiche_trace (3, fonction, "SDEC  = % e", sdec);
	Affiche_trace (3, fonction, "OBLIQ = % e", obliq);

	(void) Matrice_rotation_repere_X (obliq, M5E);

	L_sun_ecl = (sra < M_PI) 
			? acos (cos (sra) * cos (sdec))
			: 2.0 * M_PI - acos (cos (sra) * cos (sdec));
	
	(void) Matrice_rotation_repere_Z (L_sun_ecl, M5L);

	Produit_matrice (M5L, M5E, matrice);

	Affiche_matrice (3, fonction, "Matrice M5E", M5E);
	Affiche_matrice (3, fonction, "Matrice M5L", M5L);
	Affiche_matrice (3, fonction, "Matrice GEI to GSE", matrice);

EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Creation du fichier d'orbite
 *	----------------------------
 */
int	Creation_fichier_orbite (int intervalle)
{
	char *		fonction = FNAME ("Creation_fichier_orbite");
	int		code = OK;
	FILE *		pf = NULL;
	char *		stof_file;
	t_date		date;
	t_time		time;
	int		nbre_record = 0;
	int		i, unit, kode, sat, erreur, ecrits;
	double		milli, terminaison;
	double		mod_jul_day, distance, revnum;
	double		deb_time = MAXDOUBLE;
	double		end_time = MINDOUBLE;
	V3		vect_2000, vect_gei, vect_gse;
	M33		matrice;
	float		tampon [4];
	char		message [100];

	if (contexte.site_production == CSPI)
		pf = Ouverture_produit (ORBIT, "w");
	else	pf = Ouverture_fichier_temporaire ("ORBIT", "w");

	if (pf == NULL) {

		Affiche_erreur (fonction, "Creation fichier orbite impossible");
		code = ERR_fopen;
		goto EXIT;
	}

	/*	Ouverture du fichier STOF (Short Term Orbit File)
	 */
	stof_file = Nom_fichier_produit (AUX_STOF);

	if (strlen (stof_file) == 0) {

		Affiche_erreur (fonction, "Fichier STOF non defini");
		code = ERROR;
		goto EXIT;
	}
	unit = 9;

	openf_ (& unit, stof_file, & code);

	if (code != OK) {

		Affiche_erreur (fonction, "Ouverture fichier %s impossible", stof_file);
		code = ERR_fopen;
		goto EXIT;
	}

	Affiche_trace (1, fonction, "Periodicite des donnees : %d minute(s)", intervalle);

	/*	Initialisation parametres
	 */
	date.jour	= param.jour;
	date.mois	= param.mois;
	date.annee	= param.annee;
	time.heure	= 0;
	time.minute	= 0;
	time.seconde	= 0;
	time.milli	= 0;

	Date_time_to_milli (date, time, & milli);

	terminaison	= milli + MS_PAR_JOUR;

	/*	kode = 3 pour recuperer vecteur position (voir orbit.for)
	 */	
	kode = 3;		

	nbre_record	= 0;

	while (milli <= terminaison) {

		double		secondes;
		M33		mp, mpt;
		int		i, j;

		Milli_to_date_time (milli, & date, & time);

		secondes = (double) time.seconde;

		jd2000_ (& mod_jul_day,
			& date.annee, & date.mois, & date.jour,
			& time.heure, & time.minute, & secondes);
	
		Affiche_trace (3, fonction, "%s  Julian Day 2000 = %.4lf", 
			Milli_to_Ascii_time (milli),
			mod_jul_day);

		orbit_ (& mod_jul_day, & kode, & unit, 
			& erreur, & sat, & vect_2000, & revnum);

		if (erreur != 0) {

			switch (erreur) {

			case 1 :	sprintf (message, "Day %.4f too early", mod_jul_day);
					break;
			case 2 :	sprintf (message, "Day %.4f too late", mod_jul_day);
					break;
			case 3 :	sprintf (message, "Day %.4f gap in data", mod_jul_day);
					break;
			case 4 :	sprintf (message, "Wrong valus of kode %d incorrect", kode);
					break;
			case 5 :	strcpy  (message, "File content inconsistent");
					break;
			case 6 :	strcpy  (message, "Read error from data file");
					break;
			default:	strcpy  (message, "Unknown error");
					break;
			}

			Affiche_erreur (fonction, "Erreur ORBIT = %d : %s", erreur, message);
		}
		else { 	/*	Matrice passage repere GEI 2000 au repere GEI 
			 */
			pr2000_ (& mod_jul_day, & mp);

			/* 	Transposition pour probleme interface C - Fortran
			 */
			(void) Transpose_matrice (mp, mpt);

			/*	Passage repere GEI 2000 au repere GEI
			 */
			for (i = 0; i < 3; i++) {

				vect_gei [i] = 0.0;

				for (j = 0; j < 3; j++) vect_gei [i] += (mpt [i][j] * vect_2000 [j]);

				vect_gei [i] /= (double) RAYON_TERRESTRE;
			}	

			/*	Passage repere GEI au repere GSE
			 */
			code = Matrice_GEI_to_GSE (date, time, matrice);

			for (i = 0; i < 3; i++) {

				vect_gse [i]	= matrice [i][0] * vect_gei [0]
						+ matrice [i][1] * vect_gei [1]
						+ matrice [i][2] * vect_gei [2];
			}

			distance = Calcule_norme (vect_gse);

			Affiche_trace (3, fonction, "%s Distance = %15E",
				Milli_to_Ascii_time (milli),
				distance);

			Affiche_trace (3, fonction, "Position (GEI) = %15E %15E %15E",
				vect_gei [0], vect_gei [1], vect_gei [2]);

			Affiche_trace (3, fonction, "Position (GSE) = %15E %15E %15E",
				vect_gse [0], vect_gse [1], vect_gse [2]);

			tampon [0] = (float) vect_gse [0];
			tampon [1] = (float) vect_gse [1];
			tampon [2] = (float) vect_gse [2];
			tampon [3] = (float) distance;

			ecrits	= fwrite (& milli, sizeof (double), 1, pf);
			ecrits += fwrite (tampon,  sizeof (float),  4, pf);

			if (ecrits != 5) {

				Affiche_erreur (fonction, "Ecriture de %d valeurs au lieu de 5", ecrits);
				code = ERR_fwrite;
				goto EXIT;
			}
			nbre_record ++;

			if (milli < deb_time) deb_time = milli;
			if (milli > end_time) end_time = milli;
		}
		milli += (intervalle * 60.0) * 1000.0;	
	}
	closef_ (& unit);

EXIT:	if (pf != NULL)	fclose (pf);

	if (nbre_record > 0) {

		Affiche_trace (1, fonction, "Nombre enregistrements : %d", nbre_record);
		Affiche_trace (1, fonction, "Premier enregistrement : %s", Milli_to_Ascii_time (deb_time));
		Affiche_trace (1, fonction, "Dernier enregistrement : %s", Milli_to_Ascii_time (end_time));
	}

	return code;
}
