/***************************************************************************************************
 *
 *	Fichier	: $RCSfile: JSOC.c,v $
 *
 *	Version	: $Revision: 1.25 $
 *
 *	Auteur	: $Author: penou $
 *
 *	Date	: $Date: 2007/07/19 10:35:57 $
 *
 *	==========================================================================================
 *
 *	Ce module est charge des traitements specifiques au JSOC
 */

#define	MODULE_NAME	"JSOC"

#include "CISLIB.h"
#include "INTERFACE.h"
#include "CONTEXTE.h"
#include "ENTETE.h"
#include "CALIB.h"
#include "JSOC.h"


/*	Declaration prealable des fonctions de traitement des produits
 *	--------------------------------------------------------------
 */
int	Spectro_COD_12 (t_prod_header *);
int	Spectro_COD_13 (t_prod_header *);
int	Spectro_COD_14 (t_prod_header *);
int	Spectro_COD_29 (t_prod_header *);


/*	Definition de la table des fonctions de traitement a appliquer aux produits
 *	---------------------------------------------------------------------------
 */
typedef	int	(* FCT) (t_prod_header *);

typedef struct {
	t_instr		instrument;
	int		produit;
	t_produit	code;
	FCT		traitement;
}	t_descr;

static	t_descr	liste_produit [] = {
	
	{ CIS_1, 12, COD_12, Spectro_COD_12 },
	{ CIS_1, 13, COD_13, Spectro_COD_13 },
	{ CIS_1, 14, COD_14, Spectro_COD_14 },
	{ CIS_1, 29, COD_29, Spectro_COD_29 },
	{ CIS_2,  5, HIA_05, NULL },
	{ CIS_2,  6, HIA_06, NULL },
	{ CIS_2,  8, HIA_08, NULL },
	{ CIS_2, 10, HIA_10, NULL },
	{ CIS_2, 14, HIA_14, NULL },
	{ CIS_2, 15, HIA_15, NULL },
	{ CIS_2, 16, HIA_16, NULL },
	{ CIS_2, 17, HIA_17, NULL },
	{ CIS_2, 21, HIA_21, NULL },
	{ CIS_2, 22, HIA_22, NULL },
	{ CIS_2, 23, HIA_23, NULL },
	{ CIS_2, 24, HIA_24, NULL },
	{ -1,     0, 0     , NULL }
};


/*	Definition des priorites des produits a traiter par Telemetry rate et mode CIS
 *	------------------------------------------------------------------------------
 */
typedef enum { NM1 = 1, NM2, NM3, BM1, BM2, BM3 } t_tlm_rate;

typedef struct {
	t_instr		instrument;
	t_tlm_rate	tlm_rate;
	t_produit	produit [NB_CIS_MODE];
}	t_prio;

static	t_prio	priorites [] = {

	/* 		    00      01      02      03      04      05      06      07
	 *		    08      09      10      11      12      13      14      15
	 */
	{ CIS_1, NM1,	COD_13, COD_13, COD_13, COD_13, COD_13, COD_13, COD_29,     -1, 
			COD_12, COD_12, COD_12, COD_12, COD_12, COD_13, COD_12, COD_12 },
	{ CIS_1, NM2,	COD_13, COD_13, COD_13, COD_13, COD_13, COD_13, COD_29,     -1,
			COD_13, COD_12, COD_12, COD_12, COD_12, COD_13, COD_12, COD_12 },
	{ CIS_1, NM3,	COD_13, COD_13, COD_13, COD_13, COD_13, COD_13, COD_29,     -1,
			COD_13, COD_12, COD_12, COD_12, COD_12, COD_13, COD_12, COD_12 },
	{ CIS_1, BM1,	COD_13, COD_13, COD_13, COD_13, COD_13, COD_13, COD_29,     -1, 
			COD_13, COD_12, COD_12, COD_12, COD_12, COD_12, COD_12, COD_13 },
	{ CIS_1, BM2,	COD_13, COD_13, COD_13, COD_13, COD_13, COD_13, COD_29,     -1,
			COD_13, COD_12, COD_12, COD_12, COD_12, COD_13, COD_12, COD_12 },
	{ CIS_2, NM1,	    -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
			    -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1 },
	{ CIS_2, NM2,	    -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
			    -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1 },
	{ CIS_2, NM3,       -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
			    -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1 },
	{ CIS_2, BM1,	    -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
			    -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1 },
	{ CIS_2, BM2,	    -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
			    -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1 },
	{ -1 }
};


/*	Structure de donnees de memorisation des spectro
 *	------------------------------------------------
 */
typedef struct {
	double		date;
	t_instr		instrument;
	int		produit;
	int		sensit;
	int		cis_mode;
	int		tlm_rate;
	float		spectro [16];
	int		nombre;
}	t_record;


/*	Variable locales
 *	----------------
 */
static	char		buffer [33000];		
static	t_record	work;
static	int		nbre_spectro = 0;
static	float		efficacite_16  [NB_SENSITIVITE][NB_ANODE_CIS1][16];
static	float		efficacite_31  [NB_SENSITIVITE][NB_ANODE_CIS1][31];
static	float		efficacite_rpa [NB_SENSITIVITE][NB_ANODE_CIS1];
static	int		angle_88 [128];
static	int		somme_88 [NB_ANODE_CIS1] = { 4, 2, 1, 1, 1, 1, 2, 4 };
static	int		angle_24 [24];
static	int		nbphi_24 [24];


/***************************************************************************************************
 *
 *	Retourne le produit a traiter en fonction du Telemetry rate et du mode CIS
 *	--------------------------------------------------------------------------
 */
int	Produit_a_traiter (int instrument, int tlm_rate, int mode)
{
	t_prio *	ptr;

	for (ptr = priorites; ptr->instrument != -1; ptr++)

		if (ptr->instrument == instrument && ptr->tlm_rate == tlm_rate) return ptr->produit [mode];

	return -1;
}


/***************************************************************************************************
 *
 *	Verifie si la taille du produit correspond a la taille des donnees
 *	------------------------------------------------------------------
 */
int	Verifie_taille_produit (t_prod_header * entete, int size)
{
	char *		fonction = FNAME ("Verifie_taille_produit");
	int		code = OK;

	if (entete->taille_produit != size) {

		Affiche_erreur (fonction, "Taille produit : %d differente de %d octets",
			entete->taille_produit, size);
		code = ERROR;
	}
EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Initialise les informations concernant les angles transmis par produit
 *	----------------------------------------------------------------------
 *
 *	angle_88 [cellule]	numero angle solide (entre 0 et 87) correspondant aux 128 cellules
 *	angle_24 [cellule]	numero angle solide (entre 0 et 23) correspondant aux  64 cellules
 */
int	Initialise_coeff_angles (void)
{
	char *		fonction = FNAME ("Initialise_coeff_angles");
	int		code = OK;
	int		cellule = 0;
	int		angle = -1;
	int		anode, phi;

	for (anode = 0; anode < 8; anode ++) {

		for (phi = 0; phi < 16; phi ++, cellule ++) {

			if (phi % somme_88 [anode] == 0) angle ++;

			angle_88 [cellule] = angle;
		}
	}
EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Calcule les coefficients correcteurs pour passer en comptages corriges
 *	----------------------------------------------------------------------
 */
int	Initialise_coeff_correcteurs (void)
{
	char *		fonction = FNAME ("Initialise_coeff_correcteurs");
	int		code = OK;
	int		anode, energie;
	int		masse = 0;
	float		q = 1.0;

	Affiche_trace (1, fonction, "ABSEFF   = % 1.4e %1.4e", 
		cis1_absolute_efficiencies [LS][0],
		cis1_absolute_efficiencies [HS][0]);

	Affiche_trace (1, fonction, "POST ACC = % 1.4e", cis1_post_accel_volt);

	for (anode = 0; anode < NB_ANODE_CIS1; anode ++) {

		float	E, eff, M0, M1, M2, M3;

		/*	Calcul des tables d'efficacite LS en 16 et 31 energies
		 *	------------------------------------------------------
		 */
		M0 = cis1_anode_eff [LS][anode][0][masse];
		M1 = cis1_anode_eff [LS][anode][1][masse];
		M2 = cis1_anode_eff [LS][anode][2][masse];
		M3 = cis1_anode_eff [LS][anode][3][masse];

		for (energie = 0; energie < 16; energie ++) {

			E	= (cis1_energy_sweep_table_16 [0][energie].average + cis1_post_accel_volt) / 1000.0;

			eff	= (M0 + M1*E + M2*E*E + M3*E*E*E) * cis1_absolute_efficiencies [LS][masse];

			efficacite_16 [LS][anode][energie] = eff;
		}

		for (energie = 0; energie < 31; energie ++) {

			E	= (cis1_energy_sweep_table_31 [0][energie].average + cis1_post_accel_volt) / 1000.0;

			eff	= (M0 + M1*E + M2*E*E + M3*E*E*E) * cis1_absolute_efficiencies [LS][masse];

			efficacite_31 [LS][anode][energie] = eff;
		}

		/*	Calcul des tables d'efficacite HS en 16 et 31 energies
		 *	------------------------------------------------------
		 */
		M0 = cis1_anode_eff [HS][anode][0][masse];
		M1 = cis1_anode_eff [HS][anode][1][masse];
		M2 = cis1_anode_eff [HS][anode][2][masse];
		M3 = cis1_anode_eff [HS][anode][3][masse];

		for (energie = 0; energie < 16; energie ++) {

			E	= (cis1_energy_sweep_table_16 [0][energie].average + cis1_post_accel_volt) / 1000.0;

			eff	= (M0 + M1*E + M2*E*E + M3*E*E*E) * cis1_absolute_efficiencies [HS][masse];

			efficacite_16 [HS][anode][energie] = eff;
		}

		for (energie = 0; energie < 31; energie ++) {

			E	= (cis1_energy_sweep_table_31 [0][energie].average + cis1_post_accel_volt) / 1000.0;

			eff	= (M0 + M1*E + M2*E*E + M3*E*E*E) * cis1_absolute_efficiencies [HS][masse];

			efficacite_31 [HS][anode][energie] = eff;
		}

		/*	Calcul de la table d'efficacite RPA
		 *	-----------------------------------
		 */
		M0 = cis1_anode_eff [LS][anode][0][masse];
		M1 = cis1_anode_eff [LS][anode][1][masse];
		M2 = cis1_anode_eff [LS][anode][2][masse];
		M3 = cis1_anode_eff [LS][anode][3][masse];

		E = cis1_post_accel_volt / 1000.0;

		efficacite_rpa [LS][anode] = (M0 + M1*E + M2*E*E + M3*E*E*E) * cis1_absolute_efficiencies [LS][masse];

		M0 = cis1_anode_eff [HS][anode][0][masse];
		M1 = cis1_anode_eff [HS][anode][1][masse];
		M2 = cis1_anode_eff [HS][anode][2][masse];
		M3 = cis1_anode_eff [HS][anode][3][masse];

		efficacite_rpa [HS][anode] = (M0 + M1*E + M2*E*E + M3*E*E*E) * cis1_absolute_efficiencies [HS][masse];
	}

	Affiche_trace (1, fonction, "TABLE D'ENERGIE 16 E");

	for (energie = 0; energie < 16; energie ++) {

		Affiche_trace (1, fonction, "E%02d : % 1.4e", 
			energie,
			cis1_energy_sweep_table_16 [0][energie].average);
	}

	Affiche_trace (1, fonction, "COEFICIENTS POLYNOME CORRECTIF LS");

	for (anode = 0; anode < NB_ANODE_CIS1; anode ++) {

		Affiche_trace (1, fonction, "A%02d : % 1.4e % 1.4e % 1.4e % 1.4e",
			anode,
			cis1_anode_eff [LS][anode][0][masse],
			cis1_anode_eff [LS][anode][1][masse],
			cis1_anode_eff [LS][anode][2][masse],
			cis1_anode_eff [LS][anode][3][masse]);
	}

	Affiche_trace (1, fonction, "COEFICIENTS POLYNOME CORRECTIF HS");

	for (anode = 0; anode < NB_ANODE_CIS1; anode ++) {

		Affiche_trace (1, fonction, "A%02d : % 1.4e % 1.4e % 1.4e % 1.4e",
			anode,
			cis1_anode_eff [HS][anode][0][masse],
			cis1_anode_eff [HS][anode][1][masse],
			cis1_anode_eff [HS][anode][2][masse],
			cis1_anode_eff [HS][anode][3][masse]);
	}

	Affiche_trace (1, fonction, "TABLE CORRECTRICE LS 16 energies");

	for (energie = 0; energie < 16; energie ++) {

		Affiche_trace (1, fonction, "E%02d : % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e",
			energie,
			efficacite_16 [LS][0][energie],
			efficacite_16 [LS][1][energie],
			efficacite_16 [LS][2][energie],
			efficacite_16 [LS][3][energie],
			efficacite_16 [LS][4][energie],
			efficacite_16 [LS][5][energie],
			efficacite_16 [LS][6][energie],
			efficacite_16 [LS][7][energie]);
	}

	Affiche_trace (1, fonction, "TABLE CORRECTRICE HS 16 energies");

	for (energie = 0; energie < 16; energie ++) {

		Affiche_trace (1, fonction, "E%02d : % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e",
			energie,
			efficacite_16 [HS][0][energie],
			efficacite_16 [HS][1][energie],
			efficacite_16 [HS][2][energie],
			efficacite_16 [HS][3][energie],
			efficacite_16 [HS][4][energie],
			efficacite_16 [HS][5][energie],
			efficacite_16 [HS][6][energie],
			efficacite_16 [HS][7][energie]);
	}

	Affiche_trace (1, fonction, "RPA LS : % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e",
		efficacite_rpa [LS][0],
		efficacite_rpa [LS][1],
		efficacite_rpa [LS][2],
		efficacite_rpa [LS][3],
		efficacite_rpa [LS][4],
		efficacite_rpa [LS][5],
		efficacite_rpa [LS][6],
		efficacite_rpa [LS][7]);

	Affiche_trace (1, fonction, "RPA HS : % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e",
		efficacite_rpa [HS][0],
		efficacite_rpa [HS][1],
		efficacite_rpa [HS][2],
		efficacite_rpa [HS][3],
		efficacite_rpa [HS][4],
		efficacite_rpa [HS][5],
		efficacite_rpa [HS][6],
		efficacite_rpa [HS][7]);

EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Traitement du produit 12 de CODIF
 *	---------------------------------
 */
int	Spectro_COD_12 (t_prod_header * entete)
{
	char *		fonction = FNAME ("Spectro_COD_12");
	int		code = OK;
	int		cellule, energie;

	struct {
		short	K1;
		short	K2;
		int	data [88][16];
	}	data;

	if (Erreur (code = Verifie_taille_produit (entete, sizeof (data)))) goto EXIT;

	memcpy (& data, buffer, sizeof (data));

	for (energie = 0; energie < 16; energie++) {

		int	palier = (energie < 15) ? 16 : 32;
		int	nb_phi = 16;

		for (cellule = 0; cellule < 128; cellule ++) {

			int	anode	= cellule / nb_phi;
			float	delta_t	= cis1_accumulation_spin [entete->tlm_rate][entete->produit][entete->op_mode]
					* entete->pspin / 1000.0
					/ palier
					/ nb_phi;
			float	eff	= efficacite_16    [entete->sensitivity][anode][energie];
			float	g	= cis1_geom_factor [entete->sensitivity][anode][energie/2];

			if (eff != 0.0 && g != 0.0) {

				float F	= (float) data.data [angle_88 [cellule]][energie] 
					/ somme_88 [anode] 
					/ (eff * g * delta_t);

				work.spectro [energie] += F;
			}
		}
		work.spectro [energie] /= 128;
	}

EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Traitement du produit 13 de CODIF
 *	---------------------------------
 */
int	Spectro_COD_13 (t_prod_header * entete)
{
	char *		fonction = FNAME ("Spectro_COD_13");
	int		code = OK;
	int		cellule, energie;

	struct {
		short	K1;
		short	K2;
		int	data [88][31];
	}	data;

	if (Erreur (code = Verifie_taille_produit (entete, sizeof (data)))) goto EXIT;

	memcpy (& data, buffer, sizeof (data));
	
	for (energie = 0; energie < 31; energie++) {

		int	palier = 32;
		int	nb_phi = 16;

		for (cellule = 0; cellule < 128; cellule ++) {

			int	anode	= cellule / nb_phi;
			float	delta_t	= cis1_accumulation_spin [entete->tlm_rate][entete->produit][entete->op_mode]
					* entete->pspin / 1000.0
					/ palier
					/ nb_phi;
			float	eff	= efficacite_31    [entete->sensitivity][anode][energie];
			float	g	= cis1_geom_factor [entete->sensitivity][anode][energie/4];

			if (eff != 0.0 && g != 0.0) {

				float F	= (float) data.data [angle_88 [cellule]][energie] 
					/ somme_88 [anode]
					/ (eff * g * delta_t);

				work.spectro [energie] += F;
			}
		}
		work.spectro [energie] /= 128;
	}

EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Traitement du produit 14 de CODIF
 *	---------------------------------
 */
int	Spectro_COD_14 (t_prod_header * entete)
{
	char *		fonction = FNAME ("Spectro_COD_14");
	int		code = OK;
	int		cellule, energie;

	struct {
		short	K1;
		short	K2;
		int	data [24][31];
	}	data;

	if (Erreur (code = Verifie_taille_produit (entete, sizeof (data)))) goto EXIT;

	memcpy (& data, buffer, sizeof (data));
	
	/******	A TRAITER ULTERIEUREMENT ********************/

EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Traitement du produit 29 de CODIF
 *	---------------------------------
 */
int	Spectro_COD_29 (t_prod_header * entete)
{
	char *		fonction = FNAME ("Spectro_COD_29");
	int		code = OK;
	int		cellule, energie;
	int		energie_rpa = 15;
	int		masse = 0;

	struct {
		int	anode;
		int	data [4][88][16];
	}	data;

	if (Erreur (code = Verifie_taille_produit (entete, sizeof (data)))) goto EXIT;

	memcpy (& data, buffer, sizeof (data));
	
	for (energie = 0; energie < 16; energie++) {

		int	palier = 16;
		int	nb_phi = 16;

		for (cellule = 0; cellule < 128; cellule ++) {

			int	anode	= cellule / nb_phi;
			float	delta_t	= cis1_accumulation_spin [entete->tlm_rate][entete->produit][entete->op_mode]
					* entete->pspin / 1000.0
					/ palier
					/ nb_phi;
			float	eff	= efficacite_rpa   [entete->sensitivity][anode];
			float	g	= cis1_rpa_geom_factor [entete->sensitivity][anode][energie/2];

			if (eff != 0.0 && g != 0.0) {

				float F	= (float) data.data [masse][angle_88 [cellule]][energie] 
					/ somme_88 [anode] 
					/ (eff * g * delta_t);

				work.spectro [energie_rpa] += F;
			}
		}
		work.spectro [energie] /= 128;
	}

EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Initialise donnees enregistrement
 *	---------------------------------
 */
int	Initialise_spectro (t_prod_header * entete)
{
	char *		fonction = FNAME ("Initialise_spectro");
	int		code = OK;
	int		i;

	work.date	= entete->time_in_ms;
	work.instrument	= entete->instrument;
	work.produit	= entete->produit;
	work.cis_mode	= entete->op_mode;
	work.tlm_rate	= entete->tlm_rate;
	work.sensit	= entete->sensitivity;
	work.nombre	= 0;

	for (i = 0; i < 16; i++) work.spectro [i] = 0.0;
	
EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Memorise les resultats dans fichier temporaire
 *	----------------------------------------------
 */
int	Enregistre_spectro (FILE * sortie)
{
	char *		fonction = FNAME ("Enregistre_spectro");
	int		code = OK;

	if (fwrite (& work, sizeof (work), 1, sortie) != 1) {

		Affiche_erreur (fonction, "Erreur ecriture");
		code = ERR_fwrite;
	}
EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Generation spectrogramme pour un produit donne
 *	----------------------------------------------
 *
 *	Cree un fichier temporaire contenant les spectrogrammes d'un produit CODIF ou HIA donne
 *
 *	Lit les enregistrements d'un fichier issu du niveau 1
 *
 *	Si le produit a traiter pour le tlm_rate et le mode CIS correspond au produit courant, 
 *	transforme les donnees en valeurs physiques et les enregistre dans le fichier temporaire
 */
int	Traite_produit (t_descr * descr, double date_deb, double date_fin)
{
	char *		fonction = FNAME ("Traite_produit");
	int		code = OK;
	FILE *		entree = NULL;
	FILE *		sortie = NULL;
	int		lus, stat;
	t_prod_header	entete;
	char		cle [10];

	if ((entree = Ouverture_produit (descr->code, "r")) == NULL) {

		Affiche_erreur (fonction, "Lecture %s impossible", Nom_fichier_produit (descr->code));
		code = ERROR;
		goto EXIT;
	}

	sprintf (cle, "%s_%02d", Key_from_val (cis_instr, descr->instrument), descr->produit);

	if ((sortie = Ouverture_fichier_temporaire (cle, "w")) == NULL) {

		Affiche_erreur (fonction, "Ecriture fichier %s impossible", Nom_fichier_temporaire (cle));
		code = ERROR;
		goto EXIT;
	}

	stat = Lecture_entete_produit (& entete, entree);

	if (stat != OK) goto EXIT;

	if (Erreur (code = Lecture_calibration ("EFF", entete.time_in_ms))) goto EXIT;

	if (Erreur (code = Lecture_calibration (Key_from_val (cis_instr, descr->instrument), entete.time_in_ms))) goto EXIT;

	if (Erreur (code = Initialise_coeff_correcteurs ())) goto EXIT;

	while (stat == OK) {

		lus = fread (buffer, sizeof (char), entete.taille_produit, entree);

		if (lus != entete.taille_produit) {

			Affiche_erreur (fonction, "%d octets lus au lieu de %d", lus, entete.taille_produit);
			code = ERR_fread;
			goto EXIT;
		}

		if (Produit_a_traiter (entete.instrument, entete.tlm_rate, entete.op_mode) == descr->code) {

			if (entete.time_in_ms >= date_deb && entete.time_in_ms < date_fin) {

				code = Initialise_spectro (& entete);

				code = (* descr->traitement) (& entete);

				if (code == OK) code = Enregistre_spectro (sortie);
			}
		}
		stat = Lecture_entete_produit (& entete, entree);
	}

EXIT:	if (entree != NULL) fclose (entree);
	if (sortie != NULL) fclose (sortie);

	return code;
}


/***************************************************************************************************
 *
 *	Calcule intervalle d'une minute entourant une date
 *	--------------------------------------------------
 */
int	Calcule_intervalle (double date_ini, double * date_deb, double * date_fin)
{
	char *		fonction = FNAME ("Calcule_intervalle");
	int		code = OK;
	double		periode = 60000.0; 

	* date_deb = date_ini - fmod (date_ini, periode);
	* date_fin = * date_deb + periode;

EXIT:	return code;	
}


/***************************************************************************************************
 *
 *	Initialise cumul sur un intervalle
 *	----------------------------------
 */
int	Initialise_cumul (double date)
{
	char *		fonction = FNAME ("Initialise_cumul");
	int		code = OK;
	int		energie;

	for (energie = 0; energie < 16; energie++) work.spectro [energie] = 0.0;

	work.date	= date;
	work.nombre	= 0;

EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Calcule cumul sur un intervalle
 *	-------------------------------
 */
int	Calcule_cumul (t_record * ptr)
{
	char *		fonction = FNAME ("Calcule_cumul");
	int		code = OK;
	int		energie;

	for (energie = 0; energie < 16; energie++) work.spectro [energie] += ptr->spectro [energie];

	work.nombre ++;
	
EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Enregistre moyenne de l'intervalle sur disque
 *	---------------------------------------------
 */
int	Enregistre_cumul (FILE * sortie)
{
	char *		fonction = FNAME ("Enregistre_cumul");
	int		code = OK;
	int		energie;

	if (work.nombre > 0) {

		for (energie = 0; energie < 16; energie++) work.spectro [energie] /= work.nombre;

		Affiche_trace (3, fonction, "%s  Moyenne sur %d acquisitions",
			Milli_to_Ascii_time (work.date),
			work.nombre);

		Affiche_trace (3, fonction, "% 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e ",
			work.spectro [0], work.spectro [1], work.spectro [2], work.spectro [3],
			work.spectro [4], work.spectro [5], work.spectro [6], work.spectro [7]);

		Affiche_trace (3, fonction, "% 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e ",
			work.spectro [8], work.spectro [9], work.spectro [10], work.spectro [11],
			work.spectro [12], work.spectro [13], work.spectro [14], work.spectro [15]);

		if (fwrite (& work, sizeof (work), 1, sortie) != 1) {

			Affiche_erreur (fonction, "Erreur ecriture");
			code = ERR_fwrite;
			goto EXIT;
		}
		nbre_spectro ++;
	}
	else {	Affiche_erreur (fonction, "%s  Impossible calculer moyenne : duree accumulation nulle",
			Milli_to_Ascii_time (work.date));
		code = WARNING;
	}
	
EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Regroupement des enregistrements par minute
 *	-------------------------------------------
 *
 *	Pour un fichier temporaire contenant les acquisitions d'un produit CODIF ou HIA donne,
 *	regroupe les donnees a raison d'un enregistrement par minute
 *
 *	Enregistre les donnes dans un fichier commun a tous les produits
 */
int	Regroupe_produit (t_descr * descr, FILE * sortie)
{
	char *		fonction = FNAME ("Regroupe_produit");
	int		code = OK;
	FILE *		entree = NULL;
	int		lus, energie;
	char		cle [20];
	double		date_deb, date_fin;
	t_record	tmp;

	sprintf (cle, "%s_%02d", Key_from_val (cis_instr, descr->instrument), descr->produit);

	if ((entree = Ouverture_fichier_temporaire (cle, "r")) == NULL) {

		Affiche_erreur (fonction, "Lecture %s impossible", cle);
		code = ERR_fopen;
		goto EXIT;
	}

	lus = fread (& tmp, sizeof (tmp), 1, entree);

	code = Calcule_intervalle (tmp.date, & date_deb, & date_fin);

	code = Initialise_cumul (date_deb);

	while (lus == 1) {

		Affiche_trace (3, fonction, "%s %s %2d %s %s %02d %2d",
			Milli_to_Ascii_time (tmp.date),
			Key_from_val (cis_instr, tmp.instrument),
			tmp.produit,
			Key_from_val (cis_telemetry_rate, tmp.tlm_rate),
			Key_from_val (cis_sensitivity, tmp.sensit),
			tmp.cis_mode,
			tmp.nombre);

		Affiche_trace (3, fonction, "% 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e ",
			tmp.spectro [0], tmp.spectro [1], tmp.spectro [2], tmp.spectro [3],
			tmp.spectro [4], tmp.spectro [5], tmp.spectro [6], tmp.spectro [7]);

		Affiche_trace (3, fonction, "% 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e % 1.4e ",
			tmp.spectro [8], tmp.spectro [9], tmp.spectro [10], tmp.spectro [11],
			tmp.spectro [12], tmp.spectro [13], tmp.spectro [14], tmp.spectro [15]);

		code = Calcule_cumul (& tmp);

		lus = fread (& tmp, sizeof (tmp), 1, entree);

		if (lus != 1 || tmp.date > date_fin) {

			code = Enregistre_cumul (sortie);

			if (Erreur (code)) goto EXIT;

			code = Calcule_intervalle (tmp.date, & date_deb, & date_fin);

			code = Initialise_cumul (date_deb);
		}
	}

EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Fonction de comparaison utilisee par qsort
 *	------------------------------------------
 */
int	Compare_spectro (const void * gauche, const void * droite)
{
	t_record *	g = (t_record *) gauche;
	t_record *	d = (t_record *) droite;

	return g->date > d->date;
}


/***************************************************************************************************
 *
 *	Generation du fichier au format MIF
 *	-----------------------------------
 *
 *	Lecture du fichier binaire des spectrogrammes et chargement en memoire, puis tri de la table
 *	par ordre chronologique
 *
 *	Re-ecriture dans le meme fichier, mais en ASCII, d'une entete decrivant le contenu du fichier
 *	et des donnees spectrogrammes 
 */
int	Genere_fichier_MIF (void)
{
	char *		fonction = FNAME ("Genere_fichier_MIF");
	int		code = OK;
	FILE *		entree = NULL;
	FILE *		sortie = NULL;
	int		i, lus, energie;
	t_record *	table = NULL;
	time_t		secondes;
	char		start_time [30], end_time [30], gen_time [30];

	if ((entree = Ouverture_produit (JSOC_SPECTRO, "r")) == NULL) {

		Affiche_erreur (fonction, "Ouverture JSOC_SPECTRO impossible");
		code = ERR_fopen;
		goto EXIT;
	}

	if ((table = calloc (sizeof (t_record), nbre_spectro)) == NULL) {

		Affiche_erreur (fonction, "Erreur allocation dynamique de %d enregistrements", nbre_spectro);
		code = ERR_malloc;
		goto EXIT;
	}

	if ((lus = fread (table, sizeof (t_record), nbre_spectro, entree)) != nbre_spectro) {

		Affiche_erreur (fonction, "Erreur : %d spectro lus au lieu de %d", lus, nbre_spectro);
		code = ERR_fread;
		goto EXIT;
	}
	qsort (table, nbre_spectro, sizeof (t_record), Compare_spectro);

	fclose (entree);

	if ((sortie = Ouverture_produit (JSOC_SPECTRO, "w")) == NULL) {

		Affiche_erreur (fonction, "Ecriture JSOC_SPECTRO impossible");
		code = ERR_fopen;
		goto EXIT;
	}

	time (& secondes);

	strftime (gen_time, 30, "%Y-%m-%dT%TZ", localtime (& secondes));

	strcpy (start_time, Milli_to_Ascii_time (table [0].date));
	strcpy (end_time,   Milli_to_Ascii_time (table [nbre_spectro -1].date));

	fprintf (sortie, "! CIS data file for mission progress monitoring\n");
	fprintf (sortie, "! \n");
	fprintf (sortie, "! Energy Emax Emin\n");
	fprintf (sortie, "! \n");

	for (energie = 0; energie < 16; energie++) {

		fprintf (sortie, "! E%02d %10.2f %10.2f\n",
			energie,
			cis1_energy_sweep_table_16 [0][energie].upper,
			cis1_energy_sweep_table_16 [0][energie].lower);
	}
	fprintf (sortie, "! \n");
	fprintf (sortie, "! File header 1 - file details\n");
	fprintf (sortie, "! \n");
	fprintf (sortie, "WEB %03d %02d %1d %02d %s\n",
		0,
		0,
		param.satellite,
		1,
		gen_time);
	fprintf (sortie, "! \n");

	fprintf (sortie, "! File header 2 - instrument number_of_parameters start_time end_time\n");
	fprintf (sortie, "! \n");
	fprintf (sortie, "CIS 16 %s %s\n",
		start_time, end_time);
	fprintf (sortie, "! \n");

	fprintf (sortie, "! File header 3 - parameter names\n");
	fprintf (sortie, "! \n");
	fprintf (sortie, "\"Time\" ");
	fprintf (sortie, "\"E00\" \"E01\" \"E02\" \"E03\" \"E04\" \"E05\" \"E06\" \"E07\" ");
	fprintf (sortie, "\"E08\" \"E09\" \"E10\" \"E11\" \"E12\" \"E13\" \"E14\" \"E15\" \n");
	fprintf (sortie, "! \n");

	fprintf (sortie, "! File header 4 - parameter units\n");
	fprintf (sortie, "! \n");
	fprintf (sortie, "\"Time\" ");
	fprintf (sortie, "\"Ion Flux\" \"Ion Flux\" \"Ion Flux\" \"Ion Flux\" ");
	fprintf (sortie, "\"Ion Flux\" \"Ion Flux\" \"Ion Flux\" \"Ion Flux\" ");
	fprintf (sortie, "\"Ion Flux\" \"Ion Flux\" \"Ion Flux\" \"Ion Flux\" ");
	fprintf (sortie, "\"Ion Flux\" \"Ion Flux\" \"Ion Flux\" \"Ion Flux\" ");
	fprintf (sortie, "! \n");

	fprintf (sortie, "! File records\n");
	fprintf (sortie, "! \n");

	for (i = 0; i < nbre_spectro; i++) {

		fprintf (sortie, "%s ", Milli_to_Ascii_time (table [i].date));

		for (energie = 0; energie < 16; energie++)
			fprintf (sortie, "% 1.4e ", table [i].spectro [energie]);

		fprintf (sortie, "\n");
	}

EXIT:	if (entree != NULL)	fclose (entree);
	if (sortie != NULL)	fclose (sortie);
	if (table  != NULL)	free (table);

	return code;	
}


/***************************************************************************************************
 *
 *	Creation du fichier spectro specifique au JSOC
 *	----------------------------------------------
 */
int	Creation_JSOC_spectro (void)
{
	char *		fonction = FNAME ("Creation_JSOC_spectro");
	int		code = OK;
	char *		fichier;
	t_descr *	ptr;
	FILE *		sortie = NULL;
	char *		variable;
	int		instrument;
	double		date_deb, date_fin;

	if ((variable = getenv ("START_TIME")) == NULL) {

		Affiche_erreur (fonction, "Variable START_TIME inconnue");
		code = ERROR;
		goto EXIT;
	}

	date_deb = Ascii_time_to_milli (variable);

	if ((variable = getenv ("END_TIME")) == NULL) {

		Affiche_erreur (fonction, "Variable END_TIME inconnue");
		code = ERROR;
		goto EXIT;
	}

	date_fin = Ascii_time_to_milli (variable);

	if ((variable = getenv ("JSOC_INSTR")) == NULL) {

		Affiche_erreur (fonction, "Variable JSOC_INSTR inconnue");
		code = ERROR;
		goto EXIT;
	}

	if ((instrument = Val_from_key (cis_instr, variable)) == EOF) {

		Affiche_erreur (fonction, "Instrument %s inconnu", variable);
		code = ERROR;
		goto EXIT;
	}

	if ((sortie = Ouverture_produit (JSOC_SPECTRO, "w")) == NULL) {

		Affiche_erreur (fonction, "Creation JSOC_SPECTRO impossible");
		code = ERR_fopen;
		goto EXIT;
	}

	Affiche_trace (1, fonction, "Instrument traite : %s", Key_from_val (cis_instr, instrument));
	Affiche_trace (1, fonction, "Debut traitement  : %s", Milli_to_Ascii_time (date_deb));
	Affiche_trace (1, fonction, "Fin   traitement  : %s", Milli_to_Ascii_time (date_fin));

	code = Ouverture_calibration (param.satellite, Nom_fichier_produit (CALIB_CAT), contexte.version_calibration);

	if (Erreur (code)) goto EXIT;

	code = Initialise_coeff_angles ();

	nbre_spectro = 0;

	for (ptr = liste_produit; ptr->instrument != -1; ptr++) {

		if (ptr->instrument != instrument) continue;

		fichier = Nom_fichier_produit (ptr->code);

		if (strlen (fichier) == 0) {

			Affiche_erreur (fonction, "Produit %s %2d : Fichier absent", 
				Key_from_val (cis_instr, ptr->instrument), ptr->produit);
			continue;
		}

		if (ptr->traitement == NULL) {

			Affiche_erreur (fonction, "Produit %s %2d : Traitement non defini", 
				Key_from_val (cis_instr, ptr->instrument), ptr->produit);
			continue;
		}

		Affiche_trace (1, fonction, "%s", fichier);

		code = Traite_produit (ptr, date_deb, date_fin);

		if (Erreur (code)) continue;

		code = Regroupe_produit (ptr, sortie);
	}

	fclose (sortie);

	if (Erreur (code = Cloture_calibration ())) goto EXIT;;

	if (Erreur (code = Genere_fichier_MIF ())) goto EXIT;

EXIT:	if (sortie != NULL) fclose (sortie);
	return code;
}
