/***************************************************************************************************
 *
 *	Fichier	: $RCSfile: INTERFACE.c,v $
 *
 *	Version	: $Revision: 1.14 $
 *
 *	Auteur	: $Author: penou $
 *
 *	Date	: $Date: 2007/07/19 10:35:57 $
 *
 *	==========================================================================================
 *
 *	Ce module est charge de l'interface entre la chaine de traitement 
 *	et le systeme d'exploitation.
 *
 *	Il comprend notamment :
 *	- le traitement des parametres de la ligne de commande,
 *	- l'ouverture des fichiers de donnees de l'application
 *	- la gestion des fichiers temporaires
 */

#define	MODULE_NAME	"INTERFACE"

#include "CISLIB.h"
#include "INTERFACE.h"

#define	NBRE_PRODUITS	(14 + 12 + 2 + 3 + 2 + 1 + 6)


typedef struct {				/* Structure utilisee pour verifier	*/
	int		satellite;		/* la coherence des noms des fichiers	*/
	int		jour, mois, annee;
}	t_data;


typedef struct {				/* Table des fichiers de l'application	*/
	t_produit	produit;		/* - type du produit			*/
	int		option_date;		/* - date du jour necessaire		*/
	char		*format;		/* - format nom du fichier		*/
	t_filename	fichier;		/* - nom du fichier correspondant	*/
	int		genere;			/* - indicateur fichier genere		*/
	FILE		*f_desc;		/* - descripteur interne		*/
}	t_prod;


t_param	param;					/* Parametres de l'application		*/


t_prod	table_produits [NBRE_PRODUITS] = {

	/*	Produits CODIF de niveau 1	(14)
	 */
	{ COD_12,	TRUE,	"C%1d_L1_CISCOD_%8s_V%2s_3DP16E_12x.sdd" },
	{ COD_13,	TRUE,	"C%1d_L1_CISCOD_%8s_V%2s_3DP31E_13x.sdd" },
	{ COD_14,	TRUE,	"C%1d_L1_CISCOD_%8s_V%2s_3DP24A_14x.sdd" },
	{ COD_15,	TRUE,	"C%1d_L1_CISCOD_%8s_V%2s_3DA16E_15x.sdd" },
	{ COD_16,	TRUE,	"C%1d_L1_CISCOD_%8s_V%2s_3DA31E_16x.sdd" },
	{ COD_17,	TRUE,	"C%1d_L1_CISCOD_%8s_V%2s_3DHM16E_17.sdd" },
	{ COD_18,	TRUE,	"C%1d_L1_CISCOD_%8s_V%2s_3DHM31E_18.sdd" },
	{ COD_19,	TRUE,	"C%1d_L1_CISCOD_%8s_V%2s_2D4M31E_19.sdd" },
	{ COD_20,	TRUE,	"C%1d_L1_CISCOD_%8s_V%2s_2D4M16E_20.sdd" },
	{ COD_21,	TRUE,	"C%1d_L1_CISCOD_%8s_V%2s_2D2M16E_21.sdd" },
	{ COD_22,	TRUE,	"C%1d_L1_CISCOD_%8s_V%2s_2D1M31E_22.sdd" },
	{ COD_29,	TRUE,	"C%1d_L1_CISCOD_%8s_V%2s_RPA16E_29x.sdd" },
	{ COD_47,	TRUE,	"C%1d_L1_CISCOD_%8s_V%2s_3DO16E_47x.sdd" },
	{ COD_49,	TRUE,	"C%1d_L1_CISCOD_%8s_V%2s_3DO31E_49x.sdd" },

	/*	Produits HIA de niveau 1	(12)
	 */
	{ HIA_05,	TRUE,	"C%1d_L1_CISHIA_%8s_V%2s_3DM62E_5xx.sdd" },
	{ HIA_06,	TRUE,	"C%1d_L1_CISHIA_%8s_V%2s_3D31E88A_6.sdd" },
	{ HIA_08,	TRUE,	"C%1d_L1_CISHIA_%8s_V%2s_3DCOLD_8xx.sdd" },
	{ HIA_10,	TRUE,	"C%1d_L1_CISHIA_%8s_V%2s_2DAZ_10xxx.sdd" },
	{ HIA_14,	TRUE,	"C%1d_L1_CISHIA_%8s_V%2s_2DAZCLD_14.sdd" },
	{ HIA_15,	TRUE,	"C%1d_L1_CISHIA_%8s_V%2s_3DH16E_15x.sdd" },
	{ HIA_16,	TRUE,	"C%1d_L1_CISHIA_%8s_V%2s_3DH30E_16x.sdd" },
	{ HIA_17,	TRUE,	"C%1d_L1_CISHIA_%8s_V%2s_3DH62E_17x.sdd" },
	{ HIA_21,	TRUE,	"C%1d_L1_CISHIA_%8s_V%2s_3DH31E_21x.sdd" },
	{ HIA_22,	TRUE,	"C%1d_L1_CISHIA_%8s_V%2s_3DC31E_22x.sdd" },
	{ HIA_23,	TRUE,	"C%1d_L1_CISHIA_%8s_V%2s_COMP3DH_23.sdd" },
	{ HIA_24,	TRUE,	"C%1d_L1_CISHIA_%8s_V%2s_COMP3DC_24.sdd" },

	/*	Produits AUX de niveau 1	(2)
	 */
	{ AUX_STOF,	TRUE,	"C%1d_L1_CISAUX_%8s_V%2s_STOFxxxxxx.axx" },
	{ AUX_SATT,	TRUE,	"C%1d_L1_CISAUX_%8s_V%2s_SATTxxxxxx.axx" },

	/*	Produits divers de niveau 1	(3)
	 */
	{ CALIB_CAT,	FALSE,	"C%1d_L1_CISxxx_%8s_V%2s_CALCATALOG.cal" },
	{ ENVIRON,	TRUE,	"C%1d_L1_CISENV_%8s_V%2s_ENVIRONMTx.sdd" },
	{ ARCHIVE,	TRUE,	"C%1d_L1_CISARC_%8s_V%2s_ARCHIVExxx.arc" },

	/* 	Produits niveau 2		(2)
	 */
	{ PPD, 		TRUE,	"C%1d_L2_CISxxx_%8s_V%2s_PPDxxxxxxx.flt" },
	{ FGM,		TRUE,	"C%1d_L2_FGM_%8s_V%2s.asc"               },

	/* 	Fichier Contexte		(1)
	 */
	{ CONTEXT,	FALSE,	"C%1d_Lx_CISCON_%8s_V%2s_CONTEXTExx.asc" }, 	

	/*	Produits niveau 3		(6)
	 */
	{ TRACE,	TRUE,	"C%1d_L3_CISxxx_%8s_V%2s_TRACExxxxx.tra" },
	{ COD_SPECTRO,	TRUE,	"C%1d_L3_CISCOD_%8s_V%2s_CTMSPECTRO.ps"	 },
	{ HIA_SPECTRO,	TRUE,	"C%1d_L3_CISHIA_%8s_V%2s_CTMSPECTRO.ps"	 },
	{ JSOC_SPECTRO,	TRUE,	"C%1d_L3_CISxxx_%8s_V%2s_JSOCSPECTR.txt" },	/* Specifique JSOC	*/
	{ PPD_BIN,	TRUE,	"C%1d_L3_CISxxx_%8s_V%2s_PPDxxxxxxx.sdd" },	/* Specifique CESR	*/
	{ ORBIT,	TRUE,	"C%1d_L3_CISxxx_%8s_V%2s_ORBITxxxxx.sdd" }	/* Specifique CESR	*/
};


/***************************************************************************************************
 *
 *	Initialisation de la table des fichiers produits
 *	------------------------------------------------
 */
int	Initialise_table_produits (void)
{
	char *		fonction = FNAME ("Initialise_table_produits");
	int		code = OK;
	int		i;

	for (i = 0; i < NBRE_PRODUITS; i++) {

		strcpy (table_produits [i].fichier, "");
		table_produits [i].f_desc = (FILE *) NULL;
		table_produits [i].genere = FALSE;
	}
	return code;
}


/***************************************************************************************************
 *
 *	Decompose un nom de fichier
 *	---------------------------
 */
t_data * Decompose_nom_fichier (char * fichier, char * format)
{
	char *		fonction = FNAME ("Decompose_nom_fichier");
	static t_data	fd;
	char *		p;
	char		date [9], version [3];
	int		satellite, jour, mois, annee;
	t_filename	buffer;

	p = strrchr (fichier, '/');

	p = (p == NULL) ? fichier : p + 1;

	fd.satellite = fd.jour = fd.mois = fd.annee = 0;

	if (sscanf (p, format, & satellite, date, version) != 3) return (t_data *) NULL;

	fd.satellite = satellite;

	if (sscanf (date, "%4d%2d%2d", & annee, & mois, & jour) == 3) {

		fd.jour  = jour;
		fd.mois  = mois;
		fd.annee = annee;
	}

	sprintf (buffer, format, satellite, date, version);

	if (strcmp (buffer, p) == 0) return & fd;

	return (t_data *) NULL;
}


/***************************************************************************************************
 *
 *	Chargement du nom de fichier dans la table des produits
 *	-------------------------------------------------------
 */
int	Enregistre_fichier (char * fichier)
{
	char *		fonction = FNAME ("Enregistre_fichier");
	int		code = OK;
	t_data *	ptr;
	int		i, trv = -1;


	for (i=0; i < NBRE_PRODUITS && trv == -1; i++) 
		if ((ptr = Decompose_nom_fichier (fichier, table_produits [i].format)) != NULL) trv = i;

	if (trv == -1) { 

		Affiche_trace (1, fonction, "%s incorrect", fichier);
		code = WARNING;
		goto EXIT;
	}

	/*	Ne pas tenir compte de la date pour certains des fichiers
	 */
	if (table_produits [trv].option_date == FALSE) {

		ptr->annee = param.annee;
		ptr->mois  = param.mois;
		ptr->jour  = param.jour;
	}

	if (ptr->satellite != param.satellite) {

		Affiche_trace (1, fonction, "%s", fichier);
		Affiche_trace (1, fonction, "Numero satellite %d incorrect", ptr->satellite);
		code = WARNING;
		goto EXIT;
	}

	if (ptr->annee != param.annee || ptr->mois != param.mois || ptr->jour != param.jour) {

		Affiche_trace (1, fonction, "%s", fichier);
		Affiche_trace (1, fonction, "Date fichier %02d/%02d/%04d incorrecte", 
			ptr->jour, ptr->mois, ptr->annee);
		code = WARNING;
		goto EXIT;
	}

	Affiche_trace (1, fonction, "%s OK", fichier);

	strcpy (table_produits [trv].fichier, fichier);

EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Retourne le numero du produit dans la table
 *	-------------------------------------------
 */
int	Numero_produit (t_produit produit)
{
	char *		fonction = FNAME ("Numero_produit");
	int		i;

	for (i=0; i < NBRE_PRODUITS; i++) if (table_produits [i].produit == produit) return i;

	return -1;
}


/***************************************************************************************************
 *
 *	Retourne le nom du fichier produit desire
 *	-----------------------------------------
 */
char *	Nom_fichier_produit (t_produit produit)
{
	char *		fonction = FNAME ("Nom_fichier_produit");
	int		trv;
	static t_filename	file_name;

	strcpy (file_name, "");

	trv = Numero_produit (produit);

	if (trv != -1) strcpy (file_name, table_produits [trv].fichier);

	return file_name;
}


/***************************************************************************************************
 *
 *	Ouverture d'un fichier produit
 *	------------------------------
 */
FILE *	Ouverture_produit (t_produit produit, char * mode)
{
	char *		fonction = FNAME ("Ouverture_produit");
	int		numero;
	t_prod *	ptr;

	numero = Numero_produit (produit);

	if (numero == -1) 
		Affiche_trace (1, fonction, "Produit inexistant");
	else {
		ptr = & table_produits [numero];

		if (strlen (ptr->fichier) == 0) 
			Affiche_trace (1, fonction, "Produit absent");
		else {
			ptr->f_desc = fopen (ptr->fichier, mode);

			if (ptr->f_desc == NULL) 
				Affiche_trace (5, fonction, "Ouverture %s impossible", ptr->fichier);
			else {
				if (strchr (mode, 'w') != NULL || strchr (mode, 'a') != NULL) ptr->genere = TRUE;
				return (FILE *) ptr->f_desc;
			}
		}
	}
	return (FILE *) NULL;
}


/***************************************************************************************************
 *
 *	Retourne le nom du fichier temporaire associe a la cle
 *	------------------------------------------------------
 */
char *	Nom_fichier_temporaire (char * cle)
{
	char *		fonction = FNAME ("Nom_fichier_temporaire");
	static t_filename	buffer;

	sprintf (buffer, "%s/C%1d_L3_%04d%02d%02d_%s.tmp",
		param.espace_traitement, param.satellite, param.annee, param.mois, param.jour, cle);

	return  buffer;
}


/***************************************************************************************************
 *
 *	Ouverture fichier temporaire
 *	----------------------------
 */
FILE *	Ouverture_fichier_temporaire (char * cle, char * mode)
{
	char *		fonction = FNAME ("Ouverture_fichier_temporaire");
	FILE *		pf = NULL;
	char *		fichier;

	fichier = Nom_fichier_temporaire (cle);

	if ((pf = fopen (fichier, mode)) == NULL) 
		Affiche_erreur (fonction, "Ouverture fichier %s impossible", fichier);

	return (FILE *) pf;
}


/***************************************************************************************************
 *
 *	Chargement des parametres de la ligne de commande
 *	-------------------------------------------------
 */
int	Chargement_parametres (char * file_1, char * file_2, char * file_3, char * file_4, char * file_5)
{
	char *		fonction = FNAME ("Chargement_parametres");
	int		code = OK;
	t_filename	buffer;
	int		jour, mois, annee;
	FILE *		pf = NULL;
	struct stat	info;

	if (Erreur (code = Initialise_table_produits ())) goto EXIT;

	/*	Parametre d'execution
	 *	---------------------
	 */
	Affiche_trace (1, fonction, "PARAMETRES D'EXECUTION ...");

	if ((pf = fopen (file_2, "rt")) == NULL) {

		Affiche_erreur (fonction, "Ouverture fichier %s impossible", file_2);
		code = ERR_parameter_file;
		goto EXIT;
	}
	code = Read_line (buffer, pf);

	fclose (pf);

	if (code == OK && sscanf (buffer, "%4d%2d%2d", & annee, & mois, & jour) != 3) {

		Affiche_erreur (fonction, "Date traitement %s incorrecte", buffer);
		code = ERROR;
		goto EXIT;
	}
	Affiche_trace (1, fonction, "Date traitement = %02d/%02d/%04d", jour, mois, annee);

	param.annee	= annee;
	param.mois	= mois;
	param.jour	= jour;

	/*	Espace de traitement
	 *	--------------------
	 */
	Affiche_trace (1, fonction, "ESPACE DE TRAITEMENT ...");

	if (stat (file_4, & info) == -1 || ! S_ISDIR (info.st_mode)) {

		Affiche_erreur (fonction, "Acces repertoire %s impossible", file_4);
		code = ERR_tmp_directory;
		goto EXIT;
	}
	
	if (access (file_4, R_OK | W_OK) != 0) {

		Affiche_erreur (fonction, "Droits d'acces sur repertoire %s insuffisants", file_4);
		code = ERR_tmp_directory;
		goto EXIT;
	}

	Affiche_trace (1, fonction, "Acces repertoire %s OK", file_4);

	strcpy (param.espace_traitement, file_4);

	/*	Liste des fichiers d'entree
	 *	---------------------------
	 */
	Affiche_trace (1, fonction, "CHARGEMENT FICHIERS D'ENTREE ...");

	if ((pf = fopen (file_1, "rt")) == NULL) {

		Affiche_erreur (fonction, "Ouverture fichier %s impossible", file_1);
		code = ERR_input_file;
		goto EXIT;
	}
	while (Read_line (buffer, pf) == OK) {

		if (access (buffer, R_OK) != 0)
			Affiche_erreur (fonction, "%s Lecture impossible", buffer);
		else
			code = Enregistre_fichier (buffer);
	}
	fclose (pf);

	/*	Liste des fichiers attendus
	 *	---------------------------
	 */
	Affiche_trace (1, fonction, "CHARGEMENT FICHIERS ATTENDUS ...");

	if ((pf = fopen (file_3, "rt")) == NULL) {

		Affiche_erreur (fonction, "Ouverture fichier %s impossible", file_3);
		code = ERR_expected_file;
		goto EXIT;
	}
	while (Read_line (buffer, pf) == OK) {

		code = Enregistre_fichier (buffer);
	}
	fclose (pf);

	/*	Liste des fichiers generes
	 *	--------------------------
	 */
	strcpy (param.resultat, file_5);

	Affiche_trace (1, fonction, "LISTE FICHIERS GENERES ...");
	Affiche_trace (1, fonction, "Resultat dans fichier %s", file_5);

	Affiche_trace (1, fonction, "CHARGEMENT PARAMETRES OK");

EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Enregistre sur fichier la liste des fichiers generes
 *	----------------------------------------------------
 */
int	Genere_liste_sortie (char * fichier)
{
	char *		fonction = FNAME ("Genere_liste_sortie");
	int		code = 0;
	FILE *		pf = NULL;
	int		i;

	if ((pf = fopen (fichier, "wt")) == NULL) {

		Affiche_erreur (fonction, "Creation fichier %s impossible", fichier);
		code = ERR_generated_file;
		goto EXIT;
	}

	for (i = 0; i < NBRE_PRODUITS; i++) 
		if (table_produits [i].genere == TRUE) fprintf (pf, "%s\n", table_produits [i].fichier);

EXIT:	if (pf != NULL) fclose (pf);
	
	return code;
}


/***************************************************************************************************
 *
 *	Renvoie le nom de la variable representant le repertoire d'origine du produit
 *	-----------------------------------------------------------------------------
 */
int	Repertoire_origine (t_produit produit, t_filename * rep)
{
	char *		fonction = FNAME ("Repertoire_origine");
	int		code = OK;

	switch (produit) {

		case COD_12:
		case COD_13:
		case COD_14:
		case COD_15:
		case COD_16:
		case COD_17:
		case COD_18:
		case COD_19:
		case COD_20:
		case COD_21:
		case COD_22:
		case COD_29:
		case COD_47:
		case COD_49:
		case HIA_05:
		case HIA_06:
		case HIA_08:
		case HIA_10:
		case HIA_14:
		case HIA_15:
		case HIA_16:
		case HIA_17:
		case HIA_21:
		case HIA_22:
		case HIA_23:
		case HIA_24:	
		case AUX_STOF:	
		case AUX_SATT:
		case ENVIRON:
		case ARCHIVE:		strcpy ((char *) rep, "${REP_N1}");	break;

		case CALIB_CAT:		strcpy ((char *) rep, "${REP_CALIB}");	break;

		case PPD:		strcpy ((char *) rep, "${REP_N2}");	break;


		case CONTEXT:		strcpy ((char *) rep, "${REP_CTXT}");	break;

		case FGM:		strcpy ((char *) rep, "${REP_FGM}");	break;

		case TRACE:
		case COD_SPECTRO:
		case HIA_SPECTRO:	
		case JSOC_SPECTRO:	
		case PPD_BIN:
		case ORBIT:		strcpy ((char *) rep, "${REP_N3}");	break;

		default:		Affiche_erreur (fonction, "Produit %d inconnu", produit);
					code = ERROR;
	}

EXIT:	return code;
}


/***************************************************************************************************
 *
 *	Genere les masques des fichiers d'entree et attendus
 *	----------------------------------------------------
 */
int	Genere_masques_fichiers (char * fichier_entree, char * fichier_attendu)
{
	char *		fonction = FNAME ("Genere_masques_fichiers");
	int		code = OK;
	FILE *		entree = NULL;
	FILE *		sortie = NULL;
	int		i;
	char *		p;
	char *		delim;
	t_filename	format, repertoire;

	if ((entree = fopen (fichier_entree, "w")) == NULL) {

		Affiche_erreur (fonction, "Creation fichier %s impossible", fichier_entree);
		code = ERR_fopen;
		goto EXIT;
	}

	if ((sortie = fopen (fichier_attendu, "w")) == NULL) {

		Affiche_erreur (fonction, "Creation fichier %s impossible", fichier_attendu);
		code = ERR_fopen;
		goto EXIT;
	}

	for (i = 0; i < NBRE_PRODUITS; i++) {

		code = Repertoire_origine (table_produits [i].produit, & repertoire);

		if (Erreur (code)) goto EXIT;

		strcpy (format, "");

		p = table_produits [i].format;

		delim = strchr (p, '%');

		while (delim != NULL) {

			strncat (format, p, delim - p);

			if (strncmp (delim, "%1d", 3) == 0) strcat (format, "${SAT}");

			if (strncmp (delim, "%8s", 3) == 0) {

				if (table_produits [i].option_date) 
					strcat (format, "${DATE_JOUR}");
				else	strcat (format, "${DATE_QLCQ}");
			}

			if (strncmp (delim, "%2s", 3) == 0){

				if (table_produits [i].option_date)
					strcat (format, "${VERSION}");
				else	strcat (format, "${VER_MAX}");
			}

			p = (char *) delim + 3;

			delim = strchr (p, '%');
		}
		strcat (format, p);

		if (strcmp (repertoire, "${REP_N3}") == 0)
			fprintf (sortie, "%s/%s\n", repertoire, format);
		else	fprintf (entree, "%s/%s\n", repertoire, format);
	}

EXIT:	if (entree != NULL)	fclose (entree);
	if (sortie != NULL)	fclose (sortie);
	return code;
}
