;-------------------------------------------------------------------------------
;
;	Fichier	: $RCSfile: libcdf.pro,v $, v $Revision: 1.28 $
;
;	Date	: $Date: 2021/06/10 09:33:31 $
;
;	Auteur	: $Author: penou $
;
;	Version : %Z% version %I% de %M% du %G%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
FUNCTION marray_cdf,	$
;-------------------------------------------------------------------------------
	xx	; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((-)) Retourne le produit des lments d'un tableau.
;-------------------------------------------------------------------------------

	ON_ERROR,1

	n = N_ELEMENTS(xx)
	aa = LONG (xx[0])
	FOR i = 1L,n-1 DO aa *= xx[i]

	RETURN, aa

END

;-------------------------------------------------------------------------------
PRO getnbxy_cdf,	$
;-------------------------------------------------------------------------------
	var,		$	; LINT_PROTOTYPE input
	nbX,		$	; LINT_PROTOTYPE output
	nbY			; LINT_PROTOTYPE output
;-------------------------------------------------------------------------------
; ((-)) Retourne le nombre de points en X et le nombre de courbes en Y d'une variable CDF.
;
; Entrees: var: variable CDF lue
; Sorties: nbX: nombre de points en abscisse
;          nbY: nombre de courbes
;-------------------------------------------------------------------------------

	dimensions = SIZE (var, /DIMENSIONS)

	nbX = dimensions[0]
	nbY = 1L
	FOR i=1L, N_ELEMENTS(dimensions)-1 DO nbY *= dimensions[i]

END


;-------------------------------------------------------------------------------
FUNCTION transferer_cdf,	$
;-------------------------------------------------------------------------------
	data,	$	; LINT_PROTOTYPE input
	k,	$	; LINT_PROTOTYPE input
	var		; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((1:ok 2:warning)) Transfre var dans data.(k), mme si var et data n'ont pas le mme nombre d'lments.
;-------------------------------------------------------------------------------

	IF N_ELEMENTS(var) EQ N_ELEMENTS(data) THEN BEGIN
		data.(k) = REFORM(var)
		RETURN, 1
	END ELSE BEGIN
		data.(k) = -1e31
		RETURN, 2
	END

END


;-------------------------------------------------------------------------------
FUNCTION ouvrir_cdf_c,	$
;-------------------------------------------------------------------------------
	filename	; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((0:ok -1:pb)) Ouvre un fichier CDF en c.
;-------------------------------------------------------------------------------

	IF is_fdl() THEN BEGIN
		code = CALL_EXTERNAL (libname('filecdf'), 'OUVRIR_CDF_C_AUTO_GLUE', filename, /AUTO_GLUE)
	END ELSE BEGIN
		code = CALL_EXTERNAL (libname('filecdf'), 'OUVRIR_CDF_C', filename)
	END
	RETURN, code

END


;-------------------------------------------------------------------------------
FUNCTION fermer_cdf_c,	$
;-------------------------------------------------------------------------------
	filename	; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((O: toujours)) Ferme un fichier CDF en c.
;-------------------------------------------------------------------------------

	IF is_fdl() THEN BEGIN
		code = CALL_EXTERNAL (libname('filecdf'), 'FERMER_CDF_C_AUTO_GLUE', filename, /AUTO_GLUE)
	END ELSE BEGIN
		code = CALL_EXTERNAL (libname('filecdf'), 'FERMER_CDF_C', filename)
	END

	RETURN, code

END


;-------------------------------------------------------------------------------
PRO get_info_var_cdf_c,	$
;-------------------------------------------------------------------------------
	filename,	$	; LINT_PROTOTYPE input
	nom,		$	; LINT_PROTOTYPE input
	nblignes,	$	; LINT_PROTOTYPE output
	datatype,	$	; LINT_PROTOTYPE output
	nbdim,		$	; LINT_PROTOTYPE output
	dimsizes,	$	; LINT_PROTOTYPE output
	majority,	$	; LINT_PROTOTYPE output
	quick=quick		; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((-)) Retourne nlignes, datatype, nbdim, dimsizes et majority d'une variable CDF.
;-------------------------------------------------------------------------------

	IF N_ELEMENTS(quick) EQ 0 THEN quick = 0

	nblignesonly = 1L
	nblignes = 0L
	datatype = 0L
	nbdim = 0L
	dimsizes = REPLICATE (0L, 8)
	majority = 0b
	ouvrir_fermer = LONG(~quick)
	offset = 0L
	REC_START = 0L
	REC_COUNT = 0L
	taille = 0L
	type_champ = 0L
	offset_champ = 0L
	data = 0b
	info_alire = -2L
	validmin = 0.
	validmax = 0.
	scalemin = 0.
	scalemax = 0.
	zero = 0L
	IF is_fdl() THEN BEGIN
		lint_unused = CALL_EXTERNAL (libname('filecdf'), 'LIRE_VAR_CDF_C_AUTO_GLUE', filename, ouvrir_fermer, offset, REC_START, REC_COUNT, nom,	$
						data, taille, type_champ, offset_champ, nblignesonly, nblignes, info_alire, 					$
						datatype, nbdim, dimsizes, majority, validmin, validmax, scalemin, scalemax, 1L, /AUTO_GLUE)
	END ELSE BEGIN
		lint_unused = CALL_EXTERNAL (libname('filecdf'), 'LIRE_VAR_CDF_C', filename, ouvrir_fermer, offset, REC_START, REC_COUNT, nom, 			$
						data, taille, type_champ, offset_champ, nblignesonly, nblignes, info_alire, 					$
						datatype, nbdim, dimsizes, majority, validmin, validmax, scalemin, scalemax, zero)
	END
	majority = STRING(majority)

END

;-------------------------------------------------------------------------------
PRO lire_var_cdf_c,	$
;-------------------------------------------------------------------------------
	filename,	$	; LINT_PROTOTYPE input
	data,		$	; LINT_PROTOTYPE input
	nom,		$	; LINT_PROTOTYPE input
	REC_START,	$	; LINT_PROTOTYPE input
	REC_COUNT,	$	; LINT_PROTOTYPE input
	quick=quick		; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((-)) Retourne un tableau avec les donnes d'une variable CDF.
;
; Entres:
;
;	filename:	fichier CDF
;	nom:		noms de la variable CDF  lire
;	REC_START:	commencer  lire dans le CDF  l'enregistrement REC_START
;	REC_COUNT:	lire REC_COUNT enregistrements dans le fichier cdf
;
; Entrs/Sorties:
;
;	data:		tableau existant en entre, renseign en sortie
;-------------------------------------------------------------------------------

	IF N_ELEMENTS(quick) EQ 0 THEN quick = 0

	nblignes = 0L
	nblignesonly = 0L
	datatype = 0L

	tmp = SIZE (data[0])
	; tmp contient le nombre de dimension puis la taille de chaque dimension puis le code type puis le nombre d'lments
	typecode = tmp[0]+1
	CASE tmp[typecode] OF
		 1:	taille = 1 ; Byte
		 2:	taille = 2 ; Integer
		 3:	taille = 4 ; Longword integer
		 4:	taille = 4 ; Floating point
		 5:	taille = 8 ; Double-precision floating
		12:	taille = 2 ; Unsigned integer
		13:	taille = 4 ; Unsigned Longword integer
		ELSE:	taille = 0
	END
	nbdim = 0L
	dimsizes = REPLICATE (0L, 8)
	majority = 0b
	ouvrir_fermer = LONG(~quick)
	validmin = 0.
	validmax = 0.
	scalemin = 0.
	scalemax = 0.
	moins2 = -2L
	IF is_fdl() THEN BEGIN
		is_str = is_string(data)
		IF is_str THEN BEGIN
			nb_str = N_ELEMENTS (data)
			; crer une structure pour que GDL passe data par adresse
			data = { str: REPLICATE (STRING(REPLICATE(BYTE('A'),1024)), nb_str) } ; rserver 1024 bytes pour les STRING
		END
		lint_unused = CALL_EXTERNAL (libname('filecdf'), 'LIRE_VAR_CDF_C_AUTO_GLUE',								$
						filename, LONG(ouvrir_fermer), 0L, REC_START, REC_COUNT, nom, data,					$
						LONG(taille), tmp[typecode], 0L, nblignesonly, nblignes, moins2, 					$
						datatype, nbdim, dimsizes, majority, validmin, validmax, scalemin, scalemax, 1L, /AUTO_GLUE)
		IF is_str THEN BEGIN
			out = REPLICATE ('', nb_str)
			FOR i=0L,nb_str-1 DO BEGIN
				pos = STRPOS (data.str[i], STRING(255b))
				out[i] = STRMID (data.str[i], 0, pos) ; garder ce qu'il y a avant le caractre spcial 255
			END
			data = out
		END
	END ELSE BEGIN
		lint_unused = CALL_EXTERNAL (libname('filecdf'), 'LIRE_VAR_CDF_C',									$
						filename, LONG(ouvrir_fermer), 0L, REC_START, REC_COUNT, nom, data,					$
						LONG(taille), tmp[typecode], 0L, nblignesonly, nblignes, moins2, 					$
						datatype, nbdim, dimsizes, majority, validmin, validmax, scalemin, scalemax, 0L)
	END

END


;-------------------------------------------------------------------------------
FUNCTION lire_att_cdf,	$
;-------------------------------------------------------------------------------
	fd,	$	; LINT_PROTOTYPE input
	att,	$	; LINT_PROTOTYPE input
	var,	$	; LINT_PROTOTYPE input
	val		; LINT_PROTOTYPE output
;-------------------------------------------------------------------------------
; ((0:ok (1:pb)) Retourne la valeur d'un attribut d'une variable d'aprs son nom.
;-------------------------------------------------------------------------------
	ON_IOERROR, erreur

	CDF_ATTGET, fd, att, var, val

	RETURN, 0

erreur:
	RETURN, -1

END


;-------------------------------------------------------------------------------
FUNCTION lire_att_by_id_cdf,	$
;-------------------------------------------------------------------------------
	fd,	$	; LINT_PROTOTYPE input
	att,	$	; LINT_PROTOTYPE input
	var,	$	; LINT_PROTOTYPE input
	val,	$	; LINT_PROTOTYPE output
	isZ		; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((0:ok -1:pb)) Retourne la valeur d'un attribut d'une variable d'aprs son numro.
;-------------------------------------------------------------------------------

	ON_IOERROR, erreur

	CATCH, error			; pour FDL
	IF error NE 0 THEN BEGIN	; pour FDL
		RETURN, -1		; pour FDL
	END				; pour FDL


	CDF_ATTGET, fd, att, var, val, zvariable=isZ

	RETURN, 0

erreur:
	RETURN, -1

END

;-------------------------------------------------------------------------------
FUNCTION get_nblignes_cdf_c,	$
;-------------------------------------------------------------------------------
	filename,	$	; LINT_PROTOTYPE input
	nom			; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((-)) Retourne le nombre de lignes d'un fichier CDF pour une variable donne.
;-------------------------------------------------------------------------------

	nblignesonly = 1L
	nblignes = 0L
	datatype = 0L
	nbdim = 0L
	dimsizes = REPLICATE (0L, 8)
	majority = 0b
	ouvrir_fermer = 1
	validmin = 0.
	validmax = 0.
	scalemin = 0.
	scalemax = 0.
	zero = 0L
	moins2 = -2L
	IF is_fdl() THEN BEGIN
		lint_unused = CALL_EXTERNAL (libname('filecdf'), 'LIRE_VAR_CDF_C_AUTO_GLUE', filename, LONG(ouvrir_fermer), zero, zero, zero, nom, zero, 	$
						zero, zero, zero, nblignesonly, nblignes, moins2, 								$
						datatype, nbdim, dimsizes, majority, validmin, validmax, scalemin, scalemax, 1L, /AUTO_GLUE)
	END ELSE BEGIN
		lint_unused = CALL_EXTERNAL (libname('filecdf'), 'LIRE_VAR_CDF_C', filename, LONG(ouvrir_fermer), zero, zero, zero, nom, zero, 			$
						zero, zero, zero, nblignesonly, nblignes, moins2, 								$
						datatype, nbdim, dimsizes, majority, validmin, validmax, scalemin, scalemax, 0L)
	END

	RETURN, nblignes

END

;-------------------------------------------------------------------------------
PRO lire_var_cdf,	$
;-------------------------------------------------------------------------------
	id,		$	; LINT_PROTOTYPE input
	CDF_var,	$	; LINT_PROTOTYPE input
	x,		$	; LINT_PROTOTYPE output
	type,		$	; LINT_PROTOTYPE output
	REC_COUNT=rcnt,	$	; LINT_PROTOTYPE input
	REC_START=rstr		; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((-)) Retourne la valeur d'un variable CDF.
;-------------------------------------------------------------------------------

	inq = CDF_INQUIRE (id)
	vinq = CDF_VARINQ (id,CDF_var)
	type = vinq.datatype

	IF KEYWORD_SET(rstr) EQ 0 THEN rstr = 0

	IF ~vinq.is_zvar THEN BEGIN

		dims = TOTAL (vinq.dimvar)
		dimc = vinq.dimvar * inq.dim
		dimw = WHERE (dimc EQ 0)
		IF dimw[0] NE -1 THEN dimc[dimw] = 1

		IF KEYWORD_SET(rcnt) EQ 0 THEN rcnt = inq.maxrec+1
		IF (rstr+rcnt) GT (inq.maxrec+1) THEN rcnt = inq.maxrec+1 - rstr
		CDF_VARGET, id,CDF_var,x,COUNT=dimc,REC_COUNT=rcnt,REC_START=rstr
		pb = 0

	END ELSE BEGIN

		dims = TOTAL (vinq.dimvar)
		dimc = vinq.dimvar * vinq.dim
		dimw = WHERE (dimc EQ 0)
		IF dimw[0] NE -1 THEN dimc[dimw] = 1

		; L'instruction suivante est commentee sinon on peut obtenir le message suivant:
		; "CDF_CONTROL: CDF file error. NO_WRITE_ACCESS: Write access is not allowed on the CDF file(s)."
		;CDF_CONTROL,id,variable=CDF_var,/zvariable,set_padvalue=0.0

		; L'insruction qui suit plante sur certaines variables NOVARY
		CDF_CONTROL,id,variable=CDF_var,/zvariable,get_var_info=vinfo
		IF KEYWORD_SET(rcnt) EQ 0 THEN rcnt = vinfo.maxrec+1
		;IF (rstr+rcnt) GT (vinfo.maxrec+1) THEN rcnt = vinfo.maxrec+1 - rstr

		; Premier element a lire: rstr        qui doit etre compris entre 0 et vinfo.maxrec
		; Dernier element a lire: rstr+rcnt-1 qui doit etre compris entre 0 et vinfo.maxrec
		IF (rstr GE 0) AND (rstr LE vinfo.maxrec) AND (rstr+rcnt-1 GE rstr) AND (rstr+rcnt-1 LE vinfo.maxrec) THEN BEGIN
			CDF_VARGET, id,CDF_var,x,COUNT=dimc,REC_COUNT=rcnt,REC_START=rstr,/zvariable
			pb = 0
		END ELSE BEGIN
			CDF_VARGET, id,CDF_var,x,COUNT=dimc,REC_COUNT=1,REC_START=0,/zvariable
			pb = 1
		END

	END

	IF rcnt EQ 1 OR pb THEN BEGIN ; un seul enregistrement lu
		info = SIZE(x)
		nbdimensions = info[0]
		inforeform = nbdimensions EQ 0 ? [1] : [1,info[1:nbdimensions]]
		x = REFORM (nbdimensions EQ 0 ? [x]:x,inforeform)
		RETURN
	END

	sa = SIZE (x)
	IF sa[0] GT 0 THEN BEGIN
		sa = sa[1:sa[0]]
	END ELSE BEGIN
		; La variable lue ne contient aucune valeur
		x = -1
		RETURN
	END

	IF (vinq.recvar EQ 'VARY') AND (dims NE 0) THEN BEGIN
		x = REFORM (x,[marray_cdf(sa[0:(N_ELEMENTS(sa)-2)]),sa[N_ELEMENTS(sa)-1]])
		x = TRANSPOSE (x)
		sa = SHIFT (sa,1)
		x = REFORM (x,sa)
	END

	saw = WHERE (sa NE 1)
	x = REFORM (x,sa[saw])

END

;-------------------------------------------------------------------------------
PRO lire_struct_cdf_c,		$
;-------------------------------------------------------------------------------
	filename,		$	; LINT_PROTOTYPE input
	data,			$	; LINT_PROTOTYPE input
	nom1,			$	; LINT_PROTOTYPE input
	nom2,			$	; LINT_PROTOTYPE input
	info,			$	; LINT_PROTOTYPE input
	REC_START,		$	; LINT_PROTOTYPE input
	REC_COUNT,		$	; LINT_PROTOTYPE input
	offset=offset,		$	; LINT_PROTOTYPE input
	quiet=quiet,		$	; LINT_PROTOTYPE input
	validmin=validmin,	$	; LINT_PROTOTYPE output
	validmax=validmax,	$	; LINT_PROTOTYPE output
	scalemin=scalemin,	$	; LINT_PROTOTYPE output
	scalemax=scalemax		; LINT_PROTOTYPE output
;-------------------------------------------------------------------------------
; ((-)) Lit certaines variables d'un fichier CDF et crit les valeurs dans certains champs d'un tableau de structure.
;
; Entres:
;
;	filename:	fichier CDF
;	nom1:		tableau de string avec les noms des champs de la structure  lire
;	nom2:		tableau de string avec les noms des variables CDF correspondantes
;	info:		tableau de string des champs  lire (ex: '-2' pour extraire toutes les dimensions, '-1 0' pour extraire juste la premire dimension en prenant 0 pour la seconde dimension)
;	REC_START:	commencer  lire dans le CDF  l'enregistrement REC_START
;	REC_COUNT:	lire REC_COUNT enregistrements dans le fichier cdf
;	offset:		pour commencer  crire dans data  l'enregistrement offset
;
; Entrs/Sorties:
;
;	data:		tableau de structure existante en entre, renseigne en sortie
;-------------------------------------------------------------------------------

	;top1 = nbsec()

	IF N_ELEMENTS(offset) EQ 0 THEN offset = 0L
	IF N_ELEMENTS(quiet) EQ 0 THEN quiet = 0L

	nblignes = 0L
	nblignesonly = 0L
	lint_unused = get_structure_infos (data[0], taille_enreg, nbchamps, nom_champ, type_champ, offset_champ, taille_champ, nb_champ)

	nbvars = N_ELEMENTS(nom1)

	validmin = REPLICATE (0., nbvars)
	validmax = REPLICATE (0., nbvars)
	scalemin = REPLICATE (0., nbvars)
	scalemax = REPLICATE (0., nbvars)

	datatype = 0L
	nbdim = 0L
	dimsizes = REPLICATE (0L, 8)
	majority = 0b

	ouvrir_fermer = 0L ; pour ne pas ouvrir et fermer le fichier  chaque variable
	IF ouvrir_fermer EQ 0 THEN BEGIN
		code = ouvrir_cdf_c (filename)
	END

	IF is_gdl() THEN BEGIN

		; recherche nbdims dans info
		nbdims = 1L
		FOR i=0L,nbvars-1 DO BEGIN
			IF info[i] NE '' THEN BEGIN
				tmp = STRSPLIT (info[i], /EXTRACT)
				nb = N_ELEMENTS(tmp)
				IF nb GT nbdims THEN BEGIN
					nbdims = nb
				END
			END
		END

		tab_nom    = REPLICATE ('', nbvars)
		tab_type   = REPLICATE (0L, nbvars)
		tab_offset = REPLICATE (0L, nbvars)
		tab_info   = REPLICATE (0L, nbvars, nbdims)

		FOR i=0L,nbvars-1 DO BEGIN
			nochamp = (WHERE (STRUPCASE(TAG_NAMES(data)) EQ STRUPCASE(nom1[i])))[0]
			IF nochamp EQ -1 THEN BEGIN
				PRINT, 'lire_struct_cdf_c: champ ' + nom1[i] + ' introuvable dans data '
			END ELSE BEGIN
				tab_nom[i] = nom2[i]
				tab_type[i] = type_champ[nochamp]
				tab_offset[i] = offset_champ[nochamp]
				IF info[i] EQ '' THEN BEGIN
					tab_info[i,0] = -2
				END ELSE BEGIN
					tmp = STRSPLIT (info[i], /EXTRACT)
					tab_info[i,LINDGEN(N_ELEMENTS(tmp))] = LONG(tmp)
				END
			END
		END

		; lecture de toutes les variables en un seul coup

		; ex: /home/penou/DATA/CLUSTER/SOFT/CLL3/resource/omni_1.cl.gz fichier /DATA/OMNI/DATA/hro_1min/1997/omni_hro_1min_19970601_v01.cdf
		; idl = 0.068882942 sec
		; gdl = 0.19295287 sec

		code = CALL_EXTERNAL (libname('filecdf'), 'LIRE_VARS_CDF_C', filename, ouvrir_fermer, offset, REC_START, REC_COUNT, tab_nom, data,	$
					taille_enreg, tab_type, tab_offset, nblignesonly, nblignes, tab_info,						$
					datatype, nbdim, dimsizes, majority, validmin, validmax, scalemin, scalemax, 0L, nbvars)

	END ELSE BEGIN

		validmin_nochamp = 0.
		validmax_nochamp = 0.
		scalemin_nochamp = 0.
		scalemax_nochamp = 0.

		zero = 0L

		; lecture des variables les unes aprs les autres
		FOR i=0L,nbvars-1 DO BEGIN
			nochamp = (WHERE (STRUPCASE(TAG_NAMES(data)) EQ STRUPCASE(nom1[i])))[0]
			IF nochamp EQ -1 THEN BEGIN
				PRINT, 'lire_struct_cdf_c: champ ' + nom1[i] + ' introuvable dans data '
			END ELSE BEGIN
				IF info[i] EQ '' THEN BEGIN
					info_alire = -2L
				END ELSE BEGIN
					tmp = STRSPLIT (info[i], /EXTRACT)
					info_alire = LONG(tmp)
				END
				variable		= nom2[i]
				type_champ_nochamp	= type_champ[nochamp]
				offset_champ_nochamp	= offset_champ[nochamp]

				; lent en gdl
				; ex: /home/penou/DATA/CLUSTER/SOFT/CLL3/resource/omni_1.cl.gz fichier /DATA/OMNI/DATA/hro_1min/1997/omni_hro_1min_19970601_v01.cdf
				; idl = 0.0032329559 sec
				; gdl = 0.079696894 sec
				; C'est le passage des paramtres au CALL_EXTERNAL qui est lent

				IF is_fdl() THEN BEGIN
					IF type_champ[nochamp] EQ 7 THEN BEGIN
						data.(nochamp) = STRING(REPLICATE(BYTE('A'),1024)) ; rserver 1024 bytes pour les STRING
					END
					code = CALL_EXTERNAL (libname('filecdf'), 'LIRE_VAR_CDF_C_AUTO_GLUE', filename, ouvrir_fermer, offset, REC_START, REC_COUNT, variable, data,	$
								taille_enreg, type_champ_nochamp, offset_champ_nochamp, nblignesonly, nblignes, info_alire, 				$
								datatype, nbdim, dimsizes, majority, validmin_nochamp, validmax_nochamp, scalemin_nochamp, scalemax_nochamp, 1L, /AUTO_GLUE)
					IF type_champ[nochamp] EQ 7 THEN BEGIN
						pos = STRPOS (data.(i), STRING(255b))
						data.(i) = STRMID (data.(i), 0, pos) ; garder ce qu'il y a avant le caractre spcial 255

					END
				END ELSE BEGIN
					code = CALL_EXTERNAL (libname('filecdf'), 'LIRE_VAR_CDF_C', filename, ouvrir_fermer, offset, REC_START, REC_COUNT, variable, data,		$
								taille_enreg, type_champ_nochamp, offset_champ_nochamp, nblignesonly, nblignes, info_alire,				$
								datatype, nbdim, dimsizes, majority, validmin_nochamp, validmax_nochamp, scalemin_nochamp, scalemax_nochamp, zero)
				END

				validmin[i] = validmin_nochamp
				validmax[i] = validmax_nochamp
				scalemin[i] = scalemin_nochamp
				scalemax[i] = scalemax_nochamp

				IF code NE 0 AND quiet EQ 0 THEN BEGIN
					PRINT, 'lire_struct_cdf_c: ' + nom2[i] + ' in '+filename + ' returns ' + val_to_str(code)
				END
			END

		END

	END

	IF ouvrir_fermer EQ 0 THEN BEGIN
		code = fermer_cdf_c (filename)
	END

	;top2 = nbsec()
	;PRINT, 'lire_struct_cdf_c en ' + val_to_str_2decimales(top2-top1)

END

;-------------------------------------------------------------------------------
PRO lire_multi_struct_cdf_c,	$
;-------------------------------------------------------------------------------
	filenames,	$	; LINT_PROTOTYPE input
	tmp,		$	; LINT_PROTOTYPE input
	inf,		$	; LINT_PROTOTYPE input
	REC_START,	$	; LINT_PROTOTYPE input
	REC_COUNT		; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((-)) Lit dans tmp les variables dcrites dans inf d'un ou plusieurs fichier CDF dcrits dans filenames.
;-------------------------------------------------------------------------------

	IF N_ELEMENTS(filenames) EQ 1 THEN BEGIN

		; cas avec un seul fichier par jour
		top1 = nbsec()
		lire_struct_cdf_c, filenames, tmp, inf[0,*], inf[1,*], inf[2,*], REC_START, REC_COUNT
		top2 = nbsec()
		PRINT,filenames+': lire_struct_cdf_c in ' + val_to_str_2decimales(top2-top1)+' sec'

	END ELSE BEGIN

		offset = 0L
		debut = 0L
		reste = REC_COUNT ; nombre de lignes restants  lire
		lire = 0

		HELP, tmp
		FOR i=0L,N_ELEMENTS(filenames)-1 DO BEGIN

			top1 = nbsec()
			nblignes = get_nblignes_cdf_c (filenames[i], 'Epoch')
			top2 = nbsec()
			PRINT,filenames[i]+': get_nblignes_cdf_c in ' + val_to_str_2decimales(top2-top1)+' sec'

			IF nblignes EQ 0 THEN CONTINUE

			fin = debut + nblignes-1
			IF lire EQ 0 THEN BEGIN
				lire = REC_START GE debut AND REC_START LE fin
				IF lire THEN REC_START1 = REC_START - debut
			END ELSE BEGIN
				REC_START1 = 0L
			END

			IF lire THEN BEGIN

				REC_COUNT1 = reste
				IF REC_COUNT1 GT nblignes THEN REC_COUNT1 = nblignes-REC_START1

				top1 = nbsec()
				PRINT,filenames[i] + '[' + val_to_str(REC_START1) + ':' + val_to_str(REC_START1+REC_COUNT1-1)+'] -> tmp['+val_to_str(offset)+':'+val_to_str(offset+REC_COUNT1-1)+']'
				lire_struct_cdf_c, filenames[i], tmp, inf[0,*], inf[1,*], inf[2,*], REC_START1, REC_COUNT1, offset=offset
				top2 = nbsec()
				PRINT,filenames[i]+': lire_struct_cdf_c in ' + val_to_str_2decimales(top2-top1)+' sec'

				reste -= REC_COUNT1
				offset += REC_COUNT1

				IF reste EQ 0 THEN BREAK ; fini
			END

			debut += nblignes

		END

	END

END

;-------------------------------------------------------------------------------
FUNCTION read_cdf_get_symbols,	$
;-------------------------------------------------------------------------------
	FILENAME,			$	; LINT_PROTOTYPE input
	ATT,				$	; LINT_PROTOTYPE output
	NOVARY,				$	; LINT_PROTOTYPE output
	SYMBOLS,			$	; LINT_PROTOTYPE output
	FORMAT,				$	; LINT_PROTOTYPE output
	NBRECORDS,			$	; LINT_PROTOTYPE output
	NBLINES0,			$	; LINT_PROTOTYPE output
	noatt=noatt,			$	; LINT_PROTOTYPE input
	nonb=nonb,			$	; LINT_PROTOTYPE input
	selection_date=selection_date,	$	; LINT_PROTOTYPE input
	remove_ext=remove_ext,		$	; LINT_PROTOTYPE input
	nosymbols=nosymbols,		$	; LINT_PROTOTYPE input
	quiet=quiet				; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((1:ok 0:pb)) Lit un fichier CDF et retourne les symboles trouvs.
;
; Input:
; ------
; - FILENAME:			filename to read
;
; Keywords:
; ---------
; - /noatt:			don't compute ATT and NOVARY
; - /nonb:			don't compute NBRECORDS
; - /selection_date:		
; - /remove_ext:		"CAA" or "THEMIS" to remove extension in variables names
; - /nosymbols:			don't compute symbols
; - /quiet: 			no messages
;
; Output:
; -------
; - ATT:			(undefined if /noatt) array of structure { v1:'', v2:'', v3:'', v4:''} with all attributes found in header
; - NOVARY:			(undefined if /noatt) structure with variables containing a non empty DATA field
; - SYMBOLS:			structure with variables inforation found in header
; - FORMAT:			format number for Date/Time found in first line of data (5 for CEF-2)
; - NBRECORDS:			number of records found in data
; - NBLINES0:			number of lines found in header
;
; Return code:
; ------------
;	1 = OK
;	0 = PB
;-------------------------------------------------------------------------------
;
; Ordre de SIZES et des DEPEND_1, DEPEND_2, DEPEND_3:
; ---------------------------------------------------
;
; Exemple du fichier "/DATA/GEOTAIL/DATA/Kp/lep/ge_k0_lep_20010321_v01.cdf"
;
;	Pour la variable "EA_E", cdfedit donne
;		Varys		[3='F',32='T',4='T']
;		majority	'F'
;		DEPEND_1	"label_3" avec 32 valeurs
;		DEPEND_2	"sector_angle" avec 4 valeurs
;
;	La routine "lire_var_cdf_c":
;		- retourne nbdim=3 et dimsizes=[-3,32,4]
;		- crit dans un tableau de 32*4 valeurs dans l'ordre suivant:
;			- 32 valeurs pour le secteur 1
;			- puis 32 valeurs pour le secteur 2
;			- puis 32 valeurs pour le secteur 3
;			- puis 32 valeurs pour le secteur 4
;
;	En IDL, il faudra donc dclarer la variable en [32,4]
;
;	Pour rester cohrent avec read_ascii_get_structure, il faut renvoyer dans SYMBOLS:
;		SIZES		4,32
;		DEPEND_1	"sector_angle"
;		DEPEND_2	"label_3"
;
;		=> IL FAUT inverser l'ordre de SIZES et des DEPEND_[123]
;
;
; Exemple du fichier "/DATA/THEMIS/DATA/tha/l1/esa/2008/tha_l1_esa_20080101_v02.cdf"
;
;	Pour la variable "tha_eif_088x32", cdfedit donne
;		Varys		[88='T',32='T']
;		majority	'C' (ROW)
;		DEPEND_1	"tha_eif_088x32_angleno" avec 88 valeurs
;		DEPEND_2	"tha_eif_088x32_energno" avec 32 valeurs
;
;	La routine "lire_var_cdf_c":
;		- retourne nbdim=3 et dimsizes=[-3,32,4]
;		- crit dans un tableau de 32*4 valeurs dans l'ordre suivant:
;			- 32 valeurs pour l'angle 1
;			- puis 32 valeurs pour l'angle 2
;			- ...
;			- puis 32 valeurs pour l'angle 88
;
;	En IDL, il faudra donc dclarer la variable en [32,88]
;
;	Pour rester cohrent avec read_ascii_get_structure, il faut renvoyer dans SYMBOLS:
;		SIZES		88,32
;		DEPEND_1	"tha_eif_088x32_angleno"
;		DEPEND_2	"tha_eif_088x32_energno"
;
;		=> IL NE FAUT PAS inverser l'ordre de SIZES et des DEPEND_[123]
;-------------------------------------------------------------------------------

	DEBUG = 0

	top1 = nbsec()

	ON_IOERROR, error

	OLD_QUIET = !QUIET
	!QUIET = 1

	IF N_ELEMENTS(quiet) 		EQ 0 THEN quiet = 0
	IF N_ELEMENTS(noatt) 		EQ 0 THEN noatt = 0
	IF N_ELEMENTS(nosymbols) 	EQ 0 THEN nosymbols = 0
	IF N_ELEMENTS(remove_ext)	EQ 0 THEN remove_ext = 'No'
	IF N_ELEMENTS(nonb) 		EQ 0 THEN nonb = 0

	duree_ALL	= 0
	duree_NOVARY	= 0
	duree_VARY	= 0
	duree_ATT	= 0
	top1_ALL	= nbsec()
	FORMAT		= 0L
	NBLINES0	= 0L
	NBRECORDS	= 0L

	IF ~quiet THEN BEGIN
		PRINT,''
		FOR i=0L,STRLEN(FILENAME)-1 DO PRINT, '-',FORMAT='(A,$)' & PRINT, ''
		PRINT, FILENAME
		FOR i=0L,STRLEN(FILENAME)-1 DO PRINT, '-',FORMAT='(A,$)' & PRINT, ''
	END

	IF noatt THEN BEGIN
		ATT = 0 & tmp = TEMPORARY(ATT) ; destroy ATT
	END ELSE BEGIN
		ATT = { v1:'', v2:'', v3:'', v4:'' }
	END
	NOVARY = 0 & tmp = TEMPORARY(NOVARY) ; destroy NOVARY
	symbol0 = { 			$
		name:		'', 	$
		exception:	0,  	$
		nblignes:	0L,  	$
		value_type:	'', 	$
		var_type:	'', 	$
		property:	'', 	$
		sizes:		'', 	$
		data:		'', 	$
		delta_plus:	'', 	$
		delta_minus:	'', 	$
		fillval:	'', 	$
		units:		'', 	$
		fieldnam:	'', 	$
		si_conversion:	'', 	$
		depend_0:	'', 	$
		depend_1:	'', 	$
		depend_2:	'', 	$
		depend_3:	'', 	$
		depend_4:	'', 	$
		labl_ptr_1:	''  	$

	}
	SYMBOLS = [ symbol0 ]

	fd = -1 ; car sinon, si CDF_OPEN plante, fd ne sera pas defini

	quick = 1
	IF quick THEN code = ouvrir_cdf_c (FILENAME)

	IF 1 THEN BEGIN
		ndims = 0L
		nvars = 0L
		nzvars = 0L
		natts = 0L
		CDF_MAX_DIMS = 10 ; idem cdflib
		dim = REPLICATE (0L, CDF_MAX_DIMS)
		IF is_fdl() THEN BEGIN
			lint_unused = CALL_EXTERNAL (libname('filecdf'), 'MY_CDF_INQUIRE_AUTO_GLUE', ndims, nvars, nzvars, natts, dim, /AUTO_GLUE)
		END ELSE BEGIN
			lint_unused = CALL_EXTERNAL (libname('filecdf'), 'MY_CDF_INQUIRE', ndims, nvars, nzvars, natts, dim)
		END
		dim = ndims EQ 0 ? 0 : dim[0:ndims-1]
		inq = { ndims:ndims, nvars:nvars, nzvars:nzvars, natts:natts, dim:dim }
	END ELSE BEGIN
		fd = CDF_OPEN (FILENAME)
		inq = CDF_INQUIRE (fd)
	END

	attname = '' ; attname contient la liste des noms de tous les attributs possibles pour les variables
	attid   = -1 ; attname contient la liste des ids  de tous les attributs possibles pour les variables
	majority = 'C';	car majority ne sera positionn que s'il y a au moins une variable VARY
	; rapide
	FOR i=0L,inq.natts-1 DO BEGIN	; Boucle sur tous les attributs
		IF 1 THEN BEGIN
			name = ''
			scope = ''
			maxgentry = 0L
			maxentry = 0L
			maxzentry = 0L
			IF is_fdl() THEN BEGIN
				name = pchar ('')
				scope = pchar ('')
				lint_unused = CALL_EXTERNAL (libname('filecdf'), 'MY_CDF_ATTINQ_AUTO_GLUE', i, name, scope, maxgentry, maxentry, maxzentry, /AUTO_GLUE)
				name = STRING(name)
				scope = STRING(scope)
			END ELSE BEGIN
				lint_unused = CALL_EXTERNAL (libname('filecdf'), 'MY_CDF_ATTINQ', i, name, scope, maxgentry, maxentry, maxzentry)
			END
		END ELSE BEGIN
			CDF_ATTINQ, fd, i, name, scope, maxentry, maxzentry
		END
		IF scope EQ "GLOBAL_SCOPE" AND maxentry NE -1 THEN BEGIN
			; mis en commentaire sinon plantage sur /home/penou/cdaweb.gsfc.nasa.gov/pre_istp/data/isee/isee1/24or48s_fpe/1977/isee1_h2_fpe_19771030_v01.cdf
			;CDF_ATTGET, fd, name, 0, val
		END ELSE BEGIN
			IF scope EQ "VARIABLE_SCOPE" THEN BEGIN
				attname = attname[0] EQ '' ? [name] : [attname,name]
				attid   = attid[0]   EQ -1 ? [i]    : [attid,  i]
			END
		END
	END

	; Je ne m'intresse qu'aux attributs suivants
	ind = WHERE (	STRUPCASE(attname) EQ 'PROPERTY' 		OR $
			STRUPCASE(attname) EQ 'FILLVAL'			OR $
			STRUPCASE(attname) EQ 'DELTA_PLUS'		OR $
			STRUPCASE(attname) EQ 'DELTA_MINUS'		OR $
			STRUPCASE(attname) EQ 'DELTA_PLUS_VAR'		OR $
			STRUPCASE(attname) EQ 'DELTA_MINUS_VAR'		OR $
			STRUPCASE(attname) EQ 'UNITS'			OR $
			STRUPCASE(attname) EQ 'CATDESC'			OR $
			STRUPCASE(attname) EQ 'FIELDNAM'		OR $
			STRUPCASE(attname) EQ 'SI_CONVERSION'		OR $
			STRUPCASE(attname) EQ 'DEPEND_0'		OR $
			STRUPCASE(attname) EQ 'DEPEND_1'		OR $
			STRUPCASE(attname) EQ 'DEPEND_2'		OR $
			STRUPCASE(attname) EQ 'DEPEND_3'		OR $
			STRUPCASE(attname) EQ 'DEPEND_4'		OR $
			STRUPCASE(attname) EQ 'LABL_PTR_1'		OR $
			STRUPCASE(attname) EQ 'LABEL_1'			OR $
			STRUPCASE(attname) EQ 'REPRESENTATION_1'	OR $
			STRUPCASE(attname) EQ 'VAR_TYPE')
	IF ind[0] EQ -1 THEN BEGIN
		nbatt = 0
	END ELSE BEGIN
		nbatt = N_ELEMENTS(ind)
		attname = attname[ind]
		attid = attid[ind]
	END

	; ne pas continuer s'il n'y a aucune variable
	IF inq.nvars EQ 0 AND inq.nzvars EQ 0 THEN GOTO, error

	nbvar = [inq.nvars , inq.nzvars]
	SYMBOLS = REPLICATE (symbol0, nbvar[0] + nbvar[1])
	necessaire = nosymbols EQ 0

	FOR isz=0L,1 DO BEGIN ; Variables puis Zvariables

		FOR i=0L,nbvar[isz]-1 DO BEGIN

			novar = ~isz ? i : (nbvar[0]+i)
			IF 1 THEN BEGIN
				name = ''
				datatype = ''
				recvary = ''
				numelem = 0L
				dimvar = inq.ndims EQ 0 ? 0b : REPLICATE (0b, inq.ndims)
				IF is_fdl() THEN BEGIN
					name = pchar ('')
					datatype = pchar ('')
					recvary = pchar ('')
					lint_unused = CALL_EXTERNAL (libname('filecdf'), 'MY_CDF_VARINQ_AUTO_GLUE',	$
									i, isz, name, datatype, recvary, numelem, inq.ndims, dimvar, /AUTO_GLUE)
					name = STRING (name)
					datatype = STRING (datatype)
					recvary = STRING (recvary)
				END ELSE BEGIN
					lint_unused = CALL_EXTERNAL (libname('filecdf'), 'MY_CDF_VARINQ',	$
									i, isz, name, datatype, recvary, numelem, inq.ndims, dimvar)
				END
				vinq = { is_zvar:isz, name:name, datatype:datatype, numelem:numelem, recvar:recvary, dimvar:dimvar }
			END ELSE BEGIN
				vinq = CDF_VARINQ (fd,i,zvariable=isz)
			END
			SYMBOLS[novar].name = vinq.name

			CASE vinq.datatype OF
				'CDF_TIME_TT2000':	SYMBOLS[novar].value_type = 'ISO_TIME'
				'CDF_EPOCH':		SYMBOLS[novar].value_type = 'ISO_TIME'
				'CDF_EPOCH16':		SYMBOLS[novar].value_type = 'ISO_TIME'
				'CDF_CHAR':		SYMBOLS[novar].value_type = 'CHAR'
				'CDF_REAL8':		SYMBOLS[novar].value_type = 'DOUBLE'
				'CDF_DOUBLE':		SYMBOLS[novar].value_type = 'DOUBLE'
				ELSE:			SYMBOLS[novar].value_type = 'FLOAT'
			END

			IF nosymbols NE 0 THEN BEGIN
				necessaire = (SYMBOLS[novar].value_type EQ 'ISO_TIME') OR (STRMID(SYMBOLS[novar].name,STRLEN(SYMBOLS[novar].name)-5,5) EQ '_time')
				IF ~necessaire THEN CONTINUE
			END

			nblignes = 0
			IF vinq.RECVAR EQ 'NOVARY' THEN BEGIN

				IF DEBUG THEN top1_NOVARY = nbsec()		
				get_info_var_cdf_c, FILENAME, vinq.name, nblignes, datatype, nbdim, dimsizes, majority, quick=quick

				; Septembre 2012:
				; CDF_VARGET avec le mot cl /STRING ne retourne pas la mme chose sous Windows et sous Linux. Ex: pour ['x','y','z']
				;   - 3 string de 1 caractre sous Linux et MacOS
				;   - 1 string de 3 caractres sous Windows, ou ca plante sur fichier cdaweb CLUSTER FGM pour "REPRESENTATION_1"
				; => C'est pour ca que je ne l'utilise pas

				IF ~vinq.is_zvar THEN BEGIN
					IF 1 THEN BEGIN
						IF SYMBOLS[novar].value_type EQ 'ISO_TIME' THEN BEGIN
							VALEUR = 0d
						END ELSE IF SYMBOLS[novar].value_type EQ 'DOUBLE' THEN BEGIN
							VALEUR = 0d
						END ELSE IF SYMBOLS[novar].value_type EQ 'CHAR' THEN BEGIN
							VALEUR = ''
						END ELSE BEGIN
							VALEUR = 0.
						END
						dimsizes = ABS(dimsizes) ; il peut y avoir des valeurs  -1
						IF nbdim EQ 1 THEN VALEUR = REPLICATE (VALEUR, dimsizes[0])
						IF nbdim EQ 2 THEN VALEUR = REPLICATE (VALEUR, dimsizes[0], dimsizes[1])
						IF nbdim EQ 3 THEN VALEUR = REPLICATE (VALEUR, dimsizes[0], dimsizes[1], dimsizes[2])
						IF nbdim EQ 4 THEN VALEUR = REPLICATE (VALEUR, dimsizes[0], dimsizes[1], dimsizes[2], dimsizes[3])
						IF nbdim EQ 5 THEN VALEUR = REPLICATE (VALEUR, dimsizes[0], dimsizes[1], dimsizes[2], dimsizes[3], dimsizes[4])
						IF nbdim EQ 6 THEN VALEUR = REPLICATE (VALEUR, dimsizes[0], dimsizes[1], dimsizes[2], dimsizes[3], dimsizes[4], dimsizes[5])
						IF nbdim EQ 7 THEN VALEUR = REPLICATE (VALEUR, dimsizes[0], dimsizes[1], dimsizes[2], dimsizes[3], dimsizes[4], dimsizes[5], dimsizes[6])
						IF nbdim EQ 8 THEN VALEUR = REPLICATE (VALEUR, dimsizes[0], dimsizes[1], dimsizes[2], dimsizes[3], dimsizes[4], dimsizes[5], dimsizes[6], dimsizes[7])
						lire_var_cdf_c, FILENAME, VALEUR, vinq.name, 0L, 1L, quick=quick
					END ELSE BEGIN
						dimc = vinq.dimvar * inq.dim
						dimw = WHERE (dimc EQ 0)
						IF dimw[0] NE -1 THEN dimc[dimw] = 1
						CDF_VARGET, fd, vinq.name, VALEUR, COUNT=dimc, zvariable=isz
					END
				END ELSE BEGIN
					IF 1 THEN BEGIN
						IF SYMBOLS[novar].value_type EQ 'ISO_TIME' THEN BEGIN
							VALEUR = 0d
						END ELSE IF SYMBOLS[novar].value_type EQ 'DOUBLE' THEN BEGIN
							VALEUR = 0d
						END ELSE IF SYMBOLS[novar].value_type EQ 'CHAR' THEN BEGIN
							VALEUR = ''
						END ELSE BEGIN
							VALEUR = 0.
						END
						IF nbdim EQ 1 THEN VALEUR = REPLICATE (VALEUR, dimsizes[0])
						IF nbdim EQ 2 THEN VALEUR = REPLICATE (VALEUR, dimsizes[0], dimsizes[1])
						IF nbdim EQ 3 THEN VALEUR = REPLICATE (VALEUR, dimsizes[0], dimsizes[1], dimsizes[2])
						IF nbdim EQ 4 THEN VALEUR = REPLICATE (VALEUR, dimsizes[0], dimsizes[1], dimsizes[2], dimsizes[3])
						IF nbdim EQ 5 THEN VALEUR = REPLICATE (VALEUR, dimsizes[0], dimsizes[1], dimsizes[2], dimsizes[3], dimsizes[4])
						IF nbdim EQ 6 THEN VALEUR = REPLICATE (VALEUR, dimsizes[0], dimsizes[1], dimsizes[2], dimsizes[3], dimsizes[4], dimsizes[5])
						IF nbdim EQ 7 THEN VALEUR = REPLICATE (VALEUR, dimsizes[0], dimsizes[1], dimsizes[2], dimsizes[3], dimsizes[4], dimsizes[5], dimsizes[6])
						IF nbdim EQ 8 THEN VALEUR = REPLICATE (VALEUR, dimsizes[0], dimsizes[1], dimsizes[2], dimsizes[3], dimsizes[4], dimsizes[5], dimsizes[6], dimsizes[7])
						lire_var_cdf_c, FILENAME, VALEUR, vinq.name, 0L, 1L, quick=quick
					END ELSE BEGIN
						CDF_VARGET, fd, vinq.name, VALEUR, zvariable=isz
					END
				END

				IF datatype EQ 51 OR datatype EQ 52 THEN BEGIN

					; La transformation en STRING fait perdre une dimension
					; STRING([33b,34b,35b]) donne une chaine de 3 caractres
					sVALEUR = STRING(VALEUR)
					; Linux: VALEUR=BYTE[1,3]=[120,121,122] => sVALEUR = ['x','y','z']
					; Windows: VALEUR=BYTE[3]=[120,121,122] => sVALEUR = 'xyz'

					; Correction du problme sous Windows
					IF is_byte(VALEUR) THEN BEGIN
						nbelts = nbdim EQ 0 ? 1 : PRODUCT(ABS(dimsizes[0:nbdim-1]))
						IF N_ELEMENTS(sVALEUR) NE nbelts THEN BEGIN
							sVALEUR = STRING(LONG(VALEUR)) ; par encore ok car contient par exemple ['   120', '   121', '   122'] et pas ['x','y','z']
							FOR j=0L,nbelts-1 DO BEGIN
								sVALEUR[j] = STRING(BYTE(LONG(sVALEUR[j])))	; '   120' => 'x'
							END
						END
					END

				END ELSE BEGIN

					; arrive sur "cdaweb.gsfc.nasa.gov/istp_public/data/polar/pwi/pwi_h8/1996/po_h8_pwi_1996040415_v01.cdf"
					; variable "HFWR1_Flag" est de type BYTE/1

					sVALEUR = STRING(DOUBLE(VALEUR))

				END

				SIZES = LONG(val_to_str(N_ELEMENTS(sVALEUR)))

				; ex: "Look_Dir_Vctr" dans "cdaweb.gsfc.nasa.gov/istp_public/data/polar/vis/vis_ej/1996/po_ej_vis_19960526_v02.cdf"
				; pour passer de Array[3, 256, 256]  Array[196608]
				sVALEUR = REFORM(sVALEUR,SIZES)

				;	; ATTENTION: pb si sVALEUR[i] contient des ','
				sVALEUR = supprimer_caractere(sVALEUR,',')

				DATA = STRJOIN(sVALEUR,',')

				SYMBOLS[novar].SIZES = val_to_str(N_ELEMENTS(sVALEUR))
				SYMBOLS[novar].data = DATA
				IF DEBUG THEN top2_NOVARY = nbsec()		
				IF DEBUG THEN duree_NOVARY += top2_NOVARY-top1_NOVARY

			END ELSE IF vinq.RECVAR EQ 'VARY' THEN BEGIN

				IF DEBUG THEN top1_VARY = nbsec()		
				get_info_var_cdf_c, FILENAME, vinq.name, nblignes, datatype, nbdim, dimsizes, majority, quick=quick
				IF nbdim EQ 0 THEN BEGIN
					SIZES = '1'
				END ELSE BEGIN
					SIZES = ''
					IF majority EQ 'F' THEN BEGIN
						; inversion de l'ordre des dimensions si majority vaut 'F'
						FOR j=nbdim-1,0,-1 DO BEGIN
							IF dimsizes[j] GT 0 THEN SIZES += (STRLEN(SIZES) NE 0 ? ',':'')+val_to_str(dimsizes[j])
						END
					END ELSE BEGIN
						FOR j=0L,nbdim-1 DO BEGIN
							IF dimsizes[j] GT 0 THEN SIZES += (STRLEN(SIZES) NE 0 ? ',':'')+val_to_str(dimsizes[j])
						END
					END
					IF SIZES EQ '' THEN SIZES = '1'
				END
				SYMBOLS[novar].SIZES = SIZES
				IF DEBUG THEN top2_VARY = nbsec()	
				IF DEBUG THEN duree_VARY += top2_VARY-top1_VARY

			END

			SYMBOLS[novar].nblignes = nblignes
		END
	
		IF DEBUG THEN top1_ATT = nbsec()		
		IF nosymbols EQ 0 THEN BEGIN
			; boucle j puis i plus rapide que i puis j
			FOR j=0L,nbatt-1 DO BEGIN
				FOR i=0L,nbvar[isz]-1 DO BEGIN
					IF 1 THEN BEGIN
						VALEUR = ''
						IF is_fdl() THEN BEGIN
							VALEUR = pchar('')
							lint_unused = CALL_EXTERNAL (libname('filecdf'), 'MY_CDF_ATTGET_AUTO_GLUE',	$
											i, isz, pchar(attname[j]), VALEUR, /AUTO_GLUE)
							VALEUR = STRING (VALEUR)
						END ELSE BEGIN
							lint_unused = CALL_EXTERNAL (libname('filecdf'), 'MY_CDF_ATTGET',		$
											i, isz, attname[j], VALEUR)
						END
						code = VALEUR EQ '' ? -1 : 0
					END ELSE BEGIN
						code = lire_att_by_id_cdf (fd, attname[j], i, VALEUR, isz)
					END

					IF code EQ 0 THEN BEGIN
						novar = ~isz ? i : (nbvar[0]+i)
						; PATCH 09/10/2013
						; supprimer ler retours  la ligne dans les chaines de caractres
						IF is_string(VALEUR) THEN BEGIN
							remplacer, VALEUR, STRING(10b), ' '
							remplacer, VALEUR, STRING(13b), ' '
						END

						CASE STRUPCASE(attname[j]) OF
							'PROPERTY':		SYMBOLS[novar].property		= VALEUR
							'FILLVAL':		SYMBOLS[novar].fillval		= VALEUR
							'DELTA_PLUS':		SYMBOLS[novar].delta_plus	= VALEUR
							'DELTA_MINUS':		SYMBOLS[novar].delta_minus	= VALEUR
							'DELTA_PLUS_VAR':	SYMBOLS[novar].delta_plus	= VALEUR
							'DELTA_MINUS_VAR':	SYMBOLS[novar].delta_minus	= VALEUR
							'UNITS':		SYMBOLS[novar].units		= STRTRIM(VALEUR,2)

							'FIELDNAM':		BEGIN
											VALEUR = STRTRIM (VALEUR,2)
											IF VALEUR NE '' THEN SYMBOLS[novar].fieldnam		= SYMBOLS[novar].fieldnam EQ '' ? VALEUR : (SYMBOLS[novar].fieldnam EQ VALEUR ? SYMBOLS[novar].fieldnam : (SYMBOLS[novar].fieldnam+' / '+VALEUR))
										END
							'CATDESC':		BEGIN
											VALEUR = STRTRIM (VALEUR,2)
											IF VALEUR NE '' THEN SYMBOLS[novar].fieldnam		= SYMBOLS[novar].fieldnam EQ '' ? VALEUR : (SYMBOLS[novar].fieldnam EQ VALEUR ? SYMBOLS[novar].fieldnam : (SYMBOLS[novar].fieldnam+' / '+VALEUR))
										END

							'SI_CONVERSION':	SYMBOLS[novar].si_conversion	= VALEUR
							'DEPEND_0':		SYMBOLS[novar].depend_0		= VALEUR
							'DEPEND_1':		SYMBOLS[novar].depend_1		= VALEUR
							'DEPEND_2':		SYMBOLS[novar].depend_2		= VALEUR
							'DEPEND_3':		SYMBOLS[novar].depend_3		= VALEUR
							'DEPEND_4':		SYMBOLS[novar].depend_4		= VALEUR

							; on prend la chaine la plus courte (la moins dtaille)
							'LABL_PTR_1':		IF (WHERE(SYMBOLS.name EQ VALEUR))[0] NE -1 THEN SYMBOLS[novar].labl_ptr_1	= chaine_la_plus_courte(SYMBOLS[WHERE(SYMBOLS.name EQ VALEUR)].DATA,SYMBOLS[novar].labl_ptr_1)
							'LABEL_1':		IF (WHERE(SYMBOLS.name EQ VALEUR))[0] NE -1 THEN SYMBOLS[novar].labl_ptr_1	= chaine_la_plus_courte(SYMBOLS[WHERE(SYMBOLS.name EQ VALEUR)].DATA,SYMBOLS[novar].labl_ptr_1)
							'REPRESENTATION_1':	IF (WHERE(SYMBOLS.name EQ VALEUR))[0] NE -1 THEN SYMBOLS[novar].labl_ptr_1	= chaine_la_plus_courte(SYMBOLS[WHERE(SYMBOLS.name EQ VALEUR)].DATA,SYMBOLS[novar].labl_ptr_1)

							'VAR_TYPE':		SYMBOLS[novar].var_type		= STRTRIM(VALEUR,2)
							ELSE:
						END

					END
				END
			END
		END
		IF DEBUG THEN top2_ATT = nbsec()		
		IF DEBUG THEN duree_ATT = top2_ATT-top1_ATT
	END
	IF quick THEN code = fermer_cdf_c (FILENAME)
	IF DEBUG THEN top2_ALL = nbsec()
	IF DEBUG THEN duree_ALL += top2_ALL-top1_ALL
	IF DEBUG THEN PRINT,'----- FILENAME   -----  => '+FILENAME
	IF DEBUG THEN PRINT,'----- NOVARY     -----  => '+val_to_str(duree_NOVARY)
	IF DEBUG THEN PRINT,'----- VARY       -----  => '+val_to_str(duree_VARY)
	IF DEBUG THEN PRINT,'----- ATT        -----  => '+val_to_str(duree_ATT)
	IF DEBUG THEN PRINT,'----- ALL        -----  => '+val_to_str(duree_ALL)

	; Patch LABL_PTR_1
	ind = WHERE (SYMBOLS.LABL_PTR_1 NE '' AND SYMBOLS.DEPEND_1 NE '' AND SYMBOLS.DEPEND_2 EQ '' AND SYMBOLS.DEPEND_3 EQ '' AND SYMBOLS.DEPEND_4 EQ '')
	IF ind[0] NE -1 THEN BEGIN
		SYMBOLS[ind].DEPEND_1 = ''
	END

	IF necessaire THEN BEGIN
		IF majority EQ 'F' THEN BEGIN
			; Inversion de l'ordre des DEPEND_[1234]
			; Si SIZES a une seule dimension: ne rien faire
			; Si SIZES a 2 dimensions: permuter DEPEND_1/DEPEND_2
			; Si SIZES a 3 dimensions: permuter DEPEND_1/DEPEND_3
			; Si SIZES a 4 dimensions: permuter DEPEND_1/DEPEND_4 et DEPEND_2/DEPEND_3
			FOR i=0L,N_ELEMENTS(SYMBOLS)-1 DO BEGIN
				nbdims = N_ELEMENTS(string_to_data(SYMBOLS[i].SIZES))
				IF nbdims EQ 2 THEN BEGIN
					tmp = SYMBOLS[i].DEPEND_1
					SYMBOLS[i].DEPEND_1 = SYMBOLS[i].DEPEND_2
					SYMBOLS[i].DEPEND_2 = tmp
				END ELSE IF nbdims EQ 3 THEN BEGIN
					tmp = SYMBOLS[i].DEPEND_1
					SYMBOLS[i].DEPEND_1 = SYMBOLS[i].DEPEND_3
					SYMBOLS[i].DEPEND_3 = tmp
				END ELSE IF nbdims EQ 4 THEN BEGIN
					tmp = SYMBOLS[i].DEPEND_1
					SYMBOLS[i].DEPEND_1 = SYMBOLS[i].DEPEND_4
					SYMBOLS[i].DEPEND_4 = tmp
					tmp = SYMBOLS[i].DEPEND_2
					SYMBOLS[i].DEPEND_2 = SYMBOLS[i].DEPEND_3
					SYMBOLS[i].DEPEND_3 = tmp
				END
			END
		END
	END

	symbols_name = SYMBOLS.name
	symbols_name = remove_extension(remove_ext,symbols_name)

	IF DEBUG THEN top1_NBRECORDS = nbsec()
	; Dtermination du nombre de lignes en fonction de la valeur de selection_date

	NBRECORDS = -1L
	ind1 = WHERE (SYMBOLS.VALUE_TYPE EQ 'ISO_TIME' AND SYMBOLS.DATA EQ '')
	IF ind1[0] EQ -1 THEN BEGIN
		; arrive avec "C4_CP_PEA_PITCH_SPIN_DEFlux__20050101_000000_20050102_000000_V090627.cdf"
		PRINT,'*** WARNING *** NO ISO_TIME VARIABLE'
		RETURN, 0
	END
	selection = selection_date
	IF selection EQ 'Automatic' THEN selection = REFORM(SYMBOLS[ind1[0]].name)
	selection = selection[0] ; pour supprimer la dimension inutile Array[1] qui pose problme  chaine_contient
	selection = remove_extension(remove_ext,selection)


	IF STRMID(selection, STRLEN(selection)-6,6) EQ '_epoch' THEN BEGIN
		; rustine pour Themis avec codage spcial du temps
		tmp = STRMID(selection,0,STRLEN(selection)-6)+'_time'
		FOR i=0L,N_ELEMENTS(symbols_name)-1 DO BEGIN
			IF chaine_contient(symbols_name[i], tmp) THEN BEGIN
				NBRECORDS = SYMBOLS[i].nblignes
				BREAK
			END
		END
	END
	IF STRMID(selection, STRLEN(selection)-8,8) EQ '_epoch16' THEN BEGIN
		; rustine pour Themis avec codage spcial du temps
		tmp = STRMID(selection,0,STRLEN(selection)-8)+'_time'
		FOR i=0L,N_ELEMENTS(symbols_name)-1 DO BEGIN
			IF chaine_contient (symbols_name[i], tmp) THEN BEGIN
				NBRECORDS = SYMBOLS[i].nblignes
				BREAK
			END
		END
	END

	IF NBRECORDS EQ -1 THEN BEGIN
		IF ind1[0] NE -1 THEN BEGIN
			FOR i=0L, N_ELEMENTS(ind1)-1 DO BEGIN
				IF symbols_name[ind1[i]] EQ selection THEN BEGIN
					NBRECORDS = SYMBOLS[ind1[i]].nblignes
					BREAK
				END
			END
		END
	END

	IF NBRECORDS EQ -1 THEN BEGIN
		IF ind1[0] NE -1 THEN BEGIN
			FOR i=0L, N_ELEMENTS(ind1)-1 DO BEGIN
				IF chaine_contient(symbols_name[ind1[i]], selection) THEN BEGIN
					NBRECORDS = SYMBOLS[ind1[i]].nblignes
					BREAK
				END
			END
		END
	END

	IF NBRECORDS EQ -1 THEN BEGIN
		; La variable selection_date n'a pas t trouv. On prend la premire variable de type 'ISO_TIME' (ind1[0])
		IF ind1[0] NE -1 THEN BEGIN
			selection_date = symbols_name[ind1[0]]
			NBRECORDS = SYMBOLS[ind1[0]].nblignes
		END
	END

	IF NBRECORDS EQ -1 THEN BEGIN
		; Damned: aucune variable de type 'ISO_TIME'
		NBRECORDS = 0L
	END
	IF DEBUG THEN top2_NBRECORDS = nbsec()
	IF DEBUG THEN duree_NBRECORDS = top2_NBRECORDS-top1_NBRECORDS
	IF DEBUG THEN PRINT,'----- NBRECORDS  -----  => '+val_to_str(duree_NBRECORDS)

	IF DEBUG THEN top1_CORRECTION = nbsec()

	corriger_symbols, symbol0, SYMBOLS, modif

	FOR i=0L,N_ELEMENTS(SYMBOLS)-1 DO BEGIN
		IF 0 AND modif AND majority EQ 'F' THEN BEGIN
			; Inversion de l'ordre des DEPEND_[1234]
			; Si SIZES a une seule dimension: ne rien faire
			; Si SIZES a 2 dimensions: permuter DEPEND_1/DEPEND_2
			; Si SIZES a 3 dimensions: permuter DEPEND_1/DEPEND_3
			; Si SIZES a 4 dimensions: permuter DEPEND_1/DEPEND_4 et DEPEND_2/DEPEND_3
			nbdims = N_ELEMENTS(string_to_data(SYMBOLS[i].SIZES))
			IF nbdims EQ 2 THEN BEGIN
				tmp = SYMBOLS[i].DEPEND_1
				SYMBOLS[i].DEPEND_1 = SYMBOLS[i].DEPEND_2
				SYMBOLS[i].DEPEND_2 = tmp
			END ELSE IF nbdims EQ 3 THEN BEGIN
				tmp = SYMBOLS[i].DEPEND_1
				SYMBOLS[i].DEPEND_1 = SYMBOLS[i].DEPEND_3
				SYMBOLS[i].DEPEND_3 = tmp
			END ELSE IF nbdims EQ 4 THEN BEGIN
				tmp = SYMBOLS[i].DEPEND_1
				SYMBOLS[i].DEPEND_1 = SYMBOLS[i].DEPEND_4
				SYMBOLS[i].DEPEND_4 = tmp
				tmp = SYMBOLS[i].DEPEND_2
				SYMBOLS[i].DEPEND_2 = SYMBOLS[i].DEPEND_3
				SYMBOLS[i].DEPEND_3 = tmp
			END
		END
	END

	IF DEBUG THEN top2_CORRECTION = nbsec()
	IF DEBUG THEN duree_CORRECTION = top2_CORRECTION-top1_CORRECTION
	IF DEBUG THEN PRINT,'----- CORRECTION -----  => '+val_to_str(duree_CORRECTION)

	; PATCH DOUBLE -> FLOAT pour ROSINA (pour suivre la mme logique que dans les cef)
	ind = WHERE (SYMBOLS.value_type EQ 'DOUBLE')
	IF ind[0] NE -1 THEN SYMBOLS[ind].value_type = 'FLOAT'

	IF fd NE -1 THEN BEGIN
		CDF_CLOSE,fd
	END
	top2 = nbsec()
	IF ~quiet THEN PRINT,FILENAME+' read_cdf_get_symbols ***** ' + val_to_str(top2-top1)+' *****'
	!QUIET = OLD_QUIET
	RETURN, 1

error:

	PRINT,FILENAME+' read_cdf_get_symbols_error ***** '
	IF fd NE -1 THEN BEGIN
		CDF_CLOSE,fd
	END
	!QUIET = OLD_QUIET
	RETURN, 0

END

;-------------------------------------------------------------------------------
FUNCTION read_cdf_get_data,	$
;-------------------------------------------------------------------------------
	filename,					$	; LINT_PROTOTYPE input
	FORMAT,						$	; LINT_PROTOTYPE output
	NBRECORDS,					$	; LINT_PROTOTYPE output
	NBLINES0,					$	; LINT_PROTOTYPE output
	STRUCT,						$	; LINT_PROTOTYPE output
	INFO,						$	; LINT_PROTOTYPE output
	DATA,						$	; LINT_PROTOTYPE output
	rec_start=rec_start,				$	; LINT_PROTOTYPE input
	rec_count=rec_count,				$	; LINT_PROTOTYPE input
	modify_float_fillval=modify_float_fillval,	$	; LINT_PROTOTYPE input
	nomalloc=nomalloc,				$	; LINT_PROTOTYPE input
	quiet=quiet						; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((1:ok 0:pb)) Lit les donnes d'un fichier CDF dja analys et retourne un tableau de structure avec les donnes.
;
; Input:
; -----
; - filename:			filename to read
; - FORMAT:			number of format for Date/Time
; - NBRECORDS:			nbrecords
; - NBLINES0:			number of lines in header
; - STRUCT:			output from read_ascii_get_structure()
; - INFO:			output from read_ascii_get_structure()
;
; Keywords:
; ---------
; - rec_start:			start read at record rec_start (0=first)
; - rec_count:			number of records to read
; - /modify_float_fillval:	to modify FLOAT fillval by -1e31
; - /nomalloc:			if DATA has already been allocated
; - /quiet: 			no messages
;
; Output:
; -------
; - DATA:			array of STRUCT with data read in filename
;				'ISO_TIME' and 'ISO_TIME_RANGE' variables are stored as number of millsecondes since 01/01/1958 00:00:00.000 in a double
;				use date_to_tu to convert these fields to year, month, day, hour, minute, seconde
;
;
; Return code:
; ------------
;	1 = OK
;	0 = PB
;				'ERROR: INCORRECT FORMAT IN ' + filename
;-------------------------------------------------------------------------------

	lint_unused = FORMAT ; pour ne pas avoir de message de LINT
	lint_unused = NBLINES0 ; pour ne pas avoir de message de LINT

	IF N_ELEMENTS(rec_start) 		EQ 0 THEN rec_start 		= 0L
	IF N_ELEMENTS(rec_count) 		EQ 0 THEN rec_count 		= NBRECORDS-rec_start
	IF N_ELEMENTS(modify_float_fillval)	EQ 0 THEN modify_float_fillval	= 0
	IF N_ELEMENTS(nomalloc) 		EQ 0 THEN nomalloc 		= 0
	IF N_ELEMENTS(quiet) 			EQ 0 THEN quiet 		= 0

	IF rec_count LE 0 THEN RETURN, 0

	IF ~nomalloc THEN DATA = REPLICATE (STRUCT,rec_count)

	top1 = nbsec()
	TAGNAMES = TAG_NAMES(DATA)
	inf = [['','','']]

	; on peut sortir sur N_ELEMENTS(INFO.NAME) si TAGNAMES contient des champs GSM rajouts a la structure (ex: FGM cluster)

	; ex:	TAGNAMES			INFO.NAME
	;
	;	date				epoch
	;	duration			duration
	;	energy_table_1			energy_table
	; 	energy_table_2			energy_table
	;	...
	;	energy_256			energy_table
	;	delta_plus_energy_table_1	delta_plus_energy_table
	;	delta_plus_energy_table_2	delta_plus_energy_table
	;	...
	;	delta_plus_energy_table_256	delta_plus_energy_table
	;
	;	il faut mettre dans inf:
	;	=> date				epoch
	;	=> energy_table_1		energy_table
	;	=> delta_plus_energy_table_1	delta_plus_energy_table

	; ex:	TAGNAMES
	;	date				epoch
	;	duration			duration
	;	energy_table			energy_table
	;					energy_table
	;					...
	;					energy_table
	;	delta_plus_energy_table		delta_plus_energy_table
	;					delta_plus_energy_table
	;	...
	;					delta_plus_energy_table
	;
	;	il faut mettre dans inf:
	;	=> date				epoch
	;	=> energy_table			energy_table
	;	=> delta_plus_energy_table	delta_plus_energy_table
	

	i1 = 0L	; / TAGNAMES
	i2 = 0L	; / INFO.NAME

	nb1 = N_ELEMENTS(TAGNAMES)
	nb2 = N_ELEMENTS(INFO.NAME)
	WHILE 1 DO BEGIN
		val = [TAGNAMES[i1], INFO.NAME[i2], '']
		IF inf[0,0] EQ '' THEN BEGIN
			inf = [val]
		END ELSE IF val[1] NE inf[1,N_ELEMENTS(inf[0,*])-1] THEN BEGIN
			inf = [[inf],[val]]
		END
		i2 += N_ELEMENTS(DATA[0].(i1))
		IF i2 GE nb2 THEN BREAK
		i1++
		IF i1 GE nb1 THEN BREAK
	END
	lire_struct_cdf_c, filename, DATA, inf[0,*], inf[1,*], inf[2,*], rec_start, rec_count

	top2 = nbsec()

	; rustine pour Themis avec codage spcial du temps
	ind_epoch = WHERE (STRMID(INFO.name,5,6,/REVERSE_OFFSET) EQ '_epoch', nbepoch)
	FOR j=0L,nbepoch-1 DO BEGIN
		i = ind_epoch[j]
		ind = WHERE (INFO.name EQ STRMID(INFO.name[i],0,STRLEN(INFO.name[i])-6)+'_time')
		IF ind[0] NE -1 THEN BEGIN
			; var_epoch (INFO.name[i]) et var_time (INFO.name[ind[i]]) existe
			PRINT,'Compute ' + val_to_str(N_ELEMENTS(DATA)) + ' ' + INFO.name[i]
			DATA.(i) = DATA.(ind[0])*1000D + tu_to_date(1970,1,1,0,0,0)
			
		END
	END
	ind_epoch16 = WHERE (STRMID(INFO.name,5,6,/REVERSE_OFFSET) EQ '_epoch16', nbepoch16)
	FOR j=0L,nbepoch16-1 DO BEGIN
		i = ind_epoch16[j]
		ind = WHERE (INFO.name EQ STRMID(INFO.name[i],0,STRLEN(INFO.name[i])-8)+'_time')
		IF ind[0] NE -1 THEN BEGIN
			; var_epoch (INFO.name[i]) et var_time (INFO.name[ind[i]]) existe
			PRINT,'Compute ' + val_to_str(N_ELEMENTS(DATA)) + ' ' + INFO.name[i]
			DATA.(i) = DATA.(ind[0])*1000D + tu_to_date(1970,1,1,0,0,0)
			
		END
	END

	IF ~quiet THEN PRINT,'reading ' + filename + ' in '+ val_to_str_2decimales(top2-top1)+' sec'

	RETURN, 1

END 

;-------------------------------------------------------------------------------
FUNCTION lire_cdf,		$
;-------------------------------------------------------------------------------
	nom,			$	; LINT_PROTOTYPE input
	epoch,			$	; LINT_PROTOTYPE input
	epoch_reference,	$	; LINT_PROTOTYPE input
	phase,			$	; LINT_PROTOTYPE input
	noms,			$	; LINT_PROTOTYPE output
	fillval,		$	; LINT_PROTOTYPE output
	nblignes,		$	; LINT_PROTOTYPE output
	use_CDF_C_library,	$	; LINT_PROTOTYPE input
	data=data,		$	; LINT_PROTOTYPE output
	REC_COUNT=REC_COUNT,	$	; LINT_PROTOTYPE input
	REC_START=REC_START,	$	; LINT_PROTOTYPE input
	cdfdata=cdfdata,	$	; LINT_PROTOTYPE input
	only_saa=only_saa,	$	; LINT_PROTOTYPE input
	infocdf=infocdf			; LINT_PROTOTYPE output
;-------------------------------------------------------------------------------
; ((1:ok 0:pb)) Retourne les noms des champs et le nombre de lignes (si phase==1), les donnes (si phase==2), le nombre de nblignes (si phase==3) ou le champ date (phase==4).
;
; Si phase=1, on demande juste a connaitre les noms des champs et le nombre de lignes
; Si phase=2, on demande a lire reellement les donnes dans data
; Si phase=3, on demande juste a connaitre nblignes
; Si phase=4, on demande juste a connaitre le champ date
;
; Si N_ELEMENTS(cdfdata) NE 0 alors cdfdata contiendra:
; - le champ ATT
; - le champ NOVARY
; - le champ VARY
;
; epoch: nom de la variable CDF contenant la datation
; epoch_reference: '1970' si epoch contient le nb de ms/1er janvier 1970 ou '' si codage standard variable de type EPOCH
;-------------------------------------------------------------------------------
; LINT_VARIABLES cdfdata_ATT cdfdata_NOVARY cdfdata_VARY

	ON_IOERROR, erreur

	COMMON COMMON_LIRE_CDF, attvar

	IF N_ELEMENTS(only_saa) EQ 0 THEN only_saa = 0
	IF N_ELEMENTS(infocdf)  EQ 0 THEN infocdf = 0

	IF epoch NE '' AND (use_CDF_C_library OR is_gdl()) THEN BEGIN
		; Pour les fichiers CLUSTER PP_FGM et UP_FGM:
		; noms PP_FGM de 20001101  20051231: Epoch contient 'PP_FGM'
		; noms UP_FGM  partir de 20060101: Epoch contient 'UP_FGM'
		; Dans ce cas la epoch contient les 2 noms possibles spars par un blanc
		epochs = STRSPLIT (epoch, /EXTRACT)
		nom_epoch = epochs[0]
		IF phase EQ 3 THEN BEGIN
			nblignes = get_nblignes_cdf_c (nom, nom_epoch)
			IF nblignes EQ 0 AND N_ELEMENTS(epochs) EQ 2 THEN BEGIN
				nom_epoch = epochs[1]
				nblignes = get_nblignes_cdf_c (nom, nom_epoch)
			END
			RETURN, 1
		END ELSE IF phase EQ 4 THEN BEGIN
			nblignes = get_nblignes_cdf_c (nom, nom_epoch)
			IF nblignes EQ 0 AND N_ELEMENTS(epochs) EQ 2 THEN BEGIN
				nom_epoch = epochs[1]
				nblignes = get_nblignes_cdf_c (nom, nom_epoch)
			END
			data = REPLICATE (0d, nblignes)
			lire_var_cdf_c, nom, data, nom_epoch, 0L, nblignes
			IF epoch_reference NE '' THEN BEGIN
				decalage = (JULDAY (1, 1, epoch_reference) - JULDAY (1, 1, 1958))*86400d
				data = 1000d * (data + decalage)
			END
			RETURN, 1
		END
	END

	IF is_gdl() THEN BEGIN ; ok
		; sinon "% LIRE_CDF: Ambiguous: Variable is undefined: CDF_OPEN or: Function not found: CDF_OPEN"
		PRINT, 'lire_cdf(nom=',nom,' epoch=',epoch,' phase=',phase,')'
		nblignes = 0L
		RETURN, 0 ; PB
	END

	top1 = nbsec()
	noms  = ''
	id = -1 ; car sinon, si CDF_OPEN plante, id ne sera pas defini
	id = CDF_OPEN (nom)
	inq = CDF_INQUIRE (id)
	nblignes = 0L
	cdfdataon = N_ELEMENTS(cdfdata) NE 0

	IF infocdf THEN BEGIN
		PRINT
		PRINT,nom
	END

	IF (phase EQ 1 AND infocdf) OR cdfdataon THEN BEGIN
		attvar = '' ; attributs VARIABLE_SCOPE
		FOR i=0L,inq.natts-1 DO BEGIN 				; Boucle sur tous les attributs
			CDF_ATTINQ, id, i, name, scope, maxentry
			IF scope EQ "GLOBAL_SCOPE" AND maxentry NE -1 THEN BEGIN
				CDF_CONTROL, id, ATTRIBUTE=name, get_attr_info=attr_info  ; pour connaitre le nombre de ligne dans l'attribut
				val = ''
				FOR j=0L,attr_info.MAXGENTRY DO BEGIN
					IF lire_att_cdf (id, name, j, tmp) EQ 1 THEN BEGIN
						IF infocdf THEN PRINT,val_to_str(i+1), ' ', scope, ' ',name, ' = ', tmp
						val += tmp
					END
				END
				IF cdfdataon THEN BEGIN
					tmp = CREATE_STRUCT ('var', scope, 'name', name, 'val', STRING(val))
					IF N_ELEMENTS(cdfdata_ATT) EQ 0 THEN BEGIN
						cdfdata_ATT = [tmp]
					END ELSE BEGIN
						cdfdata_ATT = [cdfdata_ATT, tmp]
					END
				END
			END ELSE BEGIN
				IF infocdf THEN PRINT,val_to_str(i+1),'  ', scope, ' ',name
				IF scope EQ "VARIABLE_SCOPE" THEN BEGIN
					attvar = attvar[0] EQ '' ? [name] : [attvar,name]
				END
			END
		END
	END

	IF phase EQ 1 THEN BEGIN
		REC_START = 0
		REC_COUNT = 1
	END

	nbvar = [inq.nvars , inq.nzvars]

	; A cause des fichiers wi_h0_mfi_YYYYMMDD_vxy.cdf: contiennent 'Epoch', 'Epoch3' et 'Epoch1'
	premier_Epoch = 0	; positionne a 1 la premiere fois que l'on rencontre une variable commencant par 'Epoch'

	FOR isz=0L,1 DO BEGIN ; Variables puis Zvariables
		FOR i=0L,nbvar[isz]-1 DO BEGIN
			res1 = CDF_VARINQ (id, i, zvariable=isz)

			IF res1.name EQ 'Distribution_LABL_1' THEN CONTINUE ; pb fichier STEREO DIST
			IF res1.name EQ 'Distribution_LABL_2' THEN CONTINUE ; pb fichier STEREO DIST
			IF res1.name EQ 'Ve_LABL_1' THEN CONTINUE ; pb fichier STEREO MOMENTS
			IF res1.name EQ 'Pe_LABL_1' THEN CONTINUE ; pb fichier STEREO MOMENTS
			IF res1.name EQ 'He_LABL_1' THEN CONTINUE ; pb fichier STEREO MOMENTS

			IF phase LE 2 AND (infocdf OR cdfdataon) THEN BEGIN

				IF infocdf THEN BEGIN
					CDF_COMPRESSION, id, variable=i, zvariable=isz, GET_VAR_COMPRESSION=var_compression
					list_compression = ['No Compression','Run-Length Encoding','Huffman','Adaptive Huffman','','GZIP']
					info_compression = list_compression[var_compression]
					PRINT
					PRINT,val_to_str(i+1)+')  ' + res1.name + '  '+res1.RECVAR + ' (' + info_compression + ')' 
				END

				IF attvar[0] NE '' THEN BEGIN

					FOR j=0L,N_ELEMENTS(attvar)-1 DO BEGIN

						code = lire_att_cdf (id, attvar[j], res1.name, val)
						IF code EQ 0 THEN BEGIN
							IF infocdf THEN PRINT,'   ' + attvar[j]+' =  ', val
							IF cdfdataon THEN BEGIN
								somme_val = ''
								FOR k=0L,N_ELEMENTS(val)-1 DO BEGIN
									IF k NE 0 THEN somme_val += ' , '
									somme_val += STRING(val[k])
								END
								tmp = CREATE_STRUCT ('var', res1.name, 'name', attvar[j], 'val', somme_val)
								IF N_ELEMENTS(cdfdata_ATT) EQ 0 THEN BEGIN
									cdfdata_ATT = [tmp]
								END ELSE BEGIN
									cdfdata_ATT = [cdfdata_ATT, tmp]
								END
							END
						END

					END

				END

				IF res1.RECVAR EQ 'VARY' THEN BEGIN
					IF infocdf THEN BEGIN
						lire_var_cdf, id,res1.name,var,type
						HELP, REFORM(var)
					END
				END

				IF res1.RECVAR NE 'VARY' THEN BEGIN

					lire_var_cdf, id,res1.name,var,type
					IF infocdf THEN HELP, REFORM(var)
					;IF type EQ 'CDF_CHAR' THEN PRINT,STRING(var) ELSE PRINT, var

					IF cdfdataon THEN BEGIN
						IF N_ELEMENTS(cdfdata_NOVARY) EQ 0 THEN BEGIN
							cdfdata_NOVARY = CREATE_STRUCT (corriger_nomchamp(res1.name), var)
						END ELSE BEGIN
							cdfdata_NOVARY = CREATE_STRUCT (cdfdata_NOVARY, corriger_nomchamp(res1.name), var)
						END
					END

				END

			END

			; Si only_saa (scalar and array): je ne traite pas les matrices qui sont lourdes  grer (res1.dimvar).
			; Je traite les scalaires et les tableaux uniquement
			ok_saa = (~only_saa) OR (isz AND N_ELEMENTS(res1.dimvar) LE 1)
			IF res1.RECVAR EQ 'VARY' AND res1.datatype NE 'CDF_CHAR' AND ok_saa THEN BEGIN

				; Le champ res1.name permet-il de dterminer le nombre de lignes ?
				IF epoch NE '' THEN BEGIN
					is_epoch = res1.name EQ epoch
				END ELSE BEGIN
					is_epoch = res1.datatype EQ 'CDF_EPOCH' OR res1.datatype EQ 'CDF_EPOCH16' OR res1.datatype EQ 'CDF_TIME_TT2000'
				END

				IF phase EQ 3 OR phase EQ 4 THEN BEGIN ; On ne s'interesse qu' la variable 'Epoch'

					IF is_epoch THEN BEGIN ; Datation
						IF phase EQ 3 THEN BEGIN
							IF use_CDF_C_library THEN BEGIN
								nblignes = get_nblignes_cdf_c (nom, res1.name)
							END ELSE BEGIN
								CDF_CONTROL, id, variable=res1.name, zvariable=isz, get_var_info=vinfo
								nblignes = vinfo.maxrec+1
							END
						END ELSE IF phase EQ 4 THEN BEGIN
							IF use_CDF_C_library THEN BEGIN
								; marche bien maintenant avec la vue ace_HIRES.cl
								nblignes = get_nblignes_cdf_c (nom, res1.name)
								data = REPLICATE (0d, nblignes)
								lire_var_cdf_c, nom, data, res1.name, 0L, nblignes
								IF epoch_reference NE '' THEN BEGIN
									decalage = (JULDAY (1, 1, epoch_reference) - JULDAY (1, 1, 1958))*86400d
									data = 1000d * (data + decalage)
								END
							END ELSE BEGIN
								;PRINT,"Lecture!"
								lire_var_cdf, id,res1.name,data,type
								nblignes = N_ELEMENTS(data)
								IF epoch_reference NE '' THEN BEGIN
									decalage = (JULDAY (1, 1, epoch_reference) - JULDAY (1, 1, 1958))*86400d
									data = 1000d * (data + decalage)
								END ELSE BEGIN
									CDF_EPOCH, decalage, 1958, 1, 1, 0, 0, 0, /compute_epoch
									data -= decalage
								END
							END
						END
						CDF_CLOSE,id
						IF infocdf THEN PRINT,nom+' Phase ' + val_to_str(phase)+' done in '+val_to_str_2decimales(nbsec()-top1)+' sec'
						RETURN, 1
					END

				END ELSE IF phase EQ 1 OR phase EQ 2 THEN BEGIN

					IF infocdf THEN top3 = nbsec()
					lire_var_cdf, id, res1.name, var, type, REC_COUNT=REC_COUNT, REC_START=REC_START
					IF infocdf THEN PRINT,'Phase '+val_to_str(phase)+' read '+val_to_str(REC_COUNT)+' lines of ' + res1.name+' in '+val_to_str_2decimales(nbsec()-top3)+' sec'

					IF cdfdataon THEN BEGIN
						IF N_ELEMENTS(cdfdata_VARY) EQ 0 THEN BEGIN
							cdfdata_VARY = CREATE_STRUCT (corriger_nomchamp(res1.name), var)
						END ELSE BEGIN
							cdfdata_VARY = CREATE_STRUCT (cdfdata_VARY, corriger_nomchamp(res1.name), var)
						END
					END ELSE BEGIN

						IF ~premier_Epoch AND is_epoch THEN BEGIN ; Datation
							IF use_CDF_C_library THEN BEGIN
								nblignes = get_nblignes_cdf_c (nom, res1.name)
							END ELSE BEGIN
								CDF_CONTROL, id, variable=res1.name, zvariable=isz, get_var_info=vinfo
								nblignes = vinfo.maxrec+1
							END
							premier_Epoch = 1
						END

						IF phase EQ 2 THEN BEGIN

							; Remplacement, si necessaire, des fillval par -1E31
							IF CDF_ATTEXISTS (id, "FILLVAL", res1.name) THEN BEGIN
								;PRINT, "Lecture FILLVAL"
								CDF_ATTGET, id, 'FILLVAL', res1.name, fillval1
								IF fillval1 NE -1E31 THEN BEGIN
									ind = WHERE (var EQ fillval1)
									IF ind[0] NE -1 THEN BEGIN
										var = DOUBLE(var)
										var[ind] = -1E31
									END
								END
							END

							; Traitement du bug 'Np_cm3' + 'Vp_km_sec' + 'Tp_K' + 'He_ratio' valant -1000 dans ACE alors que
							; CDF_ATTGET retourne fillval = -1E31!
							; On supprime les valeurs negatives!
							IF res1.name EQ 'Np' OR res1.name EQ 'Vp' OR res1.name EQ 'Tpr' OR res1.name EQ 'He_ratio' THEN BEGIN
								ind = WHERE (var LT 0)
								IF ind[0] NE -1 THEN BEGIN
									var = DOUBLE(var)
									var[ind] = -1E31
								END
							END

							; Traitement du Warning: converting data to signed integers
							IF res1.datatype EQ 'CDF_UINT2' THEN BEGIN
								; Arrive par exemple avec les fichiers LEP
								ind = WHERE ((var LT 0) AND (var NE -1E31))
								IF ind[0] NE -1 THEN BEGIN
									var = LONG(var)
									var[ind] += 65536L
								END
							END

						END

						getnbxy_cdf, var,nbX,nbY
						IF phase EQ 2 AND infocdf THEN BEGIN
							HELP,var
							PRINT,'   '+res1.name+'[',val_to_str(nbX)+','+val_to_str(nbY)+']'
						END

						IF nbY EQ 1 THEN BEGIN

							nom_court = is_epoch ? 'date' : res1.name
							noms = noms[0] EQ '' ? [nom_court] : [noms, nom_court]
							IF phase EQ 2 THEN BEGIN
								IF nom_court EQ 'date' THEN BEGIN ; Codage de epoch en nb msec depuis 1/1/1958
									IF type EQ 'CDF_TIME_TT2000' THEN BEGIN
										var = tt2000_to_date (var)
									END ELSE IF type EQ 'CDF_EPOCH16' THEN BEGIN
										var = epoch16_to_date (var)
									END ELSE BEGIN
										CDF_EPOCH, decalage, 1958, 1, 1, 0, 0, 0, /compute_epoch
										var -= decalage
									END
								END
								k = N_ELEMENTS(noms)-1
								code = transferer_cdf (data, k, var)
								IF code NE 1 THEN BEGIN
									PRINT, 'WARNING: Not exactly ' + val_to_str(N_ELEMENTS(data)) + ' values in ' + res1.name
								END
								fillval = k EQ 0 ? [-1E31] : [fillval, -1E31]
							END

						END ELSE BEGIN

							var = REFORM (var, nbX, nbY)

							FOR j=1L,nbY DO BEGIN
								nom_long = res1.name+'_'+val_to_str(j)
								noms = noms[0] EQ '' ? [nom_long] : [noms, nom_long]
								IF phase EQ 2 THEN BEGIN
									k = N_ELEMENTS(noms)-1
									code = transferer_cdf (data, k, var[*,j-1])
									IF code NE 1 THEN BEGIN
										PRINT, 'WARNING: Not exactly ' + val_to_str(N_ELEMENTS(data)) + ' values in ' +  nom_long
									END
									fillval = k EQ 0 ? [-1E31] : [fillval, -1E31]
								END
							END
						END
					END

				END
			END
		END
	END

	CDF_CLOSE, id
	IF infocdf THEN PRINT,nom + ' Phase ' + val_to_str(phase)+'  done in '+val_to_str_2decimales(nbsec()-top1)+' sec'

	IF nblignes EQ 0 AND ~cdfdataon THEN RETURN,0

	IF cdfdataon THEN BEGIN
		cdfdata_existe = 0
		IF N_ELEMENTS(cdfdata_ATT) NE 0 THEN BEGIN
			cdfdata = CREATE_STRUCT ('ATT', cdfdata_ATT)
			cdfdata_existe = 1
		END
		IF N_ELEMENTS(cdfdata_NOVARY) NE 0 THEN BEGIN
			cdfdata = ~cdfdata_existe ? CREATE_STRUCT ('NOVARY', cdfdata_NOVARY) : CREATE_STRUCT (cdfdata, 'NOVARY', cdfdata_NOVARY)
			cdfdata_existe = 1
		END
		IF N_ELEMENTS(cdfdata_VARY) NE 0 THEN BEGIN
			cdfdata = ~cdfdata_existe ? CREATE_STRUCT ('VARY', cdfdata_VARY) : CREATE_STRUCT (cdfdata, 'VARY', cdfdata_VARY)
			cdfdata_existe = 1
		END
	END

	RETURN, 1

erreur:
	IF id NE -1 THEN CDF_CLOSE, id

	;dcommenter si on veut essayer d'enchainer sur la lecture de Net CDF
	;code = lire_netcdf (nom, phase, infocdf, noms, fillval, nblignes, data=data, $
	;	REC_COUNT=REC_COUNT,REC_START=REC_START, cdfdata=cdfdata)

	nblignes = 0
	RETURN, 0

END

;-------------------------------------------------------------------------------
FUNCTION read_cdf,			$
;-------------------------------------------------------------------------------
        FILENAME,			$	; LINT_PROTOTYPE input
        DATA,				$	; LINT_PROTOTYPE output
        ATT,				$	; LINT_PROTOTYPE [output]
        NOVARY,				$	; LINT_PROTOTYPE [output]
        VARNAMES,			$	; LINT_PROTOTYPE [output]
        NAMES,				$	; LINT_PROTOTYPE [output]
        quiet=quiet,			$	; LINT_PROTOTYPE input
        noatt=noatt,			$	; LINT_PROTOTYPE input
        nonb=nonb,			$	; LINT_PROTOTYPE input
        remove_ext=remove_ext,		$	; LINT_PROTOTYPE input
        selection_date=selection_date,	$	; LINT_PROTOTYPE input
        selection_data=selection_data,	$	; LINT_PROTOTYPE input
        selection_type=selection_type		; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((1:ok 0:pb)) Lit le fichier CEF donn en argument et retourne les donnes dans un tableau de structure.
;
; Input:
; -----
; - FILENAME:                   filename to read
;
; Keywords:
; ---------
; - /quiet:                     no messages
;
; Output:
; -------
; - ATT:                        array of structure { v1:'', v2:'', v3:'', v4:''} with all attributes found in header
; - NOVARY:                     structure with variables containing a non empty DATA field
; - VARNAMES:                   names of variables with empty DATA field
; - NAMES:                      names of fields in the structure
; - DATA:                       array of STRUCT with data read in FILENAME
;                               'ISO_TIME' and 'ISO_TIME_RANGE' variables are stored as number of millsecondes since 01/01/1958 00:00:00.000 in a double
;                               use date_to_tu to convert these fields to year, month, day, hour, minute, seconde
;
; Return code:
; ------------
;       1 = OK
;       0 = PB
;                               'ERROR: INCORRECT FORMAT IN ' + FILENAME
;-------------------------------------------------------------------------------

        IF N_ELEMENTS(quiet)		EQ 0 THEN quiet = 0
        IF N_ELEMENTS(selection_date)	EQ 0 THEN selection_date = 'Automatic'
        IF N_ELEMENTS(selection_data)	EQ 0 THEN selection_data = 'Automatic'
        IF N_ELEMENTS(selection_type)	EQ 0 THEN selection_type = 'count3d'

	code = read_cdf_get_symbols (FILENAME, ATT, NOVARY, SYMBOLS, format, NBRECORDS, nblignes0, $
					selection_date=selection_date, $
					quiet=quiet, noatt=noatt, nonb=nonb, remove_ext=remove_ext)

        IF ~code THEN RETURN, 0

        code = read_ascii_analyse_symbols (SYMBOLS, VARNAMES, REALVARNAMES, TYPES, SIZES, FILLVALS, UNITS, NAME_TIME, NAME_THETA, UNIT_THETA, NAME_PHI, UNIT_PHI, NAME_ENERGY, UNIT_ENERGY)
        IF ~code THEN RETURN, 0

        code = read_ascii_get_structure (VARNAMES, REALVARNAMES, TYPES, SIZES, FILLVALS, NAMES, STRUCT, INFO, quiet=quiet)
        IF ~code THEN RETURN, 0

        nbalire = NBRECORDS
        taille = get_structure_length (STRUCT)
        IF ~quiet THEN BEGIN
                msg = basename(FILENAME)+' Reading ' + val_to_str(nbalire)+' lines ('+val_to_str_2decimales(DOUBLE(nbalire)*taille/1e6)+' Mbytes)'
                PRINT, msg
        END

        code = read_cdf_get_data (FILENAME, format, NBRECORDS, nblignes0, STRUCT, INFO, DATA, /modify_float_fillval, quiet=quiet)
        RETURN, code

END

;-------------------------------------------------------------------------------
PRO afficher_info_cdf,	$
;-------------------------------------------------------------------------------
	fichier,	$ ; LINT_PROTOTYPE input
	cdfdata		  ; LINT_PROTOTYPE output
;-------------------------------------------------------------------------------
; Affiche a l'cran des informations sur les variables du fichier.
;-------------------------------------------------------------------------------

	cdfdata = 1
	lint_unused = lire_cdf (fichier,'','',1,noms,fillval,nblignes,0,cdfdata=cdfdata,/infocdf)

END

