;-------------------------------------------------------------------------------
;
;	Fichier	: $RCSfile: libascii.pro,v $, v $Revision: 1.26 $
;
;	Date	: $Date: 2021/04/27 09:48:58 $
;
;	Auteur	: $Author: penou $
;
;	Version : %Z% version %I% de %M% du %G%
;
;-------------------------------------------------------------------------------


; -----------
; Utilisation
; -----------
;
; There is:
;
; - 4 basic functions
; 	* read_ascii_get_symbols()
; 	* read_ascii_analyse_symbols()
; 	* read_ascii_get_structure()
; 	* read_ascii_get_data()
;
; - read_cef() function that calls the 4 previous functions to read an ascii file
;
;
;	FILENAME
;
;	|
;	|
;	|				\	 --------------------------------------------------------------------------------
;	|--------------------------------	|read_ascii_get_symbols(noatt=noatt,nonb=nonb,remove_ext=remove_ext,quiet=quiet)|
;	|				/	 --------------------------------------------------------------------------------
;	|					 | 
;	|					\ /
;	|
;	|	FORMAT NBRECORDS NBLINES0	SYMBOLS			ATT NOVARY
;	|
;	|		|	   		 |
;	|		|	  		\ /
;	|		|
;	|		|			 -----------------------------
;	|		|			|read_ascii_analyse_symbols()|
;	|		|			 -----------------------------
;	|		|
;	|		|	   		 |
;	|		|	  		\ /
;	|		|
;	|		|			VARNAMES TYPES SIZES FILLVALS
;	|		|
;	|		|	   		 |
;	|		|	  		\ /
;	|		|
;	|		|			 ----------------------------------------------------------
;	|		|			|read_ascii_get_structure(readnames=readnames,quiet=quiet)|
;	|		|			 ----------------------------------------------------------
;	|		|
;	|		|	   		 |
;	|		|	  		\ /
;	|		|		
;	|		|			STRUCT INFO		NAMES
;	|		|
;	|		|			 |
;	|		|			\ /
;	|		|
;	|		|		\	 ------------------------------------------------------------------------------------------------------
;	 --------------------------------	|read_ascii_get_data(rec_start=rec_start,rec_count=rec_count,modify_float_fillval=modify_float_fillval,|
;					/	|		     nomalloc=nomalloc,quiet=quiet)					               |
;						 ------------------------------------------------------------------------------------------------------
;
;						 |
;						\ /
;
;						DATA
;
;
;
;
;
;
;
;						FILENAME
;
;						 |
;						\ /
;
;	 		 		 	 -------------------------------------------------------------------------------------------------------
;						|read_cef(readnames=readnames,rec_start=rec_start,rec_count=rec_count,noatt=noatt,nodata=nodata,nob=nonb|
;					 	|         remove_ext=remove_ext,quiet=quiet)				                                |
;						 -------------------------------------------------------------------------------------------------------
;
;						 |
;						\ /
;
;						ATT NOVARY VARNAMES NAMES DATA
;
;
;

;-------------------------------------------------------------------------------
FUNCTION s_to_mot1,			$
;-------------------------------------------------------------------------------
	ligne,				$	; LINT_PROTOTYPE input
	i,				$	; LINT_PROTOTYPE output
	l					; LINT_PROTOTYPE output
;-------------------------------------------------------------------------------
; Retourne la postion et le nombre de caractres du premier mot trouv dans ligne.
;-------------------------------------------------------------------------------

	IF is_fdl() THEN BEGIN
		code = CALL_EXTERNAL (libname('libascii'),'S_TO_MOT1_AUTO_GLUE', ligne,i,l, /AUTO_GLUE)
	END ELSE BEGIN
		code = CALL_EXTERNAL (libname('libascii'),'S_TO_MOT1',           ligne,i,l)
	END
	RETURN, code

END

;-------------------------------------------------------------------------------
FUNCTION s_to_mot1_mot2,		$
;-------------------------------------------------------------------------------
	ligne,				$	; LINT_PROTOTYPE input
	i1,				$	; LINT_PROTOTYPE output
	l1,				$	; LINT_PROTOTYPE output
	i2,				$	; LINT_PROTOTYPE output
	l2					; LINT_PROTOTYPE output
;-------------------------------------------------------------------------------
; Retourne la postion et le nombre de caractres des deux premiers mots trouvs dans ligne.
;-------------------------------------------------------------------------------

	IF is_fdl() THEN BEGIN
		code = CALL_EXTERNAL (libname('libascii'),'S_TO_MOT1_MOT2_AUTO_GLUE', ligne,i1,l1,i2,l2, /AUTO_GLUE)
	END ELSE BEGIN
		code = CALL_EXTERNAL (libname('libascii'),'S_TO_MOT1_MOT2',           ligne,i1,l1,i2,l2)
	END
	RETURN, code

END

;-------------------------------------------------------------------------------
FUNCTION s_to_mot1_egal_mot2,		$
;-------------------------------------------------------------------------------
	ligne,				$	; LINT_PROTOTYPE input
	i1,				$	; LINT_PROTOTYPE output
	l1,				$	; LINT_PROTOTYPE output
	i2,				$	; LINT_PROTOTYPE output
	l2					; LINT_PROTOTYPE output
;-------------------------------------------------------------------------------
; Retourne la postion et le nombre de caractres des deux premiers mots trouvs dans ligne correspondant  mot1 = mot2.
;-------------------------------------------------------------------------------

	IF is_fdl() THEN BEGIN
		code = CALL_EXTERNAL (libname('libascii'),'S_TO_MOT1_EGAL_MOT2_AUTO_GLUE', ligne,i1,l1,i2,l2,/AUTO_GLUE)
	END ELSE BEGIN
		code = CALL_EXTERNAL (libname('libascii'),'S_TO_MOT1_EGAL_MOT2',           ligne,i1,l1,i2,l2)
	END
	RETURN, code

END


;-------------------------------------------------------------------------------
FUNCTION read_ascii_get_symbols,	$
;-------------------------------------------------------------------------------
	FILENAME,			$	; LINT_PROTOTYPE input
	ATT,				$	; LINT_PROTOTYPE output
	NOVARY,				$	; LINT_PROTOTYPE output
	SYMBOLS,			$	; LINT_PROTOTYPE output
	FORMAT,				$	; LINT_PROTOTYPE output
	NBRECORDS,			$	; LINT_PROTOTYPE output
	NBLINES0,			$	; LINT_PROTOTYPE output
	noatt=noatt,			$	; LINT_PROTOTYPE input
	nonb=nonb,			$	; LINT_PROTOTYPE input
	remove_ext=remove_ext,		$	; LINT_PROTOTYPE input
	quiet=quiet				; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((1:ok 0:pb)) Lit un fichier CEF et retourne les symboles trouvs. Utilise la variable d'environnement CEFPATH.
;
; Input:
; ------
; - FILENAME:			filename to read
;
; Keywords:
; ---------
; - /noatt:			don't compute ATT and NOVARY
; - /nonb:			don't compute NBRECORDS
; - /remove_ext:		"CAA" or "THEMIS" to remove extension in variables names
; - /quiet: 			no messages
;
; Output:
; -------
; - ATT:			(undefined if /noatt) array of structure { v1:'', v2:'', v3:'', v4:''} with all attributes found in header
; - NOVARY:			(undefined if /noatt) structure with variables containing a non empty DATA field
; - SYMBOLS:			structure with variables information found in header
; - FORMAT:			format number for Date/Time found in first line of data (5 for CEF-2)
; - NBRECORDS:			number of records found in data
; - NBLINES0:			number of lines found in header
;
; Return code:
; ------------
;	1 = OK
;	0 = PB
;-------------------------------------------------------------------------------
;
; Ordre de SIZES et des DEPEND_1, DEPEND_2, DEPEND_3:
; ---------------------------------------------------
;
; 	Exemple du fichier "/CAA_B1/CAA/ARCH/20010405/CLUSTER3/C3_CP_CIS-CODIF_HS_H1_RC__20010405_V03.cef.gz"
;
;		NAME            STRING    '3d_ions__C3_CP_CIS-CODIF_HS_H1_RC'
;		SIZES           STRING    '8,16,31'					=> dans le fichier: 31 valeurs en premier
;		DEPEND_1        STRING    'theta__C3_CP_CIS-CODIF_HS_H1_RC'		=> 8 valeurs
;		DEPEND_2        STRING    'phi__C3_CP_CIS-CODIF_HS_H1_RC'		=> 16 valeurs
;		DEPEND_3        STRING    'energy_table__C3_CP_CIS-CODIF_HS_H1_RC'	=> 31 valeurs
;
; 	En IDL, il faudra dclarer la variable en [31,16,8]: c'est le travail de read_ascii_get_structure
;-------------------------------------------------------------------------------

	;COMMON duree1,duree
	;IF N_ELEMENTS(duree) EQ 0 THEN duree = 0.0
	;topduree = nbsec()

	ON_IOERROR, error

	IF N_ELEMENTS(quiet) 		EQ 0 THEN quiet 	= 0
	IF N_ELEMENTS(noatt) 		EQ 0 THEN noatt 	= 0
	IF N_ELEMENTS(remove_ext)	EQ 0 THEN remove_ext	= 'No'
	IF N_ELEMENTS(nonb) 		EQ 0 THEN nonb 		= 0

	IF ~quiet THEN BEGIN
		PRINT,''
		FOR i=0L,STRLEN(FILENAME)-1 DO PRINT, '-',FORMAT='(A,$)' & PRINT, ''
		PRINT, FILENAME
		FOR i=0L,STRLEN(FILENAME)-1 DO PRINT, '-',FORMAT='(A,$)' & PRINT, ''
	END

	NBLINES0 	= 0L
	NBRECORDS 	= 0L
	ligne 		= ''
	go 		= 0 ; 1 pour lire les donnes et non l'entete
	FORMAT 		= -1L
	is_cef2x	= 0
	is_mini		= 0
	IF noatt THEN BEGIN
		ATT = 0 & tmp = TEMPORARY(ATT) ; destroy ATT
	END ELSE BEGIN
		ATT = { v1:'', v2:'', v3:'', v4:'' }
	END
	NOVARY = 0 & tmp = TEMPORARY(NOVARY) ; destroy NOVARY
	symbol0 = { 	$
		name:		'',	$
		exception:	0,	$

		value_type:	'',	$
		var_type:	'',	$
		property:	'',	$
		sizes:		'',	$
		data:		'',	$
		delta_plus:	'',	$
		delta_minus:	'',	$
		fillval:	'',	$
		units:		'',	$
		fieldnam:	'',	$
		si_conversion:	'',	$
		depend_0:	'',	$
		depend_1:	'',	$
		depend_2:	'',	$
		depend_3:	'',	$
		depend_4:	'',	$
		labl_ptr_1:	'',	$
		labl_ptr_2:	''	$
	}
	SYMBOLS = [ symbol0 ]
	END_OF_RECORD_MARKER = "" ; par dfaut
	names = '' ; pour fichier csv


	fd = -1
	include = { level:0L, stack:REPLICATE(-1L,5) }
	OPENR, fd, FILENAME, /GET_LUN,compress=STRMID(FILENAME,STRLEN(FILENAME)-3,3) EQ '.gz'
	include.level = 0
	include.stack[include.level] = fd

	i1 = 0L
	l1 = 0L
	i2 = 0L
	l2 = 0L

	SEP = is_windows() ? ';' : ':'

	CEFPATH = STRSPLIT ( GETENV('CEFPATH'), SEP, /EXTRACT)

	REPEAT BEGIN

		fd = include.stack[include.level]

		REPEAT BEGIN

			READF, fd, ligne
			IF include.level EQ 0 THEN NBLINES0++

			code = s_to_mot1_mot2 (ligne,i1,l1,i2,l2)
			IF code EQ -1 THEN BEGIN
				CONTINUE
			END ELSE IF code EQ -2 THEN BEGIN
				MOT = ''
			END ELSE IF code EQ 0 THEN BEGIN
				MOT = STRMID(ligne,i1,l1)
			END

			IF code EQ 0 AND (MOT EQ 'EMIN' OR MOT EQ 'EMOY' OR MOT EQ 'EMAX' OR $
					  MOT EQ 'PMIN' OR MOT EQ 'PMOY' OR MOT EQ 'PMAX' OR $
					  MOT EQ 'TMIN' OR MOT EQ 'AMIN' OR MOT EQ 'TMOY' OR $
					  MOT EQ 'AMOY' OR MOT EQ 'TMAX' OR MOT EQ 'AMAX' OR $
					  MOT EQ 'MMIN' OR MOT EQ 'MMOY' OR MOT EQ 'MMAX') THEN BEGIN

				IF ~quiet THEN PRINT, '**** ' + MOT + ' ****'

				is_mini = 1

				symbol = symbol0
				symbol.name = MOT
				symbol.exception = 1
				VALEUR =  STRMID(ligne,i2,l2)
				s = STRSPLIT (VALEUR, /EXTRACT)
				nbs = N_ELEMENTS(s)
				FOR i=0L,nbs-1 DO symbol.DATA += (i EQ 0 ? '' : ',') + s[i]
				SYMBOLS = SYMBOLS[0].name EQ '' ? [symbol] : [SYMBOLS,symbol]

			END ELSE IF code EQ 0 AND (MOT EQ 'DT') THEN BEGIN

				IF ~quiet THEN PRINT, '**** ' + MOT + ' ****'

				symbol = symbol0
				symbol.name = MOT
				symbol.exception = 1
				symbol.DATA = STRMID(ligne,i2,l2)
				symbol.UNITS = '"s"'
				symbol.SI_CONVERSION = '1.0>s'
				SYMBOLS = SYMBOLS[0].name EQ '' ? [symbol] : [SYMBOLS,symbol]

			END ELSE BEGIN

				code = s_to_mot1_egal_mot2 (ligne,i1,l1,i2,l2)
				IF code EQ 0 THEN BEGIN	; on a trouv MOT = VALEUR

					MOT = STRMID(ligne,i1,l1)
					VALEUR = STRMID(ligne,i2,l2)

					IF STRUPCASE(MOT) EQ 'INCLUDE' THEN BEGIN
						trouve = 0
						FOR i=0L,N_ELEMENTS(CEFPATH)-1 DO BEGIN
							nom = CEFPATH[i]+'/'+VALEUR
							IF fichier_existe(nom) THEN BEGIN
								OPENR, fd, nom,  /GET_LUN
								include.level++
								include.stack[include.level] = fd
								IF ~quiet THEN PRINT,'++++ INCLUDE ' + nom + ' ++++'
								trouve = 1
								BREAK
							END
						END
						IF ~trouve AND ~quiet THEN PRINT,'WARNING: INCLUDE FILE "' + VALEUR + '" NOT FOUND'
						CONTINUE
					END

					IF STRUPCASE(MOT) EQ 'START_VARIABLE' THEN BEGIN ; START_VARIABLE = nom_de_variable

						var = VALEUR
						symbol = symbol0
						symbol.name = VALEUR

						IF ~quiet THEN PRINT, '**** ' + ligne + ' ****'
						has_att = 0
						REPEAT BEGIN	; boucle entre START_VARIABLE et END_VARIABLE

							READF, fd, ligne
							IF include.level EQ 0 THEN NBLINES0++
							code = s_to_mot1_egal_mot2 (ligne,i1,l1,i2,l2)
							IF code EQ 0 THEN BEGIN	; on a trouv MOT = VALEUR

								MOT = STRMID(ligne,i1,l1)
								IF STRUPCASE(MOT) EQ 'END_VARIABLE' THEN BREAK

								VALEUR = STRMID(ligne,i2,l2)

								WHILE STRMID(VALEUR,STRLEN(VALEUR)-1,1) EQ '\' DO BEGIN
									REPEAT BEGIN
										READF, fd, ligne
										IF include.level EQ 0 THEN NBLINES0++
										code = s_to_mot1 (ligne,i1,l1)
									END UNTIL code EQ 0
									VALEUR = STRMID (VALEUR,0,STRLEN(VALEUR)-1) + STRMID(ligne,i1,l1)
								END

								CASE STRUPCASE(MOT) OF
									'VALUE_TYPE':		symbol.value_type	= VALEUR
									'PROPERTY':		symbol.property		= VALEUR
									'SIZES':		symbol.sizes		= VALEUR
									'FILLVAL':		symbol.fillval		= VALEUR
									'DATA':			symbol.data		= VALEUR
									'DELTA_PLUS':		symbol.delta_plus	= VALEUR
									'DELTA_MINUS':		symbol.delta_minus	= VALEUR
									'UNITS':		symbol.units		= STRTRIM(VALEUR,2)

									'FIELDNAM':		BEGIN
													VALEUR = STRTRIM(VALEUR,2)
													symbol.fieldnam	= symbol.fieldnam EQ '' ? VALEUR : (symbol.fieldnam EQ VALEUR ? symbol.fieldnam : (symbol.fieldnam+' / '+VALEUR))
												END
									'CATDESC':		BEGIN
													VALEUR = STRTRIM(VALEUR,2)
													symbol.fieldnam	= symbol.fieldnam EQ '' ? VALEUR : (symbol.fieldnam EQ VALEUR ? symbol.fieldnam : (symbol.fieldnam+' / '+VALEUR))
												END

									'SI_CONVERSION':	symbol.si_conversion	= VALEUR
									'DEPEND_0':		symbol.depend_0		= VALEUR
									'DEPEND_1':		symbol.depend_1		= VALEUR
									'DEPEND_2':		symbol.depend_2		= VALEUR
									'DEPEND_3':		symbol.depend_3		= VALEUR
									'DEPEND_4':		symbol.depend_4		= VALEUR

									; on prend la chaine la plus courte (la moins dtaille)
									'LABEL_1':		BEGIN
													tmp = supprimer_caractere(VALEUR,'"')
													symbol.labl_ptr_1 = chaine_la_plus_courte(tmp,symbol.labl_ptr_1)
												END
									'REPRESENTATION_1':		BEGIN
													tmp = supprimer_caractere(VALEUR,'"')
													symbol.labl_ptr_1 = chaine_la_plus_courte(tmp,symbol.labl_ptr_1)
												END
									'LABL_PTR_1':		BEGIN
													tmp = supprimer_caractere(VALEUR,'"')
													symbol.labl_ptr_1 = chaine_la_plus_courte(tmp,symbol.labl_ptr_1)
												END

									; patch tenseur de pression
									'LABEL_2':		BEGIN
													tmp = supprimer_caractere(VALEUR,'"')
													symbol.labl_ptr_2 = chaine_la_plus_courte(tmp,symbol.labl_ptr_2)
												END
									'VAR_TYPE':		symbol.var_type		= VALEUR
									'PARAMETER_TYPE':	symbol.var_type		= VALEUR
									ELSE:
								END

								has_att = 1
								IF ~noatt THEN BEGIN
									tmp = { v1:'VARIABLE', v2:var, v3:MOT, v4:VALEUR } & ATT = ATT[0].v1 EQ '' ? [tmp] : [ATT,tmp]
								END

							END

						END UNTIL EOF (fd)

						; patch tenseur de pression
						IF symbol.depend_1 EQ '' AND symbol.depend_2 EQ '' AND symbol.labl_ptr_2 NE '' THEN BEGIN
							tmp1 = STRSPLIT(STRCOMPRESS(symbol.labl_ptr_1,/REMOVE_all),',',/EXTRACT)
							tmp2 = STRSPLIT(STRCOMPRESS(symbol.labl_ptr_2,/REMOVE_ALL),',',/EXTRACT)
							nb1 = N_ELEMENTS(tmp1)
							nb2 = N_ELEMENTS(tmp2)
							labl_ptr_1 = REPLICATE('',nb1*nb2)
							FOR i1=0L,N_ELEMENTS(tmp1)-1 DO BEGIN
								FOR i2=0L,N_ELEMENTS(tmp2)-1 DO BEGIN
									labl_ptr_1[i1*nb2+i2] = tmp1[i1]+tmp2[i2]
								END
							END
							tmp = STRJOIN (labl_ptr_1,',')
							PRINT,'*** PATCH LABEL_1 LABEL_2 ***'+' '+symbol.name+' => '+tmp
							symbol.labl_ptr_1 = tmp
							symbol.sizes = val_to_str(nb1*nb2)
						END
						IF symbol.name NE '' THEN SYMBOLS = SYMBOLS[0].name EQ '' ? [symbol] : [SYMBOLS,symbol]

						IF ~has_att THEN BEGIN
						
							IF ~noatt THEN BEGIN
								tmp = { v1:'VARIABLE', v2:var, v3:'', v4:'' } ; si la variable n'a pas d'attributs
								ATT = ATT[0].v1 EQ '' ? [tmp] : [ATT,tmp]
							END

						END

					END ELSE IF STRUPCASE(MOT) EQ 'START_META' THEN BEGIN ; START_META = nom_de_variable

						var = VALEUR

						IF ~quiet AND ~noatt THEN PRINT, '**** ' + ligne + ' ****'
						REPEAT BEGIN	; boucle entre START_META et END_META
							READF, fd, ligne
							IF include.level EQ 0 THEN NBLINES0++
							code = s_to_mot1_egal_mot2 (ligne,i1,l1,i2,l2)
							IF code EQ 0 THEN BEGIN	; on a trouv MOT = VALEUR

								MOT = STRMID(ligne,i1,l1)
								VALEUR = STRMID(ligne,i2,l2)

								IF STRUPCASE(MOT) EQ 'END_META' THEN BREAK

								IF ~noatt THEN BEGIN
									tmp = { v1:'META', v2: var, v3:MOT, v4:VALEUR } & ATT = ATT[0].v1 EQ '' ? [tmp] : [ATT,tmp]
								END

							END
						END UNTIL EOF (fd)

					END ELSE BEGIN

						IF STRUPCASE(MOT) EQ 'END_OF_RECORD_MARKER' THEN BEGIN
							END_OF_RECORD_MARKER = STRMID(VALEUR,STRLEN(VALEUR)-1,1)
						END

						IF STRUPCASE(MOT) EQ 'FILE_FORMAT_VERSION' THEN BEGIN
							IF STRLEN(VALEUR) GE 6 THEN BEGIN
								is_cef2x = STRMID(VALEUR,0,6) EQ 'CEF-2.'
							END
						END

						IF ~noatt THEN BEGIN
							tmp = { v1:'GLOBAL', v2:'', v3:MOT, v4:VALEUR } & ATT = ATT[0].v1 EQ '' ? [tmp] : [ATT,tmp]
						END

					END

				END ELSE BEGIN

					IF is_cef2x THEN BEGIN

						tmp = STRSPLIT (ligne, /EXTRACT)

						IF STRUPCASE(tmp[0]) EQ 'BINARY' THEN BEGIN
							FORMAT = -1L
							NBRECORDS = LONG(tmp[1])
							go = 1
							IF include.level EQ 0 THEN NBLINES0++
							BREAK
						END

						FORMAT = 5L
						go = 1
						BREAK


					END ELSE BEGIN

						tmp = get_date_format (ligne)
						IF tmp NE -1 THEN BEGIN
							FORMAT = tmp
							go = 1
							BREAK
						END ELSE BEGIN
							IF STRUPCASE(get_extension(FILENAME)) EQ 'CSV' THEN names = STRSPLIT (ligne, ',', /EXTRACT)
						END

					END

				END

			END

		END UNTIL EOF(fd)
		FREE_LUN, fd

		include.level--

	END UNTIL include.level EQ -1

	IF is_cef2x THEN BEGIN    ; Format CEF-2.x

		corriger_symbols, symbol0, SYMBOLS, modif

	END ELSE IF is_mini THEN BEGIN      ; Format MINI

		; Soit toutes les variables sont dfinies, soit aucunes
		ind = WHERE (SYMBOLS.NAME NE '' AND SYMBOLS.DATA EQ '',nb)
		IF nb EQ 0 THEN BEGIN
			symbol = symbol0
			symbol.name = 'Epoch'
			symbol.VALUE_TYPE = 'ISO_TIME'
			SYMBOLS = SYMBOLS[0].name EQ '' ? [symbol] : [SYMBOLS,symbol]
			
			nb = 1
			s = STRSPLIT (ligne, ' '+STRING(BYTE(9))+',', /EXTRACT) ; sparateur ' ' ou '\t' ou ','
			IF FORMAT EQ 3 OR FORMAT EQ 5 OR FORMAT EQ 7 THEN BEGIN
				nbauto = N_ELEMENTS(s)
			END ELSE IF FORMAT EQ 4 OR FORMAT EQ 1 THEN BEGIN
				nbauto = N_ELEMENTS(s)-5
			END ELSE IF FORMAT EQ 6 THEN BEGIN
				nbauto = N_ELEMENTS(s) + 1
			END ELSE IF FORMAT EQ 8 THEN BEGIN
				nbauto = N_ELEMENTS(s) - 4
			END
			IF nb NE nbauto THEN BEGIN ; correction
				symbol = symbol0
				symbol.name = 'data'
				indE = WHERE (STRUPCASE(SYMBOLS.NAME) EQ 'EMIN') & IF indE[0] NE -1 THEN nbE = N_ELEMENTS(string_to_data(SYMBOLS[indE[0]].DATA))
				indT = WHERE (STRUPCASE(SYMBOLS.NAME) EQ 'TMIN') & IF indT[0] NE -1 THEN nbT = N_ELEMENTS(string_to_data(SYMBOLS[indT[0]].DATA))
				indP = WHERE (STRUPCASE(SYMBOLS.NAME) EQ 'PMIN') & IF indP[0] NE -1 THEN nbP = N_ELEMENTS(string_to_data(SYMBOLS[indP[0]].DATA))
				indA = WHERE (STRUPCASE(SYMBOLS.NAME) EQ 'AMIN') & IF indA[0] NE -1 THEN nbA = N_ELEMENTS(string_to_data(SYMBOLS[indA[0]].DATA))
				indM = WHERE (STRUPCASE(SYMBOLS.NAME) EQ 'MMIN') & IF indM[0] NE -1 THEN nbM = N_ELEMENTS(string_to_data(SYMBOLS[indM[0]].DATA))

				IF (indE[0] NE -1) AND (indT[0] NE -1) AND (indP[0] NE -1) AND (indA[0] EQ -1) THEN BEGIN

					IF indM[0] NE -1 THEN BEGIN

						; fichier ascii (time,energy,theta,phi,mass) sera vu comme (time,energy,theta,phi,mass) par cl
						symbol.SIZES = val_to_str(nbM)+','+val_to_str(nbT)+','+val_to_str(nbP)+','+val_to_str(nbE)
						symbol.DEPEND_0 = 'DATE'
						symbol.DEPEND_1 = 'MMIN'
						symbol.DEPEND_2 = 'TMIN'
						symbol.DEPEND_3 = 'PMIN'
						symbol.DEPEND_4 = 'EMIN'

					END ELSE BEGIN

						; fichier ascii (time,energy,theta,phi) sera vu comme (time,energy,theta,phi) par cl
						symbol.SIZES = val_to_str(nbT)+','+val_to_str(nbP)+','+val_to_str(nbE)
						symbol.DEPEND_0 = 'DATE'
						symbol.DEPEND_1 = 'TMIN'
						symbol.DEPEND_2 = 'PMIN'
						symbol.DEPEND_3 = 'EMIN'

					END

				END ELSE IF (indE[0] NE -1) AND (indT[0] EQ -1) AND (indP[0] EQ -1) AND (indA[0] NE -1) THEN BEGIN

					; fichier ascii (time,energy,pitchangle) sera vu comme (time,energy,theta) par cl
					symbol.SIZES = val_to_str(nbA)+','+val_to_str(nbE)
					symbol.DEPEND_0 = 'DATE'
					symbol.DEPEND_1 = 'AMIN'
					symbol.DEPEND_2 = 'EMIN'

				END ELSE IF (indE[0] NE -1) AND (indT[0] EQ -1) AND (indP[0] EQ -1) AND (indA[0] EQ -1) THEN BEGIN

					; fichier ascii (time,energy) sera vu comme (time,energy) par cl
					symbol.SIZES = val_to_str(nbE)
					symbol.DEPEND_0 = 'DATE'
					symbol.DEPEND_1 = 'EMIN'

				END ELSE IF (indE[0] EQ -1) AND (indT[0] NE -1) AND (indP[0] EQ -1) AND (indA[0] EQ -1) THEN BEGIN

					; fichier ascii (time,theta) sera vu comme (time,energy) par cl
					symbol.SIZES = val_to_str(nbT)
					symbol.DEPEND_0 = 'DATE'
					symbol.DEPEND_1 = 'TMIN'

				END ELSE IF (indE[0] EQ -1) AND (indT[0] EQ -1) AND (indP[0] NE -1) AND (indA[0] EQ -1) THEN BEGIN

					; fichier ascii (time,phi) sera vu comme (time,energy) par cl
					symbol.SIZES = val_to_str(nbP)
					symbol.DEPEND_0 = 'DATE'
					symbol.DEPEND_1 = 'PMIN'

				END ELSE IF (indE[0] EQ -1) AND (indT[0] EQ -1) AND (indP[0] EQ -1) AND (indA[0] NE -1) AND (indM[0] EQ -1) THEN BEGIN

					; fichier ascii (time,pitchangle) sera vu comme (time,energy) par cl
					symbol.SIZES = val_to_str(nbA)
					symbol.DEPEND_0 = 'DATE'
					symbol.DEPEND_1 = 'AMIN'

				END ELSE IF (indE[0] EQ -1) AND (indT[0] EQ -1) AND (indP[0] EQ -1) AND (indA[0] EQ -1) AND (indM[0] NE -1) THEN BEGIN

					; fichier ascii (time,mass) sera vu comme (time,energy) par cl
					symbol.SIZES = val_to_str(nbM)
					symbol.DEPEND_0 = 'DATE'
					symbol.DEPEND_1 = 'MMIN'
				END
				SYMBOLS = SYMBOLS[0].name EQ '' ? [symbol] : [SYMBOLS,symbol]
			END
		END
		ind = WHERE (SYMBOLS.NAME NE '' AND SYMBOLS.DATA EQ '')
		IF ind[0] NE -1 THEN BEGIN
			IF SYMBOLS[ind[0]].name NE '' THEN SYMBOLS[ind[0]].VALUE_TYPE = 'ISO_TIME'
		END
		ind = WHERE (SYMBOLS.VALUE_TYPE EQ '')
		IF ind[0] NE -1 THEN BEGIN
			SYMBOLS[ind].VALUE_TYPE = 'FLOAT'
		END

	END ELSE IF go THEN BEGIN  ; format ascii simple ou CEF-1.0

		IF FORMAT EQ 6 THEN BEGIN	; on rajoute le 1er champ 'Epoch'
			symbol = symbol0
			symbol.name = 'Epoch'
			SYMBOLS = SYMBOLS[0].name EQ '' ? [symbol] : [symbol,SYMBOLS]
		END
		nb = SYMBOLS[0].name EQ '' ? 0 : N_ELEMENTS(SYMBOLS)
		s = STRSPLIT (ligne, ' '+STRING(BYTE(9))+',', /EXTRACT) ; sparateur ' ' ou '\t' ou ','
		IF FORMAT EQ 3 OR FORMAT EQ 5 OR FORMAT EQ 7 THEN BEGIN
			nbauto = N_ELEMENTS(s)
		END ELSE IF FORMAT EQ 4 OR FORMAT EQ 1 THEN BEGIN
			nbauto = N_ELEMENTS(s)-5
		END ELSE IF FORMAT EQ 6 THEN BEGIN
			nbauto = N_ELEMENTS(s) + 1
		END ELSE IF FORMAT EQ 8 THEN BEGIN
			nbauto = N_ELEMENTS(s) - 4
		END ELSE IF FORMAT EQ 10 THEN BEGIN
			nbauto = N_ELEMENTS(s) - 1
		END

		IF nb NE nbauto THEN BEGIN ; correction

			; Methode lente avec boucle
			;FOR i=nb,nbauto-1 DO BEGIN
			;	symbol = symbol0
			;	symbol.name = 'Field_'+val_to_str(i)
			;	SYMBOLS = SYMBOLS[0].name EQ '' ? [symbol] : [SYMBOLS,symbol]
			;END

			; Methode plus rapide sans boucle
			symbol = REPLICATE (symbol0, nbauto-nb)
			symbol.name = 'Field_'+val_to_str(nb+LINDGEN(nbauto-nb))
			SYMBOLS = SYMBOLS[0].name EQ '' ? [symbol] : [SYMBOLS,symbol]
			
		END
		IF SYMBOLS[0].name NE '' THEN SYMBOLS[0].NAME = 'date'
		IF SYMBOLS[0].name NE '' THEN SYMBOLS[0].VALUE_TYPE = 'ISO_TIME'

		ind = WHERE (STRUPCASE(SYMBOLS.name) EQ 'COMMENT') & IF ind[0] NE -1 THEN SYMBOLS[ind[0]].VALUE_TYPE = 'STRING'

		ind = WHERE (SYMBOLS.VALUE_TYPE EQ '')
		IF ind[0] NE -1 THEN BEGIN
			SYMBOLS[ind].VALUE_TYPE = 'FLOAT'
		END

		IF STRUPCASE(get_extension(FILENAME)) EQ 'CSV' THEN BEGIN
			IF N_ELEMENTS(names) EQ N_ELEMENTS(SYMBOLS) THEN SYMBOLS.name = names
			IF N_ELEMENTS(s) EQ N_ELEMENTS(SYMBOLS) THEN BEGIN
				FOR i=0L,N_ELEMENTS(s)-1 DO BEGIN
					IF STRMID(s[i],0,1) EQ '"' THEN BEGIN
						SYMBOLS[i].VALUE_TYPE = 'CHAR'
					END
				END
			END
		END

	END ELSE IF ~go THEN BEGIN

		; no data
		;IF SYMBOLS[0].name NE '' THEN SYMBOLS[0].VALUE_TYPE = 'ISO_TIME'

	END

	IF go THEN BEGIN
		NBLINES0--
	END

	IF ~noatt THEN BEGIN
		ind = WHERE (ATT.v1 EQ 'VARIABLE' AND ATT.v3 EQ 'DATA' AND ATT.v4 NE '')
		IF ind[0] NE -1 THEN BEGIN
			FOR i=0L,N_ELEMENTS(ind)-1 DO BEGIN
				ajouter_champ, ATT[ind[i]].v2, string_to_data(ATT[ind[i]].v4), nom, NOVARY, taille, FIRST=i EQ 0
			END
		END
	END


	; Remplacer SIZES='' par SIZES='1' ; arrive dans les fichiers PAD d'Alain
	ind = WHERE (SYMBOLS.SIZES EQ '')
	IF ind[0] NE -1 THEN SYMBOLS[ind].SIZES = '1'


	;duree = duree+nbsec()-topduree
	;PRINT,'duree1 = ' + val_to_str_2decimales(duree)+' sec'

	IF NBRECORDS EQ 0 THEN BEGIN
		NBRECORDS = 0L
		nbspecial = REPLICATE (0L, 256)
		nbcomma = 0L
		IF ~nonb THEN BEGIN

			usecomma = is_cef2x ? (END_OF_RECORD_MARKER EQ '' ? 1L : 0L) : 0L
			ind = WHERE (STRUPCASE(SYMBOLS.VALUE_TYPE) EQ 'CHAR' AND SYMBOLS.DATA EQ '')
			usechar = ind[0] EQ -1 ? 0L : 1L

			top1 = nbsec()
			wc_l, FILENAME, NBLINES0, usecomma, usechar, NBRECORDS, nbspecial, nbcomma
			top2 = nbsec()
			IF quiet EQ 0 THEN PRINT,'wc_l ' + FILENAME + ' in ' + val_to_str_2decimales(top2-top1)+' sec'

			IF is_cef2x THEN BEGIN
				IF END_OF_RECORD_MARKER NE '' THEN BEGIN
					NBRECORDS = nbspecial[BYTE(END_OF_RECORD_MARKER)]
					NBRECORDS = NBRECORDS[0]
					;PRINT,'uses nbspecial'
				END ELSE BEGIN
					; nombre de champs dans le fichier
					nbchamps = 0L
					FOR i=0L,N_ELEMENTS(SYMBOLS)-1 DO BEGIN
						IF SYMBOLS[i].DATA EQ '' THEN BEGIN
							nbchamps += LONG(PRODUCT(LONG(SYMBOLS[i].sizes)))
						END
					END
					IF nbchamps GE 2 THEN BEGIN
						NBRECORDS = nbcomma / (nbchamps-1)
						;PRINT,'uses nbcomma'
					END ELSE BEGIN
						NBRECORDS -= NBLINES0
						;PRINT,'uses nbrecords'
					END
				END
			END ELSE BEGIN
				NBRECORDS -= NBLINES0
			END
		END
	END

	SYMBOLS.name            = remove_extension(remove_ext,SYMBOLS.name)
	SYMBOLS.data            = remove_extension(remove_ext,SYMBOLS.data)
	SYMBOLS.delta_plus      = remove_extension(remove_ext,SYMBOLS.delta_plus)
	SYMBOLS.delta_minus     = remove_extension(remove_ext,SYMBOLS.delta_minus)
	SYMBOLS.depend_0        = remove_extension(remove_ext,SYMBOLS.depend_0)
	SYMBOLS.depend_1        = remove_extension(remove_ext,SYMBOLS.depend_1)
	SYMBOLS.depend_2        = remove_extension(remove_ext,SYMBOLS.depend_2)
	SYMBOLS.depend_3        = remove_extension(remove_ext,SYMBOLS.depend_3)
	SYMBOLS.depend_4        = remove_extension(remove_ext,SYMBOLS.depend_4)
	SYMBOLS.labl_ptr_1      = remove_extension(remove_ext,SYMBOLS.labl_ptr_1)

	IF N_ELEMENTS(SYMBOLS) EQ 1 AND SYMBOLS[0].name EQ '' THEN BEGIN
		RETURN, 0 ; aucun symboles (arrive avec CEF-2.0 et fichiers include introuvables)
	END

	; PATCH DOUBLE -> FLOAT pour ROSINA
	ind = WHERE (SYMBOLS.value_type EQ 'DOUBLE')
	IF ind[0] NE -1 THEN SYMBOLS[ind].value_type = 'FLOAT'

	RETURN, 1

error:

	FOR i=0L,N_ELEMENTS(include.stack)-1 DO BEGIN
		IF include.stack[i] NE -1 THEN FREE_LUN, include.stack[i]
	END
	RETURN, 0

END


;-------------------------------------------------------------------------------
FUNCTION read_ascii_analyse_symbols,	$
;-------------------------------------------------------------------------------
	SYMBOLS,			$	; LINT_PROTOTYPE input
	REALVARNAMES,			$	; LINT_PROTOTYPE output
	VARNAMES,			$	; LINT_PROTOTYPE output
	TYPES,				$	; LINT_PROTOTYPE output
	SIZES,				$	; LINT_PROTOTYPE output
	FILLVALS,			$	; LINT_PROTOTYPE output
	UNITS,				$	; LINT_PROTOTYPE output
	NAME_TIME,			$	; LINT_PROTOTYPE output
	NAME_THETA,			$	; LINT_PROTOTYPE output
	UNIT_THETA,			$	; LINT_PROTOTYPE output
	NAME_PHI,			$	; LINT_PROTOTYPE output
	UNIT_PHI,			$	; LINT_PROTOTYPE output
	NAME_ENERGY,			$	; LINT_PROTOTYPE output
	UNIT_ENERGY,			$	; LINT_PROTOTYPE output
	remove_ext=remove_ext			; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((1:ok 0:pb)) A partir des symboles, supprime les extensions si ncessaire, et retourne des tables.
;
; Input:
; ------
; - SYMBOLS:			output from read_ascii_get_symbols()
;
; Keywords:
; ---------
; - /remove_ext:		"CAA" or "THEMIS" to remove extension in variables names
;
; Output:
; -------
; - REALVARNAMES:
; - VARNAMES:			names of variables with empty DATA field
; - TYPES:			types of variables with empty DATA field ('ISO_TIME', 'FLOAT, ...)
; - SIZES:			sizes of variables with empty DATA field
; - FILLVALS:			fillval of variables with empty DATA field
; - UNITS:			units of variables with empty DATA field
; - NAME_TIME:
; - NAME_THETA:
; - UNIT_THETA:
; - NAME_PHI:
; - UNIT_PHI:
; - NAME_ENERGY:
; - UNIT_ENERGY:
;
; Return code:
; ------------
;	1 = OK
;	0 = PB
;				'ERROR: NO VARIABLES TO READ'
;-------------------------------------------------------------------------------

	IF N_ELEMENTS(remove_ext) EQ 0 THEN remove_ext = 'No'

	; Variables sans champ DATA
	ind = WHERE (SYMBOLS.data EQ '')
	IF ind[0] EQ -1 THEN BEGIN
		PRINT, 'ERROR: NO VARIABLES TO READ'
		RETURN, 0
	END

	REALVARNAMES	= SYMBOLS[ind].name

	SYMBOLS.name		= remove_extension(remove_ext,SYMBOLS.name)
	SYMBOLS.data		= remove_extension(remove_ext,SYMBOLS.data)
	SYMBOLS.delta_plus	= remove_extension(remove_ext,SYMBOLS.delta_plus)
	SYMBOLS.delta_minus	= remove_extension(remove_ext,SYMBOLS.delta_minus)
	SYMBOLS.depend_0	= remove_extension(remove_ext,SYMBOLS.depend_0)
	SYMBOLS.depend_1	= remove_extension(remove_ext,SYMBOLS.depend_1)
	SYMBOLS.depend_2	= remove_extension(remove_ext,SYMBOLS.depend_2)
	SYMBOLS.depend_3	= remove_extension(remove_ext,SYMBOLS.depend_3)
	SYMBOLS.depend_4	= remove_extension(remove_ext,SYMBOLS.depend_4)

	VARNAMES	= SYMBOLS[ind].name
	SIZES		= SYMBOLS[ind].sizes
	TYPES		= SYMBOLS[ind].value_type
	FILLVALS	= SYMBOLS[ind].fillval
	UNITS		= SYMBOLS[ind].units

	NAME_TIME	= REPLICATE('',N_ELEMENTS(SYMBOLS[ind].depend_0))
	NAME_THETA	= REPLICATE('',N_ELEMENTS(SYMBOLS[ind].depend_1))
	UNIT_THETA	= REPLICATE('',N_ELEMENTS(SYMBOLS[ind].depend_1))
	NAME_PHI	= REPLICATE('',N_ELEMENTS(SYMBOLS[ind].depend_2))
	UNIT_PHI	= REPLICATE('',N_ELEMENTS(SYMBOLS[ind].depend_2))
	NAME_ENERGY	= REPLICATE('',N_ELEMENTS(SYMBOLS[ind].depend_3))
	UNIT_ENERGY	= REPLICATE('',N_ELEMENTS(SYMBOLS[ind].depend_3))

	RETURN, 1

END


;-------------------------------------------------------------------------------
FUNCTION read_ascii_get_structure,	$
;-------------------------------------------------------------------------------
	VARNAMES,			$	; LINT_PROTOTYPE input
	REALVARNAMES,			$	; LINT_PROTOTYPE input
	TYPES,				$	; LINT_PROTOTYPE input
	SIZES,				$	; LINT_PROTOTYPE input
	FILLVALS,			$	; LINT_PROTOTYPE input
	NAMES,				$	; LINT_PROTOTYPE output	
	STRUCT,				$	; LINT_PROTOTYPE output
	INFO,				$	; LINT_PROTOTYPE output
	readnames=readnames, 		$	; LINT_PROTOTYPE input
	quiet=quiet, 			$	; LINT_PROTOTYPE input
	trans=trans				; LINT_PROTOTYPE output
;-------------------------------------------------------------------------------
; ((1:ok 0:pb)) A partir des tables, retourne la structure d'accueil des donnes et des infos utilises pour la lecture en C.
;
; Input:
; ------
; - VARNAMES:			output of read_ascii_analyse_symbols()
; - REALVARNAMES:		output of read_ascii_analyse_symbols()
; - TYPES:			output of read_ascii_analyse_symbols()
; - SIZES:			output of read_ascii_analyse_symbols()
; - FILLVALS:			output of read_ascii_analyse_symbols()
; - UNITS:			output of read_ascii_analyse_symbols()
;
; Keywords:
; ---------
; - readnames:			list of variables to read (valid names in VARNAMES)
; - date_name:
; - data_name:
; - /quiet:			no messages
;
; Output:
; -------
; - NAMES:			names of fields in the structure
; - STRUCT:			IDL structure for one record
; - INFO:			structure used to read data
;
; Return code:
; ------------
;	1 = OK
;	0 = PB
;				'ERROR: NO VALID READNAMES FOUND'
;				'ERROR: MORE THAN ONE DATE VARIABLE'
;				'ERROR: MORE THAN ONE STRING VARIABLE'
;				'ERROR: MORE THAN ONE DATA VARIABLE'
;				'ERROR: TOO MANY DIMENSIONS '+'('+val_to_str(nbdims)+')'+'FOR VARIABLE ' + VARNAMES[ind2[i]]
;				'ERROR: TOO MANY DIMENSIONS '+'('+val_to_str(nbdims)+')'+'FOR VARIABLE ' + VARNAMES[ind3[i]]
;				'ERROR: TOO MANY DIMENSIONS '+'('+val_to_str(nbdims)+')'+'FOR VARIABLE ' + VARNAMES[ind4[i]]
;				'ERROR: TOO MANY DIMENSIONS '+'('+val_to_str(nbdims)+')'+'FOR VARIABLE ' + VARNAMES[ind6[i]]
;				'ERROR: TOO MANY DIMENSIONS '+'('+val_to_str(nbdims)+')'+'FOR VARIABLE ' + VARNAMES[ind7[0]]
;-------------------------------------------------------------------------------

	;COMMON duree3,duree
	;IF N_ELEMENTS(duree) EQ 0 THEN duree = 0.0


	DEBUG = 0

	IF N_ELEMENTS(quiet) EQ 0 THEN quiet = 0

	IF DEBUG THEN topduree = nbsec()

	nb = N_ELEMENTS(VARNAMES)

	IF N_ELEMENTS(readnames) NE 0 THEN BEGIN
		alire = REPLICATE (0, nb)
		STRUPCASE_VARNAMES = STRUPCASE(VARNAMES)
		FOR i=0L,N_ELEMENTS(readnames)-1 DO BEGIN
			ind = WHERE (STRUPCASE_VARNAMES EQ STRUPCASE(readnames[i]))
			IF ind[0] NE -1 THEN BEGIN
				; il peut y avoir plusieurs correspondances. Ex:
				; dans "cdaweb.gsfc.nasa.gov/istp_public/data/timed/see/l3a/2003/04/timed_l3a_see_20030401_v01.cdf"
				; il y a "Epoch" et "DATE", qui s'appelent tous les 2 date
				alire[ind] = 1
			END ELSE BEGIN
				PRINT, 'WARNING: UNKNOW VARIABLE "' + readnames[i] + '"'
			END
		END
	END ELSE BEGIN
		alire = REPLICATE (1, nb)
	END
	ind = WHERE (alire EQ 1, nbalire)
	IF ind[0] EQ -1 THEN BEGIN
		PRINT, 'ERROR: NO VALID READNAMES FOUND'
		RETURN, 0
	END

	dims = REPLICATE (0L, nb)
	FOR i=0L,N_ELEMENTS(VARNAMES)-1 DO BEGIN
		dims[i] = PRODUCT(LONG(string_to_data(SIZES[i])))
	END

	; ---------------------------------------------------------
	; CREATION DE LA STRUCTURE D'ACCUEIL DES DONNEES EN MEMOIRE
	; ---------------------------------------------------------

	; Ordre des champs dans cette structure afin de pouvoir y crire facilement en C:
	;
	; 	- (1)      le ou les champs avec STRUPCASE(TYPES)=='DATE' (par dfinition STRUPCASE(TYPES)='ISO_TIME' et SIZES=1)
	;	- (2) puis le ou les champs avec STRUPCASE(TYPES)=='ISO_TIME' sauf (1)
	;	- (3) puis le ou les champs avec STRUPCASE(TYPES)=='ISO_TIME_RANGE'
	;	- (4) puis le ou les champs avec STRUPCASE(TYPES)=='CHAR'
	;	- (5) puis le        champ  avec STRUPCASE(TYPES)=='STRING'
	;	- (6) puis le ou les champs avec STRUPCASE(TYPES)!='DATA' sauf (1), (2), (3), (4), (5)
	;	- (7) puis le        champ  avec STRUPCASE(TYPES)=='DATA'

	ind1 = WHERE (	STRUPCASE(TYPES) EQ 'DATE', 				nb1)

	ind2 = WHERE (	STRUPCASE(TYPES) EQ 'ISO_TIME'		AND $
			STRUPCASE(TYPES) NE 'DATE', 				nb2)

	ind3 = WHERE (	STRUPCASE(TYPES) EQ 'ISO_TIME_RANGE', 			nb3)

	ind4 = WHERE (	STRUPCASE(TYPES) EQ 'CHAR', 				nb4)

	ind5 = WHERE (	STRUPCASE(TYPES) EQ 'STRING', 				nb5)

	ind6 = WHERE (	STRUPCASE(TYPES) NE 'DATE'		AND $
			STRUPCASE(TYPES) NE 'ISO_TIME'		AND $
			STRUPCASE(TYPES) NE 'ISO_TIME_RANGE'	AND $
			STRUPCASE(TYPES) NE 'DATA'		AND $
			STRUPCASE(TYPES) NE 'STRING'		AND $
			STRUPCASE(TYPES) NE 'CHAR', 				nb6)

	ind7 = WHERE (	STRUPCASE(TYPES) EQ 'DATA', 				nb7)

	IF nb1 GE 2 THEN BEGIN
		PRINT, 'ERROR: MORE THAN ONE DATE VARIABLE'
		RETURN, 0
	END

	IF nb5 GE 2 THEN BEGIN
		PRINT, 'ERROR: MORE THAN ONE STRING VARIABLE'
		RETURN, 0
	END

	IF nb7 GE 2 THEN BEGIN
		PRINT, 'ERROR: MORE THAN ONE DATA VARIABLE'
		RETURN, 0
	END

	NAMES = ''
	nbchamps = 0L
	first = 1
	trans = -1

	; (1) 'DATE'
	IF nb1 NE 0 THEN BEGIN
		IF alire[ind1[0]] THEN BEGIN
			ajouter_champ, VARNAMES[ind1[0]], 0d, NAMES, STRUCT, taille, first=first
			trans = trans[0] EQ -1 ? ind1[0] : [trans,ind1[0]]
			first = 0
		END
		nbchamps++
	END

	; (2) 'ISO_TIME'
	IF nb2 NE 0 THEN BEGIN
		FOR i=0L,nb2-1 DO BEGIN
			IF alire[ind2[i]] THEN BEGIN
				val = 0d
				tmp = LONG(string_to_data(SIZES[ind2[i]]))
				nbdims = N_ELEMENTS(tmp)
				CASE nbdims OF
					1:	IF tmp[0] NE 1 THEN	val = REPLICATE (val, tmp[0])
					2:				val = REPLICATE (val, tmp[1],tmp[0])
					3:				val = REPLICATE (val, tmp[2],tmp[1],tmp[0])
					4:				val = REPLICATE (val, tmp[3],tmp[2],tmp[1],tmp[0])
					5:				val = REPLICATE (val, tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
					6:				val = REPLICATE (val, tmp[5],tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
					7:				val = REPLICATE (val, tmp[6],tmp[5],tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
					8:				val = REPLICATE (val, tmp[7],tmp[6],tmp[5],tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
					ELSE:	BEGIN
							PRINT, 'ERROR: TOO MANY DIMENSIONS '+'('+val_to_str(nbdims)+')'+'FOR VARIABLE ' + VARNAMES[ind2[i]]
							RETURN, 0
						END
				END
				ajouter_champ, VARNAMES[ind2[i]], val, NAMES, STRUCT, taille, first=first
				trans = trans[0] EQ -1 ? ind2[i] : [trans,ind2[i]]
				first = 0
			END
			nbchamps += dims[ind2[i]]
		END
	END

	; (3) 'ISO_TIME_RANGE'
	IF nb3 NE 0 THEN BEGIN
		FOR i=0L,nb3-1 DO BEGIN
			IF alire[ind3[i]] THEN BEGIN
				val = 0d
				tmp = LONG(string_to_data(SIZES[ind3[i]]))
				nbdims = N_ELEMENTS(tmp)
				CASE nbdims OF
					1:	val = REPLICATE (val, 2, tmp[0])
					2:	val = REPLICATE (val, 2, tmp[1],tmp[0])
					3:	val = REPLICATE (val, 2, tmp[2],tmp[1],tmp[0])
					4:	val = REPLICATE (val, 2, tmp[3],tmp[2],tmp[1],tmp[0])
					5:	val = REPLICATE (val, 2, tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
					6:	val = REPLICATE (val, 2, tmp[5],tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
					7:	val = REPLICATE (val, 2, tmp[6],tmp[5],tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
					ELSE:	BEGIN
							PRINT, 'ERROR: TOO MANY DIMENSIONS '+'('+val_to_str(nbdims)+')'+'FOR VARIABLE ' + VARNAMES[ind3[i]]
							RETURN, 0
						END
				END
				ajouter_champ, VARNAMES[ind3[i]], val, NAMES, STRUCT, taille, first=first
				trans = trans[0] EQ -1 ? ind3[i] : [trans,ind3[i]]
				first = 0
			END
			nbchamps += dims[ind3[i]]
		END
	END

	; (4) 'CHAR'
	IF nb4 NE 0 THEN BEGIN
		FOR i=0L,nb4-1 DO BEGIN
			IF alire[ind4[i]] THEN BEGIN
				val = ''
				tmp = LONG(string_to_data(SIZES[ind4[i]]))
				nbdims = N_ELEMENTS(tmp)
				CASE nbdims OF
					1:	IF tmp[0] NE 1 THEN	val = REPLICATE (val, tmp[0])
					2:				val = REPLICATE (val, tmp[1],tmp[0])
					3:				val = REPLICATE (val, tmp[2],tmp[1],tmp[0])
					4:				val = REPLICATE (val, tmp[3],tmp[2],tmp[1],tmp[0])
					5:				val = REPLICATE (val, tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
					6:				val = REPLICATE (val, tmp[5],tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
					7:				val = REPLICATE (val, tmp[6],tmp[5],tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
					8:				val = REPLICATE (val, tmp[7],tmp[6],tmp[5],tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
					ELSE:	BEGIN
							PRINT, 'ERROR: TOO MANY DIMENSIONS '+'('+val_to_str(nbdims)+')'+'FOR VARIABLE ' + VARNAMES[ind4[i]]
							RETURN, 0
						END
				END
				ajouter_champ, VARNAMES[ind4[i]], val, NAMES, STRUCT, taille, first=first
				trans = trans[0] EQ -1 ? ind4[i] : [trans,ind4[i]]
				first = 0
			END
			nbchamps += dims[ind4[i]]
		END
	END

	; (5) 'STRING'
	IF nb5 NE 0 THEN BEGIN
		IF alire[ind5[0]] THEN BEGIN
			ajouter_champ, VARNAMES[ind5[0]], '', NAMES, STRUCT, taille, first=first
			trans = trans[0] EQ -1 ? ind5[0] : [trans,ind5[0]]
			first = 0
		END
		nbchamps++
	END


	; (6)
	IF nb6 NE 0 THEN BEGIN
		FOR i=0L,nb6-1 DO BEGIN
			IF alire[ind6[i]] THEN BEGIN
				val = 0.0
				IF STRUPCASE(TYPES[ind6[i]]) EQ 'INT' THEN val = 0L
				IF STRUPCASE(TYPES[ind6[i]]) EQ 'DOUBLE' THEN val = 0d
				tmp = LONG(string_to_data(SIZES[ind6[i]]))
				nbdims = N_ELEMENTS(tmp)
				CASE nbdims OF
					1:	IF tmp[0] NE 1 THEN	val = REPLICATE (val, tmp[0])
					2:				val = REPLICATE (val, tmp[1],tmp[0])
					3:				val = REPLICATE (val, tmp[2],tmp[1],tmp[0])
					4:				val = REPLICATE (val, tmp[3],tmp[2],tmp[1],tmp[0])
					5:				val = REPLICATE (val, tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
					6:				val = REPLICATE (val, tmp[5],tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
					7:				val = REPLICATE (val, tmp[6],tmp[5],tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
					8:				val = REPLICATE (val, tmp[7],tmp[6],tmp[5],tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
					ELSE:	BEGIN
							PRINT, 'ERROR: TOO MANY DIMENSIONS '+'('+val_to_str(nbdims)+')'+'FOR VARIABLE ' + VARNAMES[ind6[i]]
							RETURN, 0
						END
				END
				ajouter_champ, VARNAMES[ind6[i]], val, NAMES, STRUCT, taille, first=first
				trans = trans[0] EQ -1 ? ind6[i] : [trans,ind6[i]]
				first = 0
			END
			nbchamps += dims[ind6[i]]
		END
	END

	; (7) 'DATA'
	IF nb7 NE 0 THEN BEGIN
		IF alire[ind7[0]] THEN BEGIN
			tmp = LONG(string_to_data(SIZES[ind7[0]]))
			nbdims = N_ELEMENTS(tmp)
			val = 0.0
			CASE nbdims OF
				1:	IF tmp[0] NE 1 THEN	val = REPLICATE (val, tmp[0])
				2:				val = REPLICATE (val, tmp[1],tmp[0])
				3:				val = REPLICATE (val, tmp[2],tmp[1],tmp[0])
				4:				val = REPLICATE (val, tmp[3],tmp[2],tmp[1],tmp[0])
				5:				val = REPLICATE (val, tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
				6:				val = REPLICATE (val, tmp[5],tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
				7:				val = REPLICATE (val, tmp[6],tmp[5],tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
				8:				val = REPLICATE (val, tmp[7],tmp[6],tmp[5],tmp[4],tmp[3],tmp[2],tmp[1],tmp[0])
				ELSE:	BEGIN
						PRINT, 'ERROR: TOO MANY DIMENSIONS '+'('+val_to_str(nbdims)+')'+'FOR VARIABLE ' + VARNAMES[ind7[0]]
						RETURN, 0
					END
			END
			names1 = NAMES
			ajouter_champ, VARNAMES[ind7[0]], val, names1, STRUCT, taille, first=first
			trans = trans[0] EQ -1 ? ind7[0] : [trans,ind7[0]]
			first = 0
		END
		nbchamps += dims[ind7[0]]
	END

	INFO = {	TYPE:		REPLICATE (BYTE('X'),	2,nbchamps),	$
			POSITION:	REPLICATE (-1L,		nbchamps),	$
			FILLVAL:	REPLICATE ('0',		nbchamps),	$
			NAME:		REPLICATE ('',		nbchamps)	$
	}

	taille_avant_champ = REPLICATE (0L, N_ELEMENTS(VARNAMES))
	FOR i=1L,N_ELEMENTS(VARNAMES)-1 DO BEGIN
		taille_avant_champ[i] = taille_avant_champ[i-1] + dims[i-1]
	END

	position = 0L
	iNAME = 0L
	; (1) 'DATE'
	IF nb1 NE 0 THEN BEGIN
		no = taille_avant_champ[ind1[0]]
		INFO.TYPE[0,no] = BYTE('D')
		INFO.TYPE[1,no] = BYTE(alire[ind1[0]])
		IF alire[ind1[0]] NE 0 THEN BEGIN
			INFO.NAME[iNAME] = REALVARNAMES[ind1[0]]
			iNAME++
			INFO.POSITION[no] = position
			INFO.FILLVAL [no] = '' ; ne pas tenir compte des fillval pour le type ISO_TIME
			position++
		END
	END

	; (2) 'ISO_TIME'
	IF nb2 NE 0 THEN BEGIN
		FOR i=0L,nb2-1 DO BEGIN
			no = taille_avant_champ[ind2[i]]
			INFO.TYPE[0,no+LINDGEN(dims[ind2[i]])] = BYTE('D')
			INFO.TYPE[1,no+LINDGEN(dims[ind2[i]])] = BYTE(alire[ind2[i]])
			IF alire[ind2[i]] NE 0 THEN BEGIN
				INFO.NAME[iNAME+LINDGEN(dims[ind2[i]])] = REALVARNAMES[ind2[i]]
				iNAME += dims[ind2[i]]
				INFO.POSITION[no+LINDGEN(dims[ind2[i]])] = position+LINDGEN(dims[ind2[i]])
				INFO.FILLVAL [no+LINDGEN(dims[ind2[i]])] = '' ; ne pas tenir compte des fillval pour le type ISO_TIME
				position += dims[ind2[i]]
			END
		END
	END

	; (3) 'ISO_TIME_RANGE'
	position = 0L
	IF nb3 NE 0 THEN BEGIN
		FOR i=0L,nb3-1 DO BEGIN
			no = taille_avant_champ[ind3[i]]
			INFO.TYPE[0,no+LINDGEN(dims[ind3[i]])] = BYTE('R')
			INFO.TYPE[1,no+LINDGEN(dims[ind3[i]])] = BYTE(alire[ind3[i]])
			IF alire[ind3[i]] NE 0 THEN BEGIN
				INFO.NAME[iNAME+LINDGEN(dims[ind3[i]])] = REALVARNAMES[ind3[i]]
				iNAME += dims[ind3[i]]
				INFO.POSITION[no+LINDGEN(dims[ind3[i]])] = position+LINDGEN(dims[ind3[i]])
				INFO.FILLVAL [no+LINDGEN(dims[ind3[i]])] = '' ; ne pas tenir compte des fillval pour le type ISO_TIME_RANGE
				position += dims[ind3[i]]
			END
		END
	END

	; (4) 'CHAR'
	position = 0L
	IF nb4 NE 0 THEN BEGIN
		FOR i=0L,nb4-1 DO BEGIN
			no = taille_avant_champ[ind4[i]]
			INFO.TYPE[0,no+LINDGEN(dims[ind4[i]])] = BYTE('C')
			INFO.TYPE[1,no+LINDGEN(dims[ind4[i]])] = BYTE(alire[ind4[i]])
			IF alire[ind4[i]] NE 0 THEN BEGIN
				INFO.NAME[iNAME+LINDGEN(dims[ind4[i]])] = REALVARNAMES[ind4[i]]
				iNAME += dims[ind4[i]]
				INFO.POSITION[no+LINDGEN(dims[ind4[i]])] = position+LINDGEN(dims[ind4[i]])
				INFO.FILLVAL [no+LINDGEN(dims[ind4[i]])] = '' ; ne pas tenir compte des fillval pour le type CHAR
				position += dims[ind4[i]]
			END
		END
	END

	; (5) 'STRING'
	position = 0L
	IF nb5 NE 0 THEN BEGIN
		no = taille_avant_champ[ind5[0]]
		INFO.TYPE[0,no] = BYTE('S')
		INFO.TYPE[1,no] = BYTE(alire[ind5[0]])
		IF alire[ind5[0]] NE 0 THEN BEGIN
			INFO.NAME[iNAME] = REALVARNAMES[ind5[0]]
			iNAME++
			INFO.POSITION[no] = position
			INFO.FILLVAL [no] = '' ; ne pas tenir compte des fillval pour le type STRING
		END
	END

	; (6)
	position = 0L
	IF nb6 NE 0 THEN BEGIN
		FOR i=0L,nb6-1 DO BEGIN
			no = taille_avant_champ[ind6[i]]
			INFO.TYPE[0,no+LINDGEN(dims[ind6[i]])] = STRUPCASE(TYPES[ind6[i]]) EQ 'INT' ? (BYTE('I'))[0] : (BYTE('F'))[0]
			INFO.TYPE[1,no+LINDGEN(dims[ind6[i]])] = (BYTE(alire[ind6[i]]))[0]
			IF alire[ind6[i]] NE 0 THEN BEGIN
				INFO.NAME[iNAME+LINDGEN(dims[ind6[i]])] = REALVARNAMES[ind6[i]]
				iNAME += dims[ind6[i]]
				INFO.POSITION[no+LINDGEN(dims[ind6[i]])] = position+LINDGEN(dims[ind6[i]])
				INFO.FILLVAL [no+LINDGEN(dims[ind6[i]])] = FILLVALS[ind6[i]]
				position += dims[ind6[i]]
			END
		END
	END

	; (7) 'DATA'
	IF nb7 NE 0 THEN BEGIN
		no = taille_avant_champ[ind7[0]]
		INFO.TYPE[0,no+LINDGEN(dims[ind7[0]])] = (BYTE('F'))[0]
		INFO.TYPE[1,no+LINDGEN(dims[ind7[0]])] = (BYTE(alire[ind7[0]]))[0]
		IF alire[ind7[0]] NE 0 THEN BEGIN
			INFO.NAME[iNAME+LINDGEN(dims[ind7[0]])] = REALVARNAMES[ind7[0]]
			INFO.POSITION[no+LINDGEN(dims[ind7[0]])] = position+LINDGEN(dims[ind7[0]])
			INFO.FILLVAL [no+LINDGEN(dims[ind7[0]])] = FILLVALS[ind7[0]]
		END
	END

	;duree = duree+nbsec()-topduree
	;PRINT,'duree3 = ' + val_to_str_2decimales(duree)+' sec'

	IF DEBUG THEN PRINT,'read_ascii_get_structure = ' + val_to_str_2decimales(nbsec()-topduree)+' sec'

	; Recherche vecteur champ magntique en GSE (pour Sauvaud) pour que cl calcule les composantes en GSM
	nbnames = N_ELEMENTS(VARNAMES)
	is_maggse = REPLICATE (0,nbnames)
	FOR i=0L,nbnames-1 DO BEGIN
		is_maggse[i] = STRMID (VARNAMES[i], 0, 13) EQ 'B_vec_xyz_gse'
	END 
	ind = WHERE (is_maggse EQ 1) 
	IF N_ELEMENTS(ind) EQ 3 THEN BEGIN
		name_xgse = VARNAMES[ind[0]]
		name_ygse = VARNAMES[ind[1]]
		name_zgse = VARNAMES[ind[2]]
		IF	STRMID(name_xgse,STRLEN(name_xgse)-2,2) EQ '_1' AND $
			STRMID(name_ygse,STRLEN(name_ygse)-2,2) EQ '_2' AND $
			STRMID(name_zgse,STRLEN(name_zgse)-2,2) EQ '_3' THEN BEGIN

			name_xgsm = STRMID(name_xgse,0,12) + 'm' + STRMID(name_xgse,13,STRLEN(name_xgse)-13)
			name_ygsm = STRMID(name_ygse,0,12) + 'm' + STRMID(name_ygse,13,STRLEN(name_ygse)-13)
			name_zgsm = STRMID(name_zgse,0,12) + 'm' + STRMID(name_zgse,13,STRLEN(name_zgse)-13)

			; Rajout des 3 composantes en GSM dans 'names' et 'struct'
			ajouter_champ, name_xgsm, 0.0, NAMES, STRUCT, taille
			ajouter_champ, name_ygsm, 0.0, NAMES, STRUCT, taille
			ajouter_champ, name_zgsm, 0.0, NAMES, STRUCT, taille

		END
	END

	RETURN, 1

END

;-------------------------------------------------------------------------------
FUNCTION read_ascii_get_data,				$
;-------------------------------------------------------------------------------
	FILENAME,					$	; LINT_PROTOTYPE input
	FORMAT,						$	; LINT_PROTOTYPE input
	NBRECORDS,					$	; LINT_PROTOTYPE input
	NBLINES0,					$	; LINT_PROTOTYPE input
	STRUCT,						$	; LINT_PROTOTYPE input
	INFO,						$	; LINT_PROTOTYPE input
	DATA,						$	; LINT_PROTOTYPE output
	rec_start=rec_start,				$	; LINT_PROTOTYPE input
	rec_count=rec_count,				$	; LINT_PROTOTYPE input
	modify_float_fillval=modify_float_fillval, 	$	; LINT_PROTOTYPE input
	nomalloc=nomalloc,				$	; LINT_PROTOTYPE input
	quiet=quiet,					$	; LINT_PROTOTYPE input
	votable=votable						; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((1:ok 0:pb)) Lit les donnes d'un fichier CEF dja analys et retourne un tableau de structure avec les donnes.
;
; Input:
; -----
; - FILENAME:			filename to read
; - FORMAT:			number of format for Date/Time
; - NBRECORDS:			nbrecords
; - NBLINES0:			number of lines in header
; - STRUCT:			output from read_ascii_get_structure()
; - INFO:			output from read_ascii_get_structure()
;
; Keywords:
; ---------
; - rec_start:			start read at record rec_start (0=first)
; - rec_count:			number of records to read
; - /modify_float_fillval:	to modify FLOAT fillval by -1e31
; - /nomalloc:			if DATA has already been allocated
; - /quiet: 			no messages
; - /votable: 			1 for VOTABLE format, 0 for CEF format
;
; Output:
; -------
; - DATA:			array of STRUCT with data read in FILENAME
;				'ISO_TIME' and 'ISO_TIME_RANGE' variables are stored as number of millsecondes since 01/01/1958 00:00:00.000 in a double
;				use date_to_tu to convert these fields to year, month, day, hour, minute, seconde
;
;
; Return code:
; ------------
;	1 = OK
;	0 = PB
;				'ERROR: INCORRECT FORMAT IN ' + FILENAME
;-------------------------------------------------------------------------------

	IF N_ELEMENTS(rec_start) 		EQ 0 THEN rec_start 		= 0L
	IF N_ELEMENTS(rec_count) 		EQ 0 THEN rec_count 		= NBRECORDS-rec_start
	IF N_ELEMENTS(modify_float_fillval)	EQ 0 THEN modify_float_fillval	= 0
	IF N_ELEMENTS(nomalloc) 		EQ 0 THEN nomalloc 		= 0
	IF N_ELEMENTS(quiet) 			EQ 0 THEN quiet 		= 0
	IF N_ELEMENTS(votable) 			EQ 0 THEN votable 		= 0

	IF rec_count EQ 0 THEN RETURN, 0

	IF ~nomalloc THEN DATA = REPLICATE (STRUCT,rec_count)
	taille = get_structure_length (DATA)
	nbchamps = N_ELEMENTS(INFO.type[1,*])

	TYPE		= INFO.TYPE[0,*]
	MEMORISE	= INFO.TYPE[1,*]
	POSITION	= INFO.POSITION
	SFILLVAL	= INFO.FILLVAL
	ind = WHERE(SFILLVAL EQ '') & IF ind[0] NE -1 THEN SFILLVAL[ind] = '-1e31'

	top1 = nbsec()
	IF is_fdl() THEN BEGIN
		; ex: projets/jason3/bug.cl
		NAMES = TAG_NAMES(DATA)
		FOR i=0L,N_ELEMENTS(NAMES)-1 DO BEGIN
			IF is_string(DATA[0].(i)) THEN BEGIN
				DATA.(i) = STRING(REPLICATE(BYTE('A'),1024)) ; rserver 1024 bytes pour les STRING
			END
		END
		code = CALL_EXTERNAL (libname('libascii'), 'READ_ASCII_AUTO_GLUE',									$
					FILENAME, nbchamps, TYPE, MEMORISE, POSITION, DATA, FORMAT, NBLINES0, 						$
					LONG(rec_count), LONG(rec_start), taille, pchar(SFILLVAL), LONG(modify_float_fillval), LONG(votable), 1L, /AUTO_GLUE)
		FOR i=0L,N_ELEMENTS(NAMES)-1 DO BEGIN
			IF is_string(DATA[0].(i)) THEN BEGIN
				pos = STRPOS (DATA.(i), STRING(255b))
				DATA.(i) = STRMID (DATA.(i), 0, pos) ; garder ce qu'il y a avant le caractre spcial 255
			END
		END
	END ELSE BEGIN
		code = CALL_EXTERNAL (libname('libascii'), 'READ_ASCII',										$
					FILENAME, nbchamps, TYPE, MEMORISE, POSITION, DATA, FORMAT, NBLINES0, 						$
					LONG(rec_count), LONG(rec_start), taille, SFILLVAL, LONG(modify_float_fillval), LONG(votable), 0L)
	END
	top2 = nbsec()

	IF ~quiet THEN PRINT,'reading ' + FILENAME + ' in '+ val_to_str_2decimales(top2-top1)+' sec'

	IF code NE 1 THEN BEGIN
		PRINT, 'ERROR LINE '+val_to_str(-code)+': INCORRECT FORMAT IN ' + FILENAME
		RETURN, 0
	END

	; Recherche vecteur champ magntique en GSE
	varnames = TAG_NAMES(DATA)
	nbnames = N_ELEMENTS(varnames)
	is_maggse = REPLICATE (0,nbnames)
	FOR i=0L,nbnames-1 DO BEGIN
		is_maggse[i] = STRMID (varnames[i], 0, 13) EQ 'B_VEC_XYZ_GSE'
	END
	ind = WHERE (is_maggse EQ 1)
	IF N_ELEMENTS(ind) EQ 3 THEN BEGIN
		name_xgse = varnames[ind[0]]
		name_ygse = varnames[ind[1]]
		name_zgse = varnames[ind[2]]
		IF	STRMID(name_xgse,STRLEN(name_xgse)-2,2) EQ '_1' AND $
			STRMID(name_ygse,STRLEN(name_ygse)-2,2) EQ '_2' AND $
			STRMID(name_zgse,STRLEN(name_zgse)-2,2) EQ '_3' THEN BEGIN

			name_xgsm = STRMID(name_xgse,0,12) + 'M' + STRMID(name_xgse,13,STRLEN(name_xgse)-13)
			name_ygsm = STRMID(name_ygse,0,12) + 'M' + STRMID(name_ygse,13,STRLEN(name_ygse)-13)
			name_zgsm = STRMID(name_zgse,0,12) + 'M' + STRMID(name_zgse,13,STRLEN(name_zgse)-13)

			; Calcul des 3 composantes en GSM (gse -> gsm) dans 'DATA'
			FILLVAL = -1E31
			top1 = nbsec()
			gse_to_gsm, FILLVAL, DATA,DATA, DATA.date, name_xgse, name_ygse, name_zgse, name_xgsm, name_ygsm, name_zgsm
			top2 = nbsec()
			IF top2-top1 GT 0.1 THEN PRINT,'gse_to_gsm in ' + val_to_str_2decimales(top2-top1)+' sec'
		END
	END

	RETURN, 1

END

;-------------------------------------------------------------------------------
PRO lire_struct_cef_c,	$
;-------------------------------------------------------------------------------
	filename,	$	; LINT_PROTOTYPE input
	DATA,		$	; LINT_PROTOTYPE input
	nom1,		$	; LINT_PROTOTYPE input
	nom2,		$	; LINT_PROTOTYPE input
	INFO,		$	; LINT_PROTOTYPE input
	rec_start,	$	; LINT_PROTOTYPE input
	rec_count		; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((-)) Lit certaines variables d'un fichier CEF et crit les valeurs dans certains champs d'un tableau de structure.
;
; Entres:
;
;	filename:	fichier CEF
;	nom1:		tableau de string avec les noms des champs de la structure  lire
;	nom2:		tableau de string avec les noms des variables CEF correspondantes
;	info:		tableau de string les champs  lire (ex: '', '* 1')
;	REC_START:	commencer  lire dans le CEF  l'enregistrement REC_START
;	REC_COUNT:	lire REC_COUNT enregistrements dans le fichier cef
;
; Entrs/Sorties:
;
;	DATA:		tableau de structure existante en entre, renseigne en sortie
;-------------------------------------------------------------------------------
; LINT_VARIABLES READNAMES

	lint_unused = nom1 ; pour ne pas avoir de message de LINT

	top1 = nbsec()

	quiet = 1
	noatt = 1
	nonb = 1
	remove_ext = 'No'

	code = read_ascii_get_symbols (filename, ATT, NOVARY, SYMBOLS, FORMAT, NBRECORDS, NBLINES0, quiet=quiet, noatt=noatt, nonb=nonb)
	IF ~code THEN RETURN

	code = read_ascii_analyse_symbols (SYMBOLS, VARNAMES, REALVARNAMES, TYPES, SIZES, FILLVALS, UNITS, NAME_TIME, NAME_THETA, UNIT_THETA, NAME_PHI, UNIT_PHI, NAME_ENERGY, UNIT_ENERGY, $
			remove_ext=remove_ext)
	IF ~code THEN RETURN

	FOR i=0L,N_ELEMENTS(nom2)-1 DO BEGIN
		ind = WHERE (SYMBOLS.NAME EQ nom2[i] AND SYMBOLS.DATA NE '')
		IF ind[0] NE -1 THEN BEGIN
			; les donnes existent dans le champ DATA et ne sont donc pas dans chaque enregistrement
			data1 = string_to_data(SYMBOLS[ind[0]].DATA)
			DATA.(i) = data1
		END ELSE BEGIN
			; les donnes sont a lire dans chaque enregistrement
			READNAMES = N_ELEMENTS(READNAMES) EQ 0 ? nom2[i] : [READNAMES , nom2[i]]
		END
	END

	IF N_ELEMENTS(READNAMES) NE 0 THEN BEGIN
		code = read_ascii_get_structure (VARNAMES, REALVARNAMES, TYPES, SIZES, FILLVALS, NAMES, STRUCT, INFO, readnames=READNAMES, quiet=quiet)
		IF ~code THEN RETURN

		IF ~quiet THEN BEGIN
			nbalire = rec_count
			taille = get_structure_length (STRUCT)
			msg = basename(filename)+' Reading ' + val_to_str(nbalire)+' lines ('+val_to_str_2decimales(DOUBLE(nbalire)*taille/1e6)+' Mbytes)'
			PRINT, msg
		END

		code = read_ascii_get_data (filename, FORMAT, NBRECORDS, NBLINES0, STRUCT, INFO, DATA, $
						rec_start=rec_start, rec_count=rec_count, /modify_float_fillval, /nomalloc, quiet=quiet)
	END

	top2 = nbsec()
	PRINT, 'lire_struct_cef_c en ' + val_to_str_2decimales(top2-top1)

END

;-------------------------------------------------------------------------------
FUNCTION read_cef, 		$
;-------------------------------------------------------------------------------
	FILENAME,		$	; LINT_PROTOTYPE input
	DATA,			$	; LINT_PROTOTYPE output
	ATT,			$	; LINT_PROTOTYPE [output]
	NOVARY,			$	; LINT_PROTOTYPE [output]
	VARNAMES,		$	; LINT_PROTOTYPE [output]
	NAMES,			$	; LINT_PROTOTYPE [output]
	quiet=quiet			; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; ((1:ok 0:pb)) Lit le fichier CEF donn en argument et retourne les donnes dans un tableau de structure.
;
; Input:
; -----
; - FILENAME:			filename to read
;
; Keywords:
; ---------
; - /quiet: 			no messages
;
; Output:
; -------
; - ATT:			array of structure { v1:'', v2:'', v3:'', v4:''} with all attributes found in header
; - NOVARY:			structure with variables containing a non empty DATA field
; - VARNAMES:			names of variables with empty DATA field
; - NAMES:			names of fields in the structure
; - DATA:			array of STRUCT with data read in FILENAME
;				'ISO_TIME' and 'ISO_TIME_RANGE' variables are stored as number of millsecondes since 01/01/1958 00:00:00.000 in a double
;				use date_to_tu to convert these fields to year, month, day, hour, minute, seconde
;
; Return code:
; ------------
;	1 = OK
;	0 = PB
;				'ERROR: INCORRECT FORMAT IN ' + FILENAME
;-------------------------------------------------------------------------------

	IF N_ELEMENTS(quiet)	EQ 0 THEN quiet = 0

	code = read_ascii_get_symbols (FILENAME, ATT, NOVARY, SYMBOLS, FORMAT, NBRECORDS, NBLINES0, quiet=quiet)
	IF ~code THEN RETURN, 0

	code = read_ascii_analyse_symbols (SYMBOLS, VARNAMES, REALVARNAMES, TYPES, SIZES, FILLVALS, UNITS, NAME_TIME, NAME_THETA, UNIT_THETA, NAME_PHI, UNIT_PHI, NAME_ENERGY, UNIT_ENERGY)
	IF ~code THEN RETURN, 0

	code = read_ascii_get_structure (VARNAMES, REALVARNAMES, TYPES, SIZES, FILLVALS, NAMES, STRUCT, INFO, quiet=quiet)
	IF ~code THEN RETURN, 0


	nbalire = NBRECORDS
	taille = get_structure_length (STRUCT)
	IF ~quiet THEN BEGIN
		msg = basename(FILENAME)+' Reading ' + val_to_str(nbalire)+' lines ('+val_to_str_2decimales(DOUBLE(nbalire)*taille/1e6)+' Mbytes)'
		PRINT, msg
	END

	code = read_ascii_get_data (FILENAME, FORMAT, NBRECORDS, NBLINES0, STRUCT, INFO, DATA, /modify_float_fillval, quiet=quiet)
	RETURN, code

END

;-------------------------------------------------------------------------------
FUNCTION read_time_comment,	$
;-------------------------------------------------------------------------------
	filename,		$	; LINT_PROTOTYPE input
	data				; LINT_PROTOTYPE output
;-------------------------------------------------------------------------------
; Retourne 1 si ok, 0 sinon et mets les donnes dans data (tableau de chaine de caractres)
;-------------------------------------------------------------------------------

	IF fichier_existe(filename) EQ 0 THEN RETURN, 0

	nblines0 = 0L
	usecomma = 0L
	usechar = 0L
	wc_l, filename, nblines0, usecomma, usechar, nblignes, nbspecial, nbcomma
	IF nblignes EQ 0 THEN RETURN, 0

	data = REPLICATE ('',nblignes)

	ligne = ''
	OPENR, fd, filename, /GET_LUN
	iligne = 0L
	WHILE ~EOF(fd) DO BEGIN
		READF, fd, ligne
		IF ligne EQ '' THEN CONTINUE
		data[iligne++] = ligne
	END
	FREE_LUN, fd

	RETURN, 1

END

