;-------------------------------------------------------------------------------
PRO str_to_yyyy_mm_dd,	$
;-------------------------------------------------------------------------------
	valeur,		$	; LINT_PROTOTYPE input
	annee,		$	; LINT_PROTOTYPE output
	mois,		$	; LINT_PROTOTYPE output
	jour			; LINT_PROTOTYPE output
;-------------------------------------------------------------------------------
; Retourne YYYY, MM et DD  partir d'un STRING "YYYY/MM/DD"
;-------------------------------------------------------------------------------

	tmp = STRSPLIT (valeur,'/',/EXTRACT)
	IF N_ELEMENTS(tmp) EQ 3 THEN BEGIN
		annee = tmp[0]
		mois = tmp[1]
		jour = tmp[2]
	END ELSE BEGIN
		annee = '????'
		mois = '??'
		jour = '??'
	END	

END

;-------------------------------------------------------------------------------
PRO print_infos_wsorb
;-------------------------------------------------------------------------------
; Affiche le contenu de infos_wsorb
;-------------------------------------------------------------------------------

	COMMON COMMON_INFOS_WSORBITE, infos_wsorb

	FOR i=0L,N_ELEMENTS(infos_wsorb)-1 DO BEGIN
	
		PRINT, '"'+infos_wsorb[i].mission_name		+ '" "' + $
			infos_wsorb[i].mission_targets		+ '" "' + $
			infos_wsorb[i].dataset_name		+ '" '  + $
			infos_wsorb[i].dataset_datastart	+ ' '   + $
			infos_wsorb[i].dataset_datastop		+ ' '   + $
			infos_wsorb[i].parameter_id
	END

END

;-------------------------------------------------------------------------------
FUNCTION get_infos_wsorb,	$
;-------------------------------------------------------------------------------
	nom_xml		; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; Lit le fichier xml (d'AMDA ou de CLWeb) et en dduit des infos sur les webservice d'orbite
;-------------------------------------------------------------------------------

	COMMON COMMON_INFOS_WSORBITE, infos_wsorb

	lint_unused = xml_to_noeud_variable_valeur (nom_xml, dataxml)

	IF N_ELEMENTS(infos_wsorb) EQ 0 THEN infos_wsorb = { mission_name:'', mission_targets: '', dataset_name:'', dataset_datastart:'', dataset_datastop: '', parameter_frame: '', parameter_id: '' }

	; liste des parameter_id de clweb_tree.xml contenant des positions (X,Y,Z)

	list = [								$
			'AC_EPHEMERIS_ORBIT_H0_SWE(17,18,19)',			$	;	gse km
			'AC_EPHEMERIS_ORBIT_H0_SWE(20,21,22)',			$	;	gsm km
			'AC_EPHEMERIS_ORBIT_1H_LOC(1,2,3)',			$	;	gse rt
			'AC_EPHEMERIS_ORBIT_H0_MFI(13,14,15)',			$	;	gse km
			'AC_EPHEMERIS_ORBIT_H0_MFI(16,17,18)',			$	;	gsm km
			
			'CHANGE2_EPHEMERIS_ORBIT_A(1,2,3)',			$	;	gse m
			'CHANGE2_EPHEMERIS_ORBIT_B(1,2,3)',			$	;	gse m

			'C1_EPHEMERIS_ORBIT(1,2,3)',				$	;	gse rt
			'C1_EPHEMERIS_ORBIT_DETAILED(1,2,3)',			$	;	GR rt
			'C1_EPHEMERIS_ORBIT_DETAILED(4,5,6)',			$	;	gse rt
			'C1_EPHEMERIS_ORBIT_DETAILED(7,8,9)',			$	;	gsm rt

			'C2_EPHEMERIS_ORBIT(1,2,3)',				$	;	gse rt
			'C2_EPHEMERIS_ORBIT_DETAILED(1,2,3)',			$	;	GR rt
			'C2_EPHEMERIS_ORBIT_DETAILED(4,5,6)',			$	;	gse rt
			'C2_EPHEMERIS_ORBIT_DETAILED(7,8,9)',			$	;	gsm rt

			'C3_EPHEMERIS_ORBIT(1,2,3)',				$	;	gse rt
			'C3_EPHEMERIS_ORBIT_DETAILED(1,2,3)',			$	;	GR rt
			'C3_EPHEMERIS_ORBIT_DETAILED(4,5,6)',			$	;	gse rt
			'C3_EPHEMERIS_ORBIT_DETAILED(7,8,9)',			$	;	gse rt

			'C4_EPHEMERIS_ORBIT(1,2,3)',				$	;	gse rt
			'C4_EPHEMERIS_ORBIT_DETAILED(1,2,3)',			$	;	GR rt
			'C4_EPHEMERIS_ORBIT_DETAILED(4,5,6)',			$	;	gse rt
			'C4_EPHEMERIS_ORBIT_DETAILED(7,8,9)',			$	;	gsm rt

			'DEM_EPHEMERIS_ORBIT(3,4,5)',				$	;	GEO m
			'DEM_EPHEMERIS_ORBIT(9,10,11)',				$	;	gei m

			'DSP1_EPHEMERIS_ORBIT(1,2,3)',				$	;	GR rt
			'DSP1_EPHEMERIS_ORBIT(4,5,6)',				$	;	gse rt
			'DSP1_EPHEMERIS_ORBIT(7,8,9)',				$	;	gsm rt

			'GE_EPHEMERIS_ORBIT_MGF(8,9,10)',			$	;	gse km
			'GE_EPHEMERIS_ORBIT_LEP(4,5,6)',			$	;	gse km

			'IMP8_EPHEMERIS_ORBIT(1,2,3)',				$	;	gsm rt

			'IT_EPHEMERIS_ORBIT_FM3I(1,2,3)',			$	;	gsm rt
			'IT_EPHEMERIS_ORBIT_MIFM(1,2,3)',			$	;	gsm rt
			'IT_EPHEMERIS_ORBIT_ELECTRON_DETAILED(2,3,4)',		$	;	GEO km
			'IT_EPHEMERIS_ORBIT_ELECTRON_DETAILED(14,15,16)',	$	;	gse rt
			'IT_EPHEMERIS_ORBIT_ELECTRON_DETAILED(17,18,19)',	$	;	gsm rt

			'MEX_EPHEMERIS_ORBIT(1,2,3)',				$	;	mso rm

			'MGS_EPHEMERIS_ORBIT(1,2,3)',				$	;	mso rm

			'SACC_EPHEMERIS_ORBIT(1,2,3)',				$	;	GEO km

			'STA_EPHEMERIS_ORBIT(1,2,3)',				$	;	GEO rt
			'STA_EPHEMERIS_ORBIT(4,5,6)',				$	;	gse rt

			'STB_EPHEMERIS_ORBIT(1,2,3)',				$	;	GEO rt
			'STB_EPHEMERIS_ORBIT(4,5,6)',				$	;	gse rt

			'THA_EPHEMERIS_ORBIT(1,2,3)',				$	;	GEI km
			'THA_EPHEMERIS_ORBIT(15,16,17)',			$	;	gse rt
			'THA_EPHEMERIS_ORBIT(18,19,20)',			$	;	gsm rt

			'THB_EPHEMERIS_ORBIT(1,2,3)',				$	;	GEI rt
			'THB_EPHEMERIS_ORBIT(15,16,17)',			$	;	gse rt
			'THB_EPHEMERIS_ORBIT(18,19,20)',			$	;	gsm rt

			'THC_EPHEMERIS_ORBIT(1,2,3)',				$	;	GEI rt
			'THC_EPHEMERIS_ORBIT(15,16,17)',			$	;	gse rt
			'THC_EPHEMERIS_ORBIT(18,19,20)',			$	;	gsm rt

			'THD_EPHEMERIS_ORBIT(1,2,3)',				$	;	GEI rt
			'THD_EPHEMERIS_ORBIT(15,16,17)',			$	;	gse rt
			'THD_EPHEMERIS_ORBIT(18,19,20)',			$	;	gsm rt

			'THE_EPHEMERIS_ORBIT(1,2,3)',				$	;	GEI rt
			'THE_EPHEMERIS_ORBIT(15,16,17)',			$	;	gse rt
			'THE_EPHEMERIS_ORBIT(18,19,20)',			$	;	gsm rt

			'VEX_EPHEMERIS_ORBIT(1,2,3)',				$	;	mso rv

			'WI_EPHEMERIS_ORBIT_K0_3DP(7,8,9)',			$	;	gse km
			'WI_EPHEMERIS_ORBIT_K0_3DP(38,39,40)',			$	;	gse rt
			'WI_EPHEMERIS_ORBIT_H0_MFI(20,21,22)',			$	;	gsm rt
			'WI_EPHEMERIS_ORBIT_H0_MFI(23,24,25)',			$	;	gse rt
			'WI_EPHEMERIS_ORBIT_OR_PRE(4,5,6)',			$	;	CGI rt
			'WI_EPHEMERIS_ORBIT_OR_PRE(4,5,6)',			$	;	GCI km
			'WI_EPHEMERIS_ORBIT_OR_PRE(10,11,12)',			$	;	gse km
			'WI_EPHEMERIS_ORBIT_OR_PRE(16,17,18)',			$	;	gsm km
			'WI_EPHEMERIS_ORBIT_OR_PRE(25,26,27)',			$	;	HEC km
			'WI_EPHEMERIS_ORBIT_H1_SWE(56,57,58)',			$	;	gse rt
			'WI_EPHEMERIS_ORBIT_K0_SWE(4,5,6)',			$	;	gse km
			'WI_EPHEMERIS_ORBIT_K0_SWE(7,8,9)',			$	;	gsm km

			'XMM_EPHEMERIS_ORBIT(1,2,3)'				$	;	GEO km

	]

	FOR i=0L,N_ELEMENTS(dataxml)-1 DO BEGIN

		IF dataxml[i].noeud EQ 'mission' THEN BEGIN
			IF dataxml[i].variable EQ 'name'   THEN mission_name   = dataxml[i].valeur
			IF dataxml[i].variable EQ 'targets' THEN mission_targets = dataxml[i].valeur
		END ELSE IF dataxml[i].noeud EQ 'dataset' THEN BEGIN
			IF dataxml[i].variable EQ 'name' THEN BEGIN
				dataset_name = dataxml[i].valeur
			END
			IF dataxml[i].variable EQ 'dataStart' THEN BEGIN
				str_to_yyyy_mm_dd, dataxml[i].valeur, annee, mois, jour
				dataset_datastart = annee+'-'+mois+'-'+jour
			END
			IF dataxml[i].variable EQ 'dataStop' THEN BEGIN
				str_to_yyyy_mm_dd, dataxml[i].valeur, annee, mois, jour
				dataset_datastop = annee+'-'+mois+'-'+jour
			END
		END ELSE IF dataxml[i].noeud EQ 'parameter' THEN BEGIN
			IF dataxml[i].variable EQ 'name' THEN BEGIN
				parameter_name = STRUPCASE(dataxml[i].valeur)
			END
			IF dataxml[i].variable EQ 'xml:id' THEN BEGIN
				ind = WHERE (list EQ dataxml[i].valeur)
				IF ind[0] NE -1 THEN BEGIN
					frames = ['GSE','GSM','GEI','GEO','GCI','HEC','MSO','VSO']
					parameter_frame = ''
					FOR j=0L,N_ELEMENTS(frames)-1 DO BEGIN
						IF chaine_contient(parameter_name,' '+frames[j]) OR chaine_contient(parameter_name,frames[j]+' ') THEN BEGIN
							parameter_frame = frames[j]
						END
					END
					IF parameter_frame EQ '' THEN BEGIN
						PRINT,'ERREUR ORBIT ' + mission_name+' '+dataset_name+' '+parameter_name
						EXIT
					END
					IF parameter_frame NE '' THEN BEGIN
						parameter_id = dataxml[i].valeur
						element = { mission_name:mission_name, mission_targets:mission_targets, dataset_name:dataset_name, dataset_datastart:dataset_datastart, dataset_datastop:dataset_datastop, parameter_frame:parameter_frame, parameter_id:parameter_id }
						infos_wsorb = infos_wsorb[0].mission_name EQ '' ? [element] : [infos_wsorb,element]
					END
				END
			END
		END
	END


	RETURN, 1

END

;-------------------------------------------------------------------------------
FUNCTION simulatedregion_to_infos,	$
;-------------------------------------------------------------------------------
	SimulatedRegion,		$	; LINT_PROTOTYPE input
	CoordinateSystemName,		$	; LINT_PROTOTYPE input
	Units					; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; Retourne,  partir de SimulatedRegion, un tableau de structure avec nom de satellite, temps dbut et temps fin.
;-------------------------------------------------------------------------------
;
; ex de SimulatedRegion:
;
;	pour LATMOS:
;		"Ganymede"
;		"Mars"
;		"Mercury"
;
;	pour FMI_HYB:
;		"Mars"
;		"Venus"
;
;	pour FMI_GUMICS:
;		"Earth"
;
;	pour SINP:
;		"Earth.Magnetosphere"
;
;	pour LESIA:
;		"Jupiter"
;
; ex de CoordinateSystemName+Units: "MSO"+"km", "VSO"+"m", "GSE"+"m"
;
;-------------------------------------------------------------------------------

	COMMON COMMON_INFOS_WSORBITE, infos_wsorb


	; Mettre dans target la premire valeur avant '.' de SimulatedRegion
	target = (STRSPLIT(SimulatedRegion,'.',/EXTRACT))[0]

	; Rechercher target dans infos_wsorb
	nb = N_ELEMENTS(infos_wsorb)

	infos = REPLICATE ({ nomsat: 'No mission available in cl', nomsat_court:'', parameter_id:'', frame: '', units: '', startdate: '???', stopdate: '???' }, nb)

	j = 0L
	; pour chaque lment de infows_wsorb
	FOR i=0L,nb-1 DO BEGIN


		; garder tous les targets qui commencent par le premier mot de SimulatedRegion
		mission_targets = STRSPLIT(infos_wsorb[i].mission_targets,' ',/EXTRACT)
		target_trouve = 0

		; pour chaque lment de mission_targets
		FOR k=0L,N_ELEMENTS(mission_targets)-1 DO BEGIN
			IF chaine_contient_debut(mission_targets[k],target) THEN BEGIN
				target_trouve = 1
				BREAK
			END
		END
		IF ~target_trouve THEN CONTINUE

		; parameter_frame doit valoir CoordinateSystemName
		IF infos_wsorb[i].parameter_frame NE STRUPCASE(CoordinateSystemName) THEN CONTINUE

		infos[j].nomsat       = infos_wsorb[i].mission_name + ' (' + infos_wsorb[i].dataset_name + ')'
		infos[j].parameter_id = infos_wsorb[i].parameter_id
		infos[j].nomsat_court = (STRSPLIT (infos_wsorb[i].parameter_id, '_', /EXTRACT))[0]
		infos[j].frame        = CoordinateSystemName
		infos[j].units        = Units
		infos[j].startdate    = infos_wsorb[i].dataset_datastart+'T00:00:00.000'
		infos[j].stopdate     = infos_wsorb[i].dataset_datastop+'T00:00:00.000'
		j++
	END

	RETURN, j EQ 0 ? infos[0] : infos[0:j-1]


END

;-------------------------------------------------------------------------------
FUNCTION analyse_impex,	$
;-------------------------------------------------------------------------------
	filename,	$	; LINT_PROTOTYPE input
	res,		$	; LINT_PROTOTYPE output
	master=master		; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; Lit les symboles dans le fichier filename (cef ou cdf) et retourne dans res la table des symboles. Retourne 1 si OK, 0 sinon.
;
; master=name pour crer le fichier master name.
;-------------------------------------------------------------------------------

	IF isxml(filename) THEN BEGIN

		code = read_votable_get_symbols (filename,SYMBOLS,FORMAT,NBRECORDS,NBLINES0,/nonb)
		IF ~code THEN RETURN, 0

		IF N_ELEMENTS(master) NE 0 THEN BEGIN
			lint_unused = ecrire_master (master, SYMBOLS)
			PRINT, 'Creation master '+ master
		END

	END ELSE IF isnetcdf(filename) THEN BEGIN

		; fichiers trop gros, pas de ISO_TIME
		RETURN, 0

		code = read_netcdf_get_symbols (filename, ATT, NOVARY, SYMBOLS, FORMAT, NBRECORDS, nblignes0, $
						selection_date='Automatic')
		IF ~code THEN RETURN, 0

		RETURN, 0

	END

        NBDIMS = REPLICATE (1L, N_ELEMENTS(SYMBOLS))
        FOR i=0L,N_ELEMENTS(SYMBOLS)-1 DO NBDIMS[i] = N_ELEMENTS(LONG(string_to_data(SYMBOLS[i].sizes)))
        DATA = SYMBOLS.DATA
        DEPEND_0 = SYMBOLS.DEPEND_0
        DEPEND_1 = SYMBOLS.DEPEND_1
        DEPEND_2 = SYMBOLS.DEPEND_2
        DEPEND_3 = SYMBOLS.DEPEND_3
        DEPEND_4 = SYMBOLS.DEPEND_4

	res = REPLICATE ({ name:'', name1:'', date:'', data:'', value_type:'', nbd:'', nbdims:'', sizes:'', type:'', fieldnam:'', units:'', var_type:'' },N_ELEMENTS(SYMBOLS))
	FOR i=0L,N_ELEMENTS(SYMBOLS)-1 DO BEGIN
		nbdim = 0
		IF DATA[i] EQ '' AND DEPEND_0[i] NE '' AND DEPEND_1[i] NE '' AND DEPEND_2[i] EQ '' AND DEPEND_3[i] EQ '' AND DEPEND_4[i] EQ '' AND NBDIMS[i] EQ 1 AND SYMBOLS[i].sizes NE '1' THEN nbdim = 1
		IF DATA[i] EQ '' AND DEPEND_0[i] NE '' AND DEPEND_1[i] NE '' AND DEPEND_2[i] NE '' AND DEPEND_3[i] EQ '' AND DEPEND_4[i] EQ '' AND NBDIMS[i] EQ 2 THEN nbdim = 2
		IF DATA[i] EQ '' AND DEPEND_0[i] NE '' AND DEPEND_1[i] NE '' AND DEPEND_2[i] NE '' AND DEPEND_3[i] NE '' AND DEPEND_4[i] EQ '' AND NBDIMS[i] EQ 3 THEN nbdim = 3
		IF DATA[i] EQ '' AND DEPEND_0[i] NE '' AND DEPEND_1[i] NE '' AND DEPEND_2[i] NE '' AND DEPEND_3[i] NE '' AND DEPEND_4[i] NE '' AND NBDIMS[i] EQ 4 THEN nbdim = 4
		res[i].name             = SYMBOLS[i].name
		res[i].name1            = corriger_nomchamp(SYMBOLS[i].name)
		res[i].nbd              = val_to_str(nbdim)
		res[i].nbdims           = val_to_str(NBDIMS[i])
		res[i].sizes            = SYMBOLS[i].sizes
		res[i].date             = STRTRIM(SYMBOLS[i].depend_0,2)
		res[i].value_type       = SYMBOLS[i].value_type
		res[i].data             = STRTRIM(SYMBOLS[i].data,2)
		res[i].fieldnam         = SYMBOLS[i].fieldnam
		res[i].units            = SYMBOLS[i].units
		res[i].var_type         = SYMBOLS[i].var_type
		IF nbdim EQ 1 THEN res[i].type = 'time_energy'
		IF nbdim EQ 2 THEN res[i].type = STRUPCASE(DEPEND_1[i]) EQ 'ENERGY' ? 'time_theta' : 'time_energy'
		IF nbdim EQ 3 THEN res[i].type = STRUPCASE(DEPEND_1[i]) EQ 'ENERGY' ? 'time_theta' : (STRUPCASE(DEPEND_2[i]) EQ 'ENERGY' ? 'time_phi' : 'time_energy')
	END

        RETURN, code

END

;-------------------------------------------------------------------------------
PRO ecrire_html,		$
;-------------------------------------------------------------------------------
	fd,			$	; LINT_PROTOTYPE input
	numero,			$	; LINT_PROTOTYPE input
	method,			$	; LINT_PROTOTYPE input
	SimulatedRegion,	$	; LINT_PROTOTYPE input
	CoordinateSystemName,	$	; LINT_PROTOTYPE input
	Units,			$	; LINT_PROTOTYPE input
	nosat,			$	; LINT_PROTOTYPE input
	nomsats,		$	; LINT_PROTOTYPE input
	resourceid,		$	; LINT_PROTOTYPE input
	day_code_link,		$	; LINT_PROTOTYPE input
	message,		$	; LINT_PROTOTYPE input
	couleur,		$	; LINT_PROTOTYPE input
	strike=strike			; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; Ecrit une ligne dans le fichier html ouvert fd.
;-------------------------------------------------------------------------------

	IF nosat NE 0 THEN RETURN ; pour n'afficher un message que pour le premier satellite

	IF couleur EQ '' THEN BEGIN
		PRINTF, fd, '<TR>'
	END ELSE BEGIN
		PRINTF, fd, '<TR bgcolor="'+couleur+'">'
	END

	PRINTF, fd, '<TD>'
	PRINTF, fd, val_to_str(numero)
	PRINTF, fd, '</TD>'

	PRINTF, fd, '<TD>'
	IF KEYWORD_SET(strike) THEN PRINTF, fd, '<s>'
	PRINTF, fd, '<b>'+method+'</b>'
	IF KEYWORD_SET(strike) THEN PRINTF, fd, '</s>'
	PRINTF, fd, '</TD>'

	PRINTF, fd, '<TD>'
	IF KEYWORD_SET(strike) THEN PRINTF, fd, '<s>'
	PRINTF, fd, '<b>'+SimulatedRegion+'</b>'
	PRINTF, fd, '<b>'+CoordinateSystemName+'</b>'
	PRINTF, fd, '<b>'+Units+'</b>'
	IF KEYWORD_SET(strike) THEN PRINTF, fd, '</s>'
	PRINTF, fd, '</TD>'

	PRINTF, fd, '<TD>'
	IF KEYWORD_SET(strike) THEN PRINTF, fd, '<s>'
	FOR i=0L,N_ELEMENTS(nomsats)-1 DO BEGIN
		IF i EQ 0 THEN BEGIN
			PRINTF, fd, '<b><u>'+nomsats[i]+'</u></b><br>'
		END ELSE BEGIN
			PRINTF, fd, '<b>'+nomsats[i]+'</b><br>'
		END
	END
	IF KEYWORD_SET(strike) THEN PRINTF, fd, '</s>'
	PRINTF, fd, '</TD>'

	PRINTF, fd, '<TD>'
	IF KEYWORD_SET(strike) THEN PRINTF, fd, '<s>'
	tmp = STRSPLIT(resourceid,'@',/EXTRACT)
	PRINTF, fd, '<b>'+STRJOIN(tmp,'<br>')+'</b>'
	IF KEYWORD_SET(strike) THEN PRINTF, fd, '</s>'
	PRINTF, fd, '</TD>'

	PRINTF, fd, '<TD>'
	PRINTF, fd, day_code_link
	PRINTF, fd, '</TD>'

	PRINTF, fd, '<TD>'
	PRINTF, fd, message
	PRINTF, fd, '</TD>'

	PRINTF, fd, '</TR>'

END


;-------------------------------------------------------------------------------
FUNCTION get_vecteur,	$
;-------------------------------------------------------------------------------
	s			; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; Decoupe la chaine de caracteres s selon ' ' ou ',' et retourne les composantes.
;-------------------------------------------------------------------------------

	IF STRPOS(s,',') NE -1 THEN BEGIN
		RETURN, STRSPLIT(s,',',/EXTRACT)
	END ELSE BEGIN
		RETURN, STRSPLIT(s,/EXTRACT)
	END

END

;-------------------------------------------------------------------------------
PRO analyse_xml,					$
;-------------------------------------------------------------------------------
	type,						$	; LINT_PROTOTYPE input
	dataxml,					$	; LINT_PROTOTYPE input
	Repository_ResourceHeader_ResourceName,		$	; LINT_PROTOTYPE output
	SimulationModels,				$	; LINT_PROTOTYPE output
	infos,						$	; LINT_PROTOTYPE output
	VOIR_GETFILEURL=VOIR_GETFILEURL				; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; Analyse les donnees XML de dataxml.
;
; type: 'LATMOS', 'FMI_HYB', 'FMI_GUMICS' ou 'SINP'
;-------------------------------------------------------------------------------

	IF N_ELEMENTS(VOIR_GETFILEURL) EQ 0 THEN VOIR_GETFILEURL = 0

	IF type EQ 'SINP' THEN BEGIN

		; Recherche Repository/ResourceHeader/Description
		data = chercher_noeuds_xml (dataxml, ['Repository','ResourceHeader','Description'])

	END ELSE BEGIN

		; Recherche Repository/ResourceHeader/ResourceName
		data = chercher_noeuds_xml (dataxml, ['Repository','ResourceHeader','ResourceName'])

	END

	Repository_ResourceHeader_ResourceName = STRJOIN (STRSPLIT(data[0].valeur,'_',/EXTRACT),' ')

	; Recherche des modeles dans SimulationModel
	debut_SimulationModel = WHERE (dataxml.noeud EQ 'SimulationModel')
	fin_SimulationModel   = WHERE (dataxml.noeud EQ '/SimulationModel')
	SimulationModels = REPLICATE ({ txt:'', desc:'' },N_ELEMENTS(debut_SimulationModel))
	FOR i=0L,N_ELEMENTS(debut_SimulationModel)-1 DO BEGIN
		data = dataxml[debut_SimulationModel[i]:fin_SimulationModel[i]]
		ResourceID = (chercher_noeuds_xml (data, ['ResourceID']))[0].valeur
		ReleaseDate = (chercher_noeuds_xml (data, ['ResourceHeader','ReleaseDate']))[0].valeur
		Description = (chercher_noeuds_xml (data, ['ResourceHeader','Description']))[0].valeur ; Description contient des retours  la ligne
		Description = STRTRIM((STRSPLIT(Description,STRING(10b),/EXTRACT))[0],2)
		SimulationModels[i].txt = ResourceID
		SimulationModels[i].desc = Description + '\nReleaseDate: '+ReleaseDate
	END

	MAX = 20000
	infos = REPLICATE ({	trie1:			'',	$
				trie2:			'',	$
				trie3:			'',	$
				trie4:			'',	$
				trie5:			'',	$
				trie6:			'',	$
				MeasurementType:	'',	$
				SimulationProduct:	'',	$
				ResourceID:		'',	$
				InputResourceID:	'',	$
				ModelID:		'',	$
				ResourceName:		'',	$
				SimulatedRegion:	'',	$
				CoordinateSystemName:	'',	$
				Units:			'',	$
				SimulationRun:		'',	$
				StartDate:		'',	$
				StopDate:		'',	$
				url:			'',	$
				desc:			''	$
			}, MAX)
	nblignes = 0L

	IF type EQ 'LESIA' THEN BEGIN
		debut_NumericalOutput = WHERE (dataxml.noeud EQ 'SimulationModel')
		fin_NumericalOutput   = WHERE (dataxml.noeud EQ '/SimulationModel')
	END ELSE BEGIN
		debut_NumericalOutput = WHERE (dataxml.noeud EQ 'NumericalOutput')
		fin_NumericalOutput   = WHERE (dataxml.noeud EQ '/NumericalOutput')
	END

	FOR i=0L,N_ELEMENTS(debut_NumericalOutput)-1 DO BEGIN
		;IF i NE 0 THEN CONTINUE
		data = dataxml[debut_NumericalOutput[i]:fin_NumericalOutput[i]]
		ResourceID = data[WHERE (data.noeud EQ 'ResourceID')].valeur
		IF type EQ 'LESIA' THEN BEGIN
			MeasurementType = 'MagneticField'
			SimulationProduct = '???'
			InputResourceID = '???'
		END ELSE BEGIN
			MeasurementType = data[WHERE (data.noeud EQ 'MeasurementType')].valeur
			SimulationProduct = data[WHERE (data.noeud EQ 'SimulationProduct')].valeur
			InputResourceID = (chercher_noeuds_xml (data, ['InputResourceID']))[0].valeur
		END

		; 9/4/2015: pas de rponse du serveur FMI_GUMICS pour InputResourceID contenant "synth_dynamical"
		IF type EQ 'FMI_GUMICS' AND chaine_contient(InputResourceID,"synth_dynamical") THEN CONTINUE

		ResourceName = data[WHERE (data.noeud EQ 'ResourceName')].valeur
		SimulatedRegion = data[WHERE (data.noeud EQ 'SimulatedRegion')].valeur

		; LATMOS: Ne garder que les /3D si on ne veut pas les GetFileURL
		IF type EQ 'LATMOS' AND VOIR_GETFILEURL EQ 0 THEN IF STRPOS(ResourceID, '/3D') EQ -1 THEN CONTINUE

		; LATMOS: je ne me sert pas de /2D
		IF type EQ 'LATMOS' THEN IF STRPOS(ResourceID, '/2D') NE -1 THEN CONTINUE

		ind = (WHERE (data.noeud EQ 'StartDate'))[0]
		StartDate = ind EQ -1 ? "???" : data[ind].valeur

		ind = (WHERE (data.noeud EQ 'StopDate'))[0]
		StopDate = ind EQ -1 ? "???" : data[ind].valeur

		tmp = STRSPLIT(ResourceID,'/',/EXTRACT)

		IF N_ELEMENTS(tmp) GE 6 THEN BEGIN
			SimulationRun = tmp[4] + '_' + tmp[3] + '_' + tmp[5]
		END ELSE IF N_ELEMENTS(tmp) GE 5 THEN BEGIN
			SimulationRun = tmp[4] + '_' + tmp[3]
		END ELSE BEGIN
			SimulationRun = STRJOIN(tmp[1:*],'_') ; LESIA
		END

		; Chercher le desc correspondant  ResourceName
		IF type EQ 'LESIA' THEN BEGIN
			debut_SimulationRun = WHERE (dataxml.noeud EQ 'SimulationModel')
			fin_SimulationRun   = WHERE (dataxml.noeud EQ '/SimulationModel')
			ModelID = 'impex://LESIA/MAG/Jupiter'
		END ELSE BEGIN
			debut_SimulationRun = WHERE (dataxml.noeud EQ 'SimulationRun')
			fin_SimulationRun   = WHERE (dataxml.noeud EQ '/SimulationRun')
		END

		CoordinateSystemName = '' ; pour LESIA
		Units = '' ; on ne sait jamais

		SolarWindp_Density = '???'
		SolarWindp_Temperature = '???'
		SolarWindp_FlowSpeed = '???'
		SolarWinde_Density = '???'
		SolarWinde_Temperature = '???'
		SolarWinde_FlowSpeed = '???'

		FOR j=0L,N_ELEMENTS(debut_SimulationRun)-1 DO BEGIN

			data1 = dataxml[debut_SimulationRun[j]:fin_SimulationRun[j]]
			ResourceID1 = (chercher_noeuds_xml (data1, ['ResourceID']))[0].valeur
			IF ResourceID1 NE InputResourceID THEN CONTINUE
			CoordinateRepresentation = (chercher_noeuds_xml (data1, ['SimulationDomain','CoordinateSystem','CoordinateRepresentation']))[0].valeur
			CoordinateSystemName = (chercher_noeuds_xml (data1, ['SimulationDomain','CoordinateSystem','CoordinateSystemName']))[0].valeur
			Units = (chercher_noeuds_xml (data1, ['SimulationDomain','Units']))[0].valeur
			ValidMin = STRSPLIT((chercher_noeuds_xml (data1, ['SimulationDomain','ValidMin']))[0].valeur,/EXTRACT)
			ValidMax = STRSPLIT((chercher_noeuds_xml (data1, ['SimulationDomain','ValidMax']))[0].valeur,/EXTRACT)
			ModelID = (chercher_noeuds_xml (data1, ['Model','ModelID']))[0].valeur
			nomodel = (WHERE (SimulationModels.txt EQ ModelID))[0]

			; PATCH 7/4/2015: bug dans LATMOS_tree.xml 'spase://IMEX/SimulationModel/LATMOS/Hybrid''
			IF nomodel EQ -1 THEN ModelID = SimulationModels[0].txt

			IMF_Units = ''
			IMF_FieldValue = ['???','???','???']
			CrustalField_FieldModel = ''
			debut_InputField = WHERE (data1.noeud EQ 'InputField')
			fin_InputField   = WHERE (data1.noeud EQ '/InputField')
			CenterDipoleField_Units = ''
			CenterDipoleField_FieldValue = ''
			CenterDipoleField_FieldModel = ''
			FOR k=0L,N_ELEMENTS(debut_InputField)-1 DO BEGIN
				IF debut_InputField[0] EQ -1 THEN CONTINUE
				IF fin_InputField[0]   EQ -1 THEN CONTINUE
				data2 = data1[debut_InputField[k]:fin_InputField[k]]
				Name = (chercher_noeuds_xml (data2, ['Name']))[0].valeur
				IF Name EQ 'IMF' THEN BEGIN
					IMF_Units = (chercher_noeuds_xml (data2, ['Units']))[0].valeur
					tmp = (chercher_noeuds_xml (data2, ['FieldValue']))[0]
					IF tmp.noeud NE '' THEN IMF_FieldValue = get_vecteur(tmp.valeur)
				END
				IF Name EQ 'CrustalField' THEN BEGIN
					CrustalField_FieldModel = get_vecteur((chercher_noeuds_xml (data2, ['FieldModel']))[0].valeur)
				END
				IF Name EQ 'Center_Dipole_Field' THEN BEGIN
					CenterDipoleField_Units = (chercher_noeuds_xml (data2, ['Units']))[0].valeur
					CenterDipoleField_FieldValue = get_vecteur((chercher_noeuds_xml (data2, ['FieldValue']))[0].valeur)
					CenterDipoleField_FieldModel = (chercher_noeuds_xml (data2, ['FieldModel']))[0].valeur
				END
				
			END

			debut_InputPopulation = WHERE (data1.noeud EQ 'InputPopulation')
			fin_InputPopulation   = WHERE (data1.noeud EQ '/InputPopulation')
			FOR k=0L,N_ELEMENTS(debut_InputPopulation)-1 DO BEGIN
				IF debut_InputPopulation[0] EQ -1 THEN CONTINUE
				IF fin_InputPopulation[0]   EQ -1 THEN CONTINUE
				data2 = data1[debut_InputPopulation[k]:fin_InputPopulation[k]]
				Name = (chercher_noeuds_xml (data2, ['Name']))[0].valeur
				IF Name EQ 'Solar Wind electrons' THEN BEGIN
					SolarWindElectrons_Density = (chercher_noeuds_xml (data2, ['PopulationDensity']))[1].valeur + ' '+(chercher_noeuds_xml (data2, ['PopulationDensity']))[0].valeur
					SolarWindElectrons_Temperature = (chercher_noeuds_xml (data2, ['PopulationTemperature']))[2].valeur + ' '+(chercher_noeuds_xml (data2, ['PopulationTemperature']))[0].valeur
					SolarWindElectrons_FlowSpeed = (chercher_noeuds_xml (data2, ['PopulationFlowSpeed']))[1].valeur + ' '+(chercher_noeuds_xml (data2, ['PopulationFlowSpeed']))[0].valeur
				END
				IF Name EQ 'Solar Wind H' THEN BEGIN
					SolarWindH_Density = (chercher_noeuds_xml (data2, ['PopulationDensity']))[1].valeur + ' '+(chercher_noeuds_xml (data2, ['PopulationDensity']))[0].valeur
					SolarWindH_Temperature = (chercher_noeuds_xml (data2, ['PopulationTemperature']))[2].valeur + ' '+(chercher_noeuds_xml (data2, ['PopulationTemperature']))[0].valeur
					SolarWindH_FlowSpeed = (chercher_noeuds_xml (data2, ['PopulationFlowSpeed']))[1].valeur + ' '+(chercher_noeuds_xml (data2, ['PopulationFlowSpeed']))[0].valeur
				END
				IF Name EQ 'Solar Wind He' THEN BEGIN
					SolarWindHe_Density = (chercher_noeuds_xml (data2, ['PopulationDensity']))[1].valeur + ' '+(chercher_noeuds_xml (data2, ['PopulationDensity']))[0].valeur
					SolarWindHe_Temperature = (chercher_noeuds_xml (data2, ['PopulationTemperature']))[2].valeur + ' '+(chercher_noeuds_xml (data2, ['PopulationTemperature']))[0].valeur
					SolarWindHe_FlowSpeed = (chercher_noeuds_xml (data2, ['PopulationFlowSpeed']))[1].valeur + ' '+(chercher_noeuds_xml (data2, ['PopulationFlowSpeed']))[0].valeur
				END
				IF Name EQ 'solarwind_H+' THEN BEGIN
					SolarWindHp_Density = (chercher_noeuds_xml (data2, ['PopulationDensity']))[1].valeur + ' '+(chercher_noeuds_xml (data2, ['PopulationDensity']))[0].valeur
					SolarWindHp_Temperature = (chercher_noeuds_xml (data2, ['PopulationTemperature']))[1].valeur + ' '+(chercher_noeuds_xml (data2, ['PopulationTemperature']))[0].valeur
					SolarWindHp_FlowSpeed = (chercher_noeuds_xml (data2, ['PopulationFlowSpeed']))[1].valeur + ' '+(chercher_noeuds_xml (data2, ['PopulationFlowSpeed']))[0].valeur
				END
				IF Name EQ 'SW_p' THEN BEGIN
					tmp = chercher_noeuds_xml (data2, ['PopulationDensity'])
					SolarWindp_Density = N_ELEMENTS(tmp) EQ 1 ? '???' : (tmp[1].valeur + ' '+tmp[0].valeur)
					tmp = chercher_noeuds_xml (data2, ['PopulationTemperature'])
					SolarWindp_Temperature = N_ELEMENTS(tmp) EQ 1 ? '???' : (tmp[1].valeur + ' '+tmp[0].valeur)
					tmp = chercher_noeuds_xml (data2, ['PopulationFlowSpeed'])
					SolarWindp_FlowSpeed = N_ELEMENTS(tmp) EQ 1 ? '???' : (tmp[1].valeur + ' '+tmp[0].valeur)
				END
				IF Name EQ 'SW_e-' THEN BEGIN
					tmp = chercher_noeuds_xml (data2, ['PopulationDensity'])
					SolarWinde_Density = N_ELEMENTS(tmp) EQ 1 ? '???' : (tmp[1].valeur + ' '+tmp[0].valeur)
					tmp = chercher_noeuds_xml (data2, ['PopulationTemperature'])
					SolarWinde_Temperature = N_ELEMENTS(tmp) EQ 1 ? '???' : (tmp[1].valeur + ' '+tmp[0].valeur)
					tmp = chercher_noeuds_xml (data2, ['PopulationFlowSpeed'])
					SolarWinde_FlowSpeed = N_ELEMENTS(tmp) EQ 1 ? '???' : (tmp[1].valeur + ' '+tmp[0].valeur)
				END
				
			END

			RegionParameter_SimulatedRegion = (chercher_noeuds_xml (data1, ['RegionParameter','SimulatedRegion']))[0].valeur
			RegionParameter_Radius = (chercher_noeuds_xml (data1, ['RegionParameter','Radius']))[1].valeur
			RegionParameter_Radius_Units = (chercher_noeuds_xml (data1, ['RegionParameter','Radius']))[0].valeur
			; Exprimer RegionParameter_Radius dans l'unit Units
			IF Units EQ 'm'  AND RegionParameter_Radius_Units EQ 'km' THEN RegionParameter_Radius *= DOUBLE(1000)
			IF Units EQ 'Re' OR Units EQ 'Rs' THEN RegionParameter_Radius = DOUBLE(1)
		END

		IF type EQ 'LESIA' THEN BEGIN
			desc = 'LESIA'
		END ELSE BEGIN
			desc = 'Reference Frame: \n'
			desc += CoordinateSystemName + ', '+ CoordinateRepresentation+'\n'
			desc += '\n'
			desc += 'Domain:\n'
			desc += 'Units: radius of '+ RegionParameter_SimulatedRegion+'\n'
			desc += 'X=['+val_to_str_2decimales(DOUBLE(ValidMin[0])/DOUBLE(RegionParameter_Radius))+','+val_to_str_2decimales(DOUBLE(ValidMax[0])/DOUBLE(RegionParameter_Radius))+']\n'
			desc += 'Y=['+val_to_str_2decimales(DOUBLE(ValidMin[1])/DOUBLE(RegionParameter_Radius))+','+val_to_str_2decimales(DOUBLE(ValidMax[1])/DOUBLE(RegionParameter_Radius))+']\n'
			desc += 'Z=['+val_to_str_2decimales(DOUBLE(ValidMin[2])/DOUBLE(RegionParameter_Radius))+','+val_to_str_2decimales(DOUBLE(ValidMax[2])/DOUBLE(RegionParameter_Radius))+']\n'
			desc += '\n'
		END

		IF type EQ 'LATMOS' THEN BEGIN
			desc += 'Upstream Plasma Properties:\n'
			desc += 'IMF ('+IMF_Units+') ['+IMF_FieldValue[0]+','+IMF_FieldValue[1]+','+IMF_FieldValue[2]+']\n'
			IF CrustalField_FieldModel NE '' THEN desc += 'CrustalField '+CrustalField_FieldModel+'\n'
			desc += '\n'
			desc += 'Solar Wind electrons:\n'
			desc += 'Density = '+SolarWindElectrons_Density+'\n'
			desc += 'Temperature = '+SolarWindElectrons_Temperature+'\n'
			desc += 'FlowSpeed = '+SolarWindElectrons_FlowSpeed+'\n'
			desc += '\n'
			desc += 'Solar Wind H:\n'
			desc += 'Density = '+SolarWindH_Density+'\n'
			desc += 'Temperature = '+SolarWindH_Temperature+'\n'
			desc += 'FlowSpeed = '+SolarWindH_FlowSpeed+'\n'
			desc += '\n'
			desc += 'Solar Wind He:\n'
			desc += 'Density = '+SolarWindHe_Density+'\n'
			desc += 'Temperature = '+SolarWindHe_Temperature+'\n'
			desc += 'FlowSpeed = '+SolarWindHe_FlowSpeed
		END

		IF type EQ 'FMI_HYB' OR type EQ 'FMI_GUMICS' THEN BEGIN
			desc += 'Upstream Plasma Properties:\n'
			desc += 'IMF ('+IMF_Units+') ['+IMF_FieldValue[0]+','+IMF_FieldValue[1]+','+IMF_FieldValue[2]+']\n'
			IF CenterDipoleField_FieldModel NE '' THEN desc += 'Center Dipole Field ('+CenterDipoleField_Units+') ['+CenterDipoleField_FieldValue+'] '+CenterDipoleField_FieldModel +'\n'
			desc += '\n'
			desc += 'Solar Wind H+:\n'
			desc += 'Density = '+SolarWindHp_Density+'\n'
			desc += 'Temperature = '+SolarWindHp_Temperature+'\n'
			desc += 'FlowSpeed = '+SolarWindHp_FlowSpeed
		END

		IF type EQ 'SINP' THEN BEGIN
			desc += 'Start date: '+StartDate+'\n'
			desc += 'Stop date: '+StopDate+'\n'
			desc += '\n'
			desc += 'Upstream Plasma Properties:\n'
			desc += 'IMF ('+IMF_Units+') ['+IMF_FieldValue[0]+','+IMF_FieldValue[1]+','+IMF_FieldValue[2]+']\n'
			desc += '\n'
			desc += 'SW p:\n'
			desc += 'Density = '+SolarWindp_Density+'\n'
			desc += 'Temperature = '+SolarWindp_Temperature+'\n'
			desc += 'FlowSpeed = '+SolarWindp_FlowSpeed+'\n'
			desc += '\n'
			desc += 'SW e-:\n'
			desc += 'Density = '+SolarWinde_Density+'\n'
			desc += 'Temperature = '+SolarWinde_Temperature+'\n'
			desc += 'FlowSpeed = '+SolarWinde_FlowSpeed
		END

		infos[nblignes].ResourceID		= ResourceID
		infos[nblignes].MeasurementType 	= MeasurementType
		infos[nblignes].SimulationProduct	= SimulationProduct
		infos[nblignes].InputResourceID		= InputResourceID
		infos[nblignes].ModelID			= ModelID
		infos[nblignes].ResourceName		= ResourceName
		infos[nblignes].SimulatedRegion		= SimulatedRegion
		infos[nblignes].SimulationRun		= SimulationRun
		infos[nblignes].desc			= desc
		infos[nblignes].StartDate		= StartDate
		infos[nblignes].StopDate		= StopDate
		infos[nblignes].CoordinateSystemName	= CoordinateSystemName
		infos[nblignes].Units			= Units

		infos[nblignes].trie1			= ModelID

		IF type EQ 'FMI_GUMICS' THEN BEGIN
			tmp1 = dirname(dirname(InputResourceID))+'/'
			tmp2 = STRMID(InputResourceID,STRLEN(tmp1))
			tmp3 = STRSPLIT (tmp2,'_',/PRESERVE_NULL,/EXTRACT)
			infos[nblignes].trie1 = ModelID
			IF N_ELEMENTS(tmp3) LT 10 THEN BEGIN
				infos[nblignes].trie2 = tmp1 + STRJOIN(tmp3,'_')
				infos[nblignes].trie3 = 'Density = ???'
				infos[nblignes].trie4 = 'Temperature = ???'
				infos[nblignes].trie5 = 'FlowSpeed = ???'
				infos[nblignes].trie6 = ''
			END ELSE BEGIN
				infos[nblignes].trie2 = tmp1 + STRJOIN(tmp3[0:9],'_')
				infos[nblignes].trie3 = 'Density = ' + tmp3[10] + '*1000000'
				infos[nblignes].trie4 = 'Temperature = ' + tmp3[11] + '*1000'
				infos[nblignes].trie5 = 'FlowSpeed = ' + tmp3[12]	+ '*1000'
				infos[nblignes].trie6 = ''
			END
		END ELSE IF type EQ 'FMI_HYB' THEN BEGIN
			infos[nblignes].trie1 = ModelID
			infos[nblignes].trie2 = InputResourceID
			infos[nblignes].trie3 = MeasurementType
			infos[nblignes].trie4 = (STRSPLIT(basename(ResourceID),'_',/EXTRACT))[0]
			infos[nblignes].trie5 = ''
			infos[nblignes].trie6 = ''
		END ELSE IF type EQ 'SINP' THEN BEGIN
			infos[nblignes].trie1 = ModelID
			infos[nblignes].trie2 = InputResourceID
			infos[nblignes].trie3 = MeasurementType
			infos[nblignes].trie4 = ResourceName
			infos[nblignes].trie5 = ''
			infos[nblignes].trie6 = ''
		END ELSE IF type EQ 'LATMOS' THEN BEGIN
			infos[nblignes].trie1 = ModelID
			infos[nblignes].trie2 = SimulationRun
			IF chaine_contient (ResourceName,'/3D') OR ResourceName EQ 'Ion_spectra' THEN BEGIN
				infos[nblignes].trie3 = MeasurementType
				infos[nblignes].trie4 = ResourceName
				infos[nblignes].trie5 = ''
				infos[nblignes].trie6 = ''
			END ELSE IF chaine_contient (ResourceName,'/') THEN BEGIN
				; ex: "Ele/MGS (0)"
				; trier par trie3="MGS" puis par trie4=MeasurementType puis par trie5="Ele" puis par trie6="0"
				tmp1 = STRSPLIT (ResourceName,'/',/EXTRACT)
				tmp2 = STRSPLIT (tmp1[1],/EXTRACT)
				infos[nblignes].trie3 = tmp2[0]
				infos[nblignes].trie4 = MeasurementType
				infos[nblignes].trie5 = tmp1[0]
				infos[nblignes].trie6 = STRMID(tmp2[1],1,STRLEN(tmp2[1])-2)
			END ELSE IF chaine_contient (ResourceName,'@') THEN BEGIN
				; ex: "Planetary O2+ @MEX (270)"
				; trier par trie3="MEX" puis par trie4=MeasurementType puis par trie5="Planetary O2+" puis par trie6="270"
				tmp1 = STRSPLIT (ResourceName,'@',/EXTRACT)
				tmp2 = STRSPLIT (tmp1[1],/EXTRACT)
				infos[nblignes].trie3 = tmp2[0]
				infos[nblignes].trie4 = MeasurementType
				infos[nblignes].trie5 = tmp1[0]
				infos[nblignes].trie6 = STRMID(tmp2[1],1,STRLEN(tmp2[1])-2)
			END ELSE BEGIN
				PRINT,'ERROR IN ResourceName: ' + ResourceName
				EXIT
			END
		END

		nblignes++
		
		PRINT,'analyse_xml: ResourceID        = ' + ResourceID
		PRINT,'analyse_xml: MeasurementType   = ' + MeasurementType
		PRINT,'analyse_xml: SimulationProduct = ' + SimulationProduct
		PRINT,'analyse_xml: InputResourceID   = ' + InputResourceID
		PRINT,'analyse_xml: ModelID           = ' + ModelID
		PRINT,'analyse_xml: ResourceName      = ' + ResourceName
		PRINT,'analyse_xml: SimulatedRegion   = ' + SimulatedRegion
		PRINT,'analyse_xml: SimulationRun     = ' + SimulationRun
		PRINT,'analyse_xml: desc              = ' + desc
		PRINT,'analyse_xml: StartDate         = ' + StartDate
		PRINT,'analyse_xml: StopDate          = ' + StopDate

		debut_Parameter = WHERE (data.noeud EQ 'Parameter')
		fin_Parameter   = WHERE (data.noeud EQ '/Parameter')
		FOR j=0L,N_ELEMENTS(debut_Parameter)-1 DO BEGIN
			tmp = data[debut_Parameter[j]:fin_Parameter[j]]
			ind = WHERE (tmp.noeud EQ 'Name')
			PRINT,'analyse_xml: Name              = '+tmp[ind].valeur
		END


		PRINT
	END

	infos = infos[LINDGEN(nblignes)]

END

;-------------------------------------------------------------------------------
PRO trier_infos,	$
;-------------------------------------------------------------------------------
	infos			; LINT_PROTOTYPE output
;-------------------------------------------------------------------------------
; Trie le tableau de structure infos avant affichage
;-------------------------------------------------------------------------------

	; il faut trier infos:
	;	- d'abord par trie1
	;	- puis par trie2
	;	- puis par trie3
	;	- puis par trie4
	;	- puis par trie5
	;	- puis par trie6

	nb = N_ELEMENTS(infos)
	FOR i1=0L,nb-1 DO BEGIN
		; comparer les autres lments avec i1
		min = i1
		FOR i2=i1+1,nb-1 DO BEGIN
			IF infos[i2].trie1 LT infos[min].trie1 THEN BEGIN
				min = i2
			END ELSE IF infos[i2].trie1 EQ infos[min].trie1 THEN BEGIN
				IF infos[i2].trie2 LT infos[min].trie2 THEN BEGIN
					min = i2
				END ELSE IF infos[i2].trie2 EQ infos[min].trie2 THEN BEGIN
					IF infos[i2].trie3 LT infos[min].trie3 THEN BEGIN
						min = i2
					END ELSE IF infos[i2].trie3 EQ infos[min].trie3 THEN BEGIN
						IF infos[i2].trie4 LT infos[min].trie4 THEN BEGIN
							min = i2
						END ELSE IF infos[i2].trie4 EQ infos[min].trie4 THEN BEGIN
							IF infos[i2].trie5 LT infos[min].trie5 THEN BEGIN
								min = i2
							END ELSE IF infos[i2].trie5 EQ infos[min].trie5 THEN BEGIN
								IF LONG(infos[i2].trie6) LT LONG(infos[min].trie6) THEN BEGIN
									min = i2
								END
							END
						END
					END
				END
			END
		END
		tmp = infos[i1]
		infos[i1] = infos[min]
		infos[min] = tmp
	END

	; Supprimer les doublons
	nbchamps = N_ELEMENTS(TAG_NAMES(infos))
	FOR i=0L,nb-1 DO BEGIN
		FOR j=i+1,nb-1 DO BEGIN
			idem = 1
			FOR k=0L,nbchamps-1 DO BEGIN
				IF infos[i].(k) NE infos[j].(k) THEN BEGIN
					idem = 0
					BREAK
				END
			END
			IF idem THEN BEGIN
				; supprimer j
				PRINT,'supprimer ' + val_to_str(j)
				infos[j].trie1 = ''
			END
		END
	END

	infos = infos[WHERE (infos.trie1 NE '')]

END


;-------------------------------------------------------------------------------
FUNCTION get_texte,				$
;-------------------------------------------------------------------------------
	filename					; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; Retourne une chaine de caractre avec les 30 premires lignes du fichier
;-------------------------------------------------------------------------------

	IF ~fichier_existe(filename) THEN RETURN, 'NO FILE'

	OPENR, fd, filename, /GET_LUN
	texte = REPLICATE ('',30)
	ligne = ''
	FOR i=1L,30 DO BEGIN
		IF EOF(fd) THEN BREAK
		READF, fd, ligne
		texte[i-1] = ligne
	END
	FREE_LUN, fd
	IF i EQ 1 THEN BEGIN
		texte = 'EMPTY FILE'
	END ELSE BEGIN
		texte = htmlspecialchars_encode (texte)
		texte = STRJOIN(texte[WHERE(texte NE '')],'<br>')
	END

	RETURN, texte

END


;-------------------------------------------------------------------------------
PRO ecrire_arbre_impex,				$
;-------------------------------------------------------------------------------
	type,					$	; LINT_PROTOTYPE input
	fd,					$	; LINT_PROTOTYPE input
	Repository_ResourceHeader_ResourceName,	$	; LINT_PROTOTYPE input
	SimulationModels,			$	; LINT_PROTOTYPE input
	infos,					$	; LINT_PROTOTYPE input
	general,				$	; LINT_PROTOTYPE input
	nomhtml,				$	; LINT_PROTOTYPE input
	VOIR_GETFILEURL=VOIR_GETFILEURL			; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; Ecrit dans le fichier fd l'arbre des donnes.
;
; type: 'LATMOS', 'FMI_HYB', 'FMI_GUMICS' ou 'SINP'
;-------------------------------------------------------------------------------

	IF N_ELEMENTS(VOIR_GETFILEURL) EQ 0 THEN VOIR_GETFILEURL = 0

	fichier1_html = '/home/penou/DATA/CLUSTER/SOFT/CLL3/clweb/log/' + nomhtml
	fichier2_html = '/net_clweb/DATA/CLUSTER/SOFT/CLL3/clweb/log/' + nomhtml

	top1html = nbsec()

	OPENW, fdhtml, fichier1_html, /GET_LUN
	PRINTF, fdhtml, '<html>'
	PRINTF, fdhtml, '<head>'
	PRINTF, fdhtml, '</head>'
	PRINTF, fdhtml, '<body>'
	PRINTF, fdhtml, date_to_str(get_date()) + '<br><br>'
	PRINTF, fdhtml, "<TABLE border=1 style='border-collapse: collapse;'>"
	PRINTF, fdhtml, '<TR>'
	PRINTF, fdhtml, '<TH>NO</TH>'
	PRINTF, fdhtml, '<TH>METHOD</TH>'
	PRINTF, fdhtml, '<TH>SIMULATEDREGION</TH>'
	PRINTF, fdhtml, '<TH>SPACECRAFT</TH>'
	PRINTF, fdhtml, '<TH>RESOURCEID</TH>'
	PRINTF, fdhtml, '<TH>DAY,CODE,LINK</TH>'
	PRINTF, fdhtml, '<TH>MESSAGE</TH>'
	PRINTF, fdhtml, '<TR>'


	niveau = 2L
	PRINTF, fd, val_to_str(niveau) + ' ' + type+' '+STRING(4b)+Repository_ResourceHeader_ResourceName

	; il faut une arborescence particulire pour GUMICS
	is_fmi_gumics = type EQ 'FMI_GUMICS'

	; mmorisation de type dans type_org
	type_org = type

	; modification de type: ex: 'FMI_HYB' -> 'fmi'
	tmp = STRSPLIT (type, '_', /EXTRACT)
	type = STRLOWCASE(tmp[0])

	sampling = type EQ 'sinp' ? '600' : '60'

	numero = 0L

	trier_infos, infos

	; Pour FMI_GUMICS
	code_ElectricField = -1
	code_IonComposition = -1
	code_MagneticField = -1
	code_ThermalPlasma = -1
	code_Spectrum = -1

	FOR imethod=0L,VOIR_GETFILEURL DO BEGIN

		; 0: getDataPointValue_Spacecraft
		; 1: getFileURL

		method = imethod EQ 0 ? (type EQ 'sinp' OR type EQ 'lesia' ? 'calculateDataPointValue_Spacecraft' : 'getDataPointValue_Spacecraft') : 'getFileURL'

		IF VOIR_GETFILEURL THEN BEGIN
			niveau = 3L
			PRINTF, fd, val_to_str(niveau) + ' ' + (method EQ 'getFileURL' ? 'Precomputed' : 'Dynamic')
			niveau = 4L
		END ELSE BEGIN 
			niveau = 3L
		END


		old_ModelID		= ''
		old_SimulationRun	= ''
		old_nomsat		= ''
		old_MeasurementType	= ''
		old_ion			= ''
		old_Density		= ''
		old_Temperature		= ''
		old_FlowSpeed		= ''
		old_Solar		= ''

		Solar = ''
		Density = ''
		Temperature = ''
		FlowSpeed = ''

		noinfo1 = 0L
		WHILE 1 DO BEGIN

			; rechercher le plus grand noinfo2 ou les champs trie1 et trie2 sont idems  ceux en noinfo1
			FOR i=noinfo1,N_ELEMENTS(infos)-1 DO BEGIN
				IF infos[i].trie1 EQ infos[noinfo1].trie1 AND infos[i].trie2 EQ infos[noinfo1].trie2 THEN BEGIN
					noinfo2 = i
				END ELSE BEGIN
					BREAK
				END
			END

			; On espre que infos[noinfo1:noinfo2].SimulatedRegion est constant
			SimulatedRegion = infos[noinfo1].SimulatedRegion
			CoordinateSystemName = infos[noinfo1].CoordinateSystemName
			Units = infos[noinfo1].Units

			; ne pas boucler sur les satellites pour getFileURL
			IF method NE 'getFileURL' THEN BEGIN
				infosws = simulatedregion_to_infos (SimulatedRegion,CoordinateSystemName,Units)
				nomsats       = infosws.nomsat
				nomsats_court = infosws.nomsat_court
				nbsats        = N_ELEMENTS(nomsats)
			END ELSE BEGIN
				nbsats = 1
			END

			FOR nosat=0L,nbsats-1 DO BEGIN

				; boucler de noinfo1 a noinfo2

				FOR noinfo=noinfo1,noinfo2 DO BEGIN

					IF method EQ 'getFileURL' THEN BEGIN
						nomsat       = infos[noinfo].trie3
						nomsat_court = nomsat
					END ELSE BEGIN
						nomsat       = nomsats[nosat]
						nomsat_court = nomsats_court[nosat]
					END
					no_mission_available = nomsat EQ 'No mission available in cl'


					sResourceID = infos[noinfo].ResourceID
					IF is_fmi_gumics THEN BEGIN
						tmp = STRSPLIT(sResourceID,'/',/EXTRACT,/PRESERVE_NULL)
						sResourceID_master = STRJOIN(tmp[[0,1,2,3,4,5,6,7,11]],'/')
						; ex: dans "spase://IMPEX/NumericalOutput/FMI/GUMICS/earth/synth_stationary/solarmax/EARTH___n_T_Vx_Bx_By_Bz__15_100_400_3p_03_15m/tilt0/H+_mstate"
						; on ne garde que "spase://IMPEX/NumericalOutput/FMI/GUMICS/earth/synth_stationary/H+_mstate" pour le master
					END
					remplacer, sResourceID, 'spase://', ''
					remplacer, sResourceID, '/', '@'
					IF is_fmi_gumics THEN BEGIN
						remplacer, sResourceID_master, 'spase://', ''
						remplacer, sResourceID_master, '/', '@'
					END ELSE BEGIN
						sResourceID_master = sResourceID
					END
				
					; Pour chaque ResourceID

					IF method EQ 'getDataPointValue_Spacecraft' OR method EQ 'calculateDataPointValue_Spacecraft' THEN BEGIN

						; ne garder que les resourceID '/3D' pour getDataPointValue_Spacecraft
						IF type EQ 'latmos' AND STRPOS(infos[noinfo].resourceID, '/3D') EQ -1 THEN BEGIN
							;ecrire_html, fdhtml, numero, method, SimulatedRegion, CoordinateSystemName, Units, nosat, nomsats, sResourceID, 'WARNING1 NOT MANAGED', '/3D ...DataPointValue_Spacecraft', '', /strike
							CONTINUE
						END

						; ne garder que les resourceID '/OnFly' pour getDataPointValue_Spacecraft
						IF type EQ 'sinp' AND STRPOS(infos[noinfo].resourceID, '/OnFly') EQ -1 THEN BEGIN
							;ecrire_html, fdhtml, numero, method, SimulatedRegion, CoordinateSystemName, Units, nosat, nomsats, sResourceID, 'WARNING1 NOT MANAGED', '/3D ...DataPointValue_Spacecraft', '', /strike
							CONTINUE
						END

						infos[noinfo].StartDate  = infosws[nosat].startdate
						infos[noinfo].StopDate   = infosws[nosat].stopdate
						parameter_id_frame_units = infosws[nosat].parameter_id+'@'+infosws[nosat].frame+'@'+infosws[nosat].units

					END ELSE IF method EQ 'getFileURL' THEN BEGIN

						; ne pas garder les '3D' du LATMOS (on n'obtient jamais de fichier)
						IF type EQ 'latmos' THEN BEGIN
							tmp = STRSPLIT(infos[noinfo].ResourceID,'/',/EXTRACT)
							IF (WHERE(tmp EQ '3D'))[0] NE -1 THEN BEGIN
								;ecrire_html, fdhtml, numero, method, SimulatedRegion, CoordinateSystemName, Units, nosat, nomsats, sResourceID, 'WARNING3 NOT MANAGED', '3D getFileURL', '', /strike
								CONTINUE
							END
						END

					END

					IF nosat EQ 0 THEN numero++

					IF ~no_mission_available AND infos[noinfo].StartDate EQ '???' THEN BEGIN
						day_code_link = '&nbsp;'
						texte = 'UNKNOW_STARTDATE'
						couleur = 'ff7f00'
						ecrire_html, fdhtml, numero, method, SimulatedRegion, CoordinateSystemName, Units, nosat, nomsats, sResourceID, day_code_link, texte, couleur
						CONTINUE
					END

					IF ~no_mission_available THEN BEGIN

						annee = STRMID(infos[noinfo].startdate,0,4)
						mois = STRMID(infos[noinfo].startdate,5,2)
						jour = STRMID(infos[noinfo].startdate,8,2)
						datemin = tu_to_date(annee,mois,jour,0,0,0)
						YYYY = annee
						MM = mois
						DD = jour
						YYYYMMDD = YYYY + MM + DD

						annee = STRMID(infos[noinfo].stopdate,0,4)
						mois = STRMID(infos[noinfo].stopdate,5,2)
						jour = STRMID(infos[noinfo].stopdate,8,2)
						datemax = tu_to_date(annee,mois,jour,0,0,0)

						reel = 0
						IF nosat EQ 0 THEN BEGIN

							; Ne faire le travail que pour le premier satellite de la liste

							; Et en plus pour GUMICS: ne le faire qu'une seule fois par MeasurementType (pas pour tous les resourceID)

							IF is_fmi_gumics THEN BEGIN
								CASE infos[noinfo].MeasurementType OF
									'ElectricField':	reel = code_ElectricField  EQ -1
									'IonComposition':	reel = code_IonComposition EQ -1
									'MagneticField':	reel = code_MagneticField  EQ -1
									'ThermalPlasma':	reel = code_ThermalPlasma  EQ -1
									'Spectrum':		reel = code_Spectrum       EQ -1
								END
							END ELSE BEGIN
								reel = 1
							END

							IF reel THEN BEGIN

								IF method EQ 'getDataPointValue_Spacecraft' OR method EQ 'calculateDataPointValue_Spacecraft' THEN BEGIN

									filename = 'http://'+type+'/'+method+'/'+sResourceID+'/'+parameter_id_frame_units+'/'+sampling+'/'
									IF type EQ 'latmos' THEN filename += '0.0/'

								END ELSE IF method EQ 'getFileURL' THEN BEGIN

									filename = 'http://'+type+'/'+method+'/'+sResourceID+'/'
	
								END
								filename += YYYY+'/'+MM+'/data_'+YYYYMMDD+'_V01.xml'

								localdir = dirname(get(general,'downloads_dir')+STRMID(filename,7))+'/'
								mkdir_p,localdir,/o_write

								link_html = filename
								traiter_debut_http_special,link_html,localdir,localname,links2,datelinks2,taillelinks2,read_timeout,tmpinfos

								file_http_copy, filename, localdir, localname=localname, error=error
								IF fichier_existe(localname) AND STRMID(localname,STRLEN(localname)-4) EQ '.tmp' THEN BEGIN
									; renommer le fichier ".tmp"
									code = renommer (localname, STRMID(localname,0,STRLEN(localname)-4))
								END
								IF STRMID(localname,STRLEN(localname)-4) EQ '.tmp' THEN BEGIN
									localname = STRMID(localname,0,STRLEN(localname)-4)
								END
								PRINT,filename+' error='+val_to_str(error)

							END

						END

						master = '/home/penou/DATA/CLUSTER/SOFT/CLL3/resource/master/impex/'+type+'/'+sResourceID_master+'.cef.gz'

						IF reel THEN BEGIN

							; Ne faire le travail que si rel vaut 1
							top1 = nbsec()
							code = analyse_impex(localname,res,master=master)
							top2 = nbsec()
							PRINT, localname+' '+val_to_str_2decimales(top2-top1)

							texte = get_texte (localname)

							IF method EQ 'getDataPointValue_Spacecraft' OR method EQ 'calculateDataPointValue_Spacecraft' THEN BEGIN

								HELP,infos[noinfo].MeasurementType

								CASE infos[noinfo].MeasurementType OF
									'ElectricField': 	BEGIN
													code_ElectricField = code
													IF code NE 0 THEN res_ElectricField = res
												END
									'IonComposition':	BEGIN
													code_IonComposition = code
													IF code NE 0 THEN res_IonComposition = res
												END
									'MagneticField':	BEGIN
													code_MagneticField = code
													IF code NE 0 THEN res_MagneticField = res
												END
									'ThermalPlasma':	BEGIN
													code_ThermalPlasma = code
													IF code NE 0 THEN res_ThermalPlasma = res
												END
									'Spectrum':	BEGIN
													code_Spectrum = code
													IF code NE 0 THEN res_Spectrum = res
												END
								END
	
							END

						END ELSE BEGIN

							; Sinon utiliser le resultat issu du premier satellite

							CASE infos[noinfo].MeasurementType OF
								'ElectricField': 	BEGIN
												code = code_ElectricField
												IF code NE 0 THEN res = res_ElectricField
											END
								'IonComposition':	BEGIN
												code = code_IonComposition
												IF code NE 0 THEN res = res_IonComposition
											END
								'MagneticField':	BEGIN
												code = code_MagneticField
												IF code NE 0 THEN res = res_MagneticField
											END
								'ThermalPlasma':	BEGIN
												code = code_ThermalPlasma
												IF code NE 0 THEN res = res_ThermalPlasma
											END
								'Spectrum':	BEGIN
												code = code_Spectrum
												IF code NE 0 THEN res = res_Spectrum
											END
							END

						END

					END

					msg = ''

					nbinfos = 0

					IF no_mission_available THEN BEGIN

						day_code_link = '&nbsp;'
						texte = 'NO_MISSION_AVAILABLE'
						couleur = 'ff7f00'
						ecrire_html, fdhtml, numero, method, SimulatedRegion, CoordinateSystemName, Units, nosat, nomsats, sResourceID, day_code_link, texte, couleur

					END ELSE IF code EQ 0 THEN BEGIN

						msg = 'ERROR '+localname
						PRINT,'  analyse_impex '+msg
						day_code_link = YYYY+'-'+MM+'-'+DD+'<br>ERROR1 CANNOT_READ<br>'+'<a href="'+link_html+'" target="_blank">link</a>'
						couleur = 'ff0000'
						IF reel THEN ecrire_html, fdhtml, numero, method, SimulatedRegion, CoordinateSystemName, Units, nosat, nomsats, sResourceID, day_code_link, texte, couleur

					END ELSE BEGIN

						; Y a t-il un champ 'ISO_TIME' ?
						ind = WHERE (res.value_type EQ 'ISO_TIME')
						IF ind[0] EQ -1 THEN BEGIN
							msg = 'NO ISO_TIME '+localname
							PRINT,'  analyse_impex '+msg
							day_code_link = YYYY+'-'+MM+'-'+DD+'<br>ERROR2 NO_ISO_TIME<br>'+'<a href="'+link_html+'" target="_blank">link</a>'
							IF reel THEN ecrire_html, fdhtml, numero, method, SimulatedRegion, CoordinateSystemName, Units, nosat, nomsats, sResourceID, day_code_link, texte, 'ff0000'
						END

						texte = STRJOIN(res.name+'['+res.sizes+']','<br>')
						sinfo = REPLICATE ({ value_type:'', name:'', name1:'', longname:'', date:'', data: '', nbd:0, nbdims:0, sizes:'', type:'', libelle:'', units:'', var_type:'' },N_ELEMENTS(res))
						FOR i=0L,N_ELEMENTS(res)-1 DO BEGIN
							IF res[i].VALUE_TYPE EQ 'ISO_TIME' THEN CONTINUE
							IF STRUPCASE(STRMID(res[i].name,0,8)) EQ 'TIME_PB5' THEN CONTINUE
							IF res[i].DATA NE '' THEN CONTINUE

							;IF res[i].value_type EQ 'FLOAT' THEN CONTINUE
							;IF res[i].value_type EQ 'INT' THEN CONTINUE
							;IF res[i].value_type EQ 'DOUBLE' THEN CONTINUE

							sinfo[nbinfos].value_type = res[i].value_type
							sinfo[nbinfos].name = res[i].name
							sinfo[nbinfos].name1 = res[i].name1
							sinfo[nbinfos].nbd = res[i].nbd
							sinfo[nbinfos].nbdims = res[i].nbdims
							sinfo[nbinfos].sizes = res[i].sizes
							sinfo[nbinfos].data = res[i].name
							sinfo[nbinfos].date = res[i].date
							sinfo[nbinfos].type = res[i].type
							sinfo[nbinfos].libelle = res[i].fieldnam
							sinfo[nbinfos].units = res[i].units
							sinfo[nbinfos].var_type = res[i].var_type
							nbinfos++
						END

						IF msg EQ '' AND nbinfos EQ 0 THEN BEGIN
							msg = 'NO FIELDS '+localname
							PRINT,'  analyse_impex '+msg
							day_code_link = YYYY+'-'+MM+'-'+DD+'<br>ERROR3 NO_FIELDS<br>'+'<a href="'+link_html+'" target="_blank">link</a>'
							IF reel THEN ecrire_html, fdhtml, numero, method, SimulatedRegion, CoordinateSystemName, Units, nosat, nomsats, sResourceID, day_code_link, texte, 'ff0000'
						END ELSE BEGIN
							day_code_link = YYYY+'-'+MM+'-'+DD+'<br>OK<br>'+'<a href="'+link_html+'" target="_blank">link</a>'
							IF reel THEN ecrire_html, fdhtml, numero, method, SimulatedRegion, CoordinateSystemName, Units, nosat, nomsats, sResourceID, day_code_link, texte, ''
						END

					END

					MeasurementType		= infos[noinfo].MeasurementType
					ResourceID		= infos[noinfo].ResourceID
					InputResourceID		= infos[noinfo].InputResourceID
					ModelID			= infos[noinfo].ModelID
					ResourceName		= infos[noinfo].ResourceName
					SimulatedRegion		= infos[noinfo].SimulatedRegion
					SimulationRun		= infos[noinfo].SimulationRun
					StartDate		= infos[noinfo].StartDate
					StopDate		= infos[noinfo].StopDate
					desc			= infos[noinfo].desc

					ion = ''
					IF type EQ 'latmos' THEN BEGIN
						IF method EQ 'getFileURL' THEN BEGIN
							ion = infos[noinfo].trie5
						END ELSE BEGIN
							ion = ResourceName
						END
					END ELSE IF type EQ 'fmi' THEN BEGIN
						IF MeasurementType EQ 'IonComposition' THEN BEGIN
							tmp = basename(ResourceID)
							tmp = STRSPLIT (tmp, '_', /EXTRACT)
							ion = tmp[0]
						END
					END

					IF type EQ 'fmi' OR type EQ 'sinp' THEN BEGIN
						; Pour le FMI je veux afficher le nom complet InputResourceID
						SimulationRun = InputResourceID
					END

					IF is_fmi_gumics THEN BEGIN
						tmp1 = dirname(dirname(SimulationRun))+'/'
						tmp2 = STRMID(SimulationRun,STRLEN(tmp1))
						tmp3 = STRSPLIT (tmp2,'_',/PRESERVE_NULL,/EXTRACT)

						IF N_ELEMENTS(tmp3) LT 10 THEN BEGIN
							Solar = tmp1 + STRJOIN(tmp3,'_')
							Density = 'Density = ???'
							Temperature = 'Temperature = ???'
							FlowSpeed = 'FlowSpeed = ???'
							SimulationRun = '???'
						END ELSE BEGIN
							Solar = tmp1 + STRJOIN(tmp3[0:9],'_')
							Density = 'Density = ' + tmp3[10] + '*1000000'
							Temperature = 'Temperature = ' + tmp3[11] + '*1000'
							FlowSpeed = 'FlowSpeed = ' + tmp3[12]	+ '*1000'
							SimulationRun = STRJOIN(tmp3[10:*],'_')
						END
					END

					t1t2 = " {"+STRMID(StartDate,8,2)+'-'+STRMID(StartDate,5,2)+'-'+STRMID(StartDate,0,4)+' '+STRMID(StopDate,8,2)+'-'+STRMID(StopDate,5,2)+'-'+STRMID(StopDate,0,4)+"}"
					IF method EQ 'getFileURL' THEN BEGIN
						tmp = ''
					END ELSE BEGIN
						tmp = t1t2
					END

					IF is_fmi_gumics THEN BEGIN
						niveau_model		= niveau
						niveau_solar 		= niveau+1
						niveau_nomsat		= niveau+2
						niveau_density 		= niveau+3
						niveau_temperature	= niveau+4
						niveau_flowspeed 	= niveau+5
						niveau_simulationrun	= niveau+6
						niveau_measurementtype	= niveau+7
						niveau_ion		= niveau+8
						niveau_max		= niveau + (MeasurementType EQ 'IonComposition' ? 9 : 8)
					END ELSE BEGIN
						niveau_solar 		= 0
						niveau_density 		= 0
						niveau_flowspeed 	= 0
						niveau_model		= niveau
						niveau_simulationrun	= niveau+1
						niveau_nomsat		= niveau+2
						niveau_measurementtype	= niveau+3
						niveau_ion		= niveau+4
						niveau_max		= niveau + (MeasurementType EQ 'IonComposition' ? 5 : 4)
					END

					IF ModelID NE old_ModelID THEN BEGIN
						nomodel = (WHERE (SimulationModels.txt EQ ModelID))[0]
						IF nomodel EQ -1 THEN nomodel = 0
						PRINTF, fd, val_to_str(niveau_model) + ' ' + SimulationModels[nomodel].txt+' '+STRING(4b)+SimulationModels[nomodel].desc
						IF is_fmi_gumics THEN old_Solar = '' ELSE old_SimulationRun = ''
					END

					IF Solar NE old_Solar THEN BEGIN
						PRINTF, fd, val_to_str(niveau_solar) + ' ' + Solar
						old_nomsat = ''
					END

					IF is_fmi_gumics EQ 0 THEN BEGIN
						IF SimulationRun NE old_SimulationRun THEN BEGIN
							PRINTF, fd, val_to_str(niveau_simulationrun) + ' ' + SimulationRun +' '+STRING(4b)+desc
							old_nomsat = ''
						END

						IF nomsat NE old_nomsat THEN BEGIN
							IF no_mission_available THEN BEGIN
								PRINTF, fd, val_to_str(0) + ' ' + nomsat
							END ELSE BEGIN
								PRINTF, fd, val_to_str(niveau_nomsat) + ' ' + nomsat + tmp + ' '+STRING(4b)+'SimulatedRegion: ' + SimulatedRegion
							END
							old_MeasurementType = ''
						END
					END ELSE BEGIN
						IF nomsat NE old_nomsat THEN BEGIN
							IF no_mission_available THEN BEGIN
								PRINTF, fd, val_to_str(0) + ' ' + nomsat
							END ELSE BEGIN
								PRINTF, fd, val_to_str(niveau_nomsat) + ' ' + nomsat + tmp + ' '+STRING(4b)+'SimulatedRegion: ' + SimulatedRegion
							END
							old_Density = ''
						END
					END

					IF Density NE old_Density THEN BEGIN
						IF ~no_mission_available THEN PRINTF, fd, val_to_str(niveau_density) + ' ' + Density
						old_Temperature = ''
					END

					IF Temperature NE old_Temperature THEN BEGIN
						IF ~no_mission_available THEN PRINTF, fd, val_to_str(niveau_temperature) + ' ' + Temperature
						old_FlowSpeed = ''
					END

					IF FlowSpeed NE old_FlowSpeed THEN BEGIN
						IF ~no_mission_available THEN PRINTF, fd, val_to_str(niveau_flowspeed) + ' ' + FlowSpeed
						old_SimulationRun = ''
					END

					IF is_fmi_gumics EQ 1 THEN BEGIN
						IF SimulationRun NE old_SimulationRun THEN BEGIN
							IF ~no_mission_available THEN PRINTF, fd, val_to_str(niveau_simulationrun) + ' ' + SimulationRun +' '+STRING(4b)+desc
							old_MeasurementType = ''
						END

					END

					IF MeasurementType NE old_MeasurementType THEN BEGIN
						IF ~no_mission_available THEN PRINTF, fd, val_to_str(niveau_measurementtype) + ' ' + MeasurementType
						IF MeasurementType EQ 'IonComposition' THEN BEGIN
							old_ion = ''
						END
					END

					IF ion NE old_ion THEN BEGIN
						IF MeasurementType EQ 'IonComposition' THEN BEGIN
							IF ~no_mission_available THEN PRINTF, fd, val_to_str(niveau_ion) + ' ' + ion
						END
					END



					FOR IMFClockAngle=0L,270,90 DO BEGIN

						IF type EQ 'latmos' THEN BEGIN
							IF method EQ 'getFileURL' THEN BEGIN
								IF IMFClockAngle NE 0 THEN CONTINUE
								PRINTF, fd, val_to_str(niveau_max) + ' ' + infos[noinfo].trie6 + ' ' + t1t2 + ' ' + STRING(4b) + 'IMFClockAngle'
							END ELSE BEGIN
								tmp = val_to_str_1decimale(IMFClockAngle)
								IF ~no_mission_available THEN PRINTF, fd, val_to_str(niveau_max) + ' ' + tmp + ' '+ STRING(4b) + 'IMFClockAngle'
							END
						END ELSE BEGIN
							; pas besoin de IMFClockAngle
							IF IMFClockAngle NE 0 THEN CONTINUE
						END


						old_ModelID		= ModelID
						old_SimulationRun	= SimulationRun
						old_nomsat		= nomsat
						old_MeasurementType	= MeasurementType
						old_ion			= ion
						old_Solar		= Solar
						old_Density		= Density
						old_Temperature		= Temperature
						old_FlowSpeed		= FlowSpeed

						IF method EQ 'getDataPointValue_Spacecraft' OR method EQ 'calculateDataPointValue_Spacecraft' THEN BEGIN
							filename = 'http://'+type+'/'+method+'/'+sResourceID+'/'+parameter_id_frame_units+'/'+sampling+'/'
							IF type EQ 'latmos' THEN filename += val_to_str_1decimale(IMFClockAngle)+'/'
						END ELSE BEGIN
							filename = 'http://'+type+'/'+method+'/'+sResourceID+'/'
						END
						filename += '$YEAR/$MONTH/data_$YYYYMMDD_V01.xml'
						master = '$RESOURCE/master/impex/'+type+'/'+sResourceID_master+'.cef.gz'

						IF msg NE '' THEN BEGIN
							IF ~no_mission_available THEN PRINTF, fd, val_to_str(0)+' '+msg
							CONTINUE
						END

						IF nbinfos NE 0 THEN BEGIN

							tab_timeseries = REPLICATE('',nbinfos)
							tab_spectros = REPLICATE('',nbinfos)
							tab_counts = REPLICATE('',nbinfos)

							FOR i=0L,nbinfos-1 DO BEGIN

								s = ""
								s1 = ""
								IF sinfo[i].nbd LE 1 AND sinfo[i].nbdims LE 1 THEN BEGIN

									s = "header legend_theta=No legend_phi=No nom_sat=" + nomsat_court + " nom_experiment=" + type_org
									s += ' selection_date='+sinfo[i].date
									IF sinfo[i].nbd NE 0 THEN BEGIN
										s += ' selection_data='+sinfo[i].data
									END
									s += " filename="+filename+'|'+master
									s += ' datemin='+val_to_str(datemin)+' datemax='+val_to_str(datemax)
									IF sinfo[i].sizes GT 1 THEN BEGIN
										s1 = '('+val_to_str(sinfo[i].sizes)+')'
									END ELSE BEGIN
										s1 = '(1)'
									END 
									value_type = ''
									IF sinfo[i].value_type EQ 'FLOAT' THEN BEGIN
									END ELSE IF sinfo[i].value_type EQ 'INT' THEN BEGIN
									END ELSE IF sinfo[i].value_type EQ 'DOUBLE' THEN BEGIN
									END ELSE BEGIN
										value_type = sinfo[i].value_type+' '
									END 
									s2 = ''
									s2 += ' ('+sinfo[i].units+')'
									timeseries = supprimer_caractere(val_to_str(0)+" "+value_type+s1+' '+sinfo[i].name+s2,'#')+"#"+supprimer_caractere("ASCII P-1 SC1 "+sinfo[i].name1+" "+s,'#')
									tab_timeseries[i] = timeseries

								END

								IF sinfo[i].nbd GE 1 THEN BEGIN

									s = "count3d type="+sinfo[i].type+" yleft_auto=Manual yleft_log=Log yleft_min=1 yleft_max=40000 cjf=count z_showdir=No z_theta=No z_phi=No z_energy=No nom_sat="+nomsat+" nom_experiment="+type_org
									s += ' selection_data='+sinfo[i].data
									s += ' selection_date='+sinfo[i].date
									s += " filename="+filename+'|'+master
									s += ' datemin='+val_to_str(datemin)+' datemax='+val_to_str(datemax)
									s1 = '('+sinfo[i].sizes+')'
									s2 = sinfo[i].longname EQ '' ? ('') : (' ('+sinfo[i].longname+')')
									s2 += ' ('+sinfo[i].libelle+')'
									spectro = supprimer_caractere(val_to_str(0)+" "+s1+' '+sinfo[i].name+s2,'#')+'#'+supprimer_caractere("ASCII P-1 SC1 "+sinfo[i].name+" "+s,'#')
									tab_spectros[i] = spectro

									s = "count2d type=time_cjf cjf=count yleft_typetitle=Wording legend_theta=No legend_phi=No legend_mass=No nom_sat="+nomsat+" nom_experiment="+type_org
									s += ' selection_data='+sinfo[i].data
									s += ' selection_date='+sinfo[i].date
									s += " filename="+filename+'|'+master
									s += ' datemin='+val_to_str(datemin)+' datemax='+val_to_str(datemax)
									s1 = '('+sinfo[i].sizes+')'
									s2 = sinfo[i].longname EQ '' ? ('') : (' ('+sinfo[i].longname+')')
									s2 += ' ('+sinfo[i].libelle+')'
									count = supprimer_caractere(val_to_str(0)+" "+s1+' '+sinfo[i].name+s2,'#')+'#'+supprimer_caractere("ASCII P-1 SC1 "+sinfo[i].name+" "+s,'#')
									tab_counts[i] = count

								END

							END
		
							ind_timeseries = WHERE (tab_timeseries NE '', nb_timeseries)
							ind_spectros = WHERE (tab_spectros NE '', nb_spectros)
							ind_counts = WHERE (tab_counts NE '', nb_counts)

							IF nb_timeseries NE 0 THEN BEGIN
								IF nb_spectros NE 0 OR nb_counts NE 0 THEN PRINTF, fd, val_to_str(niveau_max+1) + " Timeseries"
								FOR i=0L,N_ELEMENTS(ind_timeseries)-1 DO BEGIN
									IF ~no_mission_available THEN PRINTF, fd, tab_timeseries[ind_timeseries[i]]
								END
							END
							IF nb_spectros NE 0 THEN BEGIN
								PRINTF, fd, val_to_str(niveau_max+1) + " Particle spectrogram"
								FOR i=0L,N_ELEMENTS(ind_spectros)-1 DO BEGIN
									IF ~no_mission_available THEN PRINTF, fd, tab_spectros[ind_spectros[i]]
								END
							END
							IF nb_counts NE 0 THEN BEGIN
								PRINTF, fd, val_to_str(niveau_max+1) + " Particle plot"
								FOR i=0L,N_ELEMENTS(ind_spectros)-1 DO BEGIN
									IF ~no_mission_available THEN PRINTF, fd, tab_counts[ind_counts[i]]
								END
							END

						END

					END

				END

			END

			noinfo1 = noinfo2 + 1

			IF noinfo1 GE N_ELEMENTS(infos) THEN BREAK

		END

	END

	PRINTF, fdhtml, '</TABLE>'
	PRINTF, fdhtml, '<br>'

	top2html = nbsec()

	PRINTF, fdhtml, '<br><br> in ' + duree_to_str(1000d*(top2html-top1html))

	PRINTF, fdhtml, '</body>'
	PRINTF, fdhtml, '</html>'
	FREE_LUN, fdhtml

	code = copier (fichier1_html, fichier2_html)

END

;-------------------------------------------------------------------------------
PRO creer_arbre_impex, 	$
;-------------------------------------------------------------------------------
	fd, 		$	; LINT_PROTOTYPE input
	nomxml,		$	; LINT_PROTOTYPE input
	nomhtml,	$	; LINT_PROTOTYPE input
	tree=tree		; LINT_PROTOTYPE input
;-------------------------------------------------------------------------------
; Rajoute dans le fichier ouvert fd les donnes trouves dans le fichier nomxml. Cre le fichier html.
;-------------------------------------------------------------------------------

	COMMON COMMON_CL, general

	NIVEAU_MAX = 5

	code = xml_to_noeud_variable_valeur (nomxml,dataxml)
	IF code EQ 0 THEN RETURN

	IF KEYWORD_SET(tree) THEN BEGIN

		nb = N_ELEMENTS(dataxml)
		branches = REPLICATE ('',nb)

		niveau = 2L
		PRINTF, fd, val_to_str(niveau) + ' ' + nomxml + ' (tree only, no plot available)'

		; 1ere passe pour dterminer le nombre de caractres dans les feuilles
		nbcar = 0L
		FOR i=0L,nb-1 DO BEGIN

			IF STRMID(dataxml[i].noeud,0,1) NE '/' THEN BEGIN
				is_branche = dataxml[i].valeur EQ ''
				IF is_branche THEN BEGIN
					niveau++
					branches[niveau] = dataxml[i].noeud
				END ELSE IF niveau GE NIVEAU_MAX THEN BEGIN
					s = ''
					FOR j=NIVEAU_MAX+1,niveau DO s += '[' + branches[j] + ']'
					s += '[' + dataxml[i].noeud + ']'
					IF dataxml[i].variable NE '' THEN s += ' ' + dataxml[i].variable + ' ='
					nbcar = MAX ([nbcar,STRLEN(s)])
				END
			END ELSE IF dataxml[i].noeud EQ '/'+branches[niveau] THEN BEGIN
				niveau--
			END

		END

		niveau = 2L
		; 2nd passe
		FOR i=0L,nb-1 DO BEGIN

			IF STRMID(dataxml[i].noeud,0,1) NE '/' THEN BEGIN
				is_branche = dataxml[i].valeur EQ ''
				IF is_branche THEN BEGIN
					niveau++
					branches[niveau] = dataxml[i].noeud
					IF niveau LE NIVEAU_MAX THEN PRINTF, fd, val_to_str(niveau) + ' ' + dataxml[i].noeud
				END ELSE IF niveau GE NIVEAU_MAX THEN BEGIN
					s = ''
					FOR j=NIVEAU_MAX+1,niveau DO s += '[' + branches[j] + ']'
					s += '[' + dataxml[i].noeud + ']'
					IF dataxml[i].variable NE '' THEN s += ' ' + dataxml[i].variable + ' ='
					FOR k=STRLEN(s),nbcar DO s += ' '

					; Attention aux sauts de ligne (arrive dans FMI_HYB_tree_20121205.xml)
					valeur = dataxml[i].valeur
					remplacer, valeur, STRING(10b),' '
					remplacer, valeur, STRING(13b),''
					remplacer, valeur, STRING(9b),''

					s += ' ' + valeur
					PRINTF, fd, '0 '+s
				END ELSE BEGIN
					PRINTF, fd, val_to_str(niveau+1)+' '+dataxml[i].noeud+' '+dataxml[i].variable
					PRINTF, fd, '0 '+dataxml[i].valeur
				END
			END ELSE IF dataxml[i].noeud EQ '/'+branches[niveau] THEN BEGIN
				niveau--
			END

		END

	; ---------------------------------------------------
	END ELSE IF STRMID(nomxml,0,6) EQ 'LATMOS' THEN BEGIN
	; ---------------------------------------------------

		; Mettre 1 pour voir les plots possibles avec getFileURL
		VOIR_GETFILEURL = 1

		type = 'LATMOS'
		analyse_xml, type, dataxml, Repository_ResourceHeader_ResourceName, SimulationModels, infos, VOIR_GETFILEURL=VOIR_GETFILEURL

		ecrire_arbre_impex, type, fd, Repository_ResourceHeader_ResourceName, SimulationModels, infos, general, VOIR_GETFILEURL=VOIR_GETFILEURL, nomhtml

	; ---------------------------------------------------
	END ELSE IF STRMID(nomxml,0,5) EQ 'LESIA' THEN BEGIN
	; ---------------------------------------------------

		type = 'LESIA'
		analyse_xml, type, dataxml, Repository_ResourceHeader_ResourceName, SimulationModels, infos

		ecrire_arbre_impex, type, fd, Repository_ResourceHeader_ResourceName, SimulationModels, infos, general, nomhtml

	; -------------------------------------------------------------------------------------------
	END ELSE IF STRMID(nomxml,0,10) EQ 'FMI_GUMICS' OR STRMID(nomxml,0,7) EQ 'FMI_HYB' THEN BEGIN
	; -------------------------------------------------------------------------------------------

		tmp = STRSPLIT (nomxml, '_', /EXTRACT)
		type = tmp[0] + '_' + tmp[1]
		analyse_xml, type, dataxml, Repository_ResourceHeader_ResourceName, SimulationModels, infos

		ecrire_arbre_impex, type, fd, Repository_ResourceHeader_ResourceName, SimulationModels, infos, general, nomhtml

	; ---------------------------------------------------
	END ELSE IF STRMID(nomxml,0,4) EQ 'SINP' THEN BEGIN
	; ---------------------------------------------------

		type = 'SINP'
		analyse_xml, type, dataxml, Repository_ResourceHeader_ResourceName, SimulationModels, infos

		ecrire_arbre_impex, type, fd, Repository_ResourceHeader_ResourceName, SimulationModels, infos, general, nomhtml

	END

END

;-------------------------------------------------------------------------------
PRO get_arbre_impex
;-------------------------------------------------------------------------------
; Cre le fichier 'resource/arbre_impex.gz' contenant les donnes de l'arbre impex.
;-------------------------------------------------------------------------------


	;lint_unused = get_infos_wsorb ('/home/penou/DATA/CLUSTER/SOFT/CLL3/LocalDataBaseParameters.xml')
	lint_unused = get_infos_wsorb ('/home/penou/DATA/CLUSTER/SOFT/CLL3/resource/xml/clweb_tree.xml')
	print_infos_wsorb

	latmos 		= 1*1
	fmi_gumics 	= 1*1
	fmi_hyb		= 1*1
	sinp 		= 1*1
	lesia		= 1*1

	recuperer_xml	= 1*1

	OPENW, fd, '/home/penou/DATA/CLUSTER/SOFT/CLL3/resource/tree/impex_tree.txt.gz', /GET_LUN, /COMPRESS
	PRINTF, fd, val_to_str(1)+ ' IMPEX'

	IF latmos THEN BEGIN
		LATMOS_xml	= 'LATMOS_tree.xml'
		IF recuperer_xml THEN SPAWN, 'wget "http://impex.latmos.ipsl.fr/tree.xml" -O ' + LATMOS_xml
		; ne pas utiliser SPAWN, 'wget "http://impex.latmos.ipsl.fr/getTree.php" -O ' + LATMOS_xml
		creer_arbre_impex, fd, LATMOS_xml, 'get_arbre_latmos.html'
		;creer_arbre_impex, fd, LATMOS_xml,	/tree
	END

	IF fmi_gumics THEN BEGIN
		FMI_GUMICS_xml	= 'FMI_GUMICS_tree.xml'
		IF recuperer_xml THEN SPAWN, 'wget "http://impex-fp7.fmi.fi/ws/Tree_FMI_GUMICS.xml" -O ' + FMI_GUMICS_xml
		creer_arbre_impex, fd, FMI_GUMICS_xml, 'get_arbre_fmi_gumics.html'
		;creer_arbre_impex, fd, FMI_GUMICS_xml,	/tree
	END


	IF fmi_hyb THEN BEGIN
		FMI_HYB_xml	= 'FMI_HYB_tree.xml'
		IF recuperer_xml THEN SPAWN, 'wget "http://impex-fp7.fmi.fi/ws/Tree_FMI_HYB.xml" -O ' + FMI_HYB_xml
		creer_arbre_impex, fd, FMI_HYB_xml, 'get_arbre_fmi_hyb.html'
		;creer_arbre_impex, fd, FMI_HYB_xml,	/tree
	END

	IF sinp THEN BEGIN
		SINP_xml	= 'SINP_tree.xml'
		IF recuperer_xml THEN SPAWN, 'wget "http://dec1.sinp.msu.ru/~lucymu/paraboloid/SINP_tree.xml" -O ' + SINP_xml
		creer_arbre_impex, fd, SINP_xml, 'get_arbre_sinp.html'
		;creer_arbre_impex, fd, SINP_xml,	/tree
	END

	IF lesia THEN BEGIN
		LESIA_xml	= 'LESIA_tree.xml'
		SPAWN, 'wget "http://typhon.obspm.fr/maser/IMPExWS/tree_Mag.xml" -O ' + LESIA_xml
		creer_arbre_impex, fd, LESIA_xml, 'get_arbre_lesia.html'
		;creer_arbre_impex, fd, LESIA_xml,	/tree
	END

	FREE_LUN, fd

END
