/***************************************************************************************************
 *
 *	Fichier	: $RCSfile: CATALOG.c,v $
 *
 *	Version	: $Revision: 1.6 $
 *
 *	Auteur	: $Author: barthe $
 *
 *	Date	: $Date: 2020/01/15 11:06:09 $
 *
 *	==========================================================================================
 *
 *	Gestion du catalogue de calibrations
 *
 *	ATTENTION	Ce module est partage par les CLL2 et CLL3
 */

#define	MODULE_NAME	"CATALOG"

#include "CISLIB.h"
#include "CATALOG.h"

#ifndef	WIN_CDECL
#define	WIN_CDECL
#endif

typedef struct {
        char		produit [10];
        int		version;
        double		date_deb;
        double		date_fin;
        t_filename 	fichier;
}       t_cat;

static	t_cat *		catalogue = NULL;	/* table decrivant le catalogue		*/

static	t_filename	reference_calib;	/* reference version utilisee		*/

static	int		nbre_entree = 0;	/* nombre d'entrees			*/


/***************************************************************************************************
 *
 *	Fonction de comparaison pour tri du catalogue
 *	---------------------------------------------
 */
int	WIN_CDECL Compare_catalogue (const void * gauche, const void * droite)
{
	t_cat *		g = (t_cat *) gauche;
	t_cat *		d = (t_cat *) droite;
	int		cmp;

	cmp = strcmp (g->produit, d->produit);

	if (cmp != 0) return cmp;

	return (g->date_deb > d->date_deb) ? 1 : -1;
}


/***************************************************************************************************
 *
 *	Tri du catalogue, et mise a jour intervalle de validite
 *	-------------------------------------------------------
 */
t_err	Reorganisation_catalogue (void)
{	
	char *		fonction = FNAME ("Reorganisation_catalogue");
	t_err		error = OK;
	t_cat *		last;
	t_cat *		work;
	int		i;
	double		date_max = Ascii_time_to_milli ("2099-12-31T23:59:59Z");

	qsort (catalogue, nbre_entree, sizeof (t_cat), Compare_catalogue);

	last = & catalogue [0];
	
	for (i = 1; i < nbre_entree; i++) {

		work = & catalogue [i];

		if (strcmp (work->produit, last->produit) == 0)
			last->date_fin = work->date_deb; 
		else
			last->date_fin = date_max;
		last = work;
	}
	last->date_fin = date_max;

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Recherche le plus grand numero de version utilise dans le catalogue
 *	-------------------------------------------------------------------
 */
t_err	Cherche_derniere_version (FILE * entree)
{
	char *		fonction = FNAME ("Cherche_derniere_version");
	int		max = 0;
	char		buffer [200], ver [10];
	int		lus, numero = 0;

	while (Read_line (buffer, entree) == OK) {

		if (buffer [0] == '#' || strlen (buffer) == 0)	continue;		

		lus = sscanf (buffer, "%*s %*s %s %*s", ver);

		if (lus == 1 && ver [0] == 'V') numero = atoi (& ver [1]);

		if (numero > max) max = numero;
	}
	rewind (entree);

	return max;
}


/***************************************************************************************************
 *
 *	Lecture du catalogue de calibration
 *	-----------------------------------
 */
t_err	Lecture_catalogue (char * fichier, int version)
{
	char *		fonction = FNAME ("Lecture_catalogue");
	t_err		error = OK;
	FILE *		entree = NULL;
	char		var [100], deb [100], ver [100], buffer [200];
	t_filename	calib_file;

	if ((entree = fopen (fichier, "r")) == NULL) {

		Affiche_erreur (fonction, "Lecture %s impossible", fichier);
		error = ERR_fopen;
		goto EXIT;
	}

	Affiche_trace (3, fonction, "%s", fichier);

	if (version == 999) version = Cherche_derniere_version (entree);

	sprintf (reference_calib, "%s (V%03d)", fichier, version);

	Affiche_trace (3, fonction, "Version de calibration = V%03d", version);

	while (Read_line (buffer, entree) == OK) {

		if (buffer [0] == '#' || strlen (buffer) == 0)	continue;

		if (sscanf (buffer, "%s %s %s %s", var, deb, ver, calib_file) != 4) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}
		else if (version == atoi (& ver [1])) {

			catalogue = (t_cat *) realloc (catalogue, sizeof (t_cat) * (nbre_entree + 1));

			if (catalogue == NULL) {

				Affiche_erreur (fonction, "Erreur reallocation dynamique");
				error = ERR_malloc;
				goto EXIT;
			}

			strcpy (catalogue [nbre_entree].produit, var);
			strcpy (catalogue [nbre_entree].fichier, calib_file);

			catalogue [nbre_entree].version  = version;
			catalogue [nbre_entree].date_deb = Ascii_time_to_milli (deb);
			catalogue [nbre_entree].date_fin = Ascii_time_to_milli (deb);

			nbre_entree ++;
		}
	}
	
	if (nbre_entree < 1) {

		Affiche_erreur (fonction, "Pas de tables de calibrations pour la version %d", version);
		error = ERROR;
		goto EXIT;
	}

	error = Reorganisation_catalogue ();

EXIT:	if (entree != NULL) fclose (entree);
	return error;
}


/***************************************************************************************************
 *
 *	Affichage du catalogue
 *	----------------------
 */
t_err	Affiche_catalogue (void)
{
	char *		fonction = FNAME ("Affiche_catalogue");
	t_err		error = OK;
	int		i;

	Affiche_trace (3, fonction, "PRODUIT    VER  DEBUT VALIDITE        FICHIER");
	Affiche_trace (3, fonction, "-------    ---  --------------        -------");

	for (i = 0; i < nbre_entree; i++) {

		Affiche_trace (3, fonction, "%-10s %03d  %s  %s", 
			catalogue [i].produit, 
			catalogue [i].version,
			Milli_to_Ascii_time (catalogue [i].date_deb),
			catalogue [i].fichier);
	}
EXIT:	return error;
}
			

/***************************************************************************************************
 *
 *	Fermeture du catalogue
 *	----------------------
 */
t_err	Fermeture_catalogue (void)
{
	char *		fonction = FNAME ("Fermeture_catalogue");
	t_err		error = OK;

	if (nbre_entree > 0) free (catalogue);

	catalogue	= NULL;
	nbre_entree	= 0;

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Cherche entree correspondante dans catalogue
 *	--------------------------------------------
 */
t_err	Cherche_entree_catalogue (char * var, double date)
{
	int		i, trv = -1;

	for (i = 0; i < nbre_entree && trv == -1; i++) {

		if (strcmp (var, catalogue [i].produit) != 0) continue;
		if (date >= catalogue [i].date_deb && date <  catalogue [i].date_fin)	trv = i;
	}
	return trv;	
}


/***************************************************************************************************
 *
 *	Teste si la date sort des limites de validite du fichier
 *	--------------------------------------------------------
 */
t_err	Actualisation_necessaire (int indice, double date)
{
	return date < catalogue [indice].date_deb || date >= catalogue [indice].date_fin;
}		


/***************************************************************************************************
 *
 *	Retourne la date de fin de validite de l'entree
 *	-----------------------------------------------
 */
t_err	Periode_validite (int indice, double * date_deb, double * date_fin)
{
	char *		fonction = FNAME ("Periode_validite");
	t_err		error = OK;

	if (indice < 0 || indice >= nbre_entree) {

		error = ERROR;
		goto EXIT;
	}
	* date_deb = catalogue [indice].date_deb;
	* date_fin = catalogue [indice].date_fin;

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Retourne le nom du fichier correspondant
 *	----------------------------------------
 */
char * 	Fichier_calibration (int indice)
{
	return (char *) & catalogue [indice].fichier;
}	


/***************************************************************************************************
 *
 *	Retource les references de calibration
 *	--------------------------------------
 */
char * 	Reference_calibration (void)
{
	return (char *) & reference_calib;
}	
