/***************************************************************************************************
 *
 *	Fichier	: $RCSfile: CALIB.c,v $
 *
 *	Version	: $Revision: 1.12 $
 *
 *	Auteur	: $Author: barthe $
 *
 *	Date	: $Date: 2020/01/15 15:13:43 $
 *
 *	==========================================================================================
 *
 *	Ce module est charge de la gestion des tables de calibrations
 *
 *	ATTENTION	Ce module est partage par les CLL2 et CLL3
 */

#define	MODULE_NAME	"CALIB"

#include "CISLIB.h"
#include "CATALOG.h"
#include "CALIB.h"

#define	VAR(str) (strcasecmp (var, str) == 0)

/*	Signatures des fonctions de traitement des produits
 *	---------------------------------------------------
 */
t_err	Traite_HSK   (FILE * , char * );
t_err	Traite_MASS  (FILE * , char * );
t_err	Traite_COEFF (FILE * , char * );
t_err	Traite_OMC   (FILE * , char * );
t_err	Traite_CIS1  (FILE * , char * );
t_err	Traite_CIS2  (FILE * , char * );
t_err	Traite_EFF   (FILE * , char * );

typedef struct {					/* Entree table des calibrations	*/
	char		produit [10];			/* - code du produit			*/
	int		indice;				/* - indice dans catalogue		*/
	t_err		(* fonction) (FILE *, char *);	/* - fonction de traitement		*/
}	t_calib;

#define	NB_ENTREE	11				/* Taille de la table des produits	*/

static	t_calib		table [NB_ENTREE] = {		

	{ "HSK",	-1, Traite_HSK   },		/* - coefficients calcul Housekeeping	*/
	{ "MASS",	-1, Traite_MASS  },		/* - ordre des masses des moments CODIF	*/
	{ "OMC1",	-1, Traite_OMC   },		/* - coefficients calcul P07 et P09	*/
	{ "OMC2",	-1, Traite_OMC   },		/* - coefficients calcul P02 et P04	*/
	{ "P02",	-1, Traite_COEFF },		/* - calibrations produit 02 de HIA	*/
	{ "P04",	-1, Traite_COEFF },		/* - calibrations produit 04 de HIA	*/
	{ "P07",	-1, Traite_COEFF },		/* - calibrations produit 07 de CODIF	*/
	{ "P09",	-1, Traite_COEFF },		/* - calibrations produit 09 de CODIF	*/
	{ "CIS1",	-1, Traite_CIS1  },		/* - calibrations diverses de CODIF	*/
	{ "CIS2",	-1, Traite_CIS2  },		/* - calibrations diverses de HIA	*/
	{ "EFF",	-1, Traite_EFF	 }		/* - coefficients efficacite de CODIF	*/
};

static	t_filename	calib_path;			/* Repertoire fichiers de calibration	*/


/*	Coefficients de calibration des moments a bord
 *	----------------------------------------------
 */
int		k_position	[NB_MASSE_MAX];
COEFF		k_densite	[NB_MASSE_MAX][NB_ENERGY_RANGE];
V_COEFF		k_vitesse	[NB_MASSE_MAX][NB_ENERGY_RANGE];
P_COEFF		k_pression	[NB_MASSE_MAX][NB_ENERGY_RANGE];


/*	Coefficients de calibration CODIF : produit OMC1
 *	------------------------------------------------
 */
float		cis1_onboard_moment_coeff_P7	[3];
float		cis1_onboard_moment_coeff_P9	[3];


/*	Coefficients de calibration HIA : produit OMC2
 *	----------------------------------------------
 */
float		cis2_onboard_moment_coeff_P2;
float		cis2_onboard_moment_coeff_P4;


/*	Coefficients de calibration CODIF : produit CIS1
 *	------------------------------------------------
 */
int		cis1_accumulation_spin		[7][64][NB_CIS_MODE];
float		cis1_geom_factor		[NB_SENSITIVITY][NB_ANODE_CIS1][8];
float		cis1_rpa_geom_factor		[NB_SENSITIVITY][NB_ANODE_CIS1][8];
float 		cis1_anal_k_factor		[3];
float		cis1_anal_alpha_angle		[NB_SENSITIVITY];
float		cis1_post_accel_volt;
float		cis1_total_effic_normal		[NB_MASSE_MAX];
float		cis1_dead_times			[2];
float		cis1_anode_eff			[NB_SENSITIVITY][NB_ANODE_CIS1][NB_DEGRE][NB_MASSE_MAX];
COEFF		cis1_dens_corr_coeff_P7		[NB_SENSITIVITY][NB_ENERGY_RANGE];
COEFF		cis1_dens_corr_coeff_P9		[NB_SENSITIVITY][NB_ENERGY_RANGE];
float		cis1_absolute_efficiencies	[NB_SENSITIVITY][NB_MASSE_MAX];
TE_DESCR	cis1_energy_sweep_table_8 	[2][8];
TE_DESCR	cis1_energy_sweep_table_16	[2][16];
TE_DESCR	cis1_energy_sweep_table_31	[2][31];
TE_DESCR	cis1_energy_sweep_table_prom	[32];
TE_DESCR	cis1_energy_sweep_table_rpa 	[16];


/*	Coefficients de calibration HIA : produit CIS2
 *	----------------------------------------------
 */
int		cis2_accumulation_spin		[7][64][NB_CIS_MODE];
float		cis2_geom_factor		[NB_SENSITIVITY];
float		cis2_anal_k_factor		[3];
float		cis2_anal_alpha_angle		[NB_SENSITIVITY];
float		cis2_anode_eff			[NB_SENSITIVITY][NB_ANODE_CIS2][2];
float		cis2_dead_times			[2];
float		cis2_a				[NB_SENSITIVITY];
float		cis2_b				[NB_SENSITIVITY];
float		cis2_norm_e			[NB_SENSITIVITY];
float		cis2_norm_theta			[NB_SENSITIVITY];
float		cis2_MCP_fitting_param		[3][5];
TE_DESCR	cis2_energy_sweep_table_16	[11][16];
TE_DESCR	cis2_energy_sweep_table_31	[11][31];
TE_DESCR	cis2_energy_sweep_table_62	[11][62];

// AB 2016/10/19	Test pour CAA qui veut les table en 124
TE_DESCR	cis2_energy_sweep_table_124	[11][124];
// AB 2016/10/19	Test pour CAA qui veut les table en 124

TE_DESCR	cis2_energy_sweep_table_prom	[32];


/*	Variables locales de calcul
 *	---------------------------
 */
static	float	cis1_param_sw  	[2];
static	float	cis1_param_mag 	[2];
static	float	cis1_param_prom	[2];
static	float	cis1_param_sw_stop;
static	float	cis1_param_rpa 	[3];
static	float	cis2_param_sw_stop;
static	float	cis2_param_prom	[2];


/*	Coefficients dE_E de l'analyseur pour CIS1
 *	------------------------------------------
 */
static	float	CIS1_dE_E = 0.145;


/*	Coefficient dE_E de l'analyseur pour CIS2, par sensitivite
 *	----------------------------------------------------------
 */
static	float	CIS2_dE_E [NB_SENSITIVITY] = { 0.163, 0.168 };


/*	Definition de quelques symboles internes
 *	----------------------------------------
 */
static	t_symbol	Onboard_moment_products [] = {

	{ 2,	"P02",	"HIA P02 moments" 	},
	{ 4,	"P04",	"HIA P04 moments" 	},
	{ 7,	"P07",	"CODIF P07 moments" 	},
	{ 9,	"P09",	"CODIF P09 moments" 	},
	{ EOF,	"???",	"Unknown product"	}
};
	
static	t_symbol	Table_Mass_P07 [] = {

	{ 0,	"H+",		"Hydrogen"	},
	{ 1,	"O+",		"Oxygen"	},
	{ 2,	"He+",		"Helium +"	},
	{ 3,	"He++",		"Helium ++"	},
	{ EOF,	"???",		"Unknowm Mass"	}
};

static	t_symbol	Table_Mass_P09 [] = {

	{ 0,	"H+",		"Hydrogen"	},
	{ 1,	"O+",		"Oxygen"	},
	{ 2,	"He+",		"Helium +"	},
	{ 3,	"He++",		"Helium ++"	},
	{ EOF,	"???",		"Unknowm Mass"	}
};

static	t_symbol	Table_Mass_P02 [] = {

	{ 0,	"H+",		"Hydrogen"	},
	{ EOF,	"???",		"Unknowm Mass"	}
};

static	t_symbol	Table_Mass_P04 [] = {

	{ 0,	"H+",		"Hydrogen"	},
	{ 1,	"He++",		"Helium ++"	},
	{ EOF,	"???",		"Unknowm Mass"	}
};

static	t_symbol	Energy_ranges [] = {

	{ 0,	"E1",	"Energy range 0"	},
	{ 1,	"E2",	"Energy range 1"	},
	{ 2,	"E3",	"Energy range 2"	},
	{ EOF,	"???",	"Unknowm energy range"	}
};

typedef	enum	{ Var_d, Var_vx, Var_vy, Var_vz, Var_pxx, Var_pyy, Var_pzz, Var_pxy, Var_pxz, Var_pyz } t_moment;

static	t_symbol	Moments [] = {

	{ Var_d,	"D",	"Density" 	},
	{ Var_vx,	"Vx",	"Velocity (X)"	},
	{ Var_vy,	"Vy",	"Velocity (Y)"	},
	{ Var_vz,	"Vz",	"Velocity (Z)"	},
	{ Var_pxx,	"Pxx",	"Pressure (XX)"	},
	{ Var_pyy,	"Pyy",	"Pressure (YY)"	},
	{ Var_pzz,	"Pzz",	"Pressure (ZZ)"	},
	{ Var_pxy,	"Pxy",	"Pressure (XY)"	},
	{ Var_pxz,	"Pxz",	"Pressure (XZ)"	},
	{ Var_pyz,	"Pyz",	"Pressure (YZ)"	},
	{ EOF,		"???",	"Unknown moment"}
};



/***************************************************************************************************
 *
 *	Ignore lecture d'une variable tant que son format n'est pas stabilise
 *	---------------------------------------------------------------------
 */
t_err	Ignore_variable (FILE * entree, char * variable)
{
	char *		fonction = FNAME ("Ignore_variable");
	t_err		error = OK;
	char		buffer [200];

	Affiche_trace (1, fonction, "Variable %s ignoree", variable);

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;
	}
	
EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Lecture d'une variable de dimension variable
 *	--------------------------------------------
 */
t_err	Traite_variable (FILE * entree, char * variable, int dimension, float * tab)
{
	char *		fonction = FNAME ("Traite_variable");
	t_err		error = OK;
	int		lus;
	char		buffer [200];
	float		val;

	for (lus = 0; lus < dimension; lus ++) tab [lus] = 0.0;

	lus = 0;

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		if (sscanf (buffer, "%f", & val) != 1) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}
		Affiche_trace (3, fonction, "%s [%d] = %f", variable, lus, val);

		if (lus < dimension) tab [lus] = val;
		lus ++;
	}	

	if (lus != dimension) {

		Affiche_erreur (fonction, "Variable %s : %d valeurs lues au lieu de %d", variable, lus, dimension);
		error = ERROR;
		goto EXIT;
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	P A R T I E   S P E C I F I Q U E
 *
 *	A U   C A L C U L   D E S   P A R A M E T R E S   H O U S E K E E P I N G
 *
 ***************************************************************************************************
 */

typedef	struct {
	char *		libelle;
	COEFF		coeff;
}	t_hsk_param;

#define	MAX_PARAM_HSK	40

static	t_hsk_param 	hsk_param [MAX_PARAM_HSK];

static	int		nb_param_hsk = 0;


/***************************************************************************************************
 *
 *	Lecture des coefficients de calcul des parametres Housekeeping
 *	--------------------------------------------------------------
 */
t_err	Traite_HSK (FILE * entree, char * produit)
{
	char *		fonction = FNAME ("Traite_HSK");
	t_err		error = OK;
	char		buffer [200];
	int		i;

	for (i = 0; i < MAX_PARAM_HSK; i++) {

		hsk_param [i].libelle = NULL;
		hsk_param [i].coeff.a = 0.0;
		hsk_param [i].coeff.b = 0.0;
	}

	while (Read_line (buffer, entree) == OK) {

		char	variable [20];
		float	a, b;
		int	lus;

		if (strlen (buffer) == 0 || buffer [0] == '#') continue;

		Affiche_trace (5, fonction, "%s", buffer);

		lus = sscanf (buffer, "%s %f %f", variable, & a, & b);

		if (lus != 3) {

			Affiche_erreur (fonction, "%s", buffer);
			Affiche_erreur (fonction, "%d valeurs lues au lieu de 3", lus);
			error = ERROR;
			goto EXIT;
		}

		hsk_param [nb_param_hsk].libelle = (char *) strdup (variable);
		hsk_param [nb_param_hsk].coeff.a = a;
		hsk_param [nb_param_hsk].coeff.b = b;

		nb_param_hsk ++;

		if (nb_param_hsk >= MAX_PARAM_HSK) {

			Affiche_erreur (fonction, "Taille table > %d", MAX_PARAM_HSK);
			error = ERROR;
			goto EXIT;
		}
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Recherche des coefficients Housekeeping de calcul d'une variable
 *	----------------------------------------------------------------
 */
t_err	Lire_parametre_HSK (char * variable, COEFF * coeff)
{
	char *		fonction = FNAME ("Lire_parametre_HSK");
	t_err		error = OK;
	int		i;
	int		trv = -1;

	coeff->a = 0.0;
	coeff->b = 0.0;

	if (nb_param_hsk == 0) {

		Affiche_erreur (fonction, "Table des parametres HSK non initialisee");
		error = ERROR;
		goto EXIT;
	}

	for (i = 0; i < nb_param_hsk; i++) {

		if (strcasecmp (variable, hsk_param [i].libelle) == 0) {

			coeff->a = hsk_param [i].coeff.a;
			coeff->b = hsk_param [i].coeff.b;
		}
	}

	if (i > nb_param_hsk) {

		Affiche_erreur (fonction, "Variable %s inconnue", variable);
	}
EXIT:	return error;
}


/***************************************************************************************************
 *
 *	P A R T I E   S P E C I F I Q U E   
 *
 *	A U   C A L C U L   D E S   M O M E N T S   A   B O R D 
 *
 ***************************************************************************************************
 */


/***************************************************************************************************
 *
 *	Lecture de l'ordre de masses des moments CODIF
 *	----------------------------------------------
 */
t_err	Traite_MASS (FILE * entree, char * produit)
{
	char *		fonction = FNAME ("Traite_MASS");
	t_err		error = OK;
	char		buffer [200];
	char		masse [10];
	int		m, pos;
	t_symbol *	symbol;

	for (m = 0; m < NB_MASSE_MAX; m ++) k_position [m] = -1;

	while (Read_line (buffer, entree) == OK) {

		int	indice;

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		Affiche_trace (5, fonction, "%s", buffer);

		if (sscanf (buffer, "%s %d", masse, & pos) != 2) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		if ((indice = Val_from_key (Table_Mass_P07, masse)) == EOF) {

			Affiche_erreur (fonction, "Masse %s incorrecte", masse);
			error = ERROR;
			goto EXIT;
		}

		k_position [indice] = pos;
	}

	for (m = 0; m < NB_MASSE_MAX; m++) {

		Affiche_trace (3, fonction, "Masse %d : %s = %d", 
			m, 
			Key_from_val (Table_Mass_P07, m), 
			k_position [m]);
	}

EXIT:	return error;
}



/***************************************************************************************************
 *
 *	Prise en compte des coefficients ABSEFF pour le calcul des moments
 *	------------------------------------------------------------------
 */
t_err	Prise_en_compte_ABSEFF (char * produit)
{
	char *		fonction = FNAME ("Prise_en_compte_ABSEFF");
	t_err		error = OK;
	int		m, e, sensitivite;
	int		indice [4] = { 0, 3, 2, 1 };
	t_symbol *	table_masse = NULL;

	switch (Val_from_key (Onboard_moment_products, produit)) {

	case 2 :	goto EXIT;

	case 4 :	goto EXIT;

	case 7 :	sensitivite = HS;
			table_masse = Table_Mass_P07;
			break;

	case 9 :	sensitivite = LS;
			table_masse = Table_Mass_P09;
			break;

	default:	Affiche_erreur (fonction, "Produit %s incorrect", produit);
			error = ERROR;
			goto EXIT;
	}

	for (m = 0; m < NB_MASSE_MAX; m ++) {

		float	abs_eff = cis1_absolute_efficiencies [sensitivite][indice [m]];

		Affiche_trace (3, fonction, "absolute_efficiencies [%s][%-4.4s] = %f",
			Key_from_val (cis_sensitivity, sensitivite),
			Key_from_val (table_masse, m),
			abs_eff);

		Affiche_trace (3, fonction, "Avant : k_densite [%-4.4s] = % 1.4e % 1.4e % 1.4e",
			Key_from_val (table_masse, m),
			k_densite [m][0].a,
			k_densite [m][1].a,
			k_densite [m][2].a);

		for (e = 0; e < 3; e++) {

			k_densite  [m][e].a	/= abs_eff;
 
			k_vitesse  [m][e].vx.a	/= abs_eff;
			k_vitesse  [m][e].vy.a	/= abs_eff;
			k_vitesse  [m][e].vz.a	/= abs_eff;

			k_pression [m][e].pxx.a	/= abs_eff;
			k_pression [m][e].pxy.a	/= abs_eff;
			k_pression [m][e].pxz.a	/= abs_eff;
			k_pression [m][e].pyy.a	/= abs_eff;
			k_pression [m][e].pyz.a	/= abs_eff;
			k_pression [m][e].pzz.a	/= abs_eff;
		}

		Affiche_trace (3, fonction, "Apres : k_densite [%-4.4s] = % 1.4e % 1.4e % 1.4e",
			Key_from_val (table_masse, m),
			k_densite [m][0].a,
			k_densite [m][1].a,
			k_densite [m][2].a);
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Lecture des coefficients de correction des moments
 *	--------------------------------------------------
 */
t_err	Traite_COEFF (FILE * entree, char * produit)
{
	char *		fonction = FNAME ("Traite_COEFF");
	t_err		error = OK;
	char		buffer [200], variable [10], masse [10], energie [10];
	float		a, b;
	int		i, nb_coeff;
	float *		tab;
	t_symbol *	table_masse = NULL;

	nb_coeff = NB_MASSE_MAX * NB_ENERGY_RANGE * 2;

	for (tab = (float *) & k_densite , i = 0; i < nb_coeff * 1; i++) tab [i] = 0.0;
	for (tab = (float *) & k_vitesse , i = 0; i < nb_coeff * 3; i++) tab [i] = 0.0;
	for (tab = (float *) & k_pression, i = 0; i < nb_coeff * 6; i++) tab [i] = 0.0;

	switch (Val_from_key (Onboard_moment_products, produit)) {

	case 2 :	table_masse = Table_Mass_P02;
			break;
	case 4 :	table_masse = Table_Mass_P04;
			break;
	case 7 :	table_masse = Table_Mass_P07;
			break;
	case 9 :	table_masse = Table_Mass_P09;
			break;
	default :	Affiche_erreur (fonction, "Produit %s incorrect", produit);
			error = ERROR;
			goto EXIT;
	}

	while (Read_line (buffer, entree) == OK) {

		int	m, e;

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		Affiche_trace (5, fonction, "%s", buffer);

		if (sscanf (buffer, "%s %s %s %f %f", variable, masse, energie, & a, & b) != 5) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		if ((m = Val_from_key (table_masse, masse)) == EOF) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			Affiche_erreur (fonction, "Masse incorrecte : %s", masse);
			error = ERROR;
			goto EXIT;
		}

		if ((e = Val_from_key (Energy_ranges, energie)) == EOF) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			Affiche_erreur (fonction, "Energie incorrecte : %s", energie);
			error = ERROR;
			goto EXIT;
		}

		switch (Val_from_key (Moments, variable)) {

		case Var_d :	k_densite  [m][e].a = a;
				k_densite  [m][e].b = b;
				break;
		case Var_vx :	k_vitesse  [m][e].vx.a = a;
				k_vitesse  [m][e].vx.b = b;
				break;
		case Var_vy :	k_vitesse  [m][e].vy.a = a;
				k_vitesse  [m][e].vy.b = b;
				break;
		case Var_vz :	k_vitesse  [m][e].vz.a = a;
				k_vitesse  [m][e].vz.b = b;
				break;
		case Var_pxx :	k_pression [m][e].pxx.a = a;
				k_pression [m][e].pxx.b = b;
				break;
		case Var_pxy :	k_pression [m][e].pxy.a = a;
				k_pression [m][e].pxy.b = b;
				break;
		case Var_pxz :	k_pression [m][e].pxz.a = a;
				k_pression [m][e].pxz.b = b;
				break;
		case Var_pyy :	k_pression [m][e].pyy.a = a;
				k_pression [m][e].pyy.b = b;
				break;
		case Var_pyz :	k_pression [m][e].pyz.a = a;
				k_pression [m][e].pyz.b = b;
				break;
		case Var_pzz :	k_pression [m][e].pzz.a = a;
				k_pression [m][e].pzz.b = b;
				break;
		default :	Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
				Affiche_erreur (fonction, "Variable = %s", variable);
				error = ERROR;
				goto EXIT;
		}
	}

	error = Prise_en_compte_ABSEFF (produit);

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	P A R T I E   S P E C I F I Q U E
 *
 *	A U X   C O E F F I C I E N T S   D E   C A L C U L   D E S   C A L I B R A T I O N S
 *
 ***************************************************************************************************
 */


/***************************************************************************************************
 *
 *	Lecture des coefficients de calcul des calibrations des moments
 *	---------------------------------------------------------------
 */
t_err	Traite_OMC (FILE * entree, char * produit)
{
	char *		fonction = FNAME ("Traite_OMC");
	t_err		error = OK;
	char		buffer [200], var [30];

	while (Read_line (buffer, entree) == OK) {

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		if (sscanf (buffer, "BEGIN %s", var) != 1) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		if (VAR ("onboard_moment_coeff_P2")) 
			error = Traite_variable (entree, var, 1, (float *) & cis2_onboard_moment_coeff_P2);
		else
		if (VAR ("onboard_moment_coeff_P4"))
			error = Traite_variable (entree, var, 1, (float *) & cis2_onboard_moment_coeff_P4);
		else
		if (VAR ("onboard_moment_coeff_P7"))
			error = Traite_variable (entree, var, 3, (float *) & cis1_onboard_moment_coeff_P7);
		else
		if (VAR ("onboard_moment_coeff_P9"))
			error = Traite_variable (entree, var, 3, (float *) & cis1_onboard_moment_coeff_P9);
		else {
			Affiche_erreur (fonction, "Variable %s inconnue", var);
			error = ERROR;
			goto EXIT;
		}
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	P A R T I E   C O M M U N E   
 *
 *	A U X   C A L I B R A T I O N S   D E S   P R O D U I T S   C I S 1   E T   C I S 2
 *
 ***************************************************************************************************
 */


/***************************************************************************************************
 *
 *	Lecture des duree d'accumulation des produits CODIF ou HIA
 *	----------------------------------------------------------
 */
t_err	Traite_accumulation_spin (FILE * entree, t_instr instr)
{
	char *		fonction = FNAME ("Traite_accumulation_spin");
	t_err		error = OK;
	char		buffer [200], tlm_rate [10];
	int		lus, produit, mode, rate, t [16];

	for (rate = 0; rate < 7; rate ++) {

		for (produit = 0; produit < 64; produit ++) {

			for (mode = 0; mode < NB_CIS_MODE; mode ++) {

				switch (instr) {

				case CIS_1 :	cis1_accumulation_spin [rate][produit][mode] = 0;
						break;

				case CIS_2 :	cis2_accumulation_spin [rate][produit][mode] = 0;
						break;
				}
			}
		}
	}
	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		lus = sscanf (buffer, "%d %s %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
			& produit, tlm_rate, 
			& t[ 0], & t[ 1], & t[ 2], & t[ 3],
			& t[ 4], & t[ 5], & t[ 6], & t[ 7],
			& t[ 8], & t[ 9], & t[10], & t[11],
			& t[12], & t[13], & t[14], & t[15]);

		if (lus != 18) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}
		
		if ((rate = Val_from_key (cis_telemetry_rate, tlm_rate)) == EOF) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		for (mode = 0; mode < NB_CIS_MODE; mode ++) {

			switch (instr) {

			case CIS_1 :	cis1_accumulation_spin [rate][produit][mode] = t [mode];
					break;

			case CIS_2 :	cis2_accumulation_spin [rate][produit][mode] = t [mode];
					break;
			}
		}
		Affiche_trace (5, fonction, "%s", buffer);
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Calcule la valeur moyenne d'une table d'energie entre deux paliers
 *	------------------------------------------------------------------
 */
float	Moyenne (float * table, int inf, int sup)
{
	float		somme = 0.0;
	int		i;

	if (sup <= inf) return table [inf];
 
	for (i = inf; i < sup; i ++) somme += table [i];

	return somme / (sup - inf);
}


/***************************************************************************************************
 *
 *	Calcule la valeur moyenne logarithmique des deux valeurs
 *	--------------------------------------------------------
 */
float	Moyenne_log (float e1, float e2)
{
	return	exp ( (log (e1) + log (e2)) / 2.0);
}


/***************************************************************************************************
 *
 * 	P A R T I E   S P E C I F I Q U E
 *
 *	A U X   C A L I B R A T I O N S   D U   P R O D U I T   C I S 1 
 *
 ***************************************************************************************************
 */


/***************************************************************************************************
 *
 *	Lecture des coefficients de correction des moments
 *	--------------------------------------------------
 */
t_err	Traite_CIS1_dens_corr (FILE * entree, char * variable)
{
	char *		fonction = FNAME ("Traite_CIS1_dens_corr");
	t_err		error = OK;
	char		buffer [200], sensit [20], energie [20];
	float		a, b;
	int		s, e, p;

	p = -1;
	if (strcmp (variable, "dens_corr_coeff_P7") == 0) p = 7;
	if (strcmp (variable, "dens_corr_coeff_P9") == 0) p = 9;

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		if (sscanf (buffer, "%s %s %f %f", sensit, energie, & a, & b) != 4) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		s = Val_from_key (cis_sensitivity, sensit);
		e = Val_from_key (Energy_ranges, energie);

		if (s == EOF || e == EOF) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}
		switch (p) {

		case 7:	cis1_dens_corr_coeff_P7 [s][e].a = a;
			cis1_dens_corr_coeff_P7 [s][e].b = b;
			break;
		case 9:	cis1_dens_corr_coeff_P9 [s][e].a = a;
			cis1_dens_corr_coeff_P9 [s][e].b = b;
			break;
		}
	}
	
EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Lecture des facteurs de geometrie CIS1
 *	--------------------------------------
 */
t_err	Traite_CIS1_geom_factor (FILE * entree, char * variable)
{
	char *		fonction = FNAME ("Traite_CIS1_geom_factor");
	t_err		error = OK;
	char		buffer [200], sensitivite [10];
	int		lus, sensit, anode, energy;
	float		tab [8];
	int		rpa;

	rpa = (strcasecmp (variable, "rpa_geom_factor") == 0) ? TRUE : FALSE;

	for (sensit = 0; sensit < NB_SENSITIVITY; sensit ++) {

		for (anode = 0; anode < NB_ANODE_CIS1; anode ++) {

			for (energy = 0; energy < 8; energy ++) {

				if (rpa == FALSE)
					cis1_geom_factor     [sensit][anode][energy] = 0.0;
				else	cis1_rpa_geom_factor [sensit][anode][energy] = 0.0;
			}
		}
	}

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		lus = sscanf (buffer, "%s PF%d %f %f %f %f %f %f %f %f",
			sensitivite,
			& anode,
			& tab[ 0], & tab[ 1], & tab[ 2], & tab[ 3],
			& tab[ 4], & tab[ 5], & tab[ 6], & tab[ 7]);

		if (lus != 10) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		sensit = Val_from_key (cis_sensitivity, sensitivite);

		switch (sensit) {

		case LS :	if (anode < 10 || anode > 15) {

					Affiche_erreur (fonction, "Anode LS PF%d incorrecte : %d", anode);
					error = ERROR;
					goto EXIT;
				}
				anode -= 9;
				break;

		case HS :	if (anode < 1 || anode > 8) {

					Affiche_erreur (fonction, "Anode HS PF%dincorrecte : %d", anode);
					error = ERROR;
					goto EXIT;
				}
				anode -= 1;
				break;

		default :	Affiche_erreur (fonction, "Sensitivite incorrecte : %s", sensitivite);
				error = ERROR;
				goto EXIT;
		}
		for (energy = 0; energy < 8; energy ++) {

			if (rpa == FALSE) 
				cis1_geom_factor     [sensit][anode][energy] = tab [energy];
			else	cis1_rpa_geom_factor [sensit][anode][energy] = tab [energy];
		}	
	}

EXIT:	return error;
}
  

/***************************************************************************************************
 *
 *	Traite les coefficients d'efficacite des anodes
 *	-----------------------------------------------
 */
t_err	Traite_CIS1_anode_effic (FILE * entree, char * variable)
{
	char *		fonction = FNAME ("Traite_CIS1_anode_effic");
	t_err		error = OK;
	char		buffer [200], sensitivite [10];
	int		lus, sensit, anode, degre, masse;
	float		tab [4];

	for (sensit = 0; sensit < NB_SENSITIVITY; sensit ++) {

		for (anode = 0; anode < NB_ANODE_CIS1; anode ++) {

			for (degre = 0; degre < NB_DEGRE; degre ++) {

				for (masse = 0; masse < NB_MASSE_MAX; masse ++)
		
					cis1_anode_eff [sensit][anode][degre][masse] = 0.0;
			}
		}
	}

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		lus = sscanf (buffer, "%s PF%d M%d %f %f %f %f",
			sensitivite,
			& anode,
			& degre,
			& tab[ 0], & tab[ 1], & tab[ 2], & tab[ 3]);

		if (lus != 7) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		sensit = Val_from_key (cis_sensitivity, sensitivite);

		switch (sensit) {

		case LS :	if (anode < 10 || anode > 15) {

					Affiche_erreur (fonction, "Anode LS PF%d incorrecte : %d", anode);
					error = ERROR;
					goto EXIT;
				}
				anode -= 9;
				break;

		case HS :	if (anode < 1 || anode > 8) {

					Affiche_erreur (fonction, "Anode HS PF%dincorrecte : %d", anode);
					error = ERROR;
					goto EXIT;
				}
				anode -= 1;
				break;

		default :	Affiche_erreur (fonction, "Sensitivite incorrecte : %s", sensitivite);
				error = ERROR;
				goto EXIT;
		}
		if (degre < 0 || degre > 3) {

			Affiche_erreur (fonction, "Coefficient incorrect : M%d", degre);
			error = ERROR;
			goto EXIT;
		}

		for (masse = 0; masse < NB_MASSE_MAX; masse ++) 
			cis1_anode_eff [sensit][anode][degre][masse] = tab [masse];
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Generation des tables d'energies de CIS1
 *	----------------------------------------
 */
t_err	Genere_CIS1_energy_sweep_tables (void)
{
	char *		fonction = FNAME ("Genere_CIS1_energy_sweep_table");
	t_err		error = OK;
	int		i, lower, upper;
	float		fact, tmp [129];

	Affiche_trace (3, fonction, "anal_k_factor = %f", cis1_anal_k_factor [2]);
	Affiche_trace (3, fonction, "v_max sw      = %f", cis1_param_sw   [0]);
	Affiche_trace (3, fonction, "v_max mag     = %f", cis1_param_mag  [0]);
	Affiche_trace (3, fonction, "v_max prom    = %f", cis1_param_prom [0]);
	Affiche_trace (3, fonction, "k sw          = %f", cis1_param_sw   [1]);
	Affiche_trace (3, fonction, "k mag         = %f", cis1_param_mag  [1]);
	Affiche_trace (3, fonction, "k prom        = %f", cis1_param_prom [1]);

	/*	Calcul de la table magnetospheric
	 */
	for (fact = 1.0, i = 0; i < 129; i ++) {

		tmp [i] = cis1_anal_k_factor [2] * cis1_param_mag [0] * fact;
		fact   *= cis1_param_mag [1];
	}
	for (i = 0; i < 31; i++) {

		upper = i * 4;
		lower = i * 4 + 4;
		cis1_energy_sweep_table_31 [0][i].upper   = tmp [upper];
		cis1_energy_sweep_table_31 [0][i].lower   = tmp [lower];
		cis1_energy_sweep_table_31 [0][i].average = Moyenne (tmp, upper, lower);
	}
	for (i = 0; i < 16; i++) {

		upper = i * 8;
		lower = (i < 15) ? i * 8 + 8 : 124;
		cis1_energy_sweep_table_16 [0][i].upper   = tmp [upper];
		cis1_energy_sweep_table_16 [0][i].lower   = tmp [lower];
		cis1_energy_sweep_table_16 [0][i].average = Moyenne (tmp, upper, lower);
	}
	for (i = 0; i < 8; i++) {

		upper = i * 16;
		lower = (i < 7) ? i * 16 + 16 : 124;
		cis1_energy_sweep_table_8 [0][i].upper   = tmp [upper];
		cis1_energy_sweep_table_8 [0][i].lower   = tmp [lower];
		cis1_energy_sweep_table_8 [0][i].average = Moyenne (tmp, upper, lower);
	}

	/*	Calcul de la table solar wind
	 */
	for (fact = 1.0, i = 0; i < 129; i ++) {

		if (i < cis1_param_sw_stop) {

			tmp [i] = (float) cis1_anal_k_factor [2] * cis1_param_sw [0] * fact;
			fact   *= cis1_param_sw [1];
		}
		else	tmp [i] = tmp [i-1];
	}

	for (i = 0; i < 31; i++) {

		upper = i * 4;
		lower = i * 4 + 4;
		cis1_energy_sweep_table_31 [1][i].upper   = tmp [upper];
		cis1_energy_sweep_table_31 [1][i].lower   = tmp [lower];
		cis1_energy_sweep_table_31 [1][i].average = Moyenne (tmp, upper, lower);
	}
	for (i = 0; i < 16; i++) {

		upper = i * 8;
		lower = (i < 15) ? i * 8 + 8 : 124;
		cis1_energy_sweep_table_16 [1][i].upper   = tmp [upper];
		cis1_energy_sweep_table_16 [1][i].lower   = tmp [lower];
		cis1_energy_sweep_table_16 [1][i].average = Moyenne (tmp, upper, lower);
	}
	for (i = 0; i < 8; i++) {

		upper = i * 16;
		lower = (i < 7) ? i * 16 + 16 : 124;
		cis1_energy_sweep_table_8 [1][i].upper   = tmp [upper];
		cis1_energy_sweep_table_8 [1][i].lower   = tmp [lower];
		cis1_energy_sweep_table_8 [1][i].average = Moyenne (tmp, upper, lower);
	}

	/*	Calcul de la table PROM
	 */
	for (fact = 1.0, i = 0; i < 129; i ++) {

		tmp [i] = cis1_anal_k_factor [2] * cis1_param_prom [0] * fact;
		fact   *= cis1_param_prom [1];
	}

	for (i = 0; i < 32; i++) {

		upper = i * 4;
		lower = i * 4 + 4;
		cis1_energy_sweep_table_prom [i].upper   = tmp [upper];
		cis1_energy_sweep_table_prom [i].lower   = tmp [lower];
		cis1_energy_sweep_table_prom [i].average = Moyenne (tmp, upper, lower);
	}

	/*	Calcul de la table RPA
	 */
	for (i = 0; i < 16; i++) {

		float	emin = cis1_param_rpa [0] - cis1_param_rpa [1] * i;

		cis1_energy_sweep_table_rpa [i].upper   = emin + cis1_param_rpa [2];
		cis1_energy_sweep_table_rpa [i].lower   = emin;
		cis1_energy_sweep_table_rpa [i].average = emin + cis1_param_rpa [2] / 2.0;

		Affiche_trace (3, fonction, "table_rpa [%2d] = [ %5.2f, %5.2f]",
			i,
			cis1_energy_sweep_table_rpa [i].lower,
			cis1_energy_sweep_table_rpa [i].upper);
	}
EXIT:	return error;
}

/***************************************************************************************************
 *
 *	Lecture efficacites normalisees par masse
 *	-----------------------------------------
 */
t_err	Traite_CIS1_total_effic_normal (FILE * entree, char * variable)
{
	char *		fonction = FNAME ("Traite_CIS1_total_effic_normal");
	t_err		error = OK;
	char		buffer [200];
	int		i, lus, pac, selection;
	float		tab [4];

	selection = 0;

	for (i = 0; i < 4; i++) cis1_total_effic_normal [i] = 0.0;

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		lus = sscanf (buffer, "%d %f %f %f %f", 
			& pac, & tab [0], & tab [1], & tab [2], & tab [3]);

		if (lus != 5) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		/*	Recherche de la post acceleration la plus proche
		 */
		if (fabs (cis1_post_accel_volt - (float) pac) < fabs (cis1_post_accel_volt - (float) selection)) {

			selection = pac;

			for (i = 0; i < 4; i++) cis1_total_effic_normal [i] = tab [i];
		}
	}

	if (selection == 0) {

		Affiche_erreur (fonction, "Impossible trouver valeur approchee de %f", cis1_post_accel_volt);
		error = ERROR;
		goto EXIT;
	}

EXIT:	return error;
}



/***************************************************************************************************
 *
 *	Lecture des coefficients propres a CIS1
 *	---------------------------------------
 */
t_err	Traite_CIS1 (FILE * entree, char * produit)
{
	char *		fonction = FNAME ("Traite_CIS1");
	t_err		error = OK;
	char		buffer [200], var [50];

	while (Read_line (buffer, entree) == OK) {

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		if (sscanf (buffer, "BEGIN %s", var) != 1) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		if (VAR ("accumulation_spin_table"))	error = Traite_accumulation_spin (entree, CIS_1);
		else 
		if (VAR ("geom_factor"))		error = Traite_CIS1_geom_factor (entree, var);
		else 
		if (VAR ("anal_k_factor"))		error = Traite_variable (entree, var, 3, cis1_anal_k_factor);
		else 
		if (VAR ("anal_alpha_angle"))		error = Traite_variable (entree, var, 2, cis1_anal_alpha_angle);
		else 
		if (VAR ("energy_sweep_table_sw"))	error = Traite_variable (entree, var, 2, cis1_param_sw);
		else 
		if (VAR ("energy_sweep_table_mag"))	error = Traite_variable (entree, var, 2, cis1_param_mag);
		else 
		if (VAR ("energy_sweep_table_prom"))	error = Traite_variable (entree, var, 2, cis1_param_prom);
		else 
		if (VAR ("sw_energy_sweep_stop"))	error = Traite_variable (entree, var, 1, & cis1_param_sw_stop);
		else 
		if (VAR ("rpa_energy_table"))		error = Traite_variable (entree, var, 3, cis1_param_rpa);
		else 
		if (VAR ("post_accel_volt"))		error = Traite_variable (entree, var, 1, & cis1_post_accel_volt);
		else 
		if (VAR ("dead_times"))			error = Traite_variable (entree, var, 2, cis1_dead_times);
		else 
		if (VAR ("dens_corr_coeff_P7"))		error = Traite_CIS1_dens_corr (entree, var);
		else 
		if (VAR ("dens_corr_coeff_P9"))		error = Traite_CIS1_dens_corr (entree, var);
		else 
		if (VAR ("total_effic_normal"))		error = Traite_CIS1_total_effic_normal (entree, var);
		else 
		if (VAR ("anode_effic_coeff_table"))	error = Traite_CIS1_anode_effic (entree, var);
		else 
		if (VAR ("onboard_anode_effic_coeff_table"))	error = Ignore_variable (entree, var);
		else 
		if (VAR ("rpa_geom_factor"))		error = Traite_CIS1_geom_factor (entree, var);
		else {	
			Affiche_erreur (fonction, "Variable %s inconnue", var);
			error = ERROR;
		}
		if (Erreur (error)) goto EXIT;
	}

	error = Genere_CIS1_energy_sweep_tables ();

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Lecture des coefficients d'efficacite de CIS1
 *	---------------------------------------------
 */
t_err	Traite_EFF (FILE * entree, char * produit)
{
	char *		fonction = FNAME ("Traite_EFF");
	t_err		error = OK;
	char		buffer [200], var [30], sensit [10];
	float		tab [4];
	int		m;

	/*	Invalider les calibrations P07 et P09 qui dependent maintenant directement de ABSEFF
	 *	------------------------------------------------------------------------------------
	 */
	if (Erreur (Invalide_calibration ("P07"))) goto EXIT;
	if (Erreur (Invalide_calibration ("P09"))) goto EXIT;
	
	for (m = 0; m < NB_MASSE_MAX; m++) {

		cis1_absolute_efficiencies [LS][m] = 1.0;
		cis1_absolute_efficiencies [HS][m] = 1.0;
	}

	while (Read_line (buffer, entree) == OK) {

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		if (sscanf (buffer, "BEGIN %s", var) != 1) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		if (VAR ("absolute_efficiencies")) break;

		Affiche_erreur (fonction, "variable %s incorrecte", var);
		error = ERROR;
		goto EXIT;
	}

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		if (sscanf (buffer, "%s %f %f %f %f", sensit, & tab [0], & tab [1], & tab [2], & tab [3]) != 5){

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		switch (Val_from_key (cis_sensitivity, sensit)) {

		case LS:	for (m = 0; m < NB_MASSE_MAX; m++) cis1_absolute_efficiencies [LS][m] = tab [m];
				break;

		case HS:	for (m = 0; m < NB_MASSE_MAX; m++) cis1_absolute_efficiencies [HS][m] = tab [m];
				break;

		default:	Affiche_erreur (fonction, "Sensitivite incorrecte : %s", sensit);
				error = ERROR;
				goto EXIT;
		}
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	P A R T I E   S P E C I F I Q U E
 *
 *	A U X   C A L I B R A T I O N S   D U   P R O D U I T   C I S 2 
 *
 ***************************************************************************************************
 */


/***************************************************************************************************
 *
 *	Lecture des coefficients d'efficacite des anodes
 *	------------------------------------------------
 */
t_err	Traite_CIS2_anode_effic (FILE * entree, char * variable)
{
	char *		fonction = FNAME ("Traite_CIS2_anode_effic");
	t_err		error = OK;
	char		buffer [200];
	int		indice, sensit;
	float		theta, effic;

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		if (sscanf (buffer, "%d %f %f", & indice, & theta, & effic) != 3) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		if (indice < 0 || indice > 31) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		if (indice >= 16) {

			sensit = LS;
			indice = indice - 16;
		}
		else	sensit = HS;

		cis2_anode_eff [sensit][indice][0] = theta;
		cis2_anode_eff [sensit][indice][1] = effic;
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Lecture des fitting param 
 *	-------------------------
 */
t_err	Traite_CIS2_fitting_param (FILE * entree, char * variable)
{
	char *		fonction = FNAME ("Traite_CIS2_fitting_param");
	t_err		error = OK;
	char		buffer [200], sensitivite [10];
	int		sensit;
	float		a, b, norm_e, norm_theta;

	for (sensit = 0; sensit < 2; sensit ++) {

		cis2_a		[sensit] = 0.0;
		cis2_b		[sensit] = 0.0;
		cis2_norm_e	[sensit] = 0.0;
		cis2_norm_theta	[sensit] = 0.0;
	}

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		if (sscanf (buffer, "%s %f %f %f %f", sensitivite, & a, & b, & norm_e, & norm_theta) != 5) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		if ((sensit = Val_from_key (cis_sensitivity, sensitivite)) == EOF) {

			Affiche_erreur (fonction, "Sensitivite %s incorrecte", sensitivite);
			error = ERROR;
			goto EXIT;
		}

		cis2_a 		[sensit] = a;
		cis2_b		[sensit] = b;
		cis2_norm_e	[sensit] = norm_e;
		cis2_norm_theta	[sensit] = norm_theta;
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Lecture des coefficients MCP fitting param
 *	------------------------------------------
 */
t_err	Traite_CIS2_MCP_fitting_param (FILE * entree, char * variable)
{
	char *		fonction = FNAME ("Traite_CIS2_MCP_fitting_param");
	t_err		error = OK;
	char		buffer [200];
	float		tab [5];
	int		ligne, i;

	for (ligne = 0; ligne < 3; ligne ++) {

		for (i = 0; i < 5; i++) cis2_MCP_fitting_param [ligne][i] = 0.0;
	}

	ligne = 0;

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		if (sscanf (buffer, "%f %f %f %f %f", & tab[0], & tab[1], & tab[2], & tab[3], & tab[4]) != 5) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		for (i = 0; i < 5; i ++) cis2_MCP_fitting_param [ligne][i] = tab [i];

		ligne ++;
	}
	if (ligne != 3) {

		Affiche_erreur (fonction, "%d lignes lues au lieu de 3", ligne);
		error = ERROR;
		goto EXIT;
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Algorithme de correction des tables d'energies
 *	----------------------------------------------
 *
 *	Empecher le recouvrement des paliers d'energies : 
 *	si la borne inferieure du palier i est inferieure a la borne superieure du palier i+1,
 *	faire la moyenne logarithmique de ces deux bornes et l'affecter a ces deux bornes
 */
void	Corrige_table (TE_DESCR * table, int size)
{
	int		i;

	for (i = 0; i < size - 1; i++) {

		if (table [i].lower < table [i+1].upper) {

			float	value = Moyenne_log (table [i].lower, table [i+1].upper);

			table [i].lower = table [i+1].upper = value;
		}
	}
}


/***************************************************************************************************
 *
 *	Troncature de la table d'energie No 10 de CIS2
 *	----------------------------------------------
 */
t_err	Troncature_table_CIS2 (float * tmp)
{
	char *		fonction = FNAME ("Troncature_table_CIS2");
	t_err		error = OK;
	int		i, seuil, lower, upper;
	TE_DESCR *	td;
	float		dE_E = CIS2_dE_E [HS];
	float		K = 0.50;

	seuil = (int) cis2_param_sw_stop;

	Affiche_trace (3, fonction, "Troncature table 10 a partir du step %d/128", seuil);

// AB 2016/10/19	Test pour CAA qui veut les table en 124

	td = cis2_energy_sweep_table_124 [10];

	for (i = 0; i < 124; i++) {

		upper = (i < seuil) ? i : seuil;
		lower = (i + 1 < seuil) ? i + 1 : seuil;

		td [i].upper	= tmp [upper] * (1.0 + K * dE_E);
		td [i].lower	= tmp [lower] * (1.0 - K * dE_E);
		td [i].average = Moyenne_log (tmp [upper], tmp [lower]);
	}

	(void) Corrige_table (td, seuil + 1);

// AB 2016/10/19	Test pour CAA qui veut les table en 124

	td = cis2_energy_sweep_table_62 [10];

	for (i = 0; i < 62; i++) {

		upper = (i * 2 < seuil) ? i * 2 : seuil;
		lower = (i * 2 + 1 < seuil) ? i * 2 + 1 : seuil;

		td [i].upper	= tmp [upper] * (1.0 + K * dE_E);
		td [i].lower	= tmp [lower] * (1.0 - K * dE_E);
		td [i].average = Moyenne_log (tmp [upper], tmp [lower]);
	}

	(void) Corrige_table (td, seuil / 2 + 1);

	td = cis2_energy_sweep_table_31 [10];

	for (i = 0; i < 31; i++) {

		upper = (i * 4 < seuil) ? i * 4 : seuil;
		lower = (i * 4 + 3 < seuil) ? i * 4 + 3 : seuil;

		td [i].upper	= tmp [upper] * (1.0 + K * dE_E);
		td [i].lower	= tmp [lower] * (1.0 - K * dE_E);
		td [i].average = Moyenne_log (tmp [upper], tmp [lower]);
	}

	(void) Corrige_table (td, seuil / 4 + 1);

	td = cis2_energy_sweep_table_16 [10];

	for (i = 0; i < 16; i++) {

		upper = (i * 8 < seuil) ? i * 8 : seuil;
		lower = (i * 8 + 7 < seuil) ? i * 8 + 7 : seuil;

		td [i].upper	= tmp [upper] * (1.0 + K * dE_E);
		td [i].lower	= tmp [lower] * (1.0 - K * dE_E);
		td [i].average	= Moyenne_log (tmp [upper], tmp [lower]);
	}

	(void) Corrige_table (td, seuil / 8 + 1);

EXIT:	return error;
}



/***************************************************************************************************
 *
 *	Lecture des coefficients des tables d'energies de CIS2
 *	------------------------------------------------------
 */
t_err	Traite_CIS2_energy_sweep_table (FILE * entree, char * produit)
{
	char *		fonction = FNAME ("Traite_CIS2_energy_sweep_table");
	t_err		error = OK;
	char		buffer [200];
	int		table, steps, law;
	float		k, fact, emin, emax, tmp [124];
	int		i, nbre_steps, upper, lower;

	nbre_steps = 0;

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		if (sscanf (buffer, "%d %d %f %f L%d", & table, & steps, & emax, & emin, & law) != 5) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		if (table < 0 || table > 9) {

			Affiche_erreur (fonction, "Table %d incorrecte", table);
			error = ERROR;
			goto EXIT;
		}

		if (steps + nbre_steps > 124) {

			Affiche_erreur (fonction, "Depassement capacite table %d", table);
			error = ERROR;
			goto EXIT;
		}

		/*	Calcul table temporaire
		 */
		switch (law) {

		case 1:		k = (float) exp (log ((double) emin / (double) emax) / (steps - 1));

				for (fact = 1.0, i = 0; i < steps; i++) {

					tmp [nbre_steps + i] = (float) emax * fact;
					fact = fact * k;
				}
				break;

		case 2:		k = (float) exp (log ((double) emin / (double) emax) / steps);

				for (fact = 1.0, i = 0; i < steps; i++) {

					tmp [nbre_steps + i] = (float) emax * fact;
					fact = fact * k;
				}
				break;

		case 3:		for (i = 0; i < steps; i++) { 

					if (i < steps / 2)
						tmp [nbre_steps + i] = (float) emax;
					else	tmp [nbre_steps + i] = (float) emin;
				}
				break;

		case 4:		if (steps % 4 != 0) {

					Affiche_erreur (fonction, "Nombre de steps loi 4 doit etre multiple de 4");
					error = ERROR;
					goto EXIT;
				}

				k = (float) (exp (log ((double) emin / (double) emax) / (steps / 4 - 1)));

				for (fact = 1.0, i = 0; i < steps; i++) {

					tmp [nbre_steps + i] = (float) emax * fact;

					if (i % 4 == 3) fact = fact * k;
				}
				break;

		default :	Affiche_erreur (fonction, "Loi No %d inconnue", law);
				error = ERROR;	
				goto EXIT;
		}
		nbre_steps += steps;

		/*	Generation tables definitives
		 */
		if (nbre_steps == 124) {

			float		dE_E = CIS2_dE_E [(table == 0) ? HS : LS];
			float		K = 0.50;
			TE_DESCR *	td;

			td = cis2_energy_sweep_table_124 [table];

			for (i = 0; i < 124; i++) {

				upper = i;
				lower = i + 1;

				td [i].upper = tmp [upper] * (1.0 + K * dE_E);
				td [i].lower = tmp [lower] * (1.0 + K * dE_E);
				td [i].average = Moyenne_log (tmp [upper], tmp [lower]);
			}
			(void) Corrige_table (td, 124);

			td = cis2_energy_sweep_table_62 [table];

			for (i= 0; i < 62; i++) {

				upper = i * 2;
				lower = i * 2 + 1;

				td [i].upper   = tmp [upper] * (1.0 + K * dE_E);
				td [i].lower   = tmp [lower] * (1.0 - K * dE_E);
				td [i].average = Moyenne_log (tmp [upper], tmp [lower]);
			}

			(void) Corrige_table (td, 62);
				
			td = cis2_energy_sweep_table_31 [table];

			for (i = 0; i < 31; i++) {

				upper = i * 4;
				lower = i * 4 + 3;

				td [i].upper   = tmp [upper] * (1.0 + K * dE_E);
				td [i].lower   = tmp [lower] * (1.0 - K * dE_E);
				td [i].average = Moyenne_log (tmp [upper], tmp [lower]);
 			}

			(void) Corrige_table (td, 31);

			td = cis2_energy_sweep_table_16 [table];

			for (i = 0; i < 16; i++) {

				upper = i * 8;
				lower = (i < 15) ? i * 8 + 7 : 123;

				td [i].upper   = tmp [upper] * (1.0 + K * dE_E);
				td [i].lower   = tmp [lower] * (1.0 - K * dE_E);
				td [i].average = Moyenne_log (tmp [upper], tmp [lower]);
			}

			(void) Corrige_table (td, 16);

			if (table == 0) {

				error = Troncature_table_CIS2 (tmp);

				if (Erreur (error)) goto EXIT;
			}

			nbre_steps = 0;
		}
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Generation de la table PROM de CIS2
 *	-----------------------------------
 */
t_err	Genere_CIS2_energy_sweep_table_prom (void)
{
	char *		fonction = FNAME ("Genere_CIS2_energy_sweep_table_prom");
	t_err		error = OK;
	double		tmp [33];
	int		i;

	for (i = 0; i < 33; i++)
		tmp [i] = cis2_anal_k_factor [2] * cis2_param_prom [0] * pow (cis2_param_prom [1], i * 4);

	for (i = 0; i < 32; i++) {

		cis2_energy_sweep_table_prom [i].upper   = tmp [i];
		cis2_energy_sweep_table_prom [i].lower   = tmp [i+1];
		cis2_energy_sweep_table_prom [i].average = (tmp [i] + tmp [i+1]) / 2;
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Lecture des coefficients propres a CIS2
 *	---------------------------------------
 */
t_err	Traite_CIS2 (FILE * entree, char * produit)
{
	char *		fonction = FNAME ("Traite_CIS2");
	t_err		error = OK;
	char		buffer [200], var [50];

	while (Read_line (buffer, entree) == OK) {

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		if (sscanf (buffer, "BEGIN %s", var) != 1) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		if (VAR ("accumulation_spin_table"))	error = Traite_accumulation_spin (entree, CIS_2);
		else
		if (VAR ("geom_factor"))		error = Traite_variable (entree, var, 2, cis2_geom_factor);
		else
		if (VAR ("anal_k_factor"))		error = Traite_variable (entree, var, 3, cis2_anal_k_factor);
		else
		if (VAR ("anal_alpha_angle"))		error = Traite_variable (entree, var, 2, cis2_anal_alpha_angle);
		else
		if (VAR ("dead_times"))			error = Traite_variable (entree, var, 2, cis2_dead_times);
		else
		if (VAR ("sw_energy_sweep_stop"))	error = Traite_variable (entree, var, 1, & cis2_param_sw_stop);
		else
		if (VAR ("energy_sweep_table"))		error = Traite_CIS2_energy_sweep_table (entree, var);
		else
		if (VAR ("energy_sweep_table_prom"))	error = Traite_variable (entree, var, 2, cis2_param_prom);
		else
		if (VAR ("anode_effic_table"))		error = Traite_CIS2_anode_effic (entree, var);
		else
		if (VAR ("fitting_param"))		error = Traite_CIS2_fitting_param (entree, var);
		else
		if (VAR ("MCP_fitting_param"))		error = Traite_CIS2_MCP_fitting_param (entree, var);
		else {	
			Affiche_erreur (fonction, "Variable %s inconnue", var);
			error = ERROR;
		}
		if (Erreur (error)) goto EXIT;
	}

	if (Erreur (error = Genere_CIS2_energy_sweep_table_prom ())) goto EXIT;

EXIT:	return error;
}



/***************************************************************************************************
 *
 *	P A R T I E   S P E C I F I Q U E   
 *
 *	A   L A   G E S T I O N   D U   C A T A L O G U E   D E   C A L I B R A T I ON
 *
 ***************************************************************************************************
 */


/***************************************************************************************************
 *
 *	Cherche l'entree correspondant au produit
 *	-----------------------------------------
 */
t_err	Cherche_entree_produit (char * produit)
{
	char *		fonction = FNAME ("Cherche_entree_produit");
	int		i, trv = -1;

	for (i = 0; i < NB_ENTREE && trv == -1; i++) {

		if (strcmp (table [i].produit, produit) == 0) trv = i;
	}
	return trv;
}


/***************************************************************************************************
 *
 *	Initialise calibrations (methode complete)
 *	------------------------------------------
 */
t_err	Ouverture_calibration (int satellite, char * catalogue, int version)
{
	char *		fonction = FNAME ("Ouverture_calibration");
	t_err		error = OK;
	int		i;
	char		varname [30];
	char *		path;

	sprintf (varname, "CIS%d_CALPATH", satellite);

	if ((path = getenv (varname)) == NULL) {

		Affiche_erreur (fonction, "Variable %s non definie", varname);
		error = ERROR;
		goto EXIT;
	}

	strcpy (calib_path, path);

	Affiche_trace (1, fonction, "Repertoire de calibration : %s", calib_path);

	for (i = 0; i < NB_ENTREE; i++) table [i].indice = -1;

	if (Erreur (error = Lecture_catalogue (catalogue, version))) goto EXIT;

	error = Affiche_catalogue ();

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Initialise calibrations (methode simplifiee)
 *	--------------------------------------------
 */
t_err	Initialise_calibration (int satellite)
{
	char *		fonction = FNAME ("Initialise_calibration");
	t_err		error = OK;
	int		i;
	int		count;
	int		version = 999;
	t_filename	masque;

	if (Erreur (error = Check_CIS_root ())) goto EXIT;

	sprintf (calib_path, "%s/CALIB/CLUSTER%d", Get_CIS_root (), satellite);

	Affiche_trace (1, fonction, "Repertoire calibration : %s", calib_path);	

	sprintf (masque, "%s/C%d_L1_CISxxx_????????_V??_CALCATALOG.cal",
		calib_path, satellite);

	if (Erreur (error = Search_files (masque, & count))) goto EXIT;

	if (count != 1) {

		Affiche_erreur (fonction, "%s", masque);
		Affiche_erreur (fonction, "%d catalogues de calibration trouves", count);
		error = ERROR;
		goto EXIT;
	}
	
	for (i = 0; i < NB_ENTREE; i++) table [i].indice = -1;

	if (Erreur (error = Lecture_catalogue (Get_filename (count -1), version))) goto EXIT;

	error = Affiche_catalogue ();

EXIT:	Search_free ();
	return error;
}


/***************************************************************************************************
 *
 *	Cloture traitement des calibrations
 *	-----------------------------------
 */
t_err	Cloture_calibration (void)
{
	char *		fonction = FNAME ("Cloture_calibration");
	t_err		error = OK;
	int		i;

	for (i = 0; i < NB_ENTREE; i++) table [i].indice = -1;

	error = Fermeture_catalogue ();

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Lecture du fichier de calibration pour un produit
 *	-------------------------------------------------
 */
t_err	Lecture_calibration (char * produit, double date)
{
	char *		fonction = FNAME ("Lecture_calibration");
	t_err		error = OK;
	int		trv, indice;
	FILE *		entree = NULL;
	int		a_lire = FALSE;
	t_filename	fichier;

	trv = Cherche_entree_produit (produit);

	if (trv == -1) {

		Affiche_erreur (fonction, "Produit %s inconnu", produit);
		error = ERROR;
		goto EXIT;
	}

	indice = table [trv].indice;

	if (indice == -1 || Actualisation_necessaire (indice, date)) {

		indice = Cherche_entree_catalogue (produit, date);

		if (indice == -1) {

			Affiche_erreur (fonction, "Pas de calibration pour %s %s",
				produit, Milli_to_Ascii_time (date));
			error = ERROR;
			goto EXIT;
		}
		a_lire = TRUE;
	}

	if (a_lire == TRUE) {

		sprintf (fichier, "%s/%s", calib_path, Fichier_calibration (indice));

		if ((entree = fopen (fichier, "r")) == NULL) {

			Affiche_erreur (fonction, "Ouverture %s impossible", fichier);
			error = ERR_fopen;
			goto EXIT;
		}
		Affiche_trace (1, fonction, "%-10s %s  %s", 
			produit, Milli_to_Ascii_time (date), Fichier_calibration (indice));

		error = (* table [trv].fonction) (entree, produit);
	}

	if (Erreur (error))	
		table [trv].indice = -1;
	else	table [trv].indice = indice;

EXIT:	if (entree != NULL) fclose (entree);

	return error;
}


/***************************************************************************************************
 *
 *	Invalidation forcee des calibrations d'un produit
 *	-------------------------------------------------
 */
t_err	Invalide_calibration (char * produit)
{
	char *		fonction = FNAME ("Invalide_calibration");
	t_err		error = OK;
	int		indice;

	indice = Cherche_entree_produit (produit);

	if (indice == -1) {

		Affiche_erreur (fonction, "Produit de calibration %s inconnu", produit);
		error = ERROR;
		goto EXIT;
	}
	table [indice].indice = -1;

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Retourne intervalle de validite des calibrations d'un produit
 *	-------------------------------------------------------------
 */
t_err	Validite_calibration (char * produit, double * date_deb, double * date_fin)
{
	char *		fonction = FNAME ("Validite_calibration");
	t_err		error = OK;
	int		trv;

	trv = Cherche_entree_produit (produit);	

	if (trv == -1) {

		* date_deb = 0.0;
		* date_fin = 0.0;
		error = ERROR;
		goto EXIT;
	}

	error = Periode_validite (table [trv].indice, date_deb, date_fin);

	Affiche_trace (3, fonction, "Produit %s", produit);
	Affiche_trace (3, fonction, "Depuis  %s", Milli_to_Ascii_time (* date_deb));
	Affiche_trace (3, fonction, "Jusque  %s", Milli_to_Ascii_time (* date_fin));
	
EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Lecture des calibrations des moments CODIF
 *	------------------------------------------
 *
 *	Depuis le 05-02-2002, les calibrations P07 et P09 ne sont recalcules que si modification des
 *	produits CIS1 et OMC1, mais pas a chaque modification des fichiers EFF, ceci afin de minimiser
 *	leur nombre.
 *
 *	Par contre, dans le calcul des moments a bord, il convient de prendre en compte les valeurs
 *	du produit EFF, pour recalculer les coefficients des produits P07 et P09.
 *
 *	Lire produit EFF pour mettre a jour la variable cis1_absolute_efficiencies, ce qui a pour 
 *	consequence d'invalider les calibrations des produits P07 et P09.
 *
 *	Ensuite, a chaque fois qu'il est necessaire de relire les calibrations P07 ou P09, on procede
 *	a la division des coefficients par cis1_absolute_efficiencies [sensitivite][masse], pour tenir
 *	compte de la degradation temporelle de la sensitivite des instruments.
 */
t_err	Lecture_calibration_CODIF (int sensitivite, double date)
{
	char *		fonction = FNAME ("Lecture_calibration_CODIF");
	t_err		error = OK;

	error = Lecture_calibration ("MASS", date);

	if (Erreur (error)) goto EXIT;

	error = Lecture_calibration ("EFF", date);

	if (Erreur (error)) goto EXIT;

	switch (sensitivite) {

	case LS :	error = Invalide_calibration ("P07");
			error = Lecture_calibration  ("P09", date);
			break;

	case HS :	error = Invalide_calibration ("P09");
			error = Lecture_calibration  ("P07", date);
			break;

	default :	Affiche_erreur (fonction, "Efficacite incorrecte : %d", sensitivite);
			error = ERROR;
			goto EXIT;
			break;
	}

EXIT:	return error;
}
