/***************************************************************************************************
 *
 *	Fichier	: $RCSfile: CALIB.h,v $
 *
 *	Version	: $Revision: 1.2 $
 *
 *	Auteur	: $Author: barthe $
 *
 *	Date	: $Date: 2005/09/07 14:11:53 $
 *
 *	==========================================================================================
 *
 *	Ce module est charge de la gestion des tables de calibrations
 */

#ifndef __CALIB_H
#define __CALIB_H

#define	NB_MASSE_COD_07		4
#define	NB_MASSE_COD_09		4
#define	NB_MASSE_HIA_02		1
#define	NB_MASSE_HIA_04		2
#define	NB_MASSE_MAX		4

#define	NB_ENERGY_RANGE		3

#define	NB_ANODE_CIS1		8
#define	NB_ANODE_CIS2		16
#define	NB_DEGRE		4


/*	Coefficients de calibration des moments a bord 
 *	----------------------------------------------
 *	- produit P02  : produit 2 de HIA
 *	- produit P04  : produit 4 de HIA
 *	- produit P07  : produit 7 de CODIF
 *	- produit P09  : produit 9 de CODIF
 *	- produit MASS : ordre des masses CODIF dans telemesure
 */

typedef struct { float	a, b; }				COEFF;
typedef struct { COEFF	vx, vy, vz; }			V_COEFF;
typedef struct { COEFF	pxx, pxy, pxz, pyy, pyz, pzz; }	P_COEFF;
	
extern	int		k_position [NB_MASSE_MAX];			/* Ordre des masses dans la telemesure	*/
extern	COEFF		k_densite  [NB_MASSE_MAX][NB_ENERGY_RANGE];	/* Coefficients des densite		*/
extern	V_COEFF		k_vitesse  [NB_MASSE_MAX][NB_ENERGY_RANGE];	/* Coefficients des vitesses		*/
extern	P_COEFF		k_pression [NB_MASSE_MAX][NB_ENERGY_RANGE];	/* Coefficients des pressions		*/


/*	Coefficients de calibration CODIF : produit OMC1
 *	------------------------------------------------
 */
extern	float		cis1_onboard_moment_coeff_P7	[3];
extern	float		cis1_onboard_moment_coeff_P9	[3];


/*	Coefficients de calibration HIA : produit OMC2
 *	----------------------------------------------
 */
extern	float		cis2_onboard_moment_coeff_P2;
extern	float		cis2_onboard_moment_coeff_P4;


/*	Coefficients de calibration des produits CIS1 et CIS2
 *	-----------------------------------------------------
 */
typedef struct { float	upper, average, lower; }	TE_DESCR;	/* Entree table d'energie		*/


/*	Coefficients de calibration CODIF : produit CIS1
 *	------------------------------------------------
 */
extern	int     	cis1_accumulation_spin		[7][64][NB_CIS_MODE];
extern	float		cis1_geom_factor		[NB_SENSITIVITY][NB_ANODE_CIS1][8];
extern	float		cis1_rpa_geom_factor		[NB_SENSITIVITY][NB_ANODE_CIS1][8];
extern	float		cis1_anal_k_factor		[3];
extern	float		cis1_anal_alpha_angle		[NB_SENSITIVITY];
extern	float		cis1_post_accel_volt;
extern	float		cis1_total_effic_normal		[NB_MASSE_MAX];
extern	float		cis1_dead_times			[2];
extern	float		cis1_anode_eff			[NB_SENSITIVITY][NB_ANODE_CIS1][NB_DEGRE][NB_MASSE_MAX];
extern	COEFF		cis1_dens_corr_coeff_P7		[NB_SENSITIVITY][NB_ENERGY_RANGE];
extern	COEFF		cis1_dens_corr_coeff_P9		[NB_SENSITIVITY][NB_ENERGY_RANGE];
extern	TE_DESCR       	cis1_energy_sweep_table_8	[2][8];
extern	TE_DESCR       	cis1_energy_sweep_table_16	[2][16];
extern	TE_DESCR       	cis1_energy_sweep_table_31	[2][31];
extern	TE_DESCR       	cis1_energy_sweep_table_prom	[32];
extern	TE_DESCR	cis1_energy_sweep_table_rpa	[16];


/*	Coefficients de calibration CODIF : produit EFF
 *	-----------------------------------------------
 */
extern	float           cis1_absolute_efficiencies	[NB_SENSITIVITY][NB_MASSE_MAX];


/*	Coefficients de calibration HIA : produit CIS2
 *	----------------------------------------------
 */
extern	int     	cis2_accumulation_spin		[7][64][NB_CIS_MODE];
extern	float		cis2_geom_factor		[NB_SENSITIVITY];
extern	float		cis2_anal_k_factor		[3];
extern	float		cis2_anal_alpha_angle		[NB_SENSITIVITY];
extern	float		cis2_anode_eff			[NB_SENSITIVITY][NB_ANODE_CIS2][2];
extern	float		cis2_dead_times			[2];
extern	float		cis2_a				[NB_SENSITIVITY];
extern	float		cis2_b				[NB_SENSITIVITY];
extern	float		cis2_norm_e			[NB_SENSITIVITY];
extern	float		cis2_norm_theta			[NB_SENSITIVITY];
extern	float		cis2_MCP_fitting_param		[3][5];
extern	TE_DESCR	cis2_energy_sweep_table_16	[11][16];
extern	TE_DESCR	cis2_energy_sweep_table_31	[11][31];
extern	TE_DESCR	cis2_energy_sweep_table_62	[11][62];
extern	TE_DESCR	cis2_energy_sweep_table_prom	[32];


/***************************************************************************************************
 *
 *	Initialise traitement des calibrations (methode complete)
 *	---------------------------------------------------------
 */
t_err	Ouverture_calibration (int satellite, char * catalogue, int version);


/***************************************************************************************************
 *
 *	Initialise traitement des calibrations (methode simplifiee)
 *	-----------------------------------------------------------
 */
t_err	Initialise_calibration  (int satellite);


/***************************************************************************************************
 *
 *	Cloture traitement des calibrations
 *	-----------------------------------
 */
t_err	Cloture_calibration (void);


/***************************************************************************************************
 *
 *	Lecture du fichier de calibration pour un produit
 *	-------------------------------------------------
 */
t_err	Lecture_calibration (char * produit, double date);


/***************************************************************************************************
 *
 *	Recherche des coefficients Housekeeping de calcul d'une variable
 *	----------------------------------------------------------------
 */
t_err	Lire_parametre_HSK (char * variable, COEFF * coeff);


/***************************************************************************************************
 *
 *	Retourne intervalle de validite des calibrations d'un produit
 *	-------------------------------------------------------------
 */
t_err	Validite_calibration (char * produit, double * date_deb, double * date_fin);


/***************************************************************************************************
 *
 *	Invalidation forcee des calibrations d'un produit
 *	-------------------------------------------------
 */
t_err	Invalide_calibration (char * produit);


/***************************************************************************************************
 *
 *	Lecture des calibrations des moments CODIF
 *	------------------------------------------
 *
 *	Depuis le 05-02-2002, les calibrations P07 et P09 ne sont recalcules que si modification des
 *	produits CIS1 et OMC1, mais pas a chaque modification des fichiers EFF, ceci afin de minimiser
 *	leur nombre.
 *
 *	Par contre, dans le calcul des moments a bord, il convient de prendre en compte les valeurs
 *	du produit EFF, pour recalculer les coefficients des produits P07 et P09.
 *
 *	Lire produit EFF pour mettre a jour la variable cis1_absolute_efficiencies, ce qui a pour 
 *	consequence d'invalider les calibrations des produits P07 et P09.
 *
 *	Ensuite, a chaque fois qu'il est necessaire de relire les calibrations P07 ou P09, on procede
 *	a la division des coefficients par cis1_absolute_efficiencies [sensitivite][masse], pour tenir
 *	compte de la degradation temporelle de la sensitivite des instruments.
 */
t_err	Lecture_calibration_CODIF (int sensit, double date);


#endif
