/***************************************************************************************************
 *
 *	Fichier	: $RCSfile: MATHS.c,v $
 *
 *	Version	: $Revision: 1.1 $
 *
 *	Auteur	: $Author: barthe $
 *
 *	Date	: $Date: 2007/07/19 10:44:36 $
 *
 *	==========================================================================================
 *
 *	Module definissant quelques outils de calcul
 */

#define	MODULE_NAME	"MATHS"

#include "DSPLIB.h"

#define	MAX_ITER_JACOBI		50

const	float	FILL_VALUE = -1.0E+31;
const	float	EPSILON    = 1.0E-9;

/***************************************************************************************************
 *
 *	Indique si la valeur est une valeur filaire
 *	-------------------------------------------
 */
int	Fill_value (double value)
{
	return (1.0 - value / FILL_VALUE) < EPSILON;
}


/***************************************************************************************************
 *
 *	Calcule la norme d'un vecteur
 *	-----------------------------
 */
double	Norm (V3 v)
{
	return sqrt (v [0] * v [0] + v [1] * v [1] + v [2] * v [2]);
}


/***************************************************************************************************
 *
 *	Affiche contenu matrice
 *	-----------------------
 */
void	Vector_display (int level, char * fonction, char * text, V3 vector)
{
	Affiche_trace (level, fonction, "%s = % 1.4e % 1.4e % 1.4e",
		text,
		vector [0],
		vector [1],
		vector [2]);
}


/***************************************************************************************************
 *
 *	Affiche contenu matrice
 *	-----------------------
 */
void	Matrix_display (int level, char * fonction, char * text, M33 mat)
{
	int	l;

	Affiche_trace (level, fonction, "%s", text);

	for (l = 0; l < 3; l++) {

		Affiche_trace (level, fonction, "% 1.4e % 1.4e % 1.4e", 
			mat [l][0],
			mat [l][1],
			mat [l][2]);
	}
}


/***************************************************************************************************
 *
 *	Produit de 2 matrices : dst =  m1 x m2 
 *	--------------------------------------
 */
void	Matrix_product (M33 m1, M33 m2, M33 dst)
{
	int		l, c, k;

	for (l = 0; l < 3; l++) {

		for (c = 0; c < 3; c++) {

			double	sum = 0.0;

			for (k = 0; k < 3; k++) sum += m1 [l][k] * m2 [k][c];

			dst [l][c] = sum;
		}
	}
}


/***************************************************************************************************
 *
 *	Matrice transposee : dst = transpose (src)
 *	------------------------------------------
 */
void	Matrix_transpose (M33 src, M33 dst)
{
	int		i, j;

	for (i = 0; i < 3; i++) 
		for (j = 0; j < 3; j++)
			dst [i][j] = src [j][i];
}


/***************************************************************************************************
 *
 *	Matrice inverse : dst = inverse (src)
 *	-------------------------------------
 */
t_err	Matrix_inverse (M33 src, M33 dst)
{
	char *		fonction = FNAME ("Matrix_inverse");
	t_err		error = OK;
	double		deter;

	deter	= src [0][0] * (src [1][1] * src [2][2] - src [1][2] * src [2][1])
		+ src [0][1] * (src [1][2] * src [2][0] - src [1][0] * src [2][2])
		+ src [0][2] * (src [1][0] * src [2][1] - src [1][1] * src [2][0]);

	if (fabs (deter) < EPSILON) {

		Affiche_erreur (fonction, "Determinant nul, inversion matrice impossible");
		error = ERROR;
		dst [0][0] = dst [1][1] = dst [2][2] = 1.0;
		dst [1][0] = dst [2][1] = dst [0][2] = 0.0;
		dst [2][0] = dst [0][1] = dst [1][2] = 0.0;
		goto EXIT;
	}

	dst [0][0] = (src [1][1] * src [2][2] - src [1][2] * src [2][1]) / deter;
	dst [1][0] = (src [1][2] * src [2][0] - src [1][0] * src [2][2]) / deter;
	dst [2][0] = (src [1][0] * src [2][1] - src [1][1] * src [2][0]) / deter;

	dst [0][1] = (src [2][1] * src [0][2] - src [2][2] * src [0][1]) / deter;
	dst [1][1] = (src [2][2] * src [0][0] - src [2][0] * src [0][2]) / deter;
	dst [2][1] = (src [2][0] * src [0][1] - src [2][1] * src [0][0]) / deter;

	dst [0][2] = (src [0][1] * src [1][2] - src [0][2] * src [1][1]) / deter;
	dst [1][2] = (src [0][2] * src [1][0] - src [0][0] * src [1][2]) / deter;
	dst [2][2] = (src [0][0] * src [1][1] - src [0][1] * src [1][0]) / deter;

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Matrice de rotation par rapport a X d'un angle donne
 *	----------------------------------------------------
 */
void	Matrix_rotation_X_axis (double angle, M33 mat)
{
	mat [0][0] = 1.0;
	mat [1][0] = 0.0;
	mat [2][0] = 0.0;

	mat [0][1] = 0.0;
	mat [1][1] = cos (angle);
	mat [2][1] = - sin (angle);

	mat [0][2] = 0.0;
	mat [1][2] = sin (angle);
	mat [2][2] = cos (angle);
}


/***************************************************************************************************
 *
 *	Matrice de rotation par rapport a Y d'un angle donne
 *	----------------------------------------------------
 */
void	Matrix_rotation_Y_axis (double angle, M33 mat)
{
	mat [0][0] = cos (angle);
	mat [1][0] = 0.0;
	mat [2][0] = sin (angle);

	mat [0][1] = 0.0;
	mat [1][1] = 1.0;
	mat [2][1] = 0.0;

	mat [0][2] = - sin (angle);
	mat [1][2] = 0.0; 
	mat [2][2] = cos (angle);
}


/***************************************************************************************************
 *
 *	Matrice de rotation par rapport a Z d'un angle donne 
 *	----------------------------------------------------
 */
void	Matrix_rotation_Z_axis (double angle, M33 mat)
{
	mat [0][0] = cos (angle);
	mat [1][0] = - sin (angle); 
	mat [2][0] = 0.0;

	mat [0][1] = sin (angle); 
	mat [1][1] = cos (angle); 
	mat [2][1] = 0.0;

	mat [0][2] = 0.0; 
	mat [1][2] = 0.0; 
	mat [2][2] = 1.0; 
}


/***************************************************************************************************
 *
 *	Diagonalisation matrice par la methode de Jacobi
 *	------------------------------------------------
 *	Cette fonction calcule les valeurs propres et les vecteurs propres d'un matrice symetrique
 *	- in	matrice d'entree symetrique 
 *	- V	vecteur propre
 *	- out	matrice des valeurs propres
 */
t_err	Diagonalisation_Jacobi (M33 in, V3 V, M33 out)
{
	char *		fonction = FNAME ("Diagonalisation_Jacobi");
	t_err		error = OK;
	int		i, j, k, iter;
	V3		B, Z;
	double		somme, tresh, h, t, g, c, s, tau, theta;

	/* 	Initialisation de out avec la matrice unite, de V et B avec la diagonale de in
	 */
	for (i = 0; i < 3; i++) {

		for (j = 0; j < 3; j++) out [i][j] = (i == j) ? 1.0 : 0.0;
		B [i] = in [i][i];
		V [i] = in [i][i];
		Z [i] = 0.0;
	}

	iter	= 0;

	while (iter++ < MAX_ITER_JACOBI) {

		Affiche_trace (5, fonction, "Iteration %d  V = % 1.6E % 1.6E % 1.6E",
			iter, V [0], V [1], V [2]);

		/* 	Somme des valeurs absolues des termes sub-diagonaux
		 */
		somme = fabs (in [0][1]) + fabs (in [0][2]) + fabs (in [1][2]);

		if (fabs (somme) < EPSILON) break; 	/* La diagonalisation est termine */

		/* 	calcul du seuil
		 */
		tresh = (iter < 4) ? (0.2 * somme / 9.0) : 0.0;

		for (i = 0; i < 3 - 1; i++) {

			for (j = i + 1; j < 3; j++) {

				g = 100 * fabs (in [i][j]);

				/* 	Apres 4 tours de boucle, on saute la rotation 
				 *	si le terme extra-diagonal est trop petit
				 */
				if (iter > 4 && fabs (V [i]) + g == fabs (V [i]) && fabs (V [j]) + g == fabs (V [j])) {

					in [i][j] = 0;
				}
				else if (fabs (in [i][j]) > tresh) {
					h = V [j] - V [i];
					if (fabs (h) + g == fabs (h))
						t = in [i][j] / h;      /* t est egal a 1/(2 xtheta) */
					else {	
						theta = 0.5 * h / in [i][j];
						t = 1 / (fabs(theta) + sqrt(1 + theta * theta));
						if (theta < 0) t = -t;
					}
					c	= 1 / sqrt(1 + t * t);
					s	= t * c;
					tau	= s / (1 + c);
					h	= t * in [i][j];
					Z [i]	= Z [i] - h;
					Z [j]	= Z [j] + h;
					V [i]	= V [i] - h;
					V [j]	= V [j] + h;
					in [i][j]= 0.0;

					/*	Cas des rotations 0 <= k < p
					 */
					for (k = 0; k < i; k++) {

						g = in [k][i];
						h = in [k][j];
						in [k][i]= g - s * (h + g * tau);
						in [k][j]= h + s * (g - h * tau);
					}

					/* 	Cas des rotations p < k < q
					 */
					for (k = i + 1; k < j; k++) {

						g = in [i][k];
						h = in [k][j];
						in [i][k]= g - s * (h + g * tau);
						in [k][j]= h + s * (g - h * tau);
					}

					/*	Cas des rotations q < k < 3
					 */
					for (k = j + 1; k < 3; k++) {

						g = in [i][k];
						h = in [j][k];
						in [i][k]= g - s * (h + g * tau);
						in [j][k]= h + s * (g - h * tau);
					}

					for (k = 0; k < 3; k++) {

						g = out [k][i];
						h = out [k][j];
						out [k][i]= g - s * (h + g * tau);
						out [k][j]= h + s * (g - h * tau);
					}
				}
			}
		}
		/*	Mise a jour de V avec la somme des t * a [i][j] et reinitialisation Z
		 */
		for (i = 0; i < 3; i++) {

			B [i] = B [i] + Z [i];
			V [i] = B [i];
			Z [i] = 0;
		}
	}

	if (iter >= MAX_ITER_JACOBI) error = ERROR;

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Isole la valeur la plus differente des trois coordonnees d'un vecteur, 
 *	puis la plus petite et la plus grande parmi celles restantes
 *	------------------------------------------------------------
 */
t_err	Find_para (V3 vector, double * para, double * smaller, double * greater)
{
	char *		fonction = FNAME ("Find_para");
	t_err		error = OK;
	double		d0, d1, d2;
	int		indice, i, j;

	d0 = fabs (vector [0] - vector [1]);
	d1 = fabs (vector [1] - vector [2]);
	d2 = fabs (vector [2] - vector [0]);
	
	/*	Determine l'indice de la coordonnee la plus differente
	 */
	if (d0 < d1 && d0 < d1)		indice = 2;
	else if (d1 < d0 && d1 < d2)	indice = 0;
	else				indice = 1;

	* para = vector [indice];

	/*	Determine les indices des deux autres coordonnees
	 */
	i = (indice + 1) % 3;
	j = (indice + 2) % 3;

	if (vector [i] < vector [j]) {

		* smaller = vector [i];
		* greater = vector [j];
	}
	else {	* smaller = vector [j];
		* greater = vector [i];
	}
EXIT:	return error;
}
