/***************************************************************************************************
 *
 *	Fichier	: $RCSfile: INTERFACE.c,v $
 *
 *	Version	: $Revision: 1.32 $
 *
 *	Auteur	: $Author: barthe $
 *
 *	Date	: $Date: 2012/09/11 17:43:35 $
 *
 *	==========================================================================================
 *
 *	Module charge de la gestion des divers fichiers Double Star
 */

#define	MODULE_NAME	"INTERFACE"

#include "DSPLIB.h"

t_symbol dsp_data_source [] = {

	{ GSE,		"GSE",		"Ground Segment Equipment data"	},
	{ CHN,		"CHN",		"Level 1 Commissioning Data"	},
	{ L1R,		"L1R",		"Real-time level 1 data"	},
	{ L1P, 		"L1P",		"Playback level 1 data"		},
	{ L1,		"L1",		"Reconstitued level 1 data"	},
	{ L1D,		"L1D",		"Decommuted Level 1 data"	},
	{ L2,		"L2",		"Level 2 data"			},
	{ L3,		"L3",		"Level 3 data"			},
	{ EOF,		"???",		"Unknown file source"		}
};

t_symbol dsp_file_ident [] = {

	{ HIA_00,	"HIA_00",	"HIA product 00"		},
	{ HIA_01,	"HIA_01",	"HIA product 01"		},
	{ HIA_02,	"HIA_02",	"HIA onboard-momenta"		},
	{ HIA_04,	"HIA_04",	"HIA onboard momenta"		},
	{ HIA_05,	"HIA_05",	"HIA product 05"		},
	{ HIA_06,	"HIA_06",	"HIA product 06"		},
	{ HIA_07,	"HIA_07",	"HIA product 07"		},
	{ HIA_08,	"HIA_08",	"HIA product 08"		},
	{ HIA_09,	"HIA_09",	"HIA product 09"		},
	{ HIA_10,	"HIA_10",	"HIA product 10"		},
	{ HIA_11,	"HIA_11",	"HIA product 11"		},
	{ HIA_12,	"HIA_12",	"HIA product 12"		},
	{ HIA_13,	"HIA_13",	"HIA product 13"		},
	{ HIA_14,	"HIA_14",	"HIA product 14"		},
	{ HIA_15,	"HIA_15",	"HIA product 15"		},
	{ HIA_16,	"HIA_16",	"HIA product 16"		},
	{ HIA_17,	"HIA_17",	"HIA product 17"		},
	{ HIA_18,	"HIA_18",	"HIA product 18"		},
	{ HIA_19,	"HIA_19",	"HIA product 19"		},
	{ HIA_20,	"HIA_20",	"HIA product 20"		},
	{ HIA_21,	"HIA_21",	"HIA product 21"		},
	{ HIA_22,	"HIA_22",	"HIA product 22"		},
	{ HIA_23,	"HIA_23",	"HIA product 23"		},
	{ HIA_24,	"HIA_24",	"HIA product 24"		},
	{ HIA_61,	"HIA_61",	"HIA product 61"		},
	{ HIA_62,	"HIA_62",	"HIA product 62"		},
	{ NSD,		"NSD",		"HIA Normal Science Data"	},
	{ HKD,		"HKD",		"HIA Housekeeping Data"		},
	{ HPD,		"HPD",		"HIA Housekeeping Definition"	},
	{ SC_HKD,	"SC_HKD",	"Platform Housekeeping Data"	},
	{ SC_HPD,	"SC_HPD",	"Platform Housekeeping Definition" },
	{ STOF,		"STOF",		"Short term Orbit file"		},
	{ STEF,		"STEF",		"Short Term Event file"		},
	{ SATT,		"SATT",		"Spacecraft Attitude and Spin"	},
	{ TCAL,		"TCAL",		"Time Calibration file"		},
	{ CMDH,		"CMDH",		"Command History file"		},
	{ COVM,		"COVM",		"Covariance Matrix file"	},
	{ INFO,		"INFO",		"Application Info file"		},
	{ CDROM,	"CDROM",	"CDROM version information"	},
	{ PPD,		"PPD",		"Primary Parameters file"	},
	{ SPD,		"SPD",		"Summary Parameters file"	},
	{ DSDS,		"DSDS"	,	"DSDS WEB flat file"		},
	{ SR_GSE,	"SR_GSE",	"SR to GSE conversion file"	},
	{ TRACE,	"TRACE",	"Applicaion trace file"		},
	{ EOF,		"???",		"Unknown file ident"		}	
};

typedef struct {			/* Entree table des fichiers produits	*/

	t_data_source	source;		/* - origine des donnees		*/
	t_file_ident	ident;		/* - identificateur du fichier		*/
	char *		pattern;	/* - masque de recherche		*/
	FILE *		f_desc;		/* - descripteur Unix			*/
	t_filename	filename;	/* - nom du fichier			*/

}	t_prod_desc;


static	t_prod_desc	dsp_files [] = {

	/*****	Level 1 China commissioning ********************************/

	{ CHN,	HKD,	"????????_??????_????????_??????.hkd",		NULL },
	{ CHN,	NSD,	"????????_??????_????????_??????.nsd",		NULL },

	/*****	Level 1 Ground Segment Equipment ****************************/

	{ GSE,	HKD,	"????????.??????_????????.??????.hkd", 		NULL },
	{ GSE,	NSD,	"????????.??????_????????.??????.nsd",	 	NULL },

	/*****	Level 1 Real-time  DSP files ********************************/

	{ L1R,	HKD,	"HKD1_R/????????????_????????????_HH.RZ1",	NULL },

	/*****	Level 1 Reconstitued  DSP files *****************************/

	{ L1P,	HKD,	"HKD1_P/????????????_????????????_HH.PZ1",	NULL },
	{ L1P,	NSD,	"NSD1_P/????????????_????????????_HN.PZ1",	NULL },
	{ L1P,	SATT,	"AUX1_P/????????????_????????????_GA.NZ1",	NULL },

	/*****	Level 1 Playback   DSP files ********************************/

	{ L1,	STOF,	"AUX_[ABC]/%sBA.N[A-Z]1",			NULL },
	{ L1,	STEF,	"AUX_[ABC]/%sTA.N[A-Z]1",			NULL },
	{ L1,	SATT,	"AUX_[ABC]/%sGA.N[A-Z]1",			NULL },
	{ L1,	TCAL,	"AUX_[ABC]/%sLA.N[A-Z]1",			NULL },
	{ L1,	CMDH,	"AUX_[ABC]/%sIA.N[A-Z]1",			NULL },
	{ L1,	NSD,	"NSD_[ABC]/%sHN.N[A-Z]1",			NULL },
	{ L1, 	HKD,	"HKD_[ABC]/%sHH.N[A-Z]1",			NULL },
	{ L1, 	SC_HPD,	"HKD_[ABC]/%sSD.N[A-Z]1",			NULL },
	{ L1, 	SC_HKD,	"HKD_[ABC]/%sSH.N[A-Z]1",			NULL },

	/*****	Level 1 Decommuted DSP files ********************************/

	{ L1D,	HIA_00,	"T1_LD_HIA_%s_V%s_P00xxxxxxx.sci",		NULL },
	{ L1D,	HIA_01,	"T1_LD_HIA_%s_V%s_P01xxxxxxx.sci",		NULL },
	{ L1D,	HIA_02,	"T1_LD_HIA_%s_V%s_P02xxxxxxx.sci",		NULL },
	{ L1D,	HIA_04,	"T1_LD_HIA_%s_V%s_P04xxxxxxx.sci",		NULL },
	{ L1D,	HIA_05,	"T1_LD_HIA_%s_V%s_P05xxxxxxx.sci",		NULL },
	{ L1D,	HIA_06,	"T1_LD_HIA_%s_V%s_P06xxxxxxx.sci",		NULL },
	{ L1D,	HIA_07,	"T1_LD_HIA_%s_V%s_P07xxxxxxx.sci",		NULL },
	{ L1D,	HIA_08,	"T1_LD_HIA_%s_V%s_P08xxxxxxx.sci",		NULL },
	{ L1D,	HIA_09,	"T1_LD_HIA_%s_V%s_P09xxxxxxx.sci",		NULL },
	{ L1D,	HIA_10,	"T1_LD_HIA_%s_V%s_P10xxxxxxx.sci",		NULL },
	{ L1D,	HIA_11,	"T1_LD_HIA_%s_V%s_P11xxxxxxx.sci",		NULL },
	{ L1D,	HIA_12,	"T1_LD_HIA_%s_V%s_P12xxxxxxx.sci",		NULL },
	{ L1D,	HIA_13,	"T1_LD_HIA_%s_V%s_P13xxxxxxx.sci",		NULL },
	{ L1D,	HIA_14,	"T1_LD_HIA_%s_V%s_P14xxxxxxx.sci",		NULL },
	{ L1D,	HIA_15,	"T1_LD_HIA_%s_V%s_P15xxxxxxx.sci",		NULL },
	{ L1D,	HIA_16,	"T1_LD_HIA_%s_V%s_P16xxxxxxx.sci",		NULL },
	{ L1D,	HIA_17,	"T1_LD_HIA_%s_V%s_P17xxxxxxx.sci",		NULL },
	{ L1D,	HIA_18,	"T1_LD_HIA_%s_V%s_P18xxxxxxx.sci",		NULL },
	{ L1D,	HIA_19,	"T1_LD_HIA_%s_V%s_P19xxxxxxx.sci",		NULL },
	{ L1D,	HIA_20,	"T1_LD_HIA_%s_V%s_P20xxxxxxx.sci",		NULL },
	{ L1D,	HIA_21,	"T1_LD_HIA_%s_V%s_P21xxxxxxx.sci",		NULL },
	{ L1D,	HIA_22,	"T1_LD_HIA_%s_V%s_P22xxxxxxx.sci",		NULL },
	{ L1D,	HIA_23,	"T1_LD_HIA_%s_V%s_P23xxxxxxx.sci",		NULL },
	{ L1D,	HIA_24,	"T1_LD_HIA_%s_V%s_P24xxxxxxx.sci",		NULL },
	{ L1D,	HIA_61,	"T1_LD_HIA_%s_V%s_P61xxxxxxx.sci",		NULL },
	{ L1D,	HIA_62,	"T1_LD_HIA_%s_V%s_P62xxxxxxx.sci",		NULL },
	{ L1D,	HKD,	"T1_LD_HIA_%s_V%s_HKDxxxxxxx.hsk",		NULL },
	{ L1D,	HPD,	"T1_LD_HIA_%s_V%s_HPDxxxxxxx.hsk",		NULL },
	{ L1D,	SC_HKD,	"T1_LD_HIA_%s_V%s_HKDSATxxxx.hsk",		NULL },
	{ L1D,	SC_HPD,	"T1_LD_HIA_%s_V%s_HPDSATxxxx.hsk",		NULL },
	{ L1D,	STOF,	"T1_LD_HIA_%s_V%s_STOFxxxxxx.aux",		NULL },
	{ L1D,	STEF,	"T1_LD_HIA_%s_V%s_STEFxxxxxx.aux",		NULL },
	{ L1D,	SATT,	"T1_LD_HIA_%s_V%s_SATTxxxxxx.aux",		NULL },
	{ L1D,	TCAL,	"T1_LD_HIA_%s_V%s_TCALxxxxxx.aux",		NULL },
	{ L1D,	CMDH,	"T1_LD_HIA_%s_V%s_CMDHxxxxxx.aux",		NULL },
	{ L1D,	COVM,	"T1_LD_HIA_%s_V%s_COVMxxxxxx.aux",		NULL },
	{ L1D,	INFO,	"T1_LD_HIA_%s_V%s_INFOxxxxxx.asc",		NULL },
	{ L1D, 	CDROM,	"T1_LD_HIA_%s_V%s_CDROMxxxxx.asc",		NULL },
	{ L1D,	TRACE,	"T1_LD_HIA_%s_V%s_TRACExxxxx.log",		NULL },

	/*****	Level 2 DSP files *******************************************/

	{ L2,	HKD,	"T1_L2_HIA_%s_V%s_HKDxxxxxxx.hsk",		NULL },
	{ L2, 	STOF,	"T1_L2_HIA_%s_V%s_ORBITxxxxx.aux",		NULL },
	{ L2,	HIA_00,	"T1_L2_HIA_%s_V%s_P00xxxxxxx.sci",		NULL },
	{ L2,	HIA_01,	"T1_L2_HIA_%s_V%s_P01xxxxxxx.sci",		NULL },
	{ L2,	HIA_02,	"T1_L2_HIA_%s_V%s_P02xxxxxxx.sci",		NULL },
	{ L2,	HIA_04,	"T1_L2_HIA_%s_V%s_P04xxxxxxx.sci",		NULL },
	{ L2,	HIA_05,	"T1_L2_HIA_%s_V%s_P05xxxxxxx.sci",		NULL },
	{ L2,	HIA_06,	"T1_L2_HIA_%s_V%s_P06xxxxxxx.sci",		NULL },
	{ L2,	HIA_07,	"T1_L2_HIA_%s_V%s_P07xxxxxxx.sci",		NULL },
	{ L2,	HIA_08,	"T1_L2_HIA_%s_V%s_P08xxxxxxx.sci",		NULL },
	{ L2,	HIA_09,	"T1_L2_HIA_%s_V%s_P09xxxxxxx.sci",		NULL },
	{ L2,	HIA_10,	"T1_L2_HIA_%s_V%s_P10xxxxxxx.sci",		NULL },
	{ L2,	HIA_11,	"T1_L2_HIA_%s_V%s_P11xxxxxxx.sci",		NULL },
	{ L2,	HIA_12,	"T1_L2_HIA_%s_V%s_P12xxxxxxx.sci",		NULL },
	{ L2,	HIA_13,	"T1_L2_HIA_%s_V%s_P13xxxxxxx.sci",		NULL },
	{ L2,	HIA_14,	"T1_L2_HIA_%s_V%s_P14xxxxxxx.sci",		NULL },
	{ L2,	HIA_15,	"T1_L2_HIA_%s_V%s_P15xxxxxxx.sci",		NULL },
	{ L2,	HIA_16,	"T1_L2_HIA_%s_V%s_P16xxxxxxx.sci",		NULL },
	{ L2,	HIA_17,	"T1_L2_HIA_%s_V%s_P17xxxxxxx.sci",		NULL },
	{ L2,	HIA_18,	"T1_L2_HIA_%s_V%s_P18xxxxxxx.sci",		NULL },
	{ L2,	HIA_19,	"T1_L2_HIA_%s_V%s_P19xxxxxxx.sci",		NULL },
	{ L2,	HIA_20,	"T1_L2_HIA_%s_V%s_P20xxxxxxx.sci",		NULL },
	{ L2,	HIA_21,	"T1_L2_HIA_%s_V%s_P21xxxxxxx.sci",		NULL },
	{ L2,	HIA_22,	"T1_L2_HIA_%s_V%s_P22xxxxxxx.sci",		NULL },
	{ L2,	HIA_23,	"T1_L2_HIA_%s_V%s_P23xxxxxxx.sci",		NULL },
	{ L2,	HIA_24,	"T1_L2_HIA_%s_V%s_P24xxxxxxx.sci",		NULL },
	{ L2,	DSDS,	"T1_L2_HIA_%s_V%s_DSDSWEBxxx.asc",		NULL },
	{ L2,	SR_GSE,	"T1_L2_HIA_%s_V%s_SR_GSExxxx.asc",		NULL },
	{ L2,	TRACE,	"T1_L2_HIA_%s_V%s_TRACExxxxx.log",		NULL },

	/*****	Level 3 DSP files *******************************************/

	{ L3,	PPD,	"T1_L3_HIA_%s_V%s_PPDxxxxxxx.flt",		NULL },
	{ L3,	SPD,	"T1_L3_HIA_%s_V%s_SPDxxxxxxx.flt",		NULL },
	{ L3,	DSDS,	"T1_L3_HIA_%s_V%s_DSDSWEBxxx.asc",		NULL },
	{ L3,	TRACE,	"T1_L3_HIA_%s_V%s_TRACExxxxx.log",		NULL },

	/*****	End of table ************************************************/

	{ EOF,	EOF,	NULL,						NULL }
};	


/****************************************************************************************************
 *
 *	Verifie source des donnees
 *	--------------------------
 */
t_err	Check_data_source (char * str, t_data_source * source)
{
	char *		fonction = FNAME ("Check_data_source");
	t_err		error = OK;
	t_symbol *	symbol;

	* source = EOF;

	if (Unknown_symbol (symbol = Search_symbol_key (dsp_data_source, str))) {

		Affiche_erreur (fonction, "Origine donnees incorrecte : %s", str);
		error = ERROR;
		goto EXIT;
	}

	Affiche_trace (1, fonction, "%s : %s", symbol->key, symbol->lib);

	* source = symbol->val;

EXIT:	return error;	
}


/****************************************************************************************************
 *
 *	Verification que le fichier correspond a la date de traitement
 *	--------------------------------------------------------------
 *
 *	Pour certains types de fichiers, selectionner les fichiers dont la periode recouvre au moins
 *	pour partie la date de traitement
 */
static	int	File_match_period (t_data_source source, t_date date, char * fichier)
{
	char *		fonction = FNAME ("File_match_period");
	char *		ptr = basename (fichier);
	int		date_deb, date_fin, date_jour;
	
	switch (source) {

	case L1P :	;
	case L1R :	date_jour = (date.annee % 100) * 10000 + date.mois * 100 + date.jour;

			if (sscanf (ptr, "%6d%*6d_%6d%*6d_", & date_deb, & date_fin) != 2) return FALSE;

			return (date_deb > date_jour || date_fin < date_jour) ? FALSE : TRUE;

	case GSE :	date_jour = (date.annee * 10000) + date.mois * 100 + date.jour;

			if (sscanf (ptr, "%8d.%*6d_%8d_%*6d_", & date_deb, & date_fin) != 2) return FALSE;

			return (date_deb > date_jour || date_fin < date_jour) ? FALSE : TRUE;

	case CHN :	date_jour = (date.annee * 10000) + date.mois * 100 + date.jour;

			if (sscanf (ptr, "%8d_%*6d_%8d_%*6d_", & date_deb, & date_fin) != 2) return FALSE;

			return (date_deb > date_jour || date_fin < date_jour) ? FALSE : TRUE;

	default  :	Affiche_erreur (fonction, "Source %s incompatible", 
				Search_symbol_value (dsp_data_source, source)->key);

			return FALSE;
	}
}


/****************************************************************************************************
 *
 *	Retourne un chaine contenant les references du fichier
 *	------------------------------------------------------
 */
static	char *	File_reference (t_data_source source, t_file_ident ident)
{
	static char	buffer [40];
	t_symbol *	s = Search_symbol_value (dsp_data_source, source);
	t_symbol *	i = Search_symbol_value (dsp_file_ident,  ident);

	sprintf (buffer, "%s %s", s->key, i->key);

	return buffer;	
}


/****************************************************************************************************
 *
 *	Recherche une entree de la table des fichiers Double Star
 *	---------------------------------------------------------
 */
static	t_prod_desc * Get_DSP_file_entry (t_data_source source, t_file_ident ident)
{
	t_prod_desc *	ptr;

	for (ptr = dsp_files; ptr->source != EOF; ptr ++) {

		if (ptr->source == source && ptr->ident == ident) return ptr;
	}
	return (t_prod_desc *) NULL;
}


/****************************************************************************************************
 *
 *	Ouverture d'un fichier produit
 *	------------------------------
 */
FILE *	Open_DSP_file (t_data_source source, t_file_ident ident, char * mode)
{
	char *		fonction = FNAME ("Open_DSP_file");
	FILE *		file = NULL;
	t_prod_desc *	ptr = NULL;

	if ((ptr = Get_DSP_file_entry (source, ident)) == NULL) {

		Affiche_trace (1, fonction, "Erreur : Produit %s inconnu", File_reference (source, ident));
		goto EXIT;
	}

	if ((file = fopen (ptr->filename, mode)) == NULL) {

		Affiche_erreur (fonction, "Ouverture %s impossible", ptr->filename);
		Affiche_erreur (fonction, "%s", strerror (errno));
		goto EXIT;
	}
	
	ptr->f_desc = file;

EXIT:	return file;
}


/****************************************************************************************************
 *
 *	Re-ouverture d'un fichier produit
 *	---------------------------------
 */
FILE *	Reopen_DSP_file (t_data_source source, t_file_ident ident, char * mode)
{
	char *		fonction = FNAME ("Reopen_DSP_file");
	FILE *		file = NULL;
	t_prod_desc *	ptr = NULL;

	if ((ptr = Get_DSP_file_entry (source, ident)) == NULL) {

		Affiche_trace (1, fonction, "Erreur : Produit %s inconnu", File_reference (source, ident));
		goto EXIT;
	}

	/*	Si le fichier est deja ouvert on retourne son descripteur
	 */
	if (ptr->f_desc != NULL) return ptr->f_desc;

	/*	Sinon, on l'ouvre et on met a jour la table des descripteurs
	 */
	if ((file = fopen (ptr->filename, mode)) == NULL) {

		Affiche_erreur (fonction, "Ouverture %s impossible", ptr->filename);
		Affiche_erreur (fonction, "%s", strerror (errno));
		goto EXIT;
	}
	
	ptr->f_desc = file;

EXIT:	return file;
}


/****************************************************************************************************
 *
 *	Retourne la liste des fichiers pour une source, une date et un type donnes
 *	--------------------------------------------------------------------------
 */
t_err	Search_DSP_files (t_data_source source, t_date date, t_file_ident ident, t_list * list)
{
	char *		fonction = FNAME ("Search_DSP_files");
	t_err		error = OK;	
	t_prod_desc *	ptr = NULL;
	int		i, nbre;

	if ((ptr = Get_DSP_file_entry (source, ident)) == NULL) {

		Affiche_trace (1, fonction, "Erreur : Produit %s inconnu", File_reference (source, ident));
		error = WARNING;
		goto EXIT;
	}

	if (Erreur (error = Search_files (ptr->filename, & nbre))) goto EXIT;

	Affiche_trace (5, fonction, "%3d : %s", nbre, ptr->filename);

	switch (source) {

	case L1 :	/* Prendre le fichier de version la plus recente */

			if (nbre > 0) error = Add_item (list, Get_filename (nbre -1));

			break;

	case GSE:	;
	case CHN:	;
	case L1P:	;
	case L1R:	/* Selectionner les fichiers dont la periode recouvre la date de traitement */

			for (i = 0; i < nbre; i ++) {

				if (File_match_period (source, date, Get_filename (i))) {

					if (Erreur (error = Add_item (list, Get_filename (i)))) break;
				}
			}	
			break;

	defaut :	Affiche_erreur (fonction, "Source %s incompatible", 
				Search_symbol_value (dsp_data_source, source)->key);
			error = ERROR;
	}

	Search_free ();

EXIT:	return error;
}


/****************************************************************************************************
 *
 *	Retourne le nom du fichier produit specifie
 *	-------------------------------------------
 */
char *  Product_filename (t_data_source source, t_file_ident ident)
{
	t_prod_desc *	ptr = Get_DSP_file_entry (source, ident);

	if (ptr == NULL) return "UNKNOWN";

	return ptr->filename;
}


/****************************************************************************************************
 *
 *	Redirection stdout sur fichier de trace
 *	---------------------------------------
 */
t_err   Redirect_stdout (t_data_source source)
{
	char *		fonction = FNAME ("Redirect_stdout");
	t_err		error = OK;
	FILE *		sortie = NULL;

	if ((sortie = Open_DSP_file (source, TRACE, "w")) == NULL) {

		Affiche_erreur (fonction, "Redirection sur %s impossible", Product_filename (source, TRACE));
		error = ERR_fopen;
		goto EXIT;
	}

	Affiche_trace (1, fonction, "Redirection stdout sur %s", Product_filename (source, TRACE));

	if (dup2 (fileno (sortie), STDOUT_FILENO) == -1) {

		Affiche_erreur (fonction, "Erreur dup2 () : %s", strerror (errno));
		error = ERROR;
		goto EXIT;
	}

EXIT:	return error;
}


/****************************************************************************************************
 *
 *	Retourne le repertoire correspond au type des donnees
 *	-----------------------------------------------------
 */
t_err	Compute_directory (t_data_source source, t_date date, char * directory)
{
	char *		fonction = FNAME ("Compute_directory");
	t_err		error = OK;
	char		amj [9];
	int		nbre;
	t_filename	pattern;

	sprintf (amj, "%04d%02d%02d", date.annee, date.mois, date.jour);

	switch (source) {

	case GSE :	sprintf (directory, "%s/GSE", dsp_root);
			break;
	case CHN :	sprintf (directory, "%s/CSSAR", dsp_root);
			break;
	case L1R :	sprintf (directory, "%s/DATA/L1P/TC1_R", dsp_root);
			break;
	case L1P :	sprintf (directory, "%s/DATA/L1P/TC1_P", dsp_root);
			break;
	case L1  :	sprintf (pattern, "%s/DATA/L1/??????_1_[1-3][A-Z]/%s_[A-C]", dsp_root, amj + 2);

			if (Erreur (error = Search_files (pattern, & nbre))) goto EXIT;

			if (nbre > 0) strcpy (directory, Get_filename (nbre -1));

			Search_free ();
			break;
	case L1D :	sprintf (directory, "%s/DATA/L1D/%s", dsp_root, amj);
			break;
	case L2  :	sprintf (directory, "%s/DATA/L2/%s", dsp_root, amj);
			break;
	case L3  :	sprintf (directory, "%s/DATA/L3/%s", dsp_root, amj);
			break;
	default  :	error = ERROR;
			strcpy (directory, "");
			goto EXIT;
	}
EXIT:	return error;
}


/****************************************************************************************************
 *
 *	Creation repertoire destination ou purge si celui-ci n'est pas vide
 *	-------------------------------------------------------------------
 */
t_err	Clear_directory (t_data_source source, t_date date)
{
	char *		fonction = FNAME ("Clear_directory");
	t_err		error = OK;
	struct stat	info;
	int		i, nbre;
	t_filename	directory;
	t_filename	masque;

	if (Erreur (error = Compute_directory (source, date, directory))) goto EXIT;

	if (stat (directory, & info) == -1) {

		if (errno == ENOENT) {

			if (mkdir (directory, 0775) == -1) {

				Affiche_erreur ("Creation repertoire %s impossible", directory);
				Affiche_erreur (fonction, "Erreur stat () : %s", strerror (errno));
				error = ERROR;
				goto EXIT;
			}

			Affiche_trace (1, fonction, "Creation repertoire %s", directory);
			goto EXIT;
		}
		else {	Affiche_erreur (fonction, "Acces repertoire %s impossible", directory);
			Affiche_erreur (fonction, "Erreur stat () : %s", strerror (errno));
			error = ERROR;
			goto EXIT;
		}
	}

	if (S_ISDIR (info.st_mode) == FALSE) {

		Affiche_erreur (fonction, "%s n'est pas un repertoire", directory);
		error = ERROR;
		goto EXIT;
	}

	sprintf (masque, "%s/*", directory);

	if (Erreur (error = Search_files (masque, & nbre))) goto EXIT;
	
	for (i = 0; i < nbre; i ++) {

		if (unlink (Get_filename (i)) == -1) {

			Affiche_erreur (fonction, "Suppression %s impossible", Get_filename (i));
			Affiche_erreur (fonction, "Erreur unlink () : %s", strerror (errno));
			error = ERROR;
			goto EXIT;
		}
	}
	
	Search_free ();

EXIT:	return error;
}


/****************************************************************************************************
 *
 *	Recherche le repertoire des donnees L1 en fonction de la version desiree
 *	------------------------------------------------------------------------
 */
t_err	Compute_L1_directory (t_date date, char version, char * directory)
{
	char *		fonction = FNAME ("Compute_L1_directory");
	t_err		error = OK;
	t_filename	pattern;	
	int		nbre;
	char		amj [9];

	if (version == '-') version = '?';

	strcpy (directory, "");

	sprintf (amj, "%04d%02d%02d", date.annee, date.mois, date.jour);

	sprintf (pattern, "%s/DATA/L1/??????_1_[1-3]%c/%s_[A-C]", 
		dsp_root, 
		version, 
		amj + 2);

	if (Erreur (error = Search_files (pattern, & nbre)) || nbre < 1) {

		Affiche_erreur (fonction, "%s", pattern);
		Affiche_erreur (fonction, "Pas de répertoire correspondant aux données L1");
		error = ERROR;	
		goto EXIT;
	}

	if (version == '?') {

		if (nbre < 2) {

			Affiche_erreur (fonction, "Version N-1 indisponible");
			error = ERROR;
			goto EXIT;
		}
		else	strcpy (directory, Get_filename (nbre -2));
	}
	else	strcpy (directory, Get_filename (nbre -1));

EXIT:	Search_free ();
	return error;
}


/****************************************************************************************************
 *
 *	Determine le masque de recherche des fichiers de niveau L1 en fonction de la version desiree
 *	--------------------------------------------------------------------------------------------
 */
t_err	Compute_L1_filemasks (t_date date, char version)
{
	char *		fonction = FNAME ("Compute_L1_filemasks");
	t_err		error = OK;
	t_filename	directory;
	t_prod_desc *	ptr;
	char		amj [9];
	
	sprintf (amj, "%04d%02d%02d", date.annee, date.mois, date.jour);

	if (Erreur (error = Compute_L1_directory (date, version, directory))) goto EXIT;

	Affiche_trace (1, fonction, "L1 directory : %s", directory);

	for (ptr = dsp_files; ptr->source != EOF; ptr++) {

		t_filename	buffer;
		int		nbre;

		if (ptr->source != L1) continue;

		sprintf (buffer, "%s/%s", directory, ptr->pattern);

		sprintf (ptr->filename, buffer, amj + 2);
	}

EXIT:	return error;
}

/****************************************************************************************************
 *
 *	Determine le masque de recherche des fichiers pour une source et date donnees
 *	-----------------------------------------------------------------------------
 */
t_err	Compute_DSP_filemasks (t_data_source source, t_date date)
{
	char *		fonction = FNAME ("Compute_DSP_filemasks");
	t_err		error = OK;	
	char 		amj [9];
	t_filename	directory;
	t_prod_desc *	ptr;
	t_symbol *	src = Search_symbol_value (dsp_data_source, source);

	if (Erreur (error = Compute_directory (source, date, directory))) goto EXIT;

	Affiche_trace (1, fonction, "%s directory : %s", src->key, directory);

	sprintf (amj, "%04d%02d%02d", date.annee, date.mois, date.jour);

	for (ptr = dsp_files; ptr->source != EOF; ptr ++) {

		t_filename	mask;

		if (ptr->source != source) continue;

		switch (source) {
		
		case GSE:	sprintf (ptr->filename, "%s/%s", directory, ptr->pattern);
				break;
		case CHN:	sprintf (ptr->filename, "%s/%s", directory, ptr->pattern);
				break;
		case L1R:	sprintf (ptr->filename, "%s/%s", directory, ptr->pattern);
				break;
		case L1P:	sprintf (ptr->filename, "%s/%s", directory, ptr->pattern);
				break;
		case L1:	sprintf (mask, "%s/%s", directory, ptr->pattern);
				sprintf (ptr->filename, mask, amj + 2);
				break;
		default:	break;
		}	
		
		Affiche_trace (5, fonction, "%s", ptr->filename);
	}
EXIT:	return error;
}


/****************************************************************************************************
 *
 *	Determine le nom complet des fichiers pour une source et une date donnee
 *	------------------------------------------------------------------------
 */
t_err	Compute_DSP_filenames (t_data_source source, t_date date, int version)
{
	char *		fonction = FNAME ("Compute_DSP_filenames");
	t_err		error = OK;
	t_prod_desc *	ptr;
	t_filename	directory;
	char		amj [9];
	char		vmask [3];
	t_symbol *	src = Search_symbol_value (dsp_data_source, source);

	if (Erreur (error = Compute_directory (source, date, directory))) goto EXIT;

	sprintf (amj, "%04d%02d%02d", date.annee, date.mois, date.jour);

	if (version == VMAX) 
		strcpy (vmask, "??");
	else 
		sprintf (vmask, "%02d", version);

	Affiche_trace (1, fonction, "%s directory : %s", src->key, directory);

	for (ptr = dsp_files; ptr->source != EOF; ptr++) {

		t_filename	buffer;
		int		nbre;

		if (ptr->source != source) continue;

		sprintf (buffer, "%s/%s", directory, ptr->pattern);

		sprintf (ptr->filename, buffer, amj, vmask);

		if (version == VMAX) {

			if (Erreur (error = Search_files (ptr->filename, & nbre))) goto EXIT;

			if (nbre > 0) strcpy (ptr->filename, Get_filename (nbre -1));

			Search_free ();
		}

		Affiche_trace (5, fonction, "%s", ptr->filename);
	}

EXIT:	return error;
}


/****************************************************************************************************
 *
 *	Retourne le numero de version des donnees brutes utilisees
 *	----------------------------------------------------------
 */
t_err	Get_raw_data_version (t_date date, char * version)
{
	char *		fonction = FNAME ("Get_raw_data_version");
	t_err		error = OK;
	FILE *		entree = NULL;
	t_filename	buffer;
	t_filename 	tmp;

	* version = ' ';

	if ((entree = Open_DSP_file (L1D, CDROM, "r")) == NULL) {

		Affiche_erreur (fonction, "Unable to open %s file", Key_from_val (dsp_file_ident, CDROM));
		error = ERR_fopen;
		goto EXIT;
	}

	if (Read_line (buffer, entree) != OK) {

		Affiche_erreur (fonction, "Unable to read %s content", Key_from_val (dsp_file_ident, CDROM));
		error = ERR_fopen;
		goto EXIT;
	}

	strcpy (tmp, dirname (buffer));

	* version = tmp [strlen(tmp) -1];
	
EXIT:	if (entree != NULL) fclose (entree);
	return error;
}
