/***************************************************************************************************
 *
 *	Fichier	: $RCSfile: HIA.c,v $
 *
 *	Version	: $Revision: 1.11 $
 *
 *	Auteur	: $Author: barthe $
 *
 *	Date	: $Date: 2009/11/25 14:02:01 $
 *
 *	==========================================================================================
 *
 *	Module charge de la lecture des fichiers de calibration HIA
 */
 
#define	MODULE_NAME	"HIA"

#include "DSPLIB.h"
#include "HIA.h"

#define	SECTION(s)	(strcmp (section, s) == 0)


/*	Coefficients de calibration utilises par HIA
 *	--------------------------------------------
 */
int		dsp_spin_accumulation		[NB_TLM_PRODUCT][NB_DSP_MODE];
float		dsp_phase_instr			[NB_SENSITIVITY];
float		dsp_geom_factor			[NB_SENSITIVITY];
float		dsp_anal_k_factor		[3];
float		dsp_anal_alpha_angle		[NB_SENSITIVITY];
float		dsp_anode_eff			[NB_SENSITIVITY][NB_ANODE][2];
float		dsp_dead_times			[2];
float		dsp_a				[NB_SENSITIVITY];
float		dsp_b				[NB_SENSITIVITY];
float		dsp_norm_e			[NB_SENSITIVITY];
float		dsp_norm_theta			[NB_SENSITIVITY];
float		dsp_MCP_fitting_param		[3][5];
t_energy_step	dsp_energy_sweep_table_16	[11][16];
t_energy_step	dsp_energy_sweep_table_31	[11][31];
t_energy_step	dsp_energy_sweep_table_62	[11][62];
t_energy_step	dsp_energy_sweep_table_prom	[32];


/*	Variables locales au module
 *	---------------------------
 */
static	float	dsp_sw_energy_sweep_stop = -1.0;
static	float	dsp_param_prom [2];


/***************************************************************************************************
 *
 *	Lecture d'une section contenant une variable a une dimension
 *	------------------------------------------------------------
 */
static	t_err	Read_section (FILE * entree, char * variable, int size, float * values)
{
	char *		fonction = FNAME ("Read_section");
	t_err		error = OK;
	char		buffer [200];
	int		count;

	for (count = 0; count < size; count++) values [count] = 0.0;

	count = 0;

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		float 	val;
	
		if (strlen (buffer) == 0 || buffer [0] == '#') continue;

		if (sscanf (buffer, "%f", & val) != 1) {
	
			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		if (count < size) values [count] = val;

		count ++;
	}

	if (count != size) {

		Affiche_erreur (fonction, "Variable %s : %d valeurs lues au lieu de %s", 
			variable, count, size);
		error = ERROR;
		goto EXIT;
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Lecture section spin_accumulation
 *	---------------------------------
 */
static	t_err	Read_spin_accumulation (FILE * entree)
{
	char *		fonction = FNAME ("Read_spin_accumulation");
	t_err		error = OK;
	char		buffer [200];
	int		prod, mode;

	for (mode = 0; mode < NB_DSP_MODE; mode ++) {

		for (prod = 0; prod < NB_TLM_PRODUCT; prod ++) dsp_spin_accumulation [prod][mode] = 0;
	}

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		int	lus;
		int	tab [16];

		if (strlen (buffer) == 0 || buffer [0] == '#') continue;

		lus = sscanf (buffer, "%*[ \t]P%2d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
			& prod,
			& tab [ 0], & tab [ 1], & tab [ 2], & tab [ 3], 
			& tab [ 4], & tab [ 5], & tab [ 6], & tab [ 7], 
			& tab [ 8], & tab [ 9], & tab [10], & tab [11],
			& tab [12], & tab [13], & tab [14], & tab [15]);

		if (lus != 17) {

			Affiche_erreur (fonction, "%s", buffer);
			Affiche_erreur (fonction, "Ligne incorrecte : %d champs lus sur 17", lus);
			error = ERROR;
			goto EXIT;
		}

		for (mode = 0; mode < NB_DSP_MODE; mode ++) dsp_spin_accumulation [prod][mode] = tab [mode];
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Calcule la valeur moyenne d'une table d'energie entre deux paliers
 *	------------------------------------------------------------------
 */
static	float	Average (float * table, int inf, int sup)
{
	float	sum = 0.0;
	int	i;

	if (sup <= inf) return table [inf];

	for (i = inf; i < sup; i++) sum += table [i];

	return sum / (sup - inf);
}


/***************************************************************************************************
 * 
 *	Troncature table d'energie No 10 (a partir de la table 0)
 *	---------------------------------------------------------
 */
static	t_err	Compute_table_10 (float * tmp)
{
	char *		fonction = FNAME ("Compute_table_10");
	t_err		error = OK;
	int		i, trigger, lower, upper;

	trigger = (int) dsp_sw_energy_sweep_stop;

	if (dsp_sw_energy_sweep_stop < 0.0 || dsp_sw_energy_sweep_stop > 127) {

		Affiche_erreur (fonction, "Variable sw_energy_sweep_stop non lue");
		error = ERROR;
		goto EXIT;
	}

	Affiche_trace (3, fonction, "Troncature table 10 a partir du step %d/128", trigger);

	for (i = 0; i < 62; i++) {
	
		upper = (i * 2 < trigger) ? i * 2 : trigger;
		lower = (i * 2 + 2 < trigger) ? i * 2 + 2 : trigger;

		dsp_energy_sweep_table_62 [10][i].upper   = tmp [upper];
		dsp_energy_sweep_table_62 [10][i].lower   = tmp [lower];
		dsp_energy_sweep_table_62 [10][i].average = Average (tmp, upper, lower);
	}

	for (i = 0; i < 31; i++) {
	
		upper = (i * 4 < trigger) ? i * 4 : trigger;
		lower = (i * 4 + 4 < trigger) ? i * 4 + 4 : trigger;

		dsp_energy_sweep_table_31 [10][i].upper   = tmp [upper];
		dsp_energy_sweep_table_31 [10][i].lower   = tmp [lower];
		dsp_energy_sweep_table_31 [10][i].average = Average (tmp, upper, lower);
	}

	for (i = 0; i < 16; i++) {
	
		upper = (i * 8 < trigger) ? i * 8 : trigger;
		lower = (i * 8 + 8 < trigger) ? i * 8 + 8 : trigger;

		dsp_energy_sweep_table_16 [10][i].upper   = tmp [upper];
		dsp_energy_sweep_table_16 [10][i].lower   = tmp [lower];
		dsp_energy_sweep_table_16 [10][i].average = Average (tmp, upper, lower);
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Lecture section energy_sweep_table
 *	----------------------------------
 */
static	t_err	Compute_energy_sweep_tables (FILE * entree)
{
	char *		fonction = FNAME ("Compute_energy_sweep_tables");
	t_err		error = OK;
	char		buffer [200];
	int		nb_steps = 0;

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		int	table, steps, law;
		float	k, fact, emin, emax, tmp [124];
		int	i, upper, lower;

		if (strlen (buffer) == 0 || buffer [0] == '#') continue;

		if (sscanf (buffer, "%d %d %f %f L%d", & table, & steps, & emax, & emin, & law) != 5) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		if (table < 0 || table > 9) {

			Affiche_erreur (fonction, "Table %d incorrecte", table);
			error = ERROR;
			goto EXIT;
		}

		if (steps + nb_steps > 124) {

			Affiche_erreur (fonction, "Depassement capacite table %d", table);
			error = ERROR;
			goto EXIT;
		}

		switch (law) {	/*** Calcul table temporaire ***/

		case 1 :	k = (float) exp (log ((double) emin / (double) emax) / (steps - 1));

				for (fact = 1.0, i = 0; i < steps; i++) {

					tmp [nb_steps + i] = (float) emax * fact;
					fact *= k;
				}
				break;

		case 2 :	k = (float) exp (log ((double) emin / (double) emax) / steps);

				for (fact = 1.0, i = 0; i < steps; i++) {

					tmp [nb_steps + i] = (float) emax * fact;
					fact *= k;
				}
				break;

		case 3 :	for (i = 0; i < steps; i++) 
					tmp [nb_steps + i] = (i < steps / 2) ? emax : emin;
				break;

		case 4 :	if (steps % 4 != 0) {

					Affiche_erreur (fonction, "Nombre de steps non multiple de 4 (loi 4)");
					error = ERROR;
					goto EXIT;
				}

				k = (float) exp (log ((double) emin / (double) emax) / (steps / 4 - 1));

				for (fact = 1.0, i = 0; i < steps; i++) {

					tmp [nb_steps + i] = (float) emax * fact;
					if (i % 4 == 3) fact *= k;
				}
				break;

		default :	Affiche_erreur (fonction, "Loi No %d incorrecte", law);
				error = ERROR;
				goto EXIT;
		}

		nb_steps += steps;


		if (nb_steps == 124) {	/*** Generation tables definitives ***/

			for (i = 0; i < 62; i++) {

				upper = i * 2;
				lower = (i < 61) ? i * 2 + 2 : 123;

				dsp_energy_sweep_table_62 [table][i].upper   = tmp [upper];
				dsp_energy_sweep_table_62 [table][i].lower   = tmp [lower];
				dsp_energy_sweep_table_62 [table][i].average = Average (tmp, upper, lower);
			}

			for (i = 0; i < 31; i++) {

				upper = i * 4;
				lower = (i < 30) ? i * 4 + 4 : 123;

				dsp_energy_sweep_table_31 [table][i].upper   = tmp [upper];
				dsp_energy_sweep_table_31 [table][i].lower   = tmp [lower];
				dsp_energy_sweep_table_31 [table][i].average = Average (tmp, upper, lower);
			}

			for (i = 0; i < 16; i++) {

				upper = i * 8;
				lower = (i < 15) ? i * 8 + 8 : 123;

				dsp_energy_sweep_table_16 [table][i].upper   = tmp [upper];
				dsp_energy_sweep_table_16 [table][i].lower   = tmp [lower];
				dsp_energy_sweep_table_16 [table][i].average = Average (tmp, upper, lower);
			}
			
			if (table == 0) {

				if (Erreur (error = Compute_table_10 (tmp))) goto EXIT;
			}

			nb_steps = 0;
		}
	}
EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Calcul de la table d'energie mode PROM
 *	--------------------------------------
 */
static	t_err	Compute_energy_sweep_table_prom (void)
{
	char *		fonction = FNAME ("Compute_energy_sweep_table_prom");
	t_err		error = OK;
	float		tmp [33];
	int		i;

	for (i = 0; i < 33; i++) {

		tmp [i] = dsp_anal_k_factor [2] * dsp_param_prom [0] * pow (dsp_param_prom [1], i * 4);
	}

	for (i = 0; i < 32; i++) {

		dsp_energy_sweep_table_prom [i].upper = tmp [i];
		dsp_energy_sweep_table_prom [i].lower = tmp [i+1];
		dsp_energy_sweep_table_prom [i].average = (tmp [i] + tmp [i+1]) / 2;
	}
	
EXIT:	return OK;
}


/***************************************************************************************************
 *
 *	Lecture coefficients d'efficacite des anodes
 *	--------------------------------------------
 */
static	t_err	Read_anode_effic (FILE * entree)
{
	char *		fonction = FNAME ("Read_fitting_param");
	t_err		error = OK;
	char		buffer [200];

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		float	theta, effic;
		int	sensit, indice;

		if (strlen (buffer) == 0 || buffer [0] == '#') continue;

		if (sscanf (buffer, "%d %f %f", & indice, & theta, & effic) != 3) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		if (indice < 0 || indice > 31) {

			Affiche_erreur (fonction, "No anode incorrect : %d", indice);
			error = ERROR;
			goto EXIT;
		}

		if (indice >= 16) {

			sensit = LS;
			indice = indice - 16;
		}
		else	sensit = HS;

		dsp_anode_eff [sensit][indice][0] = theta;
		dsp_anode_eff [sensit][indice][1] = effic;
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Lecture coefficients de correction
 *	----------------------------------
 */
static	t_err	Read_fitting_param (FILE * entree)
{
	char *		fonction = FNAME ("Read_fitting_param");
	t_err		error = OK;
	char		buffer [200];
	t_sensitivity	s;

	for (s = LS; s <= HS; s++) {

		dsp_a          [s] = 0.0;
		dsp_b          [s] = 0.0;
		dsp_norm_e     [s] = 0.0;
		dsp_norm_theta [s] = 0.0;
	}

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		char		sensit [10];
		float		a, b, norm_e, norm_theta;
		t_symbol *	ptr;

		if (strlen (buffer) == 0 || buffer [0] == '#') continue;

		if (sscanf (buffer, "%s %f %f %f %f", sensit, & a, & b, & norm_e, & norm_theta) != 5) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		ptr = Search_symbol_key (dsp_sensitivity, sensit);

		if (Unknown_symbol (ptr)) {

			Affiche_erreur (fonction, "Valeur de sensitivite erronee : %s", sensit);
			error = ERROR;
			goto EXIT;
		}

		dsp_a          [ptr->val] = a;
		dsp_b          [ptr->val] = b;
		dsp_norm_e     [ptr->val] = norm_e;
		dsp_norm_theta [ptr->val] = norm_theta;
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Lecture des coefficients de correction des MCP
 *	----------------------------------------------
 */
static	t_err	Read_MCP_fitting_param (FILE * entree)
{
	char *		fonction = FNAME ("Read_MCP_fitting_param");
	t_err		error = OK;
	char		buffer [200];
	int		i, eband;

	for (eband = 0; eband < 3; eband++) {

		for (i = 0; i < 5; i++) dsp_MCP_fitting_param [eband][i] = 0.0;
	}

	eband = 0;

	while (Read_line (buffer, entree) == OK && strncasecmp (buffer, "END", 3) != 0) {

		float	tab [5];

		if (strlen (buffer) == 0 || buffer [0] == '#') continue;

		if (sscanf (buffer, "%f %f %f %f %f", tab, tab+1, tab+2, tab+3, tab+4) != 5) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		
		for (i = 0; i < 5; i++) dsp_MCP_fitting_param [eband][i] = tab [i];

		eband ++;
	}

	if (eband != 3) {

		Affiche_erreur (fonction, "%d lignes lus au lieu de 3", eband);
		error = ERROR;
		goto EXIT;
	}

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Lecture fichier de calibration HIA
 *	----------------------------------
 */
t_err	Read_calib_HIA (char * filename)
{
	char *		fonction = FNAME ("Read_calib_HIA");
	t_err		error = OK;
	FILE *		entree = NULL;
	char		buffer [200];
	char		section [50];
	
	if ((entree = fopen (filename, "r")) == NULL) {

		Affiche_erreur (fonction, "Lecture %s impossible", filename);
		error = ERR_fopen;
		goto EXIT;
	}

	Affiche_trace (1, fonction, "Lecture %s", filename);

	while (Read_line (buffer, entree) == OK) {

		if (strlen (buffer) == 0 || buffer [0] == '#') continue;

		if (sscanf (buffer, "BEGIN %s", section) != 1) {

			Affiche_erreur (fonction, "Ligne incorrecte : %s", buffer);
			error = ERROR;
			goto EXIT;
		}

		if (SECTION ("spin_accumulation"))	error = Read_spin_accumulation (entree);
		else
		if (SECTION ("phase_instr"))		error = Read_section (entree, section, 2, dsp_phase_instr);
		else
		if (SECTION ("geom_factor"))		error = Read_section (entree, section, 2, dsp_geom_factor);
		else
		if (SECTION ("anal_k_factor"))		error = Read_section (entree, section, 3, dsp_anal_k_factor);
		else
		if (SECTION ("anal_alpha_angle"))	error = Read_section (entree, section, 2, dsp_anal_alpha_angle);
		else
		if (SECTION ("dead_times"))		error = Read_section (entree, section, 2, dsp_dead_times);
		else
		if (SECTION ("sw_energy_sweep_stop"))	error = Read_section (entree, section, 1, & dsp_sw_energy_sweep_stop);
		else
		if (SECTION ("energy_sweep_table"))	error = Compute_energy_sweep_tables (entree);
		else 
		if (SECTION ("energy_sweep_table_prom")) error = Read_section (entree, section, 2, dsp_param_prom);
		else 
		if (SECTION ("anode_effic_table"))	error = Read_anode_effic (entree);
		else 
		if (SECTION ("fitting_param"))		error = Read_fitting_param (entree);
		else 
		if (SECTION ("MCP_fitting_param"))	error = Read_MCP_fitting_param (entree);
		else {
			Affiche_erreur (fonction, "Section %s inconnue", section);
			error = ERROR;
			goto EXIT;
		}

		if (Erreur (error)) goto EXIT;
	}

	if (Erreur (error = Compute_energy_sweep_table_prom ())) goto EXIT;

EXIT:	if (entree != NULL) fclose (entree);
	return error;
}
