/***************************************************************************************************
 *
 *	Fichier	: $RCSfile: CALIB.c,v $
 *
 *	Version	: $Revision: 1.8 $
 *
 *	Auteur	: $Author: barthe $
 *
 *	Date	: $Date: 2009/11/25 14:02:01 $
 *
 *	==========================================================================================
 *
 *	Module charge de la gestion du catalogue des fichiers de calibration.
 *
 *	Structure du fichier de calibration :
 *
 *	- type de calibration (Housekeeping, coefficient HIA)
 *	- date debut de validite
 *	- nom du fichier contenant les parametres
 */
 
#define	MODULE_NAME	"CALIB"

#include "DSPLIB.h"
#include "HIA.h"
#include "HSK.h"
#include "CALIB.h"

typedef	struct {				/* Entree du catalogue				*/

	char 		product [10];		/* - produit de calibration			*/
	double		date_deb;		/* - debut de validite				*/
	double		date_fin;		/* - fin de validite				*/
	t_filename	filename;		/* - fichier correspondant			*/

}	t_entry;

static	struct {				/* Structure de donnees du catalogue		*/

	int		size;			/* - nombre elements				*/
	t_filename	name;			/* - nom du fichier catalogue			*/
	t_entry *	item;			/* - table des elements				*/

}	catalog = { 0, "", NULL};

typedef	t_err	(* t_function) (char *);	/* Fonction de traitement			*/

typedef	struct {				/* Entree des descripteur de calibration	*/

	char *		product;		/* - type de produit				*/
	int		entry;			/* - entree correspondante du catalogue		*/
	t_function	function;		/* - fonction de traitement a appliquer		*/

}	t_caldesc;

#define	TABLE_SIZE	2

static	t_caldesc	table [TABLE_SIZE] = {	/* Table des descripteur des calibrations	*/

	{ "HSK", -1, Read_calib_HSK },
	{ "HIA", -1, Read_calib_HIA }
};


/***************************************************************************************************
 *
 *	Fonction de comparaison pour tri du catalogue
 *	---------------------------------------------
 */
static 	int	Compare (const void * left, const void * right)
{
	t_entry *	l = (t_entry *) left;
	t_entry *	r = (t_entry *) right;
	int		cmp;

	cmp = strcmp (l->product, r->product);

	if (cmp != 0) return cmp;

	return (l->date_deb > r->date_deb) ? 1 : -1;
}
	

/***************************************************************************************************
 *
 *	Calcul des dates de fin de validite de chaque fichier
 *	-----------------------------------------------------
 */
static	t_err	Compute_validity_period (void)
{
	char *		fonction = FNAME ("Compute_validity_period");
	t_err		error = OK;
	double		max_validity = Ascii_time_to_milli ("2099-01-01T00:00:00Z");
	int		i;

	for (i = 0; i < catalog.size -1 ; i++) {

		if (strcmp (catalog.item [i].product, catalog.item [i+1].product) == 0) 
			catalog.item [i].date_fin = catalog.item [i+1].date_deb;
		else	catalog.item [i].date_fin = max_validity;
	}

	catalog.item [catalog.size -1].date_fin = max_validity;

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Lecture du contenu du catalogue
 *	-------------------------------
 */
static	t_err	Read_catalog (void)
{
	char *		fonction = FNAME ("Read_calalog");
	t_err		error = OK;
	FILE *		entree = NULL;
	char		buffer [1024];
	int		i;

	if ((entree = fopen (catalog.name, "r")) == NULL) {

		Affiche_erreur (fonction, "Lecture %s impossible", catalog.name);
		error = ERR_fopen;
		goto EXIT;
	}

	while (Read_line (buffer, entree) == OK) {

		char		product [10];
		char		date [30];
		t_filename	filename;
		int		lus;

		if (buffer [0] == '#' || strlen (buffer) == 0) continue;

		lus = sscanf (buffer, "%s %s %s", product, date, filename);

		if (lus != 3) {

			Affiche_erreur (fonction, "%s", buffer);
			Affiche_erreur (fonction, "Ligne incorrecte : %2/3 champs lus", lus);
			error = ERROR;	
			goto EXIT;
		}

		catalog.item = (t_entry *) realloc (catalog.item, sizeof (t_entry) * (catalog.size + 1));

		if (catalog.item == NULL) {

			Affiche_erreur (fonction, "Echec allocation entree %d", catalog.size);
			error = ERR_malloc;
			goto EXIT;
		}

		strcpy (catalog.item [catalog.size].product, product);
		strcpy (catalog.item [catalog.size].filename, filename);
		catalog.item [catalog.size].date_deb = Ascii_time_to_milli (date);
		
		catalog.size ++;
	}

	qsort (catalog.item, catalog.size, sizeof (t_entry), Compare);

	if (Erreur (error = Compute_validity_period ())) goto EXIT;

	for (i = 0; i < catalog.size; i++) {

		Affiche_trace (2, fonction, "%-10.10s %s  %s",
			catalog.item [i].product,
			Milli_to_Ascii_time (catalog.item [i].date_deb),
			catalog.item [i].filename);
	}
	
EXIT:	if (entree != NULL) fclose (entree);
	return error;
}


/***************************************************************************************************
 *
 *	Initialisation du catalogue de calibration
 *	------------------------------------------
 */
t_err	Open_calib (void)
{
	char *		fonction = FNAME ("Open_calib");
	t_err		error = OK;
	t_filename	mask;
	int		i, count = 0;

	sprintf (mask, "%s/CALIB/T1_Lx_HIA_????????_V??_CATALOGxxx.cal", dsp_root);

	if (Erreur (error = Search_files (mask, & count))) goto EXIT;
	
	if (count != 1) {

		Affiche_erreur (fonction, "%s", mask);
		Affiche_erreur (fonction, "%d catalogues de calibration trouves", count);
		error = ERROR;
		goto EXIT;
	}

	catalog.size = 0;

	strcpy (catalog.name, Get_filename (0));

	Affiche_trace (1, fonction, "Lecture %s", catalog.name);

	if (Erreur (error = Read_catalog ())) goto EXIT;

	/*	Initialise table des calibrations lues
	 */
	for (i = 0; i < TABLE_SIZE; i++) table [i].entry = -1;

EXIT:	return error;
}


/***************************************************************************************************
 *
 *	Fermeture du catalogue de calibration
 *	-------------------------------------
 */
t_err	Close_calib (void)
{
	char *		fonction = FNAME ("Close_calib");
	t_err		error = OK;

	Affiche_trace (1, fonction, "Fermeture %s", catalog.name);

	if (catalog.item != NULL) free (catalog.item);

	catalog.size = 0;
	catalog.item = NULL;

EXIT:	return error;
}	


/***************************************************************************************************
 *
 *	Cherche entree du catalogue en fonction de la date et du produit
 *	----------------------------------------------------------------
 */
static	int	Get_catalog_entry (char * product, double milli)
{
	int	i;

	for (i = 0; i < catalog.size; i++) {

		if (strcmp (product, catalog.item [i].product) != 0) continue;
		if (milli <  catalog.item [i].date_deb) continue;
		if (milli >= catalog.item [i].date_fin) continue;

		return i;	
	}
	return -1;
}


/***************************************************************************************************
 *
 *	Lecture des calibrations pour un type de produit et une date donnes
 *	-------------------------------------------------------------------
 */
t_err	Get_calib (char * product, double milli, double * end_validity)
{
	char *		fonction = FNAME ("Get_calib");
	t_err		error = OK;
	int		prod = -1;
	int		entry = -1;
	int		i;

	* end_validity = 0.0;

	if (catalog.item == NULL) {

		Affiche_erreur (fonction, "Calibrations non initialisees");
		error = ERROR;
		goto EXIT;
	}

	for (i = 0; i < TABLE_SIZE; i++) if (strcmp (product, table [i].product) == 0) prod = i;
	
	if (prod == -1) {

		Affiche_erreur (fonction, "Produit %s incorrect", product);
		error = ERROR;
		goto EXIT;
	}

	entry = table [prod].entry;

	if (entry == -1 || milli < catalog.item [entry].date_deb || milli >= catalog.item [entry].date_fin) {

		t_filename	filename;

		entry = Get_catalog_entry (product, milli);

		if (entry == -1) {

			Affiche_erreur (fonction, "Pas de calibration pour %s %s", 
				product, Milli_to_Ascii_time (milli));
			error = ERROR;
			goto EXIT;
		}

		table [prod].entry = entry;

		* end_validity = 0.0;

		sprintf (filename, "%s/CALIB/%s", dsp_root, catalog.item [entry].filename);

		if (Erreur (error = (* table [prod].function) (filename))) goto EXIT;
	}

	* end_validity = catalog.item [entry].date_fin;

EXIT:	return error;
}
