/*------------------------------------------------------------------------------
 *
 *      Fichier : ascii2bin.c
 *
 *      Date    : 22/02/2006
 *
 *      Auteur  : Pene
 *
 *      Version : 1.0
 *
 *------------------------------------------------------------------------------
 */

#include <stdio.h>		/* FILE */
#include <stdlib.h>		/* atoi() */
#include <string.h>		/* strcmp() */
#include <math.h>		/* fabs */
#include <libgen.h>		/* basename */	


#include "temps.h"
#include "ARCH.h"

typedef struct {
	int year;
	int month;
	int day;
	int hour;
	int min;
	int sec;
	int jday;
	int jsec;

} s_date;


/*.......................................................
  .
  .		Affichage d'un message d'erreur :
  .
*/
/*---------------------------------------------------------------------------*/
void error_msg (int err,char* prg) {
/*---------------------------------------------------------------------------*/
	
	if (err != 0) { 
		printf("ERROR : %d\t\t[FAIL]\n",err);
	} else { 
		printf("\t\t\t[ OK ]\n");
	}
       	switch (err) {
		case -2:
				printf("%s : can't open output file!\n\n",prg);
				break;
		case -1:
				printf("%s : can't open input file!\n\n",prg);
				break;
		case  1:
				printf("%s : too few arguments!\n\n",prg);
				break;
		case  2:
				printf("%s : bad fields number in input file!\n\n",prg);
				break;
		case  3:
				printf("%s : bad fields number in output file!\n\n",prg);
				break;
		case  4:
				printf("%s : output file name too short!(<14 char)!\n\n",prg);
				break;
		case  5:
				printf("%s : unknown argument!\n\n",prg);
				break;
		case  6:
				printf("%s : date conflict in input file!\n\n",prg);
				break;
		case  7:
				printf("%s : Input file : Bad flield!\n\n",prg);
				break;
		case  8:
				printf("%s : input file year < 1933!\n\n",prg);
				break;
		case  9:
				printf("%s : Geomagnetic field component in input file unknown!\n\n",prg);
				break;
		case 10:
				printf("%s : not final version!\n\n",prg);
				break;
		default:
				break;
        }

}


/*.............................................................
  .
  .		generations de fichiers jours :
  .
*/
/*---------------------------------------------------------------------------*/
FILE* gen_ofile (char* output_file,FILE *o_file,s_date date,int* day_before,double* julian,int zip,int* err) {
/*---------------------------------------------------------------------------*/


	int len_ofile = strlen(output_file);

	if (len_ofile<14) {
		*err=4;
		return o_file;
	}


	if ((date.jday < *day_before) && (zip==0)) {
		*err=6;
		return o_file;
		
	}

	if ((*day_before==0) || (date.jday != *day_before)) {

		if (*day_before != 0) {
			fclose (o_file); 
		}

		/* nom du nouveau fichier de donnees : */
		char *end_ofile = strrchr (output_file, '_') - 8;
		char buf[8+1];

		sprintf (buf, "%04d%02d%02d",date.year,date.month,date.day);
		strncpy (end_ofile, buf, 8);


		/* ouverture du fichier de sortie nouvelle date */
		if ((o_file = fopen (output_file,"w+b"))==NULL) {
			*err=-2;
			return o_file;
		}
	
		*day_before=date.jday;

	}

	*julian=tu_to_date (date.year,date.month,date.day,date.hour,date.min,date.sec);

	return o_file;
	

}

/*..............................................................................
  .
  .		Lecture et ecriture des fichiers d'indices de KYOTO (ae/minute)
  .
*/
/*---------------------------------------------------------------------------*/
int ae2_treatment (char *input_file,char* output_file,FILE *i_file) {
/*---------------------------------------------------------------------------*/

	FILE* o_file=NULL;
	int len_ofile = strlen(output_file);
	int y,m,d,h,mn;	/* year, month, day, hour, minute */

	char buf[1024];
	char *buf2;
	char buftmp;
	int nb_fields;

	double julian;

	s_date date;

	unsigned char chr[3];		/* tableau de characteres */
	unsigned char chr2[8];		/* tableau de characteres */
	short i_hour[24*4];		/* hourly Mean */ 
	short i_min[24*4][60];		/* tableau des indices par minutes */ 
	char elm;			/* numero correspondant a l'element traite (pour incrementer les tableaux)*/
	short short_err = -32768;	/* valeur d'erreur dans CL pour les donnees de type short */

	for (elm=0;elm<4;elm++) {
		for (h=0;h<24;h++) {
			i_hour[h*elm]=short_err;
			for (mn=0;mn<60;mn++) {
				i_min[h*elm][mn]=short_err;
				
			}
		}
	}
	

	date.hour =0;
	date.min =0;
	date.sec =0;
	

	while (fgets(buf,sizeof(buf),i_file)!= NULL ) {

		if (buf[0]!= 'W' ) return 0;

		switch (buf[18]) {
			case 'E' : elm=0; break;
			case 'L' : elm=1; break;
			case 'O' : elm=2; break;
			case 'U' : elm=3; break;
			default : return 9;
		}

		/*........................................
		  .
		  .	Partie Lecture des fichiers :
		  .
		*/
		nb_fields=sscanf(buf,"%02s%c%06s%04d%02d%02d%c%02d",chr,&chr[0],chr2,&y,&m,&d,&chr[0],&h);
		if (nb_fields != 8) return 2;

		if (y<1933) return 8;

		buf2=buf+28;

		/* boucle sur 60min */
		for (mn=0;mn<60;mn++) {
			buf2=buf2+6;
			buftmp=buf2[6];
			buf2[6]=' ';
			nb_fields=sscanf(buf2,"%06hd",&i_min[24*elm+h][mn]);
			if ( i_min[24*elm+h][h] == 9999 ) i_min[24*elm+h][mn]=short_err;
			buf2[6]=buftmp;
		}

		buf2=buf2+6;
		nb_fields=sscanf(buf2,"%06hd",&i_hour[24*elm+h]);
		if (i_hour[24*elm+h]==9999) i_hour[24*elm+h]=short_err;

	}

	/*.............................................
	  .
	  .	Partie ecriture des fichiers :
	  .
	*/


	if (len_ofile<14) return 4;

	/* nom du nouveau fichier de donnees : */
	char *end_ofile = strrchr (output_file, '_') - 8;
	char buf0[8+1];

	sprintf (buf0, "%04d%02d%02d",y,m,d);
	strncpy (end_ofile, buf0, 8);


	/* ouverture du fichier de sortie nouvelle date */

	if ( (o_file = fopen (output_file,"w+b"))==NULL) return -2;

	for ( h=0; h<24; h++) {
		for (mn=0;mn<60;mn++) {

			// printf ("%.0f %c %d %d %d\n",julian, elm, i_tab, i_min[h],i_hour);
			julian=tu_to_date(y,m,d,h,mn,0);

			/* Ajout de AE */
			nb_fields=Endian_write(&julian,sizeof(double),1,o_file)
				 +Endian_write(&i_min[h][mn],sizeof(short),1,o_file)
				 +Endian_write(&i_hour[h],sizeof(short),1,o_file);
			if (nb_fields != 3)  return 3;

			/* Ajout de AL AO et AU */
			for ( elm=1; elm<4; elm++) {
					nb_fields=Endian_write(&i_min[24*elm+h][mn],sizeof(short),1,o_file)
						 +Endian_write(&i_hour[24*elm+h],sizeof(short),1,o_file);
				if (nb_fields != 2)  return 3;
			}

		}
	}
	fclose (o_file); 

	return 0;
}

/*..............................................................................
  .
  .		Lecture et ecriture des fichiers d'indices AE Kyoto (2007) (ae/minutes)
  .
*/
/*---------------------------------------------------------------------------*/
int ae3_treatment (char *input_file,char* output_file,FILE *i_file) {
/*---------------------------------------------------------------------------*/

	FILE* o_file=NULL;
	int len_ofile = strlen(output_file);
	int i;
	int y,m,d,h,mn;			/* year, month, day, hour */

	char buf[1024];
	char *buf2;
	char buftmp;
	int nb_fields;

	double julian;

	s_date date;

	unsigned char chr[14];		    /* tableau de characteres */
	short i_hour[31*4][24];		/* tableau des indices par heure */ 
	short i_min[31*4][24*60];		/* tableau des indices par minutes */ 
	char elm;		            	/* numero correspondant a l'element traite (pour incrementer les tableaux)*/
	char exist[31*12*4];	        	/* permet de checker si la valeur existe */

	short short_err = -32768;	/* valeur d'erreur dans CL pour les donnees de type short */

	int	vide;


	date.hour =0;
	date.min =0;
	date.sec =0;
	
	for (i=0;i<1488;i++) exist[i]=0;

	for (elm=0;elm<4;elm++) {
        for (d=0;d<31;d++) {
            for (h=0;h<24;h++) {
                i_hour[elm*31+d][h]=short_err;
                for (mn=0;mn<60;mn++) {
                    i_min[elm*31+d][h*60+mn]=short_err;
                }
            }
        }
	}
	
     

	while (fgets(buf,sizeof(buf),i_file)!= NULL ) {

		if (buf[0]!= 'A' ) return 0;

		switch (buf[18]) {
			case 'E' : elm=0; break;
			case 'L' : elm=1; break;
			case 'O' : elm=2; break;
			case 'U' : elm=3; break;
			default : return 9;
		}

		/*........................................
		  .
		  .	Partie Lecture des fichiers :
		  .
		*/
		nb_fields=sscanf(buf,"%12s%02d%02d%02d%c%02d",chr, &y,&m,&d,&chr[0],&h);
		if (nb_fields != 6) return 2;

		if (y <33 && y>=50) return 8;
		else if ( y > 30 ) y=1900+y;
		else y=2000+y;

		buf2=buf+29;

		exist[d-1]=1;

		/* boucle sur 60min */
		for (mn=0;mn<60;mn++) {
			buf2=buf2+6;
			buftmp=buf2[6];
			buf2[6]=' ';
			nb_fields=sscanf(buf2,"%06hd",&i_min[31*elm+d-1][h*60+mn]);

			//if ( i_min[31*elm+d-1][h*60+mn] == 9999 ) i_min[31*elm+d-1][h*60+mn]=short_err;
			// Correction car 99999 ne rentre pas dans un short
			if (strncmp(buf2,"99999",5) == 0) i_min[31*elm+d-1][h*60+mn]=short_err;
			//printf ("INPUT %s -> %d\n",buf2,i_min[31*elm+d-1][h*60+mn]);
			
			buf2[6]=buftmp;
		}

		buf2=buf2+6;
		nb_fields=sscanf(buf2,"%06hd",&i_hour[31*elm+d-1][h]);

		//if ( i_hour[31*elm+d-1][h] == 9999 ) i_hour[31*elm+d-1][h]=short_err;
		// Correction car 99999 ne rentre pas dans un short
		if (strncmp(buf2,"99999",5) == 0) i_hour[31*elm+d-1][h]=short_err;
		//printf ("INPUT %s -> %d\n",buf2,i_hour[31*elm+d-1][h]);

	}


    for (d=0;d<31;d++) {
	vide=1;
	for ( h=0; h<24; h++) {
		for (mn=0;mn<60;mn++) {
			if (i_min[d][h*60+mn] != short_err) {
				//printf ("d=%d h=%d mn=%d i_min %d\n", d,h,mn,i_min[d][h*60+mn]);
				vide=0;
			}
			if (i_hour[d][h] != short_err) {
				//printf ("d=%d h=%d mn=%d i_hour,%d\n",d,h,mn,i_hour[d][h]);
				vide=0;
			}
			for ( elm=1; elm<4; elm++) {
				if (i_min[elm*31+d][h*60+mn] != short_err) {
					//printf ("d=%d h=%d mn=%d elm=%d i_min %d\n",d,h,m,elm,i_min[elm*31+d][h*60+mn]);
					vide=0;
				}
				if (i_hour[elm*31+d][h] != short_err) {
					//printf ("d=%d h=%d mn=%d elm=%d i_hour %d\n",d,h,mn,elm,i_hour[elm*31+d][h]);
					vide=0;
				}
			}
		}
	}
	//printf ("vide=%d\n",vide);

	// Ne pas écrire les fichiers qui ne contiennent que des FILLVAL
        if ( exist[d]==1 && vide==0) {

            date.year=y;
            date.month=m;
            date.day=(d+1);


            /*.............................................
              .
              .	Partie ecriture des fichiers :
              .
            */

            julian=tu_to_date(date.year,date.month,date.day,0,0,0);

            if (len_ofile<14) return 4;

            /* nom du nouveau fichier de donnees : */
            char *end_ofile = strrchr (output_file, '_') - 8;
            char buf[8+1];

            sprintf (buf, "%04d%02d%02d",date.year,date.month,date.day);
            strncpy (end_ofile, buf, 8);

            /* ouverture du fichier de sortie nouvelle date */

		//printf ("output_file=%s\n", output_file);

            if ( (o_file = fopen (output_file,"w+b"))==NULL) return -2;
            //printf ("%s\n",output_file);


            for ( h=0; h<24; h++) {
                // printf ("%.0f %c %d %d\n",julian, elm, i_hour[h],i_dm);

                //--------------------------------------------------------------------------------------
                /* Ajout de AE */
                for (mn=0;mn<60;mn++) {

                    // printf ("%.0f %c %d %d %d\n",julian, elm, i_tab, i_min[h],i_hour);
                    julian=tu_to_date(date.year,date.month,date.day,h,mn,0);

                    /* Ajout de AE */
                    nb_fields=Endian_write(&julian,sizeof(double),1,o_file)
                         +Endian_write(&i_min[d][h*60+mn],sizeof(short),1,o_file)
                         +Endian_write(&i_hour[d][h],sizeof(short),1,o_file);
                    if (nb_fields != 3)  return 3;

                    /* Ajout de AL AO et AU */
                    for ( elm=1; elm<4; elm++) {
                            nb_fields=Endian_write(&i_min[elm*31+d][h*60+mn],sizeof(short),1,o_file)
                             +Endian_write(&i_hour[elm*31+d][h],sizeof(short),1,o_file);
                        if (nb_fields != 2)  return 3;
                    }

                }

            }
            fclose (o_file); 
        }
    }

	return 0;
}

/*..............................................................................
  .
  .		Lecture et ecriture des fichiers d'indices NOAA (ae/hour)
  .
*/
/*---------------------------------------------------------------------------*/
int ae_treatment (char *input_file,char* output_file,FILE *i_file) {
/*---------------------------------------------------------------------------*/

	FILE* o_file=NULL;
	int len_ofile = strlen(output_file);
	int i;
	int y,m,d,h;			/* year, month, day, hour */

	char buf[1024];
	char *buf2;
	char buftmp;
	int nb_fields;

	double julian;

	s_date date;

	unsigned char chr[4];		/* tableau de characteres */
	short i_tab[31*12*4];		/* tabular base */ 
	short i_dm[31*12*4];		/* Daily Mean */ 
	short i_hour[31*12*4][24];	/* tableau des indices par heure */ 
	char elm;			/* numero correspondant a l'element traite (pour incrementer les tableaux)*/
	char exist[31*12*4];		/* permet de checker si la valeur existe */

	short short_err = -32768;	/* valeur d'erreur dans CL pour les donnees de type short */


	date.hour =0;
	date.min =0;
	date.sec =0;
	
	for (i=0;i<1488;i++) exist[i]=0;
     

	while (fgets(buf,sizeof(buf),i_file)!= NULL ) {
		if (buf[0]!= 'A' ) return 0;

		switch (buf[7]) {
			case 'E' : elm=0; break;
			case 'L' : elm=1; break;
			case 'O' : elm=2; break;
			case 'U' : elm=3; break;
			default : return 9;
		}

		/*........................................
		  .
		  .	Partie Lecture des fichiers :
		  .
		*/
		nb_fields=sscanf(buf,"%03s%02d%02d%c%02d",chr, &y,&m,&chr[0],&d);
		if (nb_fields != 5) return 2;

		if (y <33 && y>=50) return 8;
		else if ( y > 30 ) y=1900+y;
		else y=2000+y;

		buf2=buf+16;
		buftmp=buf2[4];
		buf2[4]=' ';

		exist[31*12*elm+(m-1)*31+(d-1)]=1;

		nb_fields=sscanf(buf2,"%c%03hd",&chr[0],&i_tab[31*12*elm+(m-1)*31+(d-1)]);
		if (chr[0]=='-') i_tab[31*12*elm+(m-1)*31+(d-1)]=-i_tab[31*12*elm+(m-1)*31+(d-1)];
		if (nb_fields != 2) {
			printf("Error : %04d-%02d-%02d element nb %d ignored!\n", y, m, d,elm);
			exist[31*12*elm+(m-1)*31+(d-1)]=0;
		}
		buf2[4]=buftmp;

		/* boucle sur 24h */
		for (h=0;h<24;h++) {
			buf2=buf2+4;
			buftmp=buf2[4];
			buf2[4]=' ';
			nb_fields=sscanf(buf2,"%04hd",&i_hour[31*12*elm+(m-1)*31+(d-1)][h]);
			if (nb_fields != 1) {
				/* printf("Error : %04d-%02d-%02d element nb %d ignored!\n", y, m, d,elm);*/
				exist[31*12*elm+(m-1)*31+(d-1)]=0;
			}
			if ( i_hour[31*12*elm+(m-1)*31+(d-1)][h] == 9999 ) i_hour[31*12*elm+(m-1)*31+(d-1)][h]=short_err;
			buf2[4]=buftmp;
		}

		buf2=buf2+4;
		nb_fields=sscanf(buf2,"%04hd",&i_dm[31*12*elm+(m-1)*31+(d-1)]);
		if (nb_fields != 1) {
			printf("Error : %04d-%02d-%02d element nb %d ignored!\n", y, m, d,elm);
			exist[31*12*elm+(m-1)*31+(d-1)]=0;
		}
		if (i_dm[31*12*elm+(m-1)*31+(d-1)]==9999) i_dm[31*12*elm+(m-1)*31+(d-1)]=short_err;

	}

	for (m=0;m<12;m++) {
		for (d=0;d<31;d++) {
			if ( exist[m*31+d]==1) {

				date.year=y;
				date.month=(m+1);
				date.day=(d+1);


				/*.............................................
				  .
				  .	Partie ecriture des fichiers :
				  .
				*/

				julian=tu_to_date(date.year,date.month,date.day,0,0,0);

				if (len_ofile<14) return 4;

				/* nom du nouveau fichier de donnees : */
				char *end_ofile = strrchr (output_file, '_') - 8;
				char buf[8+1];

				sprintf (buf, "%04d%02d%02d",date.year,date.month,date.day);
				strncpy (end_ofile, buf, 8);

				/* ouverture du fichier de sortie nouvelle date */

				if ( (o_file = fopen (output_file,"w+b"))==NULL) return -2;

				for ( h=0; h<24; h++) {
					// printf ("%.0f %c %d %d %d\n",julian, elm, i_tab, i_hour[h],i_dm);

					/* Ajout de AE */
					nb_fields=Endian_write(& julian,sizeof(double),1,o_file)
						 +Endian_write(& i_tab[m*31+d],sizeof(short),1,o_file)
						 +Endian_write(& i_hour[m*31+d][h],sizeof(short),1,o_file)
						 +Endian_write(& i_dm[m*31+d],sizeof(short),1,o_file);
					if (nb_fields != 4)  return 3;

					/* Ajout de AL AO et AU */
					for ( elm=1; elm<4; elm++) {
						if (exist[31*12*elm+m*31+d]==1) {
							nb_fields=Endian_write(& i_tab[31*12*elm+m*31+d],sizeof(short),1,o_file)
								 +Endian_write(& i_hour[31*12*elm+m*31+d][h],sizeof(short),1,o_file)
								 +Endian_write(& i_dm[31*12*elm+m*31+d],sizeof(short),1,o_file);
						}
						else {
							nb_fields=Endian_write(& short_err,sizeof(short),1,o_file)
								 +Endian_write(& short_err,sizeof(short),1,o_file)
								 +Endian_write(& short_err,sizeof(short),1,o_file);
						}
						if (nb_fields != 3)  return 3;
					}

					julian=julian+3600000;
				}
				fclose (o_file); 
			}
		}
	}

	return 0;
}


/*..............................................................................
  .
  .		Lecture et ecriture des fichiers d'indices
  .
*/
/*---------------------------------------------------------------------------*/
int ind_treatment (char *input_file,char* output_file,FILE *i_file,char* typ) {
/*---------------------------------------------------------------------------*/

	FILE* o_file=NULL;
	int len_ofile = strlen(output_file);
	int i;

	char buf[1024];
	char* buf2;
	char bufr0[6],bufr1[6],bufr2[6],bufr3[6],bufr4[6];
	int nb_fields;

	double julian;

	s_date date;

	/* ...................................
	 *    Variables pour les DST        */

	unsigned char chr[4];		/* tableau de characteres */
	short i_tab;			/* tabular base */ 
	short int i_dm;			/* Daily Mean */ 
	short i_hour[24];		/* tableau des indices par heure */ 
	char elm;			/* element traite */

	/* ...................................
	 *    Variables pour les KP AP      */

	short	s_ind[2];		/* tableau de differentes valeurs sur les indices */ 
	char	c_ind[3];		/* tableau de differentes valeurs sur les indices */ 
	float	flt[2];			/* tableau de differentes valeurs sur les indices */ 
	char	c_kp[8];		/* tableau des indices kp tout les 3h */ 
	short	s_kp;			/* sommation des indices kp */ 
	short	s_ap[9];		/* tableau des indices ap tout les 3h */ 
	char	ligne[1024];

	char buftmp;

	date.hour =0;
	date.min =0;
	date.sec =0;
	
	short short_err = -32768;	/* valeur d'erreur dans CL pour les donnees de type short */

	//while (fgets(buf,sizeof(buf),i_file)!= NULL ) {
	while (! feof(i_file) ) {

		/*........................................
		  .
		  .	Partie Lecture des fichiers :
		  .
		*/

		if (strcmp(typ,"dst") == 0) {

			/* Si l'on a des donnees de type DST : */

			fgets(buf,sizeof(buf),i_file);
			if (buf[0]!= 'A' && buf[0]!= 'D' ) return 0;

			nb_fields=sscanf(buf,"%03s%02d%02d%c%02d",chr, &date.year, &date.month, &elm, &date.day);
			if (nb_fields != 5) return 2;

			buf2=buf+16;
			buftmp=buf2[4];
			buf2[4]=' ';

			nb_fields=sscanf(buf2,"%c%03hd",&chr[0],&i_tab);
			if (chr[0]=='-') i_tab=-i_tab;
			if (nb_fields != 2) return 2;

			buf2[4]=buftmp;

			for (i=0;i<24;i++) {
				buf2=buf2+4;
				buftmp=buf2[4];
				buf2[4]=' ';

				nb_fields=sscanf(buf2,"%04hd",&i_hour[i]);
				if (nb_fields != 1) return 2;
				if ( i_hour[i]==9999 ) i_hour[i]=short_err;

				buf2[4]=buftmp;
			}

			buf2=buf2+4;
			nb_fields=sscanf(buf2,"%04hd",&i_dm);
			if (nb_fields != 1) return 2;
			if ( i_dm==9999 ) i_dm=short_err;


			switch (elm) {
				case 'H' : elm=0; break;
				case 'D' : elm=1; break;
				case 'Z' : elm=2; break;
				case 'X' : elm=3; break;
				case 'Y' : elm=4; break;
				case '*' : elm=5; break;
				case 'P' : elm=6; break;
				case 'Q' : elm=7; break;
				default : return 9;
			}

		} else if (strcmp(typ,"kpap") == 0) {

			/* Si l'on a des donnees de type KPAP : */

			memset (bufr0, 0, 6);
			memset (bufr1, 0, 6);
			memset (bufr2, 0, 6);
			memset (bufr3, 0, 6);
			memset (bufr4, 0, 6);

			nb_fields=fread(&bufr0,sizeof(char),2,i_file)
				 +fread(&bufr1,sizeof(char),2,i_file)
				 +fread(&bufr2,sizeof(char),2,i_file)
				 +fread(&bufr3,sizeof(char),4,i_file)
				 +fread(&bufr4,sizeof(char),2,i_file);
			date.year=atoi(bufr0);
			date.month=atoi(bufr1);
			date.day=atoi(bufr2);
			//printf ("bufr0=%s  bufr1=%s  bufr2=%s  date.year=%d  date.month=%d  date.day=%d\n",bufr0,bufr1,bufr2,date.year,date.month,date.day);
			s_ind[0]=atoi(bufr3);
			c_ind[0]=atoi(bufr4);
			//printf ("ICI nb_fields=%d\n",nb_fields);

			if (nb_fields == 0)
				return 0;
			else if (nb_fields != 12)
				return 2;

			/* boucle sur les kp */
			for (i=0;i<8;i++) {
				memset (bufr0, 0, 6);
				nb_fields=fread(&bufr0,sizeof(char),2,i_file);
				c_kp[i]=atoi(bufr0);
				//printf ("c_kp[%d] = %d\n", i,c_kp[i]);
				if (nb_fields != 2) return 2;
			}
			memset (bufr0, 0, 6);
			nb_fields=fread(&bufr0,sizeof(char),3,i_file);
			s_kp=atoi(bufr0);
			//printf ("s_kp = %d\n", s_kp);
			if (nb_fields != 3) return 2;

			/* boucle sur les ap */
			for (i=0;i<9;i++) {
				memset (bufr0, 0, 6);
				nb_fields=fread(&bufr0,sizeof(char),3,i_file);
				s_ap[i]=atoi(bufr0);
				//printf ("s_ap[%d] = %d\n", i,s_ap[i]);
				if (nb_fields != 3) return 2;
			}

			memset (bufr0, 0, 6);
			memset (bufr1, 0, 6);
			memset (bufr2, 0, 6);
			memset (bufr3, 0, 6);
			memset (bufr4, 0, 6);

			nb_fields=fread(&bufr0,sizeof(char),3,i_file)
				 +fread(&bufr1,sizeof(char),1,i_file)
				 +fread(&bufr2,sizeof(char),3,i_file)
				 +fread(&bufr3,sizeof(char),5,i_file)
				 +fread(&bufr4,sizeof(char),1,i_file);
			flt[0]=atof(bufr0);	//printf ("flt[0]=%f\n", flt[0]);
			c_ind[1]=atoi(bufr1);	//printf ("c_ind[1]=%d\n", c_ind[1]);
			s_ind[1]=atoi(bufr2);	//printf ("s_ind[1]=%d\n", s_ind[1]);
			flt[1]=atof(bufr3);	//printf ("flt[1]=%f\n", flt[1]);
			c_ind[2]=atoi(bufr4);	//printf ("c_ind[2]=%d\n", c_ind[2]);
			//printf ("IOP nb_fields=%d\n",nb_fields);
			if (nb_fields != 13)
				return 2;
			fgets (ligne, sizeof(ligne),i_file);	// PENOU 9 main 2011: lecture de ce qui reste sur la ligne

		}

		date.year += 1900;
		if (date.year<1932) date.year += 100;
		//printf ("REGLAGE year=%02d  month=%02d  day=%02d\n", date.year, date.month, date.day);


		/*........................................
		  .
		  .	Partie ecriture des fichiers :
		  .
		*/

		julian=tu_to_date(date.year,date.month,date.day,0,0,0);

		if (len_ofile<14) return 4;

		/* nom du nouveau fichier de donnees : */
		char *end_ofile = strrchr (output_file, '_') - 8;
		char buf[8+1];

		sprintf (buf, "%04d%02d%02d",date.year,date.month,date.day);
		strncpy (end_ofile, buf, 8);

		/* ouverture du fichier de sortie nouvelle date */
		if ( (o_file = fopen (output_file,"w+b"))==NULL) {
			return -2;
		}
		

		if (strcmp(typ,"dst") == 0) {

			for (i=0; i<24; i++) {
				// printf ("%.0f %d %hd %hd %hd\n",julian, elm, i_tab, i_hour[i],i_dm);

				nb_fields=Endian_write(& julian,sizeof(double),1,o_file)
					 +Endian_write(& elm,sizeof(char),1,o_file)
					 +Endian_write(& i_tab,sizeof(short),1,o_file)
					 +Endian_write(& i_hour[i],sizeof(short),1,o_file)
					 +Endian_write(& i_dm,sizeof(short),1,o_file);

				if (nb_fields != 5)  return 3;
				julian=julian+3600000;
			}

		} else if (strcmp(typ,"kpap") == 0) {

			for (i=0; i<8; i++) {

				//printf ("%.0f %hd %hd %hd %hd %hd %hd %f %hd %hd %f %hd\n",
				//	julian, s_ind[0], c_ind[0], c_kp[i],s_kp,s_ap[i],s_ap[8],flt[0],c_ind[1],s_ind[1],flt[1],c_ind[2]);
				nb_fields=Endian_write(& julian,sizeof(double),1,o_file)
					 +Endian_write(& s_ind[0],sizeof(short),1,o_file)
					 +Endian_write(& c_ind[0],sizeof(char),1,o_file)
					 +Endian_write(& c_kp[i],sizeof(char),1,o_file)
					 +Endian_write(& s_kp,sizeof(short),1,o_file)
					 +Endian_write(& s_ap[i],sizeof(short),1,o_file)
					 +Endian_write(& s_ap[8],sizeof(short),1,o_file)
					 +Endian_write(& flt[0],sizeof(float),1,o_file)
					 +Endian_write(& c_ind[1],sizeof(char),1,o_file)
					 +Endian_write(& s_ind[1],sizeof(short),1,o_file)
					 +Endian_write(& flt[1],sizeof(float),1,o_file)
					 +Endian_write(& c_ind[2],sizeof(char),1,o_file);
				if (nb_fields != 12)  return 3;
				
				julian += 10800000;	// 3 heures

			}

		}

		fclose (o_file); 

	}

	return 0;

}


/*..............................................................................
  .
  .		Lecture et ecriture des fichiers des stats ACE et GOES
  .
*/
/*---------------------------------------------------------------------------*/
int data_treatment (char *input_file,char* output_file,FILE *i_file,char* typ) {
/*---------------------------------------------------------------------------*/


	FILE* o_file=NULL;
	int err=0;
	int i;
	int zip=0;

	char buf[1024];
	int nb_fields;

	int day_before=0;
	double julian;

	s_date date;

	int i_stat[2];		/* tableau des status */
	unsigned char stat[2];	/* tableau des status */
	float flt[11];		/* tableau des donnees float */ 

	date.sec=0;

	/* Pour les pcn non zippes on zappe les 15 premieres lignes (commentaires) : */
	if (strcmp(typ,"pcn")==0) {
		for (i=0;i<15;i++) {
			fgets(buf,sizeof(buf),i_file);
		}
	}

	if (strcmp(typ,"zippcn")==0) {
		typ="pcn";
		zip=1;
	}

     
	while (fgets(buf,sizeof(buf),i_file)!=NULL) {

//printf ("PENOU: buf=%s\n",buf);

		if (buf[0]== ':' || buf[0]== '#' ) continue;

		//........................................
		// Partie Lecture des fichiers
		//........................................

		if (strcmp(typ,"xray")==0) {

			/* Si l'on a des donnees de type GOES/xray : */

			nb_fields=sscanf(buf,"%d %d %02d %02d%02d %d %d %e %e",
			&date.year, &date.month, &date.day, &date.hour,
			&date.min, &date.jday, &date.jsec, &flt[0], &flt[1]) ;
			if (nb_fields != 9) return 2;

			/* modif. des valeurs d'erreur pour CL */
			if ( (1-fabs(flt[0]/-1.00e+05)) < 1e-6 ) flt[0] = -1e31;
			if ( (1-fabs(flt[1]/-1.00e+05)) < 1e-6 ) flt[1] = -1e31;

		} else if (strcmp(typ,"geomag")==0) {

			/* Si l'on a des donnees de type GOES/geomag : */

			nb_fields=sscanf(buf,"%d %d %02d %02d%02d %d %d %e %e %e %e",
			&date.year, &date.month, &date.day, &date.hour,
			&date.min, &date.jday, &date.jsec, &flt[0], &flt[1], &flt[2], &flt[3]) ;
			if (nb_fields != 11) return 2;

			/* modif. des valeurs d'erreur pour CL */
			for (i=0; i<4; i++) {
				if ( (1-fabs(flt[i]/-1.00e+05)) < 1e-6 ) flt[i] = -1e31;
			}

		} else if (strcmp(typ,"particle")==0) {

			/* Si l'on a des donnees de type GOES/particle : */

			nb_fields=sscanf(buf,"%d %d %02d %02d%02d %d %d %e %e %e %e %e %e %e %e %e",
			&date.year, &date.month, &date.day, &date.hour,
			&date.min, &date.jday, &date.jsec, &flt[0], &flt[1], &flt[2], &flt[3],
			&flt[4], &flt[5], &flt[6], &flt[7], &flt[8]) ;
			if (nb_fields != 16) return 2;

			/* modif. des valeurs d'erreur pour CL */
			for (i=0; i<9; i++) {
				if ( (1-fabs(flt[i]/-1.00e+05)) < 1e-6 ) flt[i] = -1e31;
			}

		} else if (strcmp(typ,"pchan")==0) {

			/* Si l'on a des donnees de type GOES/pchan : */

			nb_fields=sscanf(buf,"%d %d %02d %02d%02d %d %d %e %e %e %e %e %e %e %e %e %e %e",
			&date.year, &date.month, &date.day, &date.hour,
			&date.min, &date.jday, &date.jsec, &flt[0], &flt[1], &flt[2], &flt[3],
			&flt[4], &flt[5], &flt[6], &flt[7], &flt[8], &flt[9], &flt[10]) ;
			if (nb_fields != 18) return 2;

			/* modif. des valeurs d'erreur pour CL */
			for (i=0; i<11; i++) {
				if ( (1-fabs(flt[i]/-1.00e+05)) < 1e-6 ) flt[i] = -1e31;
			}

		} else if (strcmp(typ,"sis")==0) {

			/* Si l'on a des donnees de type ACE/sis : */

			nb_fields=sscanf(buf,"%d %d %02d %02d%02d %d %d %d %e %d %e",
			&date.year, &date.month, &date.day, &date.hour,
			&date.min, &date.jday, &date.jsec, &i_stat[0], &flt[0], &i_stat[1], &flt[1]) ;
			if (nb_fields != 11) return 2;

			/* modif. des valeurs d'erreur pour CL */
			if ( (1-fabs(flt[0]/-1.00e+05)) < 1e-6 ) flt[0] = -1e31;
			if ( (1-fabs(flt[1]/-1.00e+05)) < 1e-6 ) flt[1] = -1e31;

		} else if (strcmp(typ,"epam")==0) {

			/* Si l'on a des donnees de type ACE/epam : */

			nb_fields=sscanf(buf,"%d %d %02d %02d%02d %d %d %d %e %e %d %e %e %e %e %e %f",
			&date.year, &date.month, &date.day, &date.hour,
			&date.min, &date.jday, &date.jsec, &i_stat[0], &flt[0], &flt[1],
			&i_stat[1], &flt[2], &flt[3], &flt[4], &flt[5], &flt[6], &flt[7]);

			if (nb_fields != 17) return 2;

			/* modif. des valeurs d'erreur pour CL */
			/* modif. des valeurs d'erreur pour CL */
			for (i=0; i<7; i++) {
				if ( (1-fabs(flt[i]/-1.00e+05)) < 1e-6 ) flt[i] = -1e31;
			}
			if ( (1-fabs(flt[7]/-1.00)) < 1e-6) flt[7] = -1e31;

		} else if (strcmp(typ,"mag")==0) {

			/* Si l'on a des donnees de type ACE/mag : */

			nb_fields=sscanf(buf,"%d %d %02d %02d%02d %d %d %d %f %f %f %f %f %f",
			&date.year, &date.month, &date.day, &date.hour,
			&date.min, &date.jday, &date.jsec, &i_stat[0], &flt[0], &flt[1],
			&flt[2], &flt[3], &flt[4], &flt[5]);

			if (nb_fields != 14) return 2;

			/* modif. des valeurs d'erreur pour CL */
			for (i=0; i<6; i++) {
				if ( (1-fabs(flt[i]/-999.9)) < 1e-6 ) flt[i] = -1e31;
			}

		} else if (strcmp(typ,"swepam")==0) {

			/* Si l'on a des donnees de type ACE/swepam : */

			nb_fields=sscanf(buf,"%d %d %02d %02d%02d %d %d %d %f %f %e",
			&date.year, &date.month, &date.day, &date.hour,
			&date.min, &date.jday, &date.jsec, &i_stat[0], &flt[0], &flt[1], &flt[2]);

			if (nb_fields != 11) return 2;

			/* modif. des valeurs d'erreur pour CL */
			if ( (1-fabs(flt[0]/-9999.9)) < 1e-6 ) flt[0] = -1e31;
			if ( (1-fabs(flt[1]/-9999.9)) < 1e-6 ) flt[1] = -1e31;
			if ( (1-fabs(flt[2]/-1.e+05)) < 1e-6 ) flt[2] = -1e31;

		} else if (strcmp(typ,"loc")==0) {

			/* Si l'on a des donnees de type ACE/loc : */

			nb_fields=sscanf(buf,"%d %d %02d %02d%02d %d %d %f %f %f",
			&date.year, &date.month, &date.day, &date.hour,
			&date.min, &date.jday, &date.jsec, &flt[0], &flt[1], &flt[2]);

			if (nb_fields != 10) return 2;

			/* modif. des valeurs d'erreur pour CL */
			if ( (1-fabs(flt[0]/-999.9)) < 1e-6 ) flt[0] = -1e31;
			if ( (1-fabs(flt[1]/-999.9)) < 1e-6 ) flt[1] = -1e31;
			if ( (1-fabs(flt[2]/-999.9)) < 1e-6 ) flt[2] = -1e31;

		} else if (strcmp(typ,"pcn")==0) {

			/* Si l'on a des donnees de type PCN : */

			nb_fields=sscanf(buf," %d| %d| %d| %d| %d| %f| %f|",
			&date.year, &date.month, &date.day, &date.hour, &date.min, &flt[0], &flt[1]);

			if (nb_fields != 7) {
				return 2;
			}

			/* calcul du date.jday, celui-ci n'etant pas dispo. dans les donnees d'entree */
			err=julday (date.month,date.day,date.year,&date.jday);
			if (err!=0) return 6 ;

			/* modif. des valeurs d'erreur pour CL */
			if ( (1-fabs(flt[0]/999.0)) < 1e-6 ) flt[0] = -1e31;

		}



		//........................................
		// Partie ecriture des fichiers
		//........................................

		o_file=gen_ofile(output_file,o_file,date,&day_before,&julian,zip,&err);
		if (err != 0) return err;


		if (strcmp(typ,"xray")==0) {

			nb_fields=Endian_write(& julian,sizeof(double),1,o_file)
				 +Endian_write(& flt[0],sizeof(float),1,o_file)
				 +Endian_write(& flt[1],sizeof(float),1,o_file);
			if (nb_fields != 3)  return 3;

		} else if (strcmp(typ,"geomag")==0) {

			nb_fields=Endian_write(& julian,sizeof(double),1,o_file);
			for (i=0; i<4; i++) {
				nb_fields=nb_fields+Endian_write(& flt[i],sizeof(float),1,o_file);
			}

			if (nb_fields != 5)  return 3;

		} else if (strcmp(typ,"particle")==0) {

			nb_fields=Endian_write(& julian,sizeof(double),1,o_file);
			for (i=0; i<9; i++) {
				nb_fields=nb_fields+Endian_write(& flt[i],sizeof(float),1,o_file);
			}

			if (nb_fields != 10)  return 3;

		} else if (strcmp(typ,"pchan")==0) {

			nb_fields=Endian_write(& julian,sizeof(double),1,o_file);
			for (i=0; i<11; i++) {
				nb_fields=nb_fields+Endian_write(& flt[i],sizeof(float),1,o_file);
			}

			if (nb_fields != 12)  return 3;

		} else if (strcmp(typ,"sis")==0) {

			stat[0] = i_stat[0];
			stat[1] = i_stat[1];
			nb_fields=Endian_write(& julian,sizeof(double),1,o_file)
				 +Endian_write(& stat[0],sizeof(char),1,o_file)
				 +Endian_write(& flt[0],sizeof(float),1,o_file)
				 +Endian_write(& stat[1],sizeof(char),1,o_file)
				 +Endian_write(& flt[1],sizeof(float),1,o_file);

			if (nb_fields != 5)  return 3;

		} else if (strcmp(typ,"epam")==0) {

			stat[0] = i_stat[0];
			stat[1] = i_stat[1];
			nb_fields=Endian_write(& julian,sizeof(double),1,o_file)
				 +Endian_write(& stat[0],sizeof(char),1,o_file)
				 +Endian_write(& flt[0],sizeof(float),1,o_file)
				 +Endian_write(& flt[1],sizeof(float),1,o_file)
				 +Endian_write(& stat[1],sizeof(char),1,o_file);
			for (i=2; i<8; i++) {
				nb_fields=nb_fields+Endian_write(& flt[i],sizeof(float),1,o_file);
			}

			if (nb_fields != 11)  return 3;

		} else if (strcmp(typ,"mag")==0) {

			stat[0] = i_stat[0];
			nb_fields=Endian_write(& julian,sizeof(double),1,o_file)
				 +Endian_write(& stat[0],sizeof(char),1,o_file);
			for (i=0; i<6; i++) {
				nb_fields=nb_fields+Endian_write(& flt[i],sizeof(float),1,o_file);
			}

			if (nb_fields != 8)  return 3;

		} else if (strcmp(typ,"swepam")==0) {

			stat[0] = i_stat[0];
			nb_fields=Endian_write(& julian,sizeof(double),1,o_file)
				 +Endian_write(& stat[0],sizeof(char),1,o_file)
				 +Endian_write(& flt[0],sizeof(float),1,o_file)
				 +Endian_write(& flt[1],sizeof(float),1,o_file)
				 +Endian_write(& flt[2],sizeof(float),1,o_file);

			if (nb_fields != 5)  return 3;

		} else if (strcmp(typ,"loc")==0) {

			nb_fields=Endian_write(& julian,sizeof(double),1,o_file)
				 +Endian_write(& flt[0],sizeof(float),1,o_file)
				 +Endian_write(& flt[1],sizeof(float),1,o_file)
				 +Endian_write(& flt[2],sizeof(float),1,o_file);

			if (nb_fields != 4)  return 3;

		} else if (strcmp(typ,"pcn")==0) {

			nb_fields=Endian_write(& julian,sizeof(double),1,o_file)
				 +Endian_write(& flt[0],sizeof(float),1,o_file);

			if (nb_fields != 2)  return 3;
		}
	}
	
	return 0;
}


/*---------------------------------------------------------------------------*/
int main (int argc, char *argv[]) {
/*---------------------------------------------------------------------------*/

	char	*prg;		/* nom du programme */
	char	*input_file;	/* fichier texte a lire */
	char	*output_file;	/* fichier binaire a ecrire */
	int	err=0;		/* type d'erreur en sortie */
	char	cmd[1024];	/* commande shell pour le popen */
	FILE*	i_file;
	char	base[1024];
	char	*p;

	input_file = argv[1];
	output_file = argv[2];
	prg = argv[0];

	if (argc < 4) {

		err=1;

	} else {

		// PENOU le 27/09/2021
		// Entre le 12/12/2017 et le 28/11/2018 il y a en même temps des données xray pour GOES-14 et GOES-15
		// Cela pose des problèmes car les données GOES-14 ont l'air incorrectes, il vaut mieux utiliser celles de GOES-15
		// Solution: ne pas créer les fichiers dans cette période pour GOES-14
		// Ne pas créer de fichier goes14_?m_xray_YYYYMMDD_v01.bin si YYYYMMDD >= 20101026
		p = strstr(output_file,"goes14");
		if (p != NULL) {
			if (strncmp(p+10,"xray",4)==0) {
				if (strcmp(p+15,"20101026")>0) {
					printf ("ne pas créer le fichier\n");
					return 0;
				}
			}
		}

		// nbfields=sscanf(argv[3],"%3s",&zip);
		
		if (strcmp(argv[3],"zippcn")==0) {

			/* lancement du shell pour lire le fichier zippe */
			// sprintf (cmd, "unzip -p %s | sed '/a/d;/e/d;/A/d' | sort -n", input_file);
			sprintf (cmd, "unzip -p %s | sed '/a/d;/e/d;/A/d'", input_file);
			if ((i_file = popen (cmd,"r"))==NULL) {
				err=-1;
				error_msg(err,prg);
				return err;
			}

		} else {

			/* ouverture des fichiers */
			if ((i_file = fopen (input_file,"r"))==NULL){
				err=-1;
				error_msg(err,prg);
				return err;
			}
		}
	
		if (! (strcmp(argv[3],"xray") && strcmp(argv[3],"geomag") 
		      && strcmp(argv[3],"particle") && strcmp(argv[3],"pchan") \
		      && strcmp(argv[3],"sis") && strcmp(argv[3],"swepam") \
                      && strcmp(argv[3],"epam") && strcmp(argv[3],"loc") \
                      && strcmp(argv[3],"mag") && strcmp(argv[3],"pcn")&& strcmp(argv[3],"zippcn")) ) {

			err=data_treatment(input_file,output_file,i_file,argv[3]);	/* fichiers de type GOES, ACE & PCN */

		} else if (strcmp(argv[3],"dst")==0) {

			//	Il ne faut traiter que les fichiers sans extension. Par exemple, pour 2006, nous avons les fichiers suivants:
			//		44165 jan 24 19:45 dst2006
			//		44045 déc  5  2007 dst2006.v12
			//		44165 jan 24 19:45 dst2006.v12.corrected
			//		7140 mar  6  2006 dst2006.v2
			//		10891 avr  4  2006 dst2006.v3
			//		14521 aoû  9  2006 dst2006.v4
			//		18272 jun  5  2006 dst2006.v5
			//		21902 jui 10  2006 dst2006.v6

			//		Il suffit de traiter le fichier "dst2006"

			strcpy (base, input_file);
			if (strlen(basename(base)) != 7) {
				err=10;
			} else {
				err=ind_treatment(input_file,output_file,i_file,argv[3]);	/* fichiers de type DST */
			}			

		} else if (strcmp(argv[3],"kpap")==0) { 				/* fichiers de type KPAP */


			//	Il ne faut traiter que les fichiers sans extension. Par exemple, pour 2006, nous avons les fichiers suivants:
			//		26280 jui  6  2007 2006
			//		26280 jan  5  2007 2006.v12
			//		24048 déc  6  2006 2006.v11
			//		21888 nov 29  2006 2006.v10
			//		21888 nov 29  2006 2006.v10.rev
			//		19656 oct 11  2006 2006.v9
			//		17496 sep  7  2006 2006.v8
			//		15264 aoû 24  2006 2006.v7
			//		13032 jui 19  2006 2006.v6
			//		10872 jun  5  2006 2006.v5
			//		8640 mai  5  2006 2006.v4
			//		6480 avr 20  2006 2006.v3
			//		4248 mar  6  2006 2006.v2
			//		2232 fév  9  2006 2006.v1
			//		Il suffit de traiter le fichier "2006"

			strcpy (base, input_file);
			if (strlen(basename(base)) != 4) {
				err=10;
			} else {
				err = ind_treatment (input_file,output_file,i_file,argv[3]);
			}

		} else if (strcmp(argv[3],"ae2")==0) {

			err=ae2_treatment(input_file,output_file,i_file);

		} else if (strcmp(argv[3],"ae3")==0) {

			err=ae3_treatment(input_file,output_file,i_file);

		} else if (strcmp(argv[3],"ae")==0) {

			err=ae_treatment(input_file,output_file,i_file);

		} else {

			err=5;
		}

			
		if (strcmp(argv[3],"zippcn")==0)
			pclose(i_file);
		else
			fclose(i_file);
	}

	error_msg(err,prg);

	return err;

}
