;+
; Project     :	STEREO - IMPACT, SWEA, STE
;
; Name        : SWEASTE_270
;
; Purpose     :	Convert SWEA and STE beacon data to physical units
;
; Category    :	STEREO, IMPACT, SWEA, STE
;
; Explanation :	
;
; Syntax      :	data = SWEASTE_BEACON(PACKET)
;
; Examples    :	
;
; Inputs      :	PACKET - an IMPACT beacon telemetry packet (apid=624)
;
; Opt. Inputs :	None.
;
; Outputs     :	A structure containing SWEA and STE Beacon data
;
; Opt. Outputs:	None.
;
; Keywords    :	None
;
; Calls       :	None
;
; Common      : sweaste_b_common
;
; Restrictions:	
;
; Side effects:	None.
;
; Prev. Hist. :	None.
;
; History     :	Version 2 (preliminary), 10-Nov-2005, Peter Schroeder, Nicolas Pene
;                  UC-Berkeley, CESR.
;
; Contact     :	peters@ssl.berkeley.edu, nicolas.pene@cesr.fr
;-
;

;quick function for turn bytes into uints
function uint_from_bytes, twobytes
return, ishft(uint(twobytes[1]), 8) + uint(twobytes[0])
end

function sweaste_270, packet, compressed=compressed

;-------------------------
;   Var SWEASTE
common sweaste_b_common    ; use common block
SWEASTE_B = 45
;SWEASTE_BLOCKSIZE = 70
SWEAV0_A1 = 0.03152585d
SWEAV0_A0 = -64.56494d
;SWEAMOMCOEFFS = [22d, 10d, 4d, 19d]
;SWEATACC = 5.8d-3
;SWEAH = 1.746d-7
;SWEAK = 6.67d
;SWEAPADE0 = [6,7,8,9,10,11]
;SWEAPADE1 = [24,25,26,27,28,29]


;figure out rotation matrixfrom SWEA to Spacecraft Coordinates
if (packet.grh.spacecraft_id eq 'EA'XU) then SWEATOSC = 40d else $
   SWEATOSC = -140d

SWEATOSC = SWEATOSC/!RADEG

SWEASCrotmat = [ [1, 0, 0], [0, cos(SWEATOSC), sin(SWEATOSC)], $
   [0, -sin(SWEATOSC), cos(SWEATOSC)] ]

;grab SWEA/STE bytes from packet, swap bytes   
rawbytes = bytarr(SWEASTE_BLOCKSIZE)
rawbytes = packet.dat[SWEASTE_B:SWEASTE_B+SWEASTE_BLOCKSIZE-1]
swap,rawbytes

;raw byte values from packet
rawdatastr = { $
   SWEA_N: uint_from_bytes(rawbytes[0:1]), $
   SWEA_Vx: uint_from_bytes(rawbytes[2:3]), $
   SWEA_Vy: uint_from_bytes(rawbytes[4:5]), $
   SWEA_Vz: uint_from_bytes(rawbytes[6:7]), $
   SWEA_Pxx: uint_from_bytes(rawbytes[8:9]), $
   SWEA_Pyy: uint_from_bytes(rawbytes[10:11]), $
   SWEA_Pzz: uint_from_bytes(rawbytes[12:13]), $
   SWEA_Pxy: uint_from_bytes(rawbytes[14:15]), $
   SWEA_Pxz: uint_from_bytes(rawbytes[16:17]), $
   SWEA_Pyz: uint_from_bytes(rawbytes[18:19]), $
   SWEA_NHx: uint_from_bytes(rawbytes[20:21]), $
   SWEA_NHy: uint_from_bytes(rawbytes[22:23]), $
   SWEA_NHz: uint_from_bytes(rawbytes[24:25]), $
   SWEA_V0: uint_from_bytes(rawbytes[26:27]), $
   SWEA_MagAz: rawbytes[28], $
   SWEA_MagEl: rawbytes[29], $
   SWEA_PAD: rawbytes[30:53], $
   STE_Uns: rawbytes[54:58], $
   STE_Us: rawbytes[59:63], $
   STE_D: rawbytes[64:68], $
   STE_LLD: rawbytes[69] }

;define output structure

;if (rawdatastr.SWEA_V0 ne 2048) then print,"V0 touch",rawdatastr.SWEA_V0,SWEAV0_A1,SWEAV0_A0

;---------------------
; Log Compressed data
if keyword_set(compressed) then begin

	ret_struct = { $
	   SWEA_N: impact_moment_uncomp(rawdatastr.SWEA_N), $
	   SWEA_V: [impact_moment_uncomp(rawdatastr.SWEA_Vx), $
	      impact_moment_uncomp(rawdatastr.SWEA_Vy), $
	      impact_moment_uncomp(rawdatastr.SWEA_Vz)], $
	   SWEA_P: [impact_moment_uncomp(rawdatastr.SWEA_Pxx), $
	      impact_moment_uncomp(rawdatastr.SWEA_Pyy), $
	      impact_moment_uncomp(rawdatastr.SWEA_Pzz), $
	      impact_moment_uncomp(rawdatastr.SWEA_Pxy), $
	      impact_moment_uncomp(rawdatastr.SWEA_Pxz), $
	      impact_moment_uncomp(rawdatastr.SWEA_Pyz)], $
	   SWEA_NH: [impact_moment_uncomp(rawdatastr.SWEA_NHx), $
	      impact_moment_uncomp(rawdatastr.SWEA_NHy), $
	      impact_moment_uncomp(rawdatastr.SWEA_NHz)], $
	   SWEA_V0: double(rawdatastr.SWEA_V0)*SWEAV0_A1 + SWEAV0_A0, $
	   SWEA_MagAz: double(rawdatastr.SWEA_MagAz)*360d/256d, $
	   SWEA_MagEl: double(rawdatastr.SWEA_MagEl)*360d/256d, $
	   SWEA_PAD: bytarr(24), $
	   STE_Uns: ulonarr(5), $
	   STE_Us: ulonarr(5), $
	   STE_D: ulonarr(5), $
	   STE_LLD: impact_log_uncomp(rawdatastr.STE_LLD) }

;print,"V0 touch",rawdatastr.SWEA_V0, ret_struct.SWEA_V0

	;fill PAD and STE arrays with uncompressed data
	ret_struct.SWEA_PAD = rawdatastr.SWEA_PAD
	for i=0,4 do begin
	   ret_struct.STE_Uns[i] = impact_log_uncomp(rawdatastr.STE_Uns[i]) * 8l
	   ret_struct.STE_Us[i] = impact_log_uncomp(rawdatastr.STE_Us[i]) * 8l
	   ret_struct.STE_D[i] = impact_log_uncomp(rawdatastr.STE_D[i]) *8l
	endfor

endif else begin
;---------------------
; Uncompressed data

	ret_struct = { $
	   SWEA_N: impact_moment_uncomp(rawdatastr.SWEA_N), $
	   SWEA_V: [impact_moment_uncomp(rawdatastr.SWEA_Vx), $
	      impact_moment_uncomp(rawdatastr.SWEA_Vy), $
	      impact_moment_uncomp(rawdatastr.SWEA_Vz)], $
	   SWEA_P: [impact_moment_uncomp(rawdatastr.SWEA_Pxx), $
	      impact_moment_uncomp(rawdatastr.SWEA_Pyy), $
	      impact_moment_uncomp(rawdatastr.SWEA_Pzz), $
	      impact_moment_uncomp(rawdatastr.SWEA_Pxy), $
	      impact_moment_uncomp(rawdatastr.SWEA_Pxz), $
	      impact_moment_uncomp(rawdatastr.SWEA_Pyz)], $
	   SWEA_NH: [impact_moment_uncomp(rawdatastr.SWEA_NHx), $
	      impact_moment_uncomp(rawdatastr.SWEA_NHy), $
	      impact_moment_uncomp(rawdatastr.SWEA_NHz)], $
	   SWEA_V0: double(rawdatastr.SWEA_V0)*SWEAV0_A1 + SWEAV0_A0, $
	   SWEA_MagAz: double(rawdatastr.SWEA_MagAz)*360d/256d, $
	   SWEA_MagEl: double(rawdatastr.SWEA_MagEl)*360d/256d, $
	   SWEA_PAD: ulonarr(24), $
	   STE_Uns: ulonarr(5), $
	   STE_Us: ulonarr(5), $
	   STE_D: ulonarr(5), $
	   STE_LLD: impact_log_uncomp(rawdatastr.STE_LLD) }

	;fill PAD and STE arrays with uncompressed data
	for i=0,23 do ret_struct.SWEA_PAD[i] = impact_log_uncomp(rawdatastr.SWEA_PAD[i])
	for i=0,4 do begin
	   ret_struct.STE_Uns[i] = impact_log_uncomp(rawdatastr.STE_Uns[i]) * 8l
	   ret_struct.STE_Us[i] = impact_log_uncomp(rawdatastr.STE_Us[i]) * 8l
	   ret_struct.STE_D[i] = impact_log_uncomp(rawdatastr.STE_D[i]) *8l
	endfor

endelse

;apply moments conversions
if ret_struct.SWEA_N ne 0d then begin
   ret_struct.SWEA_N = ret_struct.SWEA_N * (SWEAK * SWEAH)^(-0.5d) * $
      2d^(-11d - SWEAMOMCOEFFS[0]) / SWEATACC
   ret_struct.SWEA_V = ret_struct.SWEA_V * 2d^(-11d - SWEAMOMCOEFFS[1]) / $
      SWEATACC / ret_struct.SWEA_N
   ret_struct.SWEA_P = ret_struct.SWEA_P * (SWEAK * SWEAH)^(0.5d) * $
      2d^(-11d - SWEAMOMCOEFFS[2]) / SWEATACC / ret_struct.SWEA_N
;
;******************************************************************************
;
;  Modified this section, based on what I think the program was actually
;  supposed to do.  William Thompson
;
;   ret_struct.SWEA_NHx = ret_struct.SWEA_NHx * (SWEAK * SWEAH) * $
;      2d^(-11d - SWEAMOMCOEFFS[3]) / SWEATACC / ret_struct.SWEA_N
;   ret_struct.SWEA_NHy = ret_struct.SWEA_NHy * (SWEAK * SWEAH) * $
;      2d^(-11d - SWEAMOMCOEFFS[3]) / SWEATACC / ret_struct.SWEA_N
;   ret_struct.SWEA_NHz = ret_struct.SWEA_NHz * (SWEAK * SWEAH) * $
;      2d^(-11d - SWEAMOMCOEFFS[3]) / SWEATACC / ret_struct.SWEA_N
;
   ret_struct.SWEA_NH[0] = ret_struct.SWEA_NH[0] * (SWEAK * SWEAH) * $
      2d^(-11d - SWEAMOMCOEFFS[3]) / SWEATACC / ret_struct.SWEA_N
   ret_struct.SWEA_NH[1] = ret_struct.SWEA_NH[1] * (SWEAK * SWEAH) * $
      2d^(-11d - SWEAMOMCOEFFS[3]) / SWEATACC / ret_struct.SWEA_N
   ret_struct.SWEA_NH[2] = ret_struct.SWEA_NH[2] * (SWEAK * SWEAH) * $
      2d^(-11d - SWEAMOMCOEFFS[3]) / SWEATACC / ret_struct.SWEA_N
;
;******************************************************************************
;
endif

;rotate SWEA moments into Spacecraft frame
ret_struct.SWEA_V = SWEASCrotmat # ret_struct.SWEA_V
ret_struct.SWEA_NH = SWEASCrotmat # ret_struct.SWEA_NH

P = ret_struct.SWEA_P
P_tensor = [ [P[0], P[3], P[4]], $
   [P[3], P[1], P[5]], $
   [P[4], P[5], P[2]] ]
P_tensor = SWEASCrotmat # P_tensor # transpose(SWEASCrotmat)
ret_struct.SWEA_P = [P_tensor[0,0], P_tensor[1,1], P_tensor[2,2],$
   P_tensor[0,1], P_tensor[0,2], P_tensor[1,2]]

return,ret_struct
end
