;+
; Project     :	STEREO - IMPACT, SWEA
;
; Name        : SWEA_270
;
; Purpose     :	Process SWEA Pkt 270 Into Physical Units
;
; Category    :	STEREO, IMPACT, SWEA
;
; Explanation :	
;
; Syntax      :	data = SWEA_270(PACKET)
;
; Examples    :	
;
; Inputs      :	PACKET - an IMPACT Beacon Telemetry packet (apid=270h)
;
; Opt. Inputs :	None.
;
; Outputs     :	A structure containing SWEA Moments data
;
; Opt. Outputs:	None.
;
; Keywords    :	None
;
; Calls       :	None
;
; Common      : swea_l1_common
;
; Restrictions:	
;
; Side effects:	None.
;
; Prev. Hist. :	None.
;
; History     :	Version 1 (preliminary), 05-Mar-2006, Peter Schroeder,
;                  UC-Berkeley.
;
; Contact     :	peters@ssl.berkeley.edu
;-
;

function swea_270, packet

common swea_l1_common ;use common block

;check checksum
chksum = 0b
hdrbytarray = bytarr(11)
load_bytarr,packet.pkt,hdrbytarray
for i = 0, 10 do chksum += hdrbytarray[i]
for i = 0, 260 do chksum += packet.dat[i]
if chksum ne 0 then return, -1


;grab SWEA bytes from packet, swap bytes   
swea_offset=44
rawbytes = packet.dat[swea_offset:swea_offset+70]
swap,rawbytes

;calculate packet time
utcpkttim = parse_stereo_pkt(packet,/pkt_date)



;---------------------------------------------------------
;
;                      SWEA & STE data

momentstr = { $
   SWEA_N: uint_from_bytes(rawbytes[0:1]), $
   SWEA_Vx: uint_from_bytes(rawbytes[2:3]), $
   SWEA_Vy: uint_from_bytes(rawbytes[4:5]), $
   SWEA_Vz: uint_from_bytes(rawbytes[6:7]), $
   SWEA_Pxx: uint_from_bytes(rawbytes[8:9]), $
   SWEA_Pyy: uint_from_bytes(rawbytes[10:11]), $
   SWEA_Pzz: uint_from_bytes(rawbytes[12:13]), $
   SWEA_Pxy: uint_from_bytes(rawbytes[14:15]), $
   SWEA_Pxz: uint_from_bytes(rawbytes[16:17]), $
   SWEA_Pyz: uint_from_bytes(rawbytes[18:19]), $
   SWEA_NHx: uint_from_bytes(rawbytes[20:21]), $
   SWEA_NHy: uint_from_bytes(rawbytes[22:23]), $
   SWEA_NHz: uint_from_bytes(rawbytes[24:25]), $
   SWEA_V0: uint_from_bytes(rawbytes[26:27]), $
   SWEA_MagAz: rawbytes[28], $
   SWEA_MagEl: rawbytes[29], $
   SWEA_PAD: rawbytes[30:53], $
   STE_Uns: rawbytes[54:58], $
   STE_Us: rawbytes[59:63], $
   STE_D: rawbytes[64:68], $
   STE_LLD: rawbytes[69] }
}

rawdatastr = { $
	Moments: replicate(momentstr,1) }

rawbytearray = bytarr(28)
rawbytearray[*] = rawbytes[44:71]

load_struct,rawbytearray,momentstr,rawdata

rawdatastr.Moments = rawdata



PAD_struct = { $
   MagAz: float(rawbytes[72])/256.*360., $
   MagEl: float(rawbytes[73])/256.*360., $
   PAD: fltarr(7,6,2) }

for e = 0,6 do $
   for d = 0,5 do $
     for a =0,1 do PAD_struct.PAD[e,d,a] = $
        impact_log_uncomp(rawbytes[74+a+2*d+12*e])

;---------------------------------------------------------
;
;                      output data


ret_mom_struct = { $
   N: 0d, $
   NVx: 0d, $
   NVy: 0d, $
   NVz: 0d, $
   NPxx: 0d, $
   NPyy: 0d, $
   NPzz: 0d, $
   NPxy: 0d, $
   NPxz: 0d, $
   NPyz: 0d, $
   NHx: 0d, $
   NHy: 0d, $
   NHz: 0d, $
   V0: 0b }


ret_struct = { $
   time: utcpkttim, $
   ISCStatus: rawbytes[0], $
   InterfActive: rawbytes[1], $
   SWEAModeID: rawbytes[2], $
   STEModeID: rawbytes[3], $
   IAutoLUT: rawbytes[4], $
   SWEADHKP: rawbytes[5], $
   MAG: replicate(MAG_Struct,1), $
   SWEASTE: replicate(SWEASTE_Struct,1), $
   SWEAMom: replicate(ret_mom_struct,1), $
   SWEAPAD: replicate(PAD_struct,1) }


;------------------
; SWEASTE output :
ret_struct.SWEASTE = SWEASTE_Struct

;------------------
; MAG output :
;ret_struct.MAG = MAG_struct

;------------------
; SWEA output :
ret_struct.SWEAMom.N = impact_moment_uncomp(rawdatastr.Moments.N)
ret_struct.SWEAMom.NVx = impact_moment_uncomp(rawdatastr.Moments.NVx)
ret_struct.SWEAMom.NVy = impact_moment_uncomp(rawdatastr.Moments.NVy)
ret_struct.SWEAMom.NVz = impact_moment_uncomp(rawdatastr.Moments.NVz)
ret_struct.SWEAMom.NPxx = impact_moment_uncomp(rawdatastr.Moments.NPxx)
ret_struct.SWEAMom.NPyy = impact_moment_uncomp(rawdatastr.Moments.NPyy)
ret_struct.SWEAMom.NPzz = impact_moment_uncomp(rawdatastr.Moments.NPzz)
ret_struct.SWEAMom.NPxy = impact_moment_uncomp(rawdatastr.Moments.NPxy)
ret_struct.SWEAMom.NPxz = impact_moment_uncomp(rawdatastr.Moments.NPxz)
ret_struct.SWEAMom.NPyz = impact_moment_uncomp(rawdatastr.Moments.NPyz)
ret_struct.SWEAMom.NHx = impact_moment_uncomp(rawdatastr.Moments.NHx)
ret_struct.SWEAMom.NHy = impact_moment_uncomp(rawdatastr.Moments.NHy)
ret_struct.SWEAMom.NHz = impact_moment_uncomp(rawdatastr.Moments.NHz)
ret_struct.SWEAMom.V0 = impact_moment_uncomp(rawdatastr.Moments.V0)
ret_struct.SWEAPAD = PAD_struct

return, ret_struct

end
