;+
; Project     :	STEREO - IMPACT, SWEA
;
; Name        : SWEA_213
;
; Purpose     :	Process SWEA Pkt 213 Into Physical Units
;
; Category    :	STEREO, IMPACT, SWEA
;
; Explanation :	
;
; Syntax      :	data = SWEA_213(PACKET)
;
; Examples    :	
;
; Inputs      :	PACKET - an IMPACT SWEA Moments Telemetry packet (apid=213h)
;
; Opt. Inputs :	None.
;
; Outputs     :	A structure containing SWEA Moments data
;
; Opt. Outputs:	None.
;
; Keywords    :	None
;
; Calls       :	None
;
; Common      : swea_l1_common
;
; Restrictions:	
;
; Side effects:	None.
;
; Prev. Hist. :	None.
;
; History     :	Version 1 (preliminary), 05-Mar-2006, Peter Schroeder,
;                  UC-Berkeley.
;
; Contact     :	peters@ssl.berkeley.edu
;-
;

function swea_213, packet

common swea_l1_common ;use common block

;check checksum
chksum = 0b
hdrbytarray = bytarr(11)
load_bytarr,packet.pkt,hdrbytarray
for i = 0, 10 do chksum += hdrbytarray[i]
for i = 0, 260 do chksum += packet.dat[i]
if chksum ne 0 then return, -1

;figure out rotation matrix from SWEA to Spacecraft Coordinates
if (packet.grh.spacecraft_id eq 'EA'XU) then SWEATOSC = 40d else $
   SWEATOSC = -140d

SWEATOSC = SWEATOSC/!RADEG

SWEASCrotmat = [ [1, 0, 0], [0, cos(SWEATOSC), sin(SWEATOSC)], $
   [0, -sin(SWEATOSC), cos(SWEATOSC)] ]

;grab SWEA bytes from packet, swap bytes   
rawbytes = packet.dat[1:*]
swap,rawbytes

;calculate packet time
utcpkttim = parse_stereo_pkt(packet,/pkt_date)

momentstr = { $
   N: uint(0), $
   NVx: uint(0), $
   NVy: uint(0), $
   NVz: uint(0), $
   NPxx: uint(0), $
   NPyy: uint(0), $
   NPzz: uint(0), $
   NPxy: uint(0), $
   NPxz: uint(0), $
   NPyz: uint(0), $
   NHx: uint(0), $
   NHy: uint(0), $
   NHz: uint(0), $
   V0: 0b, $
   SWEADHKP: 0b }

;raw byte values from packet
SWEAMomInt = packet.dat[0]
rawdatastr = { $
   SWEAModeID: rawbytes[0], $
   SWEASweepMax: uint_from_bytes(rawbytes[1:2]), $
   SWEASweepRate: uint_from_bytes(rawbytes[3:4]), $
   SWEASweepDeflStep: uint_from_bytes(rawbytes[5:6]), $
   Moments: replicate(momentstr,9), $
   Checksum: 0b }

rawbytearray = bytarr(28,9)
for i=0,8 do rawbytearray[*,i] = rawbytes[28*i+7: 28*i+34]

load_struct,rawbytearray,momentstr,rawdata

rawdatastr.Moments = rawdata

ret_mom_struct = { $
   N: 0., $
   NVx: 0., $
   NVy: 0., $
   NVz: 0., $
   NPxx: 0., $
   NPyy: 0., $
   NPzz: 0., $
   NPxy: 0., $
   NPxz: 0., $
   NPyz: 0., $
   NHx: 0., $
   NHy: 0., $
   NHz: 0., $
   V0: 0b, $
   SWEADHKP: 0b }

ret_struct = { $
   time: utcpkttim, $
   SWEAMomInt: SWEAMomInt, $
   SWEAModeID: rawdatastr.SWEAModeID, $
   SWEASweepMax: rawdatastr.SWEASweepMax, $
   SWEASweepRate: rawdatastr.SWEASweepRate, $
   SWEASweepDeflStep: rawdatastr.SWEASweepDeflStep, $
   Moments: replicate(ret_mom_struct,9) }

for i=0,8 do begin
   ret_struct.Moments[i].N = float(impact_moment_uncomp(rawdatastr.Moments[i].N))
   ret_struct.Moments[i].NVx = float(impact_moment_uncomp(rawdatastr.Moments[i].NVx))
   ret_struct.Moments[i].NVy = float(impact_moment_uncomp(rawdatastr.Moments[i].NVy))
   ret_struct.Moments[i].NVz = float(impact_moment_uncomp(rawdatastr.Moments[i].NVz))
   ret_struct.Moments[i].NPxx = float(impact_moment_uncomp(rawdatastr.Moments[i].NPxx))
   ret_struct.Moments[i].NPyy = float(impact_moment_uncomp(rawdatastr.Moments[i].NPyy))
   ret_struct.Moments[i].NPzz = float(impact_moment_uncomp(rawdatastr.Moments[i].NPzz))
   ret_struct.Moments[i].NPxy = float(impact_moment_uncomp(rawdatastr.Moments[i].NPxy))
   ret_struct.Moments[i].NPxz = float(impact_moment_uncomp(rawdatastr.Moments[i].NPxz))
   ret_struct.Moments[i].NPyz = float(impact_moment_uncomp(rawdatastr.Moments[i].NPyz))
   ret_struct.Moments[i].NHx = float(impact_moment_uncomp(rawdatastr.Moments[i].NHx))
   ret_struct.Moments[i].NHy = float(impact_moment_uncomp(rawdatastr.Moments[i].NHy))
   ret_struct.Moments[i].NHz = float(impact_moment_uncomp(rawdatastr.Moments[i].NHz))
   ret_struct.Moments[i].V0 = rawdatastr.Moments[i].V0
   ret_struct.Moments[i].SWEADHKP = rawdatastr.Moments[i].SWEADHKP
endfor


return, ret_struct

end
