;+
; Project     :	STEREO - IMPACT, SWEA
;
; Name        : SWEA_212
;
; Purpose     :	Process SWEA Pkt 212 Into Physical Units
;
; Category    :	STEREO, IMPACT, SWEA
;
; Explanation :	
;
; Syntax      :	data = SWEA_212(PACKET)
;
; Examples    :	
;
; Inputs      :	PACKET - an IMPACT SWEA PAD Telemetry packet (apid=212h)
;
; Opt. Inputs :	None.
;
; Outputs     :	A structure containing SWEA PAD data
;
; Opt. Outputs:	None.
;
; Keywords    :	None
;
; Calls       :	None
;
; Common      : swea_l1_common
;
; Restrictions:	
;
; Side effects:	None.
;
; Prev. Hist. :	None.
;
; History     :	Version 1 (preliminary), 03-Feb-2006, Peter Schroeder, Nicolas Pene
;                  UC-Berkeley & CESR
;
; Contact     :	peters@ssl.berkeley.edu, nicolas.pene@cesr.fr
;-
;

function swea_212, packet, compressed=compressed

common swea_l1_common ;use common block

;check checksum
chksum = 0b
hdrbytarray = bytarr(11)
load_bytarr,packet.pkt,hdrbytarray
for i = 0, 10 do chksum += hdrbytarray[i]
for i = 0, 260 do chksum += packet.dat[i]
if chksum ne 0 then return, -1

;compute SWEA energies
if not keyword_set(swea_energies) then swea_l1_energies

;figure out rotation matrix from SWEA to Spacecraft Coordinates
if (packet.grh.spacecraft_id eq 'EA'XU) then SWEATOSC = 40d else $
   SWEATOSC = -140d

SWEATOSC = SWEATOSC/!RADEG

SWEASCrotmat = [ [1, 0, 0], [0, cos(SWEATOSC), sin(SWEATOSC)], $
   [0, -sin(SWEATOSC), cos(SWEATOSC)] ]

;grab SWEA bytes from packet, swap bytes   
rawbytes = packet.dat[1:*]
swap,rawbytes

;calculate packet time
utcpkttim = parse_stereo_pkt(packet,/pkt_date)

PAD = {MagAz: 0b, $
   MagEl: 0b, $
   Dist: bytarr(84) }

;raw byte values from packet
rawdatastr = { $
   SWEADistInt: packet.dat[0], $
   SWEAModeID: rawbytes[0], $
   MagAz1: rawbytes[1], $
   MagEl1: rawbytes[2], $
   PAD1: rawbytes[3:86], $
   MagAz2: rawbytes[87], $
   MagEl2: rawbytes[88], $
   PAD2: rawbytes[89:172], $
   MagAz3: rawbytes[173], $
   MagEl3: rawbytes[174], $
   PAD3: rawbytes[175:258], $
   Checksum: rawbytes[259] }

utcsec = utc2sec(utcpkttim)

;---------------------
; Log Compressed data
if keyword_set(compressed) then begin
	PAD_struct1 = { $
	   time: anytim2utc(utcsec + double(rawdatastr.SWEADistInt)/2d), $
	   MagAz: float(rawdatastr.MagAz1)/256.*360., $
	   MagEl: float(rawdatastr.MagEl1)/256.*360., $
	   PAD: bytarr(7,6,2) }

	for e = 0,6 do $
	   for d = 0,5 do $
	     for a =0,1 do PAD_struct1.PAD[e,d,a] = $
		rawdatastr.PAD1[a+2*d+12*e]

	PAD_struct2 = { $
	   time: anytim2utc(utcsec + double(rawdatastr.SWEADistInt)*1.5d), $
	   MagAz: float(rawdatastr.MagAz2)/256.*360., $
	   MagEl: float(rawdatastr.MagEl2)/256.*360., $
	   PAD: bytarr(7,6,2) }

	for e = 0,6 do $
	   for d = 0,5 do $
	     for a =0,1 do PAD_struct2.PAD[e,d,a] = $
		rawdatastr.PAD2[a+2*d+12*e]

	PAD_struct3 = { $
	   time: anytim2utc(utcsec + double(rawdatastr.SWEADistInt)*2.5d), $
	   MagAz: float(rawdatastr.MagAz3)/256.*360., $
	   MagEl: float(rawdatastr.MagEl3)/256.*360., $
	   PAD: bytarr(7,6,2) }

	for e = 0,6 do $
	   for d = 0,5 do $
	     for a =0,1 do PAD_struct3.PAD[e,d,a] = $
		rawdatastr.PAD3[a+2*d+12*e]

endif else begin
;---------------------
; Uncompressed data

	PAD_struct1 = { $
	   time: anytim2utc(utcsec + double(rawdatastr.SWEADistInt)/2d), $
	   MagAz: float(rawdatastr.MagAz1)/256.*360., $
	   MagEl: float(rawdatastr.MagEl1)/256.*360., $
	   PAD: intarr(7,6,2) }

	for e = 0,6 do $
	   for d = 0,5 do $
	     for a =0,1 do PAD_struct1.PAD[e,d,a] = $
		impact_log_uncomp(rawdatastr.PAD1[a+2*d+12*e])

	PAD_struct2 = { $
	   time: anytim2utc(utcsec + double(rawdatastr.SWEADistInt)*1.5d), $
	   MagAz: float(rawdatastr.MagAz2)/256.*360., $
	   MagEl: float(rawdatastr.MagEl2)/256.*360., $
	   PAD: intarr(7,6,2) }

	for e = 0,6 do $
	   for d = 0,5 do $
	     for a =0,1 do PAD_struct2.PAD[e,d,a] = $
		impact_log_uncomp(rawdatastr.PAD2[a+2*d+12*e])

	PAD_struct3 = { $
	   time: anytim2utc(utcsec + double(rawdatastr.SWEADistInt)*2.5d), $
	   MagAz: float(rawdatastr.MagAz3)/256.*360., $
	   MagEl: float(rawdatastr.MagEl3)/256.*360., $
	   PAD: intarr(7,6,2) }

	for e = 0,6 do $
	   for d = 0,5 do $
	     for a =0,1 do PAD_struct3.PAD[e,d,a] = $
		impact_log_uncomp(rawdatastr.PAD3[a+2*d+12*e])


endelse

ret_struct = { $
 TIME: utcpkttim, $
 SWEAPADInt: rawdatastr.SWEADistInt, $
 SWEAModeID: rawdatastr.SWEAModeID, $
 SWEAPAD: replicate(PAD_struct1,3) }

ret_struct.SWEAPAD[0] = PAD_struct1
ret_struct.SWEAPAD[1] = PAD_struct2
ret_struct.SWEAPAD[2] = PAD_struct3


return, ret_struct

end
