;+
; Project     :	STEREO - IMPACT, SWEA
;
; Name        : SWEA_210
;
; Purpose     :	Process SWEA Pkt 210 Into Physical Units
;
; Category    :	STEREO, IMPACT, SWEA
;
; Explanation :	
;
; Syntax      :	data = SWEA_210(PACKET)
;
; Examples    :	
;
; Inputs      :	PACKET - an IMPACT SWEA Distribution Telemetry packet (apid=210h)
;
; Opt. Inputs :	None.
;
; Outputs     :	A structure containing SWEA Distribution data
;
; Opt. Outputs:	None.
;
; Keywords    :	None
;
; Calls       :	None
;
; Common      : swea_l1_common
;
; Restrictions:	
;
; Side effects:	None.
;
; Prev. Hist. :	None.
;
; History     :	Version 1.1 (preliminary), 03-Feb-2006, Peter Schroeder, Nicolas Pene
;                  UC-Berkeley & CESR.
;
; Contact     :	peters@ssl.berkeley.edu, nicolas.pene@cesr.fr
;-
;

function swea_210, packet, compressed=compressed, manu_init=manu_init

common swea_l1_common ;use common block
common swea_210_common, swea_dist, starttime ;need common block for distribution
common swea_210_manu_common, manu_ok

IF N_ELEMENTS(manu_ok) EQ 0 THEN manu_ok = 1
IF KEYWORD_SET(manu_init) THEN BEGIN
	; Supprime les effets de bords de swea_210_common
	manu_ok = 0
	RETURN, 0
END

;check checksum
chksum = 0b
hdrbytarray = bytarr(11)
load_bytarr,packet.pkt,hdrbytarray
for i = 0, 10 do chksum += hdrbytarray[i]
for i = 0, 260 do chksum += packet.dat[i]
if chksum ne 0 then return, -1

;compute SWEA energies
if not keyword_set(swea_energies) then swea_l1_energies

;figure out rotation matrix from SWEA to Spacecraft Coordinates
if (packet.grh.spacecraft_id eq 'EA'XU) then SWEATOSC = 40d else $
   SWEATOSC = -140d

SWEATOSC = SWEATOSC/!RADEG

SWEASCrotmat = [ [1, 0, 0], [0, cos(SWEATOSC), sin(SWEATOSC)], $
   [0, -sin(SWEATOSC), cos(SWEATOSC)] ]

;grab SWEA bytes from packet, swap bytes   
rawbytes = packet.dat[1:*]
swap,rawbytes

;calculate packet time
utcpkttim = parse_stereo_pkt(packet,/pkt_date)

;raw byte values from packet
rawdatastr = { $
   SWEADistInt: packet.dat[0], $
   Energy: rawbytes[0], $
   SWEAModeID: rawbytes[1], $
   V0: rawbytes[2], $
   Dist: bytarr(256), $
   Checksum: rawbytes[259] }

for i = 0, 255 do rawdatastr.Dist[i] = rawbytes[3+i]


;---------------------
; Log Compressed data
if keyword_set(compressed) then begin

	case rawdatastr.Energy of
	   0: begin
	      starttime = utcpkttim
	      swea_dist = bytarr(16,80)
manu_ok = 1
	      for e = 0,2 do $
		 for a = 0,79 do $
		    swea_dist[e,a] = rawdatastr.Dist[80*e + a]
	      for a = 0,15 do $
		 swea_dist[3,a] = rawdatastr.Dist[80*3 + a]
	      end
	   3: begin
	      if keyword_set(swea_dist) AND manu_ok then begin
		      for a = 16, 79 do $
			 swea_dist[3,a] = rawdatastr.Dist[80*3 + a - 256]
		      for e = 4,5 do $
			 for a = 0,79 do $
			    swea_dist[e,a] = rawdatastr.Dist[80*e + a - 256]
		      for a = 0,31 do $
			 swea_dist[6,a] = rawdatastr.Dist[80*6 + a - 256]
	      endif
	      end
	   6: begin
	      if keyword_set(swea_dist) AND manu_ok then begin
		      for a = 32, 79 do $
			 swea_dist[6,a] = rawdatastr.Dist[80*6 + a - 512]
		      for e = 7,8 do $
			 for a = 0,79 do $
			    swea_dist[e,a] = rawdatastr.Dist[80*e + a - 512]
		      for a = 0,47 do $
			 swea_dist[9,a] = rawdatastr.Dist[80*9 + a - 512]
	      endif
	      end
	   9: begin
	      if keyword_set(swea_dist) AND manu_ok then begin
		      for a = 48, 79 do $
			 swea_dist[9,a] = rawdatastr.Dist[80*9 + a - 768]
		      for e = 10, 11 do $
			 for a = 0, 79 do $
			    swea_dist[e,a] = rawdatastr.Dist[80*e + a - 768]
		      for a = 0, 63 do $
			     swea_dist[12,a] = rawdatastr.Dist[80*12 + a - 768]
	      endif
	      end
	   12: begin
	      if keyword_set(swea_dist) AND manu_ok then begin
		      for a = 64, 79 do $
			 swea_dist[12,a] = rawdatastr.Dist[80*12 + a - 1024]
		      for e = 13, 15 do $
			 for a = 0, 79 do $
			    swea_dist[e,a] = rawdatastr.Dist[80*e + a - 1024]
		      ret_struct = { $
			 time: starttime, $
			 SWEADistInt: rawdatastr.SWEADistInt, $
			 SWEAModeID: rawdatastr.SWEAModeID, $
			 distribution: swea_dist, $
			 energy: dblarr(16) }
		      for i=0,15 do ret_struct.energy[i] = $
			 total(swea_energies[where(SWEA_DIST_ENERGIES eq i,ecnt)])/ecnt
		      swea_dist = 0
		      return, ret_struct
	      endif
	      end
	   else: begin
	      print, 'Bad energy counter in ApID 210'
	      return, 0
	   end
	endcase

endif else begin
;---------------------
; Uncompressed data

	case rawdatastr.Energy of
	   0: begin
	      starttime = utcpkttim
	      swea_dist = intarr(16,80)
	      for e = 0,2 do $
		 for a = 0,79 do $
		    swea_dist[e,a] = impact_log_uncomp(rawdatastr.Dist[80*e + a])
	      for a = 0,15 do $
		 swea_dist[3,a] = impact_log_uncomp(rawdatastr.Dist[80*3 + a])
	      end
	   3: begin
	      if keyword_set(swea_dist) then begin
		      for a = 16, 79 do $
			 swea_dist[3,a] = impact_log_uncomp(rawdatastr.Dist[80*3 + a - 256])
		      for e = 4,5 do $
			 for a = 0,79 do $
			    swea_dist[e,a] = impact_log_uncomp(rawdatastr.Dist[80*e + a - 256])
		      for a = 0,31 do $
			 swea_dist[6,a] = impact_log_uncomp(rawdatastr.Dist[80*6 + a - 256])
	      endif
	      end
	   6: begin
	      if keyword_set(swea_dist) then begin
		      for a = 32, 79 do $
			 swea_dist[6,a] = impact_log_uncomp(rawdatastr.Dist[80*6 + a - 512])
		      for e = 7,8 do $
			 for a = 0,79 do $
			    swea_dist[e,a] = impact_log_uncomp(rawdatastr.Dist[80*e + a - 512])
		      for a = 0,47 do $
			 swea_dist[9,a] = impact_log_uncomp(rawdatastr.Dist[80*9 + a - 512])
	      endif
	      end
	   9: begin
	      if keyword_set(swea_dist) then begin
		      for a = 48, 79 do $
			 swea_dist[9,a] = impact_log_uncomp(rawdatastr.Dist[80*9 + a - 768])
		      for e = 10, 11 do $
			 for a = 0, 79 do $
			    swea_dist[e,a] = impact_log_uncomp(rawdatastr.Dist[80*e + a - 768])
		      for a = 0, 63 do $
			     swea_dist[12,a] = impact_log_uncomp(rawdatastr.Dist[80*12 + a - 768])
	      endif
	      end
	   12: begin
	      if keyword_set(swea_dist) then begin
		      for a = 64, 79 do $
			 swea_dist[12,a] = impact_log_uncomp(rawdatastr.Dist[80*12 + a - 1024])
		      for e = 13, 15 do $
			 for a = 0, 79 do $
			    swea_dist[e,a] = impact_log_uncomp(rawdatastr.Dist[80*e + a - 1024])
		      ret_struct = { $
			 time: starttime, $
			 SWEADistInt: rawdatastr.SWEADistInt, $
			 SWEAModeID: rawdatastr.SWEAModeID, $
			 distribution: swea_dist, $
			 energy: dblarr(16) }
		      for i=0,15 do ret_struct.energy[i] = $
			 total(swea_energies[where(SWEA_DIST_ENERGIES eq i,ecnt)])/ecnt
		      swea_dist = 0
		      return, ret_struct
	      endif
	      end
	   else: begin
	      print, 'Bad energy counter in ApID 210'
	      return, 0
	   end
	endcase
endelse

end
