;+
; Project     : STEREO - SSC
;
; Name        : STEREO_MAG_XFORM_PACKET
;
; Purpose     : MAG data calibrations.
;
; Category    : STEREO, Telemetry
;
; Explanation : Reads a STEREO PTP packet and perform MAG data calibrations based on the IMAGHKP field
;
; Syntax      : STEREO_MAG_XFORM_270 (PACKET, CAL_TYPE)
;
; Examples    : WHILE 1 DO BEGIN 
;                   READ_STEREO_PKT, UNIT, PACKET
;                   if n_elements(packet) eq 0 then break
;                      apid = parse_stereo_pkt(packet, /APID)
;                   if apid eq '270'XU then begin
;                      packet = mag_xform_packet (packet)
;                   endif
;               endwhile
;
; Inputs      : PACKET - required input which is the Telemetry Beacon Packet
;               CAL_TYPE - optinal input, defaulted to all_cal (all calibrations),
;               other valid values are: 'init_zero', 'scale_factor', 'sensor_orth_correction',
;               'frequency_response' and 'internal_cal'.
;
; Opt. Inputs : CAL_TYPE.
;
; Outputs     : PACKET = A structure containing the packet.  If no packet could
;                        be read, then this will be undefined.
;
; Opt. Outputs: None.
;
; Calls       : READ_STEREO_PKT, PARSE_STEREO_PKT
;
; Common      : None.
;
; Restrictions: The file must first be opened for read access.
;
; Side effects: None.
;
; Prev. Hist. : None.
;
; History     : Version 1, 15-Oct-2005, Alex Lam, IGPP/UCLA 
;               Version 2, 29-Oct-2005, Alex Lam, IGPP/UCLA 
;                          Calibration file will be created from reading an external file
;               Version 3, 23-Dec-2005, Alex Lam, IGPP/UCLA 
;                          Integrate update for Mario Acuna's Preliminary Calibration Report (11/15/05)
;               Version 4, 13-Feb-2006, Alex Lam, IGPP/UCLA 
;                          Add spacecraft time to the return packet
;                          Output MAG data is changed to float data type, since input unit is count,
;                          and output unit is nT (nanoteslas), it makes more sense to change the data type
;               Version 4.1, 28-Mar-2006, William Thompson, GSFC
;                          Only print if !QUIET=0.  Use IMPACT_CALIB_DATA
;                          Use LOAD_STRUCT instead of temporary file
;               Version 5, 7-Apr-2006, Alex Lam, IGPP/UCLA 
;                          Handle both Beacon Telemetry Packet (APID 270h)
;                          and MAG Telemetry Packet (APID 208h, 209h) 
;
; Contact     : Alex Lam - IGPP/UCLA
;-
;
;
; Email from Dr. Mario Acuna (11-Oct-2005)
;The parameters that we normally provide are the following:
;
; 1.      Initial zero tables for all axes and ranges
; 2.      Scale factors for all axes and ranges
; 3a.      Sensor orthogonality correction matrix (3x3)
; 3b.      Sensor rotation correction matrix (3x3)
; 4.      Frequency response information (-3dB roll-off frequency for all axes and ranges)
; 5.      Internal calibration data for all axes and ranges (nominally 1/4 scale step)
; 
; Dave Curtis has also compiled additional time response information
; which verifies some of the parameters above but I suspect that this is not 
; needed at this time.
; We do not have at the present time boom alignment information but we 
; plan to verify this after launch during roll maneuvers. You should plan for
; some ideal initial values based on UCB data. The sensor axes are 
; nominally aligned parallel to the spacecraft axes.
;
; 
;***************************************************************
; This is the entry point of the MAG calibration functions used by Peter Schroeder
function MAG_XFORM_270, packet, cal_type, quiet=quiet
;

apid = parse_stereo_pkt(packet, /APID )
if not keyword_set(quiet) then print, "APID:", APID

offset_A_r0_file = 'offset_A_r0.dat'
offset_A_r1_file = 'offset_A_r1.dat'
offset_B_r0_file = 'offset_B_r0.dat'
offset_B_r1_file = 'offset_B_r1.dat'
scale_factor_A_r0_file = 'scale_A_r0.dat'
scale_factor_A_r1_file = 'scale_A_r1.dat'
scale_factor_B_r0_file = 'scale_B_r0.dat'
scale_factor_B_r1_file = 'scale_B_r1.dat'
sensor_orth_correction_A_r0_file = 'orth_A_r0.dat'
sensor_orth_correction_A_r1_file = 'orth_A_r1.dat'
sensor_orth_correction_B_r0_file = 'orth_B_r0.dat'
sensor_orth_correction_B_r1_file = 'orth_B_r1.dat'
sensor_rotation_correction_file = 'rot.dat'
frequency_response_file = 'freq.dat'
internal_calibration_file = 'intcal.dat'

; assume a value for this now
RANGE_SENSITIVITY = 0

if not keyword_set(quiet) then print,"Testing STEREO_MAG_XFORM"

SCID = packet.grh.SPACECRAFT_ID
if not keyword_set(quiet) then print, "Spacecraft ID:", SCID

;on_error, 2
;if n_params() ne 1 then message, $
;   'Syntax: STEREO_MAG_XFORM, cal_type'
;if n_params() le 0 then cal_type = 'all_cal'
if n_elements(cal_type) eq 0 then cal_type = 'all_cal'

; Define the gains for scale factors
; set to defaulted values
;
KX = 1
KY = 1
KZ = 1
K = [KX, KY, KZ]

; Define an array for zero level calibration
; set to defaulted values
;

OX = 0
OY = 0.1
OZ = -0.1
init_zero_cal = [ OX, OY, OZ ]

; Define an array for orthonoromalize calibration
; set to defaulted values
;
a11 = 1
a12 = 0
a13 = 0
a21 = 0
a22 = 1
a23 = 0
a31 = 0
a32 = 0
a33 = 1
orth_cal = [ [a11, a12, a13], [a21, a22, a23], [a31, a32, a33] ]

; Define an array for internal/rotation calibration
; set to defaulted values
;
b11 = 1
b12 = 0
b13 = 0
b21 = 0
b22 = 1
b23 = 0
b31 = 0
b32 = 0
b33 = 1
int_cal = [ [b11, b12, b13], [b21, b22, a23], [b31, b32, b33] ]

; Original readings (Test MAG data, ADC - Count readings from the packet)
CX = 1000
CY = 2000
CZ = 3000
C = [CX, CY, CZ]

; Ciz (adjusted count after init zero calibration/adjustment has been made)
CXiz = 0
CYiz = 0
CZiz = 0
Ciz = [CXiz, CYiz, CZiz]

; Cic (adjusted count after internal calibration/adjustment has been made)
CXic = 0
CYic = 0
CZic = 0
Cic = [CXic, CYic, CZic]

; Cso (adjusted count after sensor orthogonal calibration/adjustment has been made)
CXso = 0
CYso = 0
CZso = 0
Cso = [CXso, CYso, CZso]

; Csr (adjusted count after sensor rotation calibration/adjustment has been made)
CXsr = 0
CYsr = 0
CZsr = 0
Csr = [CXsr, CYsr, CZsr]

; Cfr (adjusted count after frequency response calibration/adjustment has been made)
CXfr = 0
CYfr = 0
CZfr = 0
Cfr = [CXfr, CYfr, CZfr]


if apid eq '270'XU then begin
	if not keyword_set(quiet) then begin
		print,"using mag_load_struct_270 ..."
		print, "In mag_xform_packet, packet is: ", packet
	endif

	; initialize the return MAG structure
	get_utc, utc_int
	RET_MAG_DATA = {MAG_STRUCT_R, $
		  XX:       BYTARR(11), $
		  IMAGHKP:  0U, $
		  MAGX_01:  0.0,    $
		  MAGY_01:  0.0,    $
		  MAGZ_01:  0.0,    $
		  MAGX_02:  0.0,    $
		  MAGY_02:  0.0,    $
		  MAGZ_02:  0.0,    $
		  MAGX_03:  0.0,    $
		  MAGY_03:  0.0,    $
		  MAGZ_03:  0.0,    $
		  MAGX_04:  0.0,    $
		  MAGY_04:  0.0,    $
		  MAGZ_04:  0.0,    $
		  MAGX_05:  0.0,    $
		  MAGY_05:  0.0,    $
		  MAGZ_05:  0.0,    $
		  MAGX_06:  0.0,    $
		  MAGY_06:  0.0,    $
		  MAGZ_06:  0.0,    $
		  spacecraft_time: utc_int}

	; skip the first 2 elements (XX and IMAGHKP) in the structure
	  skip_count = 2
	; There will be 6 sets of Mag data in the Beacon Telemetry packets
	  mag_sets = 6

	MAG_DATA = mag_load_struct2_270(packet)
end

if not keyword_set(quiet) then begin
   print,"Verify return MAG_DATA ..."
   help,/structure,MAG_DATA
endif

SETS = 0
repeat begin

i=  SETS*3
CX = MAG_DATA.(i+skip_count)
CY = MAG_DATA.(i+1+skip_count)
CZ = MAG_DATA.(i+2+skip_count)
C = [CX, CY, CZ]

; In order to use the calibration function individually,
; we set the original readings to the same before the individual calibration function are called.
Ciz = C
Cic = C
Cso = C
Csr = C
Cfr = C
B = C

case cal_type of
    'init_zero': begin
    if not keyword_set(quiet) then print,"Calling init_zero"
        IF RANGE_SENSITIVITY eq 0 then begin
            IF SCID eq '0xEA'XU then begin
                if not keyword_set(quiet) then begin
                                    print, "Range is low",RANGE_SENSITIVITY
                                    print, "SCID is A"
                                endif
                FILENAME = offset_A_r0_file
            ENDIF ELSE begin
                if not keyword_set(quiet) then begin
                                    print, "Range is low",RANGE_SENSITIVITY
                                    print, "SCID is B"
                                endif
                FILENAME = offset_B_r0_file
            ENDELSE
        ENDIF ELSE begin
            IF SCID eq '0xEA'XU then begin
                if not keyword_set(quiet) then begin
                                    print, "SCID is A"
                                    print, "Range is high",RANGE_SENSITIVITY
                                endif
                FILENAME = offset_A_r1_file
            ENDIF ELSE begin
                if not keyword_set(quiet) then begin
                                    print, "SCID is B"
                                    print, "Range is high",RANGE_SENSITIVITY
                                endif
                FILENAME = offset_B_r1_file
            ENDELSE
        ENDELSE
                cal_table = findgen(3)
                cal_table = mag_load_cal_table(filename, cal_table)
        init_zero_cal = cal_table
                Ciz = mag_init_zero (C,init_zero_cal)
; We set this for putting into the RET_MAG_DATA
        B = Ciz
; Verify
if not keyword_set(quiet) then begin
    print,"Verify..."
    print, B
    print,1111111111
endif
    end

    'scale_factor': begin
    if not keyword_set(quiet) then print,"Calling scale_factor"
        IF RANGE_SENSITIVITY eq 0 then begin
            IF SCID eq '0xEA'XU then begin
                if not keyword_set(quiet) then begin
                                    print, "Range is low",RANGE_SENSITIVITY
                                    print, "SCID is A"
                                endif
                FILENAME = scale_factor_A_r0_file
            ENDIF ELSE begin

                if not keyword_set(quiet) then begin
                                    print, "Range is low",RANGE_SENSITIVITY
                                    print, "SCID is B"
                                endif
                FILENAME = scale_factor_B_r0_file
            ENDELSE
        ENDIF ELSE begin
            IF SCID eq '0xEA'XU then begin
                if not keyword_set(quiet) then begin
                                    print, "SCID is A"
                                    print, "Range is high",RANGE_SENSITIVITY
                                endif
                FILENAME = scale_factor_A_r1_file
            ENDIF ELSE begin
                if not keyword_set(quiet) then begin
                                    print, "SCID is B"
                                    print, "Range is high",RANGE_SENSITIVITY
                                endif
                FILENAME = scale_factor_B_r1_file
            ENDELSE
        ENDELSE
                cal_table = findgen(3)
                cal_table = mag_load_cal_table(filename, cal_table)
        scale_factor_cal = cal_table
                B = mag_scale_factor (Cfr,scale_factor_cal)
; We set this for putting into the RET_MAG_DATA
; Verify
if not keyword_set(quiet) then begin
    print,"Verify..."
    print, B
    print,2222222222
endif
    end

    'sensor_orth_correction': begin
    if not keyword_set(quiet) then print,"Calling sensor_orth_correction"
        IF RANGE_SENSITIVITY eq 0 then begin
            IF SCID eq '0xEA'XU then begin
                if not keyword_set(quiet) then begin
                                    print, "Range is low",RANGE_SENSITIVITY
                                    print, "SCID is A"
                                endif
                FILENAME = sensor_orth_correction_A_r0_file
            ENDIF ELSE begin
                if not keyword_set(quiet) then begin
                                    print, "Range is low",RANGE_SENSITIVITY
                                    print, "SCID is B"
                                endif
                FILENAME = sensor_orth_correction_B_r0_file
            ENDELSE

        ENDIF ELSE begin
            IF SCID eq '0xEA'XU then begin
                if not keyword_set(quiet) then begin
                                    print, "SCID is A"
                                    print, "Range is high",RANGE_SENSITIVITY
                                endif
                FILENAME = sensor_orth_correction_A_r1_file
            ENDIF ELSE begin
                if not keyword_set(quiet) then begin
                                    print, "SCID is B"
                                    print, "Range is high",RANGE_SENSITIVITY
                                endif
                FILENAME = sensor_orth_correction_B_r1_file
            ENDELSE
        ENDELSE
                cal_table = findgen(3, 3)
                cal_table = mag_load_cal_table(filename, cal_table)
        orth_cal = cal_table
                Cso = mag_sensor_orth_correction (Cic,orth_cal)
; We set this for putting into the RET_MAG_DATA
        B = Cso
; Verify
if not keyword_set(quiet) then begin
    print,"Verify..."
    print, B
    print,3333333333
endif
    end

    'sensor_rotation_correction': begin
    if not keyword_set(quiet) then print,"Calling sensor_rotation_correction"
        FILENAME = sensor_orth_correction_file
                cal_table = findgen(3, 3)
                cal_table = mag_load_cal_table(filename, cal_table)
        rot_cal = cal_table
                Csr = mag_rotation_cal (Cso,rot_cal)
; We set this for putting into the RET_MAG_DATA
        B = Csr
; Verify
if not keyword_set(quiet) then begin
    print,"Verify..."
    print, B
    print,4444444444
endif
    end

    'frequency_response': begin
    if not keyword_set(quiet) then print,"frequency_response"
        FILENAME = frequency_response_file
                cal_table = findgen(3, 3)

                cal_table = mag_load_cal_table(filename, cal_table)
        freq_resp_cal = cal_table
                Cfr = mag_frequency_response (Csr,freq_resp_cal)
; We set this for putting into the RET_MAG_DATA
        B = Cfr
; Verify
if not keyword_set(quiet) then begin
    print,"Verify..."
    print, B
    print,5555555555
endif
    end

    'internal_cal': begin
    if not keyword_set(quiet) then print,"Calling internal_cal"
        FILENAME = internal_calibration_file
                cal_table = findgen(3)
                cal_table = mag_load_cal_table(filename, cal_table)
        int_cal = cal_table
                Cic = mag_internal_cal (Ciz,int_cal)
; We set this for putting into the RET_MAG_DATA
        B = Cic
; Verify
if not keyword_set(quiet) then begin
    print,"Verify..."
    print, B
    print,6666666666
endif
    end

; This is the default cal_type, all adjustments have to apply
; The order of applying calibration adjustment is important
    'all_cal': begin
    if not keyword_set(quiet) then begin
            print,"*** Start Calling all_cal ***"
            print,"Calling all_cal"
            print,"In mag_init_zero"
        endif
        IF RANGE_SENSITIVITY eq 0 then begin
            IF SCID eq '0xEA'XU then begin
                if not keyword_set(quiet) then begin
                                    print, "Range is low",RANGE_SENSITIVITY
                                    print, "SCID is A"
                                endif
                FILENAME = offset_A_r0_file
            ENDIF ELSE begin
                if not keyword_set(quiet) then begin
                                    print, "Range is low",RANGE_SENSITIVITY
                                    print, "SCID is B"
                                endif

                FILENAME = offset_B_r0_file
            ENDELSE
        ENDIF ELSE begin
            IF SCID eq '0xEA'XU then begin
                if not keyword_set(quiet) then begin
                                    print, "SCID is A"
                                    print, "Range is high",RANGE_SENSITIVITY
                                endif
                FILENAME = offset_A_r1_file
            ENDIF ELSE begin
                if not keyword_set(quiet) then begin
                                    print, "SCID is B"
                                    print, "Range is high",RANGE_SENSITIVITY
                                endif
                FILENAME = offset_B_r1_file
            ENDELSE
        ENDELSE
        cal_table = findgen(3)
        cal_table = mag_load_cal_table(filename, cal_table)
        init_zero_cal = cal_table
        Ciz = mag_init_zero (C,init_zero_cal)
; Verify
if not keyword_set(quiet) then begin
    print,"Verify..."
	print, "SETS #: ", SETS
    PRINT, "C is: ", C
    print, "Ciz"
    print, Ciz
endif

    if not keyword_set(quiet) then print,"In mag_internal_cal"
        FILENAME = internal_calibration_file
        cal_table = findgen(3)
        cal_table = mag_load_cal_table(filename, cal_table)
        int_cal = cal_table
        Cic = mag_internal_cal (Ciz,int_cal)
; Verify
if not keyword_set(quiet) then begin
    print,"Verify..."
    PRINT, "Ciz"
    PRINT, Ciz
    print, "Cic"
    print, Cic
endif

    if not keyword_set(quiet) then print,"In mag_sensor_orth_correction"
        IF RANGE_SENSITIVITY eq 0 then begin
            IF SCID eq '0xEA'XU then begin
                if not keyword_set(quiet) then begin
                                    print, "Range is low",RANGE_SENSITIVITY

                                    print, "SCID is A"
                                endif
                FILENAME = sensor_orth_correction_A_r0_file
            ENDIF ELSE begin
                if not keyword_set(quiet) then begin
                                    print, "Range is low",RANGE_SENSITIVITY
                                    print, "SCID is B"
                                endif
                FILENAME = sensor_orth_correction_B_r0_file
            ENDELSE
        ENDIF ELSE begin
            IF SCID eq '0xEA'XU then begin
                if not keyword_set(quiet) then begin
                                    print, "SCID is A"
                                    print, "Range is high",RANGE_SENSITIVITY
                                endif
                FILENAME = sensor_orth_correction_A_r1_file
            ENDIF ELSE begin
                if not keyword_set(quiet) then begin
                                    print, "SCID is B"
                                    print, "Range is high",RANGE_SENSITIVITY
                                endif
                FILENAME = sensor_orth_correction_B_r1_file
            ENDELSE
        ENDELSE
        cal_table = findgen(3, 3)
        cal_table = mag_load_cal_table(filename, cal_table)
        orth_cal = cal_table
        Cso = mag_sensor_orth_correction (Cic,orth_cal)
; Verify
if not keyword_set(quiet) then begin
    print,"Verify..."
    PRINT, "Cic"
    PRINT, Cic
    print, "Cso"
    print, Cso
endif

    if not keyword_set(quiet) then print,"In mag_rotation_cal"
        FILENAME = sensor_rotation_correction_file
        cal_table = findgen(3, 3)
        cal_table = mag_load_cal_table(filename, cal_table)
        rot_cal = cal_table
        Csr = mag_rotation_cal (Cso,rot_cal)
; Verify
if not keyword_set(quiet) then begin
    print,"Verify..."
    PRINT, "Cso"
    PRINT, Cso
    print, "Csr"

    print, Csr
endif

    if not keyword_set(quiet) then print,"In mag_frequency_response"
        FILENAME = frequency_response_file
        cal_table = findgen(3, 3)
        cal_table = mag_load_cal_table(filename, cal_table)
        freq_cal = cal_table
        Cfr = mag_frequency_response (Csr,freq_cal)
; Verify
if not keyword_set(quiet) then begin
    print,"Verify..."
    PRINT, "Csr"
    PRINT, Csr
    print, "Cfr"
    print, Cfr
endif

    if not keyword_set(quiet) then print,"In mag_scale_factor"
        IF RANGE_SENSITIVITY eq 0 then begin
            IF SCID eq '0xEA'XU then begin
                if not keyword_set(quiet) then begin
                                    print, "Range is low",RANGE_SENSITIVITY
                                    print, "SCID is A"
                                endif
                FILENAME = scale_factor_A_r0_file
            ENDIF ELSE begin
                if not keyword_set(quiet) then begin
                                    print, "Range is low",RANGE_SENSITIVITY
                                    print, "SCID is B"
                                endif
                FILENAME = scale_factor_B_r0_file
            ENDELSE
        ENDIF ELSE begin
            IF SCID eq '0xEA'XU then begin
                if not keyword_set(quiet) then begin
                                    print, "SCID is A"
                                    print, "Range is high",RANGE_SENSITIVITY
                                endif
                FILENAME = scale_factor_A_r1_file
            ENDIF ELSE begin
                if not keyword_set(quiet) then begin
                                    print, "SCID is B"
                                    print, "Range is high",RANGE_SENSITIVITY
                                endif
                FILENAME = scale_factor_B_r1_file
            ENDELSE
        ENDELSE
        cal_table = findgen(3)
        cal_table = mag_load_cal_table(filename, cal_table)

        scale_cal = cal_table
        B = mag_scale_factor (Cfr,scale_cal)
; Verify
if not keyword_set(quiet) then begin
    print,"Verify..."
    PRINT, "Cfr"
    PRINT, Cfr
    print, "B"
    print, B
endif
    end
endcase
if not keyword_set(quiet) then print,"*** End Calling all_cal ***"

RET_MAG_DATA.(i+skip_count) = B(0)
RET_MAG_DATA.(i+1+skip_count) = B(1)
RET_MAG_DATA.(i+2+skip_count) = B(2)
RET_MAG_DATA.spacecraft_time = parse_stereo_pkt (packet, /PKT_DATE)
;print,"print Return RET_MAG_DATA "
;print,RET_MAG_DATA.spacecraft_time

SETS = SETS + 1
; The packet has 6 sets of data
endrep until SETS gt (mag_sets - 1)

; Verify
if not keyword_set(quiet) then begin
    print,"In mag_xform_packet, Verify..."
    print,"print MAG_DATA before adjustments are made"
    print,MAG_DATA
    help,/structure,MAG_DATA
    print,"print Return RET_MAG_DATA after adjustments are made"
    print,RET_MAG_DATA
    help,/structure,RET_MAG_DATA
endif

return, RET_MAG_DATA
end;
