;+
; Project     :	STEREO - SSC
;
; Name        :	ORB
;
; Purpose     :	Prints out STEREO orbital information
;
; Category    :	STEREO, Orbit
;
; Explanation :	This routine prints out "quick-look" information about the
;               positions of the two STEREO spacecraft, together with similar
;               information about Earth's position.  The information is
;               arranged by viewpoint, with the Behind spacecraft on the left,
;               the Ahead spacecraft on the right, and Earth in between.
;
;               This also serves as a good example of how the various
;               GET_STEREO_* routines work.
;
; Syntax      :	STEREO_COORD_INFO  [, DATE ]
;
; Examples    :	STEREO_COORD_INFO, '2006-05-06T11:30:00'
;
; Inputs      :	None required.
;
; Opt. Inputs :	DATE = The date and time.  This can be input in any format
;                      accepted by ANYTIM2UTC.  The default is the current
;                      date/time.
;
; Outputs     :	Information about the positions and attitude of the two STEREO
;               spacecraft are written to the screen.
;
; Opt. Outputs:	None.
;
; Keywords    : FILENAME = The name of a file to write the output to.  The
;                          default is to write to the terminal.
; Keywords    : SAT = The name of the satellite 'ahead" or 'behind'
;
;               Will also accept any LOAD_STEREO_SPICE or ANYTIM2UTC keywords.
;
; Calls       :	GET_STEREO_LONLAT, GET_STEREO_SEP_ANGLE, GET_STEREO_CARR_ROT,
;               LOAD_STEREO_SPICE
;
; Common      :	None.
;
;
;
;
; Contact     :	Pene@cesr.fr
;-

pro orb, date, sat, filename, _extra=_extra

	on_error, 2

	CATCH, error
	IF error NE 0 THEN BEGIN
		PRINT,' orb error => RETURN'
		IF unit NE -1 THEN FREE_LUN, unit
		RETURN
	END

	;
	;  Make sure that the SPICE kernels are loaded.
	;
	message = ''
	;load_stereo_spice, errmsg=message, _extra=_extra, sat=sat
	load_stereo_spice, errmsg=message, _extra=_extra, /verbose
	if message ne '' then goto, handle_error

	unit = -1
	OPENW, unit, filename, /GET_LUN, /SWAP_IF_LITTLE_ENDIAN

	annee0 = LONG(STRMID(date,0,4))
	mois0  = LONG(STRMID(date,5,2))
	jour0  = LONG(STRMID(date,8,2))

	FOR nbsec=0L,86400,60 DO BEGIN

		; tmp: nombre de secondes depuis 1/1/1958
		; 2436205: valeur de JULDAY(1,1,1958)
		tmp = 86400L*(JULDAY (mois0,jour0,annee0)-2436205) + nbsec

		julian = tmp / 86400L
		CALDAT, julian+2436205, mois, jour, annee

		reste = tmp - julian*86400
		heure  = reste / 3600
		minute = (reste - 3600*heure) / 60
		seconde = reste - 3600*heure - 60*minute

		sannee   = STRING(annee,FORMAT='(I04)')
		smois    = STRING(mois,FORMAT='(I02)')
		sjour    = STRING(jour,FORMAT='(I02)')
		sheure   = STRING(heure,FORMAT='(I02)')
		sminute  = STRING(minute,FORMAT='(I02)')
		sseconde = STRING(seconde,FORMAT='(F06.3)')

		date2 = sannee + '-' + smois + '-' + sjour + 'T' + sheure + ':' + sminute + ':' + sseconde
		;print,"orb.pro: Time => ",date2

		; nbmilli: nombre de milli secondes depuis 1/1/1958
		nbmilli = tmp * 1000d

		;
		;  Gather the information.
		;
		carr = FLOAT(get_stereo_lonlat(date2, sat, /degrees, system='Carrington')) ; en deg
		carr = [carr[1]]
		if carr lt 0 then carr = carr + 360.d0

		heeq = FLOAT(get_stereo_lonlat(date2, sat, /degrees, system='HEEQ')) ; en km
		hee  = FLOAT(get_stereo_lonlat(date2, sat, /degrees, system='HEE')) ; en km
		hci  = FLOAT(get_stereo_lonlat(date2, sat, /degrees, system='HCI')) ; en km
		if hci[1] lt 0 then hci[1] = hci[1] + 360.d0
		semi = FLOAT((6.96d5 * 648d3 / !dpi / 1.496d8) / (hee[0] / 1.496d8)) ; en arcsec (unite astronomique = 1.496d8km)

		; Stereo coord :
		geo  = FLOAT(get_stereo_coord(date2, sat, system='GEO')) ; en km
		gse  = FLOAT(get_stereo_coord(date2, sat, system='GSE')) ; en km
		gsm  = FLOAT(get_stereo_coord(date2, sat, system='GSM')) ; en km
		eroll = FLOAT(get_stereo_roll(date2, sat, system='HAE'))
		sroll = FLOAT(get_stereo_roll(date2, sat, system='HEEQ'))
		angles = FLOAT([get_stereo_sep_angle(date2, sat, 'Earth', /degrees), get_stereo_sep_angle(date2, 'A', 'B', /degrees)])
		car_rot= FLOAT(get_stereo_carr_rot(date2, sat))

		;
		;  Convert km to Rt
		;
		km2rt = 1./6378.
        
		geo = geo*km2rt
		gse = gse*km2rt
		gsm = gsm*km2rt

		hee[0] = hee[0]*km2rt
		heeq[0] = heeq[0]*km2rt
		hci[0] = hci[0]*km2rt

		WRITEU, unit, nbmilli, geo[0:2], gse[0:2], gsm[0:2], geo[3:5], hee[0], semi , hci[1], hci[2], heeq[1], heeq[2], hee[1], hee[2], carr,car_rot, eroll, sroll, angles[0], angles[1]

	END

	FREE_LUN, unit
	RETURN

	;
	;  Error handling point.
	;
	handle_error:
		PRINT, 'AIE AIE AIE'
		IF N_ELEMENTS(message) NE 0 THEN PRINT,message
		EXIT

end
