;+
; Project     :	STEREO - SSC
;
; Name        :	att
;
; Purpose     :	Prints out STEREO orbital information
;
; Category    :	STEREO, Orbit
;
; Explanation :	This routine prints out "quick-look" information about the
;               positions of the two STEREO spacecraft, together with similar
;               information about Earth's position.  The information is
;               arranged by viewpoint, with the Behind spacecraft on the left,
;               the Ahead spacecraft on the right, and Earth in between.
;
;               This also serves as a good example of how the various
;               GET_STEREO_* routines work.
;
; Syntax      :	STEREO_COORD_INFO  [, DATE ]
;
; Examples    :	STEREO_COORD_INFO, '2006-05-06T11:30:00'
;
; Inputs      :	None required.
;
; Opt. Inputs :	DATE = The date and time.  This can be input in any format
;                      accepted by ANYTIM2UTC.  The default is the current
;                      date/time.
;
; Outputs     :	Information about the positions and attitude of the two STEREO
;               spacecraft are written to the screen.
;
; Opt. Outputs:	None.
;
; Keywords    : FILENAME = The name of a file to write the output to.  The
;                          default is to write to the terminal.
; Keywords    : SAT = The name of the satellite 'ahead" or 'behind'
;
;               Will also accept any LOAD_STEREO_SPICE or ANYTIM2UTC keywords.
;
; Calls       :	GET_STEREO_LONLAT, GET_STEREO_SEP_ANGLE, GET_STEREO_CARR_ROT,
;               LOAD_STEREO_SPICE
;
; Common      :	None.
;
;
;
;
; Contact     :	Pene@cesr.fr
;-

pro att, date, sat, filename, _extra=_extra

	on_error, 2

	CATCH, error
	IF error NE 0 THEN BEGIN
		PRINT,' att error => RETURN'
		IF unit NE -1 THEN FREE_LUN, unit
		RETURN
	END

	;
	;  Make sure that the SPICE kernels are loaded.
	;
	message = ''
	;load_stereo_spice, errmsg=message, _extra=_extra, sat=sat
	load_stereo_spice, errmsg=message, _extra=_extra, /verbose
	if message ne '' then goto, handle_error

	unit = -1
	OPENW, unit, filename, /GET_LUN, /SWAP_IF_LITTLE_ENDIAN

	annee0 = LONG(STRMID(date,0,4))
	mois0  = LONG(STRMID(date,5,2))
	jour0  = LONG(STRMID(date,8,2))

	FOR nbsec=0L,86400,60 DO BEGIN

		; tmp: nombre de secondes depuis 1/1/1958
		; 2436205: valeur de JULDAY(1,1,1958)
		tmp = 86400L*(JULDAY (mois0,jour0,annee0)-2436205) + nbsec

		julian = tmp / 86400L
		CALDAT, julian+2436205, mois, jour, annee

		reste = tmp - julian*86400
		heure  = reste / 3600
		minute = (reste - 3600*heure) / 60
		seconde = reste - 3600*heure - 60*minute

		sannee   = STRING(annee,FORMAT='(I04)')
		smois    = STRING(mois,FORMAT='(I02)')
		sjour    = STRING(jour,FORMAT='(I02)')
		sheure   = STRING(heure,FORMAT='(I02)')
		sminute  = STRING(minute,FORMAT='(I02)')
		sseconde = STRING(seconde,FORMAT='(F06.3)')

		date2 = sannee + '-' + smois + '-' + sjour + 'T' + sheure + ':' + sminute + ':' + sseconde
	        ;print,"att.pro: Time => ",date2

	        att = FLOAT(TRANSPOSE(get_stereo_cmat(date2,sat, system="GSE")))

	        PRINTF, unit, FORMAT = '( A," ",A," ",A," ",A," ",A," ",A,9(" ",E) )',$
	                sannee, smois, sjour, sheure, sminute, sseconde, att

	END
	
	FREE_LUN, unit
	RETURN
	;
	;  Error handling point.
	;
	handle_error:
		PRINT, 'AIE AIE AIE'
		IF N_ELEMENTS(message) NE 0 THEN PRINT,message
		EXIT

end
