	pro tel_response_trim,input_file
;
;	program to calculate response of a particle telescope 
;	in terms of MeV deposits and flight times
;
;	Modification history:
;		6-Apr-2009	initial version; based on gasmr by Doug Hamilton  /gm
;		8-Sept-2010	add input file parameter /gm
;		10-Jun-2011	make output file name same as input_file with .txt at end /gm
;		5-Oct-2012	convert to read in TRIM program range-energy tables /gm
;		20-Dec-2012	add suggested value of 21 for boxcar smooth of .kal output /gm
;		2-Jan-2014	change name of variable from run_time to run_time_tel_response to avoid confusion with other programs /gm
;		12-Nov-2014	label tof noise spreads as sigmas;  change SSD noise to sigma (not FWHM) to compare with SIS peak fits /gm
;
;	set up common table
	@/data/joey/masongm1/Programs/idl_general/telescope_response/tel_response_trim_idl_code/tel_response_tables_trim.inc
	common ssd_defect_common, mass_array_defect, fraction_array, energy_array, defect_table_address, lun_table
	

	forward_function speed

	read, table_print, prompt="Type 1 to print out range energy tables and interpolation: "
	

	instring = ''
	title = ''
	file_name = ''
	run_time_tel_response = systime()
	mass_array = fltarr(1)
	e_inc_store = fltarr(1)
	range_file_names = strarr(100)

	input_file=strtrim(input_file,2)   ; remove any leading or trailing blanks from input file string
;	read in telescope description	
	openr, lun_input, input_file, /get_lun
	readf, lun_input, title
	readf, lun_input, angle, sigma_L_ov_L
	readf, lun_input, E_lo, E_hi, E_factor
;
;	read in absorbers
;
	readf, lun_input, n_absorbers, n_range_tables
		for i = 0, n_absorbers-1 do begin
			readf, lun_input, instring
				reads, instring,  absorber_no, thickness, type
				absorber_thickness(i) = thickness
				absorber_type(i) = type - 1
				title_loc = strpos(instring, 'title')
				absorber_title(i) = strmid(instring, title_loc+7, strlen(instring))
		endfor
;
;	check that n_range_tablees is at least as large as the number of range tables read in
	if( (max(absorber_type)+1) gt n_range_tables) then begin
		print, 'input list has maximum absorber type: ', max(absorber_type)+1, ' which is greater than # range tables: ', n_range_tables
		print, ' program stopping;
		stop
	endif

;	
;	read tof segments etc.
;
	readf, lun_input, num_tof
		for i = 0, num_tof - 1 do begin
			readf, lun_input, segment, noise
			num_tof_segments(i) = segment
			tof_noise(i) = noise
				for j = 0, segment-1 do begin
					readf, lun_input, absorber, length
					tof_absorber(i,j) = absorber - 1
					tof_length(i,j) = length
				endfor
		endfor
;
;	read energy deposit segments
	readf, lun_input, num_ssds
		for i = 0, num_ssds - 1 do begin
			readf, lun_input, ssd_in, noise
			ssd_absorber(i) = ssd_in -1
			ssd_noise(i) = noise
		endfor
;
;	define output array for kaleidagraph file:  2000 energies, tof/ssd values, 100 different species
;
	kal_array = fltarr(2000,num_tof+num_ssds, 100)   



;	read in folder containing defect table
	defect_table_address = ''
	readf, lun_input, defect_table_address
	defect_table_address = strtrim(defect_table_address)
;	read in folder containg range tables
	data_folder = ''
	readf, lun_input, data_folder
	data_folder = strtrim(data_folder)


;	now print out header data
	output_folder = strmid(input_file, 0, strpos(input_file,'/', /reverse_search)) + '/'
;	use the input file name to generate the output file name with .txt at the end   10-Jun-2011/gm
	output_file = output_folder + strmid(input_file,strpos(input_file,'/', /reverse_search)+1,strpos(input_file,'.', /reverse_search)-strpos(input_file,'/', /reverse_search)-1) + '.txt'
	output_file_kal = output_folder + strmid(input_file,strpos(input_file,'/', /reverse_search)+1,strpos(input_file,'.', /reverse_search)-strpos(input_file,'/', /reverse_search)-1) + '.kal'	
	openw, lun_output, output_file, /get_lun	
	openw, lun_kal, output_file_kal, /get_lun
	
	printf, lun_output, title, '      run time: ',run_time_tel_response
	printf, lun_kal, title, '      run time: ',run_time_tel_response
	
	printf, lun_output, angle,  sigma_L_ov_L, format="(//,'particle angle of incidence: ',f5.0,' degrees',/,'Sigma-L/L: ',f6.4)"
	printf, lun_kal, angle,  sigma_L_ov_L, format="(//,'particle angle of incidence: ',f5.0,' degrees',/,'Sigma-L/L: ',f6.4)"

	printf, lun_output, e_lo, e_hi, (e_factor-1.)*100., format="('Energy range: ',f6.3,' MeV/n to ',f7.3,' MeV/n by step size of ',f5.2,' percent'/)"
	printf, lun_kal, e_lo, e_hi, (e_factor-1.)*100., format="('Energy range: ',f6.3,' MeV/n to ',f7.3,' MeV/n by step size of ',f5.2,' percent'/)"
	
	for i = 0, n_absorbers-1 do printf, lun_output, i+1, absorber_thickness(i), absorber_type(i)+1, absorber_title(i), $
	    format="('Absorber: ',i2,'     thickness: ',f12.5,' mg/cm2      type: ',i2,'    title: ', a)"
	for i = 0, n_absorbers-1 do printf, lun_kal, i+1, absorber_thickness(i), absorber_type(i)+1, absorber_title(i), $
	    format="('Absorber: ',i2,'     thickness: ',f12.5,' mg/cm2      type: ',i2,'    title: ', a)"

	printf, lun_output, ' '
	printf, lun_kal, ' '

	for i = 0, num_tof-1 do begin
		printf, lun_output, i+1, tof_noise(i), format="('TOF ',i2,' noise spread: ',f10.6,' ns (sigma)')" 
		printf, lun_kal, i+1, tof_noise(i), format="('TOF ',i2,' noise spread: ',f10.6,' ns (sigma)')" 
		for j = 0, num_tof_segments(i)-1 do printf, lun_output, j+1,tof_absorber(i,j)+1, tof_length(i,j), format="('segment ',i2,' follows absorber # ',i2,' path length: ',f5.2,' cm')"
		for j = 0, num_tof_segments(i)-1 do printf, lun_kal, j+1,tof_absorber(i,j)+1, tof_length(i,j), format="(' segment ',i2,' follows absorber # ',i2,' path length: ',f5.2,' cm')"
		printf, lun_output, ' '
		printf, lun_kal, ' '
	endfor
	
	for i = 0, num_ssds-1 do printf, lun_output, i+1, ssd_absorber(i)+1, ssd_noise(i), format="('SSD # ',i2,' is absorber # ',i2,' noise: ',f7.2,' keV (sigma)')"
	for i = 0, num_ssds-1 do printf, lun_kal, i+1, ssd_absorber(i)+1, ssd_noise(i), format="('SSD # ',i2,' is absorber # ',i2,' noise: ',f7.2,' keV (sigma)')"	
	
	printf, lun_output, defect_table_address, format="(/,'SSD defect table used: ',a)"
	printf, lun_kal, defect_table_address, format="(/,'SSD defect table used: ',a)"

	printf, lun_output,  n_range_tables, data_folder, format="(/,i3,' range tables read from folder: ',a)"
	printf, lun_kal,  n_range_tables, data_folder, format="(/,i3, ' range tables read from folder: ',a,//)"

;
;	now process a single element
;

	while( 1 > 0 ) do begin
	readf, lun_input, instring
	if( strmid(instring,0,3) eq 'end' ) then goto, finish
	
;	find first blank in input string

	loc = strpos(instring,' ')
	element = strmid(instring,0,loc)

;	now read the atomic_mass
	reads, strmid(instring,loc,strlen(instring)), atomic_mass

;	store the mass being processed
	if( mass_array(0) eq 0. ) then mass_array = atomic_mass  else mass_array = [mass_array, atomic_mass]
	
;	special case for Deuterium -- treated like 1H except Emeas is doubled (see ACE notes 9/29/98)
;	************ commented out for TRIM input tables ***************
	deuterium_flag = 0
;	if( atomic_mass eq 2.014 ) then deuterium_flag = 1 else deuterium_flag = 0
;	if(  strpos(instring, 'euterium') gt 0 ) then deuterium_flag = 1
;	if( deuterium_flag ) then atomic_mass = 1.008    ; reset atomic mass to 1H for deuterium (but list correcltly)
	
;	write out header for this particle type
	printf, lun_output, title, '      run time: ', run_time_tel_response
	printf, lun_output, ' '
	printf, lun_output, element, atomic_mass, format="('incident particle: ',a,'    atomic mass: ',f6.2,' amu'/)"

;	now read in the data tables for this element -- take logs of input ranges & energies
;	note, program can read in tables it doesn't use -- makes handling of input file easier
	for i = 0, n_range_tables-1 do begin
		readf, lun_input, instring
;	strip off the file name from the first part of the line--
		file_name = strmid(instring,0,strpos(instring,' '))
	range_file_names(i) = file_name
;	now read the atomic mass used to generate this range table from the rest of the string
		reads, strmid(instring, strpos(instring,' '),strlen(instring)-strpos(instring,' ')), atwt_range_table 
;		print, atwt_range_table
;		range_table = data_folder + file_name + '.dat' 
		range_table = data_folder + file_name
		openr, lun_table, range_table, /get_lun
		printf, lun_output, i+1, file_name,  format="('Absorber type: ',i3,8x,' Range-energy data from file: ',a)"
;	TRIM file reading -- 
;	skip down to the first data line --
	units_e1 = ' '
	units_r1 = ' '
	while( strpos(instring, '-----------' ) eq -1 ) do begin
		readf, lun_table, instring
			if( strpos(instring, 'Target Density') ne -1 ) then begin 
			reads, instring, density, format="(t19, e12.4)"
			printf, lun_output, density, format="('Density = ',f10.4,' g/cm3')"
		endif
	endwhile
	
	for table_entry = 0L, 1000L do begin
		readf, lun_table, instring
		if( strpos(instring, '-----------' ) ne -1 ) then break
		reads, instring, e1, units_e1, r1, units_r1, format="(f7.0,x,a4,t35,f8.0,a3)"
;		print, table_entry, e1, units_e1, r1, units_r1
;  10.00 keV   4.245E-01  7.339E-01     233 A       129 A        94 A
;	now convert energy to MeV
	case units_e1 of
		'keV ':	e1/= 1000.d0
		'MeV ':
		'GeV ':	e1 *= 1000.d0
	endcase
	energies(table_entry,i) = alog(e1)    ; convert to log of total energy
					
;	now convert ranges to mg/gm2
	case units_r1 of
		' A ':		r1 *= 1.0d-8	
		' um': 	r1 *= 1.0d-4
		' mm':	r1 *= 1.0d-1
		' cm':
		' m ':		r1 *= 1.0d2
	endcase

		r1 *= 1000.d0*density ; convert from distance to mg/cm2
		ranges(table_entry,i) = alog(r1*atomic_mass/atwt_range_table)

	endfor 	; end table reading
	
		num_energies(i) = table_entry -1
	
		free_lun, lun_table

	endfor	; loop for multiple tables

;	print out tables if flag set earlier
	if( table_print eq 1 ) then print_tables_trim, output_folder, n_range_tables, atomic_mass, range_file_names


;		stop
	
	printf, lun_output, ' '
	printf, lun_output, format="($,'Einc (MeV/n), Mass (AMU), ')"
	for i = 1, num_tof do printf, lun_output, i, format="($, ' TOF-',i1,',')"
	for i = 1, num_ssds do printf, lun_output, i, format="($, ' Emeas -',i1,',')"
	printf, lun_output, ' Sigma M (AMU) (TOF-1 vs SSD-1), Sigm/M, SigE/E, 2 Sigt/T, 2 SigL/L, '


;	now step through the energy range

	e_inc = e_lo

	einc_number = 0UL
	
	while( e_inc le e_hi) do begin

;	calculate the TOFs first

		tof = fltarr(10)    ; clear the tof array
		for i = 0, num_tof-1 do begin
			for j = 0, num_tof_segments(i)-1 do begin
				loss_trim, e_inc, atomic_mass, angle, 0, tof_absorber(i,j), e_left
;				print,e_inc, atomic_mass, angle, 0, tof_absorber(i,j), e_left
				if( e_left gt 0 ) then tof(i) +=  tof_length(i,j)/(speed(e_left)*cos(angle/!radeg))
;				if e_left = 0, then one of the tof segments was not traversed, so don't save this tof value since not triggered
				if( e_left eq  0.) then tof(i) = 0.
			endfor
		endfor
	
;	now calculate the ssd deposits

		ssd = fltarr(10)
		for i = 0, num_ssds-1 do begin
			loss_trim, e_inc, atomic_mass, angle, 0, ssd_absorber(i)-1, e_ssd_in
;			print,e_inc, atomic_mass, angle, 0, ssd_absorber(i)-1, e_ssd_in
			loss_trim, e_ssd_in, atomic_mass, angle, 0, ssd_absorber(i), e_ssd_out
;			print,e_ssd_in, atomic_mass, angle, 0, ssd_absorber(i), e_ssd_out
			ssd(i) = (e_ssd_in-e_ssd_out)*atomic_mass
; stop
		endfor
;		print, ssd(0:1)


;	now fold in the ssd_defect from table
		for i = 0, num_ssds-1 do begin
			ssd_defect, ssd(i)/atomic_mass,atomic_mass,fraction
;			print, fraction
			ssd(i) *= fraction
			if( deuterium_flag ) then ssd(i) *= (2.014/1.008)
;			print, atomic_mass, fraction, i, ssd(i)
		endfor

;	write out the results
;stop
	printf, lun_output, e_inc, atomic_mass, format="($,x, 2(f9.4,','))"
	for i = 0, num_tof-1 do printf, lun_output, tof(i), format="($,x, e11.4,',')"
	for i = 0, num_ssds-1  do printf, lun_output, ssd(i), format="($,x, e11.4,',')"


;	calculate the sigmas for tof-1 vs SSD-1
	mass_sigma_array = fltarr(5)
		if( ssd(0) gt 0. ) then if( (ssd_noise(0)/1000.)/ssd(0) lt 2. ) then begin
			mass_sigma_array(4) = 2*sigma_L_ov_L
			mass_sigma_array(3) = 2*tof_noise(0)/tof(0)
			mass_sigma_array(2) = (ssd_noise(0)/(1000.))/ssd(0)    ; convert ssd_noise sigma to MeV before calculating!
			mass_sigma_array(1) = sqrt(mass_sigma_array(2)^2 + mass_sigma_array(3)^2 + mass_sigma_array(4)^2)
			mass_sigma_array(0) = atomic_mass*mass_sigma_array(1)
		endif
	
	printf, lun_output, mass_sigma_array, format="($,x, 5(e11.4,','))"
	printf, lun_output, ' '

	
;	store information for the kaleidagraph file
	if( e_inc eq E_lo ) then e_inc_store(0) = E_lo else e_inc_store = [e_inc_store,e_inc]
	
	for i = 0, num_tof-1 do kal_array(einc_number, i, n_elements(mass_array)-1 ) = tof(i)
	for i = 0, num_ssds-1 do kal_array(einc_number, i + num_tof , n_elements(mass_array)-1 ) = ssd(i)
	
	
;	bump energy
	e_inc *= e_factor
	einc_number++
	
	endwhile       ;  end of while loop for single element


	endwhile             ; end of while loop for element list

finish:	
;	truncate unused parts of the arrays 
	e_inc_store = e_inc_store(0:einc_number-1)
	kal_array_temp = fltarr(einc_number, num_tof+num_ssds, n_elements(mass_array))   
	for i = 0, einc_number-1 do for j = 0, (num_tof+num_ssds -1) do for k = 0, (n_elements(mass_array)-1) do kal_array_temp(i,j,k)=kal_array(i,j,k)
;	reset kal_array
	kal_array = 0
;	now set equal to kal_array_temp
	kal_array = kal_array_temp


;
;	write out the kaleidagraph file
;
;	first the header lines
	printf, lun_kal, n_elements(mass_array), format="(i3,' Masses:')
	for i = 0, n_elements(mass_array)-1 do printf, lun_kal, mass_array(i), format="($,f6.2,',')"
	printf, lun_kal, ' ' 
	
	printf, lun_kal, format="($,'Einc (MeV/n), ')"
	for i = 0, n_elements( mass_array )-1 do begin
		for j = 1, num_tof do printf, lun_kal, j, mass_array(i), format="($, ' TOF-',i1,' mass ',f5.1,',')"
		for j = 1, num_ssds do printf, lun_kal, j, mass_array(i), format="($, ' Emeas -',i1,' mass ',f5.1,',')"
	endfor
	printf, lun_kal, ' '
	print,' writing out kaleidagraph file: '
;	now write out the kaleidagraph data
	for i = 0, einc_number-1 do begin
		printf, lun_kal, e_inc_store(i), format="($, x, e11.4,',')"
			for j = 0, n_elements( mass_array ) -1 do begin
				for k = 0, num_tof-1 do begin
					format_string = "($," + string(num_tof, format="(i)") + "(e11.4,','))"
					printf, lun_kal, kal_array(i, k, j), format = format_string
				endfor
				for k = num_tof, num_tof+num_ssds-1 do begin
					format_string = "($," + string(num_ssds, format="(i)") + "(e11.4,','))"
					printf, lun_kal, kal_array(i, k, j), format = format_string
				endfor
			endfor
	printf, lun_kal, ' '
	endfor

;	generate a lookup table based on measured energies
	num_mass = n_elements(mass_array)	

;
;	first, fill up emeas from 10 keV to 2000 MeV by 3% steps
;
	emeas_number = floor( alog(2000./.01)/alog(1.03) + 1 )
	emeas = fltarr(emeas_number)
	emeas[0] = 0.01
	for i = 1, emeas_number-1  do emeas(i) = emeas(i-1)*1.03
	emeas_array = fltarr(emeas_number, num_tof+num_ssds, num_mass)


;	find the maximum einc numbers before 1st ssd is penetrated for each mass
	max_einc_number = fltarr(num_mass)
	for m_scan = 0, num_mass-1 do begin
		for e_scan = 0, einc_number-2 do if (kal_array(e_scan,num_tof,m_scan) gt kal_array(e_scan+1,num_tof,m_scan) ) then break
		max_einc_number(m_scan) = e_scan
	endfor


;
;	now, for each mass in the table, find TOFs at each 
;	value of the Emeas array   (this code just copied from function find_tof)
;
;	note:  ssd_num is the ssd # being used for looking up this value of e_measure
	ssd_num = 0    ; just doing the first ssd
	
	for k = 0, num_mass-1 do for j = 0, num_tof-1 do for i = 0, emeas_number-1 do begin 

;		for this mass value, find the tof values on either side of emeas
		for i_scan = 0, (max_einc_number(k)-3) do if( (emeas(i) ge kal_array(i_scan, num_tof + ssd_num, k)) and (emeas(i) lt kal_array(i_scan+1, num_tof + ssd_num, k)) ) then break
	
;		print, ' i,j,k,i_scan: ',i,j,k,i_scan, kal_array(i_scan, num_tof + ssd_num, k), kal_array(i_scan+1, num_tof + ssd_num, k)
	
		interpolate_ln, value, emeas(i), kal_array(i_scan, num_tof + ssd_num, k), kal_array(i_scan+1, num_tof + ssd_num, k), kal_array(i_scan, j, k), kal_array(i_scan+1, j, k)
;		print,value, emeas, kal_array(i_scan, num_tof + ssd_num, k), kal_array(i_scan+1, num_tof + ssd_num, k), kal_array(i_scan, j, k), kal_array(i_scan+1, j, k)	

		emeas_array(i, j, k) = value

	endfor   ; k,j,i loop

	tof1_smooth = fltarr(emeas_number)
	if( num_tof gt 1 ) then tof2_smooth = fltarr(emeas_number)
	if( num_tof gt 2 ) then tof3_smooth = fltarr(emeas_number)

	read, smooth_width, prompt="Enter the boxcar smoothing window width (odd number, suggested value: 21): "
	for j = 0, num_mass-1 do begin
		for i = 0, emeas_number-1 do begin
			tof1_smooth(i) = emeas_array(i,0,j)
			if( num_tof gt 1 ) then tof2_smooth(i) = emeas_array(i,1,j)
			if( num_tof gt 2 ) then tof3_smooth(i) = emeas_array(i,2,j)
		endfor
	tof1_smooth = smooth(tof1_smooth, smooth_width, /edge_truncate, /nan)
	if( num_tof gt 1 ) then tof2_smooth = smooth(tof2_smooth, smooth_width, /edge_truncate, /nan)
	if( num_tof gt 2 ) then tof3_smooth = smooth(tof3_smooth, smooth_width, /edge_truncate, /nan)
;	now put the smoothed values back into emeas_arry
		for i = 0, emeas_number-1 do begin
			emeas_array(i,0,j) = tof1_smooth(i)
			if( num_tof gt 1 ) then emeas_array(i,1,j) = tof2_smooth(i)
			if( num_tof gt 2 ) then emeas_array(i,2,j) = tof3_smooth(i)
		endfor
	endfor

;
;	write out kaleidagraph file
;
	output_kal_file = output_folder + strmid(input_file,strpos(input_file,'/', /reverse_search)+1,strpos(input_file,'.', /reverse_search)-strpos(input_file,'/', /reverse_search)-1) + '_emeas.txt'	
	openw, lun_kal_out, output_kal_file, /get_lun
	printf, lun_kal_out, smooth_width, format="('TOF values smoothed over window of width: ',i3)"
	printf, lun_kal_out, format="($,' Mass array:  ')"
	for i = 0, num_mass-1 do printf, lun_kal_out, mass_array(i), format="($, f9.4,',')"
	printf, lun_kal_out, ' '
	printf, lun_kal_out, format="($,' Edep,')"
	for j = 0, num_tof-1 do begin
		for i = 0, num_mass-1 do printf, lun_kal_out, j+1, mass_array(i), format="($,' Tof ',i1,' mass',f5.1,',')"
	endfor
	printf, lun_kal_out, ' '
	for i = 0, emeas_number-1 do begin
		printf, lun_kal_out, Emeas(i), format="($, e14.5,',')"
		for j = 0, num_tof-1 do begin
			for k = 0, num_mass-1 do printf, lun_kal_out, emeas_array(i,j,k), format="($,e14.5,',')"
		endfor
		printf, lun_kal_out, ' '
	endfor
	close, lun_kal_out

	tof_first = fltarr(num_tof,num_mass)
	emeas_first = fltarr(num_ssds,num_mass)
;
;	find the first non-zero tof values in the tof & Emeas arrays
;
	for k = 0, num_mass-1 do begin
		for j =  0, num_tof-1 do begin
			for i = 0, einc_number-1 do begin
				if( kal_array(i,j,k) gt 0 ) then begin
					tof_first(j,k) = i
					break
				endif
			endfor
		endfor
	endfor

	for k = 0, num_mass-1 do begin
		for j =  0, num_ssds-1 do begin
			for i = 0, einc_number-1 do begin
				if( kal_array(i,num_tof+j,k) gt 0 ) then begin
					emeas_first(j,k) = i
					break
				endif
			endfor
		endfor
	endfor


;	now save the tables etc. for use by other programs 
;	rename some of the arrays
	einc = e_inc_store
	einc_array = kal_array


;	
;	**************** variables to save file ****************
;	title:   		run title array
;	run_time_tel_response:  	run time of the tel_response program that made the .sav file
;	num_tof:  		number of TOF measurements 
;	num_ssds:		number of SSD energy measurements
;	num_mass		number of element masses calculated in this run
;	mass_array:	array of masses calculated for the run
;	einc_number:	number of energy values of incident energies
;	max_einc_number: array of maximum einc number for each element (ie max value before it penetrates the 1st ssd)
;	einc:			array with incident energy values for each used line of einc_array
;	einc_array:	array with indices (energy step, num_tof + num_ssds, masses) -- energy steps are incident energy
;	emeas_number:	number of measured energies in the ssd for the array keyed to measured energy
;	emeas:		array with measured energy values
;	emeas_array	array with indices (energy step, num_tof + num_ssds, masses) -- energy steps are measured energy
;	tof_first:		array with indices (num_tof, num_mass) of first non-zero tof value for each tof & mass
;	emeas_first:		array with indices (num_tof, num_mass) of first non-zero e_meas value for each tof & mass
	
	output_file_save = output_folder + strmid(input_file,strpos(input_file,'/', /reverse_search)+1,strpos(input_file,'.', /reverse_search)-strpos(input_file,'/', /reverse_search)-1) + '.sav'	

	save, filename=output_file_save, title, run_time_tel_response, num_tof, num_ssds, num_mass, mass_array, einc_number, max_einc_number, einc, einc_array, $
			emeas_number, emeas, emeas_array, tof_first, emeas_first
	print, 'saving file:  ',output_file_save


	close, /all

	end

	function speed, e_left

;	returns cm/ns for input energy/nucleon

	beta = sqrt(1. - (931.5/(931.5+e_left))^2)
	return, beta*29.9792458d0  ; speed in cm/ns
	end

