PRO SIT_RATES, PACKET, SITRATEDATA
;+
; Project     :	STEREO - IMPACT, SIT
;
; Name        : SIT_RATES
;
; Purpose     :	Example program accessing SIT rates data
;
; Category    :	STEREO, IMPACT, SIT
;
; Explanation :	
;
; Syntax      :	SITRATES, PACKET, SITRATEDATA
;
; Examples    :	
;
; Inputs      :	PACKET - a SIT rates telemetry packet (apid=605)
;
; Opt. Inputs :	None.
;
; Outputs     :	SITRATEDATA = A structure containing SIT rates data
;               if data is valid, undefined otherwise
;
; Opt. Outputs:	None.
;
; Keywords    :	None
;
; Calls       :	delvarx
;
; Common      : sitcom
;
; Restrictions:	
;
; Side effects:	None.
;
; Prev. Hist. :	None.
;
; History     :	Version 1 (example), 28-Jun-2006, Andrew Davis, Caltech.
;-
;

; see sit_common.pro
common sitcom

good_data = 1

; structure to hold Raw SIT data from rates packet
rawrates = { $
	DRATES:		UINTARR(8), $    ; 16-bit compressed discriminator rates
	MRATES:		UINTARR(116), $    ; 16-bit compressed matrix rates
	hvstep:		0B, $
	flags:		0B, $
	LIMHI:		0U, $
;	LUTchksum:	BYTARR(3), $       (lookup table checksum not in APID 605 -- 9/6/06/gm -- 3 bytes added to unused item below)
	unused:		BYTARR(8), $
	chksum:		0B $
}       

; decompressed rate data
DRATES =		LONARR(8)
MRATES =		LONARR(116)
hvstep =		0B
flags =			0B
LIMHI =			0U
;LUTchksum =		BYTARR(3)

; a structure to hold time in Julian day, millisec of day format
timeCDS = 		{ MJD: 0L , TIME: 0L }

; header for unpacked SIT rates data
header = { $
	SWversion: 0B, $	; Beacon processing software version (this software)
	timeCDS: timeCDS, $	; Time at start of data collection interval: Julian Day, millisec_of_day
	SCid: 0U $		; Spacecraft ID
}


      ; Note: packet checksum could be verified here

;*********************************************************
;print,' rawrates[0] status: '
;help,rawrates[0]
;help,rawrates[0],/structure
;print,' packet.dat status: '
;help,packet.dat,/structure
;print, ' ntags, n_elements(tag_names(rawrates[0])): ',  n_elements(tag_names(rawrates[0]))
;print, ' nrep, n_elements(packet.dat(0,*)): ', n_elements(packet.dat(0,*))
;print,' rawrates status: '
;help,rawrates
;help,rawrates,/structure
;*********************************************************

      ; load the raw data bytes into raw SIT rates data structure
      ; load_struct is a Solarsoft routine
	load_struct, packet.dat, rawrates[0], rawrates, /noieee, /noconv
	
;***********
;print,' rawrates status after call to load_struct: '
;help,rawrates
;help,rawrates,/structure
;*************
      ; swap bytes in 2-byte SIT items if we are running on a big-endian machine (Solaris or MacOS)
	rawrates.DRATES = swap_endian(rawrates.DRATES, /SWAP_IF_BIG_ENDIAN)
	rawrates.MRATES = swap_endian(rawrates.MRATES, /SWAP_IF_BIG_ENDIAN)
	rawrates.LIMHI = swap_endian(rawrates.LIMHI, /SWAP_IF_BIG_ENDIAN)

      ; decompress rates	     
	DRATES = decompressHetSitCounters(rawrates.DRATES)
        MRATES = decompressHetSITCounters(rawrates.MRATES)

      ; just copy the rest of the stuff...
	hvstep = rawrates.hvstep
	flags = rawrates.flags
	LIMHI = rawrates.LIMHI
;	LUTchksum = rawrates.LUTchksum
      
      ; build rates data structure header
        header.SWversion = VERSION_sit

      ; put data-collection time into header
      ; Use the /NOCORRECT option here, since STEREO spacecraft time already has
      ; leap seconds factored in. i.e. spacecraft time is UTC seconds since 1958

        header.timeCDS = tai2utc(double(packet.pkt.seconds), /NOCORRECT)

      ; Set spacecraft id in header from packet ground receipt header
	header.SCid = packet.grh.SPACECRAFT_ID

         
      ; Assemble unpacked SIT rates data structure
	sitratedata = { $
		header:header, $
		DRATES:DRATES, $
		MRATES:MRATES, $
		hvstep:hvstep, $
		flags:flags, $
		LIMHI:LIMHI $

;		flags:hvstep, $  /gm   9/6/06
;		LIMHI:hvstep $   /gm   9/6/06  (see above)
;		LUTchksum:LUTchksum $
	}


	IF (good_data eq 0) THEN DELVARX, sitratedata

END
