PRO SIT_PHA, PACKET, SITPHADATA, num_pha, apid
;+
; Project     :	STEREO - IMPACT, SIT
;
; Name        : SIT_PHA
;
; Purpose     :	Example program accessing SIT pha data
;
; Category    :	STEREO, IMPACT, SIT
;
; Explanation :	
;
; Syntax      :	SIT_PHA, PACKET, SITPHADATA
;
; Examples    :	
;
; Inputs      :	PACKET - a SIT pha telemetry packet (apids 606-616)
;
; Opt. Inputs :	None.
;
; Outputs     :	SITPHADATA = A structure containing SIT rates data
;               if data is valid, undefined otherwise
;
; Opt. Outputs:	None.
;
; Keywords    :	None
;
; Calls       :	delvarx
;
; Common      : sitcom
;
; Restrictions:	
;
; Side effects:	None.
;
; Prev. Hist. :	None.
;
; History     :	Version 1 (example), 2-Oct-2006 / gm / based on sitrates routine by Andrew Davis
;			5/17/2011	return packet structure for apid 606 even if no events (for livetime calc)
;-
;

; see sit_common.pro
common sitcom

good_data = 1


; structure to hold raw SIT data from PHA packet

rawpha = { $
	pha_event:	ulonarr(64), $
	unused:	bytarr(3), $
	num_pha:		0B, $           ; number of pha events in this packet
	chksum:		0B $
}	

;	pha data
	pha_event = ulonarr(64)


; a structure to hold time in Julian day, millisec of day format
timeCDS = 		{ MJD: 0L , TIME: 0L }

; header for unpacked SIT pha data
header = { $
	SWversion: 0B, $	; Beacon processing software version (this software)
	timeCDS: timeCDS, $	; Time at start of data collection interval: Julian Day, millisec_of_day
	SCid: 0U $		; Spacecraft ID
}


; Note: packet checksum could be verified here


; load the raw data bytes into raw SIT pha data structure
; load_struct is a Solarsoft routine


	load_struct, packet.dat, rawpha[0], rawpha, /noieee, /noconv
	
;	if there are no pha events in this packet, return unless this is apid 606, then need times to calculate live time ******************** 5/17/2011
	num_pha = rawpha.num_pha
	if((apid gt 606) and (num_pha eq 0) ) then return            

; swap bytes in 4-byte SIT pha word if we are running on a big-endian machine (Solaris or MacOS)
	rawpha.pha_event = swap_endian(rawpha.pha_event, /SWAP_IF_BIG_ENDIAN)


; just copy the rest of the stuff...
	pha_event = rawpha.pha_event
	chksum = rawpha.chksum

; build pha data structure header
header.SWversion = VERSION_sit   

; put data-collection time into header
; Use the /NOCORRECT option here, since STEREO spacecraft time already has
; leap seconds factored in. i.e. spacecraft time is UTC seconds since 1958
header.timeCDS = tai2utc(double(packet.pkt.seconds), /NOCORRECT)

; Set spacecraft id in header from packet ground receipt header
	header.SCid = packet.grh.SPACECRAFT_ID

;	extract the pha items from the pha_event word

	EXTRACT_pha, rawpha.pha_event, $
	   tof_ch, energy_ch, gain, priority, matrix_box, tof_flg_1, tof_flg_2, tof_err_proc


; Assemble SIT pha data structure
	sitphadata = { $
		header:   header, $
		tof_ch: tof_ch, $
		energy_ch: energy_ch, $
		gain: gain , $
		priority: priority, $
		matrix_box: matrix_box, $
		tof_flg_1: tof_flg_1, $
		tof_flg_2: tof_flg_2, $
		tof_err_proc: tof_err_proc, $
		num_pha:    num_pha,  $
		chksum:    chksum $
	}
;	if(sitphadata.num_pha gt 0) then print, sitphadata.num_pha, num_pha, nsav
;	num_pha = sitphadata.num_pha
;	if(total(rawpha.pha_event) ne 0) then print, ' sitphadata.num_pha: ',  sitphadata.num_pha
;	if(total(rawpha.pha_event) ne 0) then print, rawpha.pha_event
;	if(total(rawpha.pha_event) ne 0) then for i = 0,63  do if (matrix_box(i) gt 0) then	print, i, sitphadata.tof_ch(i), $
;	   sitphadata.energy_ch(i), sitphadata.tof_flg_1(i), sitphadata.tof_flg_2(i), $
;	   sitphadata.gain(i), sitphadata.tof_err_proc(i), sitphadata.matrix_box(i), priority(i)

	IF (good_data eq 0) THEN DELVARX, sitphadata


END
