PRO SIT_LISTER, inpathname
;+
; Project     :	STEREO - IMPACT, SIT
;
; Name        : SIT_LISTER
;
; Purpose     :	Program to read SIT packets sequentially and print output fi les
;
; Category    :	STEREO, IMPACT, SIT
;
; Explanation :	reads a STEREO telemetry data file and extracts data from SIT packets.
;

; History     :	Version 1 (example), 28-Jun-2006, Andrew Davis, Caltech
;			Modified from program sit_rates.pro by Andrew Davis / gm
;	Modification history:
;		26-Jun-2008	convert sit_common and sep_beacon_common to include files
;					move initialization statements from sit_common to new routine
;					read_calibration  /gm
;		May-2011		add pha rates /gm
;		10-Jan-2013	read in ephemeris before opening output files /gm
;		11-Mar-2013	change PHA time base from apid 606 to apid 605 packets /gm
;		12-Mar-2013	remove plot_pha.pro and calls /gm    -- variable removed from cfg file 3/20/2013 /gm
;		13-Jul-2015	add provision for handling missing days /gm


; ****************************************************************

	@sit_common.inc
	@sep_beacon_common.inc

	read_paths
	if (n_elements(inpathname) eq 0) then begin
		inpathname=''
		batch_out = 0
	endif else begin	
;		clean up inputpathname if fed as an input parameter  12/18/2015
		inpathname=strtrim(inpathname,2)   ; remove any leading or trailing blanks		
;		if the last character of inpathname is not a '/', then add one
		if( strmid(inpathname,strlen(inpathname)-1,1) ne '/' ) then inpathname += '/'
		batch_out = 1
;		print, 'batch job from folder : ', inpathname
	endelse		
		

;	initialize control flags --  	0 = open file and set up headers
;						1 = running value
;						2 = close file

flag_rates=0L
flag_spectra=0L
flag_hk=0L
flag_beacon=0L
flag_open_sit_file=0L
files_missed = 0L
histogram_num = 1
wrote_daily_outputs = 0
	
;	***************************************************************
;	read in the cfg file from standard directory if no foldername passed to program; otherwise from current directory
;
;	************ note:  for any single run, the cdf and times files are read from the same folder ! **********

	if (inpathname eq '' ) then control_file_path='/data/joey/masongm1/Data/work_files_STEREO/sit_lister_control/' $
		else control_file_path=inpathname
	input_file= control_file_path + 'sit_lister_cfg.dat'
;	print, ' opening cfg file: ', input_file
	openr, lun_cfg, input_file, /get_lun
	readf, lun_cfg, format='(/)'    ;  skip header line
	output_file_path=''
	readf, lun_cfg, output_file_path
	readf, lun_cfg, format='(/)'    ;  skip header line	
	readf, lun_cfg,  spacecraft_id,  avg_int,  type_out, run_checks, error_limit, bkgd_correction_on  
	readf, lun_cfg, format='(/)'    ;  skip header line
	readf, lun_cfg,  rates_out,  pha_out,  beacon_out,  hk_out, spectra_out, pha_spectra_out, pha_rates_out, histogram_out, multi_panel_out
	if( pha_out or histogram_out ) then begin 
		readf, lun_cfg, format='(/)'    ;  skip header line
		readf, lun_cfg,  pha_mlo, pha_mhi, pha_elo,  pha_ehi, bin_width
	endif

;	********************* get the list of missing days ********************
;	print, 'reading missing days list'
	get_sit_missing_days, spacecraft_id,  n_missing, mjd_missing

;
;	********************* read calibration after output types known (to see if large files can be skipped ***********	
;	print, 'reading calibration tables'
	read_calibration


;	******** pha plots setup removed from here 3/12/2013 since it didn't work when running under terminal /gm


	if( rates_out or spectra_out or pha_spectra_out or multi_panel_out or pha_rates_out) then call_process_rates = 1 else call_process_rates = 0
	if( run_checks or pha_out or multi_panel_out or (histogram_out gt 0) or (pha_spectra_out gt 0) or (pha_rates_out gt 0) ) then call_process_pha = 1 else $
		call_process_pha = 0 
	if( run_checks or hk_out ) then call_process_hk = 1 else call_process_hk = 0   ; 2/13/08
	
; ***************************************************************************
;	do an initial call to ephemeris subroutine if we will need emphemeris data later -- call with placeholder date (1/1/2013); if problem here then will quit before opening output files

	if( rates_out or  pha_out or beacon_out or pha_rates_out ) then stereo_ephemeris, 56293

		
		avg_int_hk=avg_int
		avg_int_rates=avg_int
		avg_int_pha_rates=avg_int
		avg_int_beacon=avg_int
		

	time_list_file = control_file_path + 'sit_lister_times.dat'
	if (n_elements(inpathname) eq 0) then input_file =  inpathname +'sit_lister_times.dat'    ; 6/24/08 /gm
	 
	 WHILE ( interval_number lt number_times ) DO BEGIN   ; interval_number = 0,1,2...   number_times = # lines in time list

	get_start_stop_time, time_list_file		; get start and stop days to analyze
;
;	******************** now that we have start and stop time for this plot *******************
;	******************** set flags if this is a multipanel plot during the ********************
;	******************** sit-a conjunction, in which case, substitute scaled ******************
;	******************** beacon rates for the usual rates / 12/21/2015 /gm  *******************
;	******************** need to keep doing rate processing to keep some timers going *********
	multi_panel_out_sit_a_conjunction = 0
	if( (multi_panel_out eq 4 ) and (spacecraft_id eq 0 ) and $ 
		 ( ( (start_mjd ge 56889) and (start_mjd le 57341) )   or $
		 ( ( (stop_mjd ge 56889) and (stop_mjd le 57341) ))    )   ) then begin
		multi_panel_out_sit_a_conjunction = 1
		beacon_out = 1					; turn on beacon processing
		flag_beacon = 1					;    "
	endif
;	******************** end of special flag setup for sit-a conjunction beacon rate substitute ******

	
;	printf, lun_print, ' reading new interval',   start_mjd
	
;	open all output files first time around
	if(interval_number eq 1) then begin
		open_outputs
		if( multi_panel_out gt 0 ) then multi_panel_outputs, start_mjd    ; open files for multi panel plots
	endif
	
	mjd_open = floor( start_mjd )
	
	WHILE (mjd_open le stop_mjd) DO BEGIN      ;  file opening loop
	
	open_sit_file, flag_open_sit_file, spacecraft_id, mjd_open
;	print, 'opening file for day: ',mjd_open, flag_open_sit_file

;	******** tests for missing days;  if 10 missing in a row (not on missing file list) then stop *********
	if( flag_open_sit_file eq 1	) then files_missed = 0				; reset the counter every time we find a file present
	missing_day_test = where( mjd_missing eq mjd_open )				; see if this day is on the missing day list
	if( (flag_open_sit_file eq 0) and (missing_day_test lt 0) ) $ 
							then files_missed ++					; input file wasn't found and day is not on missing day list
;	print, mjd_open, flag_open_sit_file, missing_day_test, files_missed, format="('missing files test: ', 4i6)"
	if( files_missed ge 10 ) then begin								; missing file limit reached
		printf, lun_print, '10 input files in a row missing that are not in missing days list. sit_lister quitting early'
		goto, finish
	endif
	if( flag_open_sit_file eq 0 ) then goto, nextday				; go to the next day if this file not opened

	count_reads=0L	
	packet = 0

	  WHILE 1 DO BEGIN             ; packet reading loop

newrec: 	read_stereo_pkt, lun_input, packet
	if n_elements(packet) eq 0 then break
;	compute packet checksum and print message if there is an error	
	if( run_checks) then packet_checksum, packet
;
;	compute current time
;
	current_time = tai2utc(double(packet.pkt.seconds), /NOCORRECT)
	current_mjd = double(current_time.mjd) + double(current_time.time)/(86400.d0*1000.d0)
	
	
;	compare with the current time interval (start_mjd and stop_mjd)
	if( current_mjd lt start_mjd ) then goto, newrec    ; if before start time, read next packet
	if( current_mjd gt stop_mjd) then break  ; leave packet processing loop if we've gone beyond end time

	
;	process this packet
;	calculate swoosh_energies for this packet time (they are put into the common time_bounds)
	swoosh_energies, current_mjd

	apid = parse_stereo_pkt(packet, /APID )

	proc_flag = 1   ; running value
	branch = apid
	if( (apid ge 606) and (apid le 616) ) then branch = 606

	case branch of

	577:	if( call_process_hk)  then process_hk, packet, flag_hk, proc_flag, run_checks, hk_out
		
	605: 	begin	
			if( call_process_rates )  then process_rates, packet, flag_rates, proc_flag
			if( call_process_pha ) then process_pha_times, packet, apid, run_checks,  histogram_num, proc_flag, pha_case
			end
		
	606:	if( call_process_pha ) then process_pha, packet, apid, run_checks,  histogram_num, proc_flag, pha_case

	624: if(beacon_out) then process_beacon, packet, flag_beacon, proc_flag

	else:	if(apid ne 576) then printf, lun_print, ' illegal APID: ', apid    ; APID 576 is command response; not processed; other values not expected
	
	endcase
	
	if((apid ge 577) and (apid le 616) ) then count_reads=count_reads+1
;	if(count_reads gt 100) then break


	ENDWHILE   ; loop for reading packets from one day (file) ***********

nextday:	
	if( current_mjd gt stop_mjd) then break  ; leave file processing loop if we've gone beyond end time

;	now go on to next day
	mjd_open = mjd_open + 1

;	check the multi panel plots to see if we've finished a plot interval	
	if( multi_panel_out gt 0 ) then multi_panel_outputs, current_mjd    ; open files for multi panel plots

;	************** special case:  if avg_int = 1440, write out results at day boundaries ****************
;	************** 5/17/2011 *****************
;	write out any partial sums to disk or pha plot when we cross day boundaries, unless control_flag = 1
	wrote_daily_outputs = 0
	if( (avg_int eq 1440 ) and (control_flag ge 2) ) then begin
		proc_flag = 4
		if( call_process_hk)  then process_hk, packet, flag_hk, proc_flag, run_checks, hk_out
		if(rates_out or spectra_out)  then process_rates, packet, flag_rates, proc_flag
		if( pha_out or (histogram_out gt 0) or (pha_spectra_out gt 0) or (pha_rates_out gt 0) ) then process_pha, packet, $
				apid, run_checks,  histogram_num, proc_flag
		if(beacon_out) then process_beacon, packet, flag_beacon, proc_flag
				
;	note that we just wrote outputs -- don't do again if we then move to next line
	wrote_daily_outputs = 1
	ENDIF
;	*************** end special 1440 min writeout case *******************

	ENDWHILE   ; loop for file finishing an interval / file opening  *********************************


;	write out any partial sums to disk or pha plot when we cross day boundaries, unless control_flag = 1
	if( (	wrote_daily_outputs eq 0 ) and (control_flag ge 2 ) ) then begin
		proc_flag = 4
		if( call_process_hk)  then process_hk, packet, flag_hk, proc_flag, run_checks, hk_out
		if(rates_out or spectra_out)  then process_rates, packet, flag_rates, proc_flag
		if( pha_out or (histogram_out gt 0) or (pha_spectra_out gt 0) or (pha_rates_out gt 0) ) then process_pha, packet, $
				apid, run_checks,  histogram_num, proc_flag
		if(beacon_out) then process_beacon, packet, flag_beacon, proc_flag
	ENDIF

;	bump histogram number only at end of an interval
	histogram_num ++

;	if control_flag = 3, then quit now even if there are more intervals in the file
	if( control_flag eq 3) then goto, finish
	
	ENDWHILE   ; loop for input lines ************************************

;	finish up	
	print, ' finishing up call'	
;	write out any partial sums to disk or pha plot at end of interval
	proc_flag = 4
	if( call_process_hk)  then process_hk, packet, flag_hk, proc_flag, run_checks, hk_out
	if(rates_out or spectra_out)  then process_rates, packet, flag_rates, proc_flag
	if( pha_out or (histogram_out gt 0) or (pha_spectra_out gt 0) or (pha_rates_out gt 0) ) then process_pha, packet, $
				apid, run_checks,  histogram_num, proc_flag
	if(beacon_out) then process_beacon, packet, flag_beacon, proc_flag
	

	
finish:
;	histograms are written out only at the end since all in multiple columns (one per input time line)
	if( histogram_out ) then write_hist_pha, lun_histogramlist

			if( (rates_out+pha_out+beacon_out+hk_out+spectra_out+pha_spectra_out+pha_rates_out+histogram_out+multi_panel_out) gt 0) then begin
			printf, lun_print, "sit_lister is finished at ",systime()  
			printf, lun_print,  "output files = ",    strmid(out_file,21,22,/reverse_offset)
			printf, lun_print, format="(50('*'), //)"
			close, /all     ; close all the files			
		endif else begin	
			printf, lun_print, " sit_lister is finished"
		endelse

;	send out batch file information and any error status emails:     2/23/2012 /gm
;	but don't quit due to bad boxes after 9/23/2018 /gm
	sit_lister_emails, spacecraft_id, inpathname, start_mjd, exit_status_flag

		if( exit_status_flag ne 0 ) then begin
					print, ' sit_lister exit with error status = ', exit_status_flag
;					exit, status = exit_status_flag		; ****** commented out 13-Nov-2018
		endif

;	stop

	if( bad_box_count gt 0 ) then print, 'number of bad matrix boxes encountered: ', bad_box_count

end 
