PRO SIT_HK, PACKET, SITHKDATA, num_hk
;+
; Project     :	STEREO - IMPACT, SIT
;
; Name        : SIT_HK
;
; Purpose     :	Example program accessing SIT HK data
;
; Category    :	STEREO, IMPACT, SIT
;
; Explanation :	
;
; Syntax      :	SIT_HK, PACKET, SITHKDATA
;
; Examples    :	
;
; Inputs      :	PACKET - a SIT HK telemetry packet (apid 618)
;
; Opt. Inputs :	None.
;
; Outputs     :	SITHKDATA = A structure containing SIT HK data
;               if data is valid, undefined otherwise
;
; Opt. Outputs:	None.
;
; Keywords    :	None
;
; Calls       :	delvarx
;
; Common      : sitcom
;
; Restrictions:	
;
; Side effects:	None.
;
; Prev. Hist. :	None.
;
; History     :	Version 1 (example), 12-Oct-2006 / gm / based on sitrates routine by Andrew Davis
;-
;

; see sit_common.pro
common sitcom

good_data = 1


; structure to hold raw SIT data from HK packet

rawHK = { $
	unused:	bytarr(169), $
	mf_number:	0US, $
	tof_gain:		0US, $
	tof_cal:		0US, $
	tof_cal_error:	0B,   $
	HK_item:		bytarr(8), $
	sw_version:	0US, $
	lookup_checksum:	bytarr(3), $
	unused1:		bytarr(71), $
	cHKsum:		0B $
}	

;	HK item
	HK_item = bytarr(8)
;	lookup checksum
	lookup_checksum = 0UL


; a structure to hold time in Julian day, millisec of day format
timeCDS = 		{ MJD: 0L , TIME: 0L }

; header for unpacked SIT HK data
header = { $
	SWversion: 0B, $	; Beacon processing software version (this software)
	timeCDS: timeCDS, $	; Time at start of data collection interval: Julian Day, millisec_of_day
	SCid: 0U $		; Spacecraft ID
}


; load the raw data bytes into raw SIT HK data structure
; load_struct is a Solarsoft routine


	load_struct, packet.dat, rawHK[0], rawHK, /noieee, /noconv
	
 ; make up the24-bit lookup checksum;  note:  do the calc in double precision or else the intermediate results get lost since the
 ; default data type is a 2-byte integer!
  	lookup_checksum = rawHK.lookup_checksum(0)+rawHK.lookup_checksum(1)*256. + rawHK.lookup_checksum(2)*65536.d0
 	lookup_checksum=ulong(lookup_checksum)


; swap bytes in 2-byte SIT HK words if we are running on a big-endian machine (Solaris or MacOS)
	rawHK.mf_number = swap_endian(rawHK.mf_number, /SWAP_IF_BIG_ENDIAN)
	rawHK.tof_gain = swap_endian(rawHK.tof_gain, /SWAP_IF_BIG_ENDIAN)
	rawHK.tof_cal = swap_endian(rawHK.tof_cal, /SWAP_IF_BIG_ENDIAN)
	rawHK.sw_version = swap_endian(rawHK.sw_version, /SWAP_IF_BIG_ENDIAN)



; just copy the rest of the stuff...
	mf_number = rawHK.mf_number
	tof_gain = rawHK.tof_gain
	tof_cal = rawHK.tof_cal
	tof_cal_error = rawHK.tof_cal_error
	HK_item = rawHK.HK_item
	sw_version = rawHK.sw_version
	lookup_checksum = lookup_checksum
	cHKsum = rawHK.cHKsum
	
; build HK data structure header
header.SWversion = VERSION_sit

; put data-collection time into header
; Use the /NOCORRECT option here, since STEREO spacecraft time already has
; leap seconds factored in. i.e. spacecraft time is UTC seconds since 1958
header.timeCDS = tai2utc(double(packet.pkt.seconds), /NOCORRECT)

; Set spacecraft id in header from packet ground receipt header
	header.SCid = packet.grh.SPACECRAFT_ID



; Assemble SIT HK data structure
	sitHKdata = { $
		header:   header, $
		mf_number:	mf_number, $
		tof_gain:		tof_gain, $
		tof_cal:		tof_cal, $
		tof_cal_error:	tof_cal_error,   $
		HK_item:		hk_item, $
		sw_version:	sw_version, $
		lookup_checksum:	lookup_checksum, $
		cHKsum:    cHKsum $
	}
	

	IF (good_data eq 0) THEN DELVARX, sitHKdata


END
