;+
;FUNCTION:  data_type(x)
;PURPOSE:
;   Returns the variable type (ignores dimension).
;INPUTS: x:   Any idl variable.
;OUTPUT: integer variable type:
;   0 = undefined
;   1 = byte
;   2 = integer
;   3 = long
;   4 = float
;   5 = double
;   6 = complex
;   7 = string
;   8 = structure
;   9 = double precision complex
;
;KEYWORDS:
;   STRUCTURE: When set and if input is a structure, then an array
;       of data types are returned.
;
;SEE ALSO:  "dimen", "ndimen"
;
;CREATED BY:	Davin Larson
;LAST MODIFICATION:	@(#)data_type.pro	1.7 00/07/05
;-
function data_type, val, STRUCTURE=str,n_elements=n_elem,ndimen=ndim
if keyword_set(str) then begin
   n = n_tags(val)
   if n eq 0 then return,0
   t = lonarr(n)
   n_elem = lonarr(n)
   ndim = intarr(n)
   for i=0,n-1 do begin
      t[i]=data_type(val.(i),n_elements=n_el,ndimen=nd)
      n_elem[i] = n_el
      ndim[i] = nd
   endfor
   return,t
endif
s = size(val)
dt = s[ s[0] +1 ]
if dt ne 0 then ndim = s[0] else ndim=-1
n_elem = n_elements(val)
return, dt
end

;+
;NAME:  cdf_info
;FUNCTION:   cdf_info(id)
;PURPOSE:
;  Returns a structure with useful information about a CDF file.
;  In particular the number of file records is returned in this structure.
;INPUT:
;   id:   CDF file ID.
;CREATED BY:    Davin Larson
; LAST MODIFIED: @(#)cdf_info.pro    1.9 02/11/01
; $LastChangedBy: davin-win $
; $LastChangedDate: 2006-12-15 12:31:28 -0800 (Fri, 15 Dec 2006) $
; $LastChangedRevision: 108 $
; $URL: svn+ssh://thmsvn@ambrosia.ssl.berkeley.edu/repos/ssl_general/trunk/CDF/cdf_info.pro $
;-

function cdf_info,id0,data=ret_data,attributesf=ret_attr       ;,format=format

if not keyword_set(id0) then id0 = dialog_pickfile(/multi)

;for i=0,n_elements(id0)-1 do begin

if size(/type,id0) eq 7 then begin
   if file_test(id0) then  id=cdf_open(id0)  $
   else begin
      print,'File not found: ',id0
      return,0
   endelse
endif  else id=id0

inq = cdf_inquire(id)
q = !quiet
cdf_control,id,get_filename=fn
nullp = ptr_new()
varinfo_format = {name:'',num:0, is_zvar:0, datatype:'',type:0, $
;   depend_0:'', $
   numattr:-1,  $
;   userflag1:0, $
;   userstr1:'', $
;   index:0,     $
   numelem:0, recvary:0b, numrec:0l, $
   ndimen:0, d:lonarr(6) , $
   dataptr:ptr_new(), attrptr:ptr_new()  }

nv = inq.nvars+inq.nzvars
vinfo = nv gt 0 ? replicate(varinfo_format, nv) : 0
i = 0

forward_function  cdf_var_atts    ; 4/22/2010 /gm

g_atts = cdf_var_atts(id)
num_recs =0
for zvar = 0,1 do begin   ; regular variables first, then zvariables
  nvars = zvar ? inq.nzvars : inq.nvars
  for v = 0,nvars-1 do begin
    vi = cdf_varinq(id,v,zvar=zvar)
    vinfo[i].num = v
    vinfo[i].is_zvar = zvar
    vinfo[i].name = vi.name
    vinfo[i].datatype = vi.datatype
    
    forward_function cdf_var_type   ;  4/22/2010 /gm
    
    vinfo[i].type = cdf_var_type(vi.datatype)
;    if vi.name eq 'Epoch' then num_recs = info.maxrec+1
;    if vi.datatype eq 'CDF_EPOCH' then num_recs = info.maxrec+1
    vinfo[i].numelem = vi.numelem
    recvar = vi.recvar eq 'VARY'
    vinfo[i].recvary = recvar

;    if recvar then begin
      !quiet = 1
      cdf_control,id,var=v,get_var_info=info,zvar = zvar
      !quiet = q
      nrecs = info.maxrec+1
;    endif else nrecs = -1
    vinfo[i].numrec = nrecs

    if zvar then begin
      dimen = [vi.dim]
      ndimen = total(vi.dimvar)
    endif else begin
      dimc = vi.dimvar * inq.dim
      w = where(dimc ne 0,ndimen)
      if ndimen ne 0 then dimen = dimc[w] else dimen=0
    endelse
;    if recvar then begin
;       dimen = [nrecs,dimen]
;       ndimen = ndimen+1
;    endif
    vinfo[i].ndimen = ndimen
    vinfo[i].d =  dimen
    attr = cdf_var_atts(id,vi.name)
    if keyword_set(ret_data) then begin
;       var_type=''
;       str_element,attr,'VAR_TYPE',var_type
       cdf_varget,id,vi.name,value  ;,rec_count=nrecs                ;,string= var_type eq 'metadata'
       value=reform(value,/overwrite)                             ;  get rid of trailing 1's
       vinfo[i].dataptr = ptr_new(value,/no_copy)
    endif

    if keyword_set(ret_attr) then begin
       vinfo[i].attrptr = ptr_new(attr,/no_copy)
    endif
    i = i+1
  endfor
endfor

;;attribute stuff
;atttest = 0
;
;if atttest then begin
;for a=0,inq.natts-1 do begin
;  cdf_control,id,att=a,get_attr_info=ai,get_numattrs=na
;  cdf_attinq,id,a,name,scope,maxrent,maxzent
;  scp = strmid(scope,0,1)
;  enum = 0
;  if cdf_attexists(id,a,enum) then begin
;     cdf_attget,id,a,enum,value,cdf_type=cdf_type
;    print,a,maxrent,maxzent,name,scp,value,format='(i2,i3,i3,a20,a3," ",a60)'
;  endif else begin
;    print,a,maxrent,maxzent,name,scp,format='(i2,i3,i3,a20,a3)'
;  endelse
;endfor
;endif

res = create_struct('filename',fn,'inq',inq,'g_attributes',g_atts,'nv',nv,'vars',vinfo)  ;'num_recs',num_recs,'nvars',nv
if size(/type,id0) eq 7 then cdf_close,id
;format='(i3,i2,a16,a12,i4,i4,i3,i6,i5,i6,i3,i3,i3,i3,i3,i3,i3)'
return,res
end


;+
;NAME:  cdf_var_atts
;FUNCTION:   cdf_var_atts(id [,varname[,attname]])
;PURPOSE:
;  Returns a structure that contains the attributes of a variable within
;  a CDF file. If attname is provided then it returns the value of that attribute.
;KEYWORDS:
;  DEFAULT: The default value of the attribute.
;
;INPUT:
;   id:   CDF file ID or filename.
;CREATED BY:    Davin Larson
;LAST MODIFIED: @(#)cdf_var_atts.pro    1.2 02/11/01
; $LastChangedBy: davin-win $
; $LastChangedDate: 2006-12-05 17:43:39 -0800 (Tue, 05 Dec 2006) $
; $LastChangedRevision: 52 $
; $URL: svn+ssh://thmsvn@ambrosia.ssl.berkeley.edu/repos/ssl_general/trunk/CDF/cdf_var_atts.pro $
;-

function  cdf_var_atts,id0,var,attname,default=default,zvar=zvar,names_only=names_only  ;,attributes=att
if size(/type,id0) eq 7 then id=cdf_open(id0) else id=id0
if n_elements(default) eq 0 then default=0
attstr = default
inq = cdf_inquire(id)
if inq.natts eq 0 then goto,done

if keyword_set(attname) then begin
  if cdf_attexists(id,attname,var,zvar=zvar) then cdf_attget,id,attname,var,zvar=zvar,attstr
  goto,done
endif

cdf_control,id,get_numattrs=na   ; ,get_filename=fn
;print,fn
;print,na
;printdat,inq

cdf_control,id,att=0,get_attr_info=ai
att0 = {num:0, name:'',scope:'',maxr:0, maxz:0}
att0 = create_struct(att0,ai)
att = replicate(att0,inq.natts)

for a=0,inq.natts-1 do begin
  cdf_control,id,att=a,get_attr_info=ai
  cdf_attinq,id,a,name,scope,maxrent,maxzent
;  scp = strmid(scope,0,1)
  att0.num = a
  att0.name = name
  att0.scope= strmid(scope,0,1)
  att0.maxr = maxrent
  att0.maxz = maxzent
  struct_assign,ai,att0,/nozero
  att[a] = att0
endfor

if keyword_set(var) then begin
vinq = cdf_varinq(id,var,zvar=zvar)
w = where(att.scope eq 'V',nva)
if nva ne 0 then begin
   vatt=att[w]
   for a=0,nva-1 do begin
      if cdf_attexists(id,vatt[a].num,var,zvar=zvar) eq 0 then continue
      if keyword_set(names_only) then begin
          if keyword_set(attstr) then attstr=[attstr,vatt[a].name] $
          else attstr=vatt[a].name
      endif else begin
          cdf_attget,id,vatt[a].name,var,value,zvar=zvar
          if size(/type,value) eq 7 then if  strpos(value,'>$<') ge 1 then $
              value = strsplit(/extract,value,'>$<')

          str_element,/add,attstr,vatt[a].name,value
;                   if keyword_set(attstr) then $
;                  attstr = create_struct(attstr,vatt[a].name,value)  $
;                  else attstr = create_struct(vatt[a].name,value)
      endelse
   endfor
endif

endif else begin ;global
w = where(att.scope eq 'G',nga)
if nga ne 0 then begin
   vatt=att[w]
   gentry = 0
;   attstr = {filename:fn}
   for a=0,nga-1 do begin
      if cdf_attexists(id,vatt[a].num,gentry) eq 0 then continue
      if keyword_set(names_only) then begin
          if keyword_set(attstr) then attstr=[attstr,vatt[a].name] else attstr=vatt[w].name
      endif else begin
         cdf_attget,id,vatt[a].name,gentry,value
         if keyword_set(attstr) then $
            attstr = create_struct(attstr,vatt[a].name,value)  $
         else attstr = create_struct(vatt[a].name,value)
      endelse
   endfor
endif


endelse

done:
if size(/type,id0) eq 7 then cdf_close,id
return,attstr
end




function cdf_var_type,string
stypes = 'CDF_'+strsplit(/extr,'XXX BYTE UINT1 INT1 CHAR UCHAR INT2 UINT2 INT4 UINT4 REAL4 FLOAT DOUBLE REAL8 EPOCH EPOCH16 LONG_EPOCH')
vtypes = [0,1,1,1,1,1,2,12,3,13,4,4,5,5,5,9,9]

forward_function array_union    ; 4/22/2010  /gm

type = array_union(string,stypes)
return,(vtypes[type])[0]
end




;+
;FUNCTION: ndimen
;PURPOSE:
;  Returns the number of dimensions in an array.
;INPUT:  array
;RETURNS number of dimensions  (0 for scalers,-1 for undefined)
;
;SEE ALSO:  "dimen", "data_type"
;
;CREATED BY:	Davin Larson
;LAST MODIFICATION:	@(#)ndimen.pro	1.6 97/03/10
;-
function ndimen,matrx            ;returns number of dimensions in a variable
if n_elements(matrx) eq 0 then return ,-1l
n = size(matrx)
return, n(0)
end



;+
;PROCEDURE:	loadcdf2
;PURPOSE:	
;   Loads one type of data from specified cdf file.
;INPUT:		
;	CDF_file:	the file to load data from  (or the id of an open file)
;	CDF_var:	the variable to load
;	x:		the variable to load data into
;
;KEYWORDS:
;	zvar:	 	must be set if variable to be loaded is a zvariable.
;       append:         appends data to the end of x instead of overwriting it.
;       nrecs:          number of records to be read.
;	no_shift:	if set, do not perform dimen_shift to data.
;	rec_start:	CDF record number to begin reading.
;
;CREATED BY:	Jim Byrnes, heavily modified by Davin Larson (was loadcdf)
;MODIFICATIONS:
;  96-6-26  added APPEND keyword
;LAST MODIFICATION:	@(#)loadcdf2.pro	1.5 98/08/13
;-

; The following program will load all of the data for a specific CDF file and
; variable into IDL. 
; 
;(Jim Byrnes)

pro loadcdf2,CDF_file,CDF_var,x0, zvar = zvar, $
   append=append,no_shift=no_shift,nrecs = nrecs,rec_start=rec_start

;ON_ERROR,1

;
; Open CDF file  (if neccesary)
;
if data_type(cdf_file) eq 7 then id = cdf_open(cdf_file) else id = cdf_file

;
; Get file CDF structure information
; 

inq = cdf_info(id)

;
; Get variable structure information
;

vinq = cdf_varinq(id,CDF_var, zvariable = zvar)
zvar = vinq.is_zvar
;help, vinq,/st
;help,zvar

!quiet = 1
cdf_control,id,variable=CDF_var,get_var_info=varinfo
!quiet = 0

if not keyword_set(nrecs) then nrecs = varinfo.maxrec+1
if vinq.recvar eq 'NOVARY' then nrecs = 1

dims = total(vinq.dimvar)

if keyword_set(zvar) then begin
  CDF_varget,id,CDF_var,x,REC_COUNT=nrecs,zvariable = zvar,rec_start=rec_start
endif else begin
  dimc = vinq.dimvar * inq.dim
  dimw = where(dimc eq 0,c)
  if c ne 0 then dimc(dimw) = 1
  CDF_varget,id,CDF_var,x,COUNT=dimc,REC_COUNT=nrecs,rec_start=rec_start
endelse

;help,cdf_var,x


if vinq.recvar eq 'VARY' and dims ne 0 and not keyword_set(no_shift) then $
   x = dimen_shift(x,1) 

if ndimen(x) gt 0 then x = reform(x,/overwrite)

;help,cdf_var,x

if keyword_set(append) and keyword_set(x0)  then x0=[x0,x] else x0=x

if data_type(cdf_file) eq 7 then cdf_close,id
return
end




;+
;PROCEDURE: append_array, a0, a1
;PURPOSE:
;   Append an array to another array.  Can also copy an array into a
;   subset of another.
;INPUT:
;   a0:   Array to modify.
;   a1:   Array to append to, or copy into, a0.
;KEYWORDS:
;   index:    Index of a0 at which to append or copy a1.  If index is
;     greater than the number of elements of a0, then
;     a0 is enlarged to append a1. Returns the index of the first
;     element of a0 past the section copied from a1.
;   done: If set, make a0 equal to the first index elements of a0.
;CREATED BY:    Davin Larson
;LAST MODIFIED: @(#)append_array.pro    1.6 98/08/13
;-
pro append_array,a0,a1,index=index,done=done

if arg_present(index) then begin
  if keyword_set(done) then begin
    if not keyword_set(index) then return
     a0 = a0[0:index-1]
     return
  endif
  xfactor = .5
  n0 = n_elements(a0)
  n1 = n_elements(a1)
  if not keyword_set(index) then begin
     index = 0l
     n0 = 0l
  endif
  if n1+index ge n0 then begin
     ;message,/info,"Enlarging array"
     add = floor(n0 * xfactor + n1)
     if index ne 0 then a0 = [a0,replicate(a1[0],add)] $
     else a0 = replicate(a1[0],add)
     n0=n0+add
  endif
  a0[index:index+n1-1] = a1
  index = index + n1
  return
endif


if keyword_set(a0) then a0=[a0,a1] else a0=[a1]

end





function tag_names_r,structure,data_type=all_dt

struct= structure[0]
dta = data_type(struct,/str)
if not keyword_set(dta) then return,0

tags = tag_names(struct)
ntags = n_elements(dta)

for i=0,ntags-1 do begin
   if dta[i] ne 8 then begin
      names=tags[i]
      dt = dta[i]
   endif else $
      names = tags[i]+'.'+tag_names_r(struct.(i),data_type=dt)
   append_array,all_tags,names,index=ind_tg
   append_array,all_dt  ,dt,index=ind_dt
endfor

append_array,all_tags,index=ind_tg,/done
append_array,all_dt,index=ind_dt,/done

return,all_tags
end


;+
;PROCEDURE:  str_element, struct,  tagname, value
;PURPOSE:
; Find (or add) an element of a structure.
; This procedure will not
; Input:
;   struct,  generic structure
;   tagname,    string  (tag name)
; Output:
;   value,  Named variable in which value of the structure element is returned.
; Purpose:
;   Retrieves the value of a structure element.  This function will not produce
;   an error if the tag and/or structure does not exist.
;KEYWORDS:
;  SUCCESS:  Named variable that will contain a 1 if the element was found
;     or a 0 if not found.
;  INDEX: a named variable in which the element index is returned.  The index
;     will be -2 if struct is not a structure,  -1 if the tag is not found,
;     and >= 0 if successful.
;  ADD_REPLACE:  Set this keyword to add or replace a structure element.
;  DELETE:   Set this keyword to delete the tagname.
;  CLOSEST:  Set this keyword to allow near matchs (useful with _extra)
;  VALUE: (obsolete) alternate method of returning value. (Will not work with recursion)
;Notes:
;  Value remains unchanged if the structure element does not exist.
;  If tagname contains a '.' then the structure is recursively searched and
;    index will be an array of indices.
;  If struct is an array then results may be unpredictable.
;
;Modifications:
;  5/7/97: Added recursive searching of structure hierarchy.  D. Larson
;
;CREATED BY:    Davin Larson
;FILE:  str_element.pro
;VERSION  1.10
;LAST MODIFICATION: 01/10/08
;-
pro str_element,struct,tagname,value,  $
   ADD_REPLACE = add_rep, $
   DELETE = delete, $
   CLOSEST = closest, $
   SUCCESS = success, $
   VALUE = value2,   $   ;obsolete keyword
   INDEX = index

pos = strpos(tagname,'.')
if pos ge 0 then begin
   base_name = strupcase( strmid(tagname,0,pos) )
   ext  = strmid(tagname,pos+1,100)
endif else base_name=strupcase(tagname)

;closest = 1
success = 0
if data_type(struct) ne 8 then  index = -2 else begin
   tags = tag_names(struct)
   index = (where(base_name eq tags,count))[0]
   if count gt 1 then message,/info,'More than one exact match of '+base_name+' found.'
   if count eq 0 and keyword_set(closest) then begin
      p = intarr(n_elements(tags))
      for i=0,n_elements(tags)-1 do   p[i] = strpos(base_name,tags[i])
      mx = max((p eq 0) * strlen(tags),index)
      if mx eq 0 then index=-1
 ;printdat,tags
 ;printdat,p
 ;printdat,index
      w = where(p eq 0,count)
      if count ge 2 then  $
        message,/info,'Warning: multiple close matchs of '+base_name+' found:'+string(/print,tags[w])
      if count eq 1 then message,/info,'Near match of '+base_name+' found: '+tags[index]
   endif
endelse

n = index

if pos ge 0 then begin          ; make recursive call
   if index ge 0 then new_struct= struct.(index)
   str_element,new_struct,ext,value, index=i, success=success,  $
          add_rep=add_rep,delete=delete
   if keyword_set(add_rep) then $
          str_element,struct,base_name,new_struct,/add_rep
   index = [index,i]
   return
endif

if keyword_set(add_rep) and (n_elements(value) eq 0) then delete=1

if keyword_set(delete) then begin
   delete_var = n
   add_rep = n ge 0
endif else delete_var=-1


if keyword_set(add_rep) or keyword_set(delete) then begin

  if n_elements(struct) gt 1 then begin    ; special case:  arrays of structs
    replace = keyword_set(delete)
    replace = replace or ( n lt 0 )
;    replace = replace or (data_type(value) eq 8)
    if not replace then begin
      s1 = size(struct.(n))
      s2 = size(value)
      w = where(s1 ne s2,diff_type)
      replace = replace or (diff_type ne 0)
    endif
    if not replace and (data_type(value) eq 8) then begin
       new_tags= tag_names_r(value[0],data_type=new_dt)
       old_tags= tag_names_r(struct[0].(n),data_type=old_dt)
       replace =  n_elements(new_tags) ne n_elements(old_tags)
       w = where(new_tags ne old_tags,diff_type)
       replace = replace or (diff_type ne 0)
       w = where(new_dt ne old_dt,diff_type)
       replace = replace or (diff_type ne 0)
    endif
    if replace then begin
;      message,/info,'Testing...  Adding '+base_name+' to array of structures'
      s0 = struct[0]
;      d = dimen(value)
      v0 = value[0]   ;needs work: should be all but last dimension
      str_element,/add,s0,base_name,v0,delete=delete,index=nj
;  help,nj,s0,v0,/st
      new_struct = make_array(value=s0,dim=dimen(struct))
      ntags = n_tags(new_struct)
      tags = tag_names(new_struct)
;      old_tags = tag_names(struct)
      for i=0,ntags-1 do begin
;         if i ne delete_var then begin
           str_element,s0,tags[i],index=j
           if i eq nj then new_value = value else new_value=struct.(j)
;           str_element,s0,old_tags[i],index=j
           new_struct.(i) = new_value
;         endif
      endfor
;      str_element,s0,base_name,index=j
;      if not keyword_set(delete) then new_struct.(j) = value
      struct=new_struct
    endif else begin
      struct.(n)=value
    endelse
    return
  endif


  case n of
      -2:  if n_elements(value) ne 0  then $
                struct = create_struct(idl_validname(/convert_all,base_name),value)  ; struct did not exist
      -1:  if n_elements(value) ne 0  then $
                struct = create_struct(struct,idl_validname(/convert_all,base_name),value)   ; add new tag
    else:  begin
       replace = keyword_set(delete)
       replace = replace or (data_type(value) eq 8)
       if not replace then begin
          s1 = size(struct.(n))
          s2 = size(value)
          w = where(s1 ne s2,diff_type)
          replace = replace or (diff_type ne 0)
       endif
       if replace then begin  ;       new type: replace value
          ntags = n_elements(tags)
          new_struct = 0
          for i=0,ntags-1 do begin
             if i ne delete_var then begin
               if i eq n then new_value=value else new_value=struct.(i)
               if not keyword_set(new_struct) then  $
                    new_struct = create_struct(idl_validname(/convert_all,tags[i]),new_value) $
               else new_struct = create_struct(new_struct,idl_validname(/convert_all,tags[i]),new_value)
             endif
          endfor
          struct = new_struct
       endif  else struct.(n)=value         ;same type: copy value
    endelse
  endcase
  success = 1
  if n lt 0 then index = n_tags(struct)-1

endif else begin

   if n ge 0 then begin
      value = struct.(n)
      value2 = value
      success=1
   endif

endelse

return
end




; + 
;FUNCTION array_union(A,B)
;PURPOSE:
;   Returns an array of indices of B that have elements common to A
;   The dimensions of the returned array are the same as A.
;   if an element of A is not found in B then the corresponding index is -1
; -
function  array_union,a,b
ind = replicate(-1,n_elements(a))
for i=0,n_elements(a)-1 do begin
   x = where(a(i) eq b,count)
   if count gt 0 then ind(i) = x(0)
endfor
return,ind
end





	FUNCTION IS_NOT_MISSING,ARRAY,MISSING=MISSING
;+
; Project     : SOHO - CDS
;
; Name        : IS_NOT_MISSING()
;
; Purpose     : Returns whether or not array pixels are missing
;
; Explanation : Returns a vector array containing a boolean value signalling
;               good pixels, i.e. those pixels with a finite value (e.g. not
;               NaN), and not equal to a missing pixel flag value.
;
; Use         : Result = IS_NOT_MISSING( ARRAY, <keywords> )
;
; Inputs      : ARRAY	= Array to examine for missing pixels.
;
; Opt. Inputs : None.
;
; Outputs     : Result of function is an array containing 0 for missing pixels,
;               and 1 for good pixels.
;
; Opt. Outputs: None.
;
; Keywords    : MISSING = Value flagging missing pixels.
;
; Calls       : None.
;
; Common      : None.
;
; Restrictions: None.
;
; Side effects: None.
;
; Category    : Utilities
;
; Prev. Hist. : None.
;
; Written     : William Thompson, GSFC, 29 April 2005
;
; Modified    : Version 1, William Thompson, GSFC, 29 April 2005
;
; Version     : Version 1, 29 April 2005
;-
;
	IF N_ELEMENTS(MISSING) EQ 1 THEN $
            RETURN, (ARRAY NE MISSING) AND FINITE(ARRAY) $
        ELSE RETURN, FINITE(ARRAY)
;
	END


function where_missing, array, count, missing=missing, _ref_extra=_extra
;+
; Project     : SOHO - CDS
;
; Name        : WHERE_MISSING()
;
; Purpose     : Returns the position of all missing pixels in an array.
;
; Explanation : Returns a vector array containing the position of missing
;               pixels, i.e. those pixels with a non-finite value (e.g. NaN),
;               or equal to a missing pixel flag value.
;
; Use         : Result = WHERE_MISSING( ARRAY, <keywords> )
;
; Inputs      : ARRAY	= Array to examine for missing pixels.
;
; Opt. Inputs : None.
;
; Outputs     : Result of function is a linear array containing the positions
;               of all missing pixels.
;
; Opt. Outputs: COUNT   = The number of missing pixels found.
;
; Keywords    : MISSING = Value flagging missing pixels.
;
;               COMPLEMENT, NCOMPLEMENT = Returns the position and number of
;                         non-missing pixels.  (IDL 5.4 and higher)
;
; Calls       : None.
;
; Common      : None.
;
; Restrictions: None.
;
; Side effects: None.
;
; Category    : Utilities
;
; Prev. Hist. : None.
;
; Written     : William Thompson, GSFC, 29 April 2005
;
; Modified    : Version 1, William Thompson, GSFC, 29 April 2005
;               Version 2, William Thompson, GSFC, 01-Jun-2005
;                       Use EXECUTE for pre-5.4 compatibility
;
; Version     : Version 2, 01-Jun-2005
;-
;
command = 'w = where((finite(array) ne 1)'
if n_elements(missing) eq 1 then command = command + ' or (array eq missing)'
command = command + ', count'
if !version.release ge '5.4' then command = command + ', _extra=_extra'
command = command + ')'
test = execute(command)
return, w
;
end


function where_not_missing, array, count, missing=missing, _ref_extra=_extra
;+
; Project     : SOHO - CDS
;
; Name        : WHERE_NOT_MISSING()
;
; Purpose     : Returns the position of all non-missing pixels in an array.
;
; Explanation : Returns a vector array containing the position of non-missing
;               pixels, i.e. those pixels with a finite value (i.e. not NaN),
;               and not equal to a missing pixel flag value.
;
; Use         : Result = WHERE_NOT_MISSING( ARRAY, <keywords> )
;
; Inputs      : ARRAY	= Array to examine for missing pixels.
;
; Opt. Inputs : None.
;
; Outputs     : Result of function is a linear array containing the positions
;               of all non-missing pixels.
;
; Opt. Outputs: COUNT   = The number of non-missing pixels found.
;
; Keywords    : MISSING = Value flagging missing pixels.
;
;               COMPLEMENT, NCOMPLEMENT = Returns the position and number of
;                         missing pixels.  (IDL 5.4 and higher)
;
; Calls       : None.
;
; Common      : None.
;
; Restrictions: None.
;
; Side effects: None.
;
; Category    : Utilities
;
; Prev. Hist. : None.
;
; Written     : William Thompson, GSFC, 29 April 2005
;
; Modified    : Version 1, William Thompson, GSFC, 29 April 2005
;               Version 2, William Thompson, GSFC, 01-Jun-2005
;                       Use EXECUTE for pre-5.4 compatibility
;
; Version     : Version 2, 01-Jun-2005
;-
;
command = 'w = where((finite(array) eq 1)'
if n_elements(missing) eq 1 then command = command + ' and (array ne missing)'
command = command + ', count'
if !version.release ge '5.4' then command = command + ', _extra=_extra'
command = command + ')'
test = execute(command)
return, w
;
end




	FUNCTION AVERAGE, ARRAY, DIMENSION, MISSING=MISSING
;+
; Project     : SOHO - CDS
;
; Name        : 
;	AVERAGE()
; Purpose     : 
;	Averages an array over one or all of its dimensions.
; Explanation : 
;	Calculates the average value of an array, or calculates the average
;	value over one dimension of an array as a function of all the other
;	dimensions.
; Use         : 
;	Result = AVERAGE( ARRAY )
;	Result = AVERAGE( ARRAY, DIMENSION )
;	Result = AVERAGE( ARRAY, [Dim1, Dim2, ... ] )
; Inputs      : 
;	ARRAY	  = Input array.  May be any type except string or structure.
; Opt. Inputs : 
;	DIMENSION = Optional dimension to do average over.  Valid inputs are 1
;		    through the total number of dimensions of ARRAY.
; Outputs     : 
;	The average value of the array when called with one parameter.
;
;	If DIMENSION is passed, then the result is an array with all the
;	dimensions of the input array except for the dimension specified,
;	each element of which is the average of the corresponding vector
;	in the input array.
;
;	For example, if A is an array with dimensions of (3,4,5), then the
;	command B = AVERAGE(A,2) is equivalent to
;
;		B = FLTARR(3,5)
;		FOR J = 0,4 DO BEGIN
;		    FOR I = 0,2 DO BEGIN
;			B(I,J) = TOTAL( A(I,*,J) ) / 4.
;		    ENDFOR
;		ENDFOR
;
;	It is also possible to pass an array of dimensions.  The dimensions are
;	processed in reverse order.  For example,
;
;		B = AVERAGE(A, [1,3])
;
;	is equivalent to
;
;		B = AVERAGE(A, 3)
;		B = AVERAGE(B, 1)
;
; Opt. Outputs: 
;	None.
; Keywords    : 
;	MISSING	= Value signifying missing pixels.  Any pixels with this value
;		  are not included in the average.  If there are no non-missing
;		  pixels, then MISSING is returned.
; Calls       : 
;	IS_NOT_MISSING, WHERE_MISSING, WHERE_NOT_MISSING, FLAG_MISSING
; Common      : 
;	None.
; Restrictions: 
;	The dimension specified must be valid for the array passed.
; Side effects: 
;	None.
; Category    : 
;	Utilities, Arrays.
; Prev. Hist. : 
;	Taken from an earlier routine by W. Thompson called AVG, but the
;	definition of the DIMENSION parameter is different to be consistent
;	with current usage in IDL.
; Written     : 
;	William Thompson, GSFC, 9 April 1993.
; Modified    : 
;	Version 1, William Thompson, GSFC, 9 April 1993.
;	Version 2, William Thompson, GSFC, 3 February 1996
;		Added missing keyword.
;	Version 3, 03-Jul-2000, William Thompson, GSFC
;		Modified to use /DOUBLE with TOTAL.  However, the answer
;		retains the datatype of the previous version of this routine.
;	Version 4, 14-Nov-2000, William Thompson, GSFC
;		Modified to also support version 4.
;	Version 5, 16-Nov-2001, William Thompson, GSFC
;		Allow DIMENSION to be a vector.
;	Version 6, 16-Apr-2002, William Thompson, GSFC
;		Fix bug when trailing dimensions are 1.
;       Version 7, 29-Apr-2005, William Thompson, GSFC
;               Handle NaN values
;       Version 8, 1-Jul-2005, William Thompson, GSFC
;               Fixed bug when entire array is missing
; Version     : 
;	Version 8, 1-Jul-2005
;-
;
	ON_ERROR,2
;
;  Check the input parameters.
;
	IF N_PARAMS() LT 1 THEN MESSAGE,	$
		'Syntax:  Result = AVERAGE( ARRAY  [, DIMENSION ] )'
	IF N_ELEMENTS(ARRAY) EQ 0 THEN MESSAGE,	$
		'ARRAY not defined'
;
;  Dimension not passed.  Return a simple average.  If the keyword MISSING was
;  sent, then only average together the points not equal to the missing value.
;  If there are no non-missing pixels, then return the missing value.
;
	IF N_PARAMS(0) EQ 1 THEN BEGIN
	    AVER = ARRAY(0) * 1.0
            W = WHERE_NOT_MISSING(ARRAY,MISSING=MISSING,COUNT)
            IF COUNT EQ 0 THEN AVER[0] = ARRAY[0] ELSE $
              AVER[0] = TOTAL(ARRAY[W],/DOUBLE) / COUNT
;
;  Dimension passed.  Check DIMENSION, and make sure that ARRAY is an array.
;
	END ELSE BEGIN
		IF N_ELEMENTS(DIMENSION) EQ 0 THEN BEGIN
			MESSAGE,'DIMENSION not defined'
		END ELSE IF N_ELEMENTS(DIMENSION) GT 1 THEN BEGIN
		    DIM = REVERSE(DIMENSION(SORT(DIMENSION)))
		    RETURN, AVERAGE( AVERAGE(ARRAY,DIM(0),MISSING=MISSING), $
			    DIM(1:*), MISSING=MISSING)
		ENDIF
		S = SIZE(ARRAY)
		IF S(0) EQ 0 THEN MESSAGE,'ARRAY must be an array'
;
;  Return an array collapsed along one of the dimensions.
;
		DIM = DIMENSION(0)
		IF (DIM GE 1) AND (DIM LE S(0)) THEN BEGIN
;
;  Create an array with the right dimensions, of at least floating point type.
;
		    SZ = SIZE(ARRAY)
		    TYPE = SZ(SZ(0)+1) > 4
		    SZ = SZ(WHERE(INDGEN(N_ELEMENTS(SZ)) NE DIM))
		    AVER = MAKE_ARRAY(TYPE=TYPE, DIMENSION=SZ(1:SZ(0)-1), $
			    /NOZERO)
;
;  Start by calculating the numerator, substituting 0 wherever the missing
;  pixel flag is seen.
;
                    TEMP = ARRAY
                    W = WHERE_MISSING(ARRAY, MISSING=MISSING, COUNT)
                    IF COUNT GT 0 THEN TEMP(W) = 0
                    SZ = SIZE(TEMP)
                    IF SZ(0) LT DIM THEN	$
                      AVER(0,0,0,0,0,0,0) = TEMP	ELSE $
                      AVER(0,0,0,0,0,0,0) = TOTAL(TEMP, DIM, /DOUBLE)
;
;  Next calculate the denominator as the total number of points which are good.
;  Substitute the MISSING pixel value where-ever there are no good pixels to
;  average together.
;
                    TEMP = IS_NOT_MISSING(ARRAY, MISSING=MISSING)
                    IF SZ(0) LT DIM THEN DENOM = TEMP ELSE $
                                         DENOM = TOTAL(TEMP, DIM)
                    AVER = TEMPORARY(AVER) / (DENOM > 1)
                    W = WHERE(DENOM EQ 0, COUNT)
                    IF COUNT GT 0 THEN FLAG_MISSING, AVER, W, MISSING=MISSING
		END ELSE BEGIN
			MESSAGE,'Dimension out of range'
		ENDELSE
	ENDELSE
;
	RETURN, AVER
	END


forward_function average_str_1
forward_function fill_str_zeroes

function fill_str_zeroes,d0
  dtype = data_type(d0,/struct)
  avable = [0,0,1,1,1,1,0,0,0,0,0,0,0]
  for i = 0,n_elements(dtype)-1 do begin
    if (dtype(i) ne 8) then begin
      if avable(dtype(i)) then d0.(i) = !values.f_nan else d0.(i) = 0
    endif else d0.(i) = fill_str_zeroes(d0.(i))
  endfor
return,d0
end
  
function average_str_1,data,nan=nan,median=med
   nt = n_tags(data)
   d = data(0)
   dtype = data_type(d,/struct)
   for i=0,nt-1 do begin
      if (dtype(i) ne 8) then begin
        ndim = ndimen(data.(i))
        d.(i) = average(data.(i),ndim,nan=nan,ret_median=med)
      endif else d.(i) = average_str_1(data.(i),nan=nan,median=med)
   endfor
   return,d
end

;+
;FUNCTION:	average_str(data, res)
;PURPOSE:
;	Average data in res second time segments.
;INPUTS:
;	DATA:	array of structures.  One element of structure must be TIME.
;	RES:	resolution in seconds.
;KEYWORDS:
;	NAN:	If set, treat the IEEE NAN value as missing data.
;CREATED BY:	Davin Larson
;LAST MODIFIED:	%W% %E%
;-
function average_str, data, res, nan=nan, median=med

n = n_elements(data)
d = data(0)

ind = floor(data.time / res)
start = ind(0)

ind = ind - ind(0)

max = ind(n-1)+1
d0 = data(0)

d0 = fill_str_zeroes(d0)

d = replicate(d0,max)

if max gt 1 then d.time = (dindgen(max)+5d-1+double(start))*double(res)  $
else d.time = (double(start)+5d-1)*double(res)

for i=0,max-1 do begin
   w = where(ind eq i,c)
   if c eq 1 then d(i) = data(w)
   if c gt 1 then d(i) = average_str_1(data(w),nan=nan,median=med)
endfor

report

return,d
end


pro cdf_file_names_index,name,info,set=set,get=get

common cdf_file_names_index_com,index

f = {indexformat,name:'',ptr:ptr_new()}

n = 0
i = -1
if size(/type,name) eq 7 and keyword_set(index) then  $
    i = (where(name eq index.name,n))[0]
if n ge 2 then message,'Error'

if keyword_set(set) then begin
   if keyword_set(info) then begin
     if n eq 0 then begin
        f.name = name
        f.ptr = ptr_new(info)
        index = keyword_set(index) ? [index,f] : [f]
     endif else begin
        *index[i].ptr = info
     endelse
   endif else begin           ; Delete
     if n ne 0 then begin
        ptr_free,index[i].ptr
        ni = where(name ne index.name,n)
        if n ne 0 then index=index[ni] else index=0        
     endif
   endelse
   return
endif

if keyword_set(get) then begin
   info = (n ne 0) ? *index[i].ptr : 0
   return
endif

if not keyword_set(index) then begin
   print, 'Nothing stored!'
   return
endif

if n_elements(name) eq 0 then begin
   print, transpose(["'"+ index.name +"'"]) 
   return
endif

if i ge 0 then   printdat,index[i] else print,'Name not found!'

end


pro file_YYYYMMDD_to_time,allfiles

  datepos = 16
    n = n_elements(allfiles)
    pos = strlen(allfiles.name)-datepos
    
    if n gt 1 then $
    	date = long(strmid(allfiles.name,transpose(pos),8)) $
    else date = long(strmid(allfiles.name,pos,8))

    t = replicate(time_struct(0d),n)
    t.year = date/10000
    date = date mod 10000
    t.month = date/100
    t.date = date mod 100
    allfiles.start= time_double(t)
  	
    st = sort(allfiles.start)
    allfiles = allfiles[st]
    allfiles.stop = allfiles.start + 86400d -1d
;    allfiles.stop = shift(allfiles.start,-1) - 1d
;    allfiles[n-1].stop = allfiles[n-1].start + 86400d - 1d

end


pro file_bartel_to_time,allfiles

  datepos = 12
    n = n_elements(allfiles)
    pos = strlen(allfiles.name)-datepos
    
    bartnum = long(strmid(allfiles.name,transpose(pos),4))

    allfiles.start= bartel_time(bartnum)
  	
    st = sort(allfiles.start)
    allfiles = allfiles[st]
    allfiles.stop = shift(allfiles.start,-1) - 1d
    allfiles[n-1].stop = allfiles[n-1].start + 27*86400d - 1d

end




;+
;FUNCTION: cdf_file_names
;PURPOSE:
;   Returns an array of filenames within a timerange.
;USAGE:
;   files=cdf_file_names(FORMAT,trange=trange,/verbose)
;INPUT:
;   FORMAT is a string that will be interpreted as one of two things:
;     CASE 1:  
;        e.g.    FORMAT = '/home/wind/dat/wi/3dp/k0/????/wi_k0_3dp*.cdf'
;        if FORMAT contains * or ? then filenames are returned that match that
;        pattern and for which YYYYMMDD falls within the specified timerange.
;        for example:  
;        (UNIX only)
;     CASE 2:
;        e.g.    FORMAT = 'fa_k0_ees_files'
;        The name of an indexfile that associates filenames with start and 
;        end times. If his file is not found, then the environment variable 
;        getenv('CDF_INDEX_DIR') is prepended and searched for.
;        See "make_cdf_index" for information on producing this file.
;     SPECIAL NOTE:
;        If strupcase(FORMAT) is the name of an environment varible. Then
;        the value of that environment variable is used instead.
;KEYWORDS:
;     TRANGE: 
;        Two element array specifying the time range for which data files should
;        be returned.  If not provided then "timerange" is called to provide
;        the time range.  See also "timespan".
;     NFILES:
;        Named variable that returns the number of files found.
;     VERBOSE:
;        Set to print some useful info.
;     FILEINFO:  OBSOLETE!
;        Set to a named variable that will return a table of file info.
;NOTES:
;     UNIX only!
;-
function cdf_file_names,format, pathname=pathname, use_master=use_master, $
  trange=trange,nfiles=n,fileinfo=fileinfo,reset=reset,verbose=verbose,$
  routine=routine
  
ts_ = systime(1)
tr=timerange(trange)

if keyword_set(fileinfo) then message,/info,'FILEINFO keyword is obsolete'

mfileformat = {mfileformat, name:'',start:0d,stop:0d,len:0l}
n=0


if getenv('FILE_ENV_SET') ne '1' then setfileenv
  
if size(/type,format) ne 7 then begin
    message,/info,'Input format must be a string'
    return,''
endif

if keyword_set(format) then begin
    envformat = getenv( strupcase(format) )
    if keyword_set(envformat) then begin
        if keyword_set(verbose) then $
           print,'Environment variable: ',strupcase(format),' Found.'
    endif
    if not keyword_set(envformat) then envformat = format
    
    if keyword_set(use_master) then envformat = format 
    
    if strpos(envformat,'*') ge 0 or strpos(envformat,'?') ge 0 then begin
       pathname=envformat
       indexfile = 0
    endif else begin
       if file_test(envformat,/reg,/read) then begin
          pathname = envformat
          indexfile = 1    
       endif else begin
          envformat2 = filepath(envformat,root=getenv('CDF_INDEX_DIR'))
          if file_test(envformat2,/reg,/read) then begin
             pathname = envformat2
             indexfile=1
          endif else begin
             if keyword_set(verbose) then print,'Using default path.'
             parts =  strsplit(envformat,/extr,'_')
             parts =  parts[[0,1,2]]
             pathname=getenv('BASE_DATA_DIR')+'/'+strjoin(parts,'/')+'/????/*.*'
             indexfile=0
;             message,/info,'Unable to determine PATHNAME for "'+format+'"'
;             return,''
          endelse
       endelse
    endelse
endif else begin
    message,/info,'FORMAT must be set'
    return,''
endelse

cdf_file_names_index,format,fileinfo,/get

if keyword_set(fileinfo) then begin
    reset_time = 3600.*2
    allfiles=fileinfo.allfiles
    if fileinfo.indexfile  ne indexfile  then allfiles=0
    if fileinfo.pathname   ne pathname   then allfiles=0 
    if indexfile then begin
       openr,lun,pathname,/get_lun
       stat=fstat(lun)
       free_lun,lun
;       print,'File: ',time_string(stat.mtime)
;       print,'info: ',time_string(fileinfo.timestamp)
       if stat.mtime gt fileinfo.timestamp then begin
          if keyword_set(verbose) then print,'Index file has been modified!'
          allfiles=0
       endif
    endif else begin
       if fileinfo.timestamp+reset_time lt ts_ then begin
          allfiles=0
          if keyword_set(verbose) then print,'Timer elapsed, checking for more files.'
       endif
    endelse
    if keyword_set(reset) then allfiles=0
endif


if not keyword_set(allfiles) then begin
      
  datepos = 16
  if indexfile then begin
    if keyword_set(verbose) then print,"Searching for files in: indexfile='",pathname,"'"
    on_ioerror,bad_file
    openr,lun,pathname,/get_lun
;printdat,fstat(lun)
    mf = mfileformat
    while not eof(lun) do begin
       s = ''
       readf,lun,s
       ss = strsplit(s,/extract)
       ns = n_elements(ss)

       mf.start = time_double(ss[0])
       mf.stop  = time_double(ss[1])
       mf.name  = ss[2]
       mf.len   = (ns gt 3) ? long(ss[3]) : 0
       append_array,allfiles,mf,index=n
    endwhile
    append_array,allfiles,index=n,/done
    free_lun,lun
  endif else begin
    if keyword_set(verbose) then print,"Searching for files in: pathname='",pathname,"'"
    allfnames=file_search(pathname,count=n)  ; This is the time consuming call.
    if n eq 0 then begin
       message,/info,'No files found in path: "'+pathname+'"'
       return,''
    endif
    allfiles = replicate(mfileformat,n)
    allfiles.name = allfnames
    
    if not keyword_set(routine) then begin
      if strpos(pathname,'bartel') ge 0 then routine='file_BARTEL_to_time' $
      else  routine='file_YYYYMMDD_to_time'
    endif
  
    call_procedure,routine,allfiles
    
  endelse
  
;search for and eliminate duplicate or earlier version files

  if n ge 2 then begin
    p = strpos(allfiles.name,'/',/reverse_search)
    fn1 = strmid(allfiles.name,transpose(p+1))
    st = sort(fn1)
    allfiles = allfiles[st]
    fn1=fn1[st]
    l = strlen(fn1)
    fn2 = strmid(fn1,0,transpose(l-datepos+8))
    dup = fn2 eq shift(fn2,-1)
    dup[n-1] = 0

    wdup = where(dup,ndup)
    if ndup ne 0 then begin
       if indexfile then print,'The indexfile ',pathname,' has the following duplicate entries:' $
       else print,'The following files are either duplicate or earlier version files and should be deleted:'
       print,allfiles[transpose(wdup)].name
    endif

    allfiles = allfiles[where(dup eq 0)]
    allfiles = allfiles[sort(allfiles.start)]
  endif
  fileinfo = {timestamp:ts_,pathname:pathname,indexfile:indexfile,allfiles:allfiles}
  if not keyword_set(allfiles) then fileinfo=0
endif 

cdf_file_names_index,format,fileinfo,/set

n = n_elements(allfiles) * keyword_set(allfiles)
if n eq 0 then begin
  message,/info,'No matching files found!'
  return,''
endif

w = where(allfiles.stop ge tr[0] and allfiles.start lt tr[1]  ,n)
if n ne 0 then files=allfiles[w].name else files=''

n2=0
if n ne 0 then w = where( file_test(files) ,n2)

if n2 gt 0 then files = files[w] else files=''
n=n2

if keyword_set(verbose) then $
   print,'CDF_FILE_NAMES: ',n,' of ',n_elements(allfiles),' Files found in ',systime(1)-ts_ ,' seconds'

return,files
bad_file:
  message,/info,'Unable to open index file: '+pathname
  return,''
end


;  FILENAMES:   string (array); full pathname of file(s) to be loaded.
;     (INDEXFILE, ENVIRONVAR, MASTERFILE and TIME_RANGE are ignored 
;     if this is set.)
;
;  MASTERFILE:  Full Pathname of indexfile or name of environment variable
;     giving path and filename information as defined in "get_file_names".
;     (INDEXFILE and ENVIRONVAR are ignored if this is set)
;
;  INDEXFILE:   File name (without path) of indexfile. This file
;     should be located in the directory given by ENVIRONVAR.  If not given
;     then "PICKFILE" is used to select an index file. see "make_cdf_index" for
;     information on producing this file.
;
;  ENVIRONVAR:  Name of environment variable containing directory of indexfiles
;     (default is 'CDF_INDEX_DIR')
;+
;PROCEDURE: loadallcdf
;USAGE:
;  loadallcdf,  FORMAT
;PURPOSE:
;  Loads selected CDF file variables into a data structure.
;  VARYing data is returned through the keyword: DATA.
;  NOVARY data is returned through the keyword:  NOVARDATA.
;INPUT:
;  FORMAT is a string (i.e. 'wi_k0_3dp_files') that specify the type of
;  files to be searched for.  see "cdf_file_names" for more info.
;KEYWORDS:  (all keywords are optional)
;  FILENAMES:   string (array); full pathname of file(s) to be loaded.
;     (INDEXFILE, ENVIRONVAR, MASTERFILE and TIME_RANGE are ignored 
;     if this is set.)
;  TIME_RANGE:  Two element vector specifying time range (default is to use
;     trange_full; see "TIMESPAN" or "TIMERANGE" for more info)
;
;  CDFNAMES:    Names of CDF variables to be loaded. (string array)
;  TAGNAMES:	String array of structure tag names.
;  DATA:        Named variable that data is returned in.
;  RESOLUTION:	Resolution in seconds to be returned.
;
;  NOVARNAMES:  Names of 'novary' variables to be loaded
;  NOVARDATA:   Named variable that 'novary' data is returned in.
;
;  TPLOT_NAME:  "TPLOT" string name. If set then a tplot variable is created.
;     Individual elements can be referred to as 'NAME.ELEMENT'
;  CARR_FILE:	Load Carrington rotation files.
;
;SEE ALSO:
;  "loadcdf","loadcdfstr","makecdf","make_cdf_index","get_file_names","
;VERSION:  02/04/19  loadallcdf.pro  1.27
;Created by Davin Larson,  August 1996
;-
pro loadallcdf,format, $
   indexfile=indexfile, $
   time_range=trange, $
   FILENAMES  = filenames, $
   MASTERFILE = mfile, $
;   pathname=pathname, $
;   fileinfo=fileinfo, $
;   environvar=environvar, $
   cdfnames=cdfnames, $
   tagnames=tagnames, $
   data=data, novardata=novardata, $
   resolution = res, $
   median = med, $
   filter_proc = filter_proc, $
   tplot_name=tplot_name, $
   novarnames=novarnames

verbose=1   

;if getenv('FILE_ENV_SET') ne '1' then setfileenv

if keyword_set(indexfile) then format = indexfile
if keyword_set(mfile)     then format = mfile

if keyword_set(format) then $
  filenames=cdf_file_names(format,pathname=pathname,trange=trange,  $
                  nfiles=ndays,verbose=verbose)
                  
ndays = n_elements(filenames) * keyword_set(filenames)


;if 0 and not keyword_set(filenames) then begin
;  if not keyword_set(mfile) then begin
;  	if not keyword_set(environvar) then $
;  		environvar = 'CDF_INDEX_DIR'
;  	dir = getenv(environvar)
;  	if not keyword_set(dir) then message,$
;  		'Environment variable '+environvar+$
;  		' is not defined!' ,/info
;  	if not keyword_set(indexfile) then mfile = pickfile(path=dir) $
;  	else mfile = filepath(indexfile,root_dir=dir)
;  endif
;  get_file_names,filenames,TIME_RANGE=trange,MASTERFILE=mfile,nfiles=ndays
;endif

if ndays eq 0 then begin
  data=0
  novardata=0
  print,'LOADALLCDF: No data files valid for given time range'
  return
endif

if keyword_set(pickcdfnames) then begin
   print_cdf_info,filenames(0)
   print,'Choose data quantities:'
   repeat begin
     s = ''
     read,s
     if keyword_set(s) then if keyword_set(cdfnames) then $
         cdfnames = [cdfnames,s] else cdfnames=s
   endrep until keyword_set(s) eq 0
endif

loadcdfstr,data,novardata  $
  ,file=filenames,varnames=cdfnames,tagna=tagnames  $
  ,novarnames=novarnames,/time,resolution=res,median=med,filter_proc=filter_proc

if keyword_set(tplot_name) then begin
  if data_type(tplot_name) ne 7 then begin
    message,/info,"Sorry!   Code change!"
    message,/info,"You must now supply a string name for the TPLOT_NAME keyword."
    tplot_name = 'foo'
    message,/info,"The default name is: "+tplot_name
  endif
  store_data,tplot_name,data=data
  message,/info,"The following variables can now be plotted using TPLOT:"
  print,transpose(tplot_name+'.'+tag_names(data))
endif


end



;+
;PROCEDURE:	loadcdfstr
;PURPOSE:	
;  loads data from specified cdf file into a structure.
;INPUT:		
;	x:		A named variable to return the structure in
;	novardata:	A named variable to return the non-varying
;			data.
;
;KEYWORDS:
;       FILENAMES:  [array of] CDF filename[s].  (or file id's)
;	PATH:	    CDF file path.
;       VARNAMES:   [array of] CDF variable name[s] to be loaded.
;	NOVARNAMES: [array of] CDF non-varying field names.
;       TAGNAMES:   optional array of structure tag names.
;	RESOLUTION: resolution to return in seconds.
;	APPEND:     if set, append data to the end of x.
;       TIME:     If set, will create tag TIME using the Epoch variable.
;SEE ALSO:
;  "loadcdf2", "loadallcdf", "print_cdf_info","make_cdf_index"
;
;CREATED BY:	Davin Larson 
;MODIFICATIONS:
;LAST MODIFICATION:	@(#)loadcdfstr.pro	1.22 02/04/17
;-

pro loadcdfstr,data0,novardata  $
   ,filenames=cdf_files $
   ,path=path  $
   ,varnames=cdf_vars,tagnames=tagnames $
   ,novarnames=novarnames $
   ,resolution = res $
   ,median = med $
   ,filter_proc = filter_proc $
   ,append=append,time=time

print, cdf_files
; if n_elements(cdf_files) eq 0 then cdf_files=pickfile(filter="*.cdf",path=path,get_path=path)

;on_ioerror,skip

if keyword_set(append) then index=n_elements(data0)

for num = 0,n_elements(cdf_files)-1 do begin

   cdf_file = cdf_files(num)
   id = 0
;   if not keyword_set(silent) then print,'Loading ',cdf_file
   if data_type(cdf_file) eq 7 then id = cdf_open(cdf_file) else id = cdf_file
   
   inq = cdf_inquire(id)    ;   inq = cdf_info(id)    *** /gm 4/22/2010

   if not keyword_set(cdf_vars) then begin     ;get only variables that vary
      for n=0,inq.nvars-1 do begin
         vinq=cdf_varinq(id,n)
         if vinq.recvar eq 'VARY' then append_array,cdf_vars,vinq.name
      endfor
      for n=0,inq.nzvars-1 do begin
         vinq=cdf_varinq(id,n,/zvar)
         if vinq.recvar eq 'VARY' then append_array,cdf_vars,vinq.name
      endfor
   endif

   nvars = n_elements(cdf_vars)
   if not keyword_set(tagnames) then tagnames = cdf_vars
   tagnames = strcompress(tagnames,/remove_all)

   if not keyword_set(data0) then append=0
   if not keyword_set(append)  then begin      ;define the data structure:
;      if keyword_set(time) then dat = {TIME:0.d}
      for n=0,nvars-1 do begin
          vinq = cdf_varinq(id,cdf_vars(n))
          if vinq.is_zvar then dim = vinq.dim else dim = inq.dim*vinq.dimvar
          w = where(dim,ndim)
          if ndim gt 0 then dim=dim(w) else dim=0
;print,cdf_vars(n),ndim,dim
          case vinq.datatype of
            'CDF_REAL8' :   value = !values.d_nan
            'CDF_DOUBLE':   value = !values.d_nan
            'CDF_REAL4' :   value = !values.f_nan
            'CDF_FLOAT' :   value = !values.f_nan
            'CDF_INT4'  :   value = 0l
            'CDF_INT2'  :   value = 0
            'CDF_INT1'  :   value = 0b
            'CDF_UINT1' :   value = 0b
            'CDF_CHAR'  :   value = 0b
            'CDF_UCHAR' :   value = 0b
            'CDF_EPOCH' :   value = !values.d_nan
            else        :   message ,'Invalid type,  please fix source...'
          endcase
          if ndim gt 0 then val = make_array(value=value,dim=dim)   $
          else val=value
          a = strpos(tagnames(n),'%')
          aa = strpos(tagnames(n),'*')
          if a ne -1 then begin
          	b = strlen(tagnames(n))
          	oldname = tagnames(n)
          	tagnames(n) = strmid(oldname,0,a)+'q'+strmid(oldname,a+1,b)
          endif
          if aa ne -1 then begin
           	b = strlen(tagnames(n))
          	oldname = tagnames(n)
          	tagnames(n) = strmid(oldname,0,aa)+'x'+strmid(oldname,aa+1,b)
          endif
         
          str_element,/add,dat,tagnames(n),val
      endfor
      if keyword_set(time) then begin
          w = where(tag_names(dat) eq 'TIME',c)
          if c eq 0 then str_element,/add,dat,'TIME',0.d
      endif
   endif else dat = data0(0)

   vinq = cdf_varinq(id,cdf_vars(0))
   !quiet = 1
   cdf_control,id,variable=cdf_vars(0),get_var_info=varinfo,zvar=vinq.is_zvar
   !quiet = 0
   nrecs = varinfo.maxrec+1
   data = replicate(dat,nrecs)

   del = 0

   if keyword_set(time) then begin
      if cdf_attexists(id,'DEPEND_0',cdf_vars(0),zvar=vinq.is_zvar) then $
      	cdf_attget,id,'DEPEND_0',cdf_vars(0),epochnum,zvar=vinq.is_zvar $
      	else begin
      		for thisvar=0,inq.nvars-1 do begin
      			vinq = cdf_varinq(id,thisvar)
      			if vinq.datatype eq 'CDF_EPOCH' then $
      				epochnum=vinq.name
      		endfor
      		if n_elements(epochnum) eq 0 then begin
      			for thisvar=0,inq.nzvars-1 do begin
      				vinq = cdf_varinq(id,thisvar,/zvar)
      				if vinq.datatype eq 'CDF_EPOCH' then $
      					epochnum=vinq.name
      			endfor
      		endif
      	endelse
      loadcdf2,id,epochnum,x
      epoch0 = 719528.d * 24.* 3600. * 1000.  ;Jan 1, 1970
      data.time = (x - epoch0)/1000.
   endif

   for n=0,nvars-1 do begin
       if cdf_attexists(id,'DEPEND_0',cdf_vars(n),zvar=vinq.is_zvar) then $
         cdf_attget,id,'DEPEND_0',cdf_vars(n),thisepoch,zvar=vinq.is_zvar $
         else thisepoch = epochnum
       if n eq 0 and not keyword_set(time) then epochnum = thisepoch
       if strpos(tagnames(n),'Epoch') ne -1 then thisepoch = tagnames(n)
       if thisepoch eq epochnum then begin
         loadcdf2,id,cdf_vars(n),x,/no_shift,nrecs=nrecs
         if cdf_attexists(id,'FILLVAL',cdf_vars(n),zvar=vinq.is_zvar) then $
         	begin
         	case data_type(x) of
         		4: nan = !values.f_nan
         		5: nan = !values.d_nan
         	else: nan = 0
         	endcase
         	if nan ne 0 then begin
         		cdf_attget,id,'FILLVAL',cdf_vars(n),fv,zvar=vinq.is_zvar
         		fvindx = where(x eq fv,fvcnt)
         		if fvcnt gt 0 then x(fvindx) = nan
         	endif
         endif
         str_element,/add,data,tagnames(n),x
       endif else begin
         print,'Variable '+cdf_vars(n)+' has different Epoch'
       endelse
   endfor

   if num eq 0 and keyword_set(novarnames) then begin
      novardata = 0
      novartags = strcompress(novarnames,/remove_all)
      for i=0,n_elements(novarnames)-1 do begin
         loadcdf2,id,novarnames(i),val
         str_element,/add,novardata,novartags(i),val
      endfor
   endif

   if data_type(cdf_file) eq 7 then cdf_close,id
   if keyword_set(filter_proc) then call_procedure,filter_proc,data
   if keyword_set(res) then data = average_str(data,res,/nan,median=med)
   append_array,data0,data,index=index
   append = 1
skip: 
   if id eq 0 then print,'Unable to open file: ',cdf_file
endfor

append_array,data0,index=index,/done

end





pro	l1_verification

;	Routine to calculate 1-hr averages of SIT level 1 Berkeley CDF files and put out a Kaleidagraph format file
;
;	Modification history
;		March 2008	initial version
;		14-Jul-2008	adjust paths for new directory structure /gm
;		22-Apr-2010	adjust paths for production folder use /gm
;		23-Apr-2010	modify to use l1_verification.txt file for automated runs /gm

timeCDS = { MJD: 0L , TIME: 0L }
	file_index = ['a','b']
; mjd_start=54255.   
 mjd_start=54174.   ;   3/15/07   ' day 74
; mjd_end=54517.     ; 2/21/08
 mjd_end=54184.
mjd=mjd_start

version_switch = date2mjd (2018, 9, 1)

;	flag for output files:  short_file = 1 get 4He #4 for SIT-A, O #2 for SIT-B only (else get all rates)
	short_file = 0   ; if = 0, get check files with all rates and counts

	data_folder='/data/joey/masongm1/Data/Production/STEREO/5_Berkeley_L1_checks/L1_prerelease_files/sit/'
	output_folder='/data/joey/masongm1/Data/Production/STEREO/5_Berkeley_L1_checks/L1_prerelease_check_files_2/'
	openr, lun_lastcheck, '/data/joey/masongm1/Data/Production/STEREO/0_batch_jobs/logs/last_L1_verification.txt', /get_lun


;stop

	times = fltarr(4,2)
		
	for i_sc = 0, 1 do begin
	if( i_sc eq 0 ) then spacecraft='A' else spacecraft='B'

	in_times=fltarr(4)
	readf,  lun_lastcheck, in_times   ; latest data yr, doy    last processed yr, doy
;	print, in_times
	for index = 0, 3 do times(index,i_sc) = in_times(index)
	
	mjd_start = date2mjd(times(2,i_sc), 1, times(3,i_sc))
	mjd_end = date2mjd(times(0,i_sc), 1, times(1,i_sc))
	
	if(mjd_start eq mjd_end) then goto, next_sc

;	mjd is date of next file to process
	mjd=mjd_start + 1

	outfile = output_folder + '/stereo_' + STRLOWCASE (spacecraft) + string(times(2,i_sc), times(3,i_sc), format="('_',i4,'_',i3.3,'_check.txt')")
;	print, "JM - Creating ", outfile
 
 		openw, lun_outfile, outfile, /get_lun
 		printf, lun_outfile, format='($," year, doy, mjd,mjd from epoch,number of minutes, epoch,")'
		
		if(short_file eq 1 ) then begin
	   		if (i_sc eq 0 ) then printf, lun_outfile,  format='($,"4He 137 keV/n check,")'
	   		if (i_sc eq 1 ) then printf, lun_outfile,  format='($,"O 137 keV/n check,")'		;  v02 files & new calibration 7/6/2011
		endif else begin
	 		for i=0,11 do printf, lun_outfile, i+1, format='($,"H ",i3,",")'
 			for i=0,11 do printf, lun_outfile, i+1, format='($,"counts H ",i3,",")'
  			for i=0,9 do printf, lun_outfile, i+1, format='($,"3He ",i3,",")'
  			for i=0,9 do printf, lun_outfile, i+1, format='($,"counts 3He ",i3,",")'
   			for i=0,15 do printf, lun_outfile, i+1, format='($,"4He ",i3,",")'
   			for i=0,15 do printf, lun_outfile, i+1, format='($,"counts 4He ",i3,",")'
			for i=0,16 do printf, lun_outfile, i+1, format='($,"C ",i3,",")'
    			for i=0,16 do printf, lun_outfile, i+1, format='($,"counts C ",i3,",")'
    			for i=0,15 do printf, lun_outfile, i+1, format='($,"O ",i3,",")'
     			for i=0,15 do printf, lun_outfile, i+1, format='($,"counts O ",i3,",")'
    			for i=0,15 do printf, lun_outfile, i+1, format='($,"NeS ",i3,",")' 		
    			for i=0,15 do printf, lun_outfile, i+1, format='($,"counts NeS ",i3,",")' 
       			for i=0,13 do printf, lun_outfile, i+1, format='($,"Fe ",i3,",")'
       			for i=0,13 do printf, lun_outfile, i+1, format='($,"counts Fe ",i3,",")'
       		 	for i=0,5 do printf, lun_outfile, i+1, format='($,"UH ",i3,",")'
       		 	for i=0,5 do printf, lun_outfile, i+1, format='($,"counts UH ",i3,",")'        	
        		printf, lun_outfile, format='($," time,")'
		endelse
;		write out a character to force a line feed
		printf,  lun_outfile,' ',format='(a1)'

  


while (mjd le mjd_end ) do begin


;filename='/users/masongm1/Programs/prerelease/STA_L1_SIT_20070327_V01.cdf'

date= mjd2modayrhrmn(mjd)

if (mjd ge version_switch) then file_version = '03' else file_version = '02'
; filename = '/users/masongm1/Programs/prerelease/STA_L1_SIT_' + string(date(2),date(0),date(1),format='(i4,2i02)') + '_V01.cdf'
 filename = data_folder + 'ST' + spacecraft + '_L1_SIT_' + string(date(2),date(0),date(1),format='(i4,2i02)') + '_V' + file_version + '.cdf'     ; changed to v02 July 2011 with new SIT efficiencies

;	see if this file exists -- if not go to next day           *************** 9-Jul-2010 add missing day skip /gm ************
	openr, lun_exist, filename, error = err, /get_lun
	if( err ne 0 ) then begin
		print, 'Did not find file: ', filename
		if( n_elements( lun_exist ) gt 0 ) then free_lun, lun_exist		; allow for first file to be missing 15-Oct-2015/gm
		goto, nextday
	endif
	free_lun, lun_exist	


;	print,filename

 loadallcdf, data=sitdata, file=filename
;	 print,tag_names(sitdata)


; print,n_elements(sitdata)

limit = n_elements(sitdata) - 1

;	help,sitdata,/st



		index = 0



		h_counts=fltarr(12)
		he3_counts=fltarr(10)
		he4_counts=fltarr(16)
		c_counts=fltarr(17)
		o_counts=fltarr(16)
		nes_counts=fltarr(16)
		fe_counts=fltarr(14)
		uh_counts=fltarr(6)
		
		
 		for ihr = 0, 23 do begin
; 		print,ihr,index

;	if(limit lt 1439 ) then print,i, index,limit
 			epoch_sum=sitdata(index).epoch
 			h_sum=sitdata(index).h_intensity
 			he3_sum=sitdata(index).he3_intensity
 			he4_sum=sitdata(index).he4_intensity
 			c_sum=sitdata(index).c_intensity
 			o_sum=sitdata(index).o_intensity
 			nes_sum=sitdata(index).nes_intensity
 			fe_sum=sitdata(index).fe_intensity
 			uh_sum=sitdata(index).uh_intensity
 			time_sum=sitdata(index).time
 			

 			for  j = 0,11 do if( sitdata(index).h_sigma(j) > 0 ) then  h_counts(j)=(sitdata(index).h_intensity(j)/sitdata(index).h_sigma(j))^2 else h_counts(j)=0.
 			for  j = 0,9 do if( sitdata(index).he3_sigma(j) > 0 ) then  he3_counts(j)=(sitdata(index).he3_intensity(j)/sitdata(index).he3_sigma(j))^2 else he3_counts(j)=0.
 			for  j = 0,15 do if( sitdata(index).he4_sigma(j) > 0 ) then  he4_counts(j)=(sitdata(index).he4_intensity(j)/sitdata(index).he4_sigma(j))^2 else he4_counts(j)=0.
 			for  j = 0,16 do if( sitdata(index).c_sigma(j) > 0 ) then  c_counts(j)=(sitdata(index).c_intensity(j)/sitdata(index).c_sigma(j))^2 else c_counts(j)=0.
  			for  j = 0,15 do if( sitdata(index).o_sigma(j) > 0 ) then  o_counts(j)=(sitdata(index).o_intensity(j)/sitdata(index).o_sigma(j))^2 else o_counts(j)=0.
   			for  j = 0,15 do if( sitdata(index).nes_sigma(j) > 0 ) then  nes_counts(j)=(sitdata(index).nes_intensity(j)/sitdata(index).nes_sigma(j))^2 else nes_counts(j)=0.
    			for  j = 0,13 do if( sitdata(index).fe_sigma(j) > 0 ) then  fe_counts(j)=(sitdata(index).fe_intensity(j)/sitdata(index).fe_sigma(j))^2 else fe_counts(j)=0.
    			for  j = 0,5 do if( sitdata(index).uh_sigma(j) > 0 ) then  uh_counts(j)=(sitdata(index).uh_intensity(j)/sitdata(index).uh_sigma(j))^2 else uh_counts(j)=0.
 		
 		minutes = 1.   ; we've already added in the first minute!
 		for  i = 1, 59 do begin
;			print,i,index,minutes

 				epoch_sum += sitdata(index).epoch
 				h_sum += sitdata(index).h_intensity
 				he3_sum += sitdata(index).he3_intensity
 				he4_sum += sitdata(index).he4_intensity
 				c_sum += sitdata(index).c_intensity
 				o_sum += sitdata(index).o_intensity
 				nes_sum += sitdata(index).nes_intensity
 				fe_sum += sitdata(index).fe_intensity
 				uh_sum += sitdata(index).uh_intensity
 				time_sum += sitdata(index).time
; 				print,time_sum, minutes, sitdata(index).time
 			
 				for  j = 0,11 do if( sitdata(index).h_sigma(j) > 0 ) then  h_counts(j) += (sitdata(index).h_intensity(j)/sitdata(index).h_sigma(j))^2
 				for  j = 0,9 do if( sitdata(index).he3_sigma(j) > 0 ) then  he3_counts(j) += (sitdata(index).he3_intensity(j)/sitdata(index).he3_sigma(j))^2
 				for  j = 0,15 do if( sitdata(index).he4_sigma(j) > 0 ) then  he4_counts(j) += (sitdata(index).he4_intensity(j)/sitdata(index).he4_sigma(j))^2
 				for  j = 0,16 do if( sitdata(index).c_sigma(j) > 0 ) then  c_counts(j) += (sitdata(index).c_intensity(j)/sitdata(index).c_sigma(j))^2
  				for  j = 0,15 do if( sitdata(index).o_sigma(j) > 0 ) then  o_counts(j) += (sitdata(index).o_intensity(j)/sitdata(index).o_sigma(j))^2
   				for  j = 0,15 do if( sitdata(index).nes_sigma(j) > 0 ) then  nes_counts(j) += (sitdata(index).nes_intensity(j)/sitdata(index).nes_sigma(j))^2
    				for  j = 0,13 do if( sitdata(index).fe_sigma(j) > 0 ) then  fe_counts(j) += (sitdata(index).fe_intensity(j)/sitdata(index).fe_sigma(j))^2
    				for  j = 0,5 do if( sitdata(index).uh_sigma(j) > 0 ) then  uh_counts(j) += (sitdata(index).uh_intensity(j)/sitdata(index).uh_sigma(j))^2
			
			index ++
			if(index gt limit) then goto, output
			minutes ++ 


 		endfor
; divide by minutes

output  :			if( minutes ge  0. ) then begin
				epoch_sum /= minutes 
 				h_sum /= minutes
 				he3_sum /= minutes 
 				he4_sum /= minutes 
 				c_sum /= minutes 
 				o_sum /= minutes 
 				nes_sum /= minutes 
 				fe_sum /= minutes 
 				uh_sum /= minutes 

 				time_sum /= minutes 
; 				 				print,time_sum, minutes
			ENDIF


; print out the line 			
 			decmjd = mjd*1.d0 +(index-30.)/1440.d0
 			year = date(2)
 			timecds.mjd = mjd
 			timecds.time = (index-30.)*60000.
 			doy = mjd2decdoy(timecds)
 			
 
 		if(short_file eq 1 ) then begin
	   		if (i_sc eq 0 ) then printf,lun_outfile, year,doy, decmjd, unix_mjd(time_sum), minutes, epoch_sum,he4_sum(3), $
	   			format = '(1x,2(e12.4,","), 2(e20.9,","), 250(e12.4,","))' 
	   		if (i_sc eq 1 ) then printf,lun_outfile, year,doy, decmjd, unix_mjd(time_sum), minutes, epoch_sum,o_sum(3), $    ; switched from o_sum(2) to o_sum(3) 7/6/2011
	   			format = '(1x,2(e12.4,","), 2(e20.9,","), 250(e12.4,","))' 
		endif else begin 
 				printf,lun_outfile, year,doy, decmjd, unix_mjd(time_sum), minutes, epoch_sum,h_sum,h_counts,he3_sum,he3_counts, $
 			 	he4_sum,he4_counts,c_sum,c_counts,o_sum,o_counts,nes_sum,nes_counts,fe_sum,fe_counts,time_sum,$
 			 	uh_sum,uh_counts, format = '(1x,2(e12.4,","), 2(e20.9,","), 250(e12.4,","))' 
		endelse

; 			 stop
			if(index gt limit) then goto, nextday
 			 
 			 ENDFOR

nextday:  		mjd ++

 
 ENDWHILE ; -- mjd loop
 
 next_sc:	; break if no new data
 
 endfor    ; i_sc loop termination
 
 ;	write out last_l1_verification.txt file
 ;	read in the last line of the old file first
 	last_line = ''
 	readf,  lun_lastcheck, last_line
 	free_lun, lun_lastcheck
 
; 	update the last times processed:  
 	
 	for i_sc = 0, 1 do begin
 		times(2,i_sc) = times(0,i_sc)
 		times(3,i_sc) = times(1,i_sc)
 	endfor	
 ;	now write out the new file of last data check
	openw, lun_lastcheck, '/data/joey/masongm1/Data/Production/STEREO/0_batch_jobs/logs/last_L1_verification.txt', /get_lun
	for i_sc = 0, 1 do printf, lun_lastcheck, times(*,i_sc), file_index(i_sc), format="(4i6, 5x, 'SIT-',a,'   latest data (yr,doy);  latest L1 check (yr,doy)')"
  	printf,  lun_lastcheck, last_line
;	for i_sc = 0, 1 do print, times(*,i_sc), spacecraft, format="(4i6, 5x, 'SIT-',a,'   latest data (yr,doy);  latest L1 check (yr,doy)')"
close,/all

 end
 
