PRO SEP_BEACON, PACKET, SEPBEACONDATA
;+
; Project     :	STEREO - IMPACT, SEP
;
; Name        : SEP_BEACON
;
; Purpose     :	Convert SEP beacon data to physical units (apply conversion factors)
;
; Category    :	STEREO, IMPACT, SEP
;
; Explanation :	
;
; Syntax      :	SEPBEACON, PACKET, SEPBEACONDATA
;
; Examples    :	
;
; Inputs      :	PACKET - a IMPACT beacon telemetry packet (apid=624)
;
; Opt. Inputs :	None.
;
; Outputs     :	SEPBEACONDATA = A structure containing SEP Beacon data
;               if data is valid, undefined otherwise
;
; Opt. Outputs:	None.
;
; Keywords    :	None
;
; Calls       :	delvarx
;
; Common      : sep_b
;
; Restrictions:	
;
; Side effects:	None.
;
; Prev. Hist. :	None.
;
; History     :	Version 1 (preliminary), 26-Sep-2005, Andrew Davis, Caltech.
;               Version 2 (preliminary), 09-Nov-2005, Andrew Davis, Caltech.
; 			fix error where fluxes were longs instead of doubles
;                       Rename three LET status items
;               Version 3, 02/15/2006, Andrew Davis, Caltech.
;			implement proper HET/SIT rate decompression. OK
;                       implement paths to input and temporary files. OK
;                       read in different factors file depending on spacecraft. OK
;                       decode HET livetime properly, use HET status bits. NOT OK
;               Version 4, 02/21/2006, Andrew Davis, Caltech.
;                       Add code to ensure platform-independence.
;               Version 5, 03/26/2006, Andrew Davis, Caltech.
;                       improve code to ensure platform-independence.
;               Version 6, 04/26/2006, Andrew Davis, Caltech.
;                       New factors files.
;                       Implemented load_struct per Bill Thompson to do away with temporary files.
;                       Reorganized sepbeacondata data structure
;
; Contact     :	ad@srl.caltech.edu
;-
;

; see sep_beacon_common.pro
common sep_b

good_data = 1


rawsepb = BYTARR(SEPBDATASIZE)

; structure to hold Raw SEP Beacon data
bstruct = { $
	HETRATES:	UINTARR(NHET), $    ; 16-bit compressed rates
	HETLiveTime:	0U, $
	HETStopEffic:	0U, $
	HETPenEffic:	0U, $
	HETStatus:	0U, $
	SITRATES:	UINTARR(NSIT), $
	LETRATES:	UINTARR(NLET), $
	LETLiveTime:	0U, $
	LETTrigCnts:	0U, $
	LETHazCnts:	0U, $
	LETAcceptCnts:	0U, $
	LETStatus:	0B, $
	LETMinute:	0B, $
	SEPTRATES:	UINTARR(NSEPT),  $
	SEPTStatus:	0UL, $
	SEPUnused:	0U  $
}       

; decompressed rates
decom_rates = { $
	HETRATES:	LONARR(NHET), $
	SITRATES:	LONARR(NSIT), $
	LETRATES:	LONARR(NLET), $
	SEPTRATES:	LONARR(NSEPT)  $
}

timeCDS = { MJD: 0L , TIME: 0L }

; header for SEP Beacon data structure
header = { $
	SWversion: 0B, $	; Beacon processing software version (this software)
	timeCDS: timeCDS, $	; Time at start of data collection interval: MJD, millisec_of_day
	SCid: 0U $		; Spacecraft ID
}

status = { $
	HETLiveTime:	0.0D, $   ; HET instrument livetime (out of 60 seconds)
	HETStopEffic:	0U, $     ; No info yet, standby...
	HETPenEffic:	0U, $     ; No info yet, standby...
	HETStatus:	0U, $     ; No info yet, standby...
	LETLiveTime: 0.0D, $      ; LET instrument livetime (out of 60 seconds)
	LETTrigCnts:	0L, $     ; LET Number of front-end electronics triggers
	LETHazCnts:	0L, $     ; LET Number of events w/hazard flag
	LETAcceptCnts:	0L, $     ; LET Number of accepted events
	LETCodeOK: 0B, $          ; internal code check, 1=good, 0=bad
	LETHeaterDutyCycle: 0B, $ ; heater duty cycle: 0 - 10 
	LETLeakConv: 0B, $        ; leakage current balancing, 1=OK, 0=Oops
	LETDyThState: 0B, $       ; Dynamic thresholding state; 0=normal.
                                  ; 1, 2, 3, indicate higher threshold settings
                                  ; in response to higher count-rates
	LETMinute: 0B, $          ; LET's minute counter. Rolls over at 60
	SEPTStatus:	0UL  $
}

HETflux = { $
	Electrons:	0.0D, $
	Protons1:	0.0D, $
	Protons2:	0.0D, $
	Protons3:	0.0D, $
	He1:		0.0D, $
	He2:		0.0D, $
	He3:		0.0D, $
	CNO1:		0.0D, $
	CNO2:		0.0D, $
	Fe:		0.0D $
}
SITflux = { $
	He1:		0.0D, $
	He2:		0.0D, $
	He3:		0.0D, $
	He4:		0.0D, $
	CNO1:		0.0D, $
	CNO2:		0.0D, $
	CNO3:		0.0D, $
	CNO4:		0.0D, $
	Fe1:		0.0D, $
	Fe2:		0.0D, $
	Fe3:		0.0D, $
	Fe4:		0.0D $
}
LETflux = { $
	Protons1:	0.0D, $
	Protons2A:	0.0D, $
	Protons2B:	0.0D, $
	Protons3:	0.0D, $
	He4_1:		0.0D, $
	He4_2A:		0.0D, $
	He4_2B:		0.0D, $
	He4_3A:		0.0D, $
	He4_3B:		0.0D, $
	He3_1:		0.0D, $
	He3_2:		0.0D, $
	CNO1:		0.0D, $
	CNO2:		0.0D, $
	CNO3:		0.0D, $
	Fe1:		0.0D, $
	Fe2:		0.0D, $
	Fe3:		0.0D, $
	Fe4:		0.0D $
}
SEPTflux = { $
	Electrons0N_1:	0.0D, $
	Electrons2N_1:	0.0D, $
	Electrons0E_1:	0.0D, $
	Electrons2E_1:	0.0D, $
	ElectronsSum_2:	0.0D, $
	ElectronsSum_3:	0.0D, $
	Electrons0N_4:	0.0D, $
	Electrons2N_4:	0.0D, $
	Electrons0E_4:	0.0D, $
	Electrons2E_4:	0.0D, $
	Ions1N_1:	0.0D, $
	Ions3N_1:	0.0D, $
	Ions1E_1:	0.0D, $
	Ions3E_1:	0.0D, $
	IonsSum_2:	0.0D, $
	IonsSum_3:	0.0D, $
	Ions1N_4:	0.0D, $
	Ions3N_4:	0.0D, $
	Ions1E_4:	0.0D, $
	Ions3E_4:	0.0D $
}

HETcounts = { $
	Electrons:	0L, $
	Protons1:	0L, $
	Protons2:	0L, $
	Protons3:	0L, $
	He1:		0L, $
	He2:		0L, $
	He3:		0L, $
	CNO1:	0L, $
	CNO2:	0L, $
	Fe:		0L $
}
SITcounts = { $
	He1:		0L, $
	He2:		0L, $
	He3:		0L, $
	He4:		0L, $
	CNO1:	0L, $
	CNO2:	0L, $
	CNO3:	0L, $
	CNO4:	0L, $
	Fe1:		0L, $
	Fe2:		0L, $
	Fe3:		0L, $
	Fe4:		0L $
}
LETcounts = { $
	Protons1:	0L, $
	Protons2A:	0L, $
	Protons2B:	0L, $
	Protons3:	0L, $
	He4_1:	0L, $
	He4_2A:	0L, $
	He4_2B:	0L, $
	He4_3A:	0L, $
	He4_3B:	0L, $
	He3_1:	0L, $
	He3_2:	0L, $
	CNO1:	0L, $
	CNO2:	0L, $
	CNO3:	0L, $
	Fe1:		0L, $
	Fe2:		0L, $
	Fe3:		0L, $
	Fe4:		0L $
}
SEPTcounts = { $
	Electrons0N_1:	0L, $
	Electrons2N_1:	0L, $
	Electrons0E_1:	0L, $
	Electrons2E_1:	0L, $
	ElectronsSum_2:	0L, $
	ElectronsSum_3:	0L, $
	Electrons0N_4:	0L, $
	Electrons2N_4:	0L, $
	Electrons0E_4:	0L, $
	Electrons2E_4:	0L, $
	Ions1N_1:	0L, $
	Ions3N_1:	0L, $
	Ions1E_1:	0L, $
	Ions3E_1:	0L, $
	IonsSum_2:	0L, $
	IonsSum_3:	0L, $
	Ions1N_4:	0L, $
	Ions3N_4:	0L, $
	Ions1E_4:	0L, $
	Ions3E_4:	0L $
}


      
         ; Note: IMPACT packet checksum could be verified here
      
	 rawsepb = packet.dat[SEPBDATAFSET:SEPBDATAFSET + SEPBDATASIZE -1]

         ; load the raw data bytes into SEP beacon structure
	 load_struct, rawsepb, bstruct[0], bstruct, /noieee, /noconv

      ;  swap bytes in LET and SEPT items if we are running on a little-endian architecture
	 bstruct.LETRATES = swap_endian(bstruct.LETRATES, /SWAP_IF_LITTLE_ENDIAN)
	 bstruct.SEPTRATES = swap_endian(bstruct.SEPTRATES, /SWAP_IF_LITTLE_ENDIAN)
	 bstruct.LETLiveTime = swap_endian(bstruct.LETLiveTime, /SWAP_IF_LITTLE_ENDIAN)
	 bstruct.LETTrigCnts = swap_endian(bstruct.LETTrigCnts, /SWAP_IF_LITTLE_ENDIAN)
	 bstruct.LETHazCnts = swap_endian(bstruct.LETHazCnts, /SWAP_IF_LITTLE_ENDIAN)
	 bstruct.LETAcceptCnts = swap_endian(bstruct.LETAcceptCnts, /SWAP_IF_LITTLE_ENDIAN)
	 bstruct.SEPTStatus = swap_endian(bstruct.SEPTStatus, /SWAP_IF_LITTLE_ENDIAN)
	 bstruct.SEPUnused = swap_endian(bstruct.SEPUnused, /SWAP_IF_LITTLE_ENDIAN)

      ;  swap bytes in HET and SIT items if we are big-endian
	 bstruct.HETRATES = swap_endian(bstruct.HETRATES, /SWAP_IF_BIG_ENDIAN)
	 bstruct.SITRATES = swap_endian(bstruct.SITRATES, /SWAP_IF_BIG_ENDIAN)
	 bstruct.HETLIVETIME = swap_endian(bstruct.HETLIVETIME, /SWAP_IF_BIG_ENDIAN)
	 bstruct.HETStopEffic = swap_endian(bstruct.HETStopEffic, /SWAP_IF_BIG_ENDIAN)
	 bstruct.HETPenEffic = swap_endian(bstruct.HETPenEffic, /SWAP_IF_BIG_ENDIAN)
	 bstruct.HETStatus = swap_endian(bstruct.HETStatus, /SWAP_IF_BIG_ENDIAN)


	     
	 decom_rates.HETRATES = decompressHetSitCounters(bstruct.HETRATES)
         decom_rates.SITRATES = decompressHetSITCounters(bstruct.SITRATES)
         decom_rates.LETRATES = decompressLetCounters(bstruct.LETRATES)
         decom_rates.SEPTRATES = decompressLetCounters(bstruct.SEPTRATES)
      											   
      ;   printf,-2,decom_rates.HETRATES
      
      ;  build beacon data header
         header.SWversion = VERSION

      ;  Subtract an offset to get data-collection time
      ; Use the /NOCORRECT option here, since STEREO spacecraft time already has
      ; leap seconds factored in. i.e. spacecraft time is UTC seconds since 1958
         header.timeCDS = tai2utc(double(packet.pkt.seconds) - DCFSET, /NOCORRECT)
         header.SCid = packet.grh.SPACECRAFT_ID

      ;  read calibration factors in every time, in case they change, or in case the SCid changes
         IF (header.SCid eq 235) then BEGIN
            factors_filenm = concat_dir('$IMPACT_CALIB_DATA' , 'SEP_Beacon_factors_behind.txt')
      ;	    printf, -2, factors_filenm, header.SCid
            read_sep_beacon_factors, factors_filenm
         ENDIF ELSE BEGIN
            factors_filenm = concat_dir('$IMPACT_CALIB_DATA' , 'SEP_Beacon_factors_ahead.txt')
      ;	    printf, -2, factors_filenm, header.SCid
            read_sep_beacon_factors, factors_filenm
            IF (header.SCid ne 234) then BEGIN
               printf, -2, 'sep_beacon: Warning - Bad SCid:', header.SCid, '   Using Ahead factors file'
	    ENDIF
	 ENDELSE
      ;  printf,-2,pkt_times[0], header.timeCDS
      ;  printf,-2,tai2utc(double(pkt_times[0]), /CCSDS, /NOCORRECT), $
      ;   	header.SWversion ,header.SCid, header.SciFrmVersion

      ;  build beacon status

; Need to make use of HET status info
      status.HETLiveTime = 60.0D * HETLIVETIMESCALEFACTOR * $
      		decompressHetSitCounters(bstruct.HETLiveTime)
      status.HETStopEffic = bstruct.HETStopEffic
      status.HETPenEffic = bstruct.HETPenEffic
      status.HETStatus = bstruct.HETStatus

      status.LETLiveTime = bstruct.LETLiveTime * LETLIVETIMESCALEFACTOR * 60.0D
      status.LETTrigCnts = decompressLetCounters(bstruct.LETTrigCnts)
      status.LETHazCnts = decompressLetCounters(bstruct.LETHazCnts)
      status.LETAcceptCnts = decompressLetCounters(bstruct.LETAcceptCnts)
      status.LETCodeOK = ishft(bstruct.LETStatus and '80'X,-7)
      status.LETHeaterDutyCycle = ishft(bstruct.LETStatus and '78'X,-3)
      status.LETLeakConv = ishft(bstruct.LETStatus and '4'X,-2)
      status.LETDyThState = bstruct.LETStatus and '3'X
      status.LETMinute = bstruct.LETMinute
      status.SEPTStatus = bstruct.SEPTStatus

      ; PRINTF, -2, FORMAT='(%"SEPTStatus: %x   %x")', bstruct.SEPTStatus, status.SEPTStatus

      ; build beacon counts and fluxes
      divisor = 0.0D
      tmpflux = DINDGEN(40)
        ; HET
        FOR i=0,NHET-1 DO BEGIN
	    divisor = status.HETLiveTime * $
                       (Factors.HET_fctrs[i].Ehi-Factors.HET_fctrs[i].Elo) * $
	               Factors.HET_fctrs[i].geom * $
		       Factors.HET_fctrs[i].effic
            IF divisor gt 0.0 THEN $
               tmpflux[i] = decom_rates.HETRATES[i] / divisor $
            ELSE tmpflux[i] = FILL_D
        ENDFOR
	HETflux.Electrons = tmpflux[0]
	HETflux.Protons1 = tmpflux[1]
	HETflux.Protons2 = tmpflux[2]
	HETflux.Protons3 = tmpflux[3]
	HETflux.He1 = tmpflux[4]
	HETflux.He2 = tmpflux[5]
	HETflux.He3 = tmpflux[6]
	HETflux.CNO1 = tmpflux[7]
	HETflux.CNO2 = tmpflux[8]
	HETflux.Fe = tmpflux[9]
	HETcounts.Electrons = decom_rates.HETRATES[0]
	HETcounts.Protons1 = decom_rates.HETRATES[1]
	HETcounts.Protons2 = decom_rates.HETRATES[2]
	HETcounts.Protons3 = decom_rates.HETRATES[3]
	HETcounts.He1 = decom_rates.HETRATES[4]
	HETcounts.He2 = decom_rates.HETRATES[5]
	HETcounts.He3 = decom_rates.HETRATES[6]
	HETcounts.CNO1 = decom_rates.HETRATES[7]
	HETcounts.CNO2 = decom_rates.HETRATES[8]
	HETcounts.Fe = decom_rates.HETRATES[9]

        ; SIT
        FOR i=0,NSIT-1 DO BEGIN
	    divisor = SITACCTIME * $
                       (Factors.SIT_fctrs[i].Ehi-Factors.SIT_fctrs[i].Elo) * $
	               Factors.SIT_fctrs[i].geom * $
		       Factors.SIT_fctrs[i].effic
            IF divisor gt 0.0 THEN $
               tmpflux[i] = decom_rates.SITRATES[i] / divisor $
            ELSE tmpflux[i] = FILL_D
        ENDFOR
	SITflux.He1 = tmpflux[0]
	SITflux.He2 = tmpflux[1]
	SITflux.He3 = tmpflux[2]
	SITflux.He4 = tmpflux[3]
	SITflux.CNO1 = tmpflux[4]
	SITflux.CNO2 = tmpflux[5]
	SITflux.CNO3 = tmpflux[6]
	SITflux.CNO4 = tmpflux[7]
	SITflux.Fe1 = tmpflux[8]
	SITflux.Fe2 = tmpflux[9]
	SITflux.Fe3 = tmpflux[10]
	SITflux.Fe4 = tmpflux[11]
	SITcounts.He1 = decom_rates.SITRATES[0]
	SITcounts.He2 = decom_rates.SITRATES[1]
	SITcounts.He3 = decom_rates.SITRATES[2]
	SITcounts.He4 = decom_rates.SITRATES[3]
	SITcounts.CNO1 = decom_rates.SITRATES[4]
	SITcounts.CNO2 = decom_rates.SITRATES[5]
	SITcounts.CNO3 = decom_rates.SITRATES[6]
	SITcounts.CNO4 = decom_rates.SITRATES[7]
	SITcounts.Fe1 = decom_rates.SITRATES[8]
	SITcounts.Fe2 = decom_rates.SITRATES[9]
	SITcounts.Fe3 = decom_rates.SITRATES[10]
	SITcounts.Fe4 = decom_rates.SITRATES[11]
         
        ; LET
        FOR i=0,NLET-1 DO BEGIN
	    divisor = status.LETLiveTime * $
                       (Factors.LET_fctrs[i].Ehi-Factors.LET_fctrs[i].Elo) * $
	               Factors.LET_fctrs[i].geom[status.LETDyThState] * $
		       Factors.LET_fctrs[i].effic[status.LETDyThState]
            IF divisor gt 0.0 THEN $
               tmpflux[i] = decom_rates.LETRATES[i] / divisor $
            ELSE tmpflux[i] = FILL_D
        ENDFOR
	LETflux.Protons1 = tmpflux[0]
	LETflux.Protons2A = tmpflux[1]
	LETflux.Protons2B = tmpflux[2]
	LETflux.Protons3 = tmpflux[3]
	LETflux.He4_1 = tmpflux[4]
	LETflux.He4_2A = tmpflux[5]
	LETflux.He4_2B = tmpflux[6]
	LETflux.He4_3A = tmpflux[7]
	LETflux.He4_3B = tmpflux[8]
	LETflux.He3_1 = tmpflux[9]
	LETflux.He3_2 = tmpflux[10]
	LETflux.CNO1 = tmpflux[11]
	LETflux.CNO2 = tmpflux[12]
	LETflux.CNO3 = tmpflux[13]
	LETflux.Fe1 = tmpflux[14]
	LETflux.Fe2 = tmpflux[15]
	LETflux.Fe3 = tmpflux[16]
	LETflux.Fe4 = tmpflux[17]
	LETcounts.Protons1 = decom_rates.LETRATES[0]
	LETcounts.Protons2A = decom_rates.LETRATES[1]
	LETcounts.Protons2B = decom_rates.LETRATES[2]
	LETcounts.Protons3 = decom_rates.LETRATES[3]
	LETcounts.He4_1 = decom_rates.LETRATES[4]
	LETcounts.He4_2A = decom_rates.LETRATES[5]
	LETcounts.He4_2B = decom_rates.LETRATES[6]
	LETcounts.He4_3A = decom_rates.LETRATES[7]
	LETcounts.He4_3B = decom_rates.LETRATES[8]
	LETcounts.He3_1 = decom_rates.LETRATES[9]
	LETcounts.He3_2 = decom_rates.LETRATES[10]
	LETcounts.CNO1 = decom_rates.LETRATES[11]
	LETcounts.CNO2 = decom_rates.LETRATES[12]
	LETcounts.CNO3 = decom_rates.LETRATES[13]
	LETcounts.Fe1 = decom_rates.LETRATES[14]
	LETcounts.Fe2 = decom_rates.LETRATES[15]
	LETcounts.Fe3 = decom_rates.LETRATES[16]
	LETcounts.Fe4 = decom_rates.LETRATES[17]
      
        ; SEPT
        FOR i=0,NSEPT-1 DO BEGIN
	    oflow = ishft(status.SEPTStatus,-1 * i) and 1
	    divisor = SEPTACCTIME * $
                       (Factors.SEPT_fctrs[i].Ehi-Factors.SEPT_fctrs[i].Elo) * $
	               Factors.SEPT_fctrs[i].geom * $
		       Factors.SEPT_fctrs[i].effic
            IF ((divisor gt 0.0) and (oflow eq 0)) THEN $
               tmpflux[i] = decom_rates.SEPTRATES[i] / divisor $
            ELSE BEGIN
	       tmpflux[i] = FILL_D
	       decom_rates.SEPTRATES[i] = FILL_L
	    ENDELSE
        ENDFOR
	SEPTflux.Electrons0N_1 = tmpflux[0]
	SEPTflux.Electrons2N_1 = tmpflux[1]
	SEPTflux.Electrons0E_1 = tmpflux[2]
	SEPTflux.Electrons2E_1 = tmpflux[3]
	SEPTflux.ElectronsSum_2 = tmpflux[4]
	SEPTflux.ElectronsSum_3 = tmpflux[5]
	SEPTflux.Electrons0N_4 = tmpflux[6]
	SEPTflux.Electrons2N_4 = tmpflux[7]
	SEPTflux.Electrons0E_4 = tmpflux[8]
	SEPTflux.Electrons2E_4 = tmpflux[9]
	SEPTflux.Ions1N_1 = tmpflux[10]
	SEPTflux.Ions3N_1 = tmpflux[11]
	SEPTflux.Ions1E_1 = tmpflux[12]
	SEPTflux.Ions3E_1 = tmpflux[13]
	SEPTflux.IonsSum_2 = tmpflux[14]
	SEPTflux.IonsSum_3 = tmpflux[15]
	SEPTflux.Ions1N_4 = tmpflux[16]
	SEPTflux.Ions3N_4 = tmpflux[17]
	SEPTflux.Ions1E_4 = tmpflux[18]
	SEPTflux.Ions3E_4 = tmpflux[19]
	SEPTcounts.Electrons0N_1 = decom_rates.SEPTRATES[0]
	SEPTcounts.Electrons2N_1 = decom_rates.SEPTRATES[1]
	SEPTcounts.Electrons0E_1 = decom_rates.SEPTRATES[2]
	SEPTcounts.Electrons2E_1 = decom_rates.SEPTRATES[3]
	SEPTcounts.ElectronsSum_2 = decom_rates.SEPTRATES[4]
	SEPTcounts.ElectronsSum_3 = decom_rates.SEPTRATES[5]
	SEPTcounts.Electrons0N_4 = decom_rates.SEPTRATES[6]
	SEPTcounts.Electrons2N_4 = decom_rates.SEPTRATES[7]
	SEPTcounts.Electrons0E_4 = decom_rates.SEPTRATES[8]
	SEPTcounts.Electrons2E_4 = decom_rates.SEPTRATES[9]
	SEPTcounts.Ions1N_1 = decom_rates.SEPTRATES[10]
	SEPTcounts.Ions3N_1 = decom_rates.SEPTRATES[11]
	SEPTcounts.Ions1E_1 = decom_rates.SEPTRATES[12]
	SEPTcounts.Ions3E_1 = decom_rates.SEPTRATES[13]
	SEPTcounts.IonsSum_2 = decom_rates.SEPTRATES[14]
	SEPTcounts.IonsSum_3 = decom_rates.SEPTRATES[15]
	SEPTcounts.Ions1N_4 = decom_rates.SEPTRATES[16]
	SEPTcounts.Ions3N_4 = decom_rates.SEPTRATES[17]
	SEPTcounts.Ions1E_4 = decom_rates.SEPTRATES[18]
	SEPTcounts.Ions3E_4 = decom_rates.SEPTRATES[19]
         
	; assemble SEP Beacon data structure
	sepbeacondata = {header:header, status:status, HETflux:HETflux, HETcounts:HETcounts, $
      		LETflux:LETflux, LETcounts:LETcounts, SITflux:SITflux, SITcounts:SITcounts, $
		SEPTflux:SEPTflux, SEPTcounts:SEPTcounts}

IF (good_data eq 0) THEN DELVARX, sepbeacondata

END
