function Pmunu, x, mu, nu

;+
; PMUNU
;	Returns the associated legendre polynomial P^(mu)_(nu)(x)
;
; Usage:
;	p = pmunu(x,mu,nu)
;
; Return Value:
;	p	float	The associated legendre polynomial
;
; Arguments:
;	x	float	input	The values at which to compute p
;	mu	int	input	The order of the function.
;	nu	int	input	The degree of the function.
;
; Method:
;	Explicit analytic forms for degree 1-4.
;
; Restrictions:
;	Limited to degree 0..4
;
; History:
;	Original: 11/4/95; SJT
;-


if (mu gt nu or mu lt 0 or nu lt 0 or nu gt 4) then begin
    message, /inform, "Bad order or degree of associated legendre " + $
      "function"
    return, 0.
endif

t = acos(x)
y = sin(t)
case nu of
    0: begin
        p = x
        p(*) = 1.
    end
    
    1: case mu of
        0: p = x
        1: p = y
    endcase
    
    2: case mu of
        0: p = 1.5*x^2-.5
        1: p = 1.5*x*y
        2: p = 1.5*y^2
    endcase
    
    3: case mu of
        0: p = 2.5*x^3 - 1.5*x
        1: p = .5*y*(5*x^2-1)
        2: p = 2.5*y^2*x
        3: p = 2.5*y^3
    endcase
    
    4: case mu of
        0: p = (35*x^4-30*x^2+3)/8.
        1: p = 5*y*(7*x^3-3*x)/8.
        2: p = 5*y^2*(7*x^2-1)/8.
        3: p = 35*y^3*x/8.
        4: p = 35*y^4/8.
    endcase
endcase

return, p

end
