;+
; CW_TBUTTON
;	Compound widget for toggle (state cycling) buttons.
;
; Usage:
;	bid = cw_tbutton(base, < keys >)
;
; Return Value:
;	bid	long	The widget id of the button (actually a
;			containing base)
;			
; Argument:
;	base	long	input	The id of the base widget containing
;				the button.
;
; Keywords (all input): 
;	frame	?	If set, then draw a frame round the button.
;	font	string	The font to use for the button text.
;	group_leader	long	The ID of the group leader for the button.
;	uvalue	???	The UVALUE to associate with the compound widget.
;	value	string	List of values to associate with the button
;	state	int	Initial state of the button.
;
; History:
;	Original: 25/7/95; SJT
;-

function Cw_tb_getv, id, value

;	Extract the state structure and return the state setting.

basid = widget_info(id, /child)
widget_control, basid, get_uvalue = uvs

return, uvs.state

end

pro Cw_tb_setv, id, value

;	Extract the state structure

basid = widget_info(id, /child)
widget_control, basid, get_uvalue = uvs, /no_copy

;	Unmap all sub bases

for j = 0, n_elements(uvs.basids)-1 do  $
  widget_control, uvs.basids(j), map = 0

;	Reset state and map the relevant subbase.

uvs.state = value mod n_elements(uvs.bids)
widget_control, uvs.basids(uvs.state), map = 1

widget_control, basid, set_uvalue = uvs, /no_copy

end

function Cw_tb_evf, event

;	Extract the state structure.

basid = widget_info(event.handler, /child)
widget_control, basid, get_uvalue = uvs, /no_copy

; update the state and map the appropriate button.

widget_control, uvs.basids(uvs.state), map = 0
uvs.state = (uvs.state + 1) mod n_elements(uvs.bids)
widget_control, uvs.basids(uvs.state), map = 1

;	Pass on a WIDGET_BUTTON structure with the state in the SELECT
;	field.

rev = {widget_tbutton,  $
       Id:      event.handler,  $
       Top:     event.top,  $
       Handler: event.handler,  $
       Select:  uvs.state $
      }

widget_control, basid, set_uvalue = uvs, /no_copy

return, rev

end


function Cw_tbutton, parent, $
                     frame=frame, $
                     font=font, $
                     group_leader=group_leader, $
                     uvalue=uvalue, $
                     value=value, $
                     state=state

;	Define undefined keyword values.

if (n_params() eq 0) then parent = widget_base()
if (n_elements(state) eq 0) then state = 0l
if (n_elements(group_leader) eq 0) then group_leader = 0l
if (n_elements(frame) eq 0) then frame = 0
if (n_elements(font) eq 0) then font = ''

;	This is the base that will be seen by the outside user as the
;	CW id.

base = widget_base(parent, frame = frame)

;	One subbase and one button per possible state. Need the bases
;	because mapping buttons doesn't work properly.

locs = where(value ne '', n_states)
bid = lonarr(n_states)
basid = lonarr(n_states)

;	define the bases and buttons

for j = 0, n_states-1 do begin
    basid(locs(j)) = widget_base(base)
    bid(locs(j)) = widget_button(basid(locs(j)), $
                                 value = value(locs(j)), $
                                 font = font)
endfor

;	Structure to store the state of the compound. Just need the
;	IDs of the buttons and their sub-bases (strictly only need one
;	of these as WIDGET_INFO could get one from the other but it's
;	easier this way). Will be made into the UVALUE of the subbase
;	holding the first state.

uvs = { bids: bid, $
        Basids: basid, $
        State:  0l}

widget_control, basid(0), set_uvalue = uvs

;	Set attribute of the user-visible base.

widget_control, base, set_uvalue = uvalue, group_leader = group_leader, $
  event_func = 'cw_tb_evf', func_get_value = 'cw_tb_getv', $
  pro_set_value = 'cw_tb_setv'

;	Finally set the state of the compound widget.

widget_control, base, set_value = state

return, base

end
