pro Set_archive, prefix, suffix, ula=ula, uav=uav, cmp1d=cmp1d, $
                 cmp6h=cmp6h, uaf=uaf, cuaf=cuaf, $
                 user_defined=user_defined, resolution=resolution, $
                 duration=duration, second=seconds, minutes=minutes, $
                 hours=hours, days=days, sparse=sparse, $
                 compressed=compressed

;+
; SET_ARCHIVE
;	Select whether to plot from the raw ULA archive files or from the
;	averaged UAV files.
;
; Call:
;	set_archive[, prefix, suffix]
;
; Arguments:
;	prefix	string	input	prefix for a user-defined file type.
;	suffix	string	input	suffix for a user-defined file type.
;
; Keywords:
;	ula	If set and non-zero, use ula archives.
;	uav	If set and non-zero, use uav archives.
;	cmp1d	If set and non-zero, use 1-day compressed archives.
;	cmp6h	If set and non-zero, use 6-hour compressed archives
;	uaf	If set and non-zero, use uaf filtered archives
;	cuaf	If set and non-zero, use cuaf comressed filtered
;		archives.
;	user_defined	If set and non-zero, use a user-defined
;			archive.
;	resolution	float	input	Time resolution of the archive
;					(unit given by keyword).
;	duration	int	input	Number of days data per file.
;	seconds, minutes, hours, days	If set specify units for
;					resolution in the obvious way.
;	sparse	If set and non-zero, then a user-defined
;		archive has sparse records (like UAV and UAF)
;	compressed	If set and non-zero, then a user defined
;			archive has filled records (like ULA and CUAF).
;
; History:
;	Prototype: 22/3/91; SJT
;	Include selection of rate/exponents/fluxes: 25/3/91; SJT
;	Add composition ratios: 19/7/91; SJT
;	Add X-rays: 23/7/91; SJT
;	Add number of channels to overlay: 25/7/91; SJT
;	Unified Version: 4/9/91; SJT
;	Replot locking: 13/9/91; SJT
;	Add lock for PLOT_MAG, PLOT_PAD and PLOT_SPECT: 22/7/92; SJT
;	Add 1-Day and 6-hour archives: 26/10/92; SJT
;	Add UAF archives: 23/7/93; SJT
;	Add CUAV & CUAF archives: 22/8/94; SJT
;	Thorough rebuild: 21/10/94; SJT
;-

@settings
@flags
@b_flags
@spec_flags
@pad_flags
@file_info

if ((nitem = keyword_set(ula) + keyword_set(uav) + $
     keyword_set(cmp1d) + keyword_set(cmp6h) + keyword_set(uaf) + $
     keyword_set(cuaf) + keyword_set(user_defined))) gt 1 then begin
    print, '**** ERROR **** May only give one specification of the archive'
    return
endif

if (keyword_set(seconds) + keyword_set(minutes) + keyword_set(hours) + $
    keyword_set(days) gt 1) then begin
    print, '**** ERROR **** May only specify one time unit.'
    return
endif

if n_elements(archive) eq 0 then begin
    archive = 0
    arc_prefix = 'ula'
    arc_suffix = '.rat'
    arc_resolution = 0.005
    arc_duration = 1
    arc_comp = 1b
end

Get_arctype:
if (nitem eq 0) then begin
    
    sdef = (['USER', 'ULA', 'UAV', '1D', 'UAF', '6H', 'CUAF'])(archive+1)

    stype = ''
    read, 'Read from ULA, UAV, UAF, 1D, 6H, CUAF or USER-defined archive? <' $
      +sdef+'> :_', stype

    stype = strupcase(stype)

    if stype eq '' then stype = sdef $
    else  begin
	replot_ok = 0
	pad_replot_ok = 0
	spc_replot_ok = 0
	b_replot_ok = 0
    endelse
endif else begin
    replot_ok = 0
    pad_replot_ok = 0
    spc_replot_ok = 0
    b_replot_ok = 0
    if (n_params() eq 1) then stype = strupcase(archive) $
    else if keyword_set(ula)          then stype = 'ULA' $
    else if keyword_set(uav)          then stype = 'UAV' $
    else if keyword_set(cmp1d)        then stype = '1D' $
    else if keyword_set(cmp6h)        then stype = '6H' $
    else if keyword_set(uaf)          then stype = 'UAF' $
    else if keyword_set(cuaf)         then stype = 'CUAF' $
    else if keyword_set(user_defined) then stype = 'USER'
endelse

case stype of
    'ULA': begin
        archive = 0
        arc_prefix = 'ula'
        arc_suffix = '.rat'
        arc_resolution = 0.005
        arc_duration = 1
        arc_comp = 1b
    end
    
    'UAV': begin
        archive = 1
        arc_prefix = 'uav'
        arc_suffix = '.lan'
        arc_resolution = 1.0
        arc_duration = 32
        arc_comp = 0b
    end
    
    '1D': begin
        archive = 2
        arc_prefix = 'cmp1d'
        arc_suffix = '.lan'
        arc_resolution = 24.
        arc_duration = 366
        arc_comp = 1b
    end
    'UAF': begin
        archive = 3
        arc_prefix = 'uaf'
        arc_suffix = '.lan'
        arc_resolution = 1.0
        arc_duration = 32
        arc_comp = 0b
    end

    '6H': begin
        archive = 4
        arc_prefix = 'cmp6h'
        arc_suffix = '.lan'
        arc_resolution = 6.0
        arc_duration = 32
        arc_comp = 1b
    end

    'CUAF': begin
        archive = 5
        arc_prefix = 'cuaf'
        arc_suffix = '.lan'
        arc_resolution = 1.0
        arc_duration = 366
        arc_comp = 1b
    end
    
    'USER': begin
        archive = -1
        if (n_elements(prefix) eq 0) then begin
            prefix = ''
            read, 'Enter prefix for archive file <'+arc_prefix+'> :_', $
              prefix
            if (prefix ne '') then arc_prefix = prefix
        endif else arc_prefix = prefix
        if (n_elements(suffix) eq 0) then begin
            suffix = ''
            read, 'Enter suffix for archive file <'+arc_suffix+'> :_', $
              suffix
            if (suffix ne '') then arc_suffix = suffix
        endif else arc_suffix = suffix
        
        if (not keyword_set(resolution)) then begin
            sres = ''
            dres = strtrim(string(arc_resolution), 2)
            read, 'Enter resolution of archive in hours <'+dres+'> ' + $
              ':_', sres
            if (sres ne '') then arc_resolution = float(sres)
        endif else begin
            arc_resolution = resolution
            if (keyword_set(seconds)) then $
              arc_resolution = arc_resolution/3600. $
            else if (keyword_set(minutes)) then $
              arc_resolution = arc_resolution/60. $
            else if (keyword_set(days)) then $
              arc_resolution = arc_resolution*24.
        endelse
        
        if (not keyword_set(duration)) then begin
            sdur = ''
            ddur = strtrim(string(arc_duration))
            read, 'Enter number of days data per file <'+ddur+'> :_', $
              sdur
            if (sdur ne '') then arc_duration = fix(sdur)
        endif else arc_duration = duration
        
        if (keyword_set(compressed)) then arc_comp = 1b $
        else if (keyword_set(sparse)) then arc_comp = 0b $
        else begin
            scomp = ''
            dcomp = (['S', 'F'])(arc_comp)
            read, 'Is the archive sparse (S) [e.g. UAF] or filled (F) ' + $
              '[e.g. ULA] <'+dcomp+'> :_', scomp
            case (strupcase(scomp)) of
                'S': arc_comp = 0b
                'F': arc_comp = 1b
                Else:
            endcase
        endelse
    end
    
    Else: begin
	nitem = 0
	goto, get_arctype
    endif
endcase

return
end
