pro Plopen, device, filename, landscape=landscape, portrait=portrait, $
            window=window, colour=colour, background=background, tex=tex, $
            page=page, set=set, wnumber=wnumber, big_chars=big_chars, $
            little_chars=little_chars, auto_chars=auto_chars

;+
; PLOPEN
;       Selects plot device, and if 'tek' then waits for terminal
;       to be switched into TEK mode. 
;       Allows choice of background colour and for hardcopy devices orientation
;       and filename.
;
; Call:
;       plopen[, device, file]
;
; Argument:
;	device	string	input	The device to open, any IDL device or one of
;				the recognized tektronix special cases.
;	filename string	input	File for output of hard copy. Ignored for
;				screens.
;
; Keywords:
;	landscape	input	If set and non- zero, select landscape mode 
;				for suitable devices.
;	portrait	input	If set and non- zero, select portrait mode for 
;				suitable devices.
;	window	int	input	Two element array with dimensions for window.
;	colour		input	If set and non-zero, set colour options (PS
;				only), if set to zero, no colour tables.
;	background string in	B for black ground, W for white.
;	tex		input	If set and non-zero the PS output files
;				will be suitable for inclusion in TeX files.
;	page	float	input	2-element array with size of page for hardcopy
;				plots in centimetres.
;	set		input	If set and non-zero, then call PLOTSET
;				immediately.
;	wnumber		input	The index number of the window, default is
;				zero.
;	big_chars	input	If set, then always use standard
;				character size.
;	little_chars	input	If set, then always use small
;				character size
;
; Restrictions:
;	Landscape and portrait are mutually exclusive.
;	Keywords that are meaningless for the selected device are ignored.
;	big_chars and little_chars are mutually exclusive.
;
; Common Blocks:
;       dcs             Sets a suiatable character size to get at least 
;                       80 chars/line.
;       plot_reg        size is used in computing the above.
;
; History:
;       Made to procedure: 28/1/91; SJT
;       common block DCS to hand charsize around added: 1/2/91; SJT
;       Added common plot_reg, to set plot region: 5/2/91; SJT
;       Clean up backgound prompt: 22/2/91; SJT
;       Choose filename, suitable background default: 12/3/91; SJT
;       Support for various specialized TEK emulators added: 27/3/91; SJT
;       Support for sizing of window on windowing systems: 15/4/91; SJT
;       LJ250 specific option setting added (for V2.0.13): 3/5/91; SJT
;       Select hardware font in PostScript: 10/5/91; SJT
;       Remember orientation, fix portrait mode size in PS: 17/7/91; SJT
;       Major tidy up - code had turned into spaghetti: 17/7/91; SJT
;	For screen devices - read colour table to decide bg colour: 2/8/91; SJT
;	Unified version with arguments and keywords: 3/9/91; SJT
;	Actual plot device operations dissected out into PLOTSET: 9/10/91; SJT
;	TeX option added: 18/11/91; SJT
;	Add page size option for hardcopy devices: 31/12/91; SJT
;	Support for TEK output to a file: 7/1/92; SJT
;	Added SET  keyword: 6/2/92; SJT
;	Added WNUMBER keyword for multiple windows: 20/2/92; SJT
;	Modifications to cope better with monochrome X-terminals: 7/8/92; SJT
;	Add DEVNAME for convenience of Widget interface: 7/1/93; SJT
;	Force unique filenames in unix: 17/2/93; SJT
;	Separate page size and window size: 20/4/94; SJT
;	Flip page dimensions on portrait/landscape switch: 12/5/94;
;							   SJT
;	Add big & little _chars keys: 29/6/94; SJT
;	Add LN03 "Pseudo-device": 28/7/94;SJT
;	Remove background setting from colour hardcopy devices:
;	25/7/95; SJT
;-

@plot_reg
@dcs
@plot_settings
@directories

if (keyword_set(landscape) and keyword_set(portrait)) then begin
    hs_err_msg, $
      '**** ERROR **** It is meaningless to ask for both orientations!'
    return
endif

if (keyword_set(big_chars) + keyword_set(little_chars) + $
    keyword_set(auto_chars) gt 1) then begin
    hs_err_msg, "**** ERROR **** You may only request one character " + $
      "size option."
    return
endif

if      (n_params() gt 0) then          plotdev = device $
else begin
    if (n_elements(pldname) eq 0) then pldname = strlowcase(!d.name)

get_device_type:
    plotdev = ''
    read, 'Enter plot type <'+pldname+'> :_', plotdev
    plotdev = strlowcase(plotdev)

    if (plotdev eq '') then plotdev = pldname
endelse

devname = strupcase(plotdev)

case plotdev of
    't5': begin                         ; Cifer T5 terminal
        set_plot, 'tek'
        hard = 0
        coltab = 0
        wdev = 0
	devname = 'TEK'
    end

    '4125': begin                       ; TEK 4125 emulator on VWS
        set_plot, 'tek'
        hard = 0
        coltab = 1
        wdev = 0
	devname = 'TEK'
    end

    'pericom': begin                    ; Pericom terminal
        set_plot, 'tek'
        hard = 0
        coltab = 0
        wdev = 0
	devname = 'TEK'
    end

    'xtek': begin			; MIT X11 Xterm tek mode
        set_plot, 'tek'
        hard = 0
        coltab = 0
        wdev = 0
	devname = 'TEK'
    end

    'ln03': begin			; LN03 printer, tek mode
        set_plot, 'tek'
        hard = 1
        coltab = 0
        wdev = 0
	devname = 'TEK'
    end
 
    'mac':begin      ; Macintosh
          set_plot, 'mac'
           hard = 0
           coltab = 0
          wdev = 0
      devname = 'mac' 
        end

    'ps': begin                         ; PostScript hard copy output
        set_plot, 'ps'
	if (N_elements(colour) ne 0) then ps_col = fix(colour ne 0) $
	else begin	    
            if (n_elements(ps_col) eq 0) then ps_col = 0
            coldef = (['N','Y'])(ps_col)

  get_cd:
            cans = '' & read, 'Use colour PostScript? <'+coldef+'> :_', cans
            cans = strupcase(cans)
            if (cans eq '') then cans = coldef
            if      (cans eq 'Y') then ps_col = 1 $
            else if (cans eq 'N') then ps_col = 0 $
            else                  goto, get_cd
	endelse
        if (ps_col eq 1) then begin
            coltab = 1
        endif else begin
            coltab = 0
        endelse
        hard = 1
        wdev = 0
	if (keyword_set(tex)) then pstex = 1 $
	else			   pstex = 0
    end

    'lj': begin                             ; LJ250 inkjet printer
        set_plot, 'lj'
        hard = 1
        coltab = 0
        wdev = 0
    end

    'pcl': begin
	set_plot, 'pcl'
	hard = 1
	coltab = 0
	wdev = 0
    end

    else: begin                         ; Other device.                         
        set_plot, plotdev
        wdev = (!d.flags and 256) ne 0
        hard = (!d.flags and 512) ne 0
        coltab = (!d.table_size gt 2)
    end
endcase
if (n_elements(ps_col) eq 0) then ps_col=0

if (!d.name eq 'TEK') then begin		; TEK device to file
    if (n_params() eq 2) then plotfile = filename $
    else if (plotdev eq 'ln03') then begin
	plotfile = ''
	read, 'Name of file for output <'+mk_plname(plotdev)+'> :_', plotfile
	if plotfile eq '' then plotfile = mk_plname(plotdev)
    endif else		      plotfile = ''
endif

pldname = plotdev

if (wdev) then begin                   ; Windowing device.
    if keyword_set(window) then begin
	sx = fix(window(0))
	sy = fix(window(1))
	need_new_window = 1
    endif else begin
        if (n_elements(sx) eq 0) then begin
            sx = 640
            sy = 512
        endif
	print, 'Enter window size, if <cr> is entered for the X-size, then'
	print, 'the current window size will be used, (or the default window'
	print, 'size if no window has been created). To change the Y-size only'
	print, 'enter 0 for the X-size.'
	wx = '' & read, 'X dimension of window (pixels) <' + $
                    string(sx, format="(i4)") + '> :_', wx
	if (wx eq '') then goto, no_new_window
	if ((nsx = fix(wx))) ne 0 then begin
	    sx = nsx
	    need_new_window = 1
	endif
	wy = '' & read, 'Y dimension of window (pixels) <' + $
                    string(sy, format="(i4)") + '> :_', wy
	if (wy ne '') then if ((nsy = fix(wy))) ne 0 then begin
	    sy = nsy
	    need_new_window = 1
	endif
    endelse
no_new_window:
    if keyword_set(wnumber) then begin
	windex = wnumber
	need_new_window = 1
    endif else if (n_elements(windex) eq 0) then windex = 0
endif

if (hard and plotdev ne 'ln03') then begin ; Hard copy check direction
                                           ; etc.
    if (keyword_set(landscape)) then direct = 'L' $
    else if keyword_set(portrait) then direct = 'P' $
    else begin
	if n_elements(orient) eq 0 then orient = 0
	odef = (['L','P'])(orient)

getdir:
	direct = ''
	read, 'Landscape (L) or Portrait (P) mode <'+odef+'> :_', direct
	direct = strupcase(direct)
	if (direct eq '') then direct = odef
    endelse

    case direct of
        'L': orient = 0
        'P': orient = 1
        else: goto, getdir
    endcase

    if (keyword_set(page)) then begin
	px = float(page(0))
	py = float(page(1))
    endif else if (n_elements(px) eq 0) then begin
	px = ([23.5,17.78])(orient)
	py = ([17.78,23.5])(orient)
    endif else begin
        if (orient eq 0 and px lt py) or (orient eq 1 and py lt px) $
          then begin            ; change orientation around
            tmp = px
            px = py
            py = tmp
        endif 
    endelse
    
    if (n_params() eq 2) then plotfile = filename $
    else begin
	plotfile = ''
	read, 'Name of file for output <'+mk_plname(plotdev)+'> :_', plotfile
	if plotfile eq '' then plotfile = mk_plname(plotdev)
    endelse
endif

;       Set up a simple colour table (based on the PGPLOT default)

if (coltab and not hard) then begin
    if (keyword_set(background)) then ipr = 0 else ipr = 1
Bgr:
    if (ipr eq 0) then bg = strupcase(background) $
    else begin
        tvlct, /get, r, g, b, 0
        if (r(0) lt 127b) then defbg = 'B' else defbg = 'W'
    
        bg = '' & read, 'Black (B) or White (W) background <'+defbg+'> :_', bg
        bg = strupcase(bg)
        if bg eq '' then bg = defbg
    endelse

    ipr = 1
    if (bg ne 'B' and bg ne 'W') then goto, bgr
endif else if (n_elements(bg) eq 0) then bg = 'W'

if (keyword_set(big_chars)) then char_sizes = 1 $
else if (keyword_set(little_chars)) then char_sizes = -1 $
else if (keyword_set(auto_chars)) then char_sizes = 0 $
else if (n_elements(char_sizes) eq 0) then char_sizes = 0

;		As a safety net for interactive users, make sure that there
;		will be a directory into which to put the plot file if needed.

if (hard and n_elements(pldir) eq 0) then set_dir

if keyword_set(set) then plotset

end
