function Make_fname, year, day, oldname, pha=pha, trk=trk, $
                       psum=psum, mfss=mfss, cdfdata=cdfdata, $
                       cdfmag=cdfmag, uds=uds, $
                       backwards=backwards, file_time=file_time

;+
; MAKE_FNAME
;       Produce the filename of a LAN archive file given the year and the day.
;
; Call:
;       name = make_fname(year, day[, oldname])
;
; Return Value:
;       name    string  The name of the file for the given year and day.
;
; Arguments:
;       year    int     input   The year of the data
;       day     int     input   The day of the year.
;       oldname string  input   Optional filename not to be matched.
;
; Keywords:
;       uav_arc If set then prefix is "UAV" instead of "ULA"
;       pha     If set then preferred alternative type is ".PHA" not ".RAT"
;       trk     If set then prefix is "TRK" instead of "ULA"
;       psum    If set then prefix is "PHA" instead of "ULA"
;       mfss    If set then prefix is "MFS" instead of "ULA"
;       cmp1d   If set then prefix is "CMP1D"
;       cmp6h   If set then prefix is "CMP6H"
;       uaf     If set then prefix is "UAF"
;	cuaf	If set then prefix is "CUAF"
;	cdfdata	If set then prefix is "CDF", suffix ".DAT"
;	cdfmag	If set then prefix is "CDFMAG", suffix ".DAT"
;	uds	string	input	If set, then it's value is the stem
;				name of a UDS file.
;       backwards       If set, then look backwards then forwards for a
;                       filename, else just look forwards.
;	file_time int	input	Override default file length (mainly
;				for UDS).
;
; History:
;       Prototype: 29/1/91; SJT
;       Changed to a full LAN archive name: 22/2/91; SJT
;       Accept file type, to accomodate split archives: 12/3/91; SJT
;       Accept archive type to allow ULA or UAV rates to be handled:
;                                                        22/3/91; SJT
;       Try both .lan and .rat for file type in ULA case: 19/4/91; SJT
;       Unified version: 28/8/91; SJT
;       DO directory setting in here: 3/9/91; SJT
;       Add optional check that filename is not the same as previous (passed
;               to procedure as a 3rd argument) 18/9/91; SJT
;       Prepend directory name in UNIX: 8/10/91; SJT
;       Add TRK support: 25/11/91; SJT
;       Truncate version numbers: 16/1/92; SJT
;       Add QULA support: 20/1/92; SJT
;       Remove cd (replace with form_fname): 4/2/92; SJT
;       Add support of PHA and MFS files from PHAGEN: 23/6/92; SJT
;       Add support for compressed 1d & 6h archives: 26/10/92; SJT
;       Add forward search if search fails backwards: 28/10/92; SJT
;       Change backscan length to 3 for TRK: 1/2/93; SJT
;       Add decompression of compressed file in unix: 12/3/93; SJT
;       Reduced calls to findfile: 2/7/93; SJT
;       Add UAF files: 23/7/93; SJT
;       Add backwards keyword: 6/8/93; SJT
;	Add support for GNU zipped files: 30/9/93
;	Remember filenames from one invokation to the next: 6/10/93;
;	SJT
;	Add CUA[VF] files: 22/8/94; SJT
;	Correct name list when file is decompressed: 4/10/94; SJT
;	Spin off some of the messy bits to subprograms also make I and
;	Q files automatic: 21/10/94; SJT
;	Incorporate CDF: 3/11/94; SJT
;	Extend to UDS: 3/11/94; SJT
;	Add file_time key (for SWOOPS uds): 8/11/94; SJT
;	Extra safety lock on forward search & old name match: 27/6/95;
;	SJT
;	Fix for dates beyond 2000: 1/12/95; SJT
;	Decompression now removed since HSIO now supports dynamic
;	decompression: 4/12/97; SJT
;-

@directories
@settings
@time_limits
@wid_set
@make_fname_cmn
@file_info

if (!version.os eq 'vms') then	direct = strlowcase(dir) $
else				direct = dir

iy = date100(year, /print)

extn = '.lan'
if (keyword_set(trk)) then begin
    arcname = 'trk'
    scan = 32
endif else if (keyword_set(psum)) then begin
    arcname = 'pha'
    scan = 32
endif else if (keyword_set(mfss)) then begin
    arcname = 'mfs'
    scan = 32
endif else if (keyword_set(pha)) then begin
    arcname = 'ula'
    scan = 2
    extn = '.pha'
endif else if (keyword_set(cdfdata)) then begin
    arcname = 'cdf'
    extn = '.dat'
    scan = 33
endif else if (keyword_set(cdfmag)) then begin
    arcname = 'cdfmag'
    extn = '.dat'
    scan = 33
endif else if (keyword_set(uds)) then begin
    temp = str_sep(uds, '+')
    arcname = temp(0)
    extn = temp(1)
    scan = 7                    ; Some are 7 day some are 1
endif else begin
    arcname = arc_prefix
    extn = arc_suffix
    scan = arc_duration+1
endelse

if (keyword_set(file_time)) then scan = file_time

;       Find the files likely to be appropriate to the current opening

if (n_elements(files) eq 0) then files_needed = 1b $
else files_needed = ((direct ne old_dir) or (arcname ne old_arc) $
                     or (extn ne old_extn))
first_pass = 1b

if (n_params() ne 3) then oldname = ''

Search_pass:

if (files_needed) then begin
    old_dir = direct
    old_extn = extn
    old_arc = arcname
    
    tmpl = arcname+'*'+extn
    find_matches, tmpl
    fnames_ok = 1
endif

;       Firstly, scan backwards if backwards key is set

if (keyword_set(backwards)) then begin
    for dc = day, day-scan, -1  do begin
        da = dc
        iya = iy
        while (da le 0) do begin
            iya = iy-1
            da = (365 + ((iya mod 4) eq 0)) + dc
        endwhile
        
        st = string(arcname, iya, da, extn, format = "(a,i2.2,i3.3,a)")
        fn = match_names(st)
        if (fn ne '') then goto, found
    endfor
endif

;       Then scan forwards.

look_forward:

;       Note don't re-look at the default file if a backward search
;	has been made!

da = day + keyword_set(backwards)
iya = date100(year, /print)
while (da gt (365 + ((iya mod 4) eq 0))) do begin
    da = da - (365 + ((iya mod 4) eq 0))
    iya = iya+1
endwhile

while (compare_times([iya, da,0,0], tstop) le 0) do begin

    st = string(arcname, iya, da, extn, format = "(a,i2.2,i3.3,a)")
    fn = match_names(st)
    if (fn ne '' and fn ne oldname) then goto, found

    da = da+1
    while (da gt (365 + ((iya mod 4) eq 0))) do begin
        da = da - (365 + ((iya mod 4) eq 0))
        iya = iya+1
    endwhile
endwhile

if (first_pass) then begin
    first_pass = 0b
    files_needed = 1b
    goto, search_pass
endif else if (wid_mode) then fn = mk_fn_menu(day, iy, arcname, extn) $
else begin
    fn = ''
    read, 'Default file not found, please enter name :_',fn
endelse

Found:

if (n_params() eq 3) then $
    if (fn eq oldname) then begin
        fn = ''         ; Clear the filename to prevent possibility of infinite
                        ; loop
        goto, look_forward
endif

return, fn

end

