function get_lan_record, hds, dds, tzero, icontinue, $
	trk = trk, pha = pha, psum = psum, mfss = mfss, only = only

;+
; GET_LAN_RECORD
;	Read the next record from a HISCALE data file, opening a new
;	file if needed.
;
; Usage:
;	iget = get_lan_record(hds, dds, iy, id, icontinue[, $
;		/trk, /pha, /psum, /mfss, only=only])
;
; Return Value:
;	iget	int	> 0, record code. 0 soft end, -ve hard end
;
; Arguments:
;	hds	output	Structure	The ULA record header from the record
;	dds	output	structure	The main body of the record.
;	tzero	input	int		The [y,d] of the data start.
;	icontinue in/out  byte		Set if a read has started.
;					N.B. This argument must be
;					present in the call else a new
;					read will start each time.
;
; Keywords:
;	trk	input	If set, then read from a TRKyyddd.LAN file.
;	pha	input	If set, then read from a ULAyyddd.PHA file.
;	psum	input	If set, then read from a PHAyyddd.LAN file.
;	mfss	input	If set, then read from a MFSyyddd.LAN file.
;	only	input	string	Only return records of the given type.
;
; History:
;	Original (extracted from ACQUIRE_DATA etc): 13/8/93; SJT
;	Remove toggling of monitor mode: 28/8/94; SJT
;	Make search from previous day if start is in first hour:
;	21/10/94; SJT
;	Modify end of data criteria: 24/10/94; SJT
;	Fix possible infinite loop: 7/11/94; SJT
;	Fix for dates beyond 2000: 1/12/95; SJT
;-

@time_limits
@settings
@wid_set
@file_info

common Lan_rec_local, iy, id, ily, first_flag, open_flag, file, nmax, $
  ifile, at_end

if (n_elements(icontinue) eq 0) then icontinue = 0b
back_flag = 0b

;		Set up initial file (only if not a continuation).

Try_again:

if (not icontinue) then begin

    icontinue = 1b
    at_end = 0b
    
    iy = tstart(0)
    id = tstart(1)
        
    ily = 365 + (iy mod 4 eq 0) 	; Won't work for non-400
					; century years

    if (back_flag) then begin ; Start on previous day to avoid
        id = id-1               ; possibility of missing a few minutes
        if (id eq 0) then begin
            iy = iy-1
            ily = 365 + (iy mod 4 eq 0)
            id = ily
        endif
    endif
    
;		Calculate file name and open it.

    if keyword_set(trk) then 	   file = make_fname(iy, id, /trk, /back) $
    else if keyword_set(pha) then  file = make_fname(iy, id, /pha, /back) $
    else if keyword_set(mfss) then file = make_fname(iy, id, /mfss, /back) $
    else if keyword_set(psum) then file = make_fname(iy, id, /psum, /back) $
    else                           file = make_fname(iy, id, /back)

    if (file eq '') then return, -1

    iy = tstart(0)
    id = tstart(1)
        
    ily = 365 + (iy mod 4 eq 0) 	; Won't work for non-400
					; century years

    open_lanfile, file, 0l, ifile, nmax
    if (ifile lt 0) then begin
        hs_err_msg, ['Could not open file: '+file+'!', $
		     'Error code ='+string(ifile, format="(I4)")]
        return, ifile
    endif else hs_err_msg, /noshow, 'Starting to read data from: '+file

    map_lanfile, ifile, "", imap
    if (imap ge 0) then seek_lanfile, ifile, iy, id, tstart(2), pos

    first_flag = 1
    open_flag = 1
endif

if (at_end) then begin
    close_lanfile, ifile, ec
    if (ec lt 0) then hs_err_msg, 'Error closing '+ file+ $
	': code = ' + string(ec, format="(I4)")
    return, 0l
endif

read:

read_lanfile, ifile, data, nmax, nact, nfile                ; Read data

;		End-of-file or other error detected by C-rtl.
;		Close file and open next one.

if (nfile eq -1) then begin
    close_lanfile, ifile, ec
    if (ec lt 0) then hs_err_msg, 'Error closing '+ file+ $
		': code = '+string(ec, format="(I4)")

    ido = id
    iyo = iy
    id = fix(hds.time(1,1))		; Next search time is day of
    if (id eq fix(hds.time(1,0))) then id = id+1
    iy = fix(date100(hds.time(0,1)))	; end of previous record

    if (id gt ily) then begin
        id = 1
        iy = iy+1
        ily = 365 + (iy mod 4 eq 0)
    endif
    
    if keyword_set(trk) then       file = make_fname(iy, id, /trk, file) $
    else if keyword_set(pha) then  file = make_fname(iy, id, /pha, file) $
    else if keyword_set(mfss) then file = make_fname(iy, id, /mfss, file) $
    else if keyword_set(psum) then file = make_fname(iy, id, /psum, file) $
    else                           file = make_fname(iy, id, file)
    
    if file eq '' then begin
        hs_err_msg, 'No new file specified, truncating plot.'
        return, 0l
    endif
    open_lanfile, file, 0l, ifile, nmax
    open_flag = 1

    if (ifile lt 0) then begin
        hs_err_msg, ['Could not open file: '+file+'!', $
		     'Error code ='+string(ifile, format="(I4)"), $
            	     'Truncating plot at this point.']
        return, 0l
    endif else hs_err_msg, /noshow, 'Continuing to read data from file: '+file
    goto, read
endif

;		Error detected by HSIO - request action from user.

if nfile lt 0 then begin                                    ; error
    act = err_action(nfile)
    case 1 of
        act eq 'A': return, -1
        act eq 'T': return, 0l
        act eq 'R': goto, read
    endcase
endif

;		Convert data read into structure and print message giving
;		time/type.

itype = ula_struct (data, hds, dds, only = only) ; extract structs

if (itype eq -1) then goto, read

if (open_flag and first_flag and not back_flag and $
    compare_times(hds.time(0:4, 0), tstart) gt 0) then begin
                                ;Case where data started (or may have
                                ;started before expected file.
    back_flag = 1b
    close_lanfile, ifile, ec
    if (ec lt 0) then hs_err_msg, 'Error closing '+ file+ $
		': code = '+string(ec, format="(I4)")
    icontinue = 0b
    goto, try_again
endif

message, /noname, /inform, string(string(hds.rectyp), hds.time(0:4, 0), $
        format = "(A, ' - ', i4,'/',i3.3,' ',2(i2.2,':'),f5.2)")

if (open_flag) then begin
    rc = compare_version (hds, first_flag)
    open_flag = 0
    if (rc lt 0) then return, -1 $			; Total abort
    else if (rc gt 0) then return, 0			; Truncate
endif

if itype eq 0 then  goto, read


;		Check time range and take action accordingly.

if (compare_times(hds.time(0:4, 1), tstart) le 0) then goto, read
if (compare_times(hds.time(0:4, 1), tstop) ge 0 and hds.rindex eq $
    hds.nrecord) then at_end = 1b

;****************************************************************
;* THERE NOW FOLLOWS A FUDGE TO DEAL WITH THE BAD FA4 FIELDS IN *
;* THE CMPxx FILES. IF ALL THE FA4's ARE ZERO, AND UAV IS NOT	*
;* ZERO, THEN SET THEM ALL TO 1					*
;****************************************************************

tkeys = (keyword_set(pha)+keyword_set(trk)+keyword_set(psum)+keyword_set(mfss))
if (tkeys eq 0) then if (archive eq 2 or archive eq 4) then $
  if (max(dds.fa4) eq 0.) then dds.fa4(*) = 1.0

return, itype

end
