PRO LET_PROCESS, LETL1DATA, PKT_SCid, good_frame
;+
; Project     :	STEREO - IMPACT, LET
;
; Name        : LET_PROCESS
;
; Purpose     :	From a LET science frame, extract rates data, compute LET Level 1 counts and intensities
;
; Category    :	STEREO, IMPACT, LET
;
; Explanation :	
;
; Syntax      :	LET_PROCESS, LETL1DATA, SCid
;
; Examples    :	
;
; Inputs      :	SCid
;
; Opt. Inputs :	None.
;
; Outputs     :	LETL1DATA = A structure containing LET Level 1 data, good_frame flag
;
; Opt. Outputs:	None.
;
; Keywords    :	None
;
; Calls       :	sum_rates, read_let_factors, decompressLetCounters, tai2utc, load_struct, 
;
; Common      :	let_l1
;
; Restrictions:	
;
; Side effects:	None.
;
; Prev. Hist. :	None.
;
; History     : Version 1,  4-18-2005, Andrew Davis, Caltech.
;                           Initial version, many shortcomings...
;
;               Version 2,  9-23-2005, Andrew Davis, Caltech.
;                           Handles short sci-frames (<16 packets)
;                           Livetime counter handled
;                           Sectored fluxes calculated
;               Version 3,  2/22/2006, Andrew Davis, Caltech
;                           implemented different factors file for each s/c
;                           implemented full paths to input and temporary files
;                           implemented SCid from packet GRH into LET level 1 header
;                           SciFrmCHECKsum still needs to be verified
;                           Added conditional swap_endian to handle little-endian architectures
;
;                           3/7/2006, Andrew Davis, Caltech.
;                           Changed IMPACT_CALIB_DATA to LET_L1_CALIB_DATA
;               Version 4,  4/26/2006, Andrew Davis, Caltech.
;                           New factors files
;                           Implemented load_struct per Bill Thompson to do away with temporary files
;                           Verify science frame checksum
;                           Rename calc_flux() to sum_rates()
;               Version 5,  5/10/2006, Andrew Davis, Caltech.
;                           add flag to control inclusion of raw rates into L1 data structure
;               Version 6,  1/22/2007, Andrew Davis, Caltech.
;                           Dynamic threshold status is out-of-phase by 1 minute in telemetry.
;                           Added code to account for this.
;                           Rearranged order of items in LETL1Data.header structure.
;                           Now reading in different factors files depending on time/date.
;		Version 7,  3/12/2007, Andrew Davis Caltech
;                           Added summed energy-ranges.
;                           Added code in let_level1.pro to handle last minute of each day properly.
;               Version 8,  7/02/2007, Andrew Davis, Caltech.
;                           Added, Na, Al, S, Ar, Ca, Ni.
;                           Added elemental Helium.
;                           Improved database for factors files.
;                           Added time-dependent species/energy-bin masks to allow different
;                           species/energy-bins to be masked off (filled) at different times.
;                           Added some kludges to allow some summed H and He items to be valid
;                           when their underlying energy-bins are masked as invalid...
;                           Fixed bug in reading of sectored factors files
;		Version 9,  9/4/2007 Andrew Davis, Caltech.
;                           Updated post-March29 2007 sectored geometry factors.
;		Version 11, 02/18/2013 Andrew Davis, Caltech
;			    Added NUMTRIGS correction to Livetime
;
;
; Contact     :	ad@srl.caltech.edu
;-
;

; see let_l1_common.pro
common let_l1 

; structure to hold LET science frame, minus the events
frmstruct = { $
       SCID_VER:   0B, $
       LEN:        0U, $
       SFCHECK:    0U, $
       STATUS:     0B, $
       MINUTE:     0B, $
       LIVETIME:   0U, $            
       ERATES:     UINTARR(8),   $    ; 16-bit compressed rates from here on...
       SNGRATES:   UINTARR(108), $
       MISCRATES:  UINTARR(5),   $
       EVPRATES:   UINTARR(16),  $
       COINRATES:  UINTARR(12),  $
       BUFRATES:   UINTARR(29),  $
       L2FGRATES:  UINTARR(132), $
       L2BGRATES:  UINTARR(12),  $
       L3FGRATES:  UINTARR(167), $
       L3BGRATES:  UINTARR(12),  $
       PENFGRATES: UINTARR(33),  $
       PENBGRATES: UINTARR(15),  $
       SECTORATES: UINTARR(160)  $
}       

; decompressed counters
decom_rates = { $
       ERATES:     LONARR(8),   $ 
       SNGRATES:   LONARR(108), $
       MISCRATES:  LONARR(5),   $
       EVPRATES:   LONARR(16),  $
       COINRATES:  LONARR(12),  $
       BUFRATES:   LONARR(29),  $
       L2FGRATES:  LONARR(132), $
       L2BGRATES:  LONARR(12),  $
       L3FGRATES:  LONARR(167), $
       L3BGRATES:  LONARR(12),  $
       PENFGRATES: LONARR(33),  $
       PENBGRATES: LONARR(15),  $
       SECTORATES: LONARR(160)  $
}

timeCDS = { MJD: 0L , TIME: 0L }

; header for level 1 data structure
header = { $
       timeCDS: timeCDS, $    ; UTC Time at start of data collection interval: MJD, millisec_of_day
       Livetime: 0.0D, $      ; LET livetime (seconds). Usually close to 60 seconds.
       SCid: 0U, $            ; Spacecraft ID from packet GRH header: 234=ahead, 235=behind
       SCidLET: 0B, $         ; Spacecraft ID from LET Sci Frm: 0=EM_unit, 1=FM1 (ahead), 2=FM2 (behind).
       SWversion: 0B, $       ; LET level 1 processing software version (this software)
       SciFrmVersion: 0B, $   ; LET Science Frame Version number
       CodeOK: 0B, $          ; LET internal code check, 1=good, 0=bad
       HeaterDutyCycle: 0B, $ ; heater duty cycle: 0 - 10 
       LeakConv: 0B, $        ; leakage current balancing, 1=OK, 0=Oops
       DyThState: 0B, $       ; Dynamic thresholding state; 0=normal.
                              ; 1, 2, 3, indicate higher threshold settings
			      ; in response to higher count-rates
       Minute: 0B $           ; LET's minute counter. Rolls over at 60
}

; The following energy bin thresholds are defined for the LET data:
; 1.8,2.2,2.7,3.2,3.6,4,4.5,5,6,8,10,12,15,21,27,33,40,52,70 MeV/nuc
; Nomenclature: energy bin 1.8 covers the energy range 1.8 to 2.2 MeV/nuc, etc.
unsectored_flux = { $
       H:   DBLARR(Enum[ 0]), $ ; energy bins 1.8 thru 12, inclusive 
       He3: DBLARR(Enum[ 1]), $ ; energy bins 2.2 thru 12, inclusive 
       He4: DBLARR(Enum[ 2]), $ ; energy bins 1.8 thru 12, inclusive 
       He:  DBLARR(Enum[ 1]), $ ; energy bins 2.2 thru 12, inclusive  Elemental Helium, added 06-08-2007 AJD
       C:   DBLARR(Enum[ 3]), $ ; energy bins 3.2 thru 27, inclusive 
       N:   DBLARR(Enum[ 4]), $ ; energy bins 3.2 thru 27, inclusive 
       O:   DBLARR(Enum[ 5]), $ ; energy bins 3.2 thru 27, inclusive 
       Neon: DBLARR(Enum[6]), $ ; energy bins 3.2 thru 33, inclusive  Ne is reserved for IDL...
       Na:  DBLARR(Enum[ 7]),  $ ; energy bins 6 thru 33, inclusive 
       Mg:  DBLARR(Enum[ 8]), $ ; energy bins 3.2 thru 40, inclusive 
       Al:  DBLARR(Enum[ 9]),  $ ; energy bins 6 thru 40, inclusive 
       Si:  DBLARR(Enum[10]), $ ; energy bins 3.2 thru 40, inclusive 
       S:   DBLARR(Enum[11]), $ ; energy bins 3.6 thru 40, inclusive 
       Ar:  DBLARR(Enum[12]), $ ; energy bins 3.6 thru 40, inclusive  
       Ca:  DBLARR(Enum[13]), $ ; energy bins 3.6 thru 40, inclusive 
       Fe:  DBLARR(Enum[14]), $ ; energy bins 2.7 thru 52, inclusive 
       Ni:  DBLARR(Enum[15])   $ ; energy bins 8 thru 52, inclusive 
}

unsectored_counts = { $
       H:   LONARR(Enum[ 0]), $ ; energy bins 1.8 thru 12, inclusive 
       He3: LONARR(Enum[ 1]), $ ; energy bins 2.2 thru 12, inclusive 
       He4: LONARR(Enum[ 2]), $ ; energy bins 1.8 thru 12, inclusive 
       He:  LONARR(Enum[ 1]), $ ; energy bins 2.2 thru 12, inclusive  Elemental Helium, added 06-08-2007 AJD
       C:   LONARR(Enum[ 3]), $ ; energy bins 3.2 thru 27, inclusive 
       N:   LONARR(Enum[ 4]), $ ; energy bins 3.2 thru 27, inclusive 
       O:   LONARR(Enum[ 5]), $ ; energy bins 3.2 thru 27, inclusive 
       Neon: LONARR(Enum[6]), $ ; energy bins 3.2 thru 33, inclusive  Ne is reserved for IDL...
       Na:  LONARR(Enum[ 7]),  $ ; energy bins 6 thru 33, inclusive 
       Mg:  LONARR(Enum[ 8]), $ ; energy bins 3.2 thru 40, inclusive 
       Al:  LONARR(Enum[ 9]),  $ ; energy bins 6 thru 40, inclusive 
       Si:  LONARR(Enum[10]), $ ; energy bins 3.2 thru 40, inclusive 
       S:   LONARR(Enum[11]), $ ; energy bins 3.6 thru 40, inclusive 
       Ar:  LONARR(Enum[12]), $ ; energy bins 3.6 thru 40, inclusive  
       Ca:  LONARR(Enum[13]), $ ; energy bins 3.6 thru 40, inclusive 
       Fe:  LONARR(Enum[14]), $ ; energy bins 2.7 thru 52, inclusive 
       Ni:  LONARR(Enum[15])   $ ; energy bins 8 thru 52, inclusive 
}

; fluxes in wider energy ranges, derived from the original energy ranges
summed_flux = { $
       H:   DBLARR(Esum[ 0]), $ ; 1.8 to 3.6, 4 to 6, 6 to 10, 10 to 15
       He3: DBLARR(Esum[ 1]), $ ; 4 to 6, 6 to 10, 10 to 15
       He4: DBLARR(Esum[ 2]), $ ; 1.8 to 3.6, 4 to 6, 6 to 10, 10 to 15
       He:  DBLARR(Esum[ 1]), $ ; 4 to 6, 6 to 10, 10 to 15  Elemental Helium, added 06-08-2007 AJD
       C:   DBLARR(Esum[ 3]), $ ; 4 to 6, 6 to 10, 10 to 15, 15 to 27
       N:   DBLARR(Esum[ 4]), $ ; 4 to 6, 6 to 10, 10 to 15, 15 to 27
       O:   DBLARR(Esum[ 5]), $ ; 4 to 6, 6 to 10, 10 to 15, 15 to 27
       Neon: DBLARR(Esum[6]), $ ; 4 to 6, 6 to 10, 10 to 15, 15 to 27
       Na:  DBLARR(Esum[ 7]), $ ; 10 to 15, 15 to 27
       Mg:  DBLARR(Esum[ 8]), $ ; 4 to 6, 6 to 10, 10 to 15, 15 to 27
       Al:  DBLARR(Esum[ 9]), $ ; 6 to 10, 10 to 15, 15 to 27
       Si:  DBLARR(Esum[10]), $ ; 4 to 6, 6 to 10, 10 to 15, 15 to 27, 27 to 40 
       S:   DBLARR(Esum[11]), $ ; 4 to 6, 6 to 10, 10 to 15, 15 to 27, 27 to 40 
       Ar:  DBLARR(Esum[12]), $ ; 4 to 6, 6 to 10, 10 to 15, 15 to 27, 27 to 40 
       Ca:  DBLARR(Esum[13]), $ ; 4 to 6, 6 to 10, 10 to 15, 15 to 27, 27 to 40 
       Fe:  DBLARR(Esum[14]), $ ; 4 to 6, 6 to 10, 10 to 15, 15 to 27, 27 to 40 
       Ni:  DBLARR(Esum[15])  $ ; 10 to 15, 15 to 27, 27 to 40 
}

; counts in wider energy ranges, derived from the original energy ranges
summed_counts = { $
       H:   LONARR(Esum[ 0]), $ ; 
       He3: LONARR(Esum[ 1]), $ ; 
       He4: LONARR(Esum[ 2]), $ ; 
       He:  LONARR(Esum[ 1]), $ ; Elemental Helium, added 06-08-2007 AJD
       C:   LONARR(Esum[ 3]), $ ; 
       N:   LONARR(Esum[ 4]), $ ; 
       O:   LONARR(Esum[ 5]), $ ; 
       Neon: LONARR(Esum[6]), $ ; 
       Na:  LONARR(Esum[ 7]), $ ;
       Mg:  LONARR(Esum[ 8]), $ ; 
       Al:  LONARR(Esum[ 9]), $ ;
       Si:  LONARR(Esum[10]), $ ; 
       S:   LONARR(Esum[11]), $ ; 
       Ar:  LONARR(Esum[12]), $ ; 
       Ca:  LONARR(Esum[13]), $ ; 
       Fe:  LONARR(Esum[14]), $ ; 
       Ni:  LONARR(Esum[15])  $ ;
}

sectored_flux = { $
       H_vlo:      DBLARR(16), $ ; 16 look-directions, (1.8 to 3.6 MeV)  Invalid before 2010-11-22
       H_lo:       DBLARR(16), $ ; 16 look-directions, energy bins 4, 4.5, 5 (4 to 6 MeV)
       H_hi:       DBLARR(16), $ ; 16 look-directions, (6 to 10 MeV)  Invalid before 2010-11-22
       He3_lo:     DBLARR(16), $ ; 16 look-directions, energy bins 4, 4.5, 5 (4 to 6 MeV/nuc)  Invalid after 2010-11-22
       He4_lo:     DBLARR(16), $ ; 16 look-directions, energy bins 4, 4.5, 5 (4 to 6 MeV/nuc)
       He4_hi:     DBLARR(16), $ ; 16 look-directions, energy bins 6, 8, 10 (6 to 12 MeV/nuc)
       He_lo:      DBLARR(16), $ ; 16 look-directions, energy bins 4, 4.5, 5 (4 to 6 MeV/nuc)  Invalid after 2010-11-22
       CNO_lo:     DBLARR(16), $ ; 16 look-directions, energy bins 4, 4.5, 5 (4 to 6 MeV/nuc)
       CNO_hi:     DBLARR(16), $ ; 16 look-directions, energy bins 6, 8, 10 (6 to 12 MeV/nuc)
       NeMgSi_lo:  DBLARR(16), $ ; 16 look-directions, energy bins 4, 4.5, 5 (4 to 6 MeV/nuc)
       NeMgSi_hi:  DBLARR(16), $ ; 16 look-directions, energy bins 6, 8, 10 (6 to 12 MeV/nuc)
       Fe_lo:      DBLARR(16), $ ; 16 look-directions, energy bins 4, 4.5, 5 (4 to 6 MeV/nuc)  Invalid after 2010-11-22
       Fe_hi:      DBLARR(16), $ ; 16 look-directions, energy bins 6, 8, 10 (6 to 12 MeV/nuc)  Invalid after 2010-11-22
       Fe_wide:    DBLARR(16)  $ ; 16 look-directions, (4 to 12 MeV/nuc)  Invalid before 2010-11-22
}

sectored_counts = { $
       H_vlo:      LONARR(16), $ ; 16 look-directions, (1.8 to 3.6 MeV)  Invalid before 2010-11-22
       H_lo:       LONARR(16), $ ; 16 look-directions, energy bins 4, 4.5, 5 (4 to 6 MeV)
       H_hi:       LONARR(16), $ ; 16 look-directions, (6 to 10 MeV)  Invalid before 2010-11-22
       He3_lo:     LONARR(16), $ ; 16 look-directions, energy bins 4, 4.5, 5 (4 to 6 MeV/nuc)  Invalid after 2010-11-22
       He4_lo:     LONARR(16), $ ; 16 look-directions, energy bins 4, 4.5, 5 (4 to 6 MeV/nuc)
       He4_hi:     LONARR(16), $ ; 16 look-directions, energy bins 6, 8, 10 (6 to 12 MeV/nuc)
       He_lo:      LONARR(16), $ ; 16 look-directions, energy bins 4, 4.5, 5 (4 to 6 MeV/nuc)  Invalid after 2010-11-22
       CNO_lo:     LONARR(16), $ ; 16 look-directions, energy bins 4, 4.5, 5 (4 to 6 MeV/nuc)
       CNO_hi:     LONARR(16), $ ; 16 look-directions, energy bins 6, 8, 10 (6 to 12 MeV/nuc)
       NeMgSi_lo:  LONARR(16), $ ; 16 look-directions, energy bins 4, 4.5, 5 (4 to 6 MeV/nuc)
       NeMgSi_hi:  LONARR(16), $ ; 16 look-directions, energy bins 6, 8, 10 (6 to 12 MeV/nuc)
       Fe_lo:      LONARR(16), $ ; 16 look-directions, energy bins 4, 4.5, 5 (4 to 6 MeV/nuc)  Invalid after 2010-11-22
       Fe_hi:      LONARR(16), $ ; 16 look-directions, energy bins 6, 8, 10 (6 to 12 MeV/nuc)  Invalid after 2010-11-22
       Fe_wide:    LONARR(16)  $ ; 16 look-directions, (4 to 12 MeV/nuc)  Invalid before 2010-11-22
}


      
      
      ;  load the raw data bytes into LET Sci frame structure      
         load_struct, letscifrm, frmstruct[0], frmstruct, /noieee, /noconv

      ;  swap bytes around if we are running on a little-endian architecture
         frmstruct = swap_endian(frmstruct, /SWAP_IF_LITTLE_ENDIAN)

      ;  Verify the LET Science Frame checksum
	 chksum = 0L
         FOR i=0, frmstruct.len -1 DO BEGIN
	    chksum = chksum + letscifrm[i]
	 ENDFOR
	 chksum = chksum - letscifrm[3] - letscifrm[4] 
	 chksum = (-1 * chksum) and 'FFFF'X

      ;  printf, -2, frmstruct.len, frmstruct.sfcheck, chksum

	 IF (frmstruct.sfcheck ne chksum) THEN BEGIN
	    printf, -2, 'LET_PROCESS: Warning: Bad LET science frame checksum:', frmstruct.sfcheck, chksum
	    baduttime = tai2utc(double(pkt_times[0]),/NOCORRECT)
            printf,-2,pkt_times[0],"  ",utc2str(baduttime)
	    good_frame = 0
	    RETURN
	 ENDIF

      ;  printf, -2, FORMAT='(%"SRATES: %x   %x")',frmstruct.SECTORATES[158], frmstruct.SECTORATES[159]

	 decom_rates.ERATES = decompressLetCounters(frmstruct.ERATES)
         decom_rates.SNGRATES = decompressLetCounters(frmstruct.SNGRATES)
         decom_rates.MISCRATES = decompressLetCounters(frmstruct.MISCRATES)
         decom_rates.EVPRATES = decompressLetCounters(frmstruct.EVPRATES)
         decom_rates.COINRATES = decompressLetCounters(frmstruct.COINRATES)
         decom_rates.BUFRATES = decompressLetCounters(frmstruct.BUFRATES)
         decom_rates.L2FGRATES = decompressLetCounters(frmstruct.L2FGRATES)
         decom_rates.L2BGRATES = decompressLetCounters(frmstruct.L2BGRATES)
         decom_rates.L3FGRATES = decompressLetCounters(frmstruct.L3FGRATES)
         decom_rates.L3BGRATES = decompressLetCounters(frmstruct.L3BGRATES)
         decom_rates.PENFGRATES = decompressLetCounters(frmstruct.PENFGRATES)
         decom_rates.PENBGRATES = decompressLetCounters(frmstruct.PENBGRATES)
         decom_rates.SECTORATES = decompressLetCounters(frmstruct.SECTORATES)
      											   
      ;  printf,-2,decom_rates.ERATES[0],decom_rates.ERATES[2]
      
      ;  build level 1 data header
         header.SWversion = VERSION
	 ; Use the /NOCORRECT option here, since STEREO spacecraft time already has
	 ; leap seconds factored in. i.e. spacecraft time is UTC seconds since 1958
         header.timeCDS = tai2utc(double(pkt_times[0]), /NOCORRECT)
         header.SCidLET = ishft(frmstruct.scid_ver and 'C0'X,-6)
	 header.SCid = PKT_SCid
         header.SciFrmVersion = frmstruct.scid_ver and '3F'X
         header.CodeOK = ishft(frmstruct.status and '80'X,-7)
         header.HeaterDutyCycle = ishft(frmstruct.status and '78'X,-3)
         header.LeakConv = ishft(frmstruct.status and '4'X,-2)

         IF (first_frame eq 1) then BEGIN
	    SaveDyThState = frmstruct.status and '3'X
	    header.DyThState = SaveDyThState
	 ENDIF ELSE BEGIN
	    header.DyThState = SaveDyThState
	    SaveDyThState = frmstruct.status and '3'X
         ENDELSE

         ; LT_corrected = LT + (C * NUMTRIGS)
	 ; here, C includes factor of 2 to account for ERATES being accumulated only on even minutes
	 header.Livetime = (frmstruct.LIVETIME * LIVETIMESCALEFACTOR * 60.0D) + ( LIVETIMECORRFACTOR * decom_rates.ERATES[0] )
	 IF (header.Livetime gt 60.0D) then BEGIN
	     header.Livetime = 60.0D
	 ENDIF
	 
         header.Minute = frmstruct.minute
      
      ;   printf,-2,pkt_times[0], header.timeCDS
      ;   printf,-2, header.SWversion ,header.SCidLET, header.SciFrmVersion, frmstruct.len

      ;  read calibration factors and masks
         need_to_read = 1
         sectored_need_to_read = 1
         find_let_factors_files, header.SCid, double(pkt_times[0]), factors_filenm, $
	 	sectored_factors_filenm, need_to_read, sectored_need_to_read

         IF (need_to_read) then BEGIN
         ;   print, utc2str(header.timeCDS), "  ", need_to_read, "  ", factors_filenm
	    read_let_factors, factors_filenm
         ENDIF
	 
         IF (sectored_need_to_read) then BEGIN
         ;   print, utc2str(header.timeCDS), "  ", sectored_need_to_read, "  ", sectored_factors_filenm
	    read_let_sectored_factors, sectored_factors_filenm
         ENDIF
	 
         masks_need_to_read = 1
         sectored_masks_need_to_read = 1
         find_let_masks_files, header.SCid, double(pkt_times[0]), masks_filenm, $
            sectored_masks_filenm, masks_need_to_read, sectored_masks_need_to_read
         IF (masks_need_to_read) then BEGIN
         ;   print, utc2str(header.timeCDS), "  ", masks_need_to_read, "  ", masks_filenm
	    read_let_masks, masks_filenm
         ENDIF
	 
         IF (sectored_masks_need_to_read) then BEGIN
         ;   print, utc2str(header.timeCDS), "  ", sectored_need_to_read, "  ", sectored_masks_filenm
	    read_let_sectored_masks, sectored_masks_filenm
         ENDIF
	 
         divisor = 0.0D

      ; Kludge in some special cases. NOte: assumes these bins are "1" in the masks files
      ; 15-21 MeV/n Carbon, during periods when DyThState > 0
      ;  IF (header.DyThState gt 0) then BEGIN
      ;     Masks.C_masks[9].mask = 0	   
      ;	ENDIF ELSE BEGIN
      ;     Masks.C_masks[9].mask = 1
      ;	ENDELSE
      
      ; build level 1 unsectored counts and fluxes
        ; protons 0. Do not use masks here, since some of the summed ranges are valid
	; even though the individual bins are not...
	 divisors = DBLARR(NEBINS)
	 summasks = 0.0
         n = Enum[0]
         FOR i=0,n-1 DO BEGIN
	    summasks = summasks + Masks.H_masks[i].mask
            R2idx = R2_ctrs[0,Emin[0]+i]
            R3idx = R3_ctrs[0,Emin[0]+i]
            R4idx = R4_ctrs[0,Emin[0]+i]
	    unsectored_counts.H[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.H_fctrs[i].Ehi-Factors.H_fctrs[i].Elo) * $
	               Factors.H_fctrs[i].geom[header.DyThState] * $
		       Factors.H_fctrs[i].effic[header.DyThState]
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.H[i] ge 0L)) THEN $
               unsectored_flux.H[i] = unsectored_counts.H[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.H[i] = FILL_D
	       unsectored_counts.H[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed proton energy ranges 
	 FOR i=0,Esum[0]-1 DO BEGIN
	    sums = Esumdefs.H_sums[*,i]			; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)]	; eliminate the -1 entries  
	    datatosum = unsectored_flux.H[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	;    print, nindices, nvsums

	;   Summed fluxes valid only if all contributing elements are non-fill
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.H[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_flux.H[i] =  TOTAL(intermeddata)/ TOTAL(vdivisorstouse)	; compute the intensity for the summed energy-range
	       summed_counts.H[i] = TOTAL(vcountstosum)	; add up all the counts
	    ENDIF ELSE BEGIN
	       summed_flux.H[i] =  FILL_D
	       summed_counts.H[i] = FILL_L
	    ENDELSE
	; Prior to March 29 2007, all the public proton bins are bad, and the public summed bins should be marked as bad also.
	    IF summasks LT 0.5 THEN BEGIN
	       summed_flux.H[i] =  FILL_D
	       summed_counts.H[i] = FILL_L
	    ENDIF
	 ENDFOR
 	; The 10-15 MeV public proton bin is bad even after March 29 2007
	 IF Masks.H_masks[Enum[0]-1].mask LT 0.5 THEN BEGIN
            summed_flux.H[Esum[0]-1] =  FILL_D
            summed_counts.H[Esum[0]-1] = FILL_L
	 ENDIF
	 
	; Redo protons, this time with masks   
         n = Enum[0]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[0,Emin[0]+i]
            R3idx = R3_ctrs[0,Emin[0]+i]
            R4idx = R4_ctrs[0,Emin[0]+i]
	    unsectored_counts.H[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.H_fctrs[i].Ehi-Factors.H_fctrs[i].Elo) * $
	               Factors.H_fctrs[i].geom[header.DyThState] * $
		       Factors.H_fctrs[i].effic[header.DyThState] * Masks.H_masks[i].mask
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.H[i] ge 0L)) THEN $
               unsectored_flux.H[i] = unsectored_counts.H[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.H[i] = FILL_D
	       unsectored_counts.H[i] = FILL_L
	    ENDELSE
         ENDFOR


	; He3 1
	; Do not use the masks here, as we are computing He3 for inclusion in the elemental He flux here
         n = Enum[1]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[1,Emin[1]+i]
            R3idx = R3_ctrs[1,Emin[1]+i]
            R4idx = R4_ctrs[1,Emin[1]+i]
            unsectored_counts.He3[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.He3_fctrs[i].Ehi-Factors.He3_fctrs[i].Elo) * $
	               Factors.He3_fctrs[i].geom[header.DyThState] * $
		       Factors.He3_fctrs[i].effic[header.DyThState] 
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.He3[i] ge 0L)) THEN $
               unsectored_flux.He3[i] = unsectored_counts.He3[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.He3[i] = FILL_D
	       unsectored_counts.He3[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed He3 energy ranges 
	 FOR i=0,Esum[1]-1 DO BEGIN
	    sums = Esumdefs.He3_sums[*,i]		; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)] 	; eliminate the -1 entries  
	    datatosum = unsectored_flux.He3[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.He3[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_flux.He3[i] =  TOTAL(intermeddata)/ TOTAL(vdivisorstouse)	; compute the intensity for the summed energy-range
	       summed_counts.He3[i] = TOTAL(vcountstosum)	; add up all the counts
	    ENDIF ELSE BEGIN
	       summed_flux.He3[i] =  FILL_D
	       summed_counts.He3[i] = FILL_L
	    ENDELSE
	 ENDFOR
        ; He4 2
         n = Enum[2]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[2,Emin[2]+i]
            R3idx = R3_ctrs[2,Emin[2]+i]
            R4idx = R4_ctrs[2,Emin[2]+i]
            unsectored_counts.He4[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.He4_fctrs[i].Ehi-Factors.He4_fctrs[i].Elo) * $
	               Factors.He4_fctrs[i].geom[header.DyThState] * $
		       Factors.He4_fctrs[i].effic[header.DyThState] * Masks.He4_masks[i].mask
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.He4[i] ge 0L)) THEN $
               unsectored_flux.He4[i] = unsectored_counts.He4[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.He4[i] = FILL_D
	       unsectored_counts.He4[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed He4 energy ranges
	summed_divisors_He4 = DBLARR(Esum[2])
	 FOR i=0,Esum[2]-1 DO BEGIN
	    sums = Esumdefs.He4_sums[*,i]		; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)] 	; eliminate the -1 entries  
	    datatosum = unsectored_flux.He4[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.He4[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_divisors_He4[i] = TOTAL(vdivisorstouse)	; we need to save this for the summed_He calculation
	       summed_flux.He4[i] =  TOTAL(intermeddata)/ summed_divisors_He4[i]	; compute the intensity for the summed energy-range
	       summed_counts.He4[i] = TOTAL(vcountstosum)	; add up all the counts
	    ENDIF ELSE BEGIN
	       summed_flux.He4[i] =  FILL_D
	       summed_counts.He4[i] = FILL_L
	    ENDELSE
	 ENDFOR

        ; Elemental helium - sum the appropriate He3 and He4 counts together, and compute elemental flux.
        ; As of March 29 2007, He3 events are energy-binned onboard using a mass of 4 amu rather than 3,
	; so the He3 and He4 rate-boxes can be easily combined on the ground.
	; Have to be careful, since there is no 1.8 - 2.2 MeV/nuc He3 box.
	; We use the He4 divisors already computed above...

	n = Enum[1] ; n = number of He3 boxes. Number of He4 boxes = n+1.
        FOR i=0,n-1 DO BEGIN
	   IF ( (unsectored_counts.He3[i] ge 0L) AND (unsectored_counts.He4[i+1] ge 0L) ) THEN BEGIN
	      unsectored_counts.He[i] = unsectored_counts.He3[i] + unsectored_counts.He4[i+1]
	      ; Note that this summed flux calculation is only good if the geom and efficiency
	      ; factors for each individual He3 and He4 energy range in the sum are all the same
	      unsectored_flux.He[i] = unsectored_counts.He[i] / divisors[i+1]
	   ENDIF ELSE BEGIN
	       unsectored_flux.He[i] = FILL_D
	       unsectored_counts.He[i] = FILL_L
	   ENDELSE
 	   IF ( Masks.He_masks[i].mask eq 0.0 ) THEN BEGIN
	       unsectored_flux.He[i] = FILL_D
	       unsectored_counts.He[i] = FILL_L
	   ENDIF
	ENDFOR

	; Now the summed Elemental helium. Again, there is 1 more He4 summed boxes than He3 summed boxes
        summed_He_masks = DBLARR(3) 
	summed_He_masks[0] = Masks.He_masks[0].mask + Masks.He_masks[1].mask + Masks.He_masks[2].mask
	summed_He_masks[1] = Masks.He_masks[3].mask + Masks.He_masks[4].mask
	summed_He_masks[2] = Masks.He_masks[5].mask + Masks.He_masks[6].mask
	IF ( summed_He_masks[0] ne 3.0 ) THEN  summed_He_masks[0] = 0.0
	IF ( summed_He_masks[1] ne 2.0 ) THEN  summed_He_masks[1] = 0.0
	IF ( summed_He_masks[2] ne 2.0 ) THEN  summed_He_masks[2] = 0.0
	
	FOR i=0,Esum[1]-1 DO BEGIN
	   IF ( (summed_counts.He3[i] ge 0L) AND (summed_counts.He4[i+1] ge 0L) ) THEN BEGIN
	      summed_counts.He[i] = summed_counts.He3[i] + summed_counts.He4[i+1]
	      summed_flux.He[i] = summed_counts.He[i] / summed_divisors_He4[i+1]
	   ENDIF ELSE BEGIN
	      summed_flux.He[i] = FILL_D
	      summed_counts.He[i] = FILL_L
	   ENDELSE
 	   IF ( summed_He_masks[i] eq 0.0 ) THEN BEGIN
	       summed_flux.He[i] = FILL_D
	       summed_counts.He[i] = FILL_L
	   ENDIF
	ENDFOR
	
        ; He3 1
	; Redo the He3, and Use the masks here.
         n = Enum[1]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[1,Emin[1]+i]
            R3idx = R3_ctrs[1,Emin[1]+i]
            R4idx = R4_ctrs[1,Emin[1]+i]
            unsectored_counts.He3[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.He3_fctrs[i].Ehi-Factors.He3_fctrs[i].Elo) * $
	               Factors.He3_fctrs[i].geom[header.DyThState] * $
		       Factors.He3_fctrs[i].effic[header.DyThState] * Masks.He3_masks[i].mask
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.He3[i] ge 0L)) THEN $
               unsectored_flux.He3[i] = unsectored_counts.He3[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.He3[i] = FILL_D
	       unsectored_counts.He3[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed He3 energy ranges 
	 FOR i=0,Esum[1]-1 DO BEGIN
	    sums = Esumdefs.He3_sums[*,i]		; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)] 	; eliminate the -1 entries  
	    datatosum = unsectored_flux.He3[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.He3[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_flux.He3[i] =  TOTAL(intermeddata)/ TOTAL(vdivisorstouse)	; compute the intensity for the summed energy-range
	       summed_counts.He3[i] = TOTAL(vcountstosum)	; add up all the counts
	    ENDIF ELSE BEGIN
	       summed_flux.He3[i] =  FILL_D
	       summed_counts.He3[i] = FILL_L
	    ENDELSE
	 ENDFOR

	; C 3
         n = Enum[3]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[3,Emin[3]+i]
            R3idx = R3_ctrs[3,Emin[3]+i]
            R4idx = R4_ctrs[3,Emin[3]+i]
            unsectored_counts.C[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.C_fctrs[i].Ehi-Factors.C_fctrs[i].Elo) * $
	               Factors.C_fctrs[i].geom[header.DyThState] * $
		       Factors.C_fctrs[i].effic[header.DyThState] * Masks.C_masks[i].mask
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.C[i] ge 0L)) THEN $
               unsectored_flux.C[i] = unsectored_counts.C[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.C[i] = FILL_D
	       unsectored_counts.C[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed carbon energy ranges 
	 FOR i=0,Esum[3]-1 DO BEGIN
	    sums = Esumdefs.C_sums[*,i]			; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)] 	; eliminate the -1 entries  
	    datatosum = unsectored_flux.C[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.C[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_flux.C[i] =  TOTAL(intermeddata)/ TOTAL(vdivisorstouse)	; compute the intensity for the summed energy-range
	       summed_counts.C[i] = TOTAL(vcountstosum)	; add up all the counts
	    ENDIF ELSE BEGIN
	       summed_flux.C[i] =  FILL_D
	       summed_counts.C[i] = FILL_L
	    ENDELSE
	 ENDFOR
        ; N 4
         n = Enum[4]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[4,Emin[4]+i]
            R3idx = R3_ctrs[4,Emin[4]+i]
            R4idx = R4_ctrs[4,Emin[4]+i]
            unsectored_counts.N[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.N_fctrs[i].Ehi-Factors.N_fctrs[i].Elo) * $
	               Factors.N_fctrs[i].geom[header.DyThState] * $
		       Factors.N_fctrs[i].effic[header.DyThState] * Masks.N_masks[i].mask
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.N[i] ge 0L)) THEN $
               unsectored_flux.N[i] = unsectored_counts.N[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.N[i] = FILL_D
	       unsectored_counts.N[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed nitrogen energy ranges 
	 FOR i=0,Esum[4]-1 DO BEGIN
	    sums = Esumdefs.N_sums[*,i]			; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)] 	; eliminate the -1 entries  
	    datatosum = unsectored_flux.N[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.N[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_flux.N[i] =  TOTAL(intermeddata)/ TOTAL(vdivisorstouse)	; compute the intensity for the summed energy-range
	       summed_counts.N[i] = TOTAL(vcountstosum)	; add up all the counts
	    ENDIF ELSE BEGIN
	       summed_flux.N[i] =  FILL_D
	       summed_counts.N[i] = FILL_L
	    ENDELSE
	 ENDFOR
        ; O 5
         n = Enum[5]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[5,Emin[5]+i]
            R3idx = R3_ctrs[5,Emin[5]+i]
            R4idx = R4_ctrs[5,Emin[5]+i]
            unsectored_counts.O[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.O_fctrs[i].Ehi-Factors.O_fctrs[i].Elo) * $
	               Factors.O_fctrs[i].geom[header.DyThState] * $
		       Factors.O_fctrs[i].effic[header.DyThState] * Masks.O_masks[i].mask
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.O[i] ge 0L)) THEN $
               unsectored_flux.O[i] = unsectored_counts.O[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.O[i] = FILL_D
	       unsectored_counts.O[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed oxygen energy ranges 
	 FOR i=0,Esum[5]-1 DO BEGIN
	    sums = Esumdefs.O_sums[*,i]			; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)] 	; eliminate the -1 entries  
	    datatosum = unsectored_flux.O[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.O[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_flux.O[i] =  TOTAL(intermeddata)/ TOTAL(vdivisorstouse)	; compute the intensity for the summed energy-range
	       summed_counts.O[i] = TOTAL(vcountstosum)	; add up all the counts
	    ENDIF ELSE BEGIN
	       summed_flux.O[i] =  FILL_D
	       summed_counts.O[i] = FILL_L
	    ENDELSE
	 ENDFOR
        ; Neon 6
         n = Enum[6]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[6,Emin[6]+i]
            R3idx = R3_ctrs[6,Emin[6]+i]
            R4idx = R4_ctrs[6,Emin[6]+i]
            unsectored_counts.Neon[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.Ne_fctrs[i].Ehi-Factors.Ne_fctrs[i].Elo) * $
	               Factors.Ne_fctrs[i].geom[header.DyThState] * $
		       Factors.Ne_fctrs[i].effic[header.DyThState] * Masks.Ne_masks[i].mask
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.Neon[i] ge 0L)) THEN $
               unsectored_flux.Neon[i] = unsectored_counts.Neon[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.Neon[i] = FILL_D
	       unsectored_counts.Neon[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed neon energy ranges 
	 FOR i=0,Esum[6]-1 DO BEGIN
	    sums = Esumdefs.Ne_sums[*,i]		; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)] 	; eliminate the -1 entries  
	    datatosum = unsectored_flux.Neon[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.Neon[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_flux.Neon[i] =  TOTAL(intermeddata)/ TOTAL(vdivisorstouse)	; compute the intensity for the summed energy-range
	       summed_counts.Neon[i] = TOTAL(vcountstosum)	; add up all the counts
	    ENDIF ELSE BEGIN
	       summed_flux.Neon[i] =  FILL_D
	       summed_counts.Neon[i] = FILL_L
	    ENDELSE
	 ENDFOR
        ; Na 7
         n = Enum[7]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[7,Emin[7]+i]
            R3idx = R3_ctrs[7,Emin[7]+i]
            R4idx = R4_ctrs[7,Emin[7]+i]
            unsectored_counts.Na[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.Na_fctrs[i].Ehi-Factors.Na_fctrs[i].Elo) * $
	               Factors.Na_fctrs[i].geom[header.DyThState] * $
		       Factors.Na_fctrs[i].effic[header.DyThState] * Masks.Na_masks[i].mask
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.Na[i] ge 0L)) THEN $
               unsectored_flux.Na[i] = unsectored_counts.Na[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.Na[i] = FILL_D
	       unsectored_counts.Na[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed sodium energy ranges 
	 FOR i=0,Esum[7]-1 DO BEGIN
	    sums = Esumdefs.Na_sums[*,i]		; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)] 	; eliminate the -1 entries  
	    datatosum = unsectored_flux.Na[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.Na[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_flux.Na[i] =  TOTAL(intermeddata)/ TOTAL(vdivisorstouse)	; compute the intensity for the summed energy-range
	       summed_counts.Na[i] = TOTAL(vcountstosum)	; add up all the counts
	    ENDIF ELSE BEGIN
	       summed_flux.Na[i] =  FILL_D
	       summed_counts.Na[i] = FILL_L
	    ENDELSE
	 ENDFOR
        ; Mg 8
         n = Enum[8]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[8,Emin[8]+i]
            R3idx = R3_ctrs[8,Emin[8]+i]
            R4idx = R4_ctrs[8,Emin[8]+i]
            unsectored_counts.Mg[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.Mg_fctrs[i].Ehi-Factors.Mg_fctrs[i].Elo) * $
	               Factors.Mg_fctrs[i].geom[header.DyThState] * $
		       Factors.Mg_fctrs[i].effic[header.DyThState] * Masks.Mg_masks[i].mask
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.Mg[i] ge 0L)) THEN $
               unsectored_flux.Mg[i] = unsectored_counts.Mg[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.Mg[i] = FILL_D
	       unsectored_counts.Mg[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed magnesium energy ranges 
	 FOR i=0,Esum[8]-1 DO BEGIN
	    sums = Esumdefs.Mg_sums[*,i]		; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)] 	; eliminate the -1 entries  
	    datatosum = unsectored_flux.Mg[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.Mg[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_flux.Mg[i] =  TOTAL(intermeddata)/ TOTAL(vdivisorstouse)	; compute the intensity for the summed energy-range
	       summed_counts.Mg[i] = TOTAL(vcountstosum)	; add up all the counts
	    ENDIF ELSE BEGIN
	       summed_flux.Mg[i] =  FILL_D
	       summed_counts.Mg[i] = FILL_L
	    ENDELSE
	 ENDFOR
        ; Al 9
         n = Enum[9]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[9,Emin[9]+i]
            R3idx = R3_ctrs[9,Emin[9]+i]
            R4idx = R4_ctrs[9,Emin[9]+i]
            unsectored_counts.AL[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.Al_fctrs[i].Ehi-Factors.Al_fctrs[i].Elo) * $
	               Factors.Al_fctrs[i].geom[header.DyThState] * $
		       Factors.Al_fctrs[i].effic[header.DyThState] * Masks.Al_masks[i].mask
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.Al[i] ge 0L)) THEN $
               unsectored_flux.Al[i] = unsectored_counts.Al[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.Al[i] = FILL_D
	       unsectored_counts.Al[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed aluminum energy ranges 
	 FOR i=0,Esum[9]-1 DO BEGIN
	    sums = Esumdefs.Al_sums[*,i]		; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)] 	; eliminate the -1 entries  
	    datatosum = unsectored_flux.Al[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.Al[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_flux.Al[i] =  TOTAL(intermeddata)/ TOTAL(vdivisorstouse)	; compute the intensity for the summed energy-range
	       summed_counts.Al[i] = TOTAL(vcountstosum)	; add up all the counts
	    ENDIF ELSE BEGIN
	       summed_flux.Al[i] =  FILL_D
	       summed_counts.Al[i] = FILL_L
	    ENDELSE
	 ENDFOR
        ; Si 10
         n = Enum[10]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[10,Emin[10]+i]
            R3idx = R3_ctrs[10,Emin[10]+i]
            R4idx = R4_ctrs[10,Emin[10]+i]
            unsectored_counts.Si[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.Si_fctrs[i].Ehi-Factors.Si_fctrs[i].Elo) * $
	               Factors.Si_fctrs[i].geom[header.DyThState] * $
		       Factors.Si_fctrs[i].effic[header.DyThState] * Masks.Si_masks[i].mask
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.Si[i] ge 0L)) THEN $
               unsectored_flux.Si[i] = unsectored_counts.Si[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.Si[i] = FILL_D
	       unsectored_counts.Si[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed silicon energy ranges 
	 FOR i=0,Esum[10]-1 DO BEGIN
	    sums = Esumdefs.Si_sums[*,i]		; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)] 	; eliminate the -1 entries  
	    datatosum = unsectored_flux.Si[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.Si[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_flux.Si[i] =  TOTAL(intermeddata)/ TOTAL(vdivisorstouse)	; compute the intensity for the summed energy-range
	       summed_counts.Si[i] = TOTAL(vcountstosum)	; add up all the counts
;	       IF i EQ 0 THEN BEGIN
;	          print, vsums
;	          print, indices
;	          print, divisors
;	          print, divisorstouse
;	          print, vdivisorstouse
;	       ENDIF
	    ENDIF ELSE BEGIN
	       summed_flux.Si[i] =  FILL_D
	       summed_counts.Si[i] = FILL_L
	    ENDELSE
	 ENDFOR
        ; S 11
         n = Enum[11]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[11,Emin[11]+i]
            R3idx = R3_ctrs[11,Emin[11]+i]
            R4idx = R4_ctrs[11,Emin[11]+i]
            unsectored_counts.S[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.S_fctrs[i].Ehi-Factors.S_fctrs[i].Elo) * $
	               Factors.S_fctrs[i].geom[header.DyThState] * $
		       Factors.S_fctrs[i].effic[header.DyThState] * Masks.S_masks[i].mask
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.S[i] ge 0L)) THEN $
               unsectored_flux.S[i] = unsectored_counts.S[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.S[i] = FILL_D
	       unsectored_counts.S[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed sulphur energy ranges 
	 FOR i=0,Esum[11]-1 DO BEGIN
	    sums = Esumdefs.S_sums[*,i]			; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)] 	; eliminate the -1 entries  
	    datatosum = unsectored_flux.S[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.S[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_flux.S[i] =  TOTAL(intermeddata)/ TOTAL(vdivisorstouse)	; compute the intensity for the summed energy-range
	       summed_counts.S[i] = TOTAL(vcountstosum)	; add up all the counts
	    ENDIF ELSE BEGIN
	       summed_flux.S[i] =  FILL_D
	       summed_counts.S[i] = FILL_L
	    ENDELSE
	 ENDFOR
        ; Ar 12
         n = Enum[12]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[12,Emin[12]+i]
            R3idx = R3_ctrs[12,Emin[12]+i]
            R4idx = R4_ctrs[12,Emin[12]+i]
            unsectored_counts.Ar[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.Ar_fctrs[i].Ehi-Factors.Ar_fctrs[i].Elo) * $
	               Factors.Ar_fctrs[i].geom[header.DyThState] * $
		       Factors.Ar_fctrs[i].effic[header.DyThState] * Masks.Ar_masks[i].mask
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.Ar[i] ge 0L)) THEN $
               unsectored_flux.Ar[i] = unsectored_counts.Ar[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.Ar[i] = FILL_D
	       unsectored_counts.Ar[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed argon energy ranges 
	 FOR i=0,Esum[12]-1 DO BEGIN
	    sums = Esumdefs.Ar_sums[*,i]		; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)] 	; eliminate the -1 entries  
	    datatosum = unsectored_flux.Ar[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.Ar[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_flux.Ar[i] =  TOTAL(intermeddata)/ TOTAL(vdivisorstouse)	; compute the intensity for the summed energy-range
	       summed_counts.Ar[i] = TOTAL(vcountstosum)	; add up all the counts
	    ENDIF ELSE BEGIN
	       summed_flux.Ar[i] =  FILL_D
	       summed_counts.Ar[i] = FILL_L
	    ENDELSE
	 ENDFOR
        ; Ca 13
         n = Enum[13]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[13,Emin[13]+i]
            R3idx = R3_ctrs[13,Emin[13]+i]
            R4idx = R4_ctrs[13,Emin[13]+i]
            unsectored_counts.Ca[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.Ca_fctrs[i].Ehi-Factors.Ca_fctrs[i].Elo) * $
	               Factors.Ca_fctrs[i].geom[header.DyThState] * $
		       Factors.Ca_fctrs[i].effic[header.DyThState] * Masks.Ca_masks[i].mask
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.Ca[i] ge 0L)) THEN $
               unsectored_flux.Ca[i] = unsectored_counts.Ca[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.Ca[i] = FILL_D
	       unsectored_counts.Ca[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed calcium energy ranges 
	 FOR i=0,Esum[13]-1 DO BEGIN
	    sums = Esumdefs.Ca_sums[*,i]		; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)] 	; eliminate the -1 entries  
	    datatosum = unsectored_flux.Ca[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.Ca[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_flux.Ca[i] =  TOTAL(intermeddata)/ TOTAL(vdivisorstouse)	; compute the intensity for the summed energy-range
	       summed_counts.Ca[i] = TOTAL(vcountstosum)	; add up all the counts
	    ENDIF ELSE BEGIN
	       summed_flux.Ca[i] =  FILL_D
	       summed_counts.Ca[i] = FILL_L
	    ENDELSE
	 ENDFOR
        ; Fe 14
         n = Enum[14]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[14,Emin[14]+i]
            R3idx = R3_ctrs[14,Emin[14]+i]
            R4idx = R4_ctrs[14,Emin[14]+i]
            unsectored_counts.Fe[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.Fe_fctrs[i].Ehi-Factors.Fe_fctrs[i].Elo) * $
	               Factors.Fe_fctrs[i].geom[header.DyThState] * $
		       Factors.Fe_fctrs[i].effic[header.DyThState] * Masks.Fe_masks[i].mask
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.Fe[i] ge 0L)) THEN $
               unsectored_flux.Fe[i] = unsectored_counts.Fe[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.Fe[i] = FILL_D
	       unsectored_counts.Fe[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed iron energy ranges 
	 FOR i=0,Esum[14]-1 DO BEGIN
	    sums = Esumdefs.Fe_sums[*,i]		; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)] 	; eliminate the -1 entries  
	    datatosum = unsectored_flux.Fe[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.Fe[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_flux.Fe[i] =  TOTAL(intermeddata)/ TOTAL(vdivisorstouse)	; compute the intensity for the summed energy-range
	       summed_counts.Fe[i] = TOTAL(vcountstosum)	; add up all the counts
	    ENDIF ELSE BEGIN
	       summed_flux.Fe[i] =  FILL_D
	       summed_counts.Fe[i] = FILL_L
	    ENDELSE
	 ENDFOR
        ; Ni 15
         n = Enum[15]
         FOR i=0,n-1 DO BEGIN
            R2idx = R2_ctrs[15,Emin[15]+i]
            R3idx = R3_ctrs[15,Emin[15]+i]
            R4idx = R4_ctrs[15,Emin[15]+i]
            unsectored_counts.Ni[i] = sum_rates(R2idx, R3idx, R4idx, decom_rates)
	    divisors[i] = header.Livetime * (Factors.Ni_fctrs[i].Ehi-Factors.Ni_fctrs[i].Elo) * $
	               Factors.Ni_fctrs[i].geom[header.DyThState] * $
		       Factors.Ni_fctrs[i].effic[header.DyThState] * Masks.Ni_masks[i].mask
            IF ((divisors[i] gt 0.0) AND (unsectored_counts.Ni[i] ge 0L)) THEN $
               unsectored_flux.Ni[i] = unsectored_counts.Ni[i] / divisors[i] $
            ELSE BEGIN
	       unsectored_flux.Ni[i] = FILL_D
	       unsectored_counts.Ni[i] = FILL_L
	    ENDELSE
         ENDFOR
	; summed nickel energy ranges 
	 FOR i=0,Esum[15]-1 DO BEGIN
	    sums = Esumdefs.Ni_sums[*,i]		; pick out the Ebins to sum up for this summed bin
	    vsums = sums[WHERE(sums GE 0, nvsums)] 	; eliminate the -1 entries  
	    datatosum = unsectored_flux.Ni[vsums]	; make an array of the fluxes to sum
	    indices = WHERE(datatosum GE 0, nindices)	; pick out the indices of the non-fill fluxes
	    IF nindices EQ nvsums THEN BEGIN
	       vdatatosum = datatosum[indices]		; just the non-fill fluxes
	       divisorstouse = divisors[vsums]		; make an array of the relevant divisors
	       vdivisorstouse = divisorstouse[indices]	; just the divisors for the non-fill fluxes
       	       countstosum = unsectored_counts.Ni[vsums]	; make an array of the relevant counts
	       vcountstosum = countstosum[indices]	; just the counts for the non-fill fluxes
	       intermeddata = vdatatosum * vdivisorstouse	; weight each flux by its divisor
	       summed_flux.Ni[i] =  TOTAL(intermeddata)/ TOTAL(vdivisorstouse)	; compute the intensity for the summed energy-range
	       summed_counts.Ni[i] = TOTAL(vcountstosum)	; add up all the counts
	    ENDIF ELSE BEGIN
	       summed_flux.Ni[i] =  FILL_D
	       summed_counts.Ni[i] = FILL_L
	    ENDELSE
	 ENDFOR
      
      ; build level 1 sectored counts and fluxes

         IF header.SCid EQ 234 THEN BEGIN			; Ahead
	    timestr = '2010-11-22T15:01:32.000'
	 ENDIF ELSE BEGIN
	    timestr = '2010-11-22T17:01:07.000'			; Behind
	 ENDELSE
         sectt_utc = str2utc(timestr)
	 secttime = utc2tai(sectt_utc, /NOCORRECT)
      
         IF pkt_times[0] LT secttime THEN BEGIN		; sectored data before 2011-11-22

	 sectored_counts.H_lo = decom_rates.SECTORATES[0:15]
         sectored_counts.He3_lo = decom_rates.SECTORATES[16:31]
         sectored_counts.He4_lo = decom_rates.SECTORATES[32:47]
         sectored_counts.He4_hi = decom_rates.SECTORATES[48:63]
         sectored_counts.CNO_lo = decom_rates.SECTORATES[64:79]
         sectored_counts.CNO_hi = decom_rates.SECTORATES[80:95]
         sectored_counts.NeMgSi_lo = decom_rates.SECTORATES[96:111]
         sectored_counts.NeMgSi_hi = decom_rates.SECTORATES[112:127]
         sectored_counts.Fe_lo = decom_rates.SECTORATES[128:143]
         sectored_counts.Fe_hi = decom_rates.SECTORATES[144:159]
	 FOR i=0,15 DO BEGIN
	    sectored_counts.H_vlo = FILL_L
	    sectored_counts.H_hi = FILL_L
	    sectored_counts.Fe_wide = FILL_L
	    sectored_flux.H_vlo = FILL_D
	    sectored_flux.H_hi = FILL_D
	    sectored_flux.Fe_wide = FILL_D
	 ENDFOR

         sfset = 0
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.H_lo[i] ge 0L)) THEN $
	       sectored_flux.H_lo[i] = sectored_counts.H_lo[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.H_lo[i] = FILL_D
	       sectored_counts.H_lo[i] = FILL_L
	    ENDELSE
	    IF i EQ 0 THEN BEGIN
;	       print, header.DyThState,  header.Livetime, SFactors[i+sfset].effic[header.DyThState], divisor
;	       print, SFactors[0].effic
	    ENDIF
	 ENDFOR
      
         ; Do not use masks here, as we are computing He3 for inclusion in the elemental He flux here
	 sfset = 16
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] 
	    IF ((divisor gt 0.0) AND (sectored_counts.He3_lo[i] ge 0L)) THEN $
	       sectored_flux.He3_lo[i] = sectored_counts.He3_lo[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.He3_lo[i] = FILL_D
	       sectored_counts.He3_lo[i] = FILL_L
	    ENDELSE
	 ENDFOR
      
         sfset = 32
	 sdivisors = DBLARR(16)
	 FOR i=0,15 DO BEGIN
	    sdivisors[i] = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((sdivisors[i] gt 0.0) AND (sectored_counts.He4_lo[i] ge 0L)) THEN $
	       sectored_flux.He4_lo[i] = sectored_counts.He4_lo[i] / sdivisors[i] $
	    ELSE BEGIN
	       sectored_flux.He4_lo[i] = FILL_D
	       sectored_counts.He4_lo[i] = FILL_L
	    ENDELSE
	 ENDFOR

         ; Elemental helium 4 - 6 MeV/nuc
	 FOR i=0,15 DO BEGIN
	    IF ((sectored_counts.He3_lo[i] ge 0L) AND (sectored_counts.He4_lo[i] ge 0L)) THEN BEGIN
	       sectored_counts.He_lo[i] = sectored_counts.He3_lo[i] + sectored_counts.He4_lo[i]
	       sectored_flux.He_lo[i] = sectored_counts.He_lo[i] / sdivisors[i]
	    ENDIF ELSE BEGIN
	       sectored_flux.He_lo[i] = FILL_D
	       sectored_counts.He_lo[i] = FILL_L
	    ENDELSE
	    IF ( summed_He_masks[0] eq 0.0 ) THEN BEGIN
	       sectored_flux.He_lo[i] = FILL_D
	       sectored_counts.He_lo[i] = FILL_L
	    ENDIF
	 ENDFOR
	 
         ; Redo He3, using masks
	 sfset = 16
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.He3_lo[i] ge 0L)) THEN $
	       sectored_flux.He3_lo[i] = sectored_counts.He3_lo[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.He3_lo[i] = FILL_D
	       sectored_counts.He3_lo[i] = FILL_L
	    ENDELSE
	 ENDFOR
      
         sfset = 48
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.He4_hi[i] ge 0L)) THEN $
	       sectored_flux.He4_hi[i] = sectored_counts.He4_hi[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.He4_hi[i] = FILL_D
	       sectored_counts.He4_hi[i] = FILL_L
	    ENDELSE
	 ENDFOR
      
         sfset = 64
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.CNO_lo[i] ge 0L)) THEN $
	       sectored_flux.CNO_lo[i] = sectored_counts.CNO_lo[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.CNO_lo[i] = FILL_D
	       sectored_counts.CNO_lo[i] = FILL_L
	    ENDELSE
	 ENDFOR
      
         sfset = 80
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.CNO_hi[i] ge 0L)) THEN $
	       sectored_flux.CNO_hi[i] = sectored_counts.CNO_hi[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.CNO_hi[i] = FILL_D
	       sectored_counts.CNO_hi[i] = FILL_L
	    ENDELSE
	 ENDFOR
      
         sfset = 96
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.NeMgSi_lo[i] ge 0L)) THEN $
	       sectored_flux.NeMgSi_lo[i] = sectored_counts.NeMgSi_lo[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.NeMgSi_lo[i] = FILL_D
	       sectored_counts.NeMgSi_lo[i] = FILL_L
	    ENDELSE
	 ENDFOR
      
         sfset = 112
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.NeMgSi_hi[i] ge 0L)) THEN $
	       sectored_flux.NeMgSi_hi[i] = sectored_counts.NeMgSi_hi[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.NeMgSi_hi[i] = FILL_D
	       sectored_counts.NeMgSi_hi[i] = FILL_L
	    ENDELSE
	 ENDFOR
      
         sfset = 128
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.Fe_lo[i] ge 0L)) THEN $
	       sectored_flux.Fe_lo[i] = sectored_counts.Fe_lo[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.Fe_lo[i] = FILL_D
	       sectored_counts.Fe_lo[i] = FILL_L
	    ENDELSE
	 ENDFOR
      
         sfset = 144
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.Fe_hi[i] ge 0L)) THEN $
	       sectored_flux.Fe_hi[i] = sectored_counts.Fe_hi[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.Fe_hi[i] = FILL_D
	       sectored_counts.Fe_hi[i] = FILL_L
	    ENDELSE
	 ENDFOR

	 ENDIF ELSE BEGIN		; sectored data after 2011-11-22

	 sectored_counts.H_lo = decom_rates.SECTORATES[0:15]
         sectored_counts.H_vlo = decom_rates.SECTORATES[16:31]
         sectored_counts.He4_lo = decom_rates.SECTORATES[32:47]
         sectored_counts.He4_hi = decom_rates.SECTORATES[48:63]
         sectored_counts.CNO_lo = decom_rates.SECTORATES[64:79]
         sectored_counts.CNO_hi = decom_rates.SECTORATES[80:95]
         sectored_counts.NeMgSi_lo = decom_rates.SECTORATES[96:111]
         sectored_counts.NeMgSi_hi = decom_rates.SECTORATES[112:127]
         sectored_counts.Fe_wide = decom_rates.SECTORATES[128:143]
         sectored_counts.H_hi = decom_rates.SECTORATES[144:159]
	 FOR i=0,15 DO BEGIN
	    sectored_counts.He3_lo = FILL_L
	    sectored_counts.He_lo = FILL_L
	    sectored_counts.Fe_lo = FILL_L
	    sectored_counts.Fe_hi = FILL_L
	    sectored_flux.He3_lo = FILL_D
	    sectored_flux.He_lo = FILL_D
	    sectored_flux.Fe_lo = FILL_D
	    sectored_flux.Fe_hi = FILL_D
	 ENDFOR
	 
         sfset = 0
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.H_lo[i] ge 0L)) THEN $
	       sectored_flux.H_lo[i] = sectored_counts.H_lo[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.H_lo[i] = FILL_D
	       sectored_counts.H_lo[i] = FILL_L
	    ENDELSE
	    IF i EQ 0 THEN BEGIN
;	       print, header.DyThState,  header.Livetime, SFactors[i+sfset].effic[header.DyThState], divisor
;	       print, SFactors[0].effic
	    ENDIF
	 ENDFOR
      
	 sfset = 16
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.H_vlo[i] ge 0L)) THEN $
	       sectored_flux.H_vlo[i] = sectored_counts.H_vlo[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.H_vlo[i] = FILL_D
	       sectored_counts.H_vlo[i] = FILL_L
	    ENDELSE
	 ENDFOR
      
         sfset = 32
	 sdivisors = DBLARR(16)
	 FOR i=0,15 DO BEGIN
	    sdivisors[i] = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((sdivisors[i] gt 0.0) AND (sectored_counts.He4_lo[i] ge 0L)) THEN $
	       sectored_flux.He4_lo[i] = sectored_counts.He4_lo[i] / sdivisors[i] $
	    ELSE BEGIN
	       sectored_flux.He4_lo[i] = FILL_D
	       sectored_counts.He4_lo[i] = FILL_L
	    ENDELSE
	 ENDFOR

         sfset = 48
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.He4_hi[i] ge 0L)) THEN $
	       sectored_flux.He4_hi[i] = sectored_counts.He4_hi[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.He4_hi[i] = FILL_D
	       sectored_counts.He4_hi[i] = FILL_L
	    ENDELSE
	 ENDFOR
      
         sfset = 64
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.CNO_lo[i] ge 0L)) THEN $
	       sectored_flux.CNO_lo[i] = sectored_counts.CNO_lo[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.CNO_lo[i] = FILL_D
	       sectored_counts.CNO_lo[i] = FILL_L
	    ENDELSE
	 ENDFOR
      
         sfset = 80
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.CNO_hi[i] ge 0L)) THEN $
	       sectored_flux.CNO_hi[i] = sectored_counts.CNO_hi[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.CNO_hi[i] = FILL_D
	       sectored_counts.CNO_hi[i] = FILL_L
	    ENDELSE
	 ENDFOR
      
         sfset = 96
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.NeMgSi_lo[i] ge 0L)) THEN $
	       sectored_flux.NeMgSi_lo[i] = sectored_counts.NeMgSi_lo[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.NeMgSi_lo[i] = FILL_D
	       sectored_counts.NeMgSi_lo[i] = FILL_L
	    ENDELSE
	 ENDFOR
      
         sfset = 112
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.NeMgSi_hi[i] ge 0L)) THEN $
	       sectored_flux.NeMgSi_hi[i] = sectored_counts.NeMgSi_hi[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.NeMgSi_hi[i] = FILL_D
	       sectored_counts.NeMgSi_hi[i] = FILL_L
	    ENDELSE
	 ENDFOR
      
         sfset = 128
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.Fe_wide[i] ge 0L)) THEN $
	       sectored_flux.Fe_wide[i] = sectored_counts.Fe_wide[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.Fe_wide[i] = FILL_D
	       sectored_counts.Fe_wide[i] = FILL_L
	    ENDELSE
	 ENDFOR
      
         sfset = 144
	 FOR i=0,15 DO BEGIN
	    divisor = header.Livetime * (SFactors[i+sfset].Ehi-SFactors[i+sfset].Elo) * $
	               Sgeom[i] * SFactors[i+sfset].effic[header.DyThState] * SMasks[i+sfset].mask
	    IF ((divisor gt 0.0) AND (sectored_counts.H_hi[i] ge 0L)) THEN $
	       sectored_flux.H_hi[i] = sectored_counts.H_hi[i] / divisor $
	    ELSE BEGIN
	       sectored_flux.H_hi[i] = FILL_D
	       sectored_counts.H_hi[i] = FILL_L
	    ENDELSE
	 ENDFOR
	 ENDELSE
      
      ; assemble LET level 1 data structure
         IF doext eq 1 THEN $
	    letl1data = {header:header, unsectored_flux: unsectored_flux, $
                                     unsectored_counts: unsectored_counts, $
                                     sectored_flux: sectored_flux, $
                                     sectored_counts: sectored_counts, $
                                     summed_flux: summed_flux, $
                                     summed_counts: summed_counts, $
                                     raw_rates: decom_rates $
				     } $
         ELSE $
            letl1data = {header:header, unsectored_flux: unsectored_flux, $
                                     unsectored_counts: unsectored_counts, $
                                     sectored_flux: sectored_flux, $
                                     sectored_counts: sectored_counts, $
                                     summed_flux: summed_flux, $
                                     summed_counts: summed_counts $
				     }

      ; set good_frame flag
      good_frame = 1
      first_frame = 0
      
END      
