PRO LET_LEVEL1, PACKET, LETL1DATA
;+
; Project     :	STEREO - IMPACT, LET
;
; Name        : LET_LEVEL1
;
; Purpose     :	Assemble a LET science frame from LET science packets, calls let_process
;
; Category    :	STEREO, IMPACT, LET
;
; Explanation :	
;
; Syntax      :	LET_LEVEL1, PACKET, LETL1DATA
;
; Examples    :	
;
; Inputs      :	PACKET - a LET science telemetry packet (apid=580)
;
; Opt. Inputs :	None.
;
; Outputs     :	LETL1DATA = A structure containing LET Level 1 data
;               if ready, undefined otherwise
;
; Opt. Outputs:	None.
;
; Keywords    :	None
;
; Calls       :	delvarx
;
; Common      :	let_l1
;
; Restrictions:	
;
; Side effects:	None.
;
; Prev. Hist. :	None.
;
; History     :	1-Jul-2005, Andrew Davis, Caltech - initial version
;		12-Mar-2007, Andrew Davis, Caltech - added code to handle
; 		last minute of each day properly.
;
; Contact     :	ad@srl.caltech.edu
;
;

; see let_l1_common.pro
common let_l1 

; LET usually outputs 16 packets w/ apid=580 per minute.
; These 16 packets constitute one LET "science frame".
; The packet sequence number is embedded in the 4 low-order bits
; of the subseconds field of the packet secondary header.
; However, if there are not enough events to fill 16 packets,
; LET will output less than 16 packets, possibly as low as 6.
; In normal operation, STIM events will fill the telemetry, thus
; ensuring 16 packets/frame, but we must be able to handle
; non-normal operation.

pkt_seq = packet.pkt.subsec and 15
good_frame = 0

; Note: LET packet checksum should be verified


IF (pkt_seq eq 0) THEN BEGIN
   IF (first_zero_seq eq 0) THEN BEGIN
      first_zero_seq = 1
      pkt_times[pkt_seq] = packet.pkt.seconds
      scids[pkt_seq] = packet.grh.SPACECRAFT_ID
      ; extract LET science frame data from packet
      start_loc = 0
      letscifrm[start_loc:start_loc+LETPKTDATASIZE-1] = packet.dat[0:LETPKTDATASIZE-1]
   ENDIF ELSE BEGIN
      ; process accumulated science frame, if we have at least the first 6 packets
      ; printf,-2, prev_pkt_seq
      IF (prev_pkt_seq ge 5) THEN BEGIN
         IF array_equal(pkt_times[0:prev_pkt_seq],pkt_times[0]) AND $
	    array_equal(scids[0:prev_pkt_seq],scids[0]) THEN BEGIN
            let_process, letl1data, scids[0], good_frame
	 ENDIF
      ENDIF

      ; Prepare for next frame - reset pkt timestamps and zero the letsciframe
      replicate_inplace, pkt_times, 0
      replicate_inplace, scids, 0U
      replicate_inplace, letscifrm, 0B
      pkt_times[pkt_seq] = packet.pkt.seconds
      scids[pkt_seq] = packet.grh.SPACECRAFT_ID
      ; extract LET science frame data from packet
      start_loc = 0
      letscifrm[start_loc:start_loc+LETPKTDATASIZE-1] = packet.dat[0:LETPKTDATASIZE-1]
   ENDELSE
ENDIF ELSE BEGIN
   IF (first_zero_seq eq 1) THEN BEGIN
      pkt_times[pkt_seq] = packet.pkt.seconds
      scids[pkt_seq] = packet.grh.SPACECRAFT_ID
      ; extract LET science frame data from packet
      start_loc = pkt_seq*LETPKTDATASIZE
      letscifrm[start_loc:start_loc+LETPKTDATASIZE-1] = packet.dat[0:LETPKTDATASIZE-1]
   ENDIF
   ; don't begin accumulating a sci frame until first_zero_seq eq 1
ENDELSE

; Check for end-of-day, and process last minute of day if so
utc = tai2utc(double(packet.pkt.seconds), /EXTERNAL, /NOCORRECT)
; print, utc.hour, utc.minute, prev_pkt_seq 
IF (utc.hour eq 23) AND (utc.minute eq 59) THEN BEGIN
      IF (prev_pkt_seq eq 14) THEN BEGIN
         IF array_equal(pkt_times[0:prev_pkt_seq],pkt_times[0]) AND $
	    array_equal(scids[0:prev_pkt_seq],scids[0]) THEN BEGIN
            let_process, letl1data, scids[0], good_frame
	 ENDIF
      ENDIF
ENDIF

prev_pkt_seq = pkt_seq
IF (good_frame eq 0 ) THEN DELVARX, letl1data

END
